/********************************************************************************
 *  Projektname:	AERO
 *  Filename:		autils.c
 *  Filetyp:		Modul
 ********************************************************************************
 *  WICHTIG:            Dieser Quelltext benoetigt einen Compiler, der Struktur-
 *                      Zuweisungen beherrscht (z.B. ANSI-C, gcc). Sind also z.B.
 *                      a und b Strukturen gleichen Typs, muss bei b=a; der ge-
 *                      samte Inhalt von a nach b kopiert werden. Ausserdem werden
 *                      Prototypen benutzt (gibt es i.a. nicht bei Standard-C).
 ********************************************************************************
 *  Modulname:		autils.c
 *  Version:		5
 *  letzte Aenderung:	Montag, 13. Februar 1995, 12:02:06
 *  Autor:  		Hartmut 
 *  Status:		recht funktionstuechtige Hilfsroutinen
 *										
 *  importierte Bezeichner:	-
 *  -----------------------
 *
 *  exportierte Bezeichner:
 *  -----------------------
 *  Colormap a_colormap: Farbpalette, die fuer diesen Screen gilt.
 *  
 *  ausserdem alle Routinen, siehe autils.h
 *
 *  Beschreibung:								
 *  -------------								
 *  In diesem Modul sind einige Routinen zusammengefasst, die bei der
 *  Visualisierung (Darstellung und Animation) benoetigt werden. es handelt
 *  sich dabei ausschliesslich um kleine Unterroutinen, wie Matrix- oder
 *  Vektor-Operationen.
 *										
 *  Fehler:
 *  -------
 *
 *  Versionsgeschichte:								
 *  -------------------								
 *  1:    Auskopplung der Routinen aus anzeige.c
 *  2:    MatrixVon2Punkten() hinzugekommen
 *  3:    TReal2AXKoord() hinzugekommen 
 *  4:    AQuaternionMult(), AGetColor() hinzugekommen (-> V1.3)
 *  5:    NormiereQuaternion() hinzugekommen (-> V1.5)
 ********************************************************************************/


#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "visualisierung.h"

/***********************************************************************
 *  Farbtabelle, die auf dem momentanen Screen gilt
 ***********************************************************************/

Colormap a_colormap;
TBoolean ColWarningDone = FALSE;


/***********************************************************************
 *  Fehlerroutine bei Darstellung
 ***********************************************************************/

void AFehler(char *fehlertext)
{
     fprintf(stderr, fehlertext);
}


/****************************************************************************
 *  Diese folgenden Routinen sind nur fuer die Fehlersuche gedacht, darum auch 
 *  keine ordentliche Dokumentation.
 *
 *  DumpMatrix: angegebene 4x4-Matrix ausgeben, dazu den Text
 *  DumpKoerper: Daten zu dem angegebenen Koerper ausgeben
 ***************************************************************************/

void DumpMatrix(AMatrix matrix, char *text)
{
     printf("Matrix %s: %2.6f %2.6f %2.6f %2.6f\n",text, matrix[0][0], matrix[0][1], matrix[0][2], matrix[0][3]);
     printf("Matrix %s: %2.6f %2.6f %2.6f %2.6f\n",text, matrix[1][0], matrix[1][1], matrix[1][2], matrix[1][3]);
     printf("Matrix %s: %2.6f %2.6f %2.6f %2.6f\n",text, matrix[2][0], matrix[2][1], matrix[2][2], matrix[2][3]);
     printf("Matrix %s: %2.6f %2.6f %2.6f %2.6f\n",text, matrix[3][0], matrix[3][1], matrix[3][2], matrix[3][3]);

}

void DumpKoerper(TKoerper *k)
{
     printf("Koerper: Art: %d\n", k->Art);
     printf("         pos: (%f, %f, %f)\n", k->Position[0], k->Position[1], k->Position[2]);
     printf("         size: (%f, %f, %f)\n", k->Form.Quader.KantenLaenge[0], k->Form.Quader.KantenLaenge[1], k->Form.Quader.KantenLaenge[2]);
     printf("         q:   (%f, %f, %f, %f)\n", k->q[0], k->q[1], k->q[2], k->q[3]);
}






/*****************************************************************************
 *  Funktion:       AXKoord TReal2AXKoord(TReal wert);
 *
 *  Parameter:      wert: Real-Wert, der nach AXKoord gewandelt werden soll
 *
 *  Rueckgabewert:  -
 *
 *  Importierte Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Der angegebene Real-Wert wird in den Typ AXKoord gewandelt. Wird dabei
 *  der maximale Wertebereich von AXKoord ueberschritten, wird der maximale
 *  bzw. minimale Wert selbst angenommen.
 *****************************************************************************/

AXKoord TReal2AXKoord(TReal wert)
{
     if (wert > (TReal) AXmax) return AXmax;
     if (wert < (TReal) AXmin) return AXmin;
     return (AXKoord) wert;
}




/*****************************************************************************
 *  Funktion:       void DeleteListe(void *zeiger);
 *  Parameter:      zeiger: Zeiger auf die Liste, die freigegeben werden soll
 *
 *  Rueckgabewert:  -
 *
 *  Importierte Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Liste mit Bloecken, die mit malloc() reserviert wurden, wieder freigeben.
 *  Dazu muss die Liste derart verkettet sein, dass immer der erste Eintrag
 *  jedes Elements der Zeiger auf das naechste Element ist. Folgen keine wei-
 *  teren Elemente, muss der letzte Zeiger NULL sein. Die Art der Einzelele-
 *  mente ist egal.
 *****************************************************************************/

void DeleteListe(void *zeiger)
{
     void *next;
     
     while (zeiger!=NULL)
     {
	  next = *(void **)zeiger;	  /* Zeiger auf naechstes Element merken */
	  free(zeiger);			  /* Element selbst freigeben */
	  zeiger = next;		  /* weiter mit naechstem Element */
     }
}



/*****************************************************************************
 *  Funktion:       void NormiereQuaternion(TQuaternion q);
 *
 *  Parameter:      q:  Quaternion, das normiert werden soll
 *
 *  Rueckgabewert:  q:  ist auch Rueckgabeparameter
 *
 *  Importierte Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Das Quaternion q wird normiert, so dass q[0]^2+q[1]^2+q[2]^2+q[3]^2=1.
 *  Dies ist aufgrund von Rechenungenauigkeit manchmal nicht der Fall. Da
 *  aber manchmal eine sehr hohe Genauigkeit verlangt werden muss (z.B. in
 *  EulerWinkelFromQuaternion()), kann diese Routine aufgerufen werden.
 *  Danach ist obige Beziehung (Betrag=1) gegeben, falls q[0]^2<=1 ist. Ist
 *  q[0]^2>1, dann bleibt das Quaternion unveraendert.
 *****************************************************************************/

void NormiereQuaternion(TQuaternion q)
{
     TReal skalierfaktor;

     skalierfaktor = q[1]*q[1]+q[2]*q[2]+q[3]*q[3];
     if ((q[0]*q[0] < 1.0) && !(NAHE_0(skalierfaktor)))
     {
	  skalierfaktor = sqrt((1-q[0]*q[0])/skalierfaktor);
	  q[1] *= skalierfaktor;
	  q[2] *= skalierfaktor;
	  q[3] *= skalierfaktor;
     }
}



/*****************************************************************************
 *  Funktion:       void MatrixFromQuaternion(TQuaternion q, AMatrix M);
 *
 *  Parameter:      q:  Quaternion, das in eine Matrix gewandelt werden soll
 *
 *  Rueckgabewert:  M:  Zielmatrix (4x4)
 *
 *  Importierte Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Das Quaternion q wird (moeglichst effizient) in die Matrix M gewandelt.
 *****************************************************************************/

void MatrixFromQuaternion(TQuaternion q, AMatrix M)
{
     TReal q00, q01, q02, q03,
           q12, q13, q23;		  /* Hilfsvariablen fuer Effizienz */


     q00 = q[0]*q[0];
     q01 = q[0]*q[1];
     q02 = q[0]*q[2];
     q03 = q[0]*q[3];
     q12 = q[1]*q[2];
     q13 = q[1]*q[3];
     q23 = q[2]*q[3];

     M[0][0] = 2*(q00+q[1]*q[1])-1;
     M[0][1] = 2*(q12+q03);
     M[0][2] = 2*(q13-q02);
     M[1][0] = 2*(q12-q03);
     M[1][1] = 2*(q00+q[2]*q[2])-1;
     M[1][2] = 2*(q23+q01);
     M[2][0] = 2*(q13+q02);
     M[2][1] = 2*(q23-q01);
     M[2][2] = 2*(q00+q[3]*q[3])-1;
     M[3][0] = M[3][1] = M[3][2] = M[0][3] = M[1][3] = M[2][3] = 0;
     M[3][3] = 1;
}




/*****************************************************************************
 *  Funktion:       void MatrixFrom_I_Quaternion(TQuaternion q, AMatrix M);
 *
 *  Parameter:      q:  Quaternion, dessen Inverses in eine Matrix gewandelt
 *                      werden soll
 *
 *  Rueckgabewert:  M:  Zielmatrix (4x4)
 *
 *  Importierte Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Das Quaternion (q^-1) wird (moeglichst effizient) in die Matrix M gewandelt.
 *****************************************************************************/

void MatrixFrom_I_Quaternion(TQuaternion q, AMatrix M)
{
     TReal q00, q01, q02, q03,
           q12, q13, q23;		  /* Hilfsvariablen fuer Effizienz */


     q00 = q[0]*q[0];
     q01 = q[0]*q[1];
     q02 = q[0]*q[2];
     q03 = q[0]*q[3];
     q12 = q[1]*q[2];
     q13 = q[1]*q[3];
     q23 = q[2]*q[3];

     M[0][0] = 2*(q00+q[1]*q[1])-1;
     M[0][1] = 2*(q12-q03);
     M[0][2] = 2*(q13+q02);
     M[1][0] = 2*(q12+q03);
     M[1][1] = 2*(q00+q[2]*q[2])-1;
     M[1][2] = 2*(q23-q01);
     M[2][0] = 2*(q13-q02);
     M[2][1] = 2*(q23+q01);
     M[2][2] = 2*(q00+q[3]*q[3])-1;
     M[3][0] = M[3][1] = M[3][2] = M[0][3] = M[1][3] = M[2][3] = 0;
     M[3][3] = 1;
}





/*****************************************************************************
 *  Funktion:       void AQuaternionMult(TQuaternion a, TQuaternion b, 
 *                                                          TQuaternion x)
 *  Parameter:      a, b: die beiden Quellquaternions
 *
 *  Rueckgabewert:  x:    Zielquaternion, es darf sich dabei NICHT um a
 *                        oder b handeln, da keine Kopie angelegt wird.
 *
 *  Importierte Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Das Quaternion a wird mit dem Quaternion b nach den Regeln der
 *  Quaternionmultiplikation multipliziert und das Ergebnis in x abgelegt.
 *  ACHTUNG! Da weder von a, noch von b eine Kopie angelegt wird und C bei
 *  Uebergabe von Feldern an Funktionen nur einen Zeiger uebergibt, werden
 *  bei a und b die Originalwerte verwendet. Gibt man als Ziel x also a oder
 *  b selbst an, wird das Ergebnis grob falsch!
 *****************************************************************************/

void AQuaternionMult(TQuaternion a, TQuaternion b, TQuaternion x)
{
    x[0] = a[0]*b[0] - (a[1]*b[1]+a[2]*b[2]+a[3]*b[3]);

    x[1] = a[2]*b[3]-a[3]*b[2]+a[0]*b[1]+b[0]*a[1];
    x[2] = a[3]*b[1]-a[1]*b[3]+a[0]*b[2]+b[0]*a[2];
    x[3] = a[1]*b[2]-a[2]*b[1]+a[0]*b[3]+b[0]*a[3];
}




/*****************************************************************************
 *  Funktion:       TBoolean MatrixVon2Punkten(TVektor p1, TVektor p2, AMatrix M)
 *
 *  Parameter:      p1, p2: Verbindungspunkte
 *
 *  Rueckgabewert:  TRUE:  es liess sich keine Matrix ausrechnen (Verbindung
 *                         hat Laenge 0)
 *                  FALSE: die berechnete Matrix (4x4) steht in M 
 *
 *  Importierte Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Es wird die Matrix berechnet, die einen Koerper, der bei (0,0,0) bis
 *  (0,0,1) liegt, an die Position p1 bis p2 transformiert. Dabei wird der
 *  Koerper gestreckt und entsprechend gedreht.
 *  Diese Routine wird z.B. benoetigt, dreidimensional definierte
 *  Verbindungen wie Feder und Daempfer von ihrer Defaultposition an die
 *  entsprechenden Verbindungspunkte zu transformieren.
 *****************************************************************************/

TBoolean MatrixVon2Punkten(TVektor p1, TVektor p2, AMatrix M)
{
     TVektor v;				  /* Vektor von p1 nach p2 */
     TReal laenge;			  /* Laenge dieses Vektors */

     v[0] = p2[0]-p1[0];		  /* Vektor von p1 nach p2 */
     v[1] = p2[1]-p1[1];
     v[2] = p2[2]-p1[2];
     
     laenge = sqrt(v[0]*v[0]+v[1]*v[1]+v[2]*v[2]);
     
     if (NAHE_0(laenge)) return TRUE;	  /* Laenge 0: keine Richtung bestimmbar */


     /* Wenn die Verbindung genau in z-Richtung laeuft (Drehung um 180 */
     /* Grad), kann keine Drehachse bestimmt werden. Dann eine Defaultmatrix */
     /* benutzen, die um die x-Achse dreht. */ 
     if (NAHE_0(v[2]-laenge))
     {
	  M[0][0] = 1;			  /* Default-Matrix */
	  M[0][1] = M[0][2] = 0;
	  M[0][3] = p1[0];
	  M[1][0] = M[1][2] = 0;
	  M[1][1] = -1;
	  M[1][3] = p1[1];
	  M[2][0] = M[2][1] = 0;
	  M[2][2] = -laenge;
	  M[2][3] = p1[2];
	  M[3][0] = M[3][1] = M[3][2] = 0;
	  M[3][3] = 1;
     }
     else
     {
	  TReal hilfs;

	  hilfs = laenge*(laenge+v[2]);	  /* Matrix ueber Quaternion bestimmen */
	  M[0][0] = (v[1]*v[1]+v[2]*(v[2]+laenge))/hilfs;
	  M[0][1] = -v[0]*v[1]/hilfs;
	  M[0][2] = v[0];
	  M[0][3] = p1[0];
	  M[1][0] = M[0][1];
	  M[1][1] = (v[0]*v[0]+v[2]*(v[2]+laenge))/hilfs;
	  M[1][2] = v[1];
	  M[1][3] = p1[1];
	  M[2][0] = -v[0]/laenge;
	  M[2][1] = -v[1]/laenge;
	  M[2][2] = v[2];
	  M[2][3] = p1[2];
	  M[3][0] = M[3][1] = M[3][2] = 0;
	  M[3][3] = 1;
     }

     return FALSE;
}



/*****************************************************************************
 *  Funktion:       void ANormiereVektor(TVektor v);
 *  Parameter:      v: der zu normierende Vektor
 *
 *  Rueckgabewert:  v: der normierte Vektor
 *
 *  Importierte Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Der angegebene Vektor wird auf Laenge 1 normiert. Dabei wird immer der
 *  Vektor selbst veraendert!
 *****************************************************************************/

void ANormiereVektor(TVektor v)
{
     TReal laenge;

     laenge = sqrt(v[0]*v[0]+v[1]*v[1]+v[2]*v[2]);
     v[0]  /= laenge;
     v[1]  /= laenge;
     v[2]  /= laenge;
}



/*****************************************************************************
 *  Funktion:       void AMatrixMult(AMatrix m1, AMatrix m2, AMatrix ziel);
 *  Parameter:      m1, m2: die beiden Quellmatrizen (4x4; berechnet: m1*m2)
 *
 *  Rueckgabewert:  ziel:   Zielmatrix (4x4), es darf sich dabei NICHT um m1
 *                          oder m2 handeln, da keine Kopie angelegt wird.
 *
 *  Importierte Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Die Matrix m1 wird mit der Matrix m2 multipliziert und das Ergebnis in
 *  ziel abgelegt. ACHTUNG! Da weder von m1, noch von m2 eine Kopie angelegt
 *  wird und C bei Uebergabe von Feldern an Funktionen nur einen Zeiger
 *  uebergibt, werden bei m1 und m2 die Originalwerte verwendet. Gibt man
 *  als ziel also m1 oder m2 selbst an, wird das Ergebnis grob falsch!
 *****************************************************************************/

void AMatrixMult(AMatrix m1, AMatrix m2, AMatrix ziel)
{
     int i, j;

     for (i=0; i<4; i++)
	  for (j=0; j<4; j++)
	       ziel[i][j]=m1[i][0]*m2[0][j]+m1[i][1]*m2[1][j]
		    +m1[i][2]*m2[2][j]+m1[i][3]*m2[3][j];

}




/*****************************************************************************
 *  Funktion:       void ATransformation(TVektor punkt, AMatrix M, 
 *                                                 TReal x, TReal y, TReal z)
 *  Parameter:      M:     Entsprechende Matrix
 *                  x,y,z: Koordinaten des Punktes vor der Abbildung
 *                            
 *  Rueckgabewert:  punkt: Koordinaten des abgebildeten Punktes
 *
 *  Importierte Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Multiplikation der Matrix M mit dem Punkt x,y,z, das Ergebnis wird in punkt
 *  abgelegt: punkt = M * (x,y,z)
 *****************************************************************************/

void ATransformation(TVektor punkt, AMatrix M, TReal x, TReal y, TReal z)
{
     punkt[0] = (M[0][0]*x + M[0][1]*y + M[0][2]*z + M[0][3]);
     punkt[1] = (M[1][0]*x + M[1][1]*y + M[1][2]*z + M[1][3]);
     punkt[2] = (M[2][0]*x + M[2][1]*y + M[2][2]*z + M[2][3]);
}






/*****************************************************************************
 *  Funktion:       void AGetColor(Display *display, Colormap farbtabelle, 
 *                                               XColor *farbe, char *routine)
 *  Parameter:      display,
 *                  farbtabelle,
 *                  farbe        : Parameter wie bei XAllocColor()
 *                  routine      : Name der aufrufenden Routine
 *                            
 *  Rueckgabewert:  punkt: Koordinaten des abgebildeten Punktes
 *
 *  Importierte Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Es wird versucht, mit XAllocColor() die angegebene Farbe als Read-only
 *  zu allozieren. Wenn dies nicht klappt, wird eine Warnung (Text:
 *  "WARNING! xyz(): Color Lookup Table full, can't allocate color")
 *  ausgegeben und versucht, Schwarz als Farbe zu allozieren. Klappt dies
 *  auch nicht, wird eine Fehlermeldung ausgegeben. In letzterem Fall wird
 *  keine Farbe alloziert, was in einem Programmabbruch bei einem
 *  XFreeColors() enden kann. Ansonsten kann die allozierte Farbe wieder
 *  freigegeben werden.
 *  Anmerkung: die Warnung wird nur ausgegeben, wenn das Flag ColWarningDone
 *  nicht gesetzt ist. Nach einer Ausgabe der Warnung wird das Flag
 *  automatisch gesetzt.
 *****************************************************************************/

#define TEXTLEN 200			  /* Maximale Textlaenge der Meldung */

void AGetColor(Display *display, Colormap farbtabelle, XColor *farbe, char *routine)
{
     if (!XAllocColor(display, farbtabelle, farbe))
     {
	  if (!ColWarningDone)
	  {
	       char fehlermeldung[TEXTLEN]= "WARNING! ";
	       strncat(fehlermeldung, routine, TEXTLEN);
	       fehlermeldung[TEXTLEN-1]='\0';
	       strncat(fehlermeldung, ": Color Lookup Table full, can't allocate color\n", TEXTLEN);
	       fehlermeldung[TEXTLEN-1]='\0';
	       AFehler(fehlermeldung);
	       ColWarningDone = TRUE;
	  }

	  farbe->red   = 0;
	  farbe->green = 0;
	  farbe->blue  = 0;
	  farbe->flags = DoRed | DoGreen | DoBlue;
	  if (!XAllocColor(display, farbtabelle, farbe))
	  {
	       AFehler("ERROR! Can't even allocate black; program may crash at next color change!!!!\n");
	  }
     }
}


/*****************************************************************************
 *  Funktion:       void AClearColWarningFlag(void)
 *
 *  Parameter:      -
 *                            
 *  Rueckgabewert:  -
 *
 *  Importierte Bezeichner: ColWarningDone
 *
 *  Beschreibung:
 *  -------------
 *  Das Flag wird, das angibt, ob schon eine Warnung ueber eine volle Farb-
 *  tabelle erfolgt ist, wird zurueckgesetzt, d.h. die Warnung wird beim
 *  naechsten erfolglosen Aufruf von AGetColor() wieder ausgegeben.
 *****************************************************************************/

void AClearColWarningFlag(void)
{
     ColWarningDone = FALSE;
}
