/*****************************************************************************
 *  Projektname:	AERO
 *  Filename:		inventor.c
 *  Filetyp:		Modul
 *****************************************************************************
 *  WICHTIG:            Dieser Quelltext benoetigt einen Compiler, der Struktur-
 *                      Zuweisungen beherrscht (z.B. ANSI-C, gcc). Sind also z.B.
 *                      a und b Strukturen gleichen Typs, muss bei b=a; der ge-
 *                      samte Inhalt von a nach b kopiert werden.
 *****************************************************************************
 *  Modulname:		Anzeige
 *  Version:		1
 *  letzte Aenderung:	Donnerstag, 29. August 1996, 19:26:42
 *  Autor:  		Hartmut 
 *  Status:		Test
 *
 *  imp. Bezeichner:
 *  ----------------
 *  server_tid:	        PVM-Task-ID des Inventor-Server Prozesses (invserv)
 *
 *  exp. Bezeichner:
 *  ----------------
 *  void OpenInventor():
 *          Bereitstellen des Inventorprozesses; muss immer aufgerufen
 *          werden, wenn das Animationsfenster geoeffnet wird. Kann z.B.
 *          dazu verwendet werden, ein iconisiertes Fenster zu oeffnen.
 *  void CloseInventor():
 *          Schliessen der momentanen Inventorverbindung; muss immer
 *          aufgerufen werden, wenn das Animationsfenster geschlossen wird.
 *          Kann z.B. dazu verwendet werden, ein Fenster zu iconisieren.
 *  void ExitInventor():
 *          Dem Inventor-Serverprozess wird mitgeteilt, dass er sich nun
 *          beenden muss. Danach erfolgt keine Ausgabe mehr an den Server,
 *          es wird davon ausgegangen, dass der Prozess anschliessend nicht
 *          mehr existiert. 
 *  void SetSyncMode(TBoolean onoff):
 *	    Synchronisation mit Inventor-Prozess ein- (TRUE) oder
 *	    ausschalten (FALSE)
 *  void WaitForSync():
 *          Es wird (mit einem Timeout) gewartet, bis der
 *          Inventor-Serverprozess eine Rueckmeldung fuer den Frame gibt.
 *  void ShowInventor(TKamera camera, TZustand *zustand, int modus):
 *          Anzeige des angegebenen Zustands auf dem Inventorfenster.
 *          Dabei wird durch modus die Art der Anzeige gesteuert (modus ist
 *          ein Bitvektor, gleiche Funktion wie bei AnzeigeDarstellung()):
 *              Bit 0: Fussboden zeichnen (1) oder nicht (0)
 *              Bit 1: Koordinatenachsen der Fenster zeichnen (1) oder nicht (0)
 *              Bit 2: Koordinatenachsen der einzelnen Koerper zeichnen oder
 *                     nicht (0)
 *		
 *
 *  Beschreibung:
 *  -------------
 *  Ausgabe der Objekte ueber PVM an den Inventor-Serverprozess invserv.
 *
 *  Fehler:
 *  -------
 *  
 *
 *  Versionsgeschichte:
 *  -------------------
 *  0:	Grundversion
 *  1:	Erweiterung um SetSyncMode(); kommentiert
 *
 *   in Arbeit:
 *   ----------
 * 
 *   zu tun:
 *   -------
 *   
 *****************************************************************************/

#undef INVDEBUG

#ifdef INVENTOR
#include <pvm3.h>
#include <limits.h>
#include "inventor.h"
#include "Kamera.h"
#include "grundtypen.h"
#include "koerper.h"
#include "verbindungen.h"
#include "zustand.h"
#include "anzeigeconst.h"
#include "autils.h"

/*------ Variablen ---------------------------------------------------------*/

extern int server_tid;		    /* PVM-Task-ID des Serverprozesses */

static unsigned long framenumber = 0; /* laufende Framenummer */
static unsigned int unackframes = 0;  /* unbestaetigte Frames (sliding */
				      /* window) */

static TBoolean syncflag = TRUE;    /* starte synchronisiert */



/*------ Funktionen -------------------------------------------------------*/

/*****************************************************************************
 *  Funktion:       void packinfo(TKoerper *koerper)
 *
 *  Parameter:      koerper: aktueller Koerper
 *                     
 *  Rueckgabewert:
 *
 *  Import. Bez.:
 *
 *  Beschreibung:
 *  -------------
 *  Ausgabe der Daten, die fuer alle Koerper gemeinsam sind: Position,
 *  Orientierung und Farbe.
 *****************************************************************************/

static void packinfo(TKoerper *koerper)
{
     pvm_pkdouble(koerper->Position, 3, 1);
     pvm_pkdouble(koerper->q, 4, 1);
     pvm_pkushort(&koerper->R, 1, 1);
     pvm_pkushort(&koerper->G, 1, 1);
     pvm_pkushort(&koerper->B, 1, 1);

     /* hier koennten weitere Dinge kommen, wie z.B. Material */ 
}



/*****************************************************************************
 *  Funktion:       void ProjektionKoerper(TKoerper *koerper)
 *
 *  Parameter:      koerper: Liste aller Koerper
 *                     
 *  Rueckgabewert:
 *
 *  Import. Bez.:
 *
 *  Beschreibung:
 *  -------------
 *  Ausgabe der Koerper ueber PVM: dazu werden Position, Orientierung und
 *  Farbe uebertragen. Dazu kommen koerperspezifische Angaben zur Groesse. 
 *  Anmerkung: es wird davon ausgegangen, dass die Koerperliste nicht
 *             leer ist!
 *****************************************************************************/

static void ProjektionKoerper(TKoerper *koerper)
{
     do
     {
          if (!(koerper->AStatus & HiddenON))
	  {
	       switch (koerper->Art)
	       {
		 case KUGEL:
		    pvm_packf("%d", SPHERE);
		    pvm_pkdouble(&koerper->Form.Kugel.Radius, 1, 1);
		    packinfo(koerper);
		    break;

		 case QUADER:
		    pvm_packf("%d", CUBOID);
		    pvm_pkdouble(koerper->Form.Quader.KantenLaenge, 3, 1);
		    packinfo(koerper);
		    break; 

		 case EBENE:
		    pvm_packf("%d", PLANE);
		    packinfo(koerper);
		    break;

		 case ZYLINDER:
		    pvm_packf("%d", CYLINDER);
		    pvm_pkdouble(&koerper->Form.Zylinder.Radius, 1, 1);
		    pvm_pkdouble(&koerper->Form.Zylinder.Hoehe, 1, 1);
	            packinfo(koerper);
		    break;

		 case MPUNKT:		  /* wie Nagel */
		 case PUNKT:		  /* wie Nagel */
		 case NAGEL:
		    pvm_packf("%d",  POINT);
		    packinfo(koerper);
		    break;
		    
		 case ZUSGESOBJ:	  /* rekursiv Einzelkoerper ausgeben */
		    ProjektionKoerper(koerper->Form.ZusGesObj.KoerperListe);
		    break;
		    
		 default:
		    break;
	       }
	      
	  }
           koerper = koerper->Naechster;
     } while (koerper!=NULL);
}



/*****************************************************************************
 *  Funktion:       void ProjektionVerbindungen(TVerbindung *verbindung)
 *
 *  Parameter:      verbindung: Liste aller Verbindungen
 *                     
 *  Rueckgabewert:
 *
 *  Import. Bez.:
 *
 *  Beschreibung:
 *  -------------
 *  Ausgabe der Verbindungen ueber PVM: dazu werden Anfangs- und Endpunkt
 *  der Verbindung durch entsprechende Transformationen ermittelt und
 *  zusammen mit der Verbindungsart und ggf. verbindungsspezifischer
 *  Parameter an den Inventor-Serverprozess uebermittelt.
 *  Anmerkung: es wird davon ausgegangen, dass die Verbindungsliste nicht
 *             leer ist!
 *****************************************************************************/

static void ProjektionVerbindungen(TVerbindung *verbindung)
{
    AMatrix M_verb;
    TVektor p1, p2;

    do
    {
	 /* bestimme ersten Verbindungspunkt */
	 /*   a) Koerperorientierung */
	 MatrixFromQuaternion(verbindung->Koerper1->q, M_verb);
	 
	 /*   b) Koerperposition */
	 M_verb[0][3] = verbindung->Koerper1->Position[0];
	 M_verb[1][3] = verbindung->Koerper1->Position[1];
	 M_verb[2][3] = verbindung->Koerper1->Position[2];

	 /*   c) Transformation des Punktes */
	 ATransformation(p1, M_verb, verbindung->VPunkt1[0], 
			             verbindung->VPunkt1[1],
			             verbindung->VPunkt1[2]);

	 /* bestimme zweiten Verbindungspunkt entsprechend*/
	 MatrixFromQuaternion(verbindung->Koerper2->q, M_verb);
	  
	 M_verb[0][3] = verbindung->Koerper2->Position[0];
	 M_verb[1][3] = verbindung->Koerper2->Position[1];
	 M_verb[2][3] = verbindung->Koerper2->Position[2];
	  
	 ATransformation(p2, M_verb, verbindung->VPunkt2[0], 
                                     verbindung->VPunkt2[1],
			             verbindung->VPunkt2[2]);

	 /* Sende Verbindungstyp */
	 switch (verbindung->Art)
	 {
	   case FEDER:
	      pvm_packf("%d", SPRING);
	      pvm_pkdouble(&verbindung->VerParameter.Feder.Ruhelaenge, 1, 1);
	      pvm_pkdouble(&verbindung->VerParameter.Feder.Federkonstante, 1, 1);
	      break;

	   case STANGE:
	      pvm_packf("%d", ROD);
	      break;


	   case DAEMPFER:
	      pvm_packf("%d", DAMPER);
	      break;

	   case GELENK:
	      pvm_packf("%d", JOINT);
	      break;
	
	   default:
	      break;
	 }
	
	 pvm_pkdouble(p1, 3, 1);
	 pvm_pkdouble(p2, 3, 1);

	 verbindung = verbindung->Naechste;
    } while (verbindung != NULL);
}



/*****************************************************************************
 *  Funktion:       void ShowInventor(TKamera camera, TZustand *zustand, 
 *                                                               int modus)
 *
 *  Parameter:      camera:  aktuelle Kameraeinstellung
 *                  zustand: aktueller Zustand der Welt
 *                  modus:   Bitvektor, der Anzeigemodus angibt. Wird
 *                           momentan ignoriert.
 *                             Bit 0: Fussboden zeichnen (1) oder nicht (0)
 *                             Bit 1: Koordinatenachsen der Fenster zeichnen
 *                                    (1) oder nicht (0)
 *                             Bit 2: Koordinatenachsen der einzelnen
 *                                    Koerper zeichnen (1) oder nicht (0)
 *                          
 *                     
 *  Rueckgabewert:
 *
 *  Import. Bez.:   server_tid:  PVM-Task-ID des Serverprozesses
 *
 *  Beschreibung:
 *  -------------
 *  Anzeige des angegebenen Zustands auf dem Inventorfenster. Da die
 *  Inventor-Library nicht mit dem Athena-Widget-Set zusammenarbeitet,
 *  musste der Inventorteil in einen eigenen Prozess ausgelagert werden. Der
 *  Datenaustausch erfolgt ueber PVM.
 *  Dabei werden folgende Daten fuer den Zustand erzeugt und gesendet:
 *    1. eine laufende Identifikationsnummer des Frames
 *    2. Kameraposition und -orientierung (CAMERA)
 *    3. alle Verbindungen (SPRING, ROD, DAMPER, JOINT)
 *    4. alle Koerper (SPHERE, CUBOID, PLANE, CYLINDER, POINT);
 *       zusammengesetzte Objekte werden aufgespalten
 *    5. Endekennung (NOOBJECT)
 *****************************************************************************/

/* export */
void ShowInventor(TKamera camera, TZustand *zustand, int animmodus)
{
     TKoerper *koerper;
     unsigned long ackframe;

     pvm_initsend(PvmDataDefault);

     framenumber++;
     unackframes++;
     pvm_pkulong(&framenumber, 1, 1);
     if (zustand != NULL)
     {
     	  pvm_packf("%d", CAMERA);
	  pvm_pkdouble(camera.Position, 3, 1);
	  pvm_pkdouble(camera.Richtung, 4, 1);
	  pvm_pkdouble(&camera.Zoom, 1, 1);

	  koerper = zustand->Koerper;
	  if (zustand->Verbindungen != NULL)
	  {
	       ProjektionVerbindungen(zustand->Verbindungen);
	  }
	  if (koerper != NULL)
	  {
	       ProjektionKoerper(koerper);
	  }
     }
     pvm_packf("%d", NOOBJECT);
     pvm_send(server_tid, FRAME);
}



/*****************************************************************************
 *  Funktion:       void SetSyncMode(TBoolean onoff)
 *
 *  Parameter:      onoff: Synchronisation an (TRUE) oder aus (FALSE)
 *                          
 *                     
 *  Rueckgabewert:
 *
 *  Import. Bez.:   server_tid:  PVM-Task-ID des Serverprozesses
 *
 *  Beschreibung:
 *  -------------
 *  Dem Inventorprozess wird mitgeteilt, dass jetzt eine Ausgabe erfolgt.
 *  Dieser kann dann z.B. das Inventor-Fenster oeffnen. Ausserdem wird der
 *  momentane Synchronisations-Modus uebermittelt.
 *****************************************************************************/

/* export */
void SetSyncMode(TBoolean onoff)
{
     syncflag = onoff;
     if (syncflag)
	  pvm_send(server_tid, SYNC_ON);
     else
	  pvm_send(server_tid, SYNC_OFF);
}


/*****************************************************************************
 *  Funktion:       void WaitForSync()
 *
 *  Parameter:
 *                     
 *  Rueckgabewert:
 *
 *  Import. Bez.:   server_tid:  PVM-Task-ID des Serverprozesses
 *
 *  Beschreibung:
 *  -------------
 *  Es wird gewartet, bis der Inventor-Serverprozess eine Rueckmeldung fuer
 *  den Frame gibt. Erfolgt die Meldung nicht in einer bestimmten Zeit, dann
 *  wird trotzdem zurueckgekehrt (Timeout). Erfolgt doch spaeter wieder eine
 *  Meldung, dann wird dem Inventor Zeit gegeben, "aufzuholen".
 *  Dabei kann der Inventor bis zu SLIDINGWINDOW Bilder hinterher"hinken",
 *  bei SLIDINGWINDOW=0 laeuft der Inventor bildsynchron.
 *  Kommt mehr als MAXTIMEOUTS mal hintereinander ein Timeout, dann geht die
 *  Routine davon aus, dass der Inventorprozess irgendwie gestoppt wurde
 *  (z.B. beendet). Sie kehrt dann kuenftig ohne Timeout sofort zurueck.
 *  Sollte sich der Inventorprozess spaeter doch wieder melden, kehrt die
 *  Routine zum urspruenglichen Verhalten mit Timeouts zurueck.
 *****************************************************************************/

static unsigned long ntimeouts=0;

#define SLIDINGWINDOW 3			  /* Frames */
#define WAITSTEP 0.05			  /* Sekunden */
#define TIMEOUT 1			  /* Sekunden */
#define MAXTIMEOUTS 5

void WaitForSync()
{
     unsigned int i;
     unsigned long ackframe;
     TBoolean done=FALSE;
     TBoolean unack;

     unack = (unackframes<SLIDINGWINDOW);
     if ((ntimeouts>MAXTIMEOUTS) || unack)
	  i = 1;
     else
	  i = TIMEOUT/WAITSTEP;
     do
     {
	  while ((i>0) && !pvm_probe(server_tid, ACK)) /* timeout abwarten */
	  {
	       sginap(WAITSTEP*CLK_TCK);  /* etwas warten */
	       i--;
	  }
	  if (i>0)			  /* Nachricht? */
	  {
	       ntimeouts = 0;
	       unackframes--;
	       pvm_recv(server_tid, ACK);
	       pvm_upkulong(&ackframe, 1, 1);
	       /* Wenn es ein aelteres Bild war, dann warte nochmal, */
	       /* vielleicht kommt die Meldung fuer's naechste ja noch */
	       if (ackframe>=framenumber-SLIDINGWINDOW) done=TRUE;
	  }
	  else
	  {
	       done=TRUE;
	       if (!unack) ntimeouts++;	  /* timeout */
	  }
     } while (!done);
}


/*****************************************************************************
 *  Funktion:       void OpenInventor()
 *
 *  Parameter:
 *                     
 *  Rueckgabewert:
 *
 *  Import. Bez.:   server_tid:  PVM-Task-ID des Serverprozesses
 *
 *  Beschreibung:
 *  -------------
 *  Dem Inventorprozess wird mitgeteilt, dass jetzt eine Ausgabe erfolgt.
 *  Dieser kann dann z.B. das Inventor-Fenster oeffnen. Ausserdem wird der
 *  momentane Synchronisations-Modus uebermittelt.
 *****************************************************************************/

/* export */
void OpenInventor()
{
     pvm_initsend(PvmDataDefault);
     pvm_send(server_tid, OPEN_INVENTOR);
     SetSyncMode(syncflag);
}



/*****************************************************************************
 *  Funktion:       void CloseInventor()
 *
 *  Parameter:
 *                     
 *  Rueckgabewert:
 *
 *  Import. Bez.:   server_tid:  PVM-Task-ID des Serverprozesses
 *
 *  Beschreibung:
 *  -------------
 *  Dem Inventorprozess wird mitgeteilt, dass jetzt das Animationsfenster
 *  geschlossen wurde und dass in naechster Zeit nicht mit einer Ausgabe zu
 *  rechnen ist. Dieser kann dann z.B. das Inventor-Fenster schliessen.
 *****************************************************************************/

/* export */
void CloseInventor()
{
     pvm_initsend(PvmDataDefault);
     pvm_send(server_tid, CLOSE_INVENTOR);
}



/*****************************************************************************
 *  Funktion:       void ExitInventor()
 *
 *  Parameter:
 *                     
 *  Rueckgabewert:
 *
 *  Import. Bez.:   server_tid:  PVM-Task-ID des Serverprozesses
 *
 *  Beschreibung:
 *  -------------
 *  Dem Inventor-Serverprozess wird mitgeteilt, dass er sich nun beenden
 *  muss. Danach erfolgt keine Ausgabe mehr an den Server, es wird davon
 *  ausgegangen, dass der Prozess anschliessend nicht mehr existiert.
 *****************************************************************************/

/* export */
void ExitInventor()
{
     pvm_initsend(PvmDataDefault);
     pvm_send(server_tid, EXIT_INVENTOR);
     server_tid=-1;
}


#endif

