/***************************************************************************
                      improvConfig.cpp  -  description
                             -------------------
    begin                :
    copyright            : (C) 2002 by Leon Koch
    email                : koch-la@ee.uwa.edu.au
 ***************************************************************************/

//#include "improvConfig.h"
#include "Improv.h"

improvConfig::improvConfig() {
	defaultFilename = (char *)malloc(sizeof(char)*(2+strlen(getenv("HOME"))+strlen(CONFIG_FILENAME)));
	if(defaultFilename == NULL) {
		fprintf(stderr,"Error: No memory available for configuration filename\n");
		return;
	}
	defaultFilename = strcpy(defaultFilename,getenv("HOME"));
	if(defaultFilename[strlen(defaultFilename)-1]!='/') {
		defaultFilename = strcat(defaultFilename,"/");
	}
	defaultFilename = strcat(defaultFilename,CONFIG_FILENAME);
	if(defaultFilename[strlen(defaultFilename)-1]!='\0') {
		defaultFilename = strcat(defaultFilename,"\0");
	}
	this->clear();
}

improvConfig::~improvConfig() {
	if(useSequence!=NULL) free(useSequence);
	free(this);
}

void improvConfig::load(int argc, char **argv) {
	char c;
	int windowWidth=0;
	int windowHeight=0;
	int numWindows=0;
	int numSources=0;
	int option_index=0;
	char *sequencePath=NULL;
	char *pluginPath=NULL;
	char *useSequence=NULL;

	static struct option long_options[] = {
		{"help", 0, (int *)NULL, 0},
		{0, 0, 0, 0}
	};

	optind = 1;
	
	while( (c = getopt_long(argc, argv, "x:y:p:s:n:i:u:hqc:v", long_options, &option_index)) != -1 ) {
		switch(c) {
			case 'x':
				windowWidth = atoi(optarg);
				this->setWindowWidth(windowWidth);
				break;
			case 'y':
				windowHeight = atoi(optarg);
				this->setWindowHeight(windowHeight);
				break;
			case 'p':
				pluginPath = strdup(optarg);
				this->setPluginPath(pluginPath);
				break;
			case 's':
				sequencePath = strdup(optarg);
				this->setSequencePath(sequencePath);
				break;
			case 'n':
				numWindows = atoi(optarg);
				this->setNumWindows(numWindows);
				widgets = (widgetConfig **)malloc(sizeof(widgetConfig *)*this->getNumWindows());
				if(widgets == NULL) {
					fprintf(stderr,"Error: No memory available for widget configuration\n");
					return;
				}
				break;
			case 'i':
				numSources = atoi(optarg);
				this->setNumSources(numSources);
				break;
			case 'u':
				useSequence = strdup(optarg);
				this->useSequence = useSequence;
				break;
			case 'q':
				this->cameraType = QCAM;
				break;
			case 'c':
				free(this->defaultFilename);
				this->defaultFilename = strdup(optarg);
				break;
			case 'v':
				this->cameraType = V4L2;
				break;
			case 0:
				//long_options[option_index].name)
				//if (optarg) printf (" with arg %s", optarg);
			case 'h':
			default:
				printf("\n");
				printf(" Usage:   improvqt <options>\n\n");
				printf(" Options: -x width  : Set the width of the image frames.\n");
				printf("          -y height : Set the height of the image frames.\n");
				printf("          -p path   : Set 'path' as the plugin path.\n");
				printf("          -s path   : Set 'path' as the sequence path.\n");
				printf("          -n number : Set the total number of windows.\n");
				printf("          -i number : Set the number of input windows.\n");
				printf("          -u name   : Use the image sequence 'name'.\n");
				printf("          -q        : Use the QuickCam (default is Video 4 Linux).\n");
				printf("          -v        : Use the Video 4 Linux 2 device.\n");
				printf("          -c file   : Load configuration from 'file'.\n");
				printf("\n");
				exit(0);
				break;
		}
	}
	/* check environment args */
	if(sequencePath==NULL) {
		if( (sequencePath = getenv("IMPROV_SEQUENCE_PATH")) != NULL ) {
			if(opendir(sequencePath) == NULL) {
				fprintf(stderr, "Error: Sequence path %s not found\n", sequencePath);
			}
			else {
				this->setSequencePath(sequencePath);
			}
		}
	}
	if(pluginPath==NULL) {
		if( (pluginPath = getenv("IMPROV_PLUGIN_PATH")) != NULL ) {
			if(opendir(pluginPath) == NULL) {
				fprintf(stderr, "Error: Plugin path %s not found\n", pluginPath);
			}
			else {
				this->setPluginPath(pluginPath);				
			}
		}
	}
	this->modified = false;	
}

void improvConfig::loadDefaults(void) {
	bool warning=false;
	int cols;
	// load defaults file.
	if(this->getSequencePath() == NULL) {
		fprintf(stderr, "Error: No sequence path defined - Using . as sequence path instead\n");
		this->setSequencePath(".");
		warning=true;
	}
	if(this->getPluginPath() == NULL) {
		fprintf(stderr, "Error: No pluign path defined - Using . as plugin path instead\n");
		this->setPluginPath(".");
		warning=true;
	}
	if(this->getWindowWidth()<0) this->setWindowWidth(DEFAULT_WINDOW_WIDTH);
	if(this->getWindowHeight()<0) this->setWindowHeight(DEFAULT_WINDOW_HEIGHT);
	if(this->getNumSources()<0) this->setNumSources(1);
	if(this->getNumWindows()<0) this->setNumWindows(DEFAULT_NUMBER_WINDOWS);
	if(this->getCameraType() == OTHER) this->setCameraType(V4L);
	if(this->numWidgetConfigs()==0) {
		switch(this->numWindows) {
			case 2:
			case 4:
				this->numCols = 2;
				cols = 2;
				break;
			default:
				this->numCols = 3;
				cols = 3;
				break;
		}
		this->numRows = (int)ceil((double)this->numWindows/(double)this->numCols);
		this->numWidgets = this->numWindows;
		widgets = (widgetConfig **)malloc(sizeof(widgetConfig *)*this->numWindows);
		for(int i=0;i<this->numSources; i++) {
			widgets[i] = (widgetConfig *)malloc(sizeof(widgetConfig));
			widgets[i]->id = i;
			widgets[i]->source = true;
			widgets[i]->sourceId = i;
			widgets[i]->position = 0;
			widgets[i]->row = i;
			widgets[i]->col = 0;
		}
		for(int i=this->numSources, row=0, col=0, source=-1, position=1; i<this->numWindows; i++) {
			if((row<numSources)&&(col==0)) {col=1; source++; position=1;}
			widgets[i] = (widgetConfig *)malloc(sizeof(widgetConfig));
			widgets[i]->id = i;
			widgets[i]->source = false;
			widgets[i]->sourceId = source;
			widgets[i]->position = position++;
			widgets[i]->row = row;
			widgets[i]->col = col++;
			if(col>=cols) {row++; col=0;}
		}
	}
	if(warning) {
		fprintf(stderr, "       Please set the plugin and sequence paths, in the Options dialog (Ctrl+e)\n");
	}
	this->modified = false;
}

bool improvConfig::load(void) {
	return load(defaultFilename);
}

bool improvConfig::load(char *filename) {
	int i=0, start, end;
	char *option, *value;
	char line[MAXLINE];
	FILE *fp;
	fp = fopen(filename,"r");
	if(fp != NULL) {
		while(fgets(line,MAXLINE,fp) != NULL) {
			// skip starting spaces
			while((line[i]==' ')||(line[i]=='\t')) {
				i++;
			}
			if((line[i]!='#')&&(line[i]!=';')) {
				// find the option name
				start = i;
				while((line[i]!='=')&&(line[i]!=' ')) {
					i++;
				}
				end = i;
				option = substr(line,start,end);
				// skip the in between stuff
				while((line[i]=='=')||(line[i]==' ')||(line[i]=='\t')) {
					i++;
				}
				// find the value
				start = i;
				while((line[i]!='#')&&(line[i]!='\n')&&(line[i]!='\0')) {
					i++;
				}
				end = i;
				value = substr(line,start,end);
				if((strcmp(option,"sequencePath")==0)&&(this->getSequencePath()==NULL)) {
					this->setSequencePath(value);
				}
				if((strcmp(option,"pluginPath")==0)&&(this->getPluginPath()==NULL)) {
					this->setPluginPath(value);
				}
				if((strcmp(option,"useSequence")==0)&&(this->getUseSequence()==NULL)) {
					this->setUseSequence(value);
				}
				if((strcmp(option,"windowWidth")==0)&&(this->getWindowWidth()<0)) {
					this->setWindowWidth(atoi(value));
				}
				if((strcmp(option,"windowHeight")==0)&&(this->getWindowHeight()<0)) {
					this->setWindowHeight(atoi(value));
				}
				if((strcmp(option,"numWindows")==0)&&(this->getNumWindows()<0)) {
					this->setNumWindows(atoi(value));
					widgets = (widgetConfig **)malloc(sizeof(widgetConfig *)*this->getNumWindows());
					if(widgets == NULL) {
						fprintf(stderr,"Error: No memory available for widget configuration\n");
						return false;
					}
				}
				if((strcmp(option,"numSources")==0)&&(this->getNumSources()<0)) {
					this->setNumSources(atoi(value));
				}
				if((strcmp(option,"cameraType")==0)&&(this->getCameraType()==OTHER)) {
					if(strcmp(value,"V4L")==0) this->setCameraType(V4L);
					if(strcmp(value,"QCAM")==0) this->setCameraType(QCAM);
				}
				/*if(strcmp(option,"fn")==0) {
					int row, col, position, uniqueId;
					char *pluginName;
					numPlugins++;
					if(numPlugins==1) plugins=(pluginInstance**)malloc(sizeof(pluginInstance*)*1);
					else plugins = (pluginInstance**)realloc(plugins,sizeof(pluginInstance*)*numPlugins);
					plugins[numPlugins-1] = (pluginInstance*)malloc(sizeof(pluginInstance));
					//fn = row,col,posn,uniqueid,name
					pluginName = strdup((char*)(strrchr(value,',')+1));
					sscanf(value,"%d,%d,%d,%d,", &row, &col, &position, &uniqueId);
					plugins[numPlugins-1]->row = row;
					plugins[numPlugins-1]->col = col;
					plugins[numPlugins-1]->posn = position;
					plugins[numPlugins-1]->uniqueId = uniqueId;
					plugins[numPlugins-1]->name = pluginName;
				}*/
				if(strcmp(option,"window")==0) {
					if(numWidgets<this->getNumWindows()) {
						int row=0, col=0, theBool=0, sourceId=0, position=0;
						sscanf(value,"%d,%d,%d,%d,%d", &row, &col, &theBool, &sourceId, &position);
						widgets[numWidgets] = (widgetConfig*)malloc(sizeof(widgetConfig));
						if(theBool==1) {
							widgets[numWidgets]->source = true;
							widgets[numWidgets]->position = 0;
						}
						else {
						  widgets[numWidgets]->source = false;
							widgets[numWidgets]->position = position;
						}
						widgets[numWidgets]->row = row;
						if(row+1>this->numRows) this->numRows = row+1;
						widgets[numWidgets]->col = col;
						if(col+1>this->numCols) this->numCols = col+1;
						widgets[numWidgets]->sourceId = sourceId;
						widgets[numWidgets]->id = numWidgets;
						numWidgets++;	
					}
					else {
						fprintf(stderr,"Error: Too many custom widget details in config file %s - resorting to default layout.\n", filename);
						return false;
					}
				}
			}
			i=0;
		}
		if((numWidgets>0)&&(numWidgets<this->getNumWindows())) {
			fprintf(stderr,"Error: Insufficent number of custom widget details in config file %s - resorting to default layout.\n", filename);
			return false;
		}
		this->modified = false;
		fclose(fp);
		return true;	
	}
	else {
		return false;
	}	
}

bool improvConfig::save(void) {
	this->saveAs(defaultFilename);
	return true;
}
		
bool improvConfig::saveAs(char *filename) {
	//QValueList<SeqWidget*> seqWidgetList;
	//QValueList<SeqWidget*>::iterator it;
	//PluginItem **pluginItems;
	//Improv* improv = Improv::Instance(NULL, NULL);
	FILE *fp;
	fp = fopen(filename,"w");
	if(fp != NULL) {
		fprintf(fp,"# ImprovQT v%s Config File\n", VERSION);
		fprintf(fp,"sequencePath = %s\n",this->getSequencePath());
		fprintf(fp,"pluginPath = %s\n",this->getPluginPath());
		if(this->getUseSequence() != NULL)
			fprintf(fp,"useSequence = %s\n",this->getUseSequence());
		fprintf(fp,"windowWidth = %d\n",this->getWindowWidth());
		fprintf(fp,"windowHeight = %d\n",this->getWindowHeight());
		fprintf(fp,"numWindows = %d\n", this->getNumWindows());
		fprintf(fp,"numSources = %d\n", this->getNumSources());
		if(this->getCameraType() == V4L) fprintf(fp,"cameraType = V4L\n");
		if(this->getCameraType() == QCAM) fprintf(fp,"cameraType = QCAM\n");
		fprintf(fp,"#window = row,col,source,sourceId,position\n");
		for(int i=0;i<numWidgets;i++) {
			//window = row,col,source,sourceId,position
			if(widgets[i]->source) {
				fprintf(fp,"window = %d,%d,1,%d,%d\n", widgets[i]->row,
					widgets[i]->col, widgets[i]->sourceId, widgets[i]->position);
			}
			else {
				fprintf(fp,"window = %d,%d,0,%d,%d\n", widgets[i]->row,
					widgets[i]->col, widgets[i]->sourceId, widgets[i]->position);
			}
		}
		/*seqWidgetList = improv->getSeqWidgetList();
		for(it=seqWidgetList.begin();it!=seqWidgetList.end();it++) {
			//fn = row,col,posn,uniqueid,name
			pluginItems = (*it)->getPluginItems();
			for(int i=0;i<(*it)->numPluginItems();i++) {
				widgetConfig *w = getWidgetConfig(false,(*it)->getId());
				fprintf(fp, "fn = %d,%d,%d,%d,%s\n", w->row, w->col, i,
				improv->getPluginUniqueId(pluginItems[i]->getMenuId()),
				improv->getPluginName(pluginItems[i]->getMenuId())->latin1() );
			}
		}*/
	}
	fclose(fp);
	this->modified = false;
	return true;
}

void improvConfig::clear(void) {
	// load up the vars
	this->sequencePath = NULL;
	this->pluginPath = NULL;
	this->windowWidth = -1;
	this->windowHeight = -1;
	this->numSources = -1;
	this->numWindows = -1;
	this->useSequence = NULL;
	this->cameraType = OTHER;
	this->modified = false;
	this->numPlugins = 0;
	this->numWidgets = 0;
	this->numCols = 0;
	this->numRows = 0;
}

void improvConfig::clearLayout(void) {
	this->windowWidth = -1;
	this->windowHeight = -1;
	this->numSources = -1;
	this->numWindows = -1;
	this->useSequence = NULL;
	this->numPlugins = 0;
	this->numWidgets = 0;
	this->numCols = 0;
	this->numRows = 0;
}
	
void improvConfig::setSequencePath(char *sequencePath) {
	assert(sequencePath);
	this->sequencePath = strdup(sequencePath);
	this->modified = true;
}

char *improvConfig::getSequencePath(void) {
	return this->sequencePath;
}
	
void improvConfig::setPluginPath(char *pluginPath) {
	assert(pluginPath);
	this->pluginPath = strdup(pluginPath);
	this->modified = true;
}

char *improvConfig::getPluginPath(void) {
	return this->pluginPath;
}
	
void improvConfig::setWindowWidth(int width) {
	assert(width > 0);
	this->windowWidth = width;
	this->modified = true;
}

int improvConfig::getWindowWidth(void) {
	return this->windowWidth;
}

void improvConfig::setWindowHeight(int height) {
	assert(height > 0);
	this->windowHeight = height;
	this->modified = true;
}

int improvConfig::getWindowHeight(void) {
	return this->windowHeight;
}

void improvConfig::setNumSources(int number) {
	assert(number >= 1);
	this->numSources = number;
}

int improvConfig::getNumSources(void) {
	return this->numSources;
}

void improvConfig::setNumWindows(int number) {
	assert(number > 0);
	this->numWindows = number;
	this->modified = true;
}                               

int improvConfig::getNumWindows(void) {
	return this->numWindows;
}

void improvConfig::setUseSequence(char *name) {
	if(name!=NULL) {
		this->useSequence = strdup(name);
	}
	else {
		this->useSequence = NULL;
	}
	this->modified = true;
}

char *improvConfig::getUseSequence(void) {
	return this->useSequence;
}

void improvConfig::setCameraType(imageSeqType camera) {
	this->cameraType = camera;
	this->modified = true;
}

imageSeqType improvConfig::getCameraType(void) {
	return this->cameraType;
}

bool improvConfig::isModified(void) {
	return this->modified;
}

void improvConfig::setModified(bool on) {
	this->modified = on;
}

improvConfig::widgetConfig* improvConfig::getWidgetConfig(int posn) {
	assert(posn<numWidgets);
	return widgets[posn];
}

int improvConfig::numWidgetConfigs(void) {
	return numWidgets;
}

improvConfig::widgetConfig* improvConfig::getWidgetConfig(bool source, int id) {
	for(int i=0;i<numWidgets;i++) {
		if((widgets[i]->source==source)&&(widgets[i]->id==id))
			return widgets[i];
	}
	return (improvConfig::widgetConfig*)NULL;
}

int improvConfig::getNumRows(void) {
	return numRows;
}

int improvConfig::getNumCols(void) {
	return numCols;
}

bool improvConfig::verifyWidgetConfig() {
	return verifyWidgetConfig(this->widgets,this->numWidgets,this->numWindows,this->numSources);
}

bool improvConfig::verifyWidgetConfig(widgetConfig **_widgets, int _numWidgets,
		int _numWindows, int _numSources)
{
	int sources=0;
	bool *position;
	bool end=false;
	if(_numWidgets!=_numWindows) return false;
	position = (bool *)malloc(sizeof(bool)*_numWidgets);
	for(int i=0;i<_numWidgets;i++) {
		position[i] = false;
	}
	// check number of source widgets
	for(int i=0;i<_numWidgets;i++) {
		if(_widgets[i]->source) {
			if(_widgets[i]->position==0) sources++;
			else return false;
		}
	}
	if(sources!=_numSources) return false;
	for(int i=0;i<_numSources;i++) {
		// get all the filled positions for this source
		for(int j=0;j<_numWidgets;j++) {
			if((_widgets[j]->sourceId==i)||(_widgets[j]->sourceId==-1)) position[_widgets[j]->position] = true;
			else if((_widgets[j]->sourceId>=_numSources)||(_widgets[j]->sourceId<-1)) return false;
		}
		// check that the are no discontinuities
		for(int j=0;j<_numWidgets;j++) {
			if(!position[j]) {end = true;}
			else if(end) return false;
		}
		end = false;
	}
	return true;
}

char *improvConfig::getDefaultFilename(void) {
	return defaultFilename;
}

improvConfig::pluginInstance *improvConfig::getPluginInstance(int posn) {
	assert(posn<numPlugins);
	return plugins[posn];
}

int improvConfig::numPluginInstances(void) {
	return numPlugins;
}

