#include <dlfcn.h>
#include <assert.h>
#include <string.h>
#include "ipPlugin.h"

#ifndef DEBUG
#define DEBUG 0
#endif

ipPlugin::ipPlugin(const char* plugin_file) {
	this->plugin_file = plugin_file;
	this->ready = false;
	this->phandle = NULL;

	if( init_plugin() )
		ready = true;
}

ipPlugin::~ipPlugin() 
{
	if(this->phandle)
		dlclose(this->phandle);
}

int ipPlugin::init_plugin(void)
{
	IP_Handle* (*init)(void);
	char *error;

	if( (this->phandle = dlopen( plugin_file, RTLD_LAZY)) == NULL ) {
		if(DEBUG) printf("ipPlugin: Error opening dll: >%s<\n", plugin_file);
		if( (error = dlerror()) == NULL) 
			if(DEBUG) puts("No error reported.");
		else
			fputs(error, stderr);
		return 0;
	}
	
	/*  Initialise plugin  */
	init = (IP_Handle* (*)(void)) dlsym(this->phandle, "IP_Init");
	if(!init) {
		if(DEBUG) puts("Unable to load descriptor symbol: IP_Init");
		return 0;
	}

	this->iphandle = (*init)();

	/*  Get descriptor  */
	this->desc = get_plugin_desc();

	return 1;
}

void ipPlugin::processId(int id, Picture **p_in, Picture *p_out, float *params, void *result)
{
	IP_Op* op;

	/*  Check requested id is valid */
	if( (unsigned int) id > desc->OpCount-1 || id < 0 ) {
		if(DEBUG) printf("Warning: specified operation index out of range (%d)\n", id);
		if(DEBUG) puts("Reverting to default operation 0.");
		op = desc->Ops;
	}
	else
		op = desc->Ops + id;

	//  The actual processing
	(*op->process)(this->iphandle, p_in, p_out, params,result);
}

void ipPlugin::printInfo(void)
{
	assert(this->desc);

	if(this->desc) {
		printf("Name      : %s\n", desc->Name);
		printf("Maker     : %s\n", desc->Maker);
		printf("Copyright : %s\n", desc->Copyright);
		printf("Comment   : %s\n", desc->Comment);
		printf("Number of operations available in plugin: %ld\n", desc->OpCount);
	}
}

void ipPlugin::listOps(void)
{
	IP_Op *op;
	int i,j;

	for(i=0; (unsigned int) i < desc->OpCount; i++) {
		op = desc->Ops + i;  /*  XXX pointer arithmetic, yuck  */
		printf("\n[Index %2d] %s -> %s", op->Index-1, op->Category,
				op->Operation);

		if(op->ParamCount > 0) {
			//				printf("ParamCount: %d\n", op->ParamCount);
			for(j=0; (unsigned int) j<op->ParamCount; j++)
				printf("\n           |Parameter %2d| %s", j+1, op->ParamNames[j]);
		}
	}
	putchar('\n');
}

IP_Descriptor* ipPlugin::get_plugin_desc(void)
{
	IP_Descriptor_Function get_desc;

	get_desc = (IP_Descriptor_Function) dlsym(phandle, "IP_Descriptor_get");
	if(!get_desc) {
		puts("Unable to locate the descriptor symbol!");
		return NULL;
	}

	desc = (IP_Descriptor*) (*get_desc)(0);
	return desc;
}
		
bool ipPlugin::isReady(void)
{
	return this->ready;
}
		
unsigned int ipPlugin::getNumOps(void)
{
	return desc->OpCount;
}

bool ipPlugin::getOpInfo(int id, char** category, char** name, unsigned int* num_inputs, unsigned int* num_params, char*** param_names, pluginResultType *resultType, size_t *resultSize)
{
	IP_Op *op;

	if( id<0 || id >= (int) desc->OpCount )
		return false;

	op = desc->Ops + id;  /*  XXX pointer arithmetic, yuck  */

	*category = strdup(op->Category);
	*name     = strdup(op->Operation);
	*num_inputs = op->InputCount;
	*num_params = op->ParamCount;
	*param_names = op->ParamNames;
	*resultType = op->resultType;
	*resultSize = op->resultSize;

	return true;
}

int ipPlugin::getUniqueId(void) {
	return desc->UniqueId;
}

const char *ipPlugin::getFileName(void) {
	return this->plugin_file;
}
