/**
 * Improv.h
 *
 * Singleton class, tracks global state  
 *
 */

#ifndef IMPROV_H
#define IMPROV_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dlfcn.h>
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <dirent.h>
#include <sys/types.h>
#include <getopt.h>
#include <math.h>

// QT includes
#include <qpopupmenu.h>
#include <qstatusbar.h>
#include <qaction.h>
#include <qvaluelist.h>

// Application includes, dependent upon QT
#include "AnimThread.h"
#include "ParamWidget.h"
#include "PluginMenu.h"
#include "SrcWidget.h"
#include "SeqWidget.h"
#include "ImprovQTCamOpt.h"
//#include "StatusWidget.h"

// Application includes
#include "FW/labImage.h"
#include "improv_plugin.h"
#include "improvConfig.h"
#include "imageSeq.h"
#include "ipPlugin.h"
#include "utils.h"

class Improv {
	friend class PluginMenu;
	friend class SeqWidget;
	friend class SrcWidget;
	//friend class improvConfig;

	public:
		static Improv* Instance(int argc, char **argv);

		~Improv();
	
		/** Check if the sequence is currently playing
			* @return true if the sequence is playing, false otherwise
			*/
		bool isPlaying(void);

		bool isReady(void);

		/** Start/stop the sequence from playing.
			* @param opt true to begin playback, false to stop it
			*/
		void setPlaying(bool opt);

		/** Set the source widget of the application.
			* The source widget provides the animation sequence which is processed
			* by all the sequence widgets
			*/
		void addSrcWidget(SrcWidget* sw);

		/** Return a pointer to the requested source widget.
			* @return pointer to the requested source widget.
			* @param id of the source widget
			* @see addSrcWidget
			*/
		SrcWidget* getSrcWidget(int sourceId);
		int getWidgetCount(int sourceId, int position);
		QValueList<SrcWidget*> getSrcWidgetList(void);

		/** Add a sequence widget to the list
			* This class keeps track of all the sequence widgets in the application
			* through this function.  Once a sequence is added it becomes available
			* to display the result of image processing operations.
			* @param sw pointer to SeqWidget to add
			* @see SeqWidget
			*/
		void addSeqWidget(SeqWidget* sw);

		/** Remove a sequence widget from the list
			* This class keeps track of all the sequence widgets in the application
			* through this function.
			* @param sw pointer to SeqWidget to remove
			* @see SeqWidget
			*/
		void removeSeqWidget(SeqWidget* sw);
		
		SeqWidget* getSeqWidget(int id);
		SeqWidget* getSeqWidget(int sourceId, int position, int number=0);

		QValueList<SeqWidget*> getSeqWidgetList(void);
		
		/** Return a pointer to the plugin menu.
			* Only one copy of the plugin menu is instantiated.  It is tracked by
		  * this class, and can be referenced through this function
			* @return pointer to the popup menu
			*/
		QPopupMenu* getPluginMenu(void);	

		/** Set the frame rate of playback.
			* Sets the desired rate of playback.  If possible, the source and sequence
			* widgets will update synchronously to this rate unless the CPU is overloaded,
			* in which case it will update as fast as possible.
			* @param fps desired frame rate
			*/
		void setFPS(int fps);

		/** Toggle unlimited frames per second.
			* @param toggle
			*/
		void setUnlimitedFPS(bool toggle);

		/** Copies a picture to the internal buffers of all the app's sequence widgets
			* When the source image is changed, this clears all the sequence widget internal
			* buffers which are displayed and replaces them with one of matching format and
			* attributes of the source picture.
			* @param pic pointer to source picture
			*/
		void copyPictures(Picture *pic);

		/** Iterates through one cycle of animation/processing/display
			* @param numFrames number of frames to jump (default is 1)
			*/
		void tick(int frames=1);
		
		/** Perform image processing operation on a picture, given a selected menu Id
			* @param p_in input pictures
			* @param p_out output picture
			* @param menuId menu Id selected
			* @param params array of floating point parameters
			*/
		void process(Picture** p_in, Picture* p_out, int menuId, int p_inCount=1, int bufferId=-1, float* params = NULL);

		int getPluginUniqueId(int id);

		/** Return the name of the plugin given a menu id
		 *  @param id menu id of plugin
		 *  @return string of plugin name
		 */
		QString* getPluginName(int id);

		int getPluginInputCount(int id);

		/** Return the number of input input paramters of the plugin operation 
		 *  given a menu id
		 *  @param id menu id of plugin
		 *  @return string of plugin name
		 */
		int getPluginNumParams(int id);

		QStringList* getPluginParamNames(int id);

		void activateParamWidget(int numParams, QStringList* paramNames, float* params);
		void setParamWidget(ParamWidget* pw);
		void disableParamWidget(void);

		void setStatusBar(QStatusBar* statusBar);
		bool statusBarReady(void);
		void setStatusMessage(const char* msg);
		void setStatusMessage(const char* msg, int ms);
		void setStatusFPSMessage(const char* msg);
		void setStatusResultMessage(const char* msg);
		const char *getStatusMessage(void);

		void constructPluginMenu(void);

		int getNumBuffers(void);
		void addFrameBuffer(int numFrames);
		void removeFrameBuffer(int bufferId, int menuId);

		void setCamOptAction(QAction *action);
		void setCamOpt(ImprovQTCamOpt *options);
		ImprovQTCamOpt *getCamOpt(void);
		void setCamOptOpen(bool open);
		bool camOptIsOpen(void);
		void updateCamOpt(imageSeqType type, Camera *camera);

		improvConfig* config;

	protected:
		Improv(int argc, char **argv);
		
		void setMenuId(int id);
		int  getMenuId(void);

	private:
		static Improv* _instance;

		AnimThread* animThread;
		bool        playing;

		QValueList<Plugin*> pluginList;
		PluginMenu* pluginMenu;

		// Source + Sequence widget lists.
		// lists kept in order of widget id.
		QValueList<SrcWidget*> srcWidgetList;
		QValueList<SeqWidget*> seqWidgetList;

		ParamWidget* paramWidget;

		QStatusBar* statusBar;
		//StatusWidget *statusBar;
		const char *statusBarMsg;
		const char *fpsMsg;
		const char *resultMsg;

		// keeps track of how many plugin instances there are
		//int numberPluginItems;

		// frame buffer for plugin instances
		// TODO: change this to use QValueVector to increase speed
		QValueList<Picture**> frameBuffer;

		//  Current state
		int seqId;
		int menuId;

		ImprovQTCamOpt *camOptions;
		QAction *camOptionAction;
		bool camOptOpen;

		static int seq_file_select(const struct dirent *entry);
		static int plugin_file_select(const struct dirent *entry);
};
#endif
