/*
 *   Image Processing Library
 *   Thomas Braunl, Univ. Stuttgart, 1996
 *   Additional Code by Gerrit Heitsch
 *   Adapted by Michael Rudolph
 *   derived after Braunl et al. : Parallel Image Processing, Springer 2001
 *
 *   Adapted for ImprovQT by Daniel Venkitachalam & Leon Koch (2002)
 *   32bit Colour + Additional code by Leon Koch (2002)
 *
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <assert.h>

#include "../FW/labImage.h"
#include "../improv_plugin.h"
#include "IPL_plugin.h"

#define NUMBER_OF_OPERATIONS 32

IP_Descriptor ipl_plugin =
{
	1,
	"Image Processing Library",
	"Thomas Braunl, Univ. Stuttgart",
	"1996",
	"Additional code by Gerrit Heitsch etc.\nAdapted for ImprovQT by Daniel Venkitachalam & Leon Koch\n",
	NUMBER_OF_OPERATIONS,
	NULL              /*  IP_ops, filled out by the _init function  */
};

IP_Handle IP_Init()
{
	IP_Op *op;
	int i = 0;

	/*  Fill out the IP ops descriptions  */
	ipl_plugin.Ops = calloc(NUMBER_OF_OPERATIONS, sizeof(IP_Op));

	/*  Edges  */

	/*  Laplace 3x3 function  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Edges");
	op->Operation  = strdup("Laplace\t(3x3)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_laplace;

	/*  Sobel 3x3 function  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Edges");
	op->Operation  = strdup("Sobel\t(3x3)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_sobel;

	/*  Corner 3x3 function  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Edges");
	op->Operation  = strdup("Corner\t(3x3)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_corner;

	/*  General  */

	/*  Min  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("General");
	op->Operation  = strdup("Min\t(3x3)");
	op->Index      = i;
	op->ParamCount = 0;
	op->InputCount = 1;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_min;

	/*  Max  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("General");
	op->Operation  = strdup("Max\t(3x3)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_max;

	/*  Mean function  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("General");
	op->Operation  = strdup("Mean\t(3x3)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_mean;

	/*  Median  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("General");
	op->Operation  = strdup("Median\t(3x3)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_median;

	/*  Identity  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("General");
	op->Operation  = strdup("Identity");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_identity;

	/*  Negation  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("General");
	op->Operation  = strdup("Negation");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_negation;

	/*  Noise function  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("General");
	op->Operation  = strdup("Noise\t(t)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 1;
	op->ParamNames = calloc(1, sizeof(char *));
	op->ParamNames[0] = strdup("Noise");
	op->resultType = NORESULT;
	op->process    = IPL_noise;

	/*  Show(x,y) function  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("General");
	op->Operation  = strdup("Show\t(x,y)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 2;
	op->ParamNames = calloc(2, sizeof(char *));
	op->ParamNames[0] = strdup("X");
	op->ParamNames[1] = strdup("Y");
	op->resultType = TEXT;
	op->resultSize = sizeof(char)*28;
	op->process    = IPL_show;

	/*  Format  */

	/*  Threshold function  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Format");
	op->Operation  = strdup("Threshold\t(t)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 1;
	op->ParamNames = calloc(1, sizeof(char *));
	op->ParamNames[0] = strdup("Threshold");
	op->resultType = NORESULT;
	op->process    = IPL_threshold;

	/*  Gray stretch  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Format");
	op->Operation  = strdup("Grey stretch");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_gray_stretch;

	/*  Colour to grey  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Format");
	op->Operation  = strdup("Colour to Grey");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_col2gray;

	/*  Dither  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Format");
	op->Operation  = strdup("Grey Dither");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_dither;

	/*  RGB to HSV  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Format");
	op->Operation  = strdup("RGB to HSV");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_HSVValue;

	/*  grey reduce  */	
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Format");
	op->Operation  = strdup("Grey Reduce\t(t)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 1;
	op->ParamNames = calloc(1, sizeof(char *));
	op->ParamNames[0] = strdup("Values");
	op->resultType = NORESULT;
	op->process    = IPL_gray_reduce;

	/*  Advanced  */
	
	/*  Region growing  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Advanced");
	op->Operation  = strdup("Region segmentation\t(t)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 1;
	op->ParamNames = calloc(1, sizeof(char *));
	op->ParamNames[0] = strdup("Threshold");
	op->resultType = NORESULT;
	op->process    = IPL_region_growing;

	/*  Match color range(r,g,b)  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Advanced");
	op->Operation  = strdup("Match color range\t(r,g,b)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 3;	
	op->ParamNames = calloc(3, sizeof(char *));
	op->ParamNames[0] = strdup("R");
	op->ParamNames[1] = strdup("G");
	op->ParamNames[2] = strdup("B");
	op->resultType = NORESULT;
	op->process    = IPL_match_color_range;

	/*  Binary Morphology  */

	/*  Erosion  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Binary Morphology");
	op->Operation  = strdup("Erosion");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_bin_erosion;

	/*  Dilation  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Binary Morphology");
	op->Operation  = strdup("Dilation");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_bin_dilation;

	/*  Open  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Binary Morphology");
	op->Operation  = strdup("Open");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_bin_open;

	/*  Close  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Binary Morphology");
	op->Operation  = strdup("Close");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_bin_close;

	/*  Fill(x,y)  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Binary Morphology");
	op->Operation  = strdup("Fill\t(x,y)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 2;
	op->ParamNames = calloc(2, sizeof(char *));
	op->ParamNames[0] = strdup("X");
	op->ParamNames[1] = strdup("Y");
	op->resultType = TEXT;
	op->resultSize = sizeof(char)*18;
	op->process    = IPL_bin_fill;

	/*  Connected(x,y)  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Binary Morphology");
	op->Operation  = strdup("Connected\t(x,y)");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 2;
	op->ParamNames = calloc(2, sizeof(char *));
	op->ParamNames[0] = strdup("X");
	op->ParamNames[1] = strdup("Y");
	op->resultType = TEXT;
	op->resultSize = sizeof(char)*23;
	op->process    = IPL_bin_connected;

	/*  Boundary  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Binary Morphology");
	op->Operation  = strdup("Boundary");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_bin_boundary;

	/*  Skeleton  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Binary Morphology");
	op->Operation  = strdup("Skeleton");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = TEXT;
	op->resultSize = sizeof(char)*22;
	op->process    = IPL_bin_skeleton;

	/*  Combination  */

	/*  Difference  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Combination");
	op->Operation  = strdup("Difference");
	op->Index      = i;
	op->InputCount = 2;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_difference;

	/*  AND  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Combination");
	op->Operation  = strdup("AND");
	op->Index      = i;
	op->InputCount = 2;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_and;

	/*  OR  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Combination");
	op->Operation  = strdup("OR");
	op->Index      = i;
	op->InputCount = 2;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_or;

	/*  Greyscale Morphology  */

	/*  Erosion  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Greyscale Morphology");
	op->Operation  = strdup("Erosion");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_erosion;

	/*  Dilation  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Greyscale Morphology");
	op->Operation  = strdup("Dilation");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_dilation;

	if( i != NUMBER_OF_OPERATIONS ) {
		printf("IPL_plugin.c ERROR: Mismatched number of operations:\n" \
				" (i, NUMBER_OF_OPERATIONS) = (%d, %d)\n", i, NUMBER_OF_OPERATIONS);
		exit(1);
	}

	return NULL;
}


const IP_Descriptor* IP_Descriptor_get(unsigned long Index)
{
	return &ipl_plugin;
}

void col_invert(BYTE *rgb)
{ // invert 3 consecutive RGB bytes
   *rgb     = 255 - *rgb;
   *(rgb+1) = (255 - *(rgb+1));
   *(rgb+2) = (255 - *(rgb+2));
}

pluginReturnType IPL_sobel(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int i=0;
	int grad;
	int deltaX, deltaY;

	BYTE *tmp, *imageIn;
	Picture *p_imageIn = *p_imageInputs;

	imageIn=(BYTE *)p_imageIn->data;
	tmp=(BYTE *)p_imageOut->data;

	memset(tmp, 0, p_imageIn->width);

	if(p_imageIn->format==pix_grey) {
		for (i = p_imageIn->width+1; 
				i < (p_imageIn->height-2)*p_imageIn->width; i++) {
			deltaX = 2*imageIn[i+1] + imageIn[i-p_imageIn->width+1] + 
				imageIn[i+p_imageIn->width+1] -
				2*imageIn[i-1] - imageIn[i-p_imageIn->width-1] - 
				imageIn[i+p_imageIn->width-1];

			deltaY = imageIn[i-p_imageIn->width-1] + 2*imageIn[i-p_imageIn->width] + 
				imageIn[i-p_imageIn->width+1] -
				imageIn[i+p_imageIn->width-1] - 2*imageIn[i+p_imageIn->width] - 
				imageIn[i+p_imageIn->width+1];

			grad = (abs(deltaX) + abs(deltaY)) / 3;

			if (grad > WHITE)
				grad = WHITE;

			tmp[i] = (BYTE)grad;
		}
	}
	else {
		//printf("Must be greyscale image !\n");
		return WRONGFORMAT;
	}

	memset(tmp + i, 0, p_imageIn->width);
	return NOERROR;
}

pluginReturnType IPL_mean(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int i,width,height,tmp;
	Picture *p_imageIn = *p_imageInputs;

	BYTE *imageOut,*imageIn;

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	switch(p_imageIn->format) {
		case pix_grey:
			memset(imageOut, 0, width);
			for (i =width+1; i<(height-1)*width; i++)
			{
				tmp = (int) (imageIn[i-width-1] +
						imageIn[i-width]   +
						imageIn[i-width+1] +
						imageIn[i-1]       +
						imageIn[i]         +
						imageIn[i+1]       +
						imageIn[i+width-1] +
						imageIn[i+width]   +
						imageIn[i+width+1]      )  / 9;
				imageOut[i] = (BYTE) tmp;
			}
			memset(imageOut + i, 0, width);
		  break;
		case pix_rgb24:
		case pix_bgr24:
			memset(imageOut, 0, 3*width);
			for (i = 3*width+3; i < 3*(height-2)*width; i+=3)
			{
				imageOut[i] = (BYTE)(imageIn[i-3*(width-1)] +
						imageIn[i-3*(width)]   +
						imageIn[i-3*(width+1)] +
						imageIn[i-3]       +
						imageIn[i]         +
						imageIn[i+3]       +
						imageIn[i+3*(width-1)] +
						imageIn[i+3*(width)]   +
						imageIn[i+3*(width+1)]   )  / 9;
				imageOut[i+1] = (BYTE)(imageIn[i-3*(width-1)+1] +
						imageIn[i-3*(width)+1]   +
						imageIn[i-3*(width+1)+1] +
						imageIn[i-3+1]       +
						imageIn[i+1]         +
						imageIn[i+3+1]       +
						imageIn[i+3*(width-1)+1] +
						imageIn[i+3*(width)+1]   +
						imageIn[i+3*(width+1)+1]   )  / 9;
				imageOut[i+2] = (BYTE)(imageIn[i-3*(width-1)+2] +
						imageIn[i-3*(width)+2]   +
						imageIn[i-3*(width+1)+2] +
						imageIn[i-3+2]       +
						imageIn[i+2]         +
						imageIn[i+3+2]       +
						imageIn[i+3*(width-1)+2] +
						imageIn[i+3*(width)+2]   +
						imageIn[i+3*(width+1)+2]   )  / 9;
			}
			memset(imageOut + i, 0, 3*width);
			break;
		case pix_rgb32:
		case pix_bgr32:
			memset(imageOut, 0, 4*width);
			for (i = 4*width+4; i < 4*(height-2)*width; i+=4)
			{
				imageOut[i] = (BYTE)(imageIn[i-4*(width-1)] +
						imageIn[i-4*(width)]   +
						imageIn[i-4*(width+1)] +
						imageIn[i-4]       +
						imageIn[i]         +
						imageIn[i+4]       +
						imageIn[i+4*(width-1)] +
						imageIn[i+4*(width)]   +
						imageIn[i+4*(width+1)]   )  / 9;
				imageOut[i+1] = (BYTE)(imageIn[i-4*(width-1)+1] +
						imageIn[i-4*(width)+1]   +
						imageIn[i-4*(width+1)+1] +
						imageIn[i-4+1]       +
						imageIn[i+1]         +
						imageIn[i+4+1]       +
						imageIn[i+4*(width-1)+1] +
						imageIn[i+4*(width)+1]   +
						imageIn[i+4*(width+1)+1]   )  / 9;
				imageOut[i+2] = (BYTE)(imageIn[i-4*(width-1)+2] +
						imageIn[i-4*(width)+2]   +
						imageIn[i-4*(width+1)+2] +
						imageIn[i-4+2]       +
						imageIn[i+2]         +
						imageIn[i+4+2]       +
						imageIn[i+4*(width-1)+2] +
						imageIn[i+4*(width)+2]   +
						imageIn[i+4*(width+1)+2]   )  / 9;
			}
			memset(imageOut + i, 0, 4*width);
			break;
		default:
			if(NOISY) fprintf(stderr,"IPL_mean (Mean 3x3): Not implemented for this type!\n");
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_median(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int i;
	int width, height;
	BYTE values[9];

	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *p_imageInputs;

	if(p_imageIn->format==pix_grey)
	{  
		imageIn=(BYTE *)p_imageIn->data;
		imageOut=(BYTE *)p_imageOut->data;

		width=p_imageIn->width;
		height=p_imageIn->height;

		memset(imageOut, 0, width);

		// loop over the image, starting at the second row, second column
		for (i = width+1; i < width * (height - 1); i++)
		{
			int x, y;
			int k = 0, l;
			int j;

			/*
			*    sort the 3x3 environment by simple
			*    insertion sort.
			*/

			for (y = -1; y <= 1; y++)
			{
				for (x = -1; x <= 1; x++)
				{
					BYTE val = imageIn[i + y * width + x];

					for (j = 0; j < k; j++)
						if (val < values[j])
							break;

					for (l = k; l > j; l--)
						values[l] = values[l-1];
					values[j] = val;
					k++;
				}

				imageOut[i] = values[4];
			}
		}

		memset(imageOut + i, 0, width);
	}
	else {
		//printf("Must be grayscale image !\n");
		return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_min(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int i;
	int val;
	int x, y;
	int width, height;
	static BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *p_imageInputs;

	if(p_imageIn->format==pix_grey)
	{
		imageIn=(BYTE *)p_imageIn->data;
		imageOut=(BYTE *)p_imageOut->data;

		width=p_imageIn->width;
		height=p_imageIn->height;

		memset(imageOut, 0, width);

		for (i = width+1; i < (height-1)*width; i++)
		{
			val = WHITE;

			for (y = -1; y <= 1; y++)
				for (x = -1; x <= 1; x++)
					if (imageIn[i + x + y * width] < val)
						val = imageIn[i + x + y * width];

			imageOut[i] = (BYTE)val;
		}

		memset(imageOut + i, 0, width); 
	}
	else {
		//printf("Must be grayscale image !\n");
		return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_max(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int i;
	int val;
	int x, y;
	int width, height;
	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *p_imageInputs;

	if(p_imageIn->format==pix_grey)
	{
		imageIn=(BYTE *)p_imageIn->data;
		imageOut=(BYTE *)p_imageOut->data;

		width=p_imageIn->width;
		height=p_imageIn->height;

		memset(imageOut, 0, width);

		for (i = width+1; i < (height-1)*width; i++)
		{
			val = BLACK;

			for (y = -1; y <= 1; y++)
				for (x = -1; x <= 1; x++)
					if (imageIn[i + x + y * width] > val)
						val = imageIn[i + x + y * width];

			imageOut[i] = (BYTE)val;
		}

		memset(imageOut + i, 0, width);
	}
	else {
		//printf("Must be grayscale image !\n");
		return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_identity(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int width, height;
	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *p_imageInputs;

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	if(p_imageIn->datasize==p_imageOut->datasize) {
		memcpy(imageOut, imageIn, p_imageOut->datasize);
	}
	else return WRONGFORMAT;
	return NOERROR;
}

pluginReturnType IPL_negation(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int i;
	int width, height;
	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *p_imageInputs;

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	for (i = 0; i < p_imageIn->bytes_per_pixel * width * height; i++)
		imageOut[i] = WHITE - imageIn[i];
	return NOERROR;
}

pluginReturnType IPL_gray_stretch(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int i;
	int width, height;

	BYTE maxVal = BLACK;
	BYTE minVal = WHITE;
	BYTE range;
	double factor=1.0;

	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *p_imageInputs;

	if(p_imageIn->format==pix_grey)
	{
		imageIn=(BYTE *)p_imageIn->data;
		imageOut=(BYTE *)p_imageOut->data;

		width=p_imageIn->width;
		height=p_imageIn->height;


		for (i = 0; i < width*height; i++)
		{
			if (imageIn[i] < minVal)
				minVal = imageIn[i];

			if (imageIn[i] > maxVal)
				maxVal = imageIn[i];   
		}

		range = maxVal - minVal;
		if (range == 0)
			range = 1;
		factor = (double)WHITE / range;

		for (i = 0; i < width*height; i++)
			imageOut[i] = (BYTE)(factor * (imageIn[i] - minVal));
	}
	else {
		//printf("Must be grayscale image!\n");
		return WRONGFORMAT;
	}
	return NOERROR;
}

// 2x2 ordered dithering, uses only upper left pixel in 2x2 areas     
pluginReturnType IPL_dither(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int width, height;
	int x, y, i;
	int thres = WHITE / 5;

	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *p_imageInputs;

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	if(p_imageIn->format==pix_grey)
	{   
		memcpy(imageOut, imageIn, width * height);

		for (y = 0; y < height-1; y+=2)
		{
			for (x = 0; x < width-1; x+=2)
			{
				i = x + width * y;

				if (imageIn[i] <   thres)
					imageOut[i] = BLACK;
				else
					imageOut[i] = WHITE;

				if (imageIn[i] < 3*thres)
					imageOut[i+1] = BLACK;
				else
					imageOut[i] = WHITE;

				if (imageIn[i] < 4*thres)
					imageOut[i+width] = BLACK;
				else
					imageOut[i] = WHITE;

				if (imageIn[i] < 2*thres)
					imageOut[i+width+1] = BLACK;
				else
					imageOut[i] = WHITE;
			}
		} 
	}
	else {
		//printf("Must be greyscale image!\n");
		return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_noise(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int i,pos;
	int imageSize ;
	int numberOfPixels ;
	int width, height;

	double noise;

	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *p_imageInputs;

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;

	noise = (double) params[0];

	width=p_imageIn->width;
	height=p_imageIn->height;

	imageSize = width * height; 
	numberOfPixels = (int)(imageSize * noise);

	switch(p_imageIn->format) {
		case pix_grey:
			memcpy(imageOut, imageIn, imageSize);
			for (i = 0; i < numberOfPixels; i+=2) {
				imageOut[rand() % imageSize] = WHITE;
				imageOut[rand() % imageSize] = BLACK;
			}
			break;
		case pix_rgb24:
		case pix_bgr24:
			memcpy(imageOut, imageIn, 3*imageSize);
			for (i = 0; i < numberOfPixels; i+=2) {
				pos = rand() % imageSize;
				imageOut[3*pos  ] = WHITE;
				imageOut[3*pos+1] = WHITE;
				imageOut[3*pos+2] = WHITE;
				pos = rand() % imageSize;
				imageOut[3*pos  ] = BLACK;
				imageOut[3*pos+1] = BLACK;
				imageOut[3*pos+2] = BLACK;
			}
			break;
		case pix_rgb32:
		case pix_bgr32:
			memcpy(imageOut, imageIn, 4*imageSize);
			for (i = 0; i < numberOfPixels; i+=2) {
				pos = rand() % imageSize;
				imageOut[4*pos  ] = WHITE;
				imageOut[4*pos+1] = WHITE;
				imageOut[4*pos+2] = WHITE;
				pos = rand() % imageSize;
				imageOut[4*pos  ] = BLACK;
				imageOut[4*pos+1] = BLACK;
				imageOut[4*pos+2] = BLACK;
			}
			break;
		default:
			if(NOISY) fprintf(stderr,"IPL_Noise: Not implemented for this type!\n");
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_region_growing(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int seed = 40;
	int change;
	int x, y, offset = 0, off2 = 1;
	int loopCount = 0;
	int width, height;
	BYTE reg[MAX_IMAGE_SIZE];

	BYTE *imageOut,*imageIn;
	int thresh;
	Picture *p_imageIn = *p_imageInputs;

	// thresh = 10..100
	thresh = (int) (90*params[0] + 10);

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	if(p_imageIn->format==pix_grey)
	{  
		memset(reg, 0, width * height);

		for (y = 0; y < height; y++)
			for (x = 0; x < width; x++, offset++)
				if ((x % seed == 1) && (y % seed == 1))
					reg[offset] = off2++;
		do
		{
			change = 0;

			for (y = 1; y < height-1; y++)
			{
				for (x = 1; x < width-1; x++)
				{
					offset = x + width * y;

					if (reg[offset] != 0)
					{
						if (abs(imageIn[offset] - imageIn[offset - 1]) <= thresh &&
								reg[offset - 1] < reg[offset])
							change = reg[offset - 1] = reg[offset];

						if (abs(imageIn[offset] - imageIn[offset + 1]) <= thresh &&
								reg[offset + 1] < reg[offset])
							change = reg[offset + 1] = reg[offset];

						if (abs(imageIn[offset] - imageIn[offset - width]) <= thresh &&
								reg[offset - width] < reg[offset])
							change = reg[offset - width] = reg[offset];

						if (abs(imageIn[offset] - imageIn[offset + width]) <= thresh &&
								reg[offset + width] < reg[offset])
							change = reg[offset + width] = reg[offset];
					}
				}
			}

			loopCount++;
		}
		while (change);

		memcpy(imageOut, reg, width * height);

		printf("Region Growing(s=%d,t=%d) l=%d\n", seed, thresh, loopCount);
	}
	else {
		//printf("Must be greyscale image !\n");
		return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_laplace(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int i;
	int delta;
	int width, height;

	BYTE *imageOut,*imageIn;
	Picture *p_imageIn = *p_imageInputs;

	if(p_imageIn->format==pix_grey)
	{    
		imageIn=(BYTE *)p_imageIn->data;
		imageOut=(BYTE *)p_imageOut->data;

		width=p_imageIn->width;
		height=p_imageIn->height;

		for (i = width; i < (height-1)*width; i++)
		{
			delta  = abs(4 * imageIn[i] -
					imageIn[i-1] -
					imageIn[i+1] -
					imageIn[i-width] -
					imageIn[i+width]);

			if (delta > WHITE)
				imageOut[i] = WHITE;
			else
				imageOut[i] = (BYTE)delta;
		}
	}
	else {
		//printf("Must be grayscale image !\n");
		return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_corner(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int width, height;
	int i;
	int b;
	BYTE *imageOut,*imageIn;

	Picture *p_imageIn = *p_imageInputs;
	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	if(p_imageIn->format==pix_grey)
	{  
		for (i = width; i < (height - 1) * width; i++)
		{
			b = abs(-imageIn[i-width-1] - imageIn[i-width] - imageIn[i-width+1] -
					imageIn[i-1]       + 8 * imageIn[i]   - imageIn[i+1] -
					imageIn[i+width-1] - imageIn[i+width] - imageIn[i+width+1])/9;

			if (b > WHITE)
				imageOut[i] = WHITE;
			else
				imageOut[i] = (BYTE)b;
		}
	}
	else {
		//printf("Must be greyscale image!\n");
		return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_threshold(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int i; 
	int width, height;
	BYTE *imageOut,*imageIn;
	BYTE threshold;

	Picture *p_imageIn = *p_imageInputs;
	threshold = (BYTE) (255 * params[0]);

	if(p_imageIn->format==pix_grey)
	{
		imageIn=(BYTE *)p_imageIn->data;
		imageOut=(BYTE *)p_imageOut->data;

		width=p_imageIn->width;
		height=p_imageIn->height;

		for (i = 0; i < width * height; i++)
		{
			if (imageIn[i] >= threshold)
				imageOut[i] = WHITE;
			else
				imageOut[i] = BLACK;
		}
	}
	else {
		//printf("Must be grayscale image!\n");
		return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_col2gray(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int i;
	BYTE *imageOut, *imageIn;

	Picture *p_imageIn = *p_imageInputs;

	switch(p_imageIn->format) {
		case pix_rgb24:
		case pix_bgr24:
			imageIn=(BYTE *)p_imageIn->data;
			p_imageOut->datasize=p_imageIn->datasize / 3;
			p_imageOut->bytes_per_line=p_imageIn->bytes_per_line / 3;
			p_imageOut->bytes_per_pixel=1;
			p_imageOut->width=p_imageIn->width;
			p_imageOut->height=p_imageIn->height;
  		if(p_imageOut->format!=pix_grey) {
				p_imageOut->format=pix_grey;
  			if(p_imageOut->data!=NULL)
					p_imageOut->data=(BYTE *)realloc(p_imageOut->data, p_imageOut->datasize);
				else {
					p_imageOut->data=(BYTE *)malloc(p_imageOut->datasize);
					if(p_imageOut->data == NULL) {
						fprintf(stderr,"Error: IPL_col2gray - No memory available for output image\n");
						return WRONGFORMAT;
					}
				}
			}
			imageOut=(BYTE *)p_imageOut->data;
  		for (i = 0; i < (p_imageIn->height)*p_imageIn->width; i++)
				imageOut[i] = (imageIn[3*i] + imageIn[3*i+1] + imageIn[3*i+2])/ 3 ;
  		p_imageOut->format=pix_grey;
			break;
		case pix_rgb32:
		case pix_bgr32:
			imageIn=(BYTE *)p_imageIn->data;
			p_imageOut->datasize=p_imageIn->datasize / 4;
			p_imageOut->bytes_per_line=p_imageIn->bytes_per_line / 4;
			p_imageOut->bytes_per_pixel=1;
			p_imageOut->width=p_imageIn->width;
			p_imageOut->height=p_imageIn->height;
			if(p_imageOut->format!=pix_grey) {
				p_imageOut->format=pix_grey;
  				if(p_imageOut->data!=NULL)
						p_imageOut->data=(BYTE *)realloc(p_imageOut->data, p_imageOut->datasize);
				else {
					p_imageOut->data=(BYTE *)malloc(p_imageOut->datasize);
					if(p_imageOut->data == NULL) {
						fprintf(stderr,"Error: IPL_col2gray - No memory available for output image\n");
						return WRONGFORMAT;
					}
				}
			}
			imageOut=(BYTE *)p_imageOut->data;
			for (i = 0; i < (p_imageIn->height)*p_imageIn->width; i++)
				imageOut[i] = (imageIn[4*i] + imageIn[4*i+1] + imageIn[4*i+2])/ 3 ;
			p_imageOut->format=pix_grey;
			break;
		default:
			//printf("Must be RGB Picture!\n");
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_show(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int pos, pos2;
	int width, height, x, y;
	BYTE *imageOut,*imageIn;

	Picture *p_imageIn = *p_imageInputs;
	width=p_imageIn->width;
	height=p_imageIn->height;
	
	x = (int)(params[0]*(width-1));
	y = (int)(params[1]*(height-1));

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;

	switch(p_imageIn->format) {
		case pix_grey:
			memcpy(imageOut, imageIn, width * height);
			sprintf(result,"Show(%d,%d) = %d", x, y, imageIn[x + y * width]);
			if (x > 1 && x < width - 1 && y > 1 && y < height - 1) {
				imageOut[x + y * width    ] = WHITE - imageIn[x + y * width    ];
				imageOut[x + y * width + 1] = WHITE - imageIn[x + y * width + 1];
				imageOut[x + y * width + 2] = WHITE - imageIn[x + y * width + 2];
				imageOut[x + y * width - 1] = WHITE - imageIn[x + y * width - 1];
				imageOut[x + y * width - 2] = WHITE - imageIn[x + y * width - 2];
				imageOut[x + (y+1) * width] = WHITE - imageIn[x + (y+1) * width];
				imageOut[x + (y+2) * width] = WHITE - imageIn[x + (y+2) * width];
				imageOut[x + (y-1) * width] = WHITE - imageIn[x + (y-1) * width];
				imageOut[x + (y-2) * width] = WHITE - imageIn[x + (y-2) * width];
			}
			break;
		case pix_rgb24:
		case pix_bgr24:
			memcpy(imageOut, imageIn, 3 * width * height);
			pos = 3*(x + y * width);
			sprintf(result,"Show(%d,%d) = %d %d %d", x, y, imageIn[pos], imageIn[pos+1], imageIn[pos+2]);
			if (x > 1 && x < width - 1 && y > 1 && y < height - 1) {
				col_invert(&imageOut[pos]);
				col_invert(&imageOut[pos+3]);
				col_invert(&imageOut[pos-3]);
				col_invert(&imageOut[pos+6]);
				col_invert(&imageOut[pos-6]);
				pos2 = pos + 3 * width;
				col_invert(&imageOut[pos2]);
				pos2 = pos + 6 * width;
				col_invert(&imageOut[pos2]);
				pos2 = pos - 3 * width;
				col_invert(&imageOut[pos2]);
				pos2 = pos - 6 * width;
				col_invert(&imageOut[pos2]);
			}
			break;
		case pix_rgb32:
		case pix_bgr32:
			memcpy(imageOut, imageIn, 4 * width * height);
			pos = 4*(x + y * width);
			sprintf(result,"Show(%d,%d) = %d %d %d", x, y, imageIn[pos], imageIn[pos+1], imageIn[pos+2]);
			if (x > 1 && x < width - 1 && y > 1 && y < height - 1) {
				col_invert(&imageOut[pos]);
				col_invert(&imageOut[pos+4]);
				col_invert(&imageOut[pos-4]);
				col_invert(&imageOut[pos+8]);
				col_invert(&imageOut[pos-8]);
				pos2 = pos + 4 * width;
				col_invert(&imageOut[pos2]);
				pos2 = pos + 8 * width;
				col_invert(&imageOut[pos2]);
				pos2 = pos - 4 * width;
				col_invert(&imageOut[pos2]);
				pos2 = pos - 8 * width;
				col_invert(&imageOut[pos2]);
			}
			break;
		default:
			if(NOISY) fprintf(stderr,"IPL_Show (x,y): Not implemented for this type!\n");
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_match_color_range(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int r,g,b,intensity,i;
	int width, height;
	float rf, gf, bf;
	BYTE *imageOut,*imageIn;

	Picture *p_imageIn = *p_imageInputs;
	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	rf = params[0];
	gf = params[1];
	bf = params[2];

	if(p_imageOut->format!=pix_grey) {
		p_imageOut->format = pix_grey;
		resizeImage(p_imageOut, p_imageOut->width, p_imageOut->height);
	}

	switch(p_imageIn->format) {
		case pix_rgb24:
			for (i = width; i < (height-2)*width; i++) {
				r = imageIn[3*i];
				g = imageIn[3*i+1];
				b = imageIn[3*i+2];

				intensity = r+g+b;
				if(r >= rf * intensity &&
				   g <= gf * intensity &&
				   b >= bf * intensity) {
					imageOut[i] = WHITE;
					//imageOut[3*i] = WHITE;
					//imageOut[3*i+1] = WHITE;
					//imageOut[3*i+2] = WHITE;
				}
				else {
					imageOut[i] = BLACK;
					//imageOut[3*i] = BLACK;
					//imageOut[3*i+1] = BLACK;
					//imageOut[3*i+2] = BLACK;
				}
			}
			break;
		case pix_bgr24:
			for (i = width; i < (height-2)*width; i++) {
				r = imageIn[3*i+2];
				g = imageIn[3*i+1];
				b = imageIn[3*i];

				intensity = r+g+b;
				if(r >= rf * intensity &&
				   g <= gf * intensity &&
				   b >= bf * intensity) {
					imageOut[i] = WHITE;
				}
				else {
					imageOut[i] = BLACK;
				}
			}
			break;
		case pix_rgb32:
			for (i = width; i < (height-2)*width; i++) {
				r = imageIn[4*i];
				g = imageIn[4*i+1];
				b = imageIn[4*i+2];

				intensity = r+g+b;
				if(r >= rf * intensity &&
				   g <= gf * intensity &&
				   b >= bf * intensity) {
					imageOut[i] = WHITE;
				}
				else {
					imageOut[i] = BLACK;
				}
			}
			break;
		case pix_bgr32:
			for (i = width; i < (height-2)*width; i++) {
				r = imageIn[4*i+2];
				g = imageIn[4*i+1];
				b = imageIn[4*i];

				intensity = r+g+b;
				if(r >= rf * intensity &&
				   g <= gf * intensity &&
				   b >= bf * intensity) {
					imageOut[i] = WHITE;
				}
				else {
					imageOut[i] = BLACK;
				}
			}
			break;
		default:
			if(NOISY) fprintf(stderr,"IPL_match_color_range: Must be a colour image!\n");
			return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_HSVValue(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int width, height;
	double hue, sat, val;
	double h, s, v, r, g, b;
	int index = 0;
	int pos;
	int j, k;
	BYTE *imageOut,*imageIn;

	Picture *p_imageIn = *p_imageInputs;
	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	if(p_imageOut->format!=pix_grey) {
		p_imageOut->format = pix_grey;
		resizeImage(p_imageOut, p_imageOut->width, p_imageOut->height);
	}
	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;
	width=p_imageIn->width;
	height=p_imageIn->height;

	switch(p_imageIn->format) {
		case pix_rgb24:
			for(j = 0; j < height; j++) {
				for(k = 0; k < width; k++) {
					pos = 3*(k + j * width);
					r = (double)imageIn[pos]/255;
					g = (double)imageIn[pos +1]/255;
					b = (double)imageIn[pos + 2]/255;
					IPLi_rgbToHsv(r, g, b, &h, &s, &v);
					hue = h * 255;
					sat = s * 255;
					val = v * 255;
					imageOut[index ++] = (BYTE)hue;
				}
			}
			break;
		case pix_bgr24:
			for(j = 0; j < height; j++) {
				for(k = 0; k < width; k++) {
					pos = 3*(k + j * width);
					r = (double)imageIn[pos]/255;
					g = (double)imageIn[pos +1]/255;
					b = (double)imageIn[pos + 2]/255;
					IPLi_rgbToHsv(r, g, b, &h, &s, &v);
					hue = h * 255;
					sat = s * 255;
					val = v * 255;
					imageOut[index ++] = (BYTE)hue;
				}
			}
			break;
		case pix_rgb32:
			for(j = 0; j < height; j++) {
				for(k = 0; k < width; k++) {
					pos = 4*(k + j * width);
					r = (double)imageIn[pos]/255;
					g = (double)imageIn[pos + 1]/255;
					b = (double)imageIn[pos + 2]/255;
					IPLi_rgbToHsv(r, g, b, &h, &s, &v);
					hue = h * 255;
					sat = s * 255;
					val = v * 255;
					imageOut[index ++] = (BYTE)hue;
				}
			}
			break;
		case pix_bgr32:
			for(j = 0; j < height; j++) {
				for(k = 0; k < width; k++) {
					pos = 4*(k + j * width);
					r = (double)imageIn[pos + 2]/255;
					g = (double)imageIn[pos + 1]/255;
					b = (double)imageIn[pos]/255;
					IPLi_rgbToHsv(r, g, b, &h, &s, &v);
					hue = h * 255;
					sat = s * 255;
					val = v * 255;
					imageOut[index ++] = (BYTE)hue;
				}
			}
			break;
		default:
			//printf("Must be RGB image!\n");
			return WRONGFORMAT;
	}
	return NOERROR;
}

// Convert rgb to hsv
void IPLi_rgbToHsv(double r, double g, double b, double *h, double *s, double *v)
{
  double max = MAX (r, MAX (g,b)), min = MIN(r, MIN(g,b));
  double delta = max - min;

  *v = max;
  if(max != 0.0)
    *s = delta/max;

  else
    *s = 0.0;

  if (*s == 0.0)
    *h = -1;

  else {
    if (r == max)
      *h = (g-b) / delta;

    else if (g == max)
      *h = 2 + (b - r) / delta;

    else if (b == max)
      *h = 4 + (r-g) / delta;

    *h *= 60.0;
    if(*h < 0)
      *h += 360.0;

    *h /= 360.0;
  }
}

pluginReturnType IPL_gray_reduce(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
  int i, numvals=0;
  int width, height;
  BYTE maxVal = BLACK;
  BYTE minVal = WHITE;
  BYTE range;
  BYTE *imageOut,*imageIn;
	BYTE n[256];

	Picture *p_imageIn = *p_imageInputs;
  imageIn=(BYTE *)p_imageIn->data;
  imageOut=(BYTE *)p_imageOut->data;

  width=p_imageIn->width;
  height=p_imageIn->height;

	// used to use param * 50, now uses param*1/2 number of levels in image
	for (i=0; i<width*height; i++) {
		if(n[imageIn[i]]!=1) n[imageIn[i]] = 1;
	}
	for (i=0; i<256; i++) {
		if(n[i]==1) numvals++;
	}
	numvals = params[0] * (numvals/2);

  if(numvals < 2) numvals=2;

  if(p_imageIn->format==pix_grey) {
		imageIn=(BYTE *)p_imageIn->data;
		imageOut=(BYTE *)p_imageOut->data;

		width=p_imageIn->width;
		height=p_imageIn->height;

		for (i = 0; i < width*height; i++) {
			if (imageIn[i] < minVal) minVal = imageIn[i];
			if (imageIn[i] > maxVal) maxVal = imageIn[i];
		}

		range = maxVal - minVal;

		if (range + 1 > numvals) {
			for (i = 0; i < width*height; i++) {
				int temp = ((numvals-1) * (imageIn[i] - minVal) / range);
				imageOut[i] = (BYTE)((range * temp / (numvals - 1)) + minVal);
			}
		}
		else memcpy(imageOut, imageIn, width * height);
	}
	else {
		//printf("Must be grayscale image!\n");
		return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_bin_dilation(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
  int x, y;
  int offset;
  int width, height;
  BYTE *imageOut,*imageIn;

	Picture *p_imageIn = *p_imageInputs;
  imageIn=(BYTE *)p_imageIn->data;
  imageOut=(BYTE *)p_imageOut->data;

  width=p_imageIn->width;
  height=p_imageIn->height;

  if(p_imageIn->format==pix_grey)
    {
      for (y = 1; y < height - 1; y++)
	{
        for (x = 1; x < width - 1; x++)
	{
	  offset = x + y * width;
	
	  if ((struc[0] == 1 && imageIn[offset-1 - width] == BLACK) ||
	      (struc[1] == 1 && imageIn[offset   - width] == BLACK) ||
	      (struc[2] == 1 && imageIn[offset+1 - width] == BLACK) ||
	      (struc[3] == 1 && imageIn[offset-1] == BLACK) ||
	      (struc[4] == 1 && imageIn[offset] == BLACK) ||
	      (struc[5] == 1 && imageIn[offset+1] == BLACK) ||
	      (struc[6] == 1 && imageIn[offset-1+width] == BLACK) ||
	      (struc[7] == 1 && imageIn[offset+width] == BLACK) ||
	      (struc[8] == 1 && imageIn[offset+1+width] == BLACK))
	    imageOut[offset] = BLACK;
	  else
	    imageOut[offset] = WHITE;
	}
	}
    }
	else {
		//printf("Must be greyscale image!\n");
		return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_bin_erosion(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
    int x, y;
    int offset;
    int width, height;

    BYTE *imageOut,*imageIn;

    Picture *p_imageIn = *p_imageInputs;
		imageIn=(BYTE *)p_imageIn->data;
    imageOut=(BYTE *)p_imageOut->data;

    width=p_imageIn->width;
    height=p_imageIn->height;

    if(p_imageIn->format==pix_grey)
      {
		for (y = 1; y < height - 1; y++)
	  {
	    for (x = 1; x < width - 1; x++)
	      {
		offset = x + y * width;

		if ((struc[0] == 1 && imageIn[offset-1 - width] == WHITE) ||
		    (struc[1] == 1 && imageIn[offset   - width] == WHITE) ||
		    (struc[2] == 1 && imageIn[offset+1 - width] == WHITE) ||
		    (struc[3] == 1 && imageIn[offset-1] == WHITE) ||
		    (struc[4] == 1 && imageIn[offset] == WHITE) ||
		    (struc[5] == 1 && imageIn[offset+1] == WHITE) ||
		    (struc[6] == 1 && imageIn[offset-1+width] == WHITE) ||
		    (struc[7] == 1 && imageIn[offset+width] == WHITE) ||
		    (struc[8] == 1 && imageIn[offset+1+width] == WHITE))
		  imageOut[offset] = WHITE;
		else
		  imageOut[offset] = BLACK;
	      }
	  }
      }
    else {
		//printf("Must be greyscale image!\n");
		return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_bin_open(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	Picture *p_imageIn = *p_imageInputs;
	Picture *p_temp;

	p_temp=copyImage(p_imageIn);
	clearImage(p_temp);

	//IPL_erosion(p_imageIn, p_temp, struc);
	//IPL_dilation(p_temp, p_imageOut, struc);
	IPL_bin_erosion(instance, p_imageInputs, p_temp, params, result);
	IPL_bin_dilation(instance, &p_temp, p_imageOut, params, result);

	freeImage(p_temp);
	return NOERROR;
}

pluginReturnType IPL_bin_close(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	Picture *p_imageIn = *p_imageInputs;
	Picture *p_temp;

	p_temp=copyImage(p_imageIn);
	clearImage(p_temp);

	//IPL_dilation(p_imageIn, p_temp, struc);
	//IPL_erosion(p_temp, p_imageOut, struc);
	IPL_bin_dilation(instance, p_imageInputs, p_temp, params, result);
	IPL_bin_erosion(instance, &p_temp, p_imageOut, params, result);

	freeImage(p_temp);
	return NOERROR;
}

pluginReturnType IPL_bin_fill(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	Picture *p_grow;
	Picture *p_new_grow;
	Picture *p_not_img;
	int x, y, width, height;

	int loopCount = 0;

	BYTE *imageOut, *imageIn;

	Picture *p_imageIn = *p_imageInputs;
	p_grow=copyImage(p_imageIn);
	p_new_grow=copyImage(p_imageIn);
	p_not_img=copyImage(p_imageIn);

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	x = (int)(params[0]*(width-1));
	y = (int)(params[1]*(height-1));

	//IPL_negation(p_imageIn, p_not_img);
	IPL_negation(instance, p_imageInputs, p_not_img, params, result);

	memset(p_new_grow->data, WHITE, width * height);
	p_new_grow->data[x + y * width] = BLACK;
	do {
		memcpy(p_grow->data, p_new_grow->data, width * height);
		//IPL_dilation(p_grow, p_new_grow, struc);
		IPL_bin_dilation(instance, &p_grow, p_new_grow, params, result);
		IPLi_or(p_new_grow, p_not_img, p_new_grow);
		loopCount++;
	} while (!IPLi_equal(p_grow, p_new_grow));

	memcpy(imageOut, p_grow->data, width * height);

	freeImage(p_grow);
	freeImage(p_new_grow);
	freeImage(p_not_img);

	if(loopCount<1000)
		sprintf(result,"Fill loops = %d\n", loopCount);

	return NOERROR;
}

pluginReturnType IPL_bin_connected(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	Picture *p_grow;
	Picture *p_new_grow;
	int x, y, width, height;
	BYTE *imageOut,*imageIn;
	int loopCount = 0;

	Picture *p_imageIn = *p_imageInputs;
	p_grow=copyImage(p_imageIn);
	p_new_grow=copyImage(p_imageIn);

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;

	width=p_imageIn->width;
	height=p_imageIn->height;

	x = (int)(params[0]*(width-1));
	y = (int)(params[1]*(height-1));

	memset(p_new_grow->data, WHITE, width * height);
	p_new_grow->data[x + y * width] = BLACK;
	do {
		memcpy(p_grow->data, p_new_grow->data, width * height);
		//IPL_dilation(p_grow, p_new_grow, struc);
		IPL_bin_dilation(instance, &p_grow, p_new_grow, params, result);
		IPLi_or(p_new_grow, p_imageIn, p_new_grow);
		loopCount++;
	} while (!IPLi_equal(p_grow, p_new_grow));
	
	memcpy(p_imageOut->data, p_grow->data, width * height);
	if(loopCount<1000)
		sprintf(result,"Connected loops = %d\n", loopCount);
	freeImage(p_grow);
	freeImage(p_new_grow);
	return NOERROR;
}

pluginReturnType IPL_bin_boundary(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	Picture *p_imageIn = *p_imageInputs;
	//IPL_erosion(p_imageIn, p_imageOut, struc);
	IPL_bin_erosion(instance, p_imageInputs, p_imageOut, params, result);
	//IPL_negation(p_imageOut, p_imageOut);
	IPL_negation(instance, &p_imageOut, p_imageOut, params, result);
	IPLi_or(p_imageIn, p_imageOut, p_imageOut);
	return NOERROR;
}

pluginReturnType IPL_bin_skeleton(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
  Picture *p_help;
  Picture *p_times;
  int loopCount = 0;
  int width, height;
  BYTE *imageOut,*imageIn;

	Picture *p_imageIn = *p_imageInputs;
  imageIn=(BYTE *)p_imageIn->data;
  imageOut=(BYTE *)p_imageOut->data;

  p_help=copyImage(p_imageIn);
  p_times=copyImage(p_imageIn);

  width=p_imageIn->width;
  height=p_imageIn->height;

  if(p_imageIn->format==pix_grey)
    {
  memset(p_imageOut->data, WHITE, width * height);
  memcpy(p_times->data, p_imageIn->data, width * height);

  //while(IPLi_count_nobound(p_times->data, p_help->data, BLACK) > 0)
	while(IPLi_count_nobound(p_times, BLACK) > 0)
    {
      //IPL_open(p_times, p_help, struc);
			IPL_bin_open(instance, &p_times, p_help, params, result);
      //IPL_negation(p_help, p_help);
			IPL_negation(instance, &p_help, p_help, params, result);
      IPLi_or(p_help, p_times, p_help);
      IPLi_and(p_help, p_imageOut, p_imageOut);

      memcpy(p_help->data, p_times->data, width * height);
      //IPL_erosion(p_help, p_times, struc);
			IPL_bin_erosion(instance, &p_help, p_times, params, result);

      loopCount++;
    }

	if(loopCount<1000)
		sprintf(result,"Skeleton loops = %d\n", loopCount);

  freeImage(p_times);freeImage(p_help);
    }
	else {
		//printf("Must be greyscale image !\n");
		return WRONGFORMAT;
	}
	return NOERROR;
}

int IPLi_equal(Picture *p_imageIn, Picture *p_imageOut)
{
  int i;
  int width, height;
  BYTE *imageOut,*imageIn;

  imageIn=(BYTE *)p_imageIn->data;
  imageOut=(BYTE *)p_imageOut->data;

  width=p_imageIn->width;
  height=p_imageIn->height;

  for (i = 0; i < width * height; i++)
    if (imageIn[i] != imageOut[i])
      return 0;

  return 1;
}

//int IPL_equal_nobound(Picture *p_image1, Picture *p_image2);

//int IPLi_count_nobound(Picture *p_imageIn, Picture *p_imageOut, BYTE val)
int IPLi_count_nobound(Picture *p_imageIn, BYTE val)
{
  int x, y, width, height;
  int count = 0;
  //BYTE *imageOut;
	BYTE *imageIn;

	imageIn=(BYTE *)p_imageIn->data;
  //imageOut=(BYTE *)p_imageOut->data;

	width=p_imageIn->width;
  height=p_imageIn->height;

  //memcpy(imageOut, imageIn, width * height); // identity

  for (y = 1; y < height - 1; y++)
    for (x = 1; x < width - 1; x++)
      if (imageIn[x + y * width] == val) count++;

  return count;
}

// UPTO - find circles from version 1

static BYTE markerDone;
static BYTE markerContour;
static int contourX[] = { 1,  1,  0, -1, -1, -1, 0, 1 };
static int contourY[] = { 0, -1, -1, -1,  0,  1, 1, 1 };
static Point contour[MAX_CONTOUR];
static double dcoord[MAX_CONTOUR];

////int IPLi_FindCircles(BYTE * imageIn, BYTE * imageOut, XYDistance * result)
//pluginReturnType IPL_FindCircles(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
//{
//	int i;
//	int count;
//	Point points[256];
//	BYTE *imageIn = p_imageInputs[0]->data;
//	BYTE *imageOut = p_imageOut->data;
//
//	count = IPLi_contour(imageIn, imageOut, &points[0], 8);
//	if(result == NULL)
//		//return count;
//
//	for (i = 0; i < count; i++) {
//		result[i].x = (double)points[i].x / width;
//		result[i].y = (double)(height - points[i].y) / height;
//	}
//	//return count;
//}

//int IPLi_contour(BYTE * imageIn, BYTE * imageOut, Point * result, BYTE threshold)
int IPLi_contour(Picture *pIn, Picture *pOut, Point * result, BYTE threshold)
{
	int x, y;
	int countCircles = 0;
	int width = pIn->width;
	int height = pIn->height;
	BYTE *imageIn = pIn->data;
	BYTE *imageOut = pOut->data;
	if (imageOut != NULL)
		memset(imageOut, 0, width * height);
	for (y = 1; y < height - 1; y++) {
		for (x = 1; x < width - 2; x++) {
			if (imageIn[x + y * width] != markerDone && IPLi_IsEdge(pIn, x, y, 2, 10)) {
				unsigned length = IPLi_FollowContour(pIn, x, y, threshold, &contour[0], MAX_CONTOUR);
				if (length > 4 && length < 30) {
					Point p;
					if (IPLi_CheckCircle(length, &contour[0], &p) == 0) {
						if (result != NULL)
							result[countCircles] = p;
						if (imageOut != NULL) {
							imageOut[p.x + p.y * width] = markerContour;
							while (length-- > 0) {
								imageOut[contour[length].x + contour[length].y * width] = markerContour;
							}
						}
						countCircles++;
					}
				}
			}
		}
	}
	//printf("%d circles found\n", countCircles);
	return countCircles;
}

//static int IPLi_IsEdge(BYTE * image, int x, int y, int w, BYTE threshold)
static int IPLi_IsEdge(Picture *pIn, int x, int y, int w, BYTE threshold)
{
	int i;
	int left = 0, right = 0;
	BYTE * p;
	int width = pIn->width;
	BYTE *image = pIn->data;

    if (x - w < 0 || x + w >= width)
        return 0;

    p = &image[y * width + x - w];

    for (i = 0; i < w; i++)
    {
        left  += p[i];
        right += p[i + w];
    }

    if (right - left > threshold * w)
        return 1;
    else
        return 0;
}

static int IPLi_CheckCircle(int length, Point * contour, Point * cog)
{
    int i;
    double cogX = contour[0].x;
    double cogY = contour[0].y;
    double cogDelta = 0.0;
    double cogVar = 0.0;

    if (length <= 0)
        return -1;

    /*
     *    Determine the center of gravity. Take the x/y ratio of 4/3
     *    into account.
     */

    for (i = 1; i < length; i++)
    {
        cogX += contour[i].x;
        cogY += contour[i].y;
    }

    cogY = 4.0 * cogY / (3.0 * length);
    cogX /= length;

    cog -> x = (short)cogX;
    cog -> y = (short)(cogY * 0.75);

    /*
     *
     */

    for (i = 0; i < length; i++)
    {
        double delta =  sqrt((contour[i].x - cogX) * (contour[i].x - cogX) +
                             (4.0 * contour[i].y / 3.0 - cogY) *
                             (4.0 * contour[i].y / 3.0 - cogY));
        dcoord[i] = delta;
        cogDelta += delta;
    }

    cogDelta /= length;

    for (i = 0; i < length; i++)
        cogVar += (cogDelta - dcoord[i]) * (cogDelta - dcoord[i]);
    cogVar /= length;


    if (cogVar < 0.45)
        return 0;
    else
        return 1;
}

//static int IPLi_FollowContour(BYTE * imageIn, int x, int y, BYTE threshold, Point *points, int maxLength)
static int IPLi_FollowContour(Picture *pIn, int x, int y, BYTE threshold, Point *points, int maxLength)
{
	int done = 0;
	int startX = x;
	int startY = y;
	int newX=0, newY=0;
	unsigned int direction = 6;
	unsigned int newDir = 0;
	unsigned int length = 0;
	int width = pIn->width;
	int height = pIn->height;
	BYTE *imageIn = pIn->data;

	/* follow the contour as long as the 'done' flag is not set */

	while (!done) {
		int i;
		imageIn[x + y * width] = markerDone;
		/* store the coordinates in the 'points' array if a valid
		 * pointer is supplied and 'maxLength' is not reached.
		 */
		if (points != NULL && length < maxLength) {
			points[length].x = (short)x;
			points[length].y = (short)y;
		}
		length++;
		for (i = 0; i < 8; i++) {
			newDir = (direction + i - 3) % 8;
			newX = x + contourX[newDir];
			newY = y + contourY[newDir];
			if(newX >= 0 && newY >= 0 && newX < width &&
			   newY < height && imageIn[newX + newY * width] > threshold)
				break;
		}
		if (i >= 8) done = 1;
		else {
			x = newX;
			y = newY;
			direction = newDir;
		}
		if (x == startX && y == startY) done = 1;
	}
	return length;
}

////////////////////////////////////////////////////////////
// begin leon's additional code 2002
////////////////////////////////////////////////////////////

/* Modified to handle all combinations of images
 * of different types (pix_rgb24/rgb32, pix_grey)
 */
pluginReturnType IPL_difference(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int i, j, width, height, type=1;
	BYTE *imageOut,*imageIn1,*imageIn2;
	Picture *resized;

	Picture *p_imageIn1 = p_imageInputs[0];
	Picture *p_imageIn2 = p_imageInputs[1];

	imageIn1=(BYTE *)p_imageIn1->data;
	imageIn2=(BYTE *)p_imageIn2->data;
	imageOut=(BYTE *)p_imageOut->data;
	
	width=p_imageIn1->width;
	height=p_imageIn1->height;

	if(p_imageIn2==NULL) { return WRONGFORMAT; }
	else if(((BYTE *)p_imageIn2->data)==NULL) { return WRONGFORMAT; }
	else {
		if((p_imageIn1->width!=p_imageIn2->width)||
			 (p_imageIn1->height!=p_imageIn2->height)) {
			resized = copyImage(p_imageIn2);
			resizeImage(resized,p_imageIn1->width,p_imageIn1->height);
			imageIn2 = (BYTE *)resized->data;
		}
		// find the type of combination
		if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==1)) type=0;
		if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==1)) {
			type=1;
		  imageIn1=(BYTE *)p_imageIn2->data;
			imageIn2=(BYTE *)p_imageIn1->data;
		}
		if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==4)) type=2;

		// find the difference of the images, using different cases depending upon their format.
		if(type==0) {
		  for (i=0; i<width*height; i++) {
				//delta = imageIn1[i] - imageIn2[i];
				//if (delta < 0) delta = - delta;
				imageOut[i] = (BYTE)abs(imageIn1[i] - imageIn2[i]);
			}
		}
		if(type==1) {
		  for (i=0,j=0; i<4*width*height; i+=4) {
				imageOut[i  ] = (BYTE)abs(imageIn1[j] - imageIn2[i  ]);
				imageOut[i+1] = (BYTE)abs(imageIn1[j] - imageIn2[i+1]);
				imageOut[i+2] = (BYTE)abs(imageIn1[j] - imageIn2[i+2]);
				imageOut[i+3] = (BYTE)abs(imageIn1[j] - imageIn2[i+3]);
				j++;
			}
		}
		if(type==2) {
		  for (i=0; i<4*width*height; i++)
				//delta = imageIn1[i] - imageIn2[i];
				//if (delta < 0) delta = - delta;
				imageOut[i] = (BYTE)abs(imageIn1[i] - imageIn2[i]);
		}
		return NOERROR;	
	}
}

/* Modified to handle all combinations of images
 * of different types (pix_rgb24/rgb32, pix_grey)
 */
void IPLi_or(Picture *p_imageIn1, Picture *p_imageIn2, Picture *p_imageOut)
{
	int i, j, type=1;
	int width, height;
	BYTE *imageOut,*imageIn1,*imageIn2;

	imageIn1=(BYTE *)p_imageIn1->data;
	imageIn2=(BYTE *)p_imageIn2->data;
	imageOut=(BYTE *)p_imageOut->data;

	width=p_imageIn1->width;
	height=p_imageIn1->height;

	// find the type of combination
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==1)) type=0;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==1)) {
		type=1;
		imageIn1=(BYTE *)p_imageIn2->data;
		imageIn2=(BYTE *)p_imageIn1->data;
	}
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==4)) type=2;

	// OR the images, using different cases depending upon their format.
	if(type==0) {
		for (i=0; i<width*height; i++)
			imageOut[i] = imageIn1[i] | imageIn2[i];
	}
	if(type==1) {
	  for (i=0,j=0; i<4*width*height; i+=4) {
			imageOut[i  ] = imageIn1[j] | imageIn2[i  ];
			imageOut[i+1] = imageIn1[j] | imageIn2[i+1];
			imageOut[i+2] = imageIn1[j] | imageIn2[i+2];
			imageOut[i+3] = imageIn1[j] | imageIn2[i+3];
			j++;
		}
	}
	if(type==2) {
		for (i=0; i<4*width*height; i++)
			imageOut[i] = imageIn1[i] | imageIn2[i];
	}	
}

/* Modified to handle all combinations of images
 * of different types (pix_rgb24/rgb32, pix_grey)
 */
void IPLi_and(Picture *p_imageIn1, Picture *p_imageIn2, Picture *p_imageOut)
{
	int i, j, type=1;
	int width, height;
	BYTE *imageOut,*imageIn1,*imageIn2;

	imageIn1=(BYTE *)p_imageIn1->data;
	imageIn2=(BYTE *)p_imageIn2->data;
	imageOut=(BYTE *)p_imageOut->data;

	width=p_imageIn1->width;
	height=p_imageIn1->height;

	// find the type of combination
	if((p_imageIn1->bytes_per_pixel==1)&&(p_imageIn2->bytes_per_pixel==1)) type=0;
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==1)) {
		type=1;
		imageIn1=(BYTE *)p_imageIn2->data;
		imageIn2=(BYTE *)p_imageIn1->data;
	}
	if((p_imageIn1->bytes_per_pixel==4)&&(p_imageIn2->bytes_per_pixel==4)) type=2;

	// AND the images, using different cases depending upon their format.
	if(type==0) {
		for (i=0; i<width*height; i++)
			imageOut[i] = imageIn1[i] & imageIn2[i];
	}
	if(type==1) {
		for (i=0,j=0; i<4*width*height; i+=4) {
			imageOut[i  ] = imageIn1[j] & imageIn2[i  ];
			imageOut[i+1] = imageIn1[j] & imageIn2[i+1];
			imageOut[i+2] = imageIn1[j] & imageIn2[i+2];
			imageOut[i+3] = imageIn1[j] & imageIn2[i+3];
			j++;
		}
	}
	if(type==2) {
		for (i=0; i<4*width*height; i++)
			imageOut[i] = imageIn1[i] & imageIn2[i];
	}	
}

pluginReturnType IPL_and(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int width, height;
	Picture *resized;

	Picture *p_image1 = p_imageInputs[0];
	Picture *p_image2 = p_imageInputs[1];

	width=p_image1->width;
	height=p_image1->height;

	if(p_image2==NULL) { return WRONGFORMAT; }
	else if(((BYTE *)p_image2->data)==NULL) { return WRONGFORMAT; }
	else {
		if((p_image1->width!=p_image2->width)||
			 (p_image1->height!=p_image2->height)) {
			resized = copyImage(p_image2);
			resizeImage(resized,p_image1->width,p_image1->height);
			IPLi_and(p_image1, resized, p_imageOut);
		}
		else {
			IPLi_and(p_image1, p_image2, p_imageOut);
		}
	}
	return NOERROR;
}

pluginReturnType IPL_or(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int width, height;
	Picture *resized;

	Picture *p_image1 = p_imageInputs[0];
	Picture *p_image2 = p_imageInputs[1];

	width=p_image1->width;
	height=p_image1->height;

	if(p_image2==NULL) { return WRONGFORMAT; }
	else if(((BYTE *)p_image2->data)==NULL) { return WRONGFORMAT; }
	else {
		if((p_image1->width!=p_image2->width)||
			 (p_image1->height!=p_image2->height)) {
			resized = copyImage(p_image2);
			resizeImage(resized,p_image1->width,p_image1->height);
			IPLi_or(p_image1, resized, p_imageOut);
		}
		else {
			IPLi_or(p_image1, p_image2, p_imageOut);
		}
	}
	return NOERROR;
}

/** @todo not finished! */
pluginReturnType IPL_erosion(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int i, j, width, height;
	int hasValue[WHITE+1];
	BYTE *imageOut;
	BYTE *imageIn;
	BYTE *imageResult;
	Picture *p_buf, *p_result, *p_imageIn;
	float *thresholdParam;
	thresholdParam = (float *)malloc(sizeof(float));

	p_imageIn = *p_imageInputs;
	p_buf = copyImage(p_imageIn);
	p_result = copyImage(p_imageIn);
	
	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE*)p_imageOut->data;

	clearImage(p_imageOut);

	width=p_imageIn->width;
	height=p_imageIn->height;

	// initialise
	for(i=0;i<WHITE+1;i++) {
		hasValue[i] = 0;
	}
	// find all the grey scale values present
	for(i=0;i<width*height;i++) {
		hasValue[(int)imageIn[i]] = 1;
	}

	if(p_imageIn->format==pix_grey) {
		clearImage(p_imageOut);
		for(i=BLACK;i<=WHITE;i++) {
			if(hasValue[i]==1) {
				thresholdParam[0] = (float)i/(float)255;
				//IPL_threshold(NULL, &p_imageIn, p_buf, thresholdParam, NULL);
				LK_match_grey_range(NULL, &p_imageIn, p_buf, thresholdParam, NULL);
				IPL_bin_erosion(NULL, &p_buf, p_result, NULL, NULL);
				// convert the WHITE to the required greyscale
				imageResult=(BYTE *)p_result->data;
				for(j=0;j<width*height;j++) {
					if(imageResult[j]==WHITE) imageResult[j] = (BYTE)i;
				}
				//IPLi_or(p_result,p_imageOut,p_imageOut);
			  for(j=0; j<width*height;j++) {
					if(imageResult[j]!=BLACK) imageOut[j] = imageResult[j];
					//imageOut[j]=((imageOut[j]+imageResult[j])>WHITE) ? WHITE : imageOut[j]+imageResult[j];
				}
			}
		}
	}
	else {
		if(NOISY) fprintf(stderr,"IPL_erosion: Must be greyscale image !\n");
		return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType IPL_dilation(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int i, j, width, height;
	int hasValue[WHITE+1];
	BYTE *imageOut;
	BYTE *imageIn;
	BYTE *imageResult;
	Picture *p_buf, *p_result, *p_imageIn;
	float *thresholdParam;
	thresholdParam = (float *)malloc(sizeof(float));

	p_imageIn = *p_imageInputs;
	p_buf = copyImage(p_imageIn);
	p_result = copyImage(p_imageIn);
	
	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;

	clearImage(p_imageOut);

	width=p_imageIn->width;
	height=p_imageIn->height;

	// initialise
	for(i=0;i<WHITE+1;i++) {
		hasValue[i] = 0;
	}
	// find all the grey scale values present
	for(i=0;i<width*height;i++) {
		hasValue[(int)imageIn[i]] = 1;
	}

	if(p_imageIn->format==pix_grey) {
		clearImage(p_imageOut);
		for(i=BLACK;i<=WHITE;i++) {
			if(hasValue[i]==1) {
				thresholdParam[0] = (float)i/(float)255;
				//IPL_threshold(NULL, &p_imageIn, p_buf, thresholdParam, NULL);
				LK_match_grey_range(NULL, &p_imageIn, p_buf, thresholdParam, NULL);
				IPL_bin_dilation(NULL, &p_buf, p_result, NULL, NULL);
				// convert the WHITE to the required greyscale
				imageResult=(BYTE *)p_result->data;
				for(j=0;j<width*height;j++) {
					if(imageResult[j]==WHITE) imageResult[j] = (BYTE)i;
				}
				//IPLi_or(p_result,p_imageOut,p_imageOut);
				for(j=0; j<width*height;j++) {
					if(imageResult[j]!=BLACK) imageOut[j] = imageResult[j];
					//imageOut[j]=((imageOut[j]+imageResult[j])>WHITE) ? WHITE : imageOut[j]+imageResult[j];
				}
			}
		}
	}
	else {
		if(NOISY) fprintf(stderr, "IPL_dilation: Must be greyscale image !\n");
		return WRONGFORMAT;
	}
	return NOERROR;
}

pluginReturnType LK_match_grey_range(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
  int t,i;
  int width, height;
  BYTE *imageOut,*imageIn;

	Picture *p_imageIn = *p_imageInputs;
  imageIn=(BYTE *)p_imageIn->data;
  imageOut=(BYTE *)p_imageOut->data;

  width=p_imageIn->width;
  height=p_imageIn->height;

	t = (int)WHITE*params[0];

	if(p_imageIn->format==pix_grey) {
		for (i = 0; i<height*width; i++) {
			if(imageIn[i]==t) {
				imageOut[i] = WHITE;
			}
			else {
				imageOut[i] = BLACK;
			}
		}
		return NOERROR;
	}
	else {
		return WRONGFORMAT;
	}
}
