/* improvConfig.h
 * Last modified:
 * Authors: Daniel Venkitachalam <venki-d@ee.uwa.edu.au>
 *          Leon Koch <leon@redfishsoftware.com.au>
 */
/** @class Improv
 * A Singleton class that is used to track the global state of the application
 *
 * This class is used to track the global state of the application and provide
 * communications between the classes. This class is the guts of the whole app.
 *
 */

#ifndef IMPROV_H
#define IMPROV_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <assert.h>
#include <dirent.h>
#include <dlfcn.h>
#include <getopt.h>
#include <math.h>
#include <iostream>
#include <string>
#include <sys/types.h>

// QT includes
#include <qaction.h>
#include <qevent.h>
#include <qpopupmenu.h>
#include <qstatusbar.h>
#include <qvaluelist.h>

// Application includes, dependent upon QT
#include "AnimThread.h"
#include "ParamWidget.h"
#include "PluginMenu.h"
#include "SeqWidget.h"
#include "SrcWidget.h"
#include "ImprovQTCamOpt.h"
//#include "StatusWidget.h"

// Application includes
#include "FW/labImage.h"
#include "improv_plugin.h"
#include "improvConfig.h"
#include "imageSeq.h"
#include "ipPlugin.h"
#include "utils.h"

class Improv {
	friend class PluginMenu;
	friend class SeqWidget;
	friend class SrcWidget;

	public:
		/** Returns the instance of the class, or if one does not exist, creates an instance and then returns it.
		 * @param argc The number of command line arguments.
		 * @param argv The command line arguments.
		 */
		static Improv* Instance(int argc, char **argv);

		/** Destructor. Deletes the instance of the class. */
		~Improv();
	
		/** Check if the sequence is currently playing.
		 * @return True if the sequence is playing, false otherwise.
		 * @see setPlaying
		 */
		bool isPlaying(void);

		/** Check if the application is ready to play.
		 * @return True if the all required widgets and 
		 * variables have been initialised, false otherwise.
		 */
		bool isReady(void);

		/** Start/stop the sequence from playing.
		 * @param opt True to begin playback, false to stop it.
		 * @see isPlaying
		 */
		void setPlaying(bool opt);

		/** Add a source widget to the application.
		 * The source widgets provide the animation sequence or camera 
		 * images, which are then processed by the sequence widgets.
		 * @see getSrcWidget
		 */
		void addSrcWidget(SrcWidget* sw);

		/** Return a pointer to the requested source widget.
		 * @return Pointer to the requested source widget.
		 * @param sourceId Id of the source widget.
		 * @see addSrcWidget
		 */
		SrcWidget* getSrcWidget(int sourceId);

		/** Return the number of widgets at a given position, with a given sourceId.
		 * In the application, all source widgets are at position 0 and sequence widgets are at subsequent positions.
		 * @param sourceId Id of the processing thread that the widget belongs to. -1 signifies all threads.
		 * @param position Position of the widget with in that thread.
		 * @return The number of widgets.
		 */
		int getWidgetCount(int sourceId, int position);
		
		/** Return the doubly linked list containing all the source widgets.
		 * @return The list of source widgets.
		 */
		QValueList<SrcWidget*> getSrcWidgetList(void);

		/** Add a sequence widget to the list
		 * This class keeps track of all the sequence widgets in the application
		 * through this function. Once a sequence is added it becomes available
		 * to display the result of image processing operations.
		 * @param sw pointer to SeqWidget to add
		 * @see SeqWidget
		 */
		void addSeqWidget(SeqWidget* sw);

		/** Remove a sequence widget from the list
		 * This class keeps track of all the sequence widgets in the application
		 * through this function.
		 * @param sw pointer to SeqWidget to remove
		 * @see SeqWidget
		 */
		//void removeSeqWidget(SeqWidget* sw);
		
		/** */
		SeqWidget* getSeqWidget(int id);

		/** */
		SeqWidget* getSeqWidget(int sourceId, int position, int number=0);

		/** Return the doubly linked list containing all the sequence widgets
		 * @return the list of sequence widgets
		 */
		QValueList<SeqWidget*> getSeqWidgetList(void);
		
		/** Return a pointer to the plugin menu.
		 * Only one copy of the plugin menu is instantiated.  It is tracked by
		 * this class, and can be referenced through this function
		 * @return pointer to the popup menu
		 */
		QPopupMenu* getPluginMenu(void);	

		/** Set the frame rate of playback.
		 * Sets the desired rate of playback.  If possible, the source and sequence
		 * widgets will update synchronously to this rate unless the CPU is overloaded,
		 * in which case it will update as fast as possible.
		 * @param fps desired frame rate
		 */
		void setFPS(int fps);

		/** Toggle unlimited frames per second.
		 * @param toggle
		 */
		void setUnlimitedFPS(bool toggle);

		/** Copies a picture to the internal buffers of all the app's sequence widgets
		 * When the source image is changed, this clears all the sequence widget internal
		 * buffers which are displayed and replaces them with one of matching format and
		 * attributes of the source picture.
		 * @param pic pointer to source picture
		 */
		void copyPictures(Picture *pic);

		/** Iterates through one cycle of animation/processing/display
		 * @param frames number of frames to jump (default is 1)
		 */
		void tick(int frames=1);
		
		/** Perform image processing operation on a picture, given a selected menu Id
		 * @param in input pictures
		 * @param out output picture
		 * @param menuId menu Id selected
		 * @param inCount number of input pictures
		 * @param bufferId Id of the frame buffer to use (for functions that 
		 * require multiple images). Default is -1, meaning no buffer is used.
		 * @param params array of floating point parameters
		 */
		void process(Picture** in, Picture* out, int menuId, int inCount=1, int bufferId=-1, float* params = NULL);

		/** */
		int getPluginUniqueId(int id);
		
		/** Return the name of the plugin given a menu id
		 *  @param id menu id of plugin
		 *  @return string of plugin name
		 */
		QString* getPluginName(int id);
		
		/** */
		int getPluginInputCount(int id);
		
		/** Return the number of input input paramters of the plugin operation 
		 *  given a menu id
		 *  @param id menu id of plugin
		 *  @return string of plugin name
		 */
		int getPluginNumParams(int id);
		
		/** */
		QStringList* getPluginParamNames(int id);

		/** */
		void activateParamWidget(int numParams, QStringList* paramNames, float* params);
		/** */
		void setParamWidget(ParamWidget* pw);
		/** */
		void disableParamWidget(void);

		/** */
		void setStatusBar(QStatusBar* statusBar);
		/** */
		bool statusBarReady(void);
		/** */
		void setStatusMessage(const char* msg);
		/** */
		void setStatusMessage(const char* msg, int ms);
		/** */
		void setStatusFPSMessage(const char* msg);
		/** */
		void setStatusResultMessage(const char* msg);
		/** */
		const char *getStatusMessage(void);

		/** */
		void constructPluginMenu(void);

		/** */
		int getNumBuffers(void);
		/** */
		void addFrameBuffer(int numFrames);
		/** */
		void removeFrameBuffer(int bufferId, int menuId);

		/** */
		void setCamOptAction(QAction *action);
		/** */
		void setCamOpt(ImprovQTCamOpt *options);
		/** */
		ImprovQTCamOpt *getCamOpt(void);
		/** */
		void setCamOptOpen(bool open);
		/** */
		bool camOptIsOpen(void);
		/** */
		void updateCamOpt(imageSeqType type, Camera *camera);

		/** */
		void clearSeqSelections(void);

		/** */
		improvConfig* config;

	protected:
		/** */
		Improv(int argc, char **argv);
		/** */
		void setMenuId(int id);
		/** */
		int getMenuId(void);
		/** */
		int getMenuId(int uniqueId, char *functionName);

	private:
		/** Constructor. */
		static Improv* _instance;
		/** */
		AnimThread* animThread;
		/** */
		bool playing;
		/** */
		QValueList<Plugin*> pluginList;
		/** */
		PluginMenu* pluginMenu;

		/** The doubly linked list of Source widgets.
		 * The list is kept in order of widget id. 
		 */
		QValueList<SrcWidget*> srcWidgetList;
		/** The doubly linked list of Source widgets.
		 * The list is kept in order of widget id. 
		 */
		QValueList<SeqWidget*> seqWidgetList;

		/** */
		ParamWidget* paramWidget;
		
		/** */
		QStatusBar* statusBar;
		//StatusWidget *statusBar;
		/** */
		const char *statusBarMsg;
		/** */
		const char *fpsMsg;
		/** */
		const char *resultMsg;

		// keeps track of how many plugin instances there are
		//int numberPluginItems;

		/** Frame buffer vector for plugin instances */
		QValueList<Picture**> frameBuffer;

		//  Current state
		/** */
		int seqId;
		/** */
		int menuId;

		/** */
		ImprovQTCamOpt *camOptions;
		/** */
		QAction *camOptionAction;
		/** */
		bool camOptOpen;

		/** */
		static int seq_file_select(const struct dirent *entry);
		/** */
		static int plugin_file_select(const struct dirent *entry);

};

#endif
