/* ParamWidget.cpp
 * Last modified:
 * Authors: Daniel Venkitachalam <venki-d@ee.uwa.edu.au>
 *          Leon Koch <leon@redfishsoftware.com.au>
 */

#include "Improv.h"

ParamWidget::ParamWidget(QWidget* parent, const char* name, int minimumWidth)
	: QWidget(parent, name)
{
	nameBox = new QComboBox(this);
	QWhatsThis::add(nameBox,tr("Parameter Name\n\nName of the parameter to alter"));
	valueSlider = new QSlider(0, MAX_VALUE, MAX_VALUE/100, 0, Qt::Horizontal, this, "parameter slider");
	valueSlider->setMinimumWidth(minimumWidth);
	QWhatsThis::add(valueSlider,tr("Parameter Slider\n\nSlider used to alter the parameter value"));
	valueLabel = new QLabel(this);
	valueLabel->setNum(0.0);
	valueLabel->setFixedWidth(40);
	QWhatsThis::add(valueLabel,tr("Paramter Value\n\nCurrent value of the parameter (0<t<1)"));
	
	layout = new QHBoxLayout(this);
	layout->setSpacing(15);
	layout->addWidget(nameBox);
	layout->addWidget(valueSlider);
	layout->addWidget(valueLabel);

	connect(nameBox, SIGNAL(activated(int)), this, SLOT(selectParam(int)));
	connect(valueSlider, SIGNAL(valueChanged(int)), this, SLOT(paramChanged(int)));
	this->setEnabled(false);
}
		
ParamWidget::~ParamWidget() {
	// no need to delete child widgets, Qt does it all for us
}
		
void ParamWidget::setPluginItem(PluginItem* pi)
{
	assert(pi);

	nameBox->clear();
}

void ParamWidget::activateParams(int numParams, QStringList* paramNames, 
		float* params)
{
	this->setEnabled(true);
	nameBox->clear();
	nameBox->insertStringList(*paramNames);
	this->params = params;
	valueSlider->setValue((int)(params[0]*MAX_VALUE));
}

void ParamWidget::disable(void)
{
	nameBox->clear();
	this->setEnabled(false);
}
		
void ParamWidget::selectParam(int index)
{
	valueSlider->setValue((int)(params[index]*MAX_VALUE));
	valueLabel->setNum((double)params[index]);
}

void ParamWidget::paramChanged(int value)
{
	Improv* improv;
	QValueList<SeqWidget*> seqWidgetList;
	QValueList<SeqWidget*>::iterator it;
	int index = nameBox->currentItem();
	params[index] = ((float)value / MAX_VALUE);
	updateCounter++;
	if(updateCounter==REFRESH_INTERVAL) {
		updateCounter=0;
		improv = Improv::Instance(0, NULL);
		seqWidgetList = improv->getSeqWidgetList();
		for(it = seqWidgetList.begin(); it != seqWidgetList.end(); ++it) {
			(*it)->redraw();
		}
	}
	valueLabel->setNum((double) params[index]);
}
