/* SeqWidget.cpp
 * Last modified:
 * Authors: Daniel Venkitachalam <venki-d@ee.uwa.edu.au>
 *          Leon Koch <leon@redfishsoftware.com.au>
 */

#include "Improv.h"

// #include "up.xpm"
// #include "zoom.xpm"

SeqWidget::SeqWidget( QWidget *parent, const char *name, int id, int sourceId, int position)
	: QWidget( parent, name )
{
	int width, buttonWidth;
	Improv* improv = Improv::Instance(0, NULL);

	assert(id>0);
	assert(position>0);

	width = improv->config->getWindowWidth();
	if(width<MINIMUM_BUTTON_BAR_WIDTH) {
		width = MINIMUM_BUTTON_BAR_WIDTH;
	}
	buttonWidth = (int)floor((double)width/(double)9);
		
	pixWidget    = new PixWidget(this, improv->config->getWindowWidth(), improv->config->getWindowHeight() );
	hbox         = new QHBox(this);
	popupButton  = new QPushButton("Add", this->hbox, "add");
	popupButton->setFixedWidth(buttonWidth*3);
	QWhatsThis::add(popupButton,tr("Add a function"));
	rmButton	= new QPushButton("Remove", this->hbox, "remove");
	rmButton->setFixedWidth(buttonWidth*3);
	QWhatsThis::add(rmButton,tr("Remove a function"));
	upButton	= new QPushButton("<", this->hbox, "up");
	upButton->setFixedWidth(buttonWidth);
	QWhatsThis::add(upButton,tr("Move selected function up the list"));
	downButton	= new QPushButton(">", this->hbox, "down");
	downButton->setFixedWidth(buttonWidth);
	QWhatsThis::add(downButton,tr("Move selected function down the list"));
//	zoomPixmap   = new QPixmap(zoom_xpm);
//	zoomButton   = new QPushButton(*zoomPixmap, " ", this->hbox, "zoom");
	zoomButton   = new QPushButton("Z", this->hbox, "zoom");
	zoomButton->setFixedWidth(buttonWidth);
	zoomButton->setToggleButton(true);
	QWhatsThis::add(zoomButton,tr("Toggle auto zoom"));
	
	hbox->setSpacing(0);
	//hbox->setMinimumWidth( + 1);
	hbox->setMinimumWidth(width+1);	

	popupButton->setFlat(true);
	rmButton->setEnabled(false);
	rmButton->setFlat(true);
	upButton->setFlat(true);
	downButton->setFlat(true);
	zoomButton->setFlat(true);

	chainListBox = new QListBox(this);
	//chainListBox->setFixedHeight(70);
	chainListBox->setMinimumHeight(70);
	QWhatsThis::add(chainListBox,tr("List of functions to apply"));
	this->id = id;
	this->sourceId = sourceId;
	this->position = position;
	this->picture = NULL;
	this->active = false;

	layout = new QVBoxLayout(this);
	layout->setSpacing(0);
	layout->addWidget(pixWidget);
	layout->addWidget(hbox);
	layout->addWidget(chainListBox);

	if(improv->getPluginMenu() == NULL) {
		popupButton->setPopup((new PluginMenu())->getMainMenu());
		popupButton->setEnabled(false);
	}
	else {
		popupButton->setPopup(improv->getPluginMenu());
	}
	connect(popupButton, SIGNAL(pressed()), this, SLOT(popupPressed()));
	connect(rmButton, SIGNAL(clicked()), this, SLOT(rmPressed()));
	connect(upButton, SIGNAL(clicked()), this, SLOT(upPressed()));
	connect(downButton, SIGNAL(clicked()), this, SLOT(downPressed()));
	connect(chainListBox, SIGNAL(clicked(QListBoxItem*)), this, SLOT(itemHighlighted(QListBoxItem*)));
	connect(zoomButton, SIGNAL(toggled(bool)), this, SLOT(zoomToggled(bool)));
	pixWidget->clear();
}

SeqWidget::~SeqWidget() 
{ 
	if(picture)
		freeImage(picture);
}

void SeqWidget::reload(void) {
	Improv* improv = Improv::Instance(0, NULL);
	//printf("reloading at w: %d h: %d\n", improv->config->getWindowWidth(), improv->config->getWindowHeight());
	pixWidget->resize(improv->config->getWindowWidth(),improv->config->getWindowHeight());
	pixWidget->setFixedWidth(improv->config->getWindowWidth());
	pixWidget->setFixedHeight(improv->config->getWindowHeight());
	pixWidget->clear();
	picture = NULL;
	redraw();
}

bool SeqWidget::addFunction(int menuId) {
	int bufferId=-1;
	QValueList<SeqWidget*> seqWidgetList;
	QValueList<SeqWidget*>::iterator it;
	Improv* improv = Improv::Instance(0, NULL);
	assert(menuId>=0);
	if(improv->getPluginInputCount(menuId)>1) {
		bufferId = improv->getNumBuffers();
	}
	PluginItem *new_item = new PluginItem((*(improv->getPluginName(menuId))), menuId,
		bufferId, improv->getPluginNumParams(menuId), improv->getPluginParamNames(menuId));

	if(bufferId>=0) improv->addFrameBuffer(menuId);
	printf("Op name: %s, num params: %d\n",(improv->getPluginName(menuId))->simplifyWhiteSpace().latin1(), new_item->getNumParams());
	chainListBox->insertItem(new_item, -1);
	improv->clearSeqSelections();
	chainListBox->setSelected(new_item, true);
	chainListBox->clearSelection();
	this->itemHighlighted(new_item);
	this->setActive(true);
	rmButton->setEnabled(true);
	this->redraw();
	/* redraw the rest of the seq widgets */
	seqWidgetList = improv->getSeqWidgetList();
	for(it = seqWidgetList.begin(); it != seqWidgetList.end(); ++it) {
		if((*it)->getPosition()>this->position) (*it)->redraw();
	}
	return true;
}

bool SeqWidget::addFunction(int uniqueId, char *functionName) {
	int menuId;
	Improv* improv = Improv::Instance(0, NULL);
	menuId = improv->getMenuId(uniqueId,functionName);
	if(menuId>=0) {
		return addFunction(menuId);
	}
	else return false;
}
		
void SeqWidget::popupPressed()
{
	int menuId;
	Improv* improv = Improv::Instance(0, NULL);
	
	menuId = improv->getMenuId();

	if(menuId>=0) {
		//printf("Menu Id %d, seq id %d\n", menuId, this->id);
		addFunction(menuId);		
	}
	improv->setMenuId(-1);
}

void SeqWidget::redraw(void) {
	if(active) {
		this->tick();
	}
	else {
		pixWidget->clear();
	}
}

void SeqWidget::rmPressed()
{
	Improv* improv = Improv::Instance(0, NULL);
	QValueList<SeqWidget*> seqWidgetList;
	QValueList<SeqWidget*>::iterator it;
	PluginItem *item;
	int position;

	improv->disableParamWidget();
	position = chainListBox->currentItem();
	if(position != -1) {
		item = (PluginItem *)chainListBox->item(position);
		improv->removeFrameBuffer(item->getBufferId(),item->getMenuId());
		chainListBox->removeItem(position);
	}
  improv->setStatusResultMessage(" ");
	if(chainListBox->count() == 0) {
		this->setActive(false);	
		rmButton->setEnabled(false);	
	}
	else {
		this->redraw();
	}
	/* redraw the rest of the seq widgets */
	seqWidgetList = improv->getSeqWidgetList();
	for(it = seqWidgetList.begin(); it != seqWidgetList.end(); ++it) {
		if((*it)->getPosition()>this->position) (*it)->redraw();
	}
}

void SeqWidget::upPressed()
{
	Improv* improv = Improv::Instance(0, NULL);
	QListBoxItem* current;
	QValueList<SeqWidget*> seqWidgetList;
	QValueList<SeqWidget*>::iterator it;
	
	int index = chainListBox->currentItem();
	if(index > 0) {
		current = chainListBox->item(index);
		chainListBox->setSelected(index, false);
		chainListBox->takeItem(current);
		chainListBox->insertItem(current, index-1);
		improv->clearSeqSelections();
		chainListBox->setSelected(index-1, true);
		this->redraw();
		/* redraw the rest of the seq widgets */
		seqWidgetList = improv->getSeqWidgetList();
		for(it = seqWidgetList.begin(); it != seqWidgetList.end(); ++it) {
			if((*it)->getPosition()>this->position) (*it)->redraw();
		}
	}
}

void SeqWidget::downPressed()
{
	Improv* improv = Improv::Instance(0, NULL);
	QListBoxItem* current;
	QValueList<SeqWidget*> seqWidgetList;
	QValueList<SeqWidget*>::iterator it;
		
	int index = chainListBox->currentItem();
	int count = chainListBox->count() - 1;
	if(index < count) {
		current = chainListBox->item(index);
		chainListBox->setSelected(index, false);
		chainListBox->takeItem(current);
		chainListBox->insertItem(current, index+1);
		improv->clearSeqSelections();
		chainListBox->setSelected(index+1, true);
		this->redraw();
		/* redraw the rest of the seq widgets */
		seqWidgetList = improv->getSeqWidgetList();
		for(it = seqWidgetList.begin(); it != seqWidgetList.end(); ++it) {
			if((*it)->getPosition()>this->position) (*it)->redraw();
		}
	}
}

void SeqWidget::copyPicture(Picture *pic)
{
	assert(pic);
	if(picture) freeImage(picture);
	// printf("Original picture attributes: width %d height %d\n", pic->width,
	//	pic->height);
	this->picture = copyImage(pic);
	assert(picture);
	//	printf("New picture attributes: width %d height %d\n", picture->width,
	//			picture->height);
}

void SeqWidget::tick()
{
	int loaded=0;
	int maxDataSize=0;
	int mdsIndex=0;
	bool useMBuffer=false;
	Picture *p_in=NULL;
	Picture **multiBuffer=NULL;
	Improv* improv = Improv::Instance(0, NULL);
	PluginItem *pi;
	SeqWidget *seqWidget;
	SrcWidget *srcWidget;

	if(active == true) {

		pi = (PluginItem *) chainListBox->item(0);

		// if there is only one input image source
		if(improv->getWidgetCount(sourceId,position-1)==1) {
			if(position==1) {p_in = improv->getSrcWidget(sourceId)->getCurrent();}
			else {p_in = improv->getSeqWidget(sourceId,position-1)->getCurrent();}
			// check the input images
			if(p_in==NULL) {pixWidget->clear();pixWidget->redraw();return;}
			if(p_in->data==NULL) {pixWidget->clear();pixWidget->redraw();return;}
			// check if picture is the right format/size
			if(picture) {
				if((picture->format!=p_in->format)||
				   (picture->width!=p_in->width)||
				   (picture->height!=p_in->height)) {
					freeImage(picture);
					picture = copyImage(p_in);
				}
			}
			else picture = copyImage(p_in);
			improv->process(&p_in, picture, pi->getMenuId(), 1, pi->getBufferId(), pi->getParams());
		}
		else {
			useMBuffer = true;
			multiBuffer = (Picture **)malloc(sizeof(Picture *)*improv->getWidgetCount(sourceId,position-1));
			if(position==1) {
				// get the image from each of the available src widgets
				for(int i=0;i<improv->config->getNumSources();i++) {
					srcWidget = improv->getSrcWidget(i);
					if(srcWidget->isReady()) {
						multiBuffer[loaded] = srcWidget->getCurrent();
						if(srcWidget->getCurrent()!=NULL) {
							loaded++;
							if(srcWidget->getCurrent()->datasize>maxDataSize) {
								maxDataSize = srcWidget->getCurrent()->datasize;
								mdsIndex = loaded-1;
							}
						}
					}
				}
			}
			else {
				// get the image from each of the available seq widgets
				for(int i=0;i<improv->getWidgetCount(sourceId,position-1);i++) {
					seqWidget = improv->getSeqWidget(sourceId,position-1,i);
					if(seqWidget->isActive()) {
						multiBuffer[loaded] = seqWidget->getCurrent();
						if(seqWidget->getCurrent()!=NULL) {
							loaded++;
							if(seqWidget->getCurrent()->datasize>maxDataSize) {
								maxDataSize = seqWidget->getCurrent()->datasize;
								mdsIndex = loaded-1;
							}
						}
					}
				}
			}
			if(loaded==0) {pixWidget->clear();pixWidget->redraw();return;}
			// copy the biggest image.
			p_in = multiBuffer[mdsIndex];
			// ensure that the output is big enough to handle the largest input image type.
			if(picture) {
				if((picture->format!=p_in->format)||
				   (picture->width!=p_in->width)||
				   (picture->height!=p_in->height)) {
					freeImage(picture);
					picture = copyImage(p_in);
				}
			}
			else picture = copyImage(p_in);
			improv->process(multiBuffer, picture, pi->getMenuId(), loaded, pi->getBufferId(), pi->getParams());
		}

		qApp->lock();

		for(unsigned int i=1; i<chainListBox->count(); i++) {
			Picture *buffer;
			buffer = copyImage(picture);
			pi = (PluginItem *) chainListBox->item(i);
			qApp->unlock();
			improv->process(&picture, buffer, pi->getMenuId(), 1, pi->getBufferId(), pi->getParams());
			freeImage(picture);
			picture = buffer;
			qApp->lock();
		}

		qApp->unlock();

		if(picture) {
			pixWidget->setPicture(picture);
			pixWidget->redraw();
		}
	}
}

void SeqWidget::setActive(bool value)
{
	this->active = value;
	if(active == false) {
		pixWidget->clear();
		picture=NULL;
	}
}

bool SeqWidget::isActive(void)
{
	return this->active;
}

Picture* SeqWidget::getCurrent(void)
{
	//assert(this->picture);
	return(picture);
}
		
void SeqWidget::itemHighlighted(QListBoxItem* item)
{
	Improv* improv = Improv::Instance(0, NULL);
	PluginItem* pi = (PluginItem*) item;

	// printf("itemHighlighted: Num params selected: %d\n", pi->getNumParams());

	//  Unhighlight other selected sequence items
	improv->clearSeqSelections();
	chainListBox->setSelected(item, true);

	if(!pi) return;

	if(pi->getNumParams())
		improv->activateParamWidget(pi->getNumParams(), pi->getParamNames(), pi->getParams());
	else
		improv->disableParamWidget();
}
		
int SeqWidget::getId(void)
{
	return this->id;
}

int SeqWidget::getPosition(void) {
	return this->position;
}

void SeqWidget::setPosition(int value) {
	/**  @todo check that the position is with in the range */
	this->position = value;
}

int SeqWidget::getSourceId(void) {
	return this->sourceId;
}

void SeqWidget::setSourceId(int value) {
	/** @todo check that value points to a valid source */
	this->sourceId = value;
}
		
void SeqWidget::zoomToggled(bool on)
{
	if(on == true)
		pixWidget->setScale(true);
	else
		pixWidget->setScale(false);
	if(picture!=NULL) {
		pixWidget->setPicture(picture);
		pixWidget->redraw();
	}
}

void SeqWidget::disablePopUp(void)
{
	popupButton->setEnabled(false);
}

void SeqWidget::setPopUpMenu(QPopupMenu* menu)
{
	assert(menu);
	popupButton->setPopup(menu);
	popupButton->setIsMenuButton(true);
	popupButton->setEnabled(true);
}

void SeqWidget::clearSelection(void) {
	chainListBox->clearSelection();
}

int SeqWidget::numPluginItems(void) {
	return chainListBox->count();
}

PluginItem *SeqWidget::getPluginItem(int posn) {
	assert(posn>=0&&posn<(int)chainListBox->count());
	return (PluginItem*)chainListBox->item(posn);
}

PluginItem **SeqWidget::getPluginItems(void) {
	PluginItem **pluginItems;
	pluginItems = (PluginItem**)malloc(sizeof(PluginItem*)*chainListBox->count());
	for(int i=0;i<(int)chainListBox->count();i++) {
		pluginItems[i] = (PluginItem*)chainListBox->item(i);
	}
	return pluginItems;
}

////////////////////////////////////////////////////////////////////
// PluginItem
////////////////////////////////////////////////////////////////////
		
PluginItem::PluginItem(const QString& str, int menuId, int bufferId,
		int numParams, QStringList* paramNames)
	: QListBoxText(str.simplifyWhiteSpace())
{
	this->menuId = menuId;
	this->bufferId = bufferId;
	this->numParams = numParams;
	if(numParams > 0) {
		this->params = (float *) calloc(numParams, sizeof(float));
		assert(paramNames);
		this->paramNames = paramNames;
	
		for(int i=0; i<numParams; i++)
			this->params[i] = 0.5;
	}
	else {
		this->params = 0;
		this->paramNames = 0;
	}

}

PluginItem::~PluginItem()
{ 
	if(this->params)
		free(this->params);
}

int PluginItem::getMenuId(void)
{
	return this->menuId;
}

int PluginItem::getBufferId(void)
{
	return this->bufferId;
}

int PluginItem::getNumParams(void)
{
	return this->numParams;
}

float* PluginItem::getParams(void)
{
	return this->params;
}

void PluginItem::setParams(float *params) {
	this->params = params;
}

QStringList* PluginItem::getParamNames(void)
{
	return this->paramNames;
}
		
/*void PluginItem::setParam(int param, float value)
{
	assert(param > 0 && param < (int) this->numParams);
	assert(value >= 0 && value <= 1.0);

	params[param] = value;
} */
