/* V4LCamera.cpp
 * Last modified:
 * Authors: Daniel Venkitachalam <venki-d@ee.uwa.edu.au>
 *          Leon Koch <leon@redfishsoftware.com.au>
 */

#include "V4LCamera.h"

#if defined(HAVE_V4L)

#define USE_MMAP 1
#define LARGE 1
#define MY_BLACK 0
#define MY_WHITE 255

#if !LARGE
unsigned int MY_COLUMNS =88;
unsigned int MY_ROWS =72;
#else
unsigned int MY_COLUMNS = 176; // 147 320
unsigned int MY_ROWS = 144; // 126 240
#endif

//#define MY_TYPE pix_bgr32

#define IN_TV		0
#define IN_COMPOSITE	1
#define IN_COMPOSITE2	2
#define IN_SVIDEO	3
#define IN_DEFAULT	8

#define NORM_PAL	0
#define NORM_NTSC	1
#define NORM_SECAM	2
#define NORM_DEFAULT	0

BYTE* V4LCamera::set_up_mmap (int dev, int width, int height, int fmt)
{
	BYTE *map;

	if (ioctl (dev, VIDIOCGMBUF, &vid_mbuf) == -1) {
		fprintf(stderr, "\nSorry, can't use memory map\n");
		return(NULL);
	}

	map = (BYTE*)mmap (0, vid_mbuf.size, PROT_READ|PROT_WRITE,MAP_SHARED,dev,0);
	if ((unsigned char *)-1 == (unsigned char *)map) {
		perror ("mmap()");
		return (NULL);
	}

	vid_mmap.format = fmt;
	vid_mmap.frame = 0;
	vid_mmap.width = width;
	vid_mmap.height = height;

	if (ioctl (dev, VIDIOCMCAPTURE, &vid_mmap) == -1) {
		perror ("VIDIOCMCAPTURE");
		munmap (map, vid_mbuf.size);
		return (NULL);
	}

	return (map);
}

void V4LCamera::mmap_read(Picture *p_frame, int expected_size)
{
	if (ioctl (vid_fd, VIDIOCSYNC, &vid_mmap) == -1) {
		perror ("VIDIOCSYNC");
		munmap (video_map, vid_mbuf.size);
		return;
	}

	//memcpy(p_frame->data, video_map, expected_size);

	p_frame->data=video_map;

	if (ioctl (vid_fd, VIDIOCMCAPTURE, &vid_mmap) == -1) {
		perror ("VIDIOCMCAPTURE");
		munmap (video_map, vid_mbuf.size);
		return;
	}
}

void V4LCamera::read(Picture *p_frame)
{
	int expected_size;

	expected_size = p_frame->datasize;

	if(p_frame->data == NULL) {
		p_frame->data = (BYTE*) malloc (sizeof(BYTE) * expected_size);
		//fprintf(stdout,"That thing that should not be!\n");
		if(p_frame->data == NULL) {
			fprintf(stderr,"Error: No memory available for V4L picture\n");
			return;
		}
	}

#ifdef USE_MMAP
	mmap_read(p_frame, expected_size);
#else
	size = ::read(vid_fd, p_frame->data, expected_size);

	if ((unsigned) size != expected_size) {
		std::cerr <<"V4L: read() returned " << size << " not "
			<< expected_size << " bytes" << std::endl;
		return;
	}
#endif
}

bool V4LCamera::open(void)
{
	struct video_window vid_win;
	const char *device = strdup("/dev/video0");

//	if(( !device )||(strcmp(device,"")==0))
//		device = strdup("/dev/video0");

	int temp_fd = vid_fd;

	vid_fd = ::open(device, O_RDWR);
	if (0 > vid_fd) {
		if (errno != EBADF) {
			std::cerr << "V4L: Open(" << device << ") failed: " << strerror(errno) << std::endl;
			return false;
		}
		else 
			vid_fd = temp_fd;
	}

	if (0 != ioctl(vid_fd, VIDIOCGCAP, &vid_cap)) {
		std::cerr << "V4L: Device " << device << " is not a video4linux capture device. (Failed ioctl)" << std::endl;
		::close(vid_fd);
		return false;
	}

	if (!(vid_cap.type & VID_TYPE_CAPTURE)) {
		std::cerr << "V4L: Device " << device << " is not a video capture device."	<< std::endl;
		::close(vid_fd);
		return false;
	}

	if (0 != ioctl (vid_fd, VIDIOCGPICT, &vid_pic)) {
		std::cerr << "V4L: Device " << device << " VIDIOCGPICT failed: " <<
			strerror(errno) << std::endl;
		::close(vid_fd);
		return false;
	}

	if (vid_cap.type & VID_TYPE_MONOCHROME) {
		greyscale = true;
		palette = VIDEO_PALETTE_GREY;
		vid_pic.depth = 8;
	}
	else {
		greyscale = false;
		palette = VIDEO_PALETTE_RGB32;
		vid_pic.depth = 32;
	}

	vid_pic.palette = palette;

	if(0 != ioctl (vid_fd, VIDIOCSPICT, &vid_pic)) {
		std::cerr << "V4L: Device " << device << " VIDIOCSPICT sfailed: " << strerror(errno) << std::endl;
		::close(vid_fd);

		return false;
	}

	if (0 != ioctl (vid_fd, VIDIOCGWIN, &vid_win)) {
		std::cerr << "V4L: Device " << device << " VIDIOCGWIN failed: " << strerror(errno) << std::endl;
		::close(vid_fd);
		return false;
	}

	vid_win.x=0;
	vid_win.y=0;
	vid_win.width=MY_COLUMNS;
	vid_win.height=MY_ROWS;
//	vid_win.width=vid_cap.maxwidth;
//	vid_win.height=vid_cap.maxheight;
//	fprintf(stderr,"w: %d h: %d\n", vid_cap.maxwidth, vid_cap.maxheight);

	if (0 != ioctl (vid_fd, VIDIOCSWIN, &vid_win)) {
		std::cerr << "v4l: Device " << device << " VIDIOCSWIN failed: " <<
			strerror(errno) << std::endl;
		::close(vid_fd);
		return false;
	}

	/* check to see that image dimension are possible */
	if (0 != ioctl (vid_fd, VIDIOCGWIN, &vid_win)) {
		std::cerr << "v4l: Device " << device << " VIDIOCGWIN failed: " <<
			strerror(errno) << std::endl;
		::close(vid_fd);
		return false;
	}

	if ( (vid_win.width != MY_COLUMNS) || (vid_win.height != MY_ROWS) )
		fprintf(stderr, "\nWarning: The specified image dimensions are not possible. The driver has decided you want image dimensions of %dX%d.\n"
				, vid_win.width, vid_win.height);

	width = vid_win.width;
	height = vid_win.height;
	input = IN_DEFAULT;
	norm = NORM_DEFAULT;

#ifdef USE_MMAP
	video_map = set_up_mmap (vid_fd, width, height, palette);
	if (video_map == NULL)
		return(false);
#endif

	return true;

}

void V4LCamera::setBrightness(int value)
{
	vid_pic.brightness = value;
	if (ioctl (vid_fd, VIDIOCSPICT, &vid_pic) == -1)
		perror ("ioctl (VIDIOCSPICT) - unable to set brightness\n");
}

void V4LCamera::setContrast(int value)
{
	vid_pic.contrast = value;
	if (ioctl (vid_fd, VIDIOCSPICT, &vid_pic) == -1)
		perror ("ioctl (VIDIOCSPICT) - unable to set contrast\n");
}

void V4LCamera::setSaturation(int value) {
	vid_pic.contrast = value;
	if (ioctl (vid_fd, VIDIOCSPICT, &vid_pic) == -1)
		perror ("ioctl (VIDIOCSPICT) - unable to set saturation\n");
}

void V4LCamera::setAutoBrightness(bool on) {
	/** @todo The Zoom cam we have doesn't seem to respond to this */
}

void V4LCamera::set_colour(int value)
{
	vid_pic.colour = value;
	if (ioctl (vid_fd, VIDIOCSPICT, &vid_pic) == -1)
		perror ("ioctl (VIDIOCSPICT) - unable to set colour\n");
}

int V4LCamera::getBrightness(void) {
	return vid_pic.brightness;
}

int V4LCamera::getContrast(void) {
	return vid_pic.contrast;
}

int V4LCamera::getSaturation(void) {
	return vid_pic.contrast;
}

bool V4LCamera::getAutoBrightness(void) {
	/** @todo The Zoom cam we have doesn't seem to respond to this */
	return false;
}

void V4LCamera::close(void)
{
#ifdef USE_MMAP
	munmap (video_map, vid_mbuf.size);
#endif
	if (-1 != vid_fd)
		::close(vid_fd);
	vid_fd= -1;
}

bool V4LCamera::iscolor(void)
{
	if (vid_cap.type & VID_TYPE_MONOCHROME)
		return(false);

	else
		return(true);
}

void V4LCamera::get_info(CamInfo *info)
{
	info->width = width; 
	info->height = height;
	if (iscolor() == true)
		info->bpp = 24;
	else
		info->bpp = 8; /* for the quickcam on xena(Thu Jul 13 14:39:26 WST 2000)*/

}

#endif /* HAVE_V4L */
