/*
 *   Adapted for ImprovQT 5.0 by Leon Koch (2002)
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>

#include "../FW/labImage.h"
#include "../improv_plugin.h"
#include "feat_track_plugin.h"

#define NUMBER_OF_OPERATIONS 4

IP_Descriptor ipl_plugin =
{
	4,
	"Image Processing Library",
	"Thomas Braunl, Univ. Stuttgart",
	"1996",
	"Additional code by Gerrit Heitsch etc.\nAdapted for ImprovQT by Leon Koch\n",
	NUMBER_OF_OPERATIONS,
	NULL              /*  IP_ops, filled out by the _init function  */
};

IP_Handle IP_Init()
{
	IP_Op *op;
	int i = 0;

	/*  Fill out the IP ops descriptions  */
	ipl_plugin.Ops = calloc(NUMBER_OF_OPERATIONS, sizeof(IP_Op));

	/* Gaussian Smoothing */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Advanced");
	op->Operation  = strdup("Gaussian Smooth");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;	
	op->ParamNames = NULL;
	//op->ParamCount = 2;
	//op->ParamNames = calloc(2, sizeof(char *));
	//op->ParamNames[0] = strdup("Size");
	//op->ParamNames[1] = strdup("Variance");
	op->resultType = NORESULT;
	op->process    = IPL_Gauss;

	/* Histogram Equallization */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Advanced");
	op->Operation  = strdup("Histogram Equallization");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;	
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_HistEq;

	/*  Moravec  */
	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Advanced");
	op->Operation  = strdup("Moravec");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_moravec;

	op = (IP_Op*) &(ipl_plugin.Ops[i++]);
	op->Category   = strdup("Advanced");
	op->Operation  = strdup("Track");
	op->Index      = i;
	op->InputCount = 1;
	op->ParamCount = 0;
	op->ParamNames = NULL;
	op->resultType = NORESULT;
	op->process    = IPL_Track;

	if( i != NUMBER_OF_OPERATIONS ) {
		printf("feat_track_plugin.c ERROR: Mismatched number of operations:\n" \
				" (i, NUMBER_OF_OPERATIONS) = (%d, %d)\n", i, NUMBER_OF_OPERATIONS);
		exit(1);
	}

	return NULL;
}


const IP_Descriptor* IP_Descriptor_get(unsigned long Index)
{
	return &ipl_plugin;
}

/*
 * Gaussian Smoothing
 *
 * TODO:
 *   - it would be nicer for a slider to control the size of
 *     the window.
 *   - it would be nicer for a slider to control the variance
 *     (amount of smoothing)
 */
pluginReturnType IPL_Gauss(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int size, g, var, width, height;
	BYTE *imageOut,*imageIn;

	Picture *p_imageIn = *p_imageInputs;
	width=p_imageIn->width;
	height=p_imageIn->height;

	size = 7;
	var = 2;

	//size = (int)(params[0]*MAXSIZE);
	g = (int)floor(size/2);
	//var = (int)(params[1]*MAXVAR);

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;

	if(p_imageIn->format != pix_grey) {
		//printf("Must be greyscale image!\n");
		return WRONGFORMAT;
	}

	memcpy(imageOut, imageIn, width * height);

	Gaussian_Smooth(imageIn, imageOut, g, g, width-g-1, height-g-1, width, height, size, g, var);
	return NOERROR;
}

pluginReturnType IPL_HistEq(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int width, height;
	BYTE *imageOut,*imageIn;

	Picture *p_imageIn = *p_imageInputs;
	width=p_imageIn->width;
	height=p_imageIn->height;

	imageIn=(BYTE *)p_imageIn->data;
	imageOut=(BYTE *)p_imageOut->data;

	if (p_imageIn->format == pix_rgb24)  /* using colour */
		Hist_Eq_Col(imageIn, imageOut, width, height);
	else {
		if (p_imageIn->format == pix_grey) {
    	Hist_Eq(imageIn, imageOut, width, height);
		}
		else {
			//fprintf(stderr,"\nHistogram equalization not implemented for this image format\n");
			return WRONGFORMAT;
		}
	}
	return NOERROR;
}

/*
 * Moravec's Interest Operator
 *
 * TODO:
 *   - The size of the window over which the operator is applied
 *     could be controlled with a slider.
 *   - Similarly, the size of the non-maximal suppression
 *     window could also be controlled by slider.
 *   - Currently, feature points are drawn ON the image - might
 *     want to fix.
 */

pluginReturnType IPL_moravec(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	int width, height, no_points=0;
	Picture *p_imageIn = *p_imageInputs;
	double *var;
	BYTE* imageIn = p_imageIn->data;
	BYTE* imageOut = p_imageOut->data;
	Point  point[MAX_POINTS];

	width=p_imageIn->width;
	height=p_imageIn->height;

	var = (double*)calloc( sizeof(double) * width * height, 1);

	if (p_imageIn->format != pix_grey) {
		//printf("Must be greyscale image!\n");
		return WRONGFORMAT;
	}

	interest(imageIn, M, M, width-M-1, height-M-1, var, width, height);

	non_max_suppress(var, SW, SW, width-SW-1, height-SW-1, point, &no_points, width, height);

	memcpy(imageOut, imageIn, width * height);

	draw_points(imageOut, point, no_points, width, height);

	free(var);

	return NOERROR;
}

/*
 * Tracks features detected by Moravec's interest operator.
 *
 * TODO:
 *   - feature points, the grid, and the crosshair are
 *     drawn to the image - might want to fix.
 */
pluginReturnType IPL_Track(IP_Handle instance, Picture **p_imageInputs, Picture *p_imageOut, float *params, void *result)
{
	Picture *p_imageIn = *p_imageInputs;
	BYTE* imageIn  = p_imageIn->data;
	BYTE* imageOut = p_imageOut->data;
	active grid;
	int width, height, ok;
	Point target;
	char *lum, *lum2;

	width=p_imageIn->width;
	height=p_imageIn->height;

	lum = (char *)malloc((width*height)*sizeof(char));
	lum2 = (char *)malloc((width*height)*sizeof(char));

	if (p_imageIn->format == pix_rgb24)  /* using colour */
    {
		memcpy(imageOut, imageIn, width * height * 3);
#ifdef HIST
		Hist_Eq_Col(imageIn, imageIn);
#endif
		IPLi_lum(p_imageIn, lum);
	}
	else if (p_imageIn->format == pix_grey)
    {
		memcpy(imageOut, imageIn, width * height);
#ifdef HIST
		Hist_Eq(imageIn, lum);
#else
		memcpy(lum, imageIn, width * height);
#endif
    }
	else
	{
		//fprintf(stderr, "Invalid image format. Must use rgb24 or 256 grey\n");
		return WRONGFORMAT;
	}
	if (first == TRUE)
    {
      grid.windows = malloc( sizeof(window)*(int)XX*(int)YY );

      generate_windows(&grid,width,height);

      Gaussian_Smooth(lum, lum2,
		      grid.main.x1-M, grid.main.y1-M,
		      grid.main.x2+M, grid.main.y2+M,
					width, height, 7, 3, 2);

      Detect_Prominent_Features(imageIn, lum2, &grid, width, height, 3);

#ifdef DISPLAY

      if (p_imageIn->format == pix_rgb24)
	draw_grid(grid.windows, imageOut, TRUE, width, height);
      else
	draw_grid(grid.windows, imageOut, FALSE, width, height);

#endif

      first = can_track();

      free(grid.windows);

    }

  else  /* first == FALSE */
    {

#ifdef MOTION
      copy_features();
#endif

      smooth_areas(lum, lum2, width, height, 3);

      ok = find_features(imageIn, lum2, width, height); /* features updated */

#ifdef MOTION

      if ( ok == TRUE )
	detect_motion();

#endif

    }

  direct(&target);

#ifdef DISPLAY

  if (p_imageIn->format == pix_rgb24)  /* using colour */
    {
      draw_features(imageOut, TRUE, width, height);
      draw_crosshair(imageOut, &target, TRUE, width, height);
    }

  else
    {
      draw_features(imageOut, FALSE, width, height);
      draw_crosshair(imageOut, &target, FALSE, width, height);
    }

#endif

	return NOERROR;
}

/** Internal Funtions **/

/*
 * Calculates the luminance of a rgb image
 */
void IPLi_lum(Picture *p_imageIn, BYTE* lum)
{

  BYTE* imageIn = p_imageIn->data;

  int i;

  if (p_imageIn->format != pix_rgb24)
    return;

  for (i=0; i<p_imageIn->width*p_imageIn->height; i++)
    lum[i] = (BYTE) (0.299*(double)imageIn[3*i]   +
		     0.587*(double)imageIn[3*i+1] +
		     0.114*(double)imageIn[3*i+2]);

}
