/* imageSeq.cpp
 * Last modified:
 * Authors: Daniel Venkitachalam <venki-d@ee.uwa.edu.au>
 *          Leon Koch <leon@redfishsoftware.com.au>
 */

#include "imageSeq.h"

imageSeq::imageSeq(imageSeqType type, const char *path)
{
	int i, j, numValidFrames=0;
	Camera::CamInfo info;

	assert(path);
	this->seqPath = strdup(path);
	char sinfo;
	char picfile[MAXSEQNAMELEN];

	switch(type)
	{
		case PIXMAP:
			/* Read the files from the path in alphabetical order, ignoring the "README" file */
			GetSFiles(path, &sinfo);
			if(NOISY) fprintf(stderr, "numFrames: %d sinfo: %c seqPath %s\n", numFrames, sinfo, seqPath);

			switch(sinfo) {
				case 'l':
				case 'L':
					this->loopType = LOOP;
					break;
				case 'r':
				case 'R':
				case 'b':
				case 'B':
					this->loopType = BOUNCE;
					break;
				default:
					this->loopType = NOLOOP;
					break;
			}

			pics = (Picture **) calloc(numFrames, sizeof(Picture*));
			for(i=0,j=0; i<numFrames&&j<numFrames; i++) {
				pics[i] = newImage();
				strncpy(picfile, seqPath, MAXSEQNAMELEN);
				strncat(picfile, "/", 1);
				strcat(picfile, namelist[j++]);
				if(loadImage(picfile, pics[i]) == 0) {
					if(NOISY) fprintf(stderr, "Error: Loading image >%s<\n", picfile);
					i--;
				}
				else {
					if(i>0) {
						/* check that the loaded picture is the
						   same format/size as the previous one. */
						if((pics[i-1]->width != pics[i]->width)||
						   (pics[i-1]->height != pics[i]->height)||
						   (pics[i-1]->format != pics[i]->format)) {
							if(NOISY) fprintf(stderr, "Error: Loading image >%s< - image is of different format/dimensions.\n", picfile);
							pics[i] = NULL;
							break;
						}
						else numValidFrames++;
					}
					else numValidFrames++;
				}
			}
			
			if(numValidFrames>0) {
				/* convert rgb24 image to bgr32  */
				if(pics[0]->format==pix_rgb24) {
					for(i=0;i<numValidFrames;i++) pics[i] = RGB2BGR32(pics[i]);
				}
				currentIndex = -1;
				current = pics[0];
				numFrames = numValidFrames;
				width = current->width;
				height = current->height;
				seqType = PIXMAP;
			}	
			else {
				numFrames = 0;
				current=NULL;
				seqType = OTHER;
			}		

			break;

#ifdef HAVE_V4L
		case V4L:
			camera = new V4LCamera();

			if(camera->open() == true) {
				if(NOISY) fprintf(stderr,"Opened V4L camera successfully!\n");
				cameraReady = true;
				camera->get_info(&info);
				current = newImage();
				if(info.bpp == 8)
					setImageType(current, pix_grey);
				else if(info.bpp == 24) {
					setImageType(current, pix_bgr32);
				}
				resizeImage(current, info.width, info.height);
				camera->read(current);
				assert(current);
			}
			else {
				if(NOISY) fprintf(stderr,"Error: Unsuccessful opening V4L camera\n");
				cameraReady = false;
				current = NULL;
			}

			this->seqType = V4L;
			break;
#endif

#ifdef HAVE_V4L2
		case V4L2:
			camera = new V4L2Camera();

			if(camera.open() == true) {
				if(NOISY) fprintf(stderr,"Opened V4L 2 camera successfully!\n");
				cameraReady = true;
				camera.get_info(&info);
				current = newImage();
				if(info.bpp == 8)
					setImageType(current, pix_grey);
				else
					setImageType(current, pix_bgr32);
				resizeImage(current, info.width, info.height);
				camera.read(current);
				if(info.bpp==8) MONO2RGB32(current);
				else if(info.bpp==24) RGB242RGB32(current);
				assert(current);
			}
			else {
				if(NOISY) fprintf(stderr,"Error: Unsuccessful opening V4L 2 camera\n");
				cameraReady = false;
				current = NULL;
			}

			this->seqType = V4L2;
			break;
#endif

#ifdef HAVE_QUICKCAM
		case QCAM:
			camera = new QuickCam();
			
			if(camera->open() == true) {
				cameraReady = true;
				camera->get_info(&info);
				current = newImage();
				if(info.bpp == 8)
					setImageType(current, pix_grey);
				else if(info.bpp == 24) {
					setImageType(current, pix_bgr32);
				}
				resizeImage(current, info.width, info.height);
				camera->read(current);
				assert(current);
			}
			else {
				if(NOISY) fprintf(stderr,"Error: Unsuccessful opening QuickCam camera\n");
				cameraReady = false;
				current = NULL;
			}
			
			this->seqType = QCAM;
			break;
#endif
			
		default:
			info.width = 0;
			info.height = 0;
			info.bpp = 0;
			if(NOISY) fprintf(stderr,"Error: Attempted to open unknown type of imageSeq %d\n", type);
			break;
	}

	direction = FORWARD;

}

imageSeq::~imageSeq()
{
	int i;
	free(this->seqPath);

	if(this->seqType == PIXMAP) {
		if(NOISY) fprintf(stderr,"Deleting imageSeq (pixmap)");
		for(i=0; i<numFrames; i++)
			if(pics[i]) freeImage(pics[i]);

		if(pics)
			free(pics);
		
		for(i=0;i<numFrames;i++)
			free(namelist[i]);

		free(namelist);
	}

#ifdef HAVE_V4L
	if(this->seqType == V4L && cameraReady == true) {
		if(NOISY) fprintf(stderr,"Deleting imageSeq (V4L)");
		camera->close();
		delete camera;
	}
#endif

#ifdef HAVE_V4L2
	if(this->seqType == V4L2 && cameraReady == true) {
		if(NOISY) fprintf(stderr,"Deleting imageSeq (V4L2)");
		camera->close();
		delete camera;
	}
#endif

#ifdef HAVE_QUICKCAM
	if(this->seqType == QCAM && cameraReady == true) {
		if(NOISY) fprintf(stderr,"Deleting imageSeq (QCAM)");
		camera->close();
		delete camera;
		freeImage(current);
	}
#endif

	/** @todo Free dirent's mem leak */
}

int imageSeq::file_select(const struct dirent *entry)
{
	if((strcmp(entry->d_name, ".") == 0) ||
	   (strcmp(entry->d_name, "..") == 0) ||
	   (strcmp(entry->d_name, "README") == 0))
		return 0;
	else return 1;
}

void imageSeq::GetSFiles(const char *dirpath, char *sinfo)
{ 
	int i,j;
	char full_path[300];
	char ch;
	FILE * readme;
	dirent** seqFiles;

	strcpy (full_path, dirpath);
	strcat (full_path, "/");
	seqPath = strdup(dirpath); /* copy for later file retrieval */

	numFrames = scandir(full_path, &seqFiles, file_select, alphasort);

	if (NOISY) fprintf(stderr,"GetSFiles: load path >%s< got %d files\n", full_path, numFrames);

	/* only newline in source string */
	strcpy(seqInfo, "\n");
	/* loop by default */
	*sinfo = 'L'; 

	if (numFrames > 0) { 
		namelist = (char **) calloc(numFrames, sizeof(char*) );

		for (i=0; i< numFrames; i++) { 
			namelist[i] = (char *) malloc( MAXSEQNAMELEN );
			if(namelist[i] == NULL) {	
				fprintf(stderr,"Error: No memory available for image sequence name\n");
				return;
			}
			strncpy(namelist[i], seqFiles[i]->d_name, MAXSEQNAMELEN);
			free(seqFiles[i]);
			if (NOISY) fprintf(stderr,"%3d) >%s<\n", i, namelist[i]);
		}
		free(seqFiles);

		/* Get info from README file */
		strcpy (full_path, dirpath);
		strcat (full_path, "/README");
		if ((readme = fopen(full_path, "r")) != NULL) { 
			j = 0;
			ch = ' ';
			while (j<MAXSOURCE && ch != '\n') { 
				fscanf(readme, "%c", &ch);
				seqInfo[j++] = ch;
			}
			seqInfo[j] = (char) 0;			/* string terminator */
			fscanf(readme, "%c", &ch);		/* read info */
			*sinfo = ch;
			fclose(readme);
		}
	}
}

Picture* imageSeq::tick(int frames)
{
	if((seqType != PIXMAP)&&(seqType != OTHER)) {
		if(cameraReady == true) {
			camera->read(current);
			assert(current);
			return current;
		}
		else return NULL;
	}

	switch( loopType ) {
		case BOUNCE:
			if(direction == FORWARD) {
				currentIndex = currentIndex + frames;
			}
			else {
				currentIndex = currentIndex - frames;
			}
			if(currentIndex < 0) {
				currentIndex = -currentIndex;
				if(direction==FORWARD) direction=BACKWARD;
				else direction=FORWARD;
			}
			if(currentIndex >= numFrames) {
				currentIndex = 2*(numFrames-1) - currentIndex;
				if(direction==FORWARD) direction=BACKWARD;
				else direction=FORWARD;
			}
			current = pics[currentIndex];
			break;
		case LOOP:
		case NOLOOP:
			currentIndex = currentIndex + frames;
			if(currentIndex < 0) currentIndex = numFrames-1;
			if(currentIndex >= numFrames) currentIndex = 0;
			current = pics[currentIndex];
			break;
		default:
			break;
	}

	if(loopType == BOUNCE && currentIndex == 0) {
		direction = FORWARD;
	}
	else if(loopType == BOUNCE && currentIndex == numFrames-1) {
		direction = BACKWARD;
	}

	assert(current);
	return current;
}

Picture* imageSeq::getCurrent(void)
{
	if(this == NULL) {
		return (Picture *)NULL;		
	}
	else {
		return current;
	}
}

int imageSeq::getWidth(void)
{
	return width;
}

int imageSeq::getHeight(void)
{
	return height;
}

int imageSeq::getNumFrames(void)
{
	return numFrames;
}

imageSeqType imageSeq::getSeqType(void) {
	return seqType;
}

bool imageSeq::getCameraReady(void) {
	return cameraReady;
}

Camera *imageSeq::getCamera(void) {
	if(cameraReady)
		return camera;
	else
		return NULL;
}

char *imageSeq::getInfo(void) {
	if(strncmp(seqInfo,"\n",1)==0) return (char *)NULL;
	else return (char *)seqInfo;
}
