/* improvConfig.h
 * Last modified: 
 * Authors: Leon Koch <leon@redfishsoftware.com.au>
 */
/** @class improvConfig
 * Stores all config values for easy access by all objects.
 * An instance of this class is created and maintained by the global singleton class 'Improv'.
 * @see Improv
 */

#ifndef IMPROVCONFIG_H
#define IMPROVCONFIG_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <dirent.h>
#include <getopt.h>

#include "imageSeq.h"

#define MAXLINE 512

#define CONFIG_FILENAME ".improvqt"
#define DEFAULT_WINDOW_WIDTH     169
#define DEFAULT_WINDOW_HEIGHT    134
#define DEFAULT_NUMBER_WINDOWS   6
#define MINIMUM_BUTTON_BAR_WIDTH 150

class improvConfig
{
	public:
		/** @struct widgetConfig
		 * A structure used to store the configuration information of both source and sequence widgets.
		 */
		typedef struct {
			/** Widget identifier. This is unique for all widgets. */
			int id;
			/** Indicates whether the widget is a source widget or not. */
			bool source;
			/** The source indentifier of the widget. */
			int sourceId;
			/** */
			int position;
			/** */
			int row;
			/** */
			int col;
		} widgetConfig;

		/** @struct pluginInstance
		 * A structure used to store information about an instance of an processing operation.
		 */
		typedef struct {
			/** */
			int row;
			/** */
			int col;
			/** */
			int posn;
			/** */
			int uniqueId;
			/** */
			float *params;
			/** */
			char *name;
		} pluginInstance;

		/** Constructor. */
		improvConfig();

		/** Destructor. Deletes the instance of the class. */
		~improvConfig();

		/** Load configuration values for the command line arguments.
		 * @param argc The number of command line arguments.
		 * @param argv The command line arguments.
		 */
		void load(int argc, char **argv);
	
		/** Load default configuration values.
		 */
		void loadDefaults(void);

		/** Loads config from the default file (~/CONFIG_FILENAME)
		 * @return true if successfully saved, false if it failed.
		 */
		bool load(void);

		/** Loads config from a file
		 * @return true if successfully saved, false if it failed.
		 */
		bool load(char *filename);
		
		/** Saves config to the default file (~/CONFIG_FILENAME)
		 * @return true if successfully saved, false if it failed.
		 */
		bool save(void);

		/** Saves config to a file
		 * @return true if successfully saved, false if it failed.
		 */
		bool saveAs(char *filename);

		/** Clear (or reset) all configuration details. */
		void clear(void);
		
		/** Clear (or reset) only the configuration details to do with layout. */
		void clearLayout(void);
			
		/** Set the sequence path
		 * @param sequencePath desired sequence path
		 */
		void setSequencePath(char *sequencePath);

		/** Get the sequence path
		 * @return sequence path
		 */
		char *getSequencePath(void);
	
		/** Set the plugin path
		 * @param pluginPath desired plugin path
		 */
		void addPluginPath(char *pluginPath);

		/** Return the number of plugin paths.
		 * @return The number of plugin paths.
		 */
		int getNumPluginPaths(void);

		/** Get a plugin path
		 * @param num The plugin path to return.
		 * @return The plugin path.
		 */
		char *getPluginPath(int num);

		/** Removes all plugin paths from the configuration.
		 */
		void clearPluginPaths(void);
	
		/** Set the window width
		 * @param width desired window width
		 */
		void setWindowWidth(int width);

		/** Get the window width
		 * @return window width
		 */
		int getWindowWidth(void);

		/** Set the window height
		 * @param height desired window height
		 */
		void setWindowHeight(int height);

		/** Get the window height
		 * @return window height
		 */
		int getWindowHeight(void);

		/** Set the number of image windows
		 * @param number number of windows
		 */
		void setNumSources(int number);

		/** Get the number of windows
		 * @return number of windows
		 */
		int getNumSources(void);

		/** Set the number of image windows
		 * @param number number of windows
		 */
		void setNumWindows(int number);
                          
		/** Get the number of windows
		 * @return number of windows
		 */
		int getNumWindows(void);

		/** Set the sequence to start with
		 */
		void setUseSequence(char *name);

		/** Get the sequence that was started with
		 */
		char *getUseSequence(void);

		/** Set the type of camera
		 */
		void setCameraType(imageSeqType camera);
		
		/** Get the type of camera
		 * @return enum of imageSeqType
		 */
		imageSeqType getCameraType(void);

		/** Return whether the config has been modified or not.
		 * @return True if the configuration has been modified 
		 * since initialisation of the application, false otherwise.
		 */
		bool isModified(void);

		/** Set whether the configuration has been modified or not.
		 * @param on bool value.
		 */
		void setModified(bool on);

		/** Return the widget configuration for a given position.
		 * @param posn The position of the widget.
		 * @return The widget configuration
		 * @see widgetConfig
		 */
		widgetConfig *getWidgetConfig(int posn);
		
		/** */
		widgetConfig *getWidgetConfig(bool source, int id);

		/** */
		int numWidgetConfigs(void);

		/** */
		pluginInstance *getPluginInstance(int posn);
		
		/** */
		int numPluginInstances(void);

		/** */
		int getNumRows(void);
		
		/** */
		int getNumCols(void);

		/** */
		bool verifyWidgetConfig(void);
		
		/** */
		bool verifyWidgetConfig(widgetConfig **_widgets, int _numWidgets,	int _numWindows, int _numSources);

		/** */
		char *getDefaultFilename(void);

	private:
		/** */
		char *sequencePath;
		/** */
		int numPluginPaths;
		/** */
		char **pluginPaths;
		/** */
		char *useSequence;
		/** */
		int windowWidth;
		/** */
		int windowHeight;
		/** */
		int numSources;
		/** */
		int numWindows;
		/** */
		int numRows;
		/** */
		int numCols;
		/** */
		imageSeqType cameraType;
		/** */
		bool modified;
		/** */
		char *defaultFilename;
		/** */
		int numPlugins;
		/** */
		pluginInstance **plugins;
		/** */
		int numWidgets;
		/** */
		widgetConfig **widgets;
		
};

#endif
