MODULE ga7seg;
(* Genetic Algorithm Simulation, Thomas Braunl Univ. Stuttgart 1995 *)
CONST mut_fac    =  20000;
      word_len   =      7;
      num_states = 2**word_len;
      num_gates  = num_states - 4; (* 4 input digits *)
      gene_size  = num_gates * 2 * word_len;
      fac        =     10;         (* factor for correct digit *)
      pop_size   =    200;
      max_iter   =    600;
      fit_max    = (fac*7*10 + num_gates)**2;
      fit_thres  = (fac*7*10 + num_gates DIV 2)**2;
      quiet      =   TRUE;
      debug      =  FALSE;
      debug2     =  FALSE;

(* ************************************************************** *)
(* gate coding example                                            *)
(* 1 7    2 8    0 0                                              *)
(* gate1  gate2  empty                                            *)
(* pos 8  pos 9  pos 10    --- pos 1..7 is 7 segment input        *)
(* binary: 0001 0111 0010 1000 0000 0000                          *)
(* ************************************************************** *)

CONFIGURATION pop [0..pop_size-1];
CONNECTION    left: pop[i] <-> pop[(i+1) MOD pop_size]: right;

TYPE genotype  = ARRAY[0..gene_size-1 ] OF BOOLEAN;
     phenotype = ARRAY[4..num_states-1] OF INTEGER;
     phenobool = ARRAY[4..num_states-1] OF BOOLEAN;
     statetype = ARRAY[0..num_states-1] OF BOOLEAN;
     segtype   = ARRAY[0..9],[1..7]     OF BOOLEAN;

VAR gene_pool, new_pool     : pop OF genotype;
    pheno1, pheno2          : pop OF phenotype;
    state                   : pop OF statetype;
    used                    : pop OF phenobool;
    fitness                 : pop OF INTEGER;
    best_gene               : genotype;
    best_p1, best_p2        : phenotype;
    best_s                  : statetype;
    best_u                  : phenobool;
    min_fit, max_fit, avg_fit,
    iter, best, k           : INTEGER;
    segments                : segtype;


PROCEDURE ReadData(VAR pool: segtype);
VAR pat,seg,i: INTEGER;
BEGIN
  OpenInput("segment.pat");
  IF NOT Done THEN
    WriteString("error reading data file 'segment.pat'");
    WriteLn; HALT;
  END;
  FOR pat := 0 TO 9 DO
    FOR seg := 1 TO 7 DO
      ReadInt(i);
      IF NOT Done THEN
        WriteString("error reading data file 'segment.pat'");
        WriteLn; HALT;
      END;
      pool[pat,seg] := i=1;
    END;
  END;
  CloseInput;
END ReadData;


PROCEDURE Init(VAR g: pop OF genotype);
VAR i: INTEGER;
BEGIN
  FOR i:=0 TO gene_size-1 DO
    g[i] := RandomBool(pop);
  END;
END Init;


PROCEDURE SetDigit(pat: INTEGER; VAR state: pop OF statetype);
(* set binary input digits at position 0..3 *)
VAR pos: INTEGER;
BEGIN
  FOR pos := 3 TO 0 BY -1 DO
    state[pos] := pat MOD 2 = 1;
    pat := pat DIV 2;
  END;
  FOR pos:=4 TO num_states-1 DO state[pos] := FALSE END;
  IF debug THEN
    WriteString("start:"); 
    FOR pos:=0 TO num_states-1 DO
      WriteInt(ORD(state<<1>>[pos]), 2);
      IF (pos=3) OR (pos=num_states-8) THEN Write(' ') END;
    END;
    WriteLn;
  END;
END SetDigit;


PROCEDURE CreatePheno(VAR gene: pop OF genotype;
                      VAR pheno1,pheno2: pop OF phenotype);
VAR pos, posw, k: INTEGER;
BEGIN
  FOR pos := 4 TO num_states-1 DO
    posw := 2*word_len * (pos-4);
    pheno1[pos] := 0;
    pheno2[pos] := 0;
    FOR k:=0 TO word_len-1 DO
      pheno1[pos] := 2 * pheno1[pos] + ORD(gene[posw+k]);
      pheno2[pos] := 2 * pheno2[pos] + ORD(gene[posw+word_len+k]);
    END;
    (* generate only correct connections, ie. sources less own position *)
    pheno1[pos] := pheno1[pos] MOD pos;
    pheno2[pos] := pheno2[pos] MOD pos;
  END;
END CreatePheno;


PROCEDURE Correct(pat: INTEGER; VAR state: pop OF statetype): pop OF INTEGER;
(* desired result is 7-segment display in last 7 state positions *)
(* desired output is specified in GLOBAL segments                *)
VAR points: pop OF INTEGER;
    k     : INTEGER;
BEGIN
  points := 0;
  FOR k:=1 TO 7 DO  (* test all 7 segments *)
    IF state[num_states-8+k] = segments[pat,k] THEN INC(points) END;
  END;

  IF debug THEN
    WriteString("goal"); WriteInt(pat,1); 
    WriteString(":                                                    ");
    FOR k:=1 TO 7 DO WriteInt(ORD(segments[pat,k]), 2) END;
    WriteLn;
  END;
  RETURN points;
END Correct;


PROCEDURE Evaluation(VAR gene          : pop OF genotype;
                     VAR pheno1, pheno2: pop OF phenotype;
                     VAR state         : pop OF statetype;
                     VAR used          : pop OF phenobool): pop OF INTEGER;
(* input: number in 4 digit binary format (pos. 0..3)         *)
(* output: number in 7-segment display    (pos. max-6 .. max) *)
VAR i, pat       : INTEGER;
    fit, num_used: pop OF INTEGER;
BEGIN
  CreatePheno(gene, pheno1,pheno2);

  (* check usage of gates, dangling gates can be eliminated *)
  num_used := 7;  (* seven output segments *)
  FOR i := num_states-1 TO num_states-7 BY -1 DO used[i] := TRUE END;
  FOR i := num_states-1 TO 4 BY -1 DO
    IF (pheno1[i] >= 4) AND NOT used[pheno1[i]] THEN
      used[pheno1[i]] := TRUE;
      INC(num_used);
    END;
    IF (pheno2[i] >= 4) AND NOT used[pheno2[i]] THEN
      used[pheno2[i]] := TRUE;
      INC(num_used);
    END;
  END; (* for *)

  (* perform execution for all test patterns *)
  fit := num_gates - num_used;
  FOR pat := 0 TO 9 DO
    SetDigit(pat, state);  (* set input positions 0..3 *)
    FOR i := 4 TO num_states-1 DO  (* NAND gate *)
      state[i+4] := NOT (state[pheno1[i]] AND state[pheno2[i]]);
    END;
    INC(fit, fac * Correct(pat,state));
  END; (* for pat *)
  RETURN fit*fit;  (* quadratic fitness function *)
END Evaluation;


PROCEDURE Selection(VAR old_pop,sel_pop: pop OF genotype;
                    VAR fit: pop OF INTEGER);
VAR take, wheel_of_fit: pop OF INTEGER;
    fit_val, i        : INTEGER;
BEGIN
  fit_val := 0;  (* generate fitness distribtion *)
  FOR i:=0 TO pop_size-1 DO
    INC(fit_val, fitness<:i:>);
    wheel_of_fit<:i:> := fit_val
  END;
  take := RandomInt(pop) MOD fit_val;  (* 0..sum(fit)-1 *)
  FOR i:=0 TO pop_size-1 DO
    IF take<:i:> < wheel_of_fit THEN  (* hit on wheel segment *)
      IF REDUCE.MIN(wheel_of_fit) = wheel_of_fit THEN
        sel_pop<:i:> := REDUCE.FIRST(old_pop)  (* select individuum *)
      END;
    END;
  END;
  IF debug2 THEN WriteString("after Selection"); WriteLn; Check(sel_pop) END;
END Selection;
               
            
PROCEDURE CrossOver(VAR sel_pop,new_pop: pop OF genotype);
VAR sel_neighbor, new_neighbor: pop OF genotype;
    point                     : pop OF INTEGER;
    i                         : INTEGER;
BEGIN
  IF EVEN(ID(pop))
    THEN SEND.left(sel_pop, sel_neighbor);
    ELSE (* only odd individuals work *)
    point := RandomInt(pop) MOD gene_size; (* cross-over point *)
    FOR i:=0 TO gene_size-1 DO
      IF i<point THEN new_pop[i]      := sel_pop[i];
                      new_neighbor[i] := sel_neighbor[i]
                 ELSE new_pop[i]      := sel_neighbor[i];
                      new_neighbor[i] := sel_pop[i]
      END;
    END;
    SEND.right(new_neighbor, new_pop);  (* update even individuals *)
  END;
  IF debug2 THEN WriteString("after Cross"); WriteLn; Check(new_pop) END;
END CrossOver;                             


PROCEDURE Mutation(VAR g: pop OF genotype);
VAR i: INTEGER;
BEGIN
  FOR i:=0 TO gene_size-1 DO
    IF RandomInt(pop) MOD mut_fac = 0 THEN
      g[i] := NOT g[i];
    END;
  END;
  IF debug2 THEN WriteString("after Mut"); WriteLn; Check(g) END;
END Mutation;


PROCEDURE Check(VAR g: pop OF genotype);
VAR k,l: INTEGER;
BEGIN
  WriteString("Check population:"); WriteLn;
  FOR l:=1 TO pop_size DO
    FOR k:=0 TO gene_size-1 DO
      WriteInt(ORD(g<<l>>[k]), 1);
      IF (k+1) MOD word_len = 0 THEN Write(' ') END;
    END;
    WriteLn;
  END;
END Check;


PROCEDURE CheckBest(VAR g: genotype; VAR p1,p2: phenotype;
                    VAR s: statetype; VAR u: phenobool);
VAR i, num: INTEGER;
BEGIN
  IF debug THEN
    WriteString("stop :");
    FOR i:=0 TO num_states-1 DO
      WriteInt(ORD(s[i]),2);
      IF (i=3) OR (i=num_states-8) THEN Write(' ') END;
    END;
    WriteLn;
  END;

  IF debug2 THEN
    WriteString("best: ");
    FOR i:=0 TO gene_size-1 DO
      WriteInt(ORD(g[i]), 1);
      IF (i+1) MOD word_len = 0 THEN Write(' ') END;
    END;
    WriteLn;
  END;

  num := 0;
  FOR i:=4 TO num_states-1 DO
    (* print only valid gates *)
    IF u[i] THEN
      Write('g'); WriteInt(i,3); Write('(');
      WriteInt(p1[i],2); Write(',');
      WriteInt(p2[i],2); WriteString(')  ');
      INC(num);
      IF num MOD 5 = 0 THEN WriteLn END;
    END; (* if *)
  END;
  WriteLn;
END CheckBest;


BEGIN (* main *)
  ReadData(segments);
  Init(gene_pool);
  iter := 0;
  WriteString("population size: "); WriteInt(pop_size, 6); WriteLn;
  WriteString("fitness max    : "); WriteInt(fit_max  ,6); WriteLn;
  WriteString("fitness thres  : "); WriteInt(fit_thres,6); WriteLn;
  WriteLn;
  REPEAT
    INC(iter);
    fitness := Evaluation(gene_pool, pheno1, pheno2, state, used);
    max_fit := REDUCE.MAX(fitness);
    min_fit := REDUCE.MIN(fitness);
    avg_fit := REDUCE.SUM(fitness) DIV pop_size;
    IF max_fit = fitness THEN
      best      := REDUCE.FIRST(ID(pop));
      best_gene := gene_pool<<best>>;
      best_p1   := pheno1   <<best>>;
      best_p2   := pheno2   <<best>>;
      best_s    := state    <<best>>;
      best_u    := used     <<best>>;
    END;
    IF quiet THEN
      WriteInt(min_fit,7); WriteInt(avg_fit,7); WriteInt(max_fit,7);
     ELSE
      WriteString("generation "); WriteInt(iter,4); WriteLn;
      CheckBest(best_gene, best_p1, best_p2, best_s, best_u);
      WriteString("min fitness: ");   WriteInt(min_fit,6); 
      WriteString("  avg fitness: "); WriteInt(avg_fit,6);
      WriteString("  max fitness: "); WriteInt(max_fit,6); WriteLn;
      IF debug2 THEN Check(gene_pool) END;
    END;
    WriteLn;

    Selection (gene_pool, new_pool, fitness);
    CrossOver (new_pool, gene_pool);
    Mutation  (gene_pool);
  UNTIL (iter = max_iter) OR (max_fit >= fit_thres);
END ga7seg.

