/*
**  Interface module between Parallaxis-III and X11R5 Graphics Library.
**
**  Written by Joerg Stippa in September 1994.
**  Contact address: stippa@hermes.informatik.uni-stuttgart.de
**		 or: stippa@isa.de
*/

/*
**  All graphic output is terminated by calling XFlush() to be sure
**  graphic appears on the screen before normal computation continues.
**  The best way to achieve this would be calling XSync(), but then
**  there is a waste of time, assuming all commands will end correct.
**  When debugging graphic output you may call XSynchronize() in
**  init_x() to run the X window connection in synchronous mode.
**  I.e. all calls to Xlib are round trips and are very slow!
*/


#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include <stdlib.h>
#if !defined (_AIX)
#include <alloca.h>
#endif

#include "SYSTEM_C.h"	/* Parallaxis-III general definitions */
#include "Graphics.h"	/* Definitions for the graphics routines */

#ifndef numberof
#define numberof(arr)	(sizeof(arr) / sizeof(arr[0]))
#endif

#ifndef abs
#define abs(val)	((val) < 0 ? -(val) : (val))
#endif

#ifndef FONT_ENV_VAR	/* default environment variable containing draw-font */
#define FONT_ENV_VAR	"PARALLAXIS_FONT"
#endif

#define COLOR_CACHE_ENTRIES	256


/* Macro to check if a window is in use. On failure set error variable,
   otherwise clear previous error. */
#define CheckForActiveWindow \
    if (!selectW) { \
	Graphics_GraphicsError = Graphics_GrNoWindowSelected; \
	return; } \
    else Graphics_GraphicsError = Graphics_GrOK

/* The same as above, but can return a result. */
#define CheckForActiveWindowReturn(retval) \
    if (!selectW) { \
	Graphics_GraphicsError = Graphics_GrNoWindowSelected; \
	return retval; } \
    else Graphics_GraphicsError = Graphics_GrOK


/* Generalize a data type for a pixel */
typedef unsigned long Pixel;	/* look for type of pixel in XColor struct! */


/* Global variable for error indication in Graphics-Module */
CARDINAL Graphics_GraphicsError;


/* Data structure for window management */
struct WindowList
{
    struct WindowList	*nextW;
    Window		wPointer;	/* X11 handle */
    Colormap		cmap;		/* the color map for this window */
#ifndef NO_COLOR_CACHE
    XColor		*colorCache;	/* a trick for speed up */
    int			*colorSlot;	/* index of next avail slot */
#endif
    int			winNum;		/* internal number */
    int			width, height;	/* size */
    int			pixel;		/* current color */
    int			x, y;		/* current drawing position */
};

/* Variables for window management */
static struct WindowList   *firstW = NULL,
			   *selectW = NULL,
			   *lastW = NULL,
			    maxW;

/* X-specific variables */
static Display	    *display;
static GC	    gc;
static XFontStruct  *fontStruct;
static Screen	    *screen;
static int	    screenwidth;
static int	    screenheight;
#ifndef NO_COLOR_CACHE
static XColor	    defColorCache[COLOR_CACHE_ENTRIES];	/* for speed up */
static int	    defColorSlot;	/* index of next avail slot */
#endif



static Graphics_window _OpenWindow ARGS((CHAR title[], CARDINAL O_1, CARDINAL width, CARDINAL height, int private_cmap));
static int init_x ARGS((void));
static int lookupColor ARGS((struct WindowList *win, XColor *color));
static void _SetArea ARGS((Pixel *, int x, int y, CONFIGURATION Config));
static void _SetAreaXYZ ARGS((Pixel *, int x, int y, int zoom, CONFIGURATION Config));
static void _SetAreaXYS ARGS((Pixel *, int x, int y, int shrink, CONFIGURATION Config));




Graphics_window Graphics_OpenWindow
#ifdef __STDC__
(CHAR title[], CARDINAL O_1, CARDINAL width, CARDINAL height)
#else
(title, O_1, width, height)
CHAR title[];
CARDINAL O_1;
CARDINAL width, height;
#endif
{
    return _OpenWindow (title, O_1, width, height, 0);
}



Graphics_window Graphics_OpenWindowP
#ifdef __STDC__
(CHAR title[], CARDINAL O_1, CARDINAL width, CARDINAL height)
#else
(title, O_1, width, height)
CHAR title[];
CARDINAL O_1;
CARDINAL width, height;
#endif
{
    return _OpenWindow (title, O_1, width, height, 1);
}



/*
**  Create a new window with given size in width x height.
**  Name the window as 'title'.
**  Positioning is handled by the user via mouse.
*/
static Graphics_window _OpenWindow
#ifdef __STDC__
(CHAR title[], CARDINAL O_1, CARDINAL width, CARDINAL height, int private_cmap)
#else
(title, O_1, width, height, private_cmap)
CHAR title[];
CARDINAL O_1;
CARDINAL width, height;
int private_cmap;
#endif
{
    struct WindowList	    *newW;
    XSetWindowAttributes    winAttrs;
    XSizeHints		    sizeHints;
    XEvent		    event;

    Graphics_GraphicsError = Graphics_GrOK;

    /* Display already opened? */
    if (display == NULL)
	if (init_x () < 0)	/* no, init all the stuff */
	{
	    Graphics_GraphicsError = Graphics_GrWindowCreationFailure;
	    return -3;
	}

    /* Window's size ok? */
    if (width > screenwidth || width <= 0 || height > screenheight || height <= 0)
    {
	Graphics_GraphicsError = Graphics_GrWrongWindowSize;
	return -1;
    }

    /* Get memory for new window structure */
    if ((newW = (struct WindowList *) malloc (sizeof (struct WindowList)))
	== NULL)
    {
	Graphics_GraphicsError = Graphics_GrMemoryTrouble;
	return -2;
    }

    newW->cmap = 0;

    /* Assign a new window id */

    newW->winNum = lastW ? lastW->winNum + 1 : 1;

    /* Initialise new window structure */

    newW->nextW = NULL;

    newW->width = width;
    newW->height = height;

    newW->pixel = BlackPixelOfScreen (screen);

    newW->x = 0;
    newW->y = 0;

    /* Create is now */
    newW->wPointer = XCreateSimpleWindow (display, RootWindowOfScreen (screen),
					  newW->x, newW->y,
					  newW->width, newW->height, 2,
					  BlackPixelOfScreen (screen),
					  WhitePixelOfScreen (screen));
    /* Windowattributes */
    winAttrs.event_mask = ExposureMask;
    winAttrs.bit_gravity = NorthWestGravity;
    winAttrs.backing_store = WhenMapped;

    XChangeWindowAttributes (display, newW->wPointer,
			     CWEventMask | CWBitGravity | CWBackingStore,
			     &winAttrs);

    /* Colormap for new window */
    if (private_cmap)
    {
	newW->cmap = XCreateColormap (display, newW->wPointer,
				      DefaultVisualOfScreen (screen),
				      AllocNone);
	if (newW->cmap)
	{
	    XColor col;

#ifndef NO_COLOR_CACHE
	    newW->colorCache = malloc (COLOR_CACHE_ENTRIES * sizeof (XColor));
	    if (newW->colorCache != NULL)
	    {
		newW->colorSlot = malloc (sizeof (*newW->colorSlot));
		if (newW->colorSlot == NULL)
		{
		    free (newW->colorCache);
		    newW->colorCache = NULL;
		}
		else
		  *newW->colorSlot = COLOR_CACHE_ENTRIES - 1;
	    }
#endif

	    if (BlackPixelOfScreen (screen) < WhitePixelOfScreen (screen))
	    {
		col.red = col.green = col.blue = 0;
		lookupColor (newW, &col);
		col.red = col.green = col.blue = 0xFFFF;
		lookupColor (newW, &col);
	    }
	    else
	    {
		col.red = col.green = col.blue = 0xFFFF;
		lookupColor (newW, &col);
		col.red = col.green = col.blue = 0;
		lookupColor (newW, &col);
	    }
	}
    }
    if (newW->cmap == 0)
    {
	newW->cmap = DefaultColormapOfScreen (screen);
#ifndef NO_COLOR_CACHE
	newW->colorCache = defColorCache;
	newW->colorSlot = &defColorSlot;
#endif
    }
    XSetWindowColormap (display, newW->wPointer, newW->cmap);

    /* Insert new window into window list */

    selectW = newW;

    if (lastW)
    {
	lastW->nextW = newW;
	lastW = newW;
    }
    else
	firstW = lastW = newW;

    /* Set the name of the window and its icon */
    XStoreName (display, newW->wPointer, title);
    XSetIconName (display, newW->wPointer, title);

    /* Inhibit resizeing */
    sizeHints.flags = PMinSize | PMaxSize;
    sizeHints.min_width = sizeHints.max_width = newW->width;
    sizeHints.min_height = sizeHints.max_height = newW->height;
    XSetWMNormalHints (display, newW->wPointer, &sizeHints);

    /* show window */
    XMapWindow (display, newW->wPointer); /* Window sichtbar machen */

    /* Wait until window is really visible */
    /* Neccessary for servers not supporting backing store */
    XWindowEvent (display, newW->wPointer, ExposureMask, &event);

    return newW->winNum;	/* Window identifier */
}



/*
**  Destroy a window and release all memory
*/
void Graphics_CloseWindow
#ifdef __STDC__
(CARDINAL window_num)
#else
(window_num)
CARDINAL window_num;
#endif
{
    struct WindowList	    *help, *oldhelp;

    /* Search for the window id */
    for (help = firstW, oldhelp = NULL;
	 help != lastW && help->winNum < window_num;
	 oldhelp = help, help = help->nextW)
	;

    if (!(help && help->winNum == window_num)) /* window id ok? */
	Graphics_GraphicsError = Graphics_GrNotExistingWindow;
    else
    {
	Graphics_GraphicsError = Graphics_GrOK;

	/* remove the window from the list of displayed windows
	   check if it is the currently active one, and choose another
	   to be the new active window. */

	if (oldhelp)
	    oldhelp->nextW = help->nextW;

	if (help == firstW)
	    firstW = help->nextW;

	if (help == selectW)
	    selectW = oldhelp ? oldhelp : firstW;

	if (help == lastW)
	    lastW = oldhelp;

	/* Destroy the window */
	XDestroyWindow (display, help->wPointer);
	XFlush (display);

	/* Free private colormap */
	if (help->cmap != DefaultColormapOfScreen (screen))
	    XFreeColormap (display, help->cmap);

	/* Free memory of corresponding management information */
	free (help);
    }
}



/*
**  Choose a window to be the active one.
*/

void Graphics_SelectWindow
#ifdef __STDC__
(CARDINAL window_num)
#else
(window_num)
CARDINAL window_num;
#endif
{
    struct WindowList *help;

    Graphics_GraphicsError = Graphics_GrOK;

    /* Look for special case 'zero'. This means, activate the last
       window created. Otherwise find the window id and set it active. */

    if (window_num == 0 && display != (Display *) 0)
	selectW = &maxW;
    else
    {
	for (help = firstW; help != lastW && help->winNum < window_num;
	    help = help->nextW)
	    ;

	if (help && help->winNum == window_num) /* Window id found? */
	    selectW = help;
	else
	{
	    Graphics_GraphicsError = Graphics_GrNotExistingWindow;
	    selectW = NULL;
	}
    }
}



/*
**  Query the X window system for the current screen size
*/
void Graphics_GetScreenSize
#ifdef __STDC__
(CARDINAL *width, CARDINAL *height)
#else
(width, height)
CARDINAL *width, *height;
#endif
{
    /* Already initialized? No, then return 'zero' */

    if (display == (Display *) 0)
	*width = *height = 0;
    else
	*width = screenwidth,	/* these variables are set in init_x() */
	*height = screenheight;
}



/*
**  Query the system for the size of the active window.
*/

void Graphics_GetWindowSize
#ifdef __STDC__
(CARDINAL *width, CARDINAL *height)
#else
(width, height)
CARDINAL *width, *height;
#endif
{
    /* Got an active window? No, then return 'zero' */

    if (selectW == (struct WindowList *) 0)
	*width = *height = 0;
    else
	*width = selectW->width,	/* set on creation, never changed */
	*height = selectW->height;
}



/*
**  Receive a single linear array of color information in c.
**  PE configuration is given in OPEN_VECTOR.
**  Copy color info and make it suitable for X. Finally call common code
**  for filling an area (max 2-dim.) appropriately.
**  Color's RGB range between 'zero' and 255 inclusive.
*/

void Graphics_SetArea
#ifdef __STDC__
(ImageIO_color c[], CONFIGURATION OPEN_VECTOR)
#else
(c, OPEN_VECTOR)
ImageIO_color c[];
CONFIGURATION OPEN_VECTOR;
#endif
{
    int used_malloc = 0;
    Pixel *color;

    CheckForActiveWindow;

    /* Try using the stack */
    color = alloca (LEN (OPEN_VECTOR) * sizeof (Pixel));

    if (!color)
	/* Okay, I will use system memory */
	if ((color = malloc (LEN (OPEN_VECTOR) * sizeof (Pixel))))
	    used_malloc++;

    if (color)
    {
	register int i, pe;
	register Pixel *cp = color;
	XColor col;

	for (pe = 0, i = LEN (OPEN_VECTOR); --i >= 0; c++, cp++, pe++)
	    if (IS_ACTIVE (OPEN_VECTOR, pe))
	    {
		/* Translate color info into X11 values.
		   Could be done by multiplying with 257, but looks worse.
		   This way 0 maps to lowest (0), and 255 maps to highest
		   (65535). */
		col.red   = (c->red << 8) + c->red;
		col.green = (c->green << 8) + c->green;
		col.blue  = (c->blue << 8) + c->blue;
		/* Query for the requested color */
		*cp = lookupColor (selectW, &col) != 0
		      ? col.pixel
		      : BlackPixelOfScreen (screen);
	    }

	/* Now do the real work */
	_SetArea (color, 0, 0, OPEN_VECTOR);
    }

    if (used_malloc)
	free (color);
}



/*
**  The same as for Graphics_SetArea(), but handles grey scale.
**  Grey ranges between 'zero' and 255 inclusive.
*/

void Graphics_SetgArea
#ifdef __STDC__
(ImageIO_gray c[], CONFIGURATION OPEN_VECTOR)
#else
(c, OPEN_VECTOR)
ImageIO_gray c[];
CONFIGURATION OPEN_VECTOR;
#endif
{
    int used_malloc = 0;
    Pixel *color;

    CheckForActiveWindow;

    color = alloca (LEN (OPEN_VECTOR) * sizeof (Pixel));

    if (!color)
	if ((color = malloc (LEN (OPEN_VECTOR) * sizeof (Pixel))))
	    used_malloc++;

    if (color)
    {
	register int i, pe;
	register Pixel *cp = color, prev_pixel = 0xaffe;
	register ImageIO_gray prev = 0xdeadbeef;
	XColor col;

	for (pe = 0, i = LEN (OPEN_VECTOR); --i >= 0; c++, cp++, pe++)
	    if (IS_ACTIVE (OPEN_VECTOR, pe))
	    {
		if (prev == *c)
		    *cp = prev_pixel;
		else
		{
		    prev = *c;
		    /* convert grey to color. For an explanation of
		       the algorithm see Graphics_SetArea above. */
		    col.red = col.green = col.blue = (*c << 8) + *c;
		    prev_pixel = *cp = lookupColor (selectW, &col) != 0
				       ? col.pixel
				       : BlackPixelOfScreen (screen);
		}
	    }

	_SetArea (color, 0, 0, OPEN_VECTOR);
    }

    if (used_malloc)
	free (color);
}



/*
** Even fewer colors are used, i.e. black and white presentation.
** Anything else is like Graphics_SetArea().
*/

void Graphics_SetbArea
#ifdef __STDC__
(ImageIO_binary c[], CONFIGURATION OPEN_VECTOR)
#else
(c, OPEN_VECTOR)
ImageIO_binary c[];
CONFIGURATION OPEN_VECTOR;
#endif
{
    int used_malloc = 0;
    Pixel *color;

    CheckForActiveWindow;

    color = alloca (LEN (OPEN_VECTOR) * sizeof (Pixel));

    if (!color)
	if ((color = malloc (LEN (OPEN_VECTOR) * sizeof (Pixel))))
	    used_malloc++;

    if (color)
    {
	register int i;
	register Pixel *cp = color;
	register Pixel bp = BlackPixelOfScreen (screen);
	register Pixel wp = WhitePixelOfScreen (screen);

	for (i = LEN (OPEN_VECTOR); --i >= 0; )
	    *cp++ = *c++ ? bp : wp;

	_SetArea (color, 0, 0, OPEN_VECTOR);
    }

    if (used_malloc)
	free (color);
}



/*
**  Now, some pictures are too small or too big to enjoy.
**  Handle zooming/shrinking.
*/
void Graphics_SetAreaXYZ
#ifdef __STDC__
(ImageIO_color c[], INTEGER x, INTEGER y, INTEGER zoom, CONFIGURATION OPEN_VECTOR)
#else
(c, x, y, zoom, OPEN_VECTOR)
ImageIO_color c[];
INTEGER x;
INTEGER y;
INTEGER zoom;
CONFIGURATION OPEN_VECTOR;
#endif
{
    int used_malloc = 0;
    Pixel *pixel;

    CheckForActiveWindow;

    /* Shrinking by '1' is the same as enlarging by '1'. */

    if (zoom == -1)
	zoom = 1;

    pixel = alloca (LEN (OPEN_VECTOR) * sizeof (Pixel));

    if (!pixel)
	if ((pixel = malloc (LEN (OPEN_VECTOR) * sizeof (Pixel))))
	    used_malloc++;

    if (pixel)
    {
	register int i, pe;
	register Pixel *cp = pixel;
	XColor col;

	for (pe = 0, i = LEN (OPEN_VECTOR); --i >= 0; c++, cp++, pe++)
	    if (IS_ACTIVE (OPEN_VECTOR, pe))
	    {
		/* and again fit colors into X11
		   map 0 to zero and 255 to 65535 */
		col.red   = (c->red << 8) + c->red;
		col.green = (c->green << 8) + c->green;
		col.blue  = (c->blue << 8) + c->blue;
		*cp = lookupColor (selectW, &col) != 0
		      ? col.pixel
		      : BlackPixelOfScreen (screen);
	    }

	if (zoom > 0)
	    if (zoom == 1)	/* no zoom, use faster routine */
		_SetArea (pixel, x, y, OPEN_VECTOR);
	    else
		_SetAreaXYZ (pixel, x, y, zoom, OPEN_VECTOR);
	else			/* shrink data */
	    _SetAreaXYS (pixel, x, y, -zoom, OPEN_VECTOR);
    }

    if (used_malloc)
	free (pixel);
}



/*
**  Have a look at Graphics_SetAreaXYZ.
**  Should be the same code, but with grey scale than with color.
*/

void Graphics_SetgAreaXYZ
#ifdef __STDC__
(ImageIO_gray c[], INTEGER x, INTEGER y, INTEGER zoom, CONFIGURATION OPEN_VECTOR)
#else
(c, x, y, zoom, OPEN_VECTOR)
ImageIO_gray c[];
INTEGER x;
INTEGER y;
INTEGER zoom;
CONFIGURATION OPEN_VECTOR;
#endif
{
    int used_malloc = 0;
    Pixel *pixel;

    CheckForActiveWindow;

    if (zoom == -1)
	zoom = 1;

    pixel = alloca (LEN (OPEN_VECTOR) * sizeof (Pixel));

    if (!pixel)
	if ((pixel = malloc (LEN (OPEN_VECTOR) * sizeof (Pixel))))
	    used_malloc++;

    if (pixel)
    {
	register int i, pe;
	register Pixel *cp = pixel, prev_pixel = 0xaffe;
	register ImageIO_gray prev = 0xdeadbeef;
	XColor col;

	for (pe = 0, i = LEN (OPEN_VECTOR); --i >= 0; c++, cp++, pe++)
	    if (IS_ACTIVE (OPEN_VECTOR, pe))
	    {
		if (prev == *c)
		    *cp = prev_pixel;
		else
		{
		    prev = *c;
		    col.red = col.green = col.blue = (*c << 8) + *c;
		    prev_pixel = *cp = lookupColor (selectW, &col) != 0
				       ? col.pixel
				       : BlackPixelOfScreen (screen);
		}
	    }

	if (zoom > 0)
	    if (zoom == 1)
		_SetArea (pixel, x, y, OPEN_VECTOR);
	    else
		_SetAreaXYZ (pixel, x, y, zoom, OPEN_VECTOR);
	else
	    _SetAreaXYS (pixel, x, y, -zoom, OPEN_VECTOR);
    }

    if (used_malloc)
	free (pixel);
}



/*
**  You should know what I want to say: ...
*/

void Graphics_SetbAreaXYZ
#ifdef __STDC__
(ImageIO_binary c[], INTEGER x, INTEGER y, INTEGER zoom, CONFIGURATION OPEN_VECTOR)
#else
(c, x, y, zoom, OPEN_VECTOR)
ImageIO_binary c[];
INTEGER x;
INTEGER y;
INTEGER zoom;
CONFIGURATION OPEN_VECTOR;
#endif
{
    register int i;
    int used_malloc = 0;
    Pixel *pixel;

    CheckForActiveWindow;

    if (zoom == -1)
	zoom = 1;

    pixel = alloca (LEN (OPEN_VECTOR) * sizeof (Pixel));

    if (!pixel)
	if ((pixel = malloc (LEN (OPEN_VECTOR) * sizeof (Pixel))))
	    used_malloc++;

    if (pixel)
    {
	register Pixel bp = BlackPixelOfScreen (screen);
	register Pixel wp = WhitePixelOfScreen (screen);
	register Pixel *cp = pixel;

	for (i = LEN (OPEN_VECTOR); --i >= 0; )
	    *cp++ = *c++ ? bp : wp;

	if (zoom > 0)
	    if (zoom == 1)
		_SetArea (pixel, x, y, OPEN_VECTOR);
	    else
		_SetAreaXYZ (pixel, x, y, zoom, OPEN_VECTOR);
	else
	    _SetAreaXYS (pixel, x, y, -zoom, OPEN_VECTOR);
    }

    if (used_malloc)
	free (pixel);
}



/*
**  Hey, something new has to be looked at!
**  In this case no difference between CONFIGURATIONs is made, all are
**  treated as linear arrays. Start drawing pixels at the left of the
**  window and expand to the right. The result will be a horizontal
**  line with 'line' pixel distance from the top of the window.
*/

void Graphics_SetLine
#ifdef __STDC__
(ImageIO_color c[], CARDINAL line, CONFIGURATION OPEN_VECTOR)
#else
(c, line, OPEN_VECTOR)
ImageIO_color c[];
CARDINAL line;
CONFIGURATION OPEN_VECTOR;
#endif
{
    register int i, pe, x;
    XColor color;

    CheckForActiveWindow;

    /* Drawing below the window makes no sense! */
    if (line >= selectW->height)
    {
	Graphics_GraphicsError = Graphics_GrWrongCoordinates;
	return;
    }

    for (x = pe = 0, i = LEN (OPEN_VECTOR); --i >= 0; c++, pe++)
    {
	if (IS_ACTIVE (OPEN_VECTOR, pe))
	{
	    /* Oh, this code has been explained two times right now! */
	    color.red	= (c->red << 8) + c->red;
	    color.green = (c->green << 8) + c->green;
	    color.blue	= (c->blue << 8) + c->blue;
	    if (lookupColor (selectW, &color) != 0)
	    {
		XSetForeground (display, gc, color.pixel);
		XDrawPoint (display, selectW->wPointer, gc, x, line);
	    }
	}

	/* Stop drawing if fallen out of window */
	if (++x == selectW->width)
	    break;
    }

    XFlush (display);
}



/*
**  Yes, you guess right. The same as above, but with gray scale
**  colors.
*/

void Graphics_SetgLine
#ifdef __STDC__
(ImageIO_gray c[], CARDINAL line, CONFIGURATION OPEN_VECTOR)
#else
(c, line, OPEN_VECTOR)
ImageIO_gray c[];
CARDINAL line;
CONFIGURATION OPEN_VECTOR;
#endif
{
    register int i, pe, x;
    XColor color;

    CheckForActiveWindow;

    /* Line inside window? */
    if (line >= selectW->height)
    {
	Graphics_GraphicsError = Graphics_GrWrongCoordinates;
	return;
    }

    for (x = pe = 0, i = LEN (OPEN_VECTOR); --i >= 0; pe++)
    {
	if (IS_ACTIVE (OPEN_VECTOR, pe))
	{
	    color.red = color.green = color.blue = (*c << 8) + *c;
	    c++;
	    if (lookupColor (selectW, &color) != 0)
	    {
		XSetForeground (display, gc, color.pixel);
		XDrawPoint (display, selectW->wPointer, gc, x, line);
	    }
	}

	/* Stop drawing if leaving window */
	if (++x == selectW->width)
	    break;
    }

    XFlush (display);
}



/*
**  "No, not again!" you might say, but it is true. The same, but for
**  black and white.
*/

void Graphics_SetbLine
#ifdef __STDC__
(ImageIO_binary c[], CARDINAL line, CONFIGURATION OPEN_VECTOR)
#else
(c, line, OPEN_VECTOR)
ImageIO_binary c[];
CARDINAL line;
CONFIGURATION OPEN_VECTOR;
#endif
{
    register int i, pe, x;

    CheckForActiveWindow;

    /* Line inside window? */
    if (line >= selectW->height)
    {
	Graphics_GraphicsError = Graphics_GrWrongCoordinates;
	return;
    }

    XSetForeground (display, gc, BlackPixelOfScreen (screen));

    for (x = pe = 0, i = LEN (OPEN_VECTOR); --i >= 0; c++, pe++)
    {
	if (IS_ACTIVE (OPEN_VECTOR, pe))
	    if (*c != 0)
		XDrawPoint (display, selectW->wPointer, gc, x, line);

	/* fallen out of the window? */
	if (++x == selectW->width)
	    break;
    }

    XFlush (display);
}



/*
**  Like SetAreaXYZ there should be the same for SetLine (and SetColumn).
**  Just create a new configuration describing a single line (column)
**  and call SetAreaXYZ.
*/

void Graphics_SetLineZ
#ifdef __STDC__
(ImageIO_color c[], CARDINAL line, INTEGER offset, INTEGER zoom, CONFIGURATION OPEN_VECTOR)
#else
(c, line, offset, zoom, OPEN_VECTOR)
ImageIO_color c[];
CARDINAL line;
INTEGER offset;
INTEGER zoom;
CONFIGURATION OPEN_VECTOR;
#endif
{
    CONFIGURATION tmp_line_config;

    INIT_CONFIGURATION ("tmp_line_config", &tmp_line_config, 2);
    SET_BOUNDS (&tmp_line_config, 0, 0, 0);
    SET_BOUNDS (&tmp_line_config, 1, 0, LEN (OPEN_VECTOR) - 1);
    COPY_STACKS (&tmp_line_config, &OPEN_VECTOR);

    Graphics_SetAreaXYZ (c, offset, line, zoom, tmp_line_config);
    FREE_CONFIGURATION (&tmp_line_config, FALSE);
}



/*
**  Just like SetLineZ, but for gray values.
*/

void Graphics_SetgLineZ
#ifdef __STDC__
(ImageIO_gray c[], CARDINAL line, INTEGER offset, INTEGER zoom, CONFIGURATION OPEN_VECTOR)
#else
(c, line, offset, zoom, OPEN_VECTOR)
ImageIO_gray c[];
CARDINAL line;
INTEGER offset;
INTEGER zoom;
CONFIGURATION OPEN_VECTOR;
#endif
{
    CONFIGURATION tmp_line_config;

    INIT_CONFIGURATION ("tmp_line_config", &tmp_line_config, 2);
    SET_BOUNDS (&tmp_line_config, 0, 0, 0);
    SET_BOUNDS (&tmp_line_config, 1, 0, LEN (OPEN_VECTOR) - 1);
    COPY_STACKS (&tmp_line_config, &OPEN_VECTOR);

    Graphics_SetgAreaXYZ (c, offset, line, zoom, tmp_line_config);
    FREE_CONFIGURATION (&tmp_line_config, FALSE);
}



/*
**  Don't feel worried. Here is the routine for black and white pixels.
*/

void Graphics_SetbLineZ
#ifdef __STDC__
(ImageIO_binary c[], CARDINAL line, INTEGER offset, INTEGER zoom, CONFIGURATION OPEN_VECTOR)
#else
(c, line, offset, zoom, OPEN_VECTOR)
ImageIO_binary c[];
CARDINAL line;
INTEGER offset;
INTEGER zoom;
CONFIGURATION OPEN_VECTOR;
#endif
{
    CONFIGURATION tmp_line_config;

    INIT_CONFIGURATION ("tmp_line_config", &tmp_line_config, 2);
    SET_BOUNDS (&tmp_line_config, 0, 0, 0);
    SET_BOUNDS (&tmp_line_config, 1, 0, LEN (OPEN_VECTOR) - 1);
    COPY_STACKS (&tmp_line_config, &OPEN_VECTOR);

    Graphics_SetbAreaXYZ (c, offset, line, zoom, tmp_line_config);
    FREE_CONFIGURATION (&tmp_line_config, FALSE);
}



/*
**  How amazing. The above routines not for lines but for columns.
**  Do I really need to explain what's going on?
**  You should have understood the above code, so I will not comment
**  the following routines for drawing columns instead of rows.
*/

void Graphics_SetColumn
#ifdef __STDC__
(ImageIO_color c[], CARDINAL col, CONFIGURATION OPEN_VECTOR)
#else
(c, col, OPEN_VECTOR)
ImageIO_color c[];
CARDINAL col;
CONFIGURATION OPEN_VECTOR;
#endif
{
    register int i, pe, y;
    XColor color;

    CheckForActiveWindow;

    if (col >= selectW->width)
    {
	Graphics_GraphicsError = Graphics_GrWrongCoordinates;
	return;
    }

    for (y = pe = 0, i = LEN (OPEN_VECTOR); --i >= 0; c++, pe++)
    {
	if (IS_ACTIVE (OPEN_VECTOR, pe))
	{
	    color.red	= (c->red << 8) + c->red;
	    color.green = (c->green << 8) + c->green;
	    color.blue	= (c->blue << 8) + c->blue;
	    if (lookupColor (selectW, &color) != 0)
	    {
		XSetForeground (display, gc, color.pixel);
		XDrawPoint (display, selectW->wPointer, gc, col, y);
	    }
	}

	if (++y == selectW->height)
	    break;
    }

    XFlush (display);
}



void Graphics_SetgColumn
#ifdef __STDC__
(ImageIO_gray c[], CARDINAL col, CONFIGURATION OPEN_VECTOR)
#else
(c, col, OPEN_VECTOR)
ImageIO_gray c[];
CARDINAL col;
CONFIGURATION OPEN_VECTOR;
#endif
{
    register int i, pe, y;
    XColor color;

    CheckForActiveWindow;

    if (col >= selectW->width)
    {
	Graphics_GraphicsError = Graphics_GrWrongCoordinates;
	return;
    }

    for (y = pe = 0, i = LEN (OPEN_VECTOR); --i >= 0; pe++)
    {
	if (IS_ACTIVE (OPEN_VECTOR, pe))
	{
	    color.red = color.green = color.blue = (*c << 8) + *c;
	    c++;
	    if (lookupColor (selectW, &color) != 0)
	    {
		XSetForeground (display, gc, color.pixel);
		XDrawPoint (display, selectW->wPointer, gc, col, y);
	    }
	}

	if (++y == selectW->height)
	    break;
    }

    XFlush (display);
}



void Graphics_SetbColumn
#ifdef __STDC__
(ImageIO_binary c[], CARDINAL col, CONFIGURATION OPEN_VECTOR)
#else
(c, col, OPEN_VECTOR)
ImageIO_binary c[];
CARDINAL col;
CONFIGURATION OPEN_VECTOR;
#endif
{
    register int i, pe, y;

    CheckForActiveWindow;

    if (col >= selectW->width)
    {
	Graphics_GraphicsError = Graphics_GrWrongCoordinates;
	return;
    }

    XSetForeground (display, gc, BlackPixelOfScreen (screen));

    for (y = pe = 0, i = LEN (OPEN_VECTOR); --i >= 0; c++, pe++)
    {
	if (IS_ACTIVE (OPEN_VECTOR, pe))
	    if (*c != 0)
		XDrawPoint (display, selectW->wPointer, gc, col, y);

	if (++y == selectW->height)
	    break;
    }

    XFlush (display);
}



void Graphics_SetColumnZ
#ifdef __STDC__
(ImageIO_color c[], CARDINAL col, INTEGER offset, INTEGER zoom, CONFIGURATION OPEN_VECTOR)
#else
(c, col, offset, zoom, OPEN_VECTOR)
ImageIO_color c[];
CARDINAL col;
INTEGER offset;
INTEGER zoom;
CONFIGURATION OPEN_VECTOR;
#endif
{
    CONFIGURATION tmp_col_config;

    INIT_CONFIGURATION ("tmp_col_config", &tmp_col_config, 2);
    SET_BOUNDS (&tmp_col_config, 0, 0, LEN (OPEN_VECTOR) - 1);
    SET_BOUNDS (&tmp_col_config, 1, 0, 0);
    COPY_STACKS (&tmp_col_config, &OPEN_VECTOR);

    Graphics_SetAreaXYZ (c, col, offset, zoom, tmp_col_config);
    FREE_CONFIGURATION (&tmp_col_config, FALSE);
}



void Graphics_SetgColumnZ
#ifdef __STDC__
(ImageIO_gray c[], CARDINAL col, INTEGER offset, INTEGER zoom, CONFIGURATION OPEN_VECTOR)
#else
(c, col, offset, zoom, OPEN_VECTOR)
ImageIO_gray c[];
CARDINAL col;
INTEGER offset;
INTEGER zoom;
CONFIGURATION OPEN_VECTOR;
#endif
{
    CONFIGURATION tmp_col_config;

    INIT_CONFIGURATION ("tmp_col_config", &tmp_col_config, 2);
    SET_BOUNDS (&tmp_col_config, 0, 0, LEN (OPEN_VECTOR) - 1);
    SET_BOUNDS (&tmp_col_config, 1, 0, 0);
    COPY_STACKS (&tmp_col_config, &OPEN_VECTOR);

    Graphics_SetgAreaXYZ (c, col, offset, zoom, tmp_col_config);
    FREE_CONFIGURATION (&tmp_col_config, FALSE);
}



void Graphics_SetbColumnZ
#ifdef __STDC__
(ImageIO_binary c[], CARDINAL col, INTEGER offset, INTEGER zoom, CONFIGURATION OPEN_VECTOR)
#else
(c, col, offset, zoom, OPEN_VECTOR)
ImageIO_binary c[];
CARDINAL col;
INTEGER offset;
INTEGER zoom;
CONFIGURATION OPEN_VECTOR;
#endif
{
    CONFIGURATION tmp_col_config;

    INIT_CONFIGURATION ("tmp_col_config", &tmp_col_config, 2);
    SET_BOUNDS (&tmp_col_config, 0, 0, LEN (OPEN_VECTOR) - 1);
    SET_BOUNDS (&tmp_col_config, 1, 0, 0);
    COPY_STACKS (&tmp_col_config, &OPEN_VECTOR);

    Graphics_SetbAreaXYZ (c, col, offset, zoom, tmp_col_config);
    FREE_CONFIGURATION (&tmp_col_config, FALSE);
}



/*
**  Start a request for the color used in the subsequent drawing
**  commands. This is a scalar routine, requiring only one color!
*/

void Graphics_SetColor
#ifdef __STDC__
(ImageIO_color c)
#else
(c)
ImageIO_color c;
#endif
{
    XColor col;

    CheckForActiveWindow;

    col.red   = (c.red << 8) + c.red;
    col.green = (c.green << 8) + c.green;
    col.blue  = (c.blue << 8) + c.blue;
    if (lookupColor (selectW, &col) != 0)
	selectW->pixel = col.pixel;
    else
	Graphics_GraphicsError = Graphics_GrWrongColors;
}



/*
**  Set a gray scale color for subsequent drawing commands.
*/

void Graphics_SetgColor
#ifdef __STDC__
(ImageIO_gray c)
#else
(c)
ImageIO_gray c;
#endif
{
    ImageIO_color color;

    color.red = color.green = color.blue = (c << 8) + c;
    Graphics_SetColor (color);
}



/*
**  Handle black and white request.
*/

void Graphics_SetbColor
#ifdef __STDC__
(ImageIO_binary c)
#else
(c)
ImageIO_binary c;
#endif
{
    Graphics_SetColor (c ? ImageIO_c_black : ImageIO_c_white);
}



/*
**  Just a little action, drawing a single point in the color set by
**  Graphics_Set[gb]Color().
*/

void Graphics_SetPixel
#ifdef __STDC__
(CARDINAL x, CARDINAL y)
#else
(x, y)
CARDINAL x;
CARDINAL y;
#endif
{
    CheckForActiveWindow;

    XSetForeground (display, gc, selectW->pixel);
    XDrawPoint (display, selectW->wPointer, gc, x, y);
    XFlush (display);
}



/*
**  The other way round. Get the value of the pixel at position (x,y).
**  Only X11 color translation is performed, no gray scale or even
**  black and white scaling.
*/

ImageIO_color Graphics_GetPixel
#ifdef __STDC__
(CARDINAL x, CARDINAL y)
#else
(x, y)
CARDINAL x;
CARDINAL y;
#endif
{
    ImageIO_color rgb_pixel;
    XImage  *pixel;

    rgb_pixel = ImageIO_c_white;

    /* If no window is active, return white (the standard background
       color). */
    CheckForActiveWindowReturn (rgb_pixel);

    /* Requested position inside window? */
    if (x >= selectW->width || y >= selectW->height)
    {
	Graphics_GraphicsError = Graphics_GrWrongCoordinates;
	return rgb_pixel;
    }

    /* Call X and have a look at the values */
    if (!(pixel = XGetImage (display, selectW->wPointer,
			     x, y, 1, 1, AllPlanes, XYPixmap)))
	Graphics_GraphicsError = Graphics_GrWrongCoordinates;
    else
    {
	XColor color;

	color.pixel = XGetPixel (pixel, 0, 0);
	XQueryColor (display, selectW->cmap, &color);
	rgb_pixel.red	= color.red / 256;
	rgb_pixel.green = color.green / 256;
	rgb_pixel.blue	= color.blue / 256;
	XDestroyImage (pixel);
    }

    /* If all failed, rgb_pixel is still ImageIO_c_white. */
    return rgb_pixel;
}



/*
**  Draw an ambigious line. Done by the front end. No parallel
**  mechanisms used.
*/

void Graphics_Line
#ifdef __STDC__
(CARDINAL x1, CARDINAL y1, CARDINAL x2, CARDINAL y2)
#else
(x1, y1, x2, y2)
CARDINAL x1, y1, x2, y2;
#endif
{
    CheckForActiveWindow;

    /* Set new foreground, given with Graphics_Set[gb]Color. */
    XSetForeground (display, gc, selectW->pixel);
    XDrawLine (display, selectW->wPointer, gc,
	       x1, y1, x2, y2);
    XFlush (display);
}



/*
**  Set new drawing position. With no active window, do nothing, but
**  indicate an error.
*/

void Graphics_DrawAt
#ifdef __STDC__
(CARDINAL x, CARDINAL y)
#else
(x, y)
CARDINAL x, y;
#endif
{
    CheckForActiveWindow;

    selectW->x = x;
    selectW->y = y;
}



/*
**  All the Graphics_Draw... routines act like the PASCAL, Modula-2
**  and Parallaxis 'Write' operations. For a complete syntax refer to
**  the corresponding manuals (anyway, you should always RTFM).
*/

/*
**  Print a single character. Done by translating the char into a
**  string and calling DrawString().
*/

void Graphics_Draw
#ifdef __STDC__
(CHAR c)
#else
(c)
CHAR c;
#endif
{
    CHAR str[2];

    str [0] = c;
    str [1] = '\0';

    Graphics_DrawString (str, 2);
}



/*
**  This function does the real work. A string is drawn at the current
**  drawing position, string length is computed by X using the standard
**  font description. The string length/width is added to the horizontal
**  drawing position. There is no wrap at the right side of the
**  window. Responsibility for full visibility of text must be managed by
**  the user.
*/

void Graphics_DrawString
#ifdef __STDC__
(CHAR s[], CARDINAL O_2)
#else
(s, O_2)
CHAR s[];
CARDINAL O_2;
#endif
{
    int dummy;
    int xStringDirection;
    XCharStruct	xStringLen;

    CheckForActiveWindow;

    XSetForeground (display, gc, selectW->pixel);
    XDrawString (display, selectW->wPointer, gc,
		 selectW->x, selectW->y + fontStruct->ascent, s, strlen (s));
    XTextExtents (fontStruct, s, strlen (s),
		  &xStringDirection, &dummy, &dummy, &xStringLen);
    dummy = xStringLen.width;
    if (xStringDirection == FontRightToLeft)
	dummy = -dummy;
    selectW->x += dummy;

    XFlush (display);
}



/*
**  Transfer a INTEGER into a string via sprintf and call DrawString.
**  No additional efforts in making the code fast. This code is small
**  and easy to maintain.
*/

void Graphics_DrawInt
#ifdef __STDC__
(INTEGER i, INTEGER l)
#else
(i, l)
INTEGER i, l;
#endif
{
    CHAR str[256];

    if (l < numberof (str))
    {
	sprintf (str, "%*ld", (int) l, i);
	Graphics_DrawString (str, numberof (str));
    }
}



/*
**  Write a CARDINAL (unsigned int) the way used for INTEGERs.
*/

void Graphics_DrawCard
#ifdef __STDC__
(CARDINAL c, CARDINAL l)
#else
(c, l)
CARDINAL c, l;
#endif
{
    CHAR str[256];

    if (l < numberof (str))
    {
	sprintf (str, "%*lu", (int) l, c);
	Graphics_DrawString (str, numberof (str));
    }
}



/*
**  Do it with REAL (float or double) values.
*/

void Graphics_DrawReal
#ifdef __STDC__
(REAL r, CARDINAL l)
#else
(r, l)
REAL r;
CARDINAL l;
#endif
{
    CHAR str[256];

    if (l < numberof (str))
    {
	sprintf (str, "%*f", (int) l, r);
	Graphics_DrawString (str, numberof (str));
    }
}



/*
**  The same as above, but in fixed point representation, means in a
**  predefined format. DrawReal() writes the value unformatted.
*/

void Graphics_DrawFixPt
#ifdef __STDC__
(REAL r, CARDINAL l, CARDINAL m)
#else
(r, l, m)
REAL r;
CARDINAL l, m;
#endif
{
    CHAR str[256];

    if (l < numberof (str))
    {
	sprintf (str, "%*.*f", (int) l, (int) m, r);
	Graphics_DrawString (str, numberof (str));
    }
}



/*
**  And finally, do a verbose output for boolean values. Standard text
**  is TRUE and FALSE, all in upper case.
*/

void Graphics_DrawBool
#ifdef __STDC__
(BOOLEAN b, CARDINAL l)
#else
(b, l)
BOOLEAN b;
CARDINAL l;
#endif
{
    CHAR str[256];

    if (l < numberof (str))
    {
	sprintf (str, "%-*s", (int) l, b ? "TRUE" : "FALSE");
	Graphics_DrawString (str, numberof (str));
    }
}



/*
**  Force a line wrap!
**  Could be done with DrawAt(), but makes no sense for such a short
**  function to call another short function.
*/

void Graphics_DrawLn
#ifdef __STDC__
(void)
#else
()
#endif
{
    /*
    **	could also be done via
    **	DrawAt (0, selectW->y + fontStruct->ascent + fontStruct->descent);
    */

    CheckForActiveWindow;

    selectW->x = 0;
    selectW->y += fontStruct->ascent + fontStruct->descent;
}



/*
**  Module initialization phase. Must be called before any Graphics_...()
**  call. For sanity, OpenWindow() checks for initialized Graphics, and
**  calls init_x() on demand. This should never happen, but who knows!
**  BEGIN_Graphics() should be called by the main of the Parallaxis program.
*/

void BEGIN_Graphics
#ifdef __STDC__
(void)
#else
()
#endif
{
    static BOOLEAN isInitialised = FALSE;

    if (!isInitialised)
    {
	isInitialised = TRUE;

	init_x ();
    }
}



/*
**  Initialize the X window connection. Query for global values and store
**  them. Also sets the font used in drawing. The default color cache will
**  also be initialized.
*/
static int init_x
#ifdef __STDC__
(void)
#else
()
#endif
{
    XGCValues gc_values;
    char *fontName;
    Font font;
    char *displayName = getenv ("DISPLAY");
#ifndef NO_COLOR_CACHE
    register int i;
#endif

    if (!(display = XOpenDisplay (displayName)))
    {
	fprintf (stderr, "Can't open display '%s'\n", displayName);
	Graphics_GraphicsError = Graphics_GrWindowCreationFailure;
	return -1;
    }

    screen = DefaultScreenOfDisplay (display);

    screenwidth = WidthOfScreen (screen);
    screenheight = HeightOfScreen (screen);
    gc = DefaultGC (display, DefaultScreen (display));

    /* If other font than default-font should be used,
       set actual font to it */
    if ((fontName = (char *) getenv (FONT_ENV_VAR)) == NULL)
	fontName = "fixed";
    if ((font = XLoadFont (display, fontName)) != (Font) 0)
	XSetFont (display, gc, font);
    else
	fprintf (stderr, "Warning: font %s couldn't be loaded\n", fontName);

    XGetGCValues (display, gc, GCFont, &gc_values);
    fontStruct = XQueryFont (display, gc_values.font);

    maxW.width	= screenwidth;
    maxW.height = screenheight;

#ifndef NO_COLOR_CACHE
#define BLACK (255 << 8) + 255
    /* Initialize default color cache with black and white */
    i = numberof (defColorCache);

    --i;
    defColorCache[i].red = defColorCache[i].green = defColorCache[i].blue = BLACK;
    defColorCache[i].pixel = WhitePixelOfScreen (screen);

    --i;
    defColorCache[i].red = defColorCache[i].green = defColorCache[i].blue = 0;
    defColorCache[i].pixel = BlackPixelOfScreen (screen);

    defColorSlot = i - 1;

    while (--i >= 0)
	defColorCache[i].pixel = BlackPixelOfScreen (screen);
#undef BLACK
#endif

    return 1;
}



/*
**  Because XAllocColor() is a real bad guy consuming a lot of time,
**  and because we are using read-only colors, a small cache is installed.
**  So, if an already used color is requested there is a cache hit, and
**  the server needs not to be asked for the pixel value.
**  The cache is realized as a linear array. Searching is also
**  linear. I didn't know how to sort RGB values and apply binary
**  search. When the cache is not filled, linear searching is faster than
**  doing a bsearch(). Maybe in a future version I will change it.
**
**  When XAllocColor() fails, then have a look at the current colormap loaded
**  and search for the closest color found in the table.
**
**  return 0 on failure
**	   1 on success
*/

static int lookupColor
#ifdef __STDC__
(struct WindowList *win, XColor *color)
#else
(win, color)
struct WindowList *win;
XColor *color;
#endif
{
#ifdef NO_COLOR_CACHE
    return XAllocColor (display, win->cmap, color);
#else
#define r color->red
#define g color->green
#define b color->blue
    register int i = COLOR_CACHE_ENTRIES;
    int closest;
    unsigned long min_cdist, cdist;
    XColor current_cmap[256];
    Colormap cmap = win->cmap;
    XColor *colorCache = win->colorCache;
    int *colorSlot = win->colorSlot;

    if (colorCache)
    {
	while (--i > *colorSlot)
	{
	    if (colorCache[i].red == color->red
		&& colorCache[i].green == color->green
		&& colorCache[i].blue == color->blue)
	    {
		color->pixel = colorCache[i].pixel;
		return 1;
	    }
	}

	colorCache[*colorSlot] = *color; /* remember requested RGB, not real! */

	if (XAllocColor (display, cmap, color) != 0)
	{
	    colorCache[*colorSlot].pixel = color->pixel;
#ifdef DEBUG
	    printf ("slot %3d: color (%3d/%3d/%3d) = %lu\n",
		    *colorSlot,
		    colorCache[*colorSlot].red,
		    colorCache[*colorSlot].green,
		    colorCache[*colorSlot].blue,
		    colorCache[*colorSlot].pixel);
#endif
	    if (*colorSlot != 0)	/* cache full? */
		(*colorSlot)--;		/* next overriding current result */

	    return 1;
	}
    }
    else
    {
	if (XAllocColor (display, cmap, color) != 0)
	    return 1;
    }

    /* Get current color map */
    for (i = numberof (current_cmap); --i >= 0; )
	current_cmap[i].pixel = i;
    XQueryColors (display, cmap, current_cmap, numberof (current_cmap));

    /* now look for the color closest to the desired one */
    min_cdist = 200000L;
    closest = -1;
    for (i = numberof (current_cmap); --i >= 0; )
    {
	cdist = abs ((r >> 8) - (current_cmap[i].red >> 8))
		+ abs ((g >> 8) - (current_cmap[i].green >> 8))
		+ abs ((b >> 8) - (current_cmap[i].blue >> 8));
	if (cdist < min_cdist)
	    min_cdist = cdist, closest = i;
    }

    if (closest != -1)
    {
	colorCache[*colorSlot].pixel
	  = color->pixel = current_cmap[closest].pixel;
        if (*colorSlot != 0)
	    (*colorSlot)--;

	return 1;
    }

    return 0;
#undef r
#undef g
#undef b
#endif
}



/*
**  Fill the window with values given in a linear array. If 'Config' tells
**  that a rectangle is used, draw as rectangle. All other CONFIGURATIONs
**  are treated as linear field.
*/

static void _SetArea
#ifdef __STDC__
(Pixel *c, int x, int y, CONFIGURATION Config)
#else
(c, x, y, Config)
Pixel *c;
int x;
int y;
CONFIGURATION Config;
#endif
{
    register int i, pe;
    int xsave, skip;

    /* how many values can be skipped, when drawing is outside of
       window? Should be faster, than looping without drawing, or
       drawing but clipped by X window */
    skip = Config.Rank >= 2 ? Config.DimLen[1] - selectW->width + x : 0;

    for (xsave = x, pe = 0, i = LEN (Config); --i >= 0; c++, pe++)
    {
	if (IS_ACTIVE (Config, pe))
	{
	    XSetForeground (display, gc, *c);
	    XDrawPoint (display, selectW->wPointer, gc, x, y);
	}

	if (++x == selectW->width)
	{
	    /* crossed the border, so skip values and increase y-position */
	    c += skip;
	    pe += skip;
	    i -= skip;
	    x = xsave;
	    if (++y == selectW->height)
		break;
	}
	else if (Config.Rank >= 2 && x - xsave == Config.DimLen[1])
	{
	    /* window is larger than field, so wrap here, no skipping
	       required! */
	    x = xsave;
	    if (++y == selectW->height)
		break;
	}
    }

    XFlush (display);
}



/*
**  Enlarge Area by factor 'zoom', starting at offset 'x/y'.
**  This function in called from Set[gb]AreaXYZ() if zooming is
**  requested.
**  'c' contains translated color information for each PE.
*/
static void _SetAreaXYZ
#ifdef __STDC__
(Pixel *c, int x, int y, int zoom, CONFIGURATION Config)
#else
(c, x, y, zoom, Config)
Pixel *c;
int x;
int y;
int zoom;
CONFIGURATION Config;
#endif
{
    register int i, pe;
    int xsave, skip, border;

    if (Config.Rank >= 2)
	border = Config.DimLen[1] * zoom,
	skip = (border - selectW->width + x) / zoom;
    else
	border = selectW->width,
	skip = 0;

    for (xsave = x, pe = 0, i = LEN (Config); --i >= 0; c++, pe++)
    {
	if (IS_ACTIVE (Config, pe))
	{
	    XSetForeground (display, gc, *c);
	    /* Using filled rectangles is faster than drawing zoom^2
	       single dots. Nice trick for speedup. */

	    XFillRectangle (display, selectW->wPointer, gc,
			    x, y, zoom, zoom);
	}

	x += zoom;

	/* can skip some pixels if wrapping happens */
	if (x >= selectW->width)
	{
	    c += skip;
	    pe += skip;
	    i -= skip;
	    x = xsave;
	    y += zoom;
	    if (y >= selectW->height)
		break;
	}
	else if (x - xsave >= border)
	{
	    /* no wrap, but pixel line ends here, so switch to next
	       line */
	    x = xsave;
	    y += zoom;
	    if (y >= selectW->height)
		break;
	}
    }

    XFlush (display);
}



/*
**  Shrink Area by factor zoom, starting at offset x/y.
**  Shrinking is done the following way:
**	shrink == 2 : print each 2nd point
**	shrink == 3 : print each 3rd point
**	and so on
**  Attention: shrink is always positive!
**  'c' contains translated color information for each PE.
*/
static void _SetAreaXYS
#ifdef __STDC__
(Pixel *c, int x, int y, int shrink, CONFIGURATION Config)
#else
(c, x, y, shrink, Config)
Pixel *c;
int x;
int y;
int shrink;
CONFIGURATION Config;
#endif
{
    register int i, pe;
    int xsave, skip, border;

    /* Skipping is a little bit more complex in shrink mode. Think
       about it in a silent minute. You should get the same code.
       When skipping, lines can be skipped complete. This is the
       term added to 'skip'. */
    if (Config.Rank >= 2)
	border = Config.DimLen[1] / shrink,
	skip = Config.DimLen[1] - (selectW->width - x) * shrink - shrink,
	skip += Config.DimLen[1] * shrink;
    else
	border = selectW->width,
	skip = 0;

    for (xsave = x, pe = 0, i = LEN (Config);
	 i > 0;
	 i -= shrink, c += shrink, pe += shrink)
    {
	if (IS_ACTIVE (Config, pe))
	{
	    XSetForeground (display, gc, *c);
	    XDrawPoint (display, selectW->wPointer, gc, x, y);
	}

	/* should skip pixels? */
	if (++x >= selectW->width)
	{
	    c += skip;
	    pe += skip;
	    i -= skip;
	    x = xsave;
	    if (++y == selectW->height)
		break;
	}
	else if (x >= border)
	{
	    /* scan line ends here, skip next 'shrink' lines */
	    c += Config.DimLen[1] * shrink;
	    pe += Config.DimLen[1] * shrink;
	    i -= Config.DimLen[1] * shrink;
	    x = xsave;
	    if (++y == selectW->height)
		break;
	}
    }

    XFlush (display);
}
