IMPLEMENTATION MODULE nn;
(* Neural Network Simulator                                *)
(* Thomas Braunl, Univ Stuttgart '95                       *)
(* further read.: Zell: Simulation Neuronaler Netze, 1994  *)
 
 
PROCEDURE activate(x: weights OF REAL): weights OF REAL;
BEGIN (* logistic activation function *)
  RETURN 1.0 / (1.0 + exp(-x))
END activate;


PROCEDURE InitWeights(VAR strength1, strength2: weights OF REAL;
                      insize, outsize: INTEGER);
(* random initialization of connection weights *)
BEGIN
  IF (DIM(weights,2) > insize+1) OR (DIM(weights,1) = UPPER(weights,1))
    THEN strength1 := 0.0
    ELSE strength1 := RandomReal(weights) / 2.0;
  END;
  IF DIM(weights,2) > outsize
    THEN strength2 := 0.0
    ELSE strength2 := RandomReal(weights) / 2.0;
  END;
END InitWeights;


PROCEDURE NetStep (VAR inval, hidval, outval: weights OF REAL;
                   insize, outsize: INTEGER;
                   strength1, strength2: weights OF REAL);
(* single step execution of neural data propagation *)
VAR s1val, s2val: weights OF REAL;
BEGIN  (* single step execution of neural data propagation *)
  IF DIM(weights,1) = 1 THEN                       (* first column only *)
    inval<:insize+1, 1:> := 1.0;                   (* set "on" neuron *)
    SEND.forward1     (inval, s1val);              (* broadcast *)
  END;
  IF DIM(weights,2) <= insize+1 THEN
    SEND.forward2:#SUM(s1val*strength1, hidval);   (* reduce *)
  END;
  IF DIM(weights,2) = 1 THEN                       (* first row only *)
    hidval := activate(hidval);
    hidval<:1,UPPER(weights,1):> := 1.0;           (* set "on" neuron *)
    SEND.forward3     (hidval, s2val);             (* broadcast *)
  END;
  IF DIM(weights,2) <= outsize THEN
    SEND.forward4:#SUM(s2val*strength2, outval);   (* reduce *)
  END;
  IF DIM(weights,1) = 1 THEN                       (* first column only *)
    outval := activate(outval);
  END;
END NetStep;

 

PROCEDURE BackProp(VAR strength1, strength2: weights OF REAL;
                   insize, outsize: INTEGER;
                   SetPattern: patproc; maxpat: INTEGER;
                   eta: REAL; VAR totalerr: REAL; debug: BOOLEAN);
(* error back-propagatipon algorithm                                     *)
(* Proc. SetPattern provides maxpat training patterns (input and output) *)
VAR pattern                                    : INTEGER;
    inval,hidval,outval, teach,diffout,outerr, diffhid,
    diff2,hid2,delta2,   diff1,hid1,delta1,sen1: weights OF REAL;
BEGIN
  totalerr := 0.0;
  delta1   := 0.0;
  delta2   := 0.0;
  FOR pattern:=0 TO maxpat-1 DO
    SetPattern(pattern,inval,teach);
    NetStep(inval,hidval,outval,insize,outsize,strength1,strength2);
    IF (DIM(weights,2) <= outsize) AND (DIM(weights,1) = 1) THEN
      outerr   := teach - outval;
      diffout  := outerr * (1.0 - outval) * outval;
      totalerr := totalerr + REDUCE.SUM(outerr*outerr);
      SEND.back4(diffout, diff2);
    END;
    (* change weights in layer 2 *)
    IF DIM(weights,2) = 1 THEN
      SEND.forward3(hidval, hid2);
    END;
    IF DIM(weights,2) <= outsize THEN
      delta2 := eta * hid2 * diff2;
      SEND.back3:#SUM(strength2 * diff2, diffhid);
    END;

    (* change weights in layer 1 *)
    IF DIM(weights,2) = 1 THEN
      SEND.back2     (diffhid, diff1);
      SEND.back2     (hidval, hid1);
    END;
    IF (DIM(weights,2) <= insize+1) AND (DIM(weights,1) = 1) THEN
      SEND.forward1  (inval, sen1);
    END;
    IF DIM(weights,2) <= insize+1 THEN
      delta1 := eta * sen1 * hid1 * (1.0-hid1) * diff1;
    END;

    strength2 := strength2 + delta2;
    strength1 := strength1 + delta1;

    IF debug THEN 
      WriteString("BackProp  "); WriteLn;                         
      WriteString("input :   "); WriteLn; WriteFixPt(inval,6,2);
      WriteString("hidden:   "); WriteLn; WriteFixPt(hidval,6,2);
      WriteString("output:   "); WriteLn; WriteFixPt(outval,6,2);
      WriteString("teach :   "); WriteLn; WriteFixPt(teach,6,2);
      WriteString("total :   "); WriteLn; WriteFixPt(totalerr,6,2); WriteLn;
      WriteString("delta1:   "); WriteLn; WriteFixPt(delta1,6,3);
      WriteString("delta2:   "); WriteLn; WriteFixPt(delta2,6,3);
      WriteString("strength1:"); WriteLn; WriteFixPt(strength1,6,3);
      WriteString("strength2:"); WriteLn; WriteFixPt(strength2,6,3);
    END;
  END; (* for *)
  totalerr  := totalerr / FLOAT(maxpat);
END BackProp;

END nn.
