MODULE nn_7seg;
(* Neural Network Simulator                                          *)
(* Thomas Braunl, Univ Stuttgart '95                                 *)
(* debug modus: command line -debug                                  *)
(* application: learning the coding of a 7-segment numerical display *)

FROM nn IMPORT weights, InitWeights, NetStep, BackProp;
 
CONST MaxInput   =     7; (* number of neurons          *)
      MaxHidden  =     5; (* number of neurons          *)
      MaxOutput  =    10; (* number of neurons          *)
      MaxPattern =    10; (* number of trainig patterns *)
      MaxSteps   =  1000; (* training steps             *)
      eps        =  0.10; (* max error                  *)
      eta        =  1.00; (* connection strength change *)

CONFIGURATION w = weights[1..MaxOutput],[1..MaxHidden+1];
(* rows   : 1 .. max(MaxInput+1, MaxOutput) *)
(* columns: 1 .. MaxHidden+1                *)

TYPE pattype = ARRAY[0..MaxPattern-1],[0..MaxInput-1] OF REAL; 

VAR strength1,strength2,teach: w OF REAL;
    sensor,hidval,outval,verr: w OF REAL;
    step,pos                 : INTEGER;
    debug                    : BOOLEAN;
    err,maxout               : REAL;
    pool                     : pattype;
    c1,c2                    : ARRAY[1..2] OF CHAR;


PROCEDURE ReadData(VAR pool: pattype);
(* read patterns from file *)
VAR pat,pix,i: INTEGER;
BEGIN
  OpenInput("segment.pat");
  IF NOT Done THEN
    WriteString("error reading data file 'segment.pat'");
    WriteLn; HALT;
  END;
  FOR pat := 0 TO MaxPattern-1 DO
    FOR pix := 0 TO MaxInput-1 DO
      ReadInt(i);
      IF NOT Done THEN
        WriteString("error reading data file 'segment.pat'");
        WriteLn; HALT;
      END;
      pool[pat,pix] := FLOAT(i);
    END;
  END;
  CloseInput;
END ReadData;


PROCEDURE SetPattern(pat: INTEGER; VAR val,teach: w OF REAL);
(* set learning pattern from GLOBAL pool *)
BEGIN
  val   := 0.0;
  teach := 0.0;
  IF DIM(w,1) = 1 THEN  (* first column *)
    LOAD(val,pool[pat]);
    teach := FLOAT(ORD(DIM(w,2) = pat+1)); (* 1..10 *)
  END;
END SetPattern;


PROCEDURE ShowPattern(val: w OF REAL);
BEGIN
  WriteLn;
  IF val<:1,1:> > 0.5 THEN WriteString(" - ") END; WriteLn;
  IF val<:2,1:> > 0.5 THEN Write("|") ELSE Write(" ") END; Write(" ");
  IF val<:3,1:> > 0.5 THEN Write("|") END; WriteLn;
  IF val<:4,1:> > 0.5 THEN WriteString(" - ") END; WriteLn;
  IF val<:5,1:> > 0.5 THEN Write("|") ELSE Write(" ") END; Write(" ");
  IF val<:6,1:> > 0.5 THEN Write("|") END; WriteLn;
  IF val<:7,1:> > 0.5 THEN WriteString(" - ") END; WriteLn;
END ShowPattern;


BEGIN  (* main program *)
  WriteString('Start NN Simulator ');
  debug := argc() > 0;
  IF debug THEN WriteString('debug on') END;
  WriteLn;
  ReadData(pool);
  WriteString('training (t) or load weights (l) ? : ');
  ReadString(c1);
  IF CAP(c1[1])#'T'
   THEN
     (* ready input from file *)
     OpenInput("");
     ReadReal(strength1);
     IF NOT Done THEN 
       WriteString("error reading data file"); WriteLn; HALT;
     END;
     ReadReal(strength2);
     IF NOT Done THEN 
       WriteString("error reading data file"); WriteLn; HALT;
     END;
     CloseInput;
     IF debug THEN WriteFixPt(strength1,8,3);
                   WriteFixPt(strength2,8,3);
     END;
   ELSE
     (* training from scratch *)
     WriteString('start training'); WriteLn;
     InitWeights(strength1, strength2, MaxInput, MaxOutput);
     IF debug THEN WriteFixPt(strength1,8,3);
                   WriteFixPt(strength2,8,3);
     END;

     WriteString('Learning Phase'); WriteLn;
     step := 0;
     REPEAT
       INC(step);
       BackProp(strength1, strength2, MaxInput, MaxOutput,
                SetPattern, MaxPattern, eta, err, debug);
       IF step MOD 10 = 0 THEN
         WriteString("step: ");    WriteInt(step,4);
         WriteString("  error: "); WriteFixPt(err,6,3); WriteLn;
       END;
     UNTIL (err < eps) OR (step >= MaxSteps);

     WriteString("do you want to save weights (y/n) ? : ");
     ReadString(c2);
     IF CAP(c2[1])#'N' THEN
       OpenOutput("");
       WriteReal(strength1, 20); WriteLn;
       WriteReal(strength2, 20); WriteLn;
       CloseOutput;
     END; (* if c2 *)
   END; (* if c1 *)
 
  WriteString('Testing Phase'); WriteLn;
  FOR step := 0 TO MaxPattern - 1 DO
    SetPattern(step, sensor, teach);
    ShowPattern(sensor);
    NetStep(sensor,hidval,outval, MaxInput,MaxOutput, strength1,strength2);
    IF (DIM(w,2) <= MaxOutput) AND (DIM(w,1) = 1) THEN
      verr   := teach - outval;
      err    := REDUCE.SUM(verr*verr);
      maxout := REDUCE.MAX(outval);
      IF outval=maxout THEN pos := REDUCE.FIRST(DIM(w,2)) - 1 END;
    END;
    WriteString("winner: "); WriteInt(pos,3); 
    IF pos = step THEN WriteString(" ** OK **")
                  ELSE WriteString(" -- no --")
    END;
    WriteLn;
    WriteString("outval: "); WriteLn;
    IF DIM(w,1) = 1 THEN WriteFixPt(outval,6,2) END;
    WriteString("error : "); WriteFixPt(err,6,2); WriteLn;
  END;

  WriteString('Application Phase'); WriteLn;
  REPEAT
    WriteString("continue (y/n) ? ");
    ReadString(c1);
    IF CAP(c1[1])#'N' THEN
      WriteString('Enter input neuron values: ');
      IF (DIM(w,2) <= MaxInput) AND (DIM(w,1) = 1) THEN
        ReadReal(sensor)
      END;
      ShowPattern(sensor);
      NetStep(sensor,hidval,outval, MaxInput,MaxOutput, strength1,strength2);
      IF (DIM(w,2) <= MaxOutput) AND (DIM(w,1) = 1) THEN
        maxout := REDUCE.MAX(outval);
        IF outval=maxout THEN pos := REDUCE.FIRST(DIM(w,2)) - 1 END;
      END;
      WriteString("winner: "); WriteInt(pos,3); WriteLn;
      WriteString("outval: "); WriteLn;
      IF DIM(w,1) = 1 THEN WriteFixPt(outval,6,2) END;
    END;
  UNTIL CAP(c1[1])='N';
  WriteString('Stop  NN Simulator'); WriteLn;
END nn_7seg.

