MODULE ga;
(* Genetic Algorithm Simulation, Thomas Braunl Univ. Stuttgart 1994 *)
CONST pop_size  =     40;
      gen_size  =     10;
      max_iter  =    100;
      fit_thres = (2**gen_size - 7)**2;
      debug     =  FALSE;

CONFIGURATION pop [0..pop_size-1];
CONNECTION    left: pop[i] <-> pop[(i+1) MOD pop_size]: right;

TYPE gen = ARRAY[1..gen_size] OF BOOLEAN;

VAR gen_pool, new_pool      : pop OF gen;
    fitness                 : pop OF INTEGER;
    best_gen                : gen;
    min_fit, max_fit, sum_fit, avg_fit,
    iter                    : INTEGER;

PROCEDURE Init(VAR g: pop OF gen);
VAR i: INTEGER;
BEGIN
  FOR i:=1 TO gen_size DO
    g[i] := RandomBool(pop);
  END;
END Init;


PROCEDURE Evaluation(VAR g: pop OF gen): pop OF INTEGER;
VAR i: INTEGER;
    f: pop OF INTEGER;
BEGIN
  f := 0;
  FOR i:=1 TO gen_size DO
    f := 2*f + ORD(g[i]);  (* binary to integer *)
  END;
  RETURN f*f;  (* square function *)
END Evaluation;


PROCEDURE Selection(VAR old_pop,sel_pop: pop OF gen; VAR fit: pop OF INTEGER);
VAR take, wheel_of_fit: pop OF INTEGER;
    fit_val, i        : INTEGER;
BEGIN
  fit_val := 0;  (* generate fitness distribtion *)
  FOR i:=0 TO pop_size-1 DO
    INC(fit_val, fitness<:i:>);
    wheel_of_fit<:i:> := fit_val
  END;
  take := RandomInt(pop) MOD fit_val;  (* 0..sum_fit-1 *)
  FOR i:=0 TO pop_size-1 DO
    IF take<:i:> < wheel_of_fit THEN  (* hit on wheel segment *)
      IF REDUCE.MIN(wheel_of_fit) = wheel_of_fit THEN
        sel_pop<:i:> := REDUCE.FIRST(old_pop)  (* select individuum *)
      END;
    END;
  END;
  IF debug THEN WriteString("after Selection"); WriteLn; Check(sel_pop) END;
END Selection;
               
            
PROCEDURE CrossOver(VAR sel_pop,new_pop: pop OF gen);
VAR sel_neighbor, new_neighbor: pop OF gen;
    point                     : pop OF INTEGER;
    i                         : INTEGER;
BEGIN
  IF EVEN(ID(pop))
    THEN SEND.left(sel_pop, sel_neighbor);
    ELSE (* only odd individuals work *)
    point := RandomInt(pop) MOD gen_size; (* cross-over point *)
    FOR i:=1 TO gen_size DO               (* 0..pop-size-1 *)
      IF i<point THEN new_pop[i]      := sel_pop[i];
                      new_neighbor[i] := sel_neighbor[i]
                 ELSE new_pop[i]      := sel_neighbor[i];
                      new_neighbor[i] := sel_pop[i]
      END;
    END;
    SEND.right(new_neighbor, new_pop);  (* update even individuals *)
  END;
  IF debug THEN WriteString("after Cross"); WriteLn; Check(new_pop) END;
END CrossOver;                             


PROCEDURE Mutation(VAR g: pop OF gen);
CONST mut_fac = 1000;
VAR i: INTEGER;
BEGIN
  FOR i:=1 TO gen_size DO
    IF RandomInt(pop) MOD mut_fac = 0 THEN
      g[i] := NOT g[i];
    END;
  END;
  IF debug THEN WriteString("after Mut"); WriteLn; Check(g) END;
END Mutation;


PROCEDURE Check(VAR g: pop OF gen);
VAR k,l: INTEGER;
BEGIN
  FOR l:=1 TO pop_size DO
    FOR k:=1 TO gen_size DO WriteBool(g<<l>>[k], 6); END;
    WriteLn;
  END;
END Check;


BEGIN (* main *)
  Init(gen_pool);
  iter := 0;
  REPEAT
    INC(iter);
    fitness := Evaluation(gen_pool);
    max_fit := REDUCE.MAX(fitness);
    min_fit := REDUCE.MIN(fitness);
    IF max_fit = fitness THEN best_gen := REDUCE.FIRST(gen_pool) END;
    sum_fit := REDUCE.SUM(fitness); 
    avg_fit := sum_fit DIV pop_size;
    WriteString("generation ");    WriteInt(iter,2);
    WriteString(" min fitness: "); WriteInt(min_fit,7); 
    WriteString("  avg fitness: "); WriteInt(avg_fit,7);
    WriteString("  max fitness: "); WriteInt(max_fit,7); WriteLn;
    IF debug THEN Check(gen_pool) END;

    Selection (gen_pool, new_pool, fitness);
    CrossOver (new_pool, gen_pool);
    Mutation  (gen_pool);
  UNTIL (iter = max_iter) OR (max_fit > fit_thres);
END ga.

