MODULE nn;
(* Neural Network Simulator                                *)
(* Thomas Braunl, Univ Stuttgart '95                       *)
(* debug modus: command line -debug                        *)
(* further read.: Zell: Simulation Neuronaler Netze, 1994  *)
 
CONST MaxInput   =     7; (* number of neurons          *)
      MaxHidden  =     5; (* number of neurons          *)
      MaxOutput  =    10; (* number of neurons          *)
      MaxPattern =    10; (* number of trainig patterns *)
      MaxSteps   =  1000; (* training steps             *)
      eps        =  0.10; (* max error                  *)
      eta        =  1.00; (* connection strength change *)

CONFIGURATION input [1..MaxInput+1];  (* "on" neuron in last pos *)
CONFIGURATION hidden[1..MaxHidden+1]; (* "on" neuron in last pos *)
CONFIGURATION output[0..MaxOutput-1];

CONFIGURATION weight1[1..MaxInput+1] ,[1..MaxHidden];
CONFIGURATION weight2[1..MaxHidden+1],[0..MaxOutput-1];

CONNECTION forward1: input  [i]    -> weight1[i,*];
           forward2: weight1[i,j] <-> {j<=MaxHidden}
                                      hidden[j]    :back2;
           forward3: hidden [k]   <-> weight2[k,*] :back3;
           forward4: weight2[k,l] <-> output[l]    :back4;

 
VAR strength1  : weight1 OF REAL;
    strength2  : weight2 OF REAL;
   
    sensor     : input  OF REAL;
    hidval     : hidden OF REAL;
    outval,verr: output OF REAL;

    step,pos   : INTEGER;
    debug      : BOOLEAN;
    err,maxout : REAL;
    ch         : ARRAY[1..2] OF CHAR;


PROCEDURE activate(x: VECTOR OF REAL): VECTOR OF REAL;
BEGIN (* logistic activation function *)
  RETURN 1.0 / (1.0 + exp(-x))
END activate;


PROCEDURE NetStep;
VAR s1val: weight1 OF REAL;
    s2val: weight2 OF REAL;
BEGIN  (* single step execution of neural data propagation *)
  sensor<:MaxInput+1:>  := 1.0;                    (* set "on" neuron *)
  SEND.forward1     (sensor, s1val);               (* broadcast *)
  SEND.forward2:#SUM(s1val*strength1, hidval);     (* reduce    *)
  hidval := activate(hidval);
  hidval<:MaxHidden+1:> := 1.0;                    (* set "on" neuron *)
  SEND.forward3     (hidval, s2val);               (* broadcast *)
  SEND.forward4:#SUM(s2val*strength2, outval);     (* reduce    *)
  outval := activate(outval);
END NetStep;

 
PROCEDURE error(val: output OF REAL; digit: INTEGER): output OF REAL;
VAR teach: output OF REAL;
BEGIN (* overall error difference of digit activation *)
  teach := 0.0;
  teach<:digit:> := 1.0;
  RETURN teach - val;
END error;


PROCEDURE BackProp(VAR totalerr: REAL);
(* error back-propagatipon algorithm *)
VAR digit                 : INTEGER;
    diffout,outerr        : output  OF REAL;
    diff2,hid2,delta2     : weight2 OF REAL;
    diffhid               : hidden  OF REAL;
    diff1,hid1,delta1,sen1: weight1 OF REAL;
BEGIN
  totalerr := 0.0;
  FOR digit:=0 TO MaxPattern-1 DO
    setPattern(sensor,digit);
    NetStep;
    outerr   := error(outval,digit);
    diffout  := outerr * (1.0 - outval) * outval;
    totalerr := totalerr + REDUCE.SUM(outerr*outerr);

    (* change weights in layer 2 *)
    SEND.back4(diffout, diff2);
    SEND.forward3(hidval, hid2);
    delta2 := eta * hid2 * diff2;

    (* change weights in layer 1 *)
    SEND.back3:#SUM(strength2 * diff2, diffhid);
    SEND.back2     (diffhid, diff1);
    SEND.back2     (hidval, hid1);
    SEND.forward1  (sensor, sen1);
    delta1 := eta * sen1 * hid1 * (1.0-hid1) * diff1;

    strength2 := strength2 + delta2;
    strength1 := strength1 + delta1;

    IF debug THEN 
      WriteString("BackProp"); WriteLn;                         
      WriteString("input : "); WriteFixPt(sensor,6,2);
      WriteString("hidden: "); WriteFixPt(hidval,6,2);
      WriteString("output: "); WriteFixPt(outval,6,2);
      WriteString("total : "); WriteFixPt(totalerr,6,2); WriteLn;
      WriteString("delta1: "); WriteLn; WriteFixPt(delta1,6,3);
      WriteString("delta2: "); WriteLn; WriteFixPt(delta2,6,3);
      WriteString("strength1: "); WriteLn; WriteFixPt(strength1,6,3);
      WriteString("strength2: "); WriteLn; WriteFixPt(strength2,6,3);
    END;
  END; (* for *)
  totalerr  := totalerr / FLOAT(MaxPattern);
END BackProp;


PROCEDURE setPattern(VAR val: input OF REAL; pat: INTEGER);
(* generate test patterns for digits 0..9 in 7 segment display *)
(* segment labeling:    -   1    *)
(*                     | |  2 3  *)
(*                      -   4    *)
(*                     | |  5 6  *)
(*                      -   7    *)
TYPE seg = ARRAY[1..7] OF REAL;
VAR  s: seg;
BEGIN
  CASE pat OF
           (*   1    2    3    4    5    6    7  *)
   0: s := seg(1.0, 1.0, 1.0, 0.0, 1.0, 1.0, 1.0) |
   1: s := seg(0.0, 0.0, 1.0, 0.0, 0.0, 1.0, 0.0) |
   2: s := seg(1.0, 0.0, 1.0, 1.0, 1.0, 0.0, 1.0) |
   3: s := seg(1.0, 0.0, 1.0, 1.0, 0.0, 1.0, 1.0) |
   4: s := seg(0.0, 1.0, 1.0, 1.0, 0.0, 1.0, 0.0) |
   5: s := seg(1.0, 1.0, 0.0, 1.0, 0.0, 1.0, 1.0) |
   6: s := seg(1.0, 1.0, 0.0, 1.0, 1.0, 1.0, 1.0) |
   7: s := seg(1.0, 0.0, 1.0, 0.0, 0.0, 1.0, 0.0) |
   8: s := seg(1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0) |
   9: s := seg(1.0, 1.0, 1.0, 1.0, 0.0, 1.0, 1.0)
   ELSE WriteString("Error in setPattern"); WriteInt(pat,1); 
        WriteLn; HALT;
  END;
  LOAD(val,s);
END setPattern;

PROCEDURE showPattern(val: input OF REAL);
BEGIN
  WriteLn;
  IF val<:1:> > 0.5 THEN WriteString(" - ") END; WriteLn;
  IF val<:2:> > 0.5 THEN Write("|") ELSE Write(" ") END; Write(" ");
  IF val<:3:> > 0.5 THEN Write("|") END; WriteLn;
  IF val<:4:> > 0.5 THEN WriteString(" - ") END; WriteLn;
  IF val<:5:> > 0.5 THEN Write("|") ELSE Write(" ") END; Write(" ");
  IF val<:6:> > 0.5 THEN Write("|") END; WriteLn;
  IF val<:7:> > 0.5 THEN WriteString(" - ") END; WriteLn;
END showPattern;


BEGIN  (* main program *)
  WriteString('Start NN Simulator ');
  debug := argc() > 0;
  IF debug THEN WriteString('debug on') END;
  WriteLn;
  WriteString('start training'); WriteLn;
  (* init connection values *)
  strength1 := RandomReal(weight1) / 2.0;
  strength2 := RandomReal(weight2) / 2.0;
  IF debug THEN
    WriteString("Init"); WriteLn;
    WriteString("strength1: "); WriteLn; WriteFixPt(strength1,6,2);
    WriteString("strength2: "); WriteLn; WriteFixPt(strength2,6,2);
  END;       

  (* create training patterns for 7-segment display *)
  WriteString('Learning Phase'); WriteLn;
  step     := 0;
  REPEAT
    INC(step);
    BackProp(err);
    IF step MOD 10 = 0 THEN
      WriteString("step: ");    WriteInt(step,4);
      WriteString("  error: "); WriteFixPt(err,6,2); WriteLn;
    END;
  UNTIL (err < eps) OR (step >= MaxSteps);
 
  WriteString('Testing Phase'); WriteLn;
  FOR step := 0 TO MaxPattern - 1 DO
    setPattern(sensor,step);
    showPattern(sensor);
    NetStep;
    verr := error(outval,step);
    err    := REDUCE.SUM(verr*verr);
    maxout := REDUCE.MAX(outval);
    IF outval=maxout THEN pos := REDUCE.FIRST(DIM(output,1)) END;
    WriteString("winner: "); WriteInt(pos,3); 
    IF pos = step THEN WriteString(" ** OK **")
                  ELSE WriteString(" -- no --")
    END;
    WriteLn;
    WriteString("outval: "); WriteFixPt(outval,6,2);
    WriteString("error : "); WriteFixPt(err,6,2); WriteLn;
  END;

  WriteString('Application Phase'); WriteLn;
  REPEAT
    WriteString("continue (y/n) ? ");
    ReadString(ch);
    IF CAP(ch[1])#'N' THEN
      WriteString('Enter input neuron values: ');
      IF DIM(sensor,1) <= MaxInput THEN ReadReal(sensor) END;
      showPattern(sensor);
      NetStep;
      maxout := REDUCE.MAX(outval);
      IF outval=maxout THEN pos := REDUCE.FIRST(DIM(output,1)) END;
      WriteString("winner: "); WriteInt(pos,3); WriteLn;
      WriteString("outval: "); WriteFixPt(outval,6,2);
    END;
  UNTIL CAP(ch[1])='N';
  WriteString('Stop  NN Simulator'); WriteLn;
END nn.

