MODULE xautobahn;
(* traffic jam simulation, X version   *)
(* Thomas Braunl, Univ. Stuttgart 1995 *)
FROM ImageIO  IMPORT gray, color, g_white, c_red, c_green;
FROM Graphics IMPORT OpenWindow, CloseWindow, SetgLine, SetLine, window;

CONST max_cars  =  100;       (* number of cars on autobahn          *)
      steps     =  300;       (* number of simulation steps          *)
      max_speed =    0.005;   (* maximum speed of overall distance   *)
      max_accel =    0.0001;  (* maximum accelaration in a time step *)
      min_dist  =    0.002;   (* car slows down below this distance  *)
      rand_fac  =    0.00003; (* random acceleration factor          *)
      width     =  300;
      height    =  steps;

CONFIGURATION cars[0..max_cars-1];
CONNECTION next: cars[i] <-> cars[(i+1) MOD max_cars] :back;

CONFIGURATION street[0..width-1];

VAR pos, dist, speed, accel: cars OF REAL; (* car position is [0..1] *)
    collision              : cars OF BOOLEAN;
    num                    : street OF INTEGER;
    col,my_car             : street OF BOOLEAN;
    time, z                : INTEGER;
    win                    : window;
    ch                     : CHAR;

PROCEDURE min(a: cars OF REAL; b: REAL): cars OF REAL;
VAR h: cars OF REAL;
BEGIN
  IF a<b THEN h:=a
         ELSE h:=b
  END;
  RETURN h;
END min;


BEGIN
  win       := OpenWindow("autobahn",width,height);
  pos       := FLOAT(DIM(cars,1)) / FLOAT(max_cars); (* init eq. dist. [0..1[ *)
  speed     := 0.0;
  collision := FALSE;
  my_car    := FALSE;
  FOR time := 1 TO steps DO
    num := 0;
    col := FALSE;
    FOR z := 0 TO max_cars-1 DO  (* count cars in each section *)
      INC(num<: TRUNC(pos<:z:> * FLOAT(width)) :>);
      IF collision<:z:> THEN col <: TRUNC(pos<:z:> * FLOAT(width)) :> := TRUE
      END;
    END;
    num := 20 *num;
    IF num > g_white THEN num := g_white END;
    SetgLine(num, time);
    IF col THEN SetLine(street(c_red), time) END;      (* display col. red *)
    my_car := DIM(street,1) = TRUNC(pos<:0:> * FLOAT(width));
    IF my_car THEN SetLine(street(c_green), time) END; (* disp. my car green *)

    dist := MOVE.back(pos) - pos;
    IF dist < 0.0 THEN dist := dist + 1.0 END;  (* close street to loop *)

    collision := dist < min_dist;
    IF collision THEN speed := 0.0;  (* pos unchanged *)
     ELSE (* no collision *)
      (* accel proportional to distance-diff. plus small random term *)
      accel := max_accel + rand_fac * (RandomReal(cars)-0.5);
      IF dist < min_dist THEN accel := - max_accel END;

      speed := min(speed + accel, max_speed);
      IF speed < 0.0 THEN speed := 0.0 END; (* do not back up on autobahn ! *)

      IF speed < dist THEN pos := pos + speed 
                      ELSE speed := 0.0
      END;
      IF pos >= 1.0 THEN pos := pos - 1.0 END;      (* end of car ring *)
    END;
  END;
  WriteString("Press RETURN for termination"); WriteLn;
  Read(ch);
  CloseWindow(win);
END xautobahn.

