IMPLEMENTATION MODULE Corners;
(* algorithms for contour line based detection of grayvalue corners    *)
(* EXPORTED PROCEDURES:                                                *)
(*   1.) detMeanvar          (corner detector)                         *)
(*   2.) detPairvar          (corner detector)                         *)
(*   3.) marksqr_binary      (overlay for corner visualization)        *)
(*   4.) marksqr_gray        (overlay for corner visualization)        *)
(*   5.) marksqr_color       (overlay for corner visualization)        *)
(* LANGUAGE: Parallaxis Version 3                                      *)
(* CREATED: Aug. 95  Wolfgang Rapf                                     *)
(* RELATED FILES: Corners.pd                                           *)
(* REMARKS:                                                            *)
(* CHANGES:                                                            *) 
 
FROM ImageIO IMPORT gray, color, binary, b_black, b_white;
FROM Trans IMPORT gray2color;
FROM Local IMPORT grid, right, left, up, down,
                  median_5x5, sobel_x_5x5, sobel_y_5x5;
FROM Thinning IMPORT lueWang;


(* calculate min. & max. INTEGER value in NxN-neigbourhood *)
PROCEDURE int_minmax_nxn(radius: CARDINAL;
                         input: grid OF INTEGER;
                         VAR min, max: grid OF INTEGER);
VAR i: INTEGER;
    v_min, v_max, temp1, temp2: grid OF INTEGER;
BEGIN 
  min := input;
  max := input;
  IF radius > 0
  THEN  
    v_min := input;
    v_max := input;
    (* maximum and minimum of a column (upper half) *)
    temp1 := MOVE.down(input);
    IF temp1 < v_min THEN v_min := temp1 END;
    IF temp1 > v_max THEN v_max := temp1 END;
    FOR i := 2 TO radius DO
      temp1 := MOVE.down(temp1);
      IF temp1 < v_min THEN v_min := temp1 END;
      IF temp1 > v_max THEN v_max := temp1 END;
    END;
    (* maximum and minimum of a column (lower half) *)
    temp1 := MOVE.up(input);
    IF temp1 < v_min THEN v_min := temp1 END;
    IF temp1 > v_max THEN v_max := temp1 END;
    FOR i := 2 TO radius DO
      temp1 := MOVE.up(temp1);
      IF temp1 < v_min THEN v_min := temp1 END;
      IF temp1 > v_max THEN v_max := temp1 END;
    END;
    (* maximum und minimum of all columns (left half) *)
    temp1 := MOVE.right(v_min);
    temp2 := MOVE.right(v_max);
    IF temp1 < v_min THEN min := temp1 ELSE min := v_min END; 
    IF temp2 > v_max THEN max := temp2 ELSE max := v_max END;
    FOR i := 2 TO radius DO
      temp1 := MOVE.right(temp1);
      temp2 := MOVE.right(temp2);
      IF temp1 < min THEN min := temp1 END; 
      IF temp2 > max THEN max := temp2 END;
    END;
    (* maximum und minimum of all columns (right half) *)
    temp1 := MOVE.left(v_min);
    temp2 := MOVE.left(v_max);
    IF temp1 < min THEN min := temp1 END; 
    IF temp2 > max THEN max := temp2 END;
    FOR i := 2 TO radius DO
      temp1 := MOVE.left(temp1);
      temp2 := MOVE.left(temp2);
      IF temp1 < min THEN min := temp1 END; 
      IF temp2 > max THEN max := temp2 END;
    END;
  END; (* IF radius > 0 *)
END int_minmax_nxn;


(* calculate median INTEGER vector of 3x3-neighbourhood; only vectors,    *)
(* whose components both differ from zero are considered; for application *)
(* along edge lines only                                                  *)
PROCEDURE edge_grad_median_3x3(VAR dx, dy: grid OF INTEGER);
VAR count: grid OF CARDINAL;
    h_temp_x, h_temp_y, v_temp_x, v_temp_y: grid OF INTEGER;
    sort_field: grid OF ARRAY[1..9] OF RECORD x, y: INTEGER END;

        PROCEDURE push_grad(dx, dy: grid OF INTEGER;
                            VAR count: grid OF CARDINAL);
        (* insert gradient vector into sorted list *)
        VAR do_insert: grid OF BOOLEAN;
            i, j: grid OF INTEGER;
        BEGIN
          IF (dx <> 0) OR (dy <> 0)     (* ignore zero vectors *)
          THEN
            IF (count = 0)
            THEN sort_field[1].x := dx;   (* insert first element *)
                 sort_field[1].y := dy;
            ELSE
              (* insert x component *)
              i := 0; 
              REPEAT
                INC(i);
              UNTIL (dx < sort_field[i].x) OR (i > count);
              FOR j:=count TO i BY -1 DO  (* create a gap for insertion *)
                IF j < 9 THEN sort_field[j+1].x := sort_field[j].x END
              END;
              IF i < 10 THEN sort_field[i].x := dx END;
              (* insert y component *)
              i := 0;
              REPEAT
                INC(i);
              UNTIL (dy < sort_field[i].y) OR (i > count);
              FOR j:=count TO i BY -1 DO  (* create a gap for insertion *)
                IF j < 9 THEN sort_field[j+1].y := sort_field[j].y END
              END;
              IF i < 10 THEN sort_field[i].y := dy END;
            END;

            INC(count);
          END;
        END push_grad;
 
BEGIN (* edge_grad_median_3x3 *)
  FOR count := 1 TO 9 DO
    sort_field[count].x := 0;
    sort_field[count].y := 0;
  END;
  count := 0;

  (* left column *)
  h_temp_x := MOVE.right(dx);
  h_temp_y := MOVE.right(dy);
  push_grad(h_temp_x, h_temp_y, count);
  v_temp_x := MOVE.up(h_temp_x);
  v_temp_y := MOVE.up(h_temp_y);
  push_grad(v_temp_x, v_temp_y, count);
  v_temp_x := MOVE.down(h_temp_x);
  v_temp_y := MOVE.down(h_temp_y);
  push_grad(v_temp_x, v_temp_y, count);
  (* center column *)  
  v_temp_x := MOVE.up(dx);
  v_temp_y := MOVE.up(dy);
  push_grad(v_temp_x, v_temp_y, count);
  push_grad(dx, dy, count);
  v_temp_x := MOVE.down(dx);
  v_temp_y := MOVE.down(dy);
  push_grad(v_temp_x, v_temp_y, count);
  (* right column *)
  h_temp_x := MOVE.left(dx);
  h_temp_y := MOVE.left(dy);
  push_grad(h_temp_x, h_temp_y, count);
  v_temp_x := MOVE.up(h_temp_x);
  v_temp_y := MOVE.up(h_temp_y);
  push_grad(v_temp_x, v_temp_y, count);
  v_temp_x := MOVE.down(h_temp_x);
  v_temp_y := MOVE.down(h_temp_y);
  push_grad(v_temp_x, v_temp_y, count); 

  IF ((count > 0) AND ((dx <> 0) OR (dy <> 0))) OR (count > 5)
  THEN dx := sort_field[(count + 1) DIV 2].x;
       dy := sort_field[(count + 1) DIV 2].y
  END;

END edge_grad_median_3x3;


(* calculate local curvature measurement based on local vector variances; *)
(* input: dx, dy: local gradient vector along an edge line (object        *)
(*                contour line). For use as curvature measurement, all    *)
(*                vectors have to be  globally normed to equal length !   *)
(* vectors in 3x3-environment are considered, ignoring zero vectors;      *)
(* returns mean of the squares of the vector products of all considered   *)
(* vectors with the central vector. This value can be interpreted as a    *)
(* measurement of the local vector irregularity, in case of edge lines as *)
(* a measurement of the local edge curvature.                             *)
(* remark: only apply to edge lines.                                      *)
PROCEDURE curv_mvar_3x3(dx, dy: grid OF INTEGER): grid OF CARDINAL;
VAR sum, count: grid OF CARDINAL;
    h_temp_dx, h_temp_dy, v_temp_dx, v_temp_dy: grid OF INTEGER;
 
        PROCEDURE sum_mvar(dx, dy, ext_dx, ext_dy: grid OF INTEGER;
                           VAR sum, count: grid OF CARDINAL);
        (* add vector product to global sum; zero vectors will be ignored *) 
        BEGIN IF (ext_dx <> 0) OR (ext_dy <> 0)
              THEN sum := sum + (dx * ext_dy - dy * ext_dx) ** 2;
                   INC(count);
              END;
        END sum_mvar;
 
BEGIN (* curv_mvar3x3 *)
  sum := 0;
  count := 0;
  (* consider neighbour vectors of left column *)
  h_temp_dx := MOVE.right(dx);
  h_temp_dy := MOVE.right(dy);
  sum_mvar(dx, dy, h_temp_dx, h_temp_dy, sum, count);
  v_temp_dx := MOVE.up(h_temp_dx);
  v_temp_dy := MOVE.up(h_temp_dy);
  sum_mvar(dx, dy, v_temp_dx, v_temp_dy, sum, count);
  h_temp_dx := MOVE.down(h_temp_dx);
  h_temp_dy := MOVE.down(h_temp_dy);
  sum_mvar(dx, dy, h_temp_dx, h_temp_dy, sum, count);
  (* consider upper and lower neighbour vector *)
  v_temp_dx := MOVE.up(dx);
  v_temp_dy := MOVE.up(dy);
  sum_mvar(dx, dy, v_temp_dx, v_temp_dy, sum, count);
  v_temp_dx := MOVE.down(dx);
  v_temp_dy := MOVE.down(dy);
  sum_mvar(dx, dy, v_temp_dx, v_temp_dy, sum, count);
  (* consider neighbour vectors of right column *)
  h_temp_dx := MOVE.left(dx);
  h_temp_dy := MOVE.left(dy);
  sum_mvar(dx, dy, h_temp_dx, h_temp_dy, sum, count);
  v_temp_dx := MOVE.up(h_temp_dx);
  v_temp_dy := MOVE.up(h_temp_dy);
  sum_mvar(dx, dy, v_temp_dx, v_temp_dy, sum, count);
  h_temp_dx := MOVE.down(h_temp_dx);
  h_temp_dy := MOVE.down(h_temp_dy);
  sum_mvar(dx, dy, h_temp_dx, h_temp_dy, sum, count);
 
  IF (count > 0) AND ((dx <> 0) OR (dy <> 0))
  THEN sum :=  sum DIV count 
  ELSE sum := 0;
  END;
  RETURN sum;
END curv_mvar_3x3;


(* calculate local curvature measurement based on local vector variances;   *)
(* one pair of gradient vectors will be returned as corner attributes       *)
(* input: dx, dy: local grayvalue gradient vector along an edge line (ob-   *)
(*                ject contour line). For use as curvature measurement,     *)
(*                all vectors have to be  globally normed to equal length;  *)
(*                Contour line is expected to be thinned to width 1 (apply  *)
(*                a skeletonizing algorithm first, e.g. of Lue and Wang );  *)
(*                gradient vectors should be smoothed before thinning;      *)
(* vectors in the border of the 5x5-environment are considered, ignoring    *)
(* zero vectors; (all locations, which don't belong to the contour line are *)
(* expected to be labelled as zero vectors in the input dx, dy);            *)
(* returns (dx1 - dx2) ** 2 + (dy1 - dy2) ** 2, where (dx1,dy1)/(dx2,dy2)   *)
(* denote the vector at the location in the border of the 5x5-neighbourhood,*)
(* where the contour line enters/leaves the environment. This value can be  *)
(* interpreted as a measurement of the local edge line curvature.           *)
(* remark: only apply to edge lines.                                        *)
(*                                                                          *)
(* distances:          order of passing through the neighbours:             *)
(*                                   (S = start; visited positions)         *)
(*      4 3 2 3 4       3  4  5  6  7                                       *)
(*      3 2 1 2 3       2           8                                       *)
(*      2 1 0 1 2       1  0  S     9                                       *)
(*      3 2 1 2 3      16          10                                       *)
(*      4 3 2 3 4      15 14 13 12 11                                       *)
(*                                                                          *)
PROCEDURE curv_pvar_5x5(dx, dy: grid OF INTEGER;
                        VAR dx1, dy1, dx2, dy2: grid OF INTEGER)
                      : grid OF CARDINAL;
VAR x_temp, y_temp, x_init, y_init, branch_nr,
    old_dist, start_pos: grid OF INTEGER;
    branches: grid OF ARRAY[1..2] OF RECORD x, y: INTEGER END;
    hit_branch: grid OF BOOLEAN;
    curvature: grid OF CARDINAL;
 
        PROCEDURE locate_ends(dx, dy: grid OF INTEGER;
                              dist: INTEGER);
        BEGIN
          IF (dx <> 0) OR (dy <> 0)
          THEN
            IF NOT hit_branch THEN INC(branch_nr) END;
            hit_branch := TRUE;
            IF dist > old_dist
            THEN
              old_dist := dist;
              IF branch_nr <= 2
              THEN branches[branch_nr].x := dx;
                   branches[branch_nr].y := dy;
              END;
            END;
          ELSE 
            hit_branch := FALSE;
            old_dist := 0
          END;
        END locate_ends;
 
BEGIN (* curv_pvar_5x5 *)
  branch_nr := 0;
  old_dist := 0;
  hit_branch := FALSE;

  x_temp := MOVE.right(dx);           (* position 0 *)
  y_temp := MOVE.right(dy);
  x_temp := MOVE.right(x_temp);       (* position 1 *)
  y_temp := MOVE.right(y_temp);
  locate_ends(x_temp, y_temp, 2);
  IF hit_branch THEN start_pos := 1 ELSE start_pos := 0 END;
 
  x_init := MOVE.up(x_temp);          (* position 16 *)
  y_init := MOVE.up(y_temp);
  IF (start_pos = 1) AND ((x_init <> 0) OR (y_init <> 0))
  THEN old_dist := 3;
       start_pos := 16;
       branches[1].x := dx;
       branches[1].y := dy;
  END;
 
  x_init := MOVE.up(x_init);          (* position 15 *)
  y_init := MOVE.up(y_init);
  IF (start_pos = 16) AND ((x_init <> 0) OR (y_init <> 0))
  THEN old_dist := 4;
       start_pos := 15;
       branches[branch_nr].x := dx;
       branches[branch_nr].y := dy
  END;
 
  x_temp := MOVE.down(x_temp);       (* position 2 *)
  y_temp := MOVE.down(y_temp);
  locate_ends(x_temp, y_temp, 3);
  x_temp := MOVE.down(x_temp);       (* position 3 *)
  y_temp := MOVE.down(y_temp);
  locate_ends(x_temp, y_temp, 4);
  x_temp := MOVE.left(x_temp);       (* position 4 *)
  y_temp := MOVE.left(y_temp);
  locate_ends(x_temp, y_temp, 3);
  x_temp := MOVE.left(x_temp);       (* position 5 *)
  y_temp := MOVE.left(y_temp);
  locate_ends(x_temp, y_temp, 2);
  x_temp := MOVE.left(x_temp);       (* position 6 *)
  y_temp := MOVE.left(y_temp);
  locate_ends(x_temp, y_temp, 3);
  x_temp := MOVE.left(x_temp);       (* position 7 *)
  y_temp := MOVE.left(y_temp);
  locate_ends(x_temp, y_temp, 4);
  x_temp := MOVE.up(x_temp);         (* position 8 *)
  y_temp := MOVE.up(y_temp);
  locate_ends(x_temp, y_temp, 3);
  x_temp := MOVE.up(x_temp);         (* position 9 *)
  y_temp := MOVE.up(y_temp);
  locate_ends(x_temp, y_temp, 2);
  x_temp := MOVE.up(x_temp);         (* position 10 *)
  y_temp := MOVE.up(y_temp);
  locate_ends(x_temp, y_temp, 3);
  x_temp := MOVE.up(x_temp);         (* position 11 *)
  y_temp := MOVE.up(y_temp);
  locate_ends(x_temp, y_temp, 4);
  x_temp := MOVE.right(x_temp);      (* position 12 *)
  y_temp := MOVE.right(y_temp);
  locate_ends(x_temp, y_temp, 3);
  x_temp := MOVE.right(x_temp);      (* position 13 *)
  y_temp := MOVE.right(y_temp);
  locate_ends(x_temp, y_temp, 2);
  x_temp := MOVE.right(x_temp);      (* position 14 *)
  y_temp := MOVE.right(y_temp);
  locate_ends(x_temp, y_temp, 3);
  x_temp := MOVE.right(x_temp);      (* position 15 *)
  y_temp := MOVE.right(y_temp);
  IF start_pos = 15 THEN hit_branch := TRUE END;
  locate_ends(x_temp, y_temp, 4);
  x_temp := MOVE.down(x_temp);       (* position 16 *)
  y_temp := MOVE.down(y_temp);
  IF start_pos = 16 THEN hit_branch := TRUE END;
  locate_ends(x_temp, y_temp, 3);
 
  IF (branch_nr = 2) AND ((dx <> 0) OR (dy <> 0))
  THEN dx1 := branches[1].x; dy1 := branches[1].y;
       dx2 := branches[2].x; dy2 := branches[2].y;
       curvature := (dx1 - dx2) ** 2 + (dy1 - dy2) ** 2;
  ELSE curvature := 0; dx1 := 0; dy1 := 0; dx2 := 0; dy2 := 0;
  END;
 
  RETURN(curvature); 
END curv_pvar_5x5;
 

(* calculate mean CARDINAL value of 3x3-neighbourhood; only vectors, whose *)
(* components both differ from zero are considered; for application along  *)
(* edge lines only                                                         *)
PROCEDURE edge_card_mean_3x3(input: grid OF CARDINAL): grid OF CARDINAL;
VAR  sum, count, h_temp, v_temp: grid OF CARDINAL;
 
        PROCEDURE sum_avg(input: grid OF CARDINAL;
                          VAR sum, count: grid OF CARDINAL);
        BEGIN IF (input <> 0)
              THEN sum := sum + input;
                   INC(count);
              END;
        END sum_avg;
BEGIN (* edge_card_mean_3x3 *)
  sum := 0;
  count := 0;
  (* left column *)
  h_temp := MOVE.right(input);
  sum_avg(h_temp, sum, count);
  v_temp := MOVE.up(h_temp);
  sum_avg(v_temp, sum, count);
  h_temp := MOVE.down(h_temp);
  sum_avg(h_temp, sum, count);
  (* center column *)
  v_temp := MOVE.up(input);
  sum_avg(v_temp, sum, count);
  sum_avg(input,sum, count);
  v_temp := MOVE.down(input);
  sum_avg(v_temp, sum, count);
  (* right column *)
  h_temp := MOVE.left(input);
  sum_avg(h_temp, sum, count);
  v_temp := MOVE.up(h_temp);
  sum_avg(v_temp, sum, count);
  h_temp := MOVE.down(h_temp);
  sum_avg(h_temp, sum, count);
 
  IF (count > 0) AND (input <> 0)
  THEN sum := sum DIV count
  ELSE sum := 0
  END;
  RETURN sum;
END edge_card_mean_3x3;
 

(* detect local peaks in a CARDINAL image; parameters:           *)
(* radius (range 2,3,.. typical 5)                               *)
(* global_thresh (range: 0.0 to 1.0, typical 0.1)                *)
(* local_thresh (range: 0.0 to 1.0, typical 0.8 to 1.0)          *)
(* returns TRUE, if                                              *)
(*       1.) input => NEARMAX                                    *)
(*  and  2.) input * local_thresh > FARMAX                       *)
(*  and  3.) input > GLOBALMAX * global_thresh                   *)
(* NEARMAX = maximum value in 3x3-neighbourhood                  *)
(* FARMAX = maximum value in NxN-neighbourhood (given by radius) *) 
(*          without contained 3x3-neighbourhood                  *)
(* GLOBALMAX = global maximum of input                           *)
(* remark: this operation should only be applied to smoothed     *)
(*         CARDINAL images                                       *)
PROCEDURE card_is_peak_nxn(radius: CARDINAL;
                           global_thresh, local_thresh: REAL;
                           input: grid OF CARDINAL)
                         : grid OF BOOLEAN;
VAR max_global: INTEGER;
    max, v_max, h_max, far_max, near_max, temp, i: grid OF CARDINAL;
    is_max: grid OF BOOLEAN;

BEGIN (* card_is_peak_nxn *)
  IF radius > 0
  THEN v_max := input;
 
       (* maximum of a column (upper half) *)
       temp := MOVE.down(input);
       IF temp > v_max THEN v_max := temp END;
       FOR i := 2 TO radius DO
         temp := MOVE.down(temp);
         IF temp > v_max THEN v_max := temp END;
       END;
 
       (* maximum of a column (lower half) *)
       temp := MOVE.up(input);
       IF temp > v_max THEN v_max := temp END;
       FOR i := 2 TO radius DO
         temp := MOVE.up(temp);
         IF temp > v_max THEN v_max := temp END;
       END;
 
       (* maximum of left outer columns *)
       temp := MOVE.right(v_max);
       IF radius > 1 THEN temp := MOVE.right(temp) END;
       far_max := temp;
       FOR i := 3 TO radius DO
         temp := MOVE.right(temp);
         IF temp > far_max THEN far_max := temp END;
       END;
    
       (* maximum of right outer columns *)
       temp := MOVE.left(v_max);
       IF radius > 1 THEN temp := MOVE.left(temp) END;
       IF temp > far_max THEN far_max := temp END;
       FOR i := 3 TO radius DO
         temp := MOVE.left(temp);
         IF temp > far_max THEN far_max := temp END;
       END;
 
       (* maximum of center columns *)
       h_max := input;
       temp := MOVE.right(input);
       IF temp > h_max THEN h_max := temp END;
       temp := MOVE.left(input);
       IF temp > h_max THEN h_max := temp END;
       
       (* maxima of direct and indirect neighbourhood *)
       IF radius = 1
       THEN near_max := input;
            temp := MOVE.down(input);
            IF temp > far_max THEN far_max := temp END;
            temp := MOVE.up(input);
            IF temp > far_max THEN far_max := temp END;
       ELSE near_max := h_max;
            temp := MOVE.down(h_max);
            IF temp > near_max THEN near_max := temp END;
            FOR i := 2 TO radius DO
              temp := MOVE.down(temp);
              IF temp > far_max THEN far_max := temp END;
            END;
            h_max := MOVE.up(h_max);
            IF h_max > near_max THEN near_max := h_max END;
            FOR i := 2 TO radius DO
              h_max := MOVE.up(h_max);
              IF h_max > far_max THEN far_max := h_max END;
            END;
      END;
  ELSE near_max := input; far_max := input;
  END;
 
  IF far_max > near_max THEN max := far_max ELSE max := near_max END;
  max_global := REDUCE.MAX(max);

  IF  (input = max)
  AND (far_max <  CARDINAL(FLOAT(input) * local_thresh))
  AND (input > CARDINAL(FLOAT(max_global) * global_thresh))
  THEN is_max := TRUE;
  ELSE is_max := FALSE;
  END;
  RETURN is_max;
END card_is_peak_nxn;


(* reduce a spot of several equal maximal points to a single point *)
PROCEDURE concen_peaks_3x3(input: grid OF BOOLEAN): grid OF BOOLEAN;
VAR changed: BOOLEAN;
    p_num, p_max, old_num, global_max, local_rand, rand_max: grid OF INTEGER;
    is_peak: grid OF BOOLEAN;
 
        PROCEDURE count_peaks_3x3(VAR is_peak: grid OF BOOLEAN)
                                : grid OF CARDINAL;
        VAR v_count, count, c_temp: grid OF CARDINAL; 
            b_temp: grid OF BOOLEAN;
        BEGIN
          (* count peak labels in a column *)
          IF is_peak THEN v_count := 1 ELSE v_count := 0 END; 
          b_temp := MOVE.up(is_peak);
          IF b_temp THEN INC(v_count) END; 
          b_temp := MOVE.down(is_peak);
          IF b_temp THEN INC(v_count) END; 
 
          (* sum of counters (all columns) *)
          c_temp := MOVE.right(v_count);
          count := v_count + c_temp;
          c_temp := MOVE.left(count);
          count := count + c_temp;
 
          RETURN count;
        END count_peaks_3x3;


        PROCEDURE int_max_3x3(input: grid OF INTEGER): grid OF INTEGER;
        VAR max_val, temp: grid OF INTEGER;
        BEGIN
          (* vertical maximum *)
          max_val := MOVE.down(input);
          IF input > max_val THEN max_val := input END;
          temp := MOVE.up(max_val);
          IF temp > max_val THEN max_val := temp  END;
          (* horizontal maximum *)
          temp := MOVE.left(max_val);
          IF temp > max_val THEN max_val := temp END;
          temp := MOVE.right(max_val);
          IF temp > max_val THEN max_val := temp END;
          RETURN(max_val)
        END int_max_3x3;


BEGIN (* concen_peaks_3x3 *)
  is_peak := input;
  p_num := count_peaks_3x3(is_peak);
  old_num := p_num;
  p_max := int_max_3x3(p_num);
  changed := REDUCE.OR(p_num > 1);
  WHILE changed DO
    IF (p_num <> p_max) THEN is_peak := FALSE END;
    p_num := count_peaks_3x3(is_peak);
    p_max := int_max_3x3(p_num);
    changed := REDUCE.OR(old_num <> p_num);
    global_max := REDUCE.MAX(p_max);
    IF (NOT changed) AND (global_max > 1) (* arbitrary elimination of peaks *)
    THEN IF is_peak 
         THEN local_rand := ABS(RandomInt(grid));
         ELSE local_rand := -1;         
         END;
         rand_max := int_max_3x3(local_rand);
         IF local_rand <> rand_max THEN is_peak := FALSE END;
         p_num := count_peaks_3x3(is_peak);
         p_max := int_max_3x3(p_num);
         global_max := REDUCE.MAX(p_max);
         changed := REDUCE.OR(old_num <> p_num);
    END;
    old_num := p_num;
  END;
  RETURN is_peak;
END concen_peaks_3x3;


(* overlay for point labelling in binary images: 2 squares of selected *)
(* color and size; the larger one contains the smaller one             *)
PROCEDURE marksqr_binary(old_image: grid OF binary; labeling: grid OF BOOLEAN;
                         value1: binary; size1: CARDINAL;
                         value2: binary; size2: CARDINAL)
                         : grid OF binary;
VAR front_size, back_size, i: CARDINAL;
    front_value, back_value, return_pixel: grid OF binary;
    v_contains, area_contains, temp: grid OF BOOLEAN;
BEGIN 
  IF size2 > size1 
  THEN front_size  := size1;
       front_value := value1;
       back_size   := size2;
       back_value  := value2;
  ELSE front_size  := size2;
       front_value := value2;
       back_size   := size1;
       back_value  := value1;
  END;

  return_pixel := old_image;
   
  (* column of height back_size: exists a label ? *)
  IF back_size > 0 
  THEN v_contains := labeling; temp := labeling;
  ELSE v_contains := FALSE;    temp := FALSE;
  END;
 
  FOR i:=1 TO (back_size+1) DIV 2 DO
    temp := MOVE.down(temp);
    v_contains := v_contains OR temp; 
  END;

  temp := labeling;
  FOR i:=1 TO back_size DIV 2 DO
    temp := MOVE.up(temp);
    v_contains := v_contains OR temp; 
  END;
 
  (* all columns of height back_size *)
  area_contains := v_contains;
  temp := v_contains;

  FOR i:=1 TO (back_size+1) DIV 2 DO
    temp := MOVE.right(temp);
    area_contains := area_contains OR temp;
  END;

  temp := v_contains;
  FOR i:=1 TO back_size DIV 2 DO
    temp := MOVE.left(temp);
    area_contains := area_contains OR temp;
  END;

  IF area_contains THEN return_pixel :=  back_value END;

  (* column of height front_size: exists a label ? *)
  IF front_size > 0 
  THEN v_contains := labeling; temp := labeling;
  ELSE v_contains := FALSE;    temp := FALSE;
  END;
 
  FOR i:=1 TO (front_size+1) DIV 2 DO
    temp := MOVE.down(temp);
    v_contains := v_contains OR temp; 
  END;

  temp := labeling;
  FOR i:=1 TO front_size DIV 2 DO
    temp := MOVE.up(temp);
    v_contains := v_contains OR temp; 
  END;
 
  (* all columns of height front_size *)
  area_contains := v_contains;
  temp := v_contains;

  FOR i:=1 TO (front_size+1) DIV 2 DO
    temp := MOVE.right(temp);
    area_contains := area_contains OR temp;
  END;

  temp := v_contains;
  FOR i:=1 TO front_size DIV 2 DO
    temp := MOVE.left(temp);
    area_contains := area_contains OR temp;
  END;

  IF area_contains THEN return_pixel := front_value END;

  RETURN(return_pixel);  
END marksqr_binary;


(* overlay for point labelling in grayvalue images: 2 squares of selected *)
(* color and size; the larger one contains the smaller one                *)
PROCEDURE marksqr_gray(old_image: grid OF gray; labeling: grid OF BOOLEAN;
                       value1: gray; size1: CARDINAL;
                       value2: gray; size2: CARDINAL)
                       : grid OF gray;
VAR front_size, back_size, i: CARDINAL;
    front_value, back_value, return_pixel: grid OF gray;
    v_contains, area_contains, temp: grid OF BOOLEAN;
BEGIN 
  IF size2 > size1 
  THEN front_size  := size1;
       front_value := value1;
       back_size   := size2;
       back_value  := value2;
  ELSE front_size  := size2;
       front_value := value2;
       back_size   := size1;
       back_value  := value1;
  END;

  return_pixel := old_image;
   
  (* column of height back_size: exists a label ? *)
  IF back_size > 0 
  THEN v_contains := labeling; temp := labeling;
  ELSE v_contains := FALSE;    temp := FALSE;
  END;
 
  FOR i:=1 TO (back_size+1) DIV 2 DO
    temp := MOVE.down(temp);
    v_contains := v_contains OR temp; 
  END;

  temp := labeling;
  FOR i:=1 TO back_size DIV 2 DO
    temp := MOVE.up(temp);
    v_contains := v_contains OR temp; 
  END;
 
  (* all columns of height back_size *)
  area_contains := v_contains;
  temp := v_contains;

  FOR i:=1 TO (back_size+1) DIV 2 DO
    temp := MOVE.right(temp);
    area_contains := area_contains OR temp;
  END;

  temp := v_contains;
  FOR i:=1 TO back_size DIV 2 DO
    temp := MOVE.left(temp);
    area_contains := area_contains OR temp;
  END;

  IF area_contains THEN return_pixel :=  back_value END;

  (* column of height front_size: exists a label ? *)
  IF front_size > 0 
  THEN v_contains := labeling; temp := labeling;
  ELSE v_contains := FALSE;    temp := FALSE;
  END;
 
  FOR i:=1 TO (front_size+1) DIV 2 DO
    temp := MOVE.down(temp);
    v_contains := v_contains OR temp; 
  END;

  temp := labeling;
  FOR i:=1 TO front_size DIV 2 DO
    temp := MOVE.up(temp);
    v_contains := v_contains OR temp; 
  END;
 
  (* all columns of height front_size *)
  area_contains := v_contains;
  temp := v_contains;

  FOR i:=1 TO (front_size+1) DIV 2 DO
    temp := MOVE.right(temp);
    area_contains := area_contains OR temp;
  END;

  temp := v_contains;
  FOR i:=1 TO front_size DIV 2 DO
    temp := MOVE.left(temp);
    area_contains := area_contains OR temp;
  END;

  IF area_contains THEN return_pixel := front_value END;

  RETURN(return_pixel);
END marksqr_gray;


(* overlay for point labelling in color images: 2 squares of selected *)
(* color and size; the larger one contains the smaller one             *)
PROCEDURE marksqr_color(old_image: grid OF color; labeling: grid OF BOOLEAN;
                       value1: color; size1: CARDINAL;
                       value2: color; size2: CARDINAL)
                       : grid OF color;
VAR front_size, back_size, i: CARDINAL;
    front_value, back_value, return_pixel: grid OF color;
    v_contains, area_contains, temp: grid OF BOOLEAN;
BEGIN 
  IF size2 > size1 
  THEN front_size  := size1;
       front_value := value1;
       back_size   := size2;
       back_value  := value2;
  ELSE front_size  := size2;
       front_value := value2;
       back_size   := size1;
       back_value  := value1;
  END;

  return_pixel := old_image;
   
  (* column of height back_size: exists a label ? *)
  IF back_size > 0 
  THEN v_contains := labeling; temp := labeling;
  ELSE v_contains := FALSE;    temp := FALSE;
  END;
 
  FOR i:=1 TO (back_size+1) DIV 2 DO
    temp := MOVE.down(temp);
    v_contains := v_contains OR temp; 
  END;

  temp := labeling;
  FOR i:=1 TO back_size DIV 2 DO
    temp := MOVE.up(temp);
    v_contains := v_contains OR temp; 
  END;
 
  (* all columns of height back_size *)
  area_contains := v_contains;
  temp := v_contains;

  FOR i:=1 TO (back_size+1) DIV 2 DO
    temp := MOVE.right(temp);
    area_contains := area_contains OR temp;
  END;

  temp := v_contains;
  FOR i:=1 TO back_size DIV 2 DO
    temp := MOVE.left(temp);
    area_contains := area_contains OR temp;
  END;

  IF area_contains THEN return_pixel :=  back_value END;

  (* column of height front_size: exists a label ? *)
  IF front_size > 0 
  THEN v_contains := labeling; temp := labeling;
  ELSE v_contains := FALSE;    temp := FALSE;
  END;
 
  FOR i:=1 TO (front_size+1) DIV 2 DO
    temp := MOVE.down(temp);
    v_contains := v_contains OR temp; 
  END;

  temp := labeling;
  FOR i:=1 TO front_size DIV 2 DO
    temp := MOVE.up(temp);
    v_contains := v_contains OR temp; 
  END;
 
  (* all columns of height front_size *)
  area_contains := v_contains;
  temp := v_contains;

  FOR i:=1 TO (front_size+1) DIV 2 DO
    temp := MOVE.right(temp);
    area_contains := area_contains OR temp;
  END;

  temp := v_contains;
  FOR i:=1 TO front_size DIV 2 DO
    temp := MOVE.left(temp);
    area_contains := area_contains OR temp;
  END;

  IF area_contains THEN return_pixel := front_value END;

  RETURN(return_pixel);
END marksqr_color;



(* corner detection using gradient vector variance curvature measurement *)
PROCEDURE detMeanvar(minmax_radius: CARDINAL;
                     global_thresh, local_thresh: REAL;
                     peak_radius: CARDINAL;
                     peak_global_thresh, peak_local_thresh: REAL;
                     image: grid OF gray)
                     : grid OF BOOLEAN;
VAR med_image: grid OF gray;
    gv_dx, gv_dy: grid OF INTEGER;
    grad_stren, local_min, local_max, global_max,
    local_curv, length: grid OF CARDINAL;
    is_edge, is_peak, is_corner: grid OF BOOLEAN;

BEGIN (* detMeanvar *)
  (* edge preserving image smoothing *)
  med_image := median_5x5(image);

  (* calculate grayvalue gradients *) 
  gv_dx := (sobel_x_5x5(med_image) + 512) DIV 1024;
  gv_dy := (sobel_y_5x5(med_image) + 512) DIV 1024;

  (* calculate gradient strength *)
  grad_stren := CARDINAL(sqrt (FLOAT(gv_dx ** 2 + gv_dy ** 2)));

  (* eliminate regions with low gradient strength *)
  int_minmax_nxn(minmax_radius, grad_stren, local_min, local_max);
  global_max := REDUCE.MAX(grad_stren);
  IF(grad_stren < CARDINAL(FLOAT(global_max) * global_thresh))
  OR(grad_stren < CARDINAL((local_thresh * FLOAT(local_max) 
                         + (1.0 - local_thresh) * FLOAT(local_min))))
  THEN is_edge := FALSE; gv_dx := 0; gv_dy := 0; grad_stren := 0
  ELSE is_edge := TRUE
  END;

  (* egalize gradient vector lengths to 128 *)
  IF grad_stren <> 0
  THEN gv_dx := (gv_dx * 128 + INTEGER(grad_stren) DIV 2) 
                DIV INTEGER(grad_stren);
       gv_dy := (gv_dy * 128 + INTEGER(grad_stren) DIV 2)
                DIV INTEGER(grad_stren);
  END;

  (* smoothing of gradient vectors along the edges *)
  edge_grad_median_3x3(gv_dx, gv_dy);

  (* egalize gradient vector lengths to 128 *)
  length := CARDINAL(sqrt (FLOAT(gv_dx ** 2 + gv_dy ** 2)));
  IF length <> 0
  THEN gv_dx := (gv_dx * 128 + INTEGER(length) DIV 2)
                DIV INTEGER(length);
       gv_dy := (gv_dy * 128 + INTEGER(length) DIV 2)
                DIV INTEGER(length);
  END;

  (* calculate local curvature measurement along the edges *)
  local_curv := curv_mvar_3x3(gv_dx, gv_dy);

  (* smoothing of curvature measurement along the edges *)
  local_curv := edge_card_mean_3x3(local_curv);

  (* locate maxima of curvature measurement *)
  is_peak := card_is_peak_nxn(peak_radius, peak_global_thresh,
                              peak_local_thresh, local_curv);

  (* reduce corner labels to singular points *)
  is_corner := concen_peaks_3x3(is_peak);

  RETURN is_corner;
END detMeanvar;



(* corner detection using gradient vector variance curvature measurement *)
PROCEDURE detPairvar(minmax_radius: CARDINAL;
                     global_thresh, local_thresh: REAL;
                     peak_radius: CARDINAL;
                     peak_global_thresh, peak_local_thresh: REAL;
                     image: grid OF gray;
                     VAR dx1, dy1, dx2, dy2: grid OF INTEGER)
                     : grid OF BOOLEAN;
VAR med_image: grid OF gray;
    bin_image: grid OF binary; 
    gv_dx, gv_dy: grid OF INTEGER;
    grad_stren, local_min, local_max, global_max,
    local_curv, length: grid OF CARDINAL;
    is_edge, is_peak, is_corner: grid OF BOOLEAN;

BEGIN (* detPairvar *)
  (* edge preserving image smoothing *)
  med_image := median_5x5(image);

  (* calculate grayvalue gradients *) 
  gv_dx := (sobel_x_5x5(med_image) + 512) DIV 1024;
  gv_dy := (sobel_y_5x5(med_image) + 512) DIV 1024;

  (* calculate gradient strength *)
  grad_stren := CARDINAL(sqrt (FLOAT(gv_dx ** 2 + gv_dy ** 2)));

  (* eliminate regions with low gradient strength *)
  int_minmax_nxn(minmax_radius, grad_stren, local_min, local_max);
  global_max := REDUCE.MAX(grad_stren);
  IF(grad_stren < CARDINAL(FLOAT(global_max) * global_thresh))
  OR(grad_stren < CARDINAL((local_thresh * FLOAT(local_max) 
                         + (1.0 - local_thresh) * FLOAT(local_min))))
  THEN is_edge := FALSE; gv_dx := 0; gv_dy := 0; grad_stren := 0
  ELSE is_edge := TRUE
  END;

  (* egalize gradient vector lengths to 128 *)
  IF grad_stren <> 0
  THEN gv_dx := (gv_dx * 128 + INTEGER(grad_stren) DIV 2) 
                DIV INTEGER(grad_stren);
       gv_dy := (gv_dy * 128 + INTEGER(grad_stren) DIV 2)
                DIV INTEGER(grad_stren);
  END;

  (* smoothing of gradient vectors along the edges *)
  edge_grad_median_3x3(gv_dx, gv_dy);

  (* egalize gradient vector lengths to 128 *)
  length := CARDINAL(sqrt (FLOAT(gv_dx ** 2 + gv_dy ** 2)));
  IF length <> 0
  THEN gv_dx := (gv_dx * 128 + INTEGER(length) DIV 2)
                DIV INTEGER(length);
       gv_dy := (gv_dy * 128 + INTEGER(length) DIV 2)
                DIV INTEGER(length);
  END;

  (* thinning of contour line to width 1 *)
  IF is_edge THEN bin_image := b_black ELSE bin_image := b_white END;
  bin_image := lueWang(bin_image);
  IF bin_image = b_black
  THEN is_edge := TRUE 
  ELSE is_edge := FALSE; gv_dx := 0; gv_dy := 0; grad_stren := 0
  END;

  (* calculate local curvature measurement along the edges *)
  local_curv := curv_pvar_5x5(gv_dx, gv_dy, dx1, dy1, dx2, dy2);

  (* smoothing of curvature measurement along the edges *)
  local_curv := edge_card_mean_3x3(local_curv);

  (* locate maxima of curvature measurement *)
  is_peak := card_is_peak_nxn(peak_radius, peak_global_thresh,
                              peak_local_thresh, local_curv);

  (* reduce corner labels to singular points *)
  is_corner := concen_peaks_3x3(is_peak);

  RETURN is_corner;
END detPairvar;


BEGIN (* no init necessary *)
END Corners.

