IMPLEMENTATION MODULE FourierTransform;
(* Stefan Feyrer, Univ. Stuttgart 1995 *)

(* ------------------------------------------------------------------------ *)

FROM ImageIO           IMPORT gray;
FROM Local             IMPORT grid, left, right, up, down;
FROM ComplexArithmetic IMPORT complex, complex_abs, complex_add, complex_sub,
                              complex_mult;

(* ------------------------------------------------------------------------ *)

TYPE direction_type = (vertical, horizontal);

(* ------------------------------------------------------------------------ *)

PROCEDURE minimum (a, b: VECTOR OF CARDINAL): VECTOR OF CARDINAL;

  VAR c: VECTOR OF CARDINAL;

  BEGIN (* minimum *)
    IF a < b THEN
      c := a;
    ELSE
      c := b;
    END; (* IF *)
    RETURN c; 
  END minimum;

(* ------------------------------------------------------------------------ *)
(* Computes the x-th power of w=e^(-2*pi*i/size).                           *)

PROCEDURE w_power (x: VECTOR OF CARDINAL; size: CARDINAL): VECTOR OF complex;

  VAR result: VECTOR OF complex;
      temp:   VECTOR OF REAL;

  BEGIN (* w_power *)
    temp := -2.0 * pi * FLOAT (x) / FLOAT (size);
    result.re := cos (temp);
    result.im := sin (temp);
    RETURN result;
  END w_power;

(* ------------------------------------------------------------------------ *)
(* Returns the smallest power of two which is greater or equal n.           *)

PROCEDURE get_size (n: CARDINAL): CARDINAL;

  VAR result: CARDINAL;

  BEGIN (* get_size *)
    result := 1;
    WHILE result < n DO
      result := result * 2;
    END; (* WHILE *)
    RETURN result;
  END get_size;

(* ------------------------------------------------------------------------ *)
(* Returns the bit-reverse number of n.                                     *)

PROCEDURE reverse_number (n:    VECTOR OF CARDINAL;
                          size: CARDINAL): VECTOR OF CARDINAL;

  VAR result: VECTOR OF CARDINAL;
      power:  CARDINAL;
      i:      CARDINAL;

  BEGIN (* reverse_number *)
    IF n < size THEN
      result := 0;
      power := size;
      FOR i := 1 TO size DO
        power := power DIV 2;
        IF ODD (n) THEN
          result := result + power;
        END; (* IF *)
        n := n DIV 2;
      END; (* FOR *)
    ELSE
      result := n;
    END; (* IF *)
    RETURN result;
  END reverse_number;

(* ------------------------------------------------------------------------ *)
(* Exchanges the image values according to the bit-reverse function.        *)

PROCEDURE reverse (VAR image:     grid OF complex;
                       size:      CARDINAL;
                       direction: direction_type);

  VAR rev: grid OF CARDINAL;

  BEGIN (* reverse *)
    IF direction = horizontal THEN
      rev := reverse_number (DIM (grid, 1)-LOWER (grid, 1), size) + LOWER (grid, 1);
      SEND.<:*, rev:> (image, image);
    ELSE
      rev := reverse_number (DIM (grid, 2)-LOWER (grid, 2), size) + LOWER (grid, 2);
      SEND.<:rev, *:> (image, image);
    END; (* IF *)
  END reverse;

(* ------------------------------------------------------------------------ *)
(* Performs the actual transformation step.                                 *)

PROCEDURE transform (VAR image:     grid OF complex;
                         size:      CARDINAL;
                         direction: direction_type);

  VAR two_power_m: CARDINAL;
      log_of_size: CARDINAL;
      m:           CARDINAL;
      p:           CARDINAL;
      q:           CARDINAL;
      e:           grid OF complex;
      image_plus:  grid OF complex;
      temp:        grid OF complex;
      zero_dim:    grid OF CARDINAL;

  BEGIN (* transform *)
    two_power_m := 1;
    log_of_size := TRUNC (ln (FLOAT (size)) / ln (2.0));
    IF direction = horizontal THEN
      zero_dim := DIM (grid, 1) - LOWER (grid, 1);
      FOR m := 1 TO log_of_size DO
        p := two_power_m;
        two_power_m := two_power_m * 2;
        q := size DIV two_power_m;
        e := w_power ((zero_dim MOD two_power_m) * q, size);
        SEND.left:p (image, image_plus);
        IF zero_dim MOD two_power_m = zero_dim MOD p THEN
          temp := complex_mult (image_plus, e);
          image_plus := complex_sub (image, temp);
          image := complex_add (image, temp);
          SEND.right:p (image_plus, image);
        END; (* IF *)
      END; (* FOR *)
    ELSE
      zero_dim := DIM (grid, 2) - LOWER (grid, 2);
      FOR m := 1 TO log_of_size DO
        p := two_power_m;
        two_power_m := two_power_m * 2;
        q := size DIV two_power_m;
        e := w_power ((zero_dim MOD two_power_m) * q, size);
        SEND.down:p (image, image_plus);
        IF zero_dim MOD two_power_m # zero_dim MOD p THEN
          temp := complex_mult (image_plus, e);
          image_plus := complex_sub (image, temp);
          image := complex_add (image, temp);
          SEND.up:p (image_plus, image);
        END; (* IF *)
      END; (* FOR *)
    END; (* IF *)
  END transform;

(* ------------------------------------------------------------------------ *)
(* Performs the Fourier transform of a complex image. If 'division' is      *)
(* TRUE, the result is divided by the length of the transformation vector.  *)

PROCEDURE complex_fft (VAR image:     grid OF complex;
                           width:     CARDINAL;
                           height:    CARDINAL;
                           direction: direction_type;
                           division:  BOOLEAN);

  VAR size: CARDINAL;

  BEGIN (* complex_fft *)
    IF direction = horizontal THEN
      size := get_size (width);
      IF width < DIM (grid, 1)-LOWER (grid, 1)+1 <= size THEN
        image.re := 0.0;
        image.im := 0.0;
      END; (* IF *)
    ELSE
      size := get_size (height);
      IF height < DIM (grid, 2)-LOWER (grid, 2)+1 <= size THEN
        image.re := 0.0;
        image.im := 0.0;
      END; (* IF *)
    END; (* IF *)
    reverse (image, size, direction);
    transform (image, size, direction);
    IF division THEN
      image.re := image.re / FLOAT (size);
      image.im := image.im / FLOAT (size);
    END; (* IF *)
  END complex_fft;

(* ------------------------------------------------------------------------ *)
(* Fast-Fourier-Transform (forward)                                         *)

PROCEDURE fft (image: grid OF gray; width, height: CARDINAL): grid OF complex;

  VAR result: grid OF complex;

  BEGIN (* fft *)
    result.re  := FLOAT (image);
    result.im  := 0.0;
    complex_fft (result, width, height, horizontal, TRUE);
    complex_fft (result, width, height, vertical,   TRUE);
    RETURN result;
  END fft;

(* ------------------------------------------------------------------------ *)
(* Fast-Fourier-Transform (inverse)                                         *)

PROCEDURE inverse_fft (image:         grid OF complex; 
                       width, height: CARDINAL): grid OF gray;

  VAR result: grid OF gray;

  BEGIN (* inverse_fft *)
    image.im := - image.im;
    complex_fft (image, width, height, horizontal, FALSE);
    complex_fft (image, width, height, vertical,   FALSE);
    result := minimum (CARDINAL (complex_abs (image)), grid (MAX (gray)));
    RETURN result;
  END inverse_fft;

(* ------------------------------------------------------------------------ *)
(* Returns the Fourier spectrum of 'image' as a gray scale image.           *)
(* The parameters 'centred' and 'log_scale' determine the kind of           *)
(* representation of the Fourier spectrum:                                  *)
(* centred = TRUE:    high frequencies are displayed in the middle          *)
(*                    of the spectrum image                                 *)
(* centred = FALSE:   low frequencies are displayed in the middle of        *)
(*                    the spectrum image                                    *)
(* log_scale = TRUE:  logarithmic scaled spectrum                           *)
(* log_scale = FALSE: linearly scaled spectrum                              *)

PROCEDURE fourier_spectrum (image:     grid OF gray;
                            width:     CARDINAL;
                            height:    CARDINAL;
                            centred:   BOOLEAN;
                            log_scale: BOOLEAN): grid OF gray;

  VAR temp_image:    grid OF complex;
      result:        grid OF gray;
      log_pixel:     grid OF REAL;
      max_log_pixel: REAL;

  BEGIN (* fourier_spectrum *)
    temp_image.re := FLOAT (image);
    temp_image.im := 0.0;

    (* To get a centered representation, the image has to be inverted at    *)
    (* 'the black sqares of the chessboard'.                                *)

    IF centred THEN
      IF (DIM (grid, 1) + DIM (grid, 2)) MOD 2 = 1 THEN
        temp_image.re := -temp_image.re;
      END; (* IF *)
    END; (* IF *)
    complex_fft (temp_image, width, height, horizontal, TRUE);
    complex_fft (temp_image, width, height, vertical,   TRUE);
    IF log_scale THEN
      log_pixel := ln (1.0 + complex_abs (temp_image));
      max_log_pixel := REDUCE.MAX (log_pixel);
      result := CARDINAL (log_pixel/max_log_pixel*255.0);
    ELSE
      result := minimum (CARDINAL (complex_abs (temp_image)), grid (MAX (gray)));
    END; (* IF *)
    RETURN result;
  END fourier_spectrum;

(* ------------------------------------------------------------------------ *)

END FourierTransform.