IMPLEMENTATION MODULE Gauss;
(* Michael Lampe, Univ. Stuttgart, March 1995 *)

FROM Local   IMPORT grid, left, right, up, down;
FROM ImageIO IMPORT gray;

(*****************************************************************************
     'sgn' computes the sign of a real valued vector
******************************************************************************)
PROCEDURE sgn(x: VECTOR OF REAL): VECTOR OF INTEGER;
  VAR s: VECTOR OF INTEGER;
BEGIN
  IF x>0.0 THEN
    s:=1;
  ELSIF x<0.0 THEN
    s:=-1;
  ELSE
    s:=0;
  END;
  RETURN s;
END sgn;


(***************************************************************************
     'sum_r' computes local sum of REALS
****************************************************************************)
PROCEDURE sum_r(x: grid OF REAL; size: grid OF CARDINAL): grid OF REAL;
  VAR i,l: grid OF CARDINAL;
      s,t1,t2: grid OF REAL;
BEGIN
  l:=(size-1) DIV 2;
  s:=x; t1:=x; t2:=x;
  FOR i:=1 TO l DO
    t1:=MOVE.left(t1);
    t2:=MOVE.right(t2);
    s:=s+t1+t2;
  END;
  t1:=s; t2:=s;
  FOR i:=1 TO l DO
    t1:=MOVE.up(t1);
    t2:=MOVE.down(t2);
    s:=s+t1+t2;
  END;
  RETURN s;
END sum_r;


(******************************************************************************
     'coeff_LoG' computes the coefficients for a Laplacian of Gaussian filter
******************************************************************************)
PROCEDURE coeff_LoG(VAR coeff1, coeff2: grid OF coeff_type;
                    VAR nr_coeff      : grid OF CARDINAL;
                    sigma             : REAL);
VAR c1,c2: coeff_type;
    k1,k2,k3,k4,i2,s1,s2,n: REAL;
    i,l: CARDINAL;
BEGIN
  k1:=1.0/(sigma*sqrt(2.0*pi));    (* constants to calculate 1D Gaussian   *)
  k2:=1.0/(sigma*sigma);           (* function G and its second derivative *)
  k3:=-0.5*k2;
  k4:=k2*k2;
  l:=round(3.0*sqrt(2.0)*sigma);   (* number of coefficients to compute, *)
  s1:=0.0;                         (*                experimental result *)  
  s2:=s1;
  FOR i:=0 TO l DO                 (* compute coefficients *)
    i2:=FLOAT(i*i);
    c1[i]:=k1*exp(k3*i2);          (* ...=G(i)   *)
    c2[i]:=(k4*i2-k2)*c1[i];       (* ...=G''(i) *)
    s1:=s1+2.0*c1[i];
    s2:=s2+2.0*c2[i];
  END;
  n:=FLOAT(2*l+1);                 (* correct coefficients *)
  s1:=(1.0-s1+c1[0])/n;           
  s2:=(s2-c2[0])/n;
  FOR i:=0 TO l DO
    c1[i]:=c1[i]+s1;               (* sum c1[] must be 1.0 *)
    c2[i]:=c2[i]-s2;               (* sum c2[] must be 0.0 *)
  END;
  coeff1:=grid(c1);
  coeff2:=grid(c2);
  nr_coeff:=l;
END coeff_LoG;

(******************************************************************************
     'LoG' filters an intensity image with a LoG operator 
******************************************************************************)
PROCEDURE LoG(img               : grid OF gray;
              VAR coeff1, coeff2: grid OF coeff_type;
              nr_coeff          : grid OF CARDINAL
             ): grid OF REAL;
VAR imgf1, imgf2, imgf3, imgf4, t1, t2: grid OF REAL;
    z: grid OF CARDINAL;
BEGIN                                   (* using the separability the LoG *)
  imgf1:=FLOAT(img); imgf2:=imgf1;      (* operator is computed with four *)
  imgf3:=imgf1;  imgf4:=imgf1;          (* 1D convolutions                *)
  t1:=coeff2[0]*imgf1;
  FOR z:=1 TO nr_coeff DO
    imgf1:=MOVE.up(imgf1);
    imgf2:=MOVE.down(imgf2);
    t1:=t1+coeff2[z]*(imgf1+imgf2);
  END;
  imgf1:=t1;  imgf2:=t1;
  t1:=coeff1[0]*imgf1;
  FOR z:=1 TO nr_coeff DO
    imgf1:=MOVE.left(imgf1);
    imgf2:=MOVE.right(imgf2);
    t1:=t1+coeff1[z]*(imgf1+imgf2);
  END;
  t2:=coeff2[0]*imgf3;
  FOR z:=1 TO nr_coeff DO
    imgf3:=MOVE.left(imgf3);
    imgf4:=MOVE.right(imgf4);
    t2:=t2+coeff2[z]*(imgf3+imgf4);
  END;
  imgf3:=t2;  imgf4:=t2;
  t2:=coeff1[0]*imgf3;
  FOR z:=1 TO nr_coeff DO
    imgf3:=MOVE.up(imgf3);
    imgf4:=MOVE.down(imgf4);
    t2:=t2+coeff1[z]*(imgf3+imgf4);
  END;
  RETURN t1+t2;
END LoG;

(******************************************************************************
     'coeff_Gauss' computes the coefficients for a Gauss filter
******************************************************************************)
PROCEDURE coeff_Gauss(VAR coeff   : grid OF coeff_type;
                      VAR nr_coeff: grid OF CARDINAL;
		      sigma       : REAL);
VAR c1,c2: coeff_type;
    k1,k2,i2,s1,n: REAL;
    i,l: CARDINAL;
BEGIN                               (* derived form coeff_LoG *)
  k1:=1.0/(sigma*sqrt(2.0*pi));
  k2:=-0.5/(sigma*sigma);
  l:=round(3.0*sqrt(2.0)*sigma);
  s1:=0.0;
  FOR i:=0 TO l DO
    i2:=FLOAT(i*i);
    c1[i]:=k1*exp(k2*i2);
    s1:=s1+2.0*c1[i];
  END;
  n:=FLOAT(2*l+1);
  s1:=(1.0-s1+c1[0])/n;
  FOR i:=0 TO l DO
    c1[i]:=c1[i]+s1;
  END;
  coeff:=grid(c1);
  nr_coeff:=l;
END coeff_Gauss;

(******************************************************************************
     'Gauss' filters an intensity image with a Gaussian operator
******************************************************************************)
PROCEDURE gauss(img      : grid OF gray;
                VAR coeff: grid OF coeff_type;
                nr_coeff : grid OF CARDINAL
               ): grid OF gray;

VAR imgf1, imgf2, t1: grid OF REAL;
    z: grid OF CARDINAL;
BEGIN                                   (* derived from LoG *)
  imgf1:=FLOAT(img); imgf2:=imgf1;
  t1:=coeff[0]*imgf1;
  FOR z:=1 TO nr_coeff DO
    imgf1:=MOVE.up(imgf1);
    imgf2:=MOVE.down(imgf2);
    t1:=t1+coeff[z]*(imgf1+imgf2);
  END;
  imgf1:=t1;  imgf2:=t1;
  t1:=coeff[0]*imgf1;
  FOR z:=1 TO nr_coeff DO
    imgf1:=MOVE.left(imgf1);
    imgf2:=MOVE.right(imgf2);
    t1:=t1+coeff[z]*(imgf1+imgf2);
  END;
  RETURN round(t1);
END gauss;

(******************************************************************************
     'DoG' filters an intensity image with a difference of Gaussian operator
******************************************************************************)
PROCEDURE DoG(img: grid OF gray; 
              VAR coeff1: grid OF coeff_type; nr_coeff1: grid OF CARDINAL;
              VAR coeff2: grid OF coeff_type; nr_coeff2: grid OF CARDINAL
             ): grid OF INTEGER;
BEGIN
  RETURN gauss(img, coeff1, nr_coeff1) - gauss(img, coeff2, nr_coeff2);
END DoG;

(******************************************************************************
     'gauss_edge' detets edges in a gauss filtered image
******************************************************************************)
PROCEDURE gauss_edge(f_img: grid OF INTEGER): grid OF BOOLEAN;
VAR l,r,u,d: grid OF INTEGER;
BEGIN
  l:=MOVE.left(f_img);
  r:=MOVE.right(f_img);
  u:=MOVE.up(f_img);             (* zero crossings mark edges *)
  d:=MOVE.down(f_img);
  RETURN (f_img>0) AND ((l<0) OR (r<0) OR (u<0) OR (d<0)) OR
         (f_img=0) AND ((l<0) AND (r>0) OR (l>0) AND (r<0) OR
                        (u<0) AND (d>0) OR (u>0) AND (d<0));
END gauss_edge;

(******************************************************************************
    'energy' computes local energie of a filtered image
******************************************************************************)
PROCEDURE energy(f_img:grid OF REAL; p_size: grid OF CARDINAL): grid OF REAL;
BEGIN
  RETURN sum_r(f_img*f_img, p_size);  (* sum of squares in patch *)
END energy; 
                                            
END Gauss.
