IMPLEMENTATION MODULE Hist;
(* Braunl, Univ. Stuttgart, Nov. 1994 *)

FROM ImageIO IMPORT color, gray, binary,
                    g_black, g_white;
FROM Local   IMPORT grid;


PROCEDURE gen_histogram_n(img: grid OF gray; VAR num: gray): histogram;
(* gen. histogram by iterating over all pixels, return also number of entries *)
VAR res                   : histogram;
    px, py, last_x, last_y: INTEGER;
BEGIN
  last_x := REDUCE.LAST(DIM(grid,1));  (* max. column *)
  last_y := REDUCE.LAST(DIM(grid,2));  (* max. row    *)

  FOR px:=0 TO 255 DO res[px]:=0 END;
  FOR py:= LOWER(grid,2) TO last_y DO
    FOR px:= LOWER(grid,1) TO last_x DO
      INC(res[img<:py,px:>]);
    END;
  END;
  num := 0;
  FOR px:=0 TO 255 DO
    IF res[px] > 0 THEN INC(num) END;
  END;
  RETURN res;
END gen_histogram_n;


PROCEDURE gen_histogram(img: grid OF gray): histogram;
(* generate histogram by iterating over all gray values *)
VAR res: histogram;
    g  : gray;
BEGIN
  FOR g:= g_black TO g_white DO
    res[g] := REDUCE.SUM(ORD(img=g));
  END;
  RETURN res;
END gen_histogram;


PROCEDURE draw_histogram(VAR his: histogram; VAR canvas: grid OF gray;
          x_origin, y_origin, width, height: CARDINAL; foreground: gray);
(* display histogram in gray vector *)
VAR max_val, x_fac, pos: INTEGER;
    g                  : gray;
    val                : histogram;
BEGIN
  IF width < 1 THEN width := 1 END;  (* avoid endless loop *)
  x_fac := 1;
  WHILE x_fac * width < 256 DO x_fac := 2*x_fac  (* find power of 2 *)
  END;

  max_val := 0;
  FOR g:=g_black TO g_white DIV x_fac DO
    val[g] := 0;
    FOR pos := g*x_fac TO (g+1)*x_fac - 1 DO
      INC(val[g],his[pos])
    END;
    IF val[g] > max_val THEN max_val:=val[g] END;
  END;

  FOR g:=g_black TO g_white DIV x_fac DO
    IF (DIM(grid,1) - LOWER(grid,1) = x_origin + g) AND
       (y_origin >= DIM(grid,2)-LOWER(grid,2) >= y_origin -
                                                 val[g]*height DIV max_val)
      THEN canvas := foreground
    END;
  END;
END draw_histogram;

 
PROCEDURE equal_histogram(img : grid OF gray; his: histogram): grid OF gray;
(* remap gray values of complete image according to histogram *)
VAR res: grid OF gray;
    g  : gray;
    sum: CARDINAL;
BEGIN
  sum := 0;
  FOR g:=g_black TO g_white DO
    IF img = g THEN res := g_white * sum DIV LEN(grid) END;
    INC(sum, his[g]);
  END;
  RETURN res;
END equal_histogram;


PROCEDURE equal_histogram_n(img : grid OF gray): grid OF gray;
(* remap gray values according to histogram, gather new histogram data *)
VAR res     : grid OF gray;
    g,gpos  : gray;
    size,sum: CARDINAL;
BEGIN
  size := REDUCE.SUM(grid(1));  (* get number of active PEs *)
  sum  := 0;
  FOR g:=g_black TO g_white DO
    gpos := g_white * sum DIV size;
    IF img = g THEN
      res := gpos;
      INC(sum, REDUCE.SUM(grid(1)));
    END;
  END;
  RETURN res;
END equal_histogram_n;

END Hist.

