IMPLEMENTATION MODULE ImageIO;
(* Thomas Braunl, Sep. 1994 *)
(* ********************** *)
(* auxiliary procedures   *)
(* ********************** *)
TYPE string = ARRAY[1..100] OF CHAR;

  PROCEDURE error(mes: ARRAY OF CHAR);
  (* display errormessage and halt *)
  BEGIN
    WriteString("ERROR: "); WriteString(mes); WriteLn;
    HALT;
  END error;

  PROCEDURE read_i(VAR i: INTEGER);
  (* read integer from image file, disregards comments *)
  VAR s : string;
  BEGIN
   ReadInt(i);
   WHILE NOT Done DO
    WHILE (termCH <> EOL) DO
      ReadString(s);
      IF NOT Done THEN error("image comment") END;
    END;
    ReadInt(i);
   END
  END read_i;

(* ********************** *)
(* procedures for READING *)
(* ********************** *)

PROCEDURE read_c_image (VAR im: VECTOR OF color;  filename: ARRAY OF CHAR;             
                        VAR width,height: CARDINAL);
VAR ch                    : CHAR;
    x_pos,y_pos,pos,maxcol: INTEGER;
    max_height,max_width  : CARDINAL;
    magic                 : string;

BEGIN
  max_width  := UPPER(im,1) - LOWER(im,1) + 1;
  max_height := UPPER(im,2) - LOWER(im,2) + 1;
  OpenInput(filename);
  IF NOT Done THEN error("file not found") END;
  WriteString("reading color  image '");
  WriteString(filename); WriteString("' .. ");
  ReadString(magic);
  IF magic <> "P6" THEN error("wrong file format - please use ppm P6"); END;
  read_i(width); read_i(height); read_i(maxcol);  
  Read(ch) ; (* for whitespace *)
  WriteInt(height,1); WriteString(" x "); WriteInt(width,1);
  WriteString(" Pixels - max color "); WriteInt(maxcol,1); WriteLn;
  IF (height>max_height) OR (width>max_width) THEN 
    error("image too large")
  END;

  IF maxcol=255 THEN  (* standard *)
   FOR y_pos := 1 TO height DO
    FOR x_pos := 1 TO width DO
      pos := (y_pos-1) * max_width + x_pos;
      Read(ch); im.red  <<pos>> := ORD(ch); 
      Read(ch); im.green<<pos>> := ORD(ch); 
      Read(ch); im.blue <<pos>> := ORD(ch);
    END
   END
  ELSE  (* special *)
   FOR y_pos := 1 TO height DO
    FOR x_pos := 1 TO width DO
      pos := (y_pos-1) * max_width + x_pos;
      Read(ch); im.red  <<pos>> := (ORD(ch)*255) DIV maxcol; 
      Read(ch); im.green<<pos>> := (ORD(ch)*255) DIV maxcol;
      Read(ch); im.blue <<pos>> := (ORD(ch)*255) DIV maxcol;
    END;
   END;
  END; (* IF *)

  CloseInput;
  IF (DIM(im,1)-LOWER(im,1) >= width) OR (DIM(im,2)-LOWER(im,2) >= height) THEN
    im := c_white;  (* clear remaining space *)
  END;
END read_c_image;


PROCEDURE read_g_image (VAR im: VECTOR OF gray;   filename: ARRAY OF CHAR;          
                        VAR width,height: CARDINAL);
VAR ch                    : CHAR;
    x_pos,y_pos,pos,maxcol: INTEGER;
    max_height,max_width  : CARDINAL;
    magic                 : string;

BEGIN
  max_width  := UPPER(im,1) - LOWER(im,1) + 1;
  max_height := UPPER(im,2) - LOWER(im,2) + 1;
  OpenInput(filename);
  IF NOT Done THEN error("file not found") END;
  WriteString("reading gray   image '"); 
  WriteString(filename); WriteString("' .. ");
  ReadString(magic);
  IF magic <> "P5" THEN error("wrong file format - please use pgm P5"); END;
  read_i(width); read_i(height); read_i(maxcol);
  Read(ch) ; (* for whitespace *)
  WriteInt(height,1); WriteString(" x "); WriteInt(width,1);
  WriteString(" Pixels - max color "); WriteInt(maxcol,1); WriteLn;
  IF (height>max_height) OR (width>max_width) THEN 
    error("image too large")
  END;

  IF maxcol=255 THEN  (* standard *)
   FOR y_pos := 1 TO height DO
    FOR x_pos := 1 TO width DO
      pos := (y_pos-1) * max_width + x_pos;
      Read(ch); im<<pos>> := ORD(ch);
    END
   END
  ELSE  (* special *)
   FOR y_pos := 1 TO height DO
    FOR x_pos := 1 TO width DO
      pos := (y_pos-1) * max_width + x_pos;
      Read(ch); im<<pos>> := (ORD(ch)*255) DIV maxcol;
    END
   END
  END; (* IF *)

  CloseInput;
  IF (DIM(im,1)-LOWER(im,1) >= width) OR (DIM(im,2)-LOWER(im,2) >= height) THEN
    im := g_white;  (* clear remaining space *)
  END;
END read_g_image;


PROCEDURE read_b_image (VAR im: VECTOR OF binary; filename: ARRAY OF CHAR;
                        VAR width,height: CARDINAL);
VAR ch                    : CHAR;
    count,num,maxcol,
    pos,x_pos,y_pos,
    max_height,max_width  : INTEGER;
    magic                 : string;

BEGIN
  max_width  := UPPER(im,1) - LOWER(im,1) + 1;
  max_height := UPPER(im,2) - LOWER(im,2) + 1;
  OpenInput(filename);
  IF NOT Done THEN error("file not found") END;
  WriteString("reading binary image '"); 
  WriteString(filename); WriteString("' .. ");
  ReadString(magic);
  IF magic <> "P4" THEN error("wrong file format - please use pbm P4"); END;
  read_i(width); read_i(height);
  Read(ch) ; (* for whitespace *)
  WriteInt(height,1); WriteString(" x "); WriteInt(width,1); 
  WriteString(" Pixels"); WriteLn;
  IF (height>max_height) OR (width>max_width) THEN 
    error("image too large")
  END;

  y_pos := 1;
  x_pos := 1;
  count := 0;
  WHILE y_pos <= height DO
    IF count=0 THEN
      Read(ch); num := ORD(ch);
      count := 8;
    END;
    pos := (y_pos-1) * max_width + x_pos;
    im<<pos>> :=  num >= 2**7;   (* test high bit   *)
    DEC(count);
    num := (2*num) MOD 256; (* shift left byte *)
    INC(x_pos);
    IF x_pos > width THEN
      INC(y_pos);
      x_pos := 1;
      count := 0;
    END;
  END;

  CloseInput;
  IF (DIM(im,1)-LOWER(im,1) >= width) OR (DIM(im,2)-LOWER(im,2) >= height) THEN
    im := b_white;  (* clear remaining space *)
  END;
END read_b_image;


(* ********************** *)
(* procedures for WRITING *)
(* ********************** *)

PROCEDURE write_c_image (im: VECTOR OF color;  filename: ARRAY OF CHAR;  
                         width,height: CARDINAL);
CONST maxcol = 255;
VAR   max_width, pos,x_pos,y_pos: INTEGER;

BEGIN
  max_width  := UPPER(im,1) - LOWER(im,1) + 1;
  WriteString("writing color  image '") ; 
  WriteString( filename ); WriteString("' .. " );
  WriteInt(height,1); WriteString(" x "); WriteInt(width,1);
  WriteString(" Pixels - max color "); WriteInt(maxcol,1); WriteLn;
  OpenOutput( filename );
  WriteString ("P6"); WriteLn;    (* RAWBIT option, not ASCII *)
  WriteString ("# CREATOR: Parallaxis-III Univ. Stuttgart"); WriteLn;
  WriteInt(width,1); Write(' '); WriteInt(height,1); WriteLn;
  WriteInt(maxcol,1); WriteLn;    (* max color-component value *)
  
  FOR y_pos := 1 TO height DO
   FOR x_pos := 1 TO width DO
    pos := (y_pos-1) * max_width + x_pos;
    WITH im<<pos>> DO
      Write(CHR(red)); Write(CHR(green)); Write(CHR(blue));
    END;
   END;
  END;
  CloseOutput;
END write_c_image;


PROCEDURE write_g_image (im: VECTOR OF gray;   filename: ARRAY OF CHAR;  
                         width,height: CARDINAL);                
CONST maxcol = 255;
VAR   max_width, pos,x_pos,y_pos: INTEGER;

BEGIN
  max_width  := UPPER(im,1) - LOWER(im,1) + 1;
  WriteString("writing gray   image '") ;
  WriteString( filename ); WriteString("' .. " );
  WriteInt(height,1); WriteString(" x "); WriteInt(width,1);
  WriteString(" Pixels - max color "); WriteInt(maxcol,1); WriteLn;
  OpenOutput( filename );
  WriteString ("P5"); WriteLn;    (* RAWBIT option, not ASCII *)
  WriteString ("# CREATOR: Parallaxis-III Univ. Stuttgart"); WriteLn;
  WriteInt(width,1); Write(' '); WriteInt(height,1); WriteLn;
  WriteInt(maxcol,1); WriteLn;    (* max color-component value *)

  FOR y_pos := 1 TO height DO
   FOR x_pos := 1 TO width DO
    pos := (y_pos-1) * max_width + x_pos;
    Write(CHR(im<<pos>>));
   END;
  END;
  CloseOutput;
END write_g_image;


PROCEDURE write_b_image (im: VECTOR OF binary; filename: ARRAY OF CHAR; 
                         width,height: CARDINAL);
CONST maxcol = 255;
VAR   max_width, count,
      x_pos,y_pos,pos  : INTEGER;
      b8               : [0..255];

BEGIN
  max_width  := UPPER(im,1) - LOWER(im,1) + 1;
  WriteString("writing binary image '") ;
  WriteString( filename ); WriteString("' .. " );
  WriteInt(height,1); WriteString(" x "); WriteInt(width,1);
  WriteString(" Pixels"); WriteLn;
  OpenOutput( filename );
  WriteString ("P4"); WriteLn;    (* RAWBIT option, not ASCII *)
  WriteString ("# CREATOR: Parallaxis-III Univ. Stuttgart"); WriteLn;
  WriteInt(width,1); Write(' '); WriteInt(height,1); WriteLn;

  y_pos := 1;
  x_pos := 1;
  count := 0;
  b8 := 0;
  WHILE y_pos <= height DO
    pos := (y_pos-1) * max_width + x_pos;
    b8 := 2*b8 + ORD(im<<pos>>);  (* shift left and set lower bit *)
    INC(count);
    IF count=8 THEN
      Write(CHR(b8));
      count := 0;  b8 := 0;
    END;
    INC(x_pos);
    IF x_pos > width THEN
      INC(y_pos);
      x_pos := 1;
      IF count>0 THEN
        b8 := b8 * 2**(8-count);   (* shift left several steps *)
        Write(CHR(b8));
        count := 0;  b8 := 0;
      END;
    END;
  END;
  IF count>0 THEN
    b8 := b8 * 2**(8-count);   (* shift left several steps *)
    Write(CHR(b8));
  END;
  CloseOutput;
END write_b_image;

BEGIN  (* no init necessary *)
END ImageIO.

