IMPLEMENTATION MODULE Local;
(* Braunl, Jochum, Schaper, Univ. Stuttgart, Sep. 1994 *)

FROM ImageIO IMPORT gray, g_white, binary;

PROCEDURE sum_3x3(img: grid OF gray): grid OF INTEGER;
(* returns sum of local 3x3 neighborhood area *)
VAR res: grid OF INTEGER;
BEGIN
  res:= img + MOVE.right(img) + MOVE.left(img); (* horizontal *)
  res:= res + MOVE.down(res)  + MOVE.up(res);   (* vertical   *)
  RETURN res;
END sum_3x3;


PROCEDURE sum_5x5(img: grid OF gray): grid OF INTEGER;
(* returns sum of local 5x5 neighborhood area *)
VAR temp1, temp2, res: grid OF INTEGER;
BEGIN
  (* horizontal sum *)
  temp1 := MOVE.left (img);
  temp2 := MOVE.right(img);
  res:= img + temp1 + temp2 + MOVE.left(temp1) + MOVE.right(temp2);
  (* vertical sum *)
  temp1 := MOVE.up  (res);
  temp2 := MOVE.down(res);
  res := res + temp1 + temp2 + MOVE.up(temp1) + MOVE.down(temp2);
  RETURN res;
END sum_5x5;


PROCEDURE sum_op (img: grid OF gray; op_size: CARDINAL): grid OF INTEGER;
(* returns sum of local opxop neighborhood area *)
VAR res,temp1,temp2 : grid OF INTEGER;
    i,j             : INTEGER;
BEGIN
  res := img;
  (* horizontal sum *)
  temp1 := img;
  temp2 := img;
  FOR i:=1 TO (op_size DIV 2)-1 DO
    temp1 := MOVE.right(temp1);
    temp2 := MOVE.left (temp2); 
    res   := res + temp1 + temp2;
  END;
  (* vertical sum *)
  temp1 := res;
  temp2 := res;
  FOR i:=1 TO (op_size DIV 2)-1 DO
    temp1 := MOVE.down(temp1);
    temp2 := MOVE.up  (temp2);
    res   := res + temp1 + temp2;
  END; 
  RETURN res;
END sum_op;

 
PROCEDURE laplace_3x3(img: grid OF gray): grid OF INTEGER;
(*  0 -1  0  Laplace operator    *)
(* -1  4 -1  in 3x3 neighborhood *)
(*  0 -1  0                      *)
BEGIN
  RETURN 4*img -MOVE.up(img) -MOVE.down(img) -MOVE.left(img) -MOVE.right(img);
END laplace_3x3;


PROCEDURE sobel_x_3x3(img: grid OF gray): grid OF INTEGER;
(* -1  0  1  Sobel operator in x-direction *)
(* -2  0  2  in 3x3 neighborhood           *)
(* -1  0  1                                *)
VAR col: grid OF INTEGER;
BEGIN
  col := 2*img + MOVE.up(img) + MOVE.down(img); 
  RETURN MOVE.left(col) - MOVE.right(col);
END sobel_x_3x3;


PROCEDURE sobel_y_3x3(img: grid OF gray): grid OF INTEGER;
(*  1  2  1  Sobel operator in y-direction *)
(*  0  0  0  in 3x3 neighborhood           *)
(* -1 -2 -1                                *)
VAR row: grid OF INTEGER;
BEGIN
  row := 2*img + MOVE.left(img) + MOVE.right(img); 
  RETURN MOVE.down(row) - MOVE.up(row);
END sobel_y_3x3;


PROCEDURE sobel_x_5x5(img: grid OF gray): grid OF INTEGER;
(* -10 -10   0  10  10  Sobel operator in x-direction *)
(* -17 -17   0  17  17  in 5x5 neighborhood           *)
(* -20 -20   0  20  20                                *)
(* -17 -17   0  17  17                                *)
(* -10 -10   0  10  10                                *)
VAR col,t1,t2: grid OF INTEGER;
BEGIN
  t1 := MOVE.up  (img);
  t2 := MOVE.down(img);
  col := 20*img + 17*(t1+t2) + 10*(MOVE.up(t1)+MOVE.down(t2));
  t1 := MOVE.left (col);
  t2 := MOVE.right(col);
  RETURN t1 + MOVE.left(t1) - t2 - MOVE.right(t2);
END sobel_x_5x5;


PROCEDURE sobel_y_5x5(img: grid OF gray): grid OF INTEGER;
(*  10  17  20  17  10  Sobel operator in y-direction *)
(*  10  17  20  17  10  in 5x5 neighborhood           *)
(*  0    0   0   0   0                                *)
(* -10 -17 -20 -17 -10                                *)
(* -10 -17 -20 -17 -10                                *)
VAR row,t1,t2: grid OF INTEGER;
BEGIN
  t1 := MOVE.left (img);
  t2 := MOVE.right(img);
  row := 20*img + 17*(t1+t2) + 10*(MOVE.left(t1)+MOVE.right(t2));
  t1 := MOVE.up  (row);
  t2 := MOVE.down(row);
  RETURN t2 + MOVE.down(t2) - t1 - MOVE.up(t1);
END sobel_y_5x5;


(* auxiliary procedure *)
  PROCEDURE limit2gray(img: grid OF INTEGER): grid OF gray;
  (* limit positive integer values into interval [0..c_white] *)
  VAR res : grid OF gray;
      tmax: INTEGER;
  BEGIN (* assume img > 0 ! *)
    tmax := REDUCE.MAX(img);
    IF tmax > g_white THEN res := g_white * img DIV tmax
                      ELSE res := img
    END;
    RETURN res;
  END limit2gray;


PROCEDURE edges_sobel_3x3(img: grid OF gray; VAR strength,direction: grid OF gray);
(* Sobel edge strength and edge direction *)
VAR dx,dy: grid OF INTEGER;
BEGIN
  dx := sobel_x_3x3(img);
  dy := sobel_y_3x3(img);
  strength := limit2gray( ABS(dx) + ABS(dy) );
  direction:= round( (arctan2(FLOAT(dy),FLOAT(dx)) + pi) / (2.0*pi) * 255.0 );
END edges_sobel_3x3;


PROCEDURE edges_sobel_5x5(img: grid OF gray; VAR strength,direction: grid OF gray);
(* Sobel edge strength and edge direction *)
VAR dx,dy: grid OF INTEGER;
BEGIN
  dx := sobel_x_3x3(img);
  dy := sobel_y_3x3(img);
  strength := limit2gray( ABS(dx) + ABS(dy) );
  direction:= round( (arctan2(FLOAT(dy),FLOAT(dx)) + pi) / (2.0*pi) * 255.0 );
END edges_sobel_5x5;


PROCEDURE min_3x3(img: grid OF gray): grid OF gray;
(* minimum of 3x3 matrix *)
VAR res,t1,t2: grid OF gray;
BEGIN
  t1  := MOVE.left (img);
  t2  := MOVE.right(img);
  res := img;
  IF res > t1 THEN res:= t1 END;
  IF res > t2 THEN res:= t2 END;
  t1  := MOVE.up   (res);
  t2  := MOVE.down (res);                      
  IF res > t1 THEN res:= t1 END;
  IF res > t2 THEN res:= t2 END;
  RETURN res;
END min_3x3;


PROCEDURE min_op(img: grid OF gray; op_size: CARDINAL): grid OF gray;
(* minimum of op_size x op_size neighborhood *)
VAR res, t1,t2: grid OF gray;
    i         : INTEGER;
BEGIN
  t1  := img;
  t2  := img;
  res := img;
  (*horizontal minimum *)
  FOR i:=1 TO (op_size DIV 2)-1 DO
    t1 := MOVE.right(t1);
    t2 := MOVE.left (t2); 
    IF res > t1 THEN res:= t1 END;
    IF res > t2 THEN res:= t2 END;
  END;
  t1 := res;
  t2 := res;
  (* vertical minimum *)
  FOR i:=1 TO (op_size DIV 2)-1 DO
    t1 := MOVE.up  (t1);
    t2 := MOVE.down(t2); 
    IF res > t1 THEN res:= t1 END;
    IF res > t2 THEN res:= t2 END;
  END;
  RETURN res;
END min_op;


PROCEDURE max_3x3(img: grid OF gray): grid OF gray;
(* maximum of 3x3 matrix *)
VAR res,t1,t2: grid OF gray;
BEGIN
  t1  := MOVE.left (img);
  t2  := MOVE.right(img);
  res := img;
  IF res < t1 THEN res:= t1 END;
  IF res < t2 THEN res:= t2 END;
  t1  := MOVE.up   (res);
  t2  := MOVE.down (res);                      
  IF res < t1 THEN res:= t1 END;
  IF res < t2 THEN res:= t2 END;
  RETURN res;
END max_3x3;


PROCEDURE max_op(img: grid OF gray; op_size: CARDINAL): grid OF gray;
(* maximum of op_size x op_size neighborhood *)
VAR res, t1,t2: grid OF gray;
    i         : INTEGER;
BEGIN
  t1  := img;
  t2  := img;
  res := img;
  (*horizontal minimum *)
  FOR i:=1 TO (op_size DIV 2)-1 DO
    t1 := MOVE.right(t1);
    t2 := MOVE.left (t2);
    IF res < t1 THEN res:= t1 END;
    IF res < t2 THEN res:= t2 END;
  END;
  t1 := res;
  t2 := res;
  (* vertical minimum *)
  FOR i:=1 TO (op_size DIV 2)-1 DO
    t1 := MOVE.up  (t1);
    t2 := MOVE.down(t2);
    IF res < t1 THEN res:= t1 END;
    IF res < t2 THEN res:= t2 END;
  END;
  RETURN res;
END max_op;


PROCEDURE mean_3x3(img: grid OF gray): grid OF gray;
(* mean value of 3x3 neighborhood *)
BEGIN
  RETURN sum_3x3(img) DIV 9
END mean_3x3;
 
 
PROCEDURE mean_5x5(img: grid OF gray): grid OF gray;
(* mean value of 5x5 neighborhood *)
BEGIN
  RETURN sum_5x5(img) DIV 25
END mean_5x5;


PROCEDURE mean_op(img: grid OF gray; op_size: CARDINAL): grid OF gray;
BEGIN
  RETURN sum_op(img,op_size) DIV (op_size*op_size)
END mean_op;


(* auxiliary procedure *)
  PROCEDURE swap(VAR a,b: grid OF gray);
  VAR tmp: grid OF gray;
  BEGIN
    tmp:=a; a:=b; b:=tmp;
  END swap;


PROCEDURE median_3x3(img: grid OF gray): grid OF gray;
(* median of 3x3 matrix *)
VAR a,b,c : grid OF ARRAY[1..4] OF INTEGER;
    res   : grid OF gray;
    count : [1..5];
    i,j,k : grid OF [1..5];
BEGIN
  a[1] := MOVE.left (img);
  a[2] := img;
  a[3] := MOVE.right(img);
  a[4] := 256;  (* stopper *)
  (* sort 3 elems. with 3 comparisons/swaps *)
  IF a[1] > a[2] THEN swap(a[1],a[2]) END;
  IF a[2] > a[3] THEN swap(a[2],a[3]) END;
  IF a[1] > a[2] THEN swap(a[1],a[2]) END;

  (* send results up and down *)
  SEND.up  (a,b);
  SEND.down(a,c);
 
  (* merge lists, take 5th-smallest element   *)
  i:=1; j:=1; k:=1;  (* indices *)
  FOR count:=1 TO 5 DO
    IF a[i] < b[j] THEN
      IF a[i] < c[k]
        THEN res := a[i]; INC(i);
        ELSE res := c[k]; INC(k);
      END
     ELSIF b[j] < c[k]
        THEN res := b[j]; INC(j);
        ELSE res := c[k]; INC(k);
    END; (* if *)
  END; (* for *)
  RETURN res;
END median_3x3;

 
PROCEDURE median_5x5(img: grid OF gray): grid OF gray;
(* median of 5x5 matrix *)
VAR list   : grid OF ARRAY[1..5],[1..6] OF INTEGER;
    index  : grid OF ARRAY[1..5] OF [1..6];
    res    : grid OF gray;
    i      : CARDINAL;
    j,slist: grid OF CARDINAL;
BEGIN
  (* load horizontal data with 4 data exchanges *)
  list[1,2] := MOVE.left (img);
  list[1,3] := img;
  list[1,4] := MOVE.right(img);
  list[1,1] := MOVE.left (list[1,2]);
  list[1,5] := MOVE.right(list[1,4]);
  list[1,6] := 256;  (* stopper *)

  (* sort 5 elements *)
  FOR i:=4 TO 1 BY -1 DO
    res := list[1,i];
    j := i+1;
    WHILE list[1,j] < res DO
      list[1,j-1] := list[1,j];
      INC(j);
    END;
    list[1,j-1] := res;
  END; (* sort *)

  (* send results up and down *)
  SEND.up  (list[1],list[2]);
  SEND.up  (list[2],list[3]);
  SEND.down(list[1],list[4]);
  SEND.down(list[4],list[5]);

  (* merge lists, take 13th-smallest element   *)
  FOR i:=1 TO 5 DO index[i] := 1 END; (* indices *)
  FOR i:=1 TO 13 DO
    res := list[1,index[1]]; slist := 1;
    FOR j:=2 TO 5 DO
      IF list[j,index[j]] < res THEN
        res := list[j,index[j]]; slist := j
      END;
    END;
    INC(index[slist])
  END; (* merge *)

  RETURN res;
END median_5x5;


PROCEDURE median_3x3fast(img: grid OF gray): grid OF gray;
(* fast approximation of true median 3x3 matrix   *)
(* median in y-direction of median in x-direction *)
VAR a: grid OF ARRAY[1..3] OF gray;
BEGIN
  a[1] := MOVE.left (img);
  a[2] := img;
  a[3] := MOVE.right(img);

  (* sort 3 elems. with 3 comparisons/swaps *)
  IF a[1] > a[2] THEN swap(a[1],a[2]) END;
  IF a[2] > a[3] THEN swap(a[2],a[3]) END;
  IF a[1] > a[2] THEN swap(a[1],a[2]) END;
 
  (* send median in x-direction up and down *)
  SEND.up  (a[2],a[1]);
  SEND.down(a[2],a[3]);
 
  (* sort 3 elems. with 3 comparisons/swaps *)
  IF a[1] > a[2] THEN swap(a[1],a[2]) END;
  IF a[2] > a[3] THEN swap(a[2],a[3]) END;
  IF a[1] > a[2] THEN swap(a[1],a[2]) END;

  RETURN a[2];
END median_3x3fast;


PROCEDURE median_5x5fast(img: grid OF gray): grid OF gray;
(* median of 5x5 matrix *)
VAR list: grid OF ARRAY[1..6] OF gray;
    res : grid OF gray;
    i   : CARDINAL;
    j   : grid OF CARDINAL;
BEGIN
  (* load horizontal data with 4 data exchanges *)
  list[2] := MOVE.left (img);
  list[3] := img;
  list[4] := MOVE.right(img);
  list[1] := MOVE.left (list[2]);
  list[5] := MOVE.right(list[4]);
  list[6] := 256; (* stopper *)
 
  (* sort 5 elements *)
  FOR i:=4 TO 1 BY -1 DO
    res := list[i];
    j := i+1;
    WHILE list[j] < res DO
      list[j-1] := list[j];
      INC(j);
    END;
    list[j-1] := res;
  END; (* sort *)
 
  (* send x-median up and down *)
  SEND.up  (list[3],list[1]);
  SEND.up  (list[1],list[2]);
  SEND.down(list[3],list[4]);
  SEND.down(list[4],list[5]);

  (* sort 5 elements *)
  FOR i:=4 TO 1 BY -1 DO
    res := list[i];
    j := i+1;               
    WHILE list[j] < res DO   
      list[j-1] := list[j];
      INC(j);
    END;               
    list[j-1] := res;
  END; (* sort *)

  RETURN list[3];
END median_5x5fast;


PROCEDURE dither_ordered(img: grid OF gray): grid OF binary;
(* ordered dithering with 2x2 patterns *)
CONST thres = g_white DIV 5;
VAR res: grid OF binary;
BEGIN
  IF ODD(DIM(grid,2)) AND ODD(DIM(grid,1)) THEN
    res := img < thres;  (* upper left corner *)
    SEND.right (img < 3*thres,res);
    SEND.down  (img < 4*thres,res);
    SEND.down_r(img < 2*thres,res);
  END;
  RETURN res;
END dither_ordered;


PROCEDURE open_op(img: grid OF gray; op_size: CARDINAL): grid OF gray;
(* min then max *)
BEGIN
  RETURN max_op( min_op(img,op_size), op_size);
END open_op;


PROCEDURE close_op(img: grid OF gray; op_size: CARDINAL): grid OF gray;
(* max then min *)
BEGIN
  RETURN min_op( max_op(img,op_size), op_size);
END close_op;


PROCEDURE top_hat_op(img: grid OF gray; op_size: CARDINAL): grid OF gray;	
(* image minus open *)
VAR res: grid OF gray;
BEGIN
  RETURN img - open_op(img,op_size);
END top_hat_op; 

END Local.

