IMPLEMENTATION MODULE MatchFeat;
(* feature-based matching algorithm for calculation of displacement vectors *)
(* EXPORTED PROCEDURES: matchFeatures                                       *)
(* LANGUAGE: Parallaxis Version 3                                           *)
(* CREATED: Jan. 95  Wolfgang Rapf                                          *)
(* RELATED FILES: MatchFeat.pd, Corners.pd, Corners.pm                      *)
(* REMARKS:                                                                 *)
(* CHANGES:                                                                 *) 

FROM Local IMPORT grid, right, left, up, down; 
FROM OpticalFlow IMPORT rvec2;

TYPE feature_type = RECORD is_feature: BOOLEAN;
                           dx1, dy1, dx2, dy2: INTEGER
                    END;

TYPE prop_data_type = RECORD is_feature: BOOLEAN;
                             dx1, dy1, dx2, dy2, dxr, dyr: INTEGER
                      END;


PROCEDURE search_backwards(radius: INTEGER;
                           feature_a: grid OF feature_type;
                           VAR feature_b: grid OF prop_data_type);
 
VAR i, j, xpos: INTEGER;
    h_temp, v_temp: grid OF feature_type;
    min_diff: grid OF INTEGER;
 
        PROCEDURE hold_match(is_feature: grid OF BOOLEAN;
                             feature: grid OF feature_type;
                             xpos, ypos: INTEGER;
                             VAR min_diff: grid OF INTEGER);
        VAR diff: grid OF INTEGER;
        BEGIN
          IF is_feature AND feature.is_feature

            THEN diff := (feature.dy2 - feature_b.dy2) ** 2
		       + (feature.dx2 - feature_b.dx2) ** 2
	    	       + (feature.dy1 - feature_b.dy1) ** 2
       		       + (feature.dx1 - feature_b.dx1) ** 2;

            IF diff < min_diff THEN
              min_diff := diff;
              feature_b.dxr := xpos;
              feature_b.dyr := ypos;
            END;
          END;
        END hold_match;

BEGIN (* search_backwards *)
  feature_b.dxr := 0;
  feature_b.dyr := 0;
  min_diff := MAX(INTEGER);
  IF radius > 0 THEN
    FOR i := 0 TO 2 * radius DO
      IF i = 0 THEN h_temp := feature_a; xpos := 0;
      ELSE IF i <= radius THEN h_temp := MOVE.left(h_temp); xpos:= i;
      ELSE IF i = radius + 1 THEN h_temp := MOVE.right(feature_a);
                                  xpos := -1;
      ELSE h_temp := MOVE.right(h_temp); xpos := radius - i;
      END; END; END;

      (* upper half of a column *)
      v_temp := h_temp;
      hold_match(feature_b.is_feature, v_temp, xpos, 0, min_diff);
      FOR j := 1 TO radius DO
        v_temp := MOVE.down(v_temp);
        hold_match(feature_b.is_feature, v_temp, xpos, -j, min_diff);
      END;
      v_temp := h_temp;

      (* lower half of a column *)
      FOR j := 1 TO radius DO
        v_temp := MOVE.up(v_temp);
        hold_match(feature_b.is_feature, v_temp, xpos, j, min_diff);
      END;
    END; (* FOR *)
  END; (* IF *)
END search_backwards;


PROCEDURE search_forward(radius: INTEGER;
		         feature: grid OF feature_type;
                         prop_data: grid OF prop_data_type;
		         VAR dx, dy: grid OF INTEGER);
VAR i, j, xpos: INTEGER;
    min_diff, diff: grid OF INTEGER;
    h_temp_prop, v_temp_prop: grid OF prop_data_type;

	PROCEDURE hold_match_prop(feature: grid OF feature_type;
                                  prop_data: grid OF prop_data_type;
			          xpos, ypos: INTEGER;
			          VAR min_diff, dx, dy: grid OF INTEGER);
	VAR diff: grid OF INTEGER;
	BEGIN(* hold_match_prop *)
          IF feature.is_feature AND prop_data.is_feature THEN
	    IF  (prop_data.dyr = -ypos)
	    AND (prop_data.dxr = -xpos)
            THEN diff := (prop_data.dy2 - feature.dy2) ** 2
		       + (prop_data.dx2 - feature.dx2) ** 2
	    	       + (prop_data.dy1 - feature.dy1) ** 2
       		       + (prop_data.dx1 - feature.dx1) ** 2;
                 IF diff < min_diff
                 THEN min_diff := diff;
                      dx := xpos;
                      dy := ypos;
                 END;
            END;
          END;          
	END hold_match_prop;

BEGIN (* search_forward *)
  dx := 0;
  dy := 0;
  min_diff := MAX(INTEGER);
  IF radius > 0 THEN
    FOR i := 0 TO 2 * radius DO
      IF i = 0 THEN h_temp_prop := prop_data; 
		    xpos := 0;
      ELSE IF i <= radius THEN h_temp_prop := MOVE.left(h_temp_prop);
			       xpos:= i;
      ELSE IF i = radius + 1 THEN h_temp_prop := MOVE.right(prop_data);
				  xpos := -1;
      ELSE h_temp_prop := MOVE.right(h_temp_prop);
	   xpos := radius - i;
      END; END; END;

      (* upper half of a column *)
      v_temp_prop := h_temp_prop;
      hold_match_prop(feature, v_temp_prop, xpos, 0, min_diff, dx, dy);
      FOR j := 1 TO radius DO
        v_temp_prop := MOVE.down(v_temp_prop);
       hold_match_prop(feature, v_temp_prop, xpos, -j, min_diff, dx, dy);
      END;

      (* lower half of a column *)
      v_temp_prop := h_temp_prop;
      FOR j := 1 TO radius DO
        v_temp_prop := MOVE.up(v_temp_prop);
	hold_match_prop(feature, v_temp_prop, xpos, j, min_diff, dx, dy);
      END;
    END; (* FOR *)
  END; (* IF *)
END search_forward;


PROCEDURE matchFeatures(radius: CARDINAL;
                        is_feature_a, is_feature_b: grid OF BOOLEAN;
		        dx1_a, dy1_a, dx2_a, dy2_a,
                        dx1_b, dx2_b, dy1_b, dy2_b: grid OF INTEGER)
                        : grid OF rvec2;
VAR dx, dy, dx_r, dy_r, dir_vect, feat_vect: grid OF INTEGER;
    vect: grid OF rvec2;
    feature_a: grid OF feature_type;
    prop_data: grid OF prop_data_type;
  
BEGIN (* matchFeatures *)
  (* prepare RECORD variables for far propagation of feature data *) 
  feature_a.is_feature := is_feature_a;
  feature_a.dx1 := dx1_a;
  feature_a.dy1 := dy1_a;
  feature_a.dx2 := dx2_a;
  feature_a.dy2 := dy2_a;

  prop_data.is_feature := is_feature_b;
  prop_data.dx1 := dx1_b;
  prop_data.dy1 := dy1_b;
  prop_data.dx2 := dx2_b;
  prop_data.dy2 := dy2_b;
  prop_data.dxr := 0;
  prop_data.dyr := 0;

  (* map correspondent features; base location for search: *)
  (* features of image b                                   *)
  search_backwards(radius, feature_a, prop_data);

  (* map correspondent features; base location for search: *)
  (* features of image a                                   *)
  search_forward(radius, feature_a, prop_data, dx, dy);

  vect.x := FLOAT(dx);
  vect.y := FLOAT(dy);

  RETURN(vect);
END matchFeatures;

BEGIN (* no init necessary *)
END MatchFeat.


