IMPLEMENTATION MODULE OpticalFlow;
(* algorithms for calculation of optical flow by differential methods: *)
(* algorithm of Horn and Schunck                                       *)
(* EXPORTED PROCEDURES:                                                *)
(*   1.) hornSchunck                                                   *)
(*   2.) hornSchunckMed                                                *)
(*   3.) write_VFFA                                                    *)
(* LANGUAGE: Parallaxis Version 3                                      *)
(* CREATED: Oct. 94  Wolfgang Rapf                                     *)
(* RELATED FILES: OpticalFlow.pd                                       *)
(* REMARKS:                                                            *)
(* CHANGES:                                                            *) 

FROM ImageIO IMPORT gray; 
FROM Local IMPORT grid, right, left, up, down,
                  sobel_x_3x3, sobel_y_3x3, mean_3x3;


PROCEDURE hornSchunck(alpha2, iters	 : CARDINAL;
		      pixel_t1, pixel_t2 : grid OF gray)
		     			 : grid OF rvec2;
(* calculation of optical flow by the algorithm of Horn and Schunck  *)
(* parameters: (all input)                                           *)
(*      alpha2: CARDINAL   standard 2500,    determines influence of *)
(*                         smoothness constraint                     *)
(*      iters:  CARDINAL standard 10..500                            *)
(*                       max. number of iterations                   *)
(*      pixel_t1,                                                    *)
(*      pixel_t2: grid OF gray      input grayvalue images (t1, t2)  *)
(* returns: image of 2-dim. displacement vectors (grid OF rvec2)     *)

VAR iterCnt: CARDINAL;
    dx_envir, dy_envir, a, b: grid OF REAL;	
    gv_dx, gv_dy, gv_dt: grid OF INTEGER;
    flow_field: grid OF rvec2;

	PROCEDURE laplace_envir(input: grid OF REAL): grid OF REAL;
        (* modification of laplace-operator: center value not considered *)
	VAR ret_val: grid OF REAL;
	BEGIN
          ret_val := 2.0*input + MOVE.right(input) + MOVE.left(input);
	  ret_val := (2.0*ret_val - 4.0*input + MOVE.up(ret_val)
                                  + MOVE.down(ret_val))/12.0; 
	  RETURN ret_val;
	END laplace_envir;

BEGIN (* hornSchunck *)
  (* calculate spatial and temporal gradients *)
  gv_dx := (sobel_x_3x3((pixel_t1 + pixel_t2 + 1) DIV 2) + 4) DIV 8;
  gv_dy := (sobel_y_3x3((pixel_t1 + pixel_t2 + 1) DIV 2) + 4) DIV 8;
  gv_dt := mean_3x3(pixel_t2) - mean_3x3(pixel_t1);

  (* init result variables and start iteration *)
  flow_field.x := 0.0;
  flow_field.y := 0.0;

  FOR iterCnt := 0 TO iters DO 
    dx_envir := laplace_envir(flow_field.x);
    dy_envir := laplace_envir(flow_field.y);
    a := FLOAT(gv_dx) * dx_envir + FLOAT(gv_dy) * dy_envir + FLOAT(gv_dt);
    b := FLOAT(alpha2 + gv_dx ** 2 + gv_dy ** 2) ;
    flow_field.x := dx_envir - (FLOAT(gv_dx) * a)/b;
    flow_field.y := dy_envir - (FLOAT(gv_dy) * a)/b;
  END (* FOR *);

  RETURN flow_field;
END hornSchunck; 


PROCEDURE hornSchunckMed(alpha2, iters		: CARDINAL;
		         pixel_t1, pixel_t2	: grid OF gray)
		     				: grid OF rvec2;
(* calculation of optical flow by the algorithm of Horn and Schunck  *)
(* smoothness constraint replaced by medianess constraint            *)
(* parameters: (all input)                                           *)
(*      alpha2: CARDINAL   standard 2500,    determines influence of *)
(*                         smoothness constraint                     *)
(*      iters:  CARDINAL standard 10..500                            *)
(*                       max. number of iterations                   *)
(*      pixel_t1,                                                    *)
(*      pixel_t2: grid OF gray      input grayvalue images (t1, t2)  *)
(* returns: image of 2-dim. displacement vectors (grid OF rvec2)     *)

VAR iterCnt: CARDINAL;
    dx_envir, dy_envir, a, b: grid OF REAL;	
    gv_dx, gv_dy, gv_dt: grid OF INTEGER;
    flow_field: grid OF rvec2;

       (* auxiliary procedure *)
       PROCEDURE swap(VAR a,b: grid OF REAL);
       VAR tmp: grid OF REAL;
       BEGIN
         tmp:=a; a:=b; b:=tmp;
       END swap;

       PROCEDURE median_3x3r(img: grid OF REAL): grid OF REAL;
       (* median of 3x3 matrix, for REAL values *)
       VAR a,b,c      : grid OF ARRAY[1..4] OF REAL;
           res        : grid OF REAL;
           count      : [1..5];
           i,j,k      : grid OF [1..5];
       BEGIN
         a[1] := MOVE.left(img);
         a[2] := img;
         a[3] := MOVE.right(img);
         a[4] := MAX(REAL);  (* stopper *)

         (* sort 3 elems. with 3 comparisons/swaps *)
         IF a[1] > a[2] THEN swap(a[1],a[2]) END;
         IF a[2] > a[3] THEN swap(a[2],a[3]) END;
         IF a[1] > a[2] THEN swap(a[1],a[2]) END;

         (* send results up and down *)
         SEND.up  (a,b);
         SEND.down(a,c);
 
         (* merge lists, take 5th-smallest element   *)
         i:=1; j:=1; k:=1;  (* indices *)
         FOR count:=1 TO 5 DO
           IF a[i] < b[j] THEN
             IF a[i] < c[k]
             THEN res := a[i]; INC(i);
             ELSE res := c[k]; INC(k);
             END
           ELSIF b[j] < c[k]
             THEN res := b[j]; INC(j);
             ELSE res := c[k]; INC(k);
           END; (* if *)
         END; (* for *)
         RETURN res;
       END median_3x3r;


BEGIN (* hornSchunckMed *)
  (* calculate spatial and temporal gradients *)
  gv_dx := (sobel_x_3x3((pixel_t1 + pixel_t2) DIV 2) + 4) DIV 8;
  gv_dy := (sobel_y_3x3((pixel_t1 + pixel_t2) DIV 2) + 4) DIV 8;
  gv_dt := mean_3x3(pixel_t2) - mean_3x3(pixel_t1);

  (* init result variables and start iteration *)
  flow_field.x := 0.0;
  flow_field.y := 0.0;

  FOR iterCnt := 0 TO iters DO 
    dx_envir := median_3x3r(flow_field.x);
    dy_envir := median_3x3r(flow_field.y);
    a := FLOAT(gv_dx) * dx_envir + FLOAT(gv_dy) * dy_envir + FLOAT(gv_dt);
    b := FLOAT(alpha2 + gv_dx ** 2 + gv_dy ** 2) ;
    flow_field.x := dx_envir - (FLOAT(gv_dx) * a)/b;
    flow_field.y := dy_envir - (FLOAT(gv_dy) * a)/b;
  END (* FOR *);

  RETURN flow_field;
END hornSchunckMed; 


PROCEDURE write_VFFA(vec_field: VECTOR OF rvec2;
                     width, height: CARDINAL;
                     filename: ARRAY OF CHAR;
                     scale, one_of: CARDINAL);
(* write optical flow field in VFFA Format                             *)
(* parameters:                                                         *)
(*      vec_field: VECTOR OF rvec2   input vector field                *)
(*      width, height: CARDINAL      dimensions of vector field        *)
(*      filename: ARRAY OF CHAR      output filename                   *)
(*      scale: CARDINAL              VFFA specific: image scale factor *)
(*                                   standard 1                        *)
(*      one_of: CARDINAL             prepare file for visualisation:   *)
(*                                   zero all vectors, where           *) 
(*                                   (x_pos MOD one_of) != 0  or       *)
(*                                   (y_pos MOD one_of) != 0           *)
VAR max_width, pos, x_pos, y_pos: CARDINAL;

BEGIN
  WriteString("writing vectorfield '"); WriteString(filename); 
  WriteString("`, 1 of "); WriteInt(one_of, 1); WriteLn;
  OpenOutput(filename);
  WriteString("VFFA "); WriteInt(width, 4); Write(' '); 
  WriteInt(height, 4); Write(' ');
  WriteInt(scale,1);    (* grid distance *)
  Write(' ');
  WriteLn;

  max_width  := UPPER(vec_field, 1) - LOWER(vec_field, 1) + 1;
  FOR y_pos := 1 TO height DO
    FOR x_pos := 1 TO width DO
      pos := (y_pos-1) * max_width + x_pos;
      WITH vec_field<<pos>> DO
        IF ((y_pos MOD one_of) = 0) AND ((x_pos MOD one_of) = 0)
        THEN 
          WriteReal(x, 4); Write(' ');
          WriteReal(y, 4);
        ELSE
          WriteReal(0.0 , 4); Write(' ');
          WriteReal(0.0 , 4);
        END;
        Write(' ');
      END;
    END;
    WriteLn;
  END;
  CloseOutput;
END write_VFFA;

BEGIN (* no init necessary *)
END OpticalFlow.



