IMPLEMENTATION MODULE Textures;
(* Stefan Feyrer, Univ. Stuttgart 1995 *)

(* ------------------------------------------------------------------------ *)

FROM ImageIO IMPORT gray, g_black, g_white;
FROM Local   IMPORT left, up, up_r, up_l, grid;

(* ------------------------------------------------------------------------ *)
(* Returns the transposed matrix.                                           *)

PROCEDURE transpose (matrix: co_occ_grid OF REAL): co_occ_grid OF REAL;

  VAR temp: co_occ_grid OF REAL;

  BEGIN (* transpose *)
    SEND.<:g_white-DIM (co_occ_grid, 1),
           g_white-DIM (co_occ_grid, 2):> (matrix, temp);
    RETURN temp;
  END transpose;

(* ------------------------------------------------------------------------ *)
(* Returns the cooccurrence matrix of the considered gray level image       *)
(* for a translation of one pixel in the desired direction.                 *)

PROCEDURE co_occur (image:     grid OF gray; 
                    width:     CARDINAL;
                    height:    CARDINAL;
                    direction: direction_type): co_occ_grid OF REAL;

  VAR moved: grid OF gray;
      
  BEGIN (* co_occur *)
    CASE direction OF
    | horizontal:      moved  := MOVE.left (image);
                       width  := width-1;
    | vertical:        moved  := MOVE.up (image);
                       height := height-1;
    | diagonal_up_r:   moved  := MOVE.up_r (image);
                       moved  := MOVE.left (moved);
                       width  := width-1;
                       height := height-1;
    | diagonal_down_r: moved  := MOVE.up_l (image);
                       width  := width-1;
                       height := height-1;
    END; (* CASE *)
    RETURN co_occur_general (image, moved, width, height);
  END co_occur;

(* ------------------------------------------------------------------------ *)
(* Returns the cooccurrence matrix of the considered gray level image       *)
(* for the desired translation.                                             *)

PROCEDURE co_occur_general (image:  grid OF gray;
                            moved:  grid OF gray; 
                            width:  CARDINAL;
                            height: CARDINAL): co_occ_grid OF REAL;

  CONNECTION grid2co_occ: grid[i,j] -> co_occ_grid[i,j];
  
  VAR one_co_occ:    co_occ_grid OF REAL;
      index_1:       co_occ_grid OF CARDINAL;
      index_2:       co_occ_grid OF CARDINAL;
      one_grid:      grid OF REAL;
      co_occur_help: grid OF REAL;
      co_occur:      co_occ_grid OF REAL;

  BEGIN (* co_occur_general *)
    co_occur := 0.0;
    IF LEN (grid, 1) = LEN (grid, 2) THEN
      IF g_white - g_black >= LEN (grid, 1) THEN
        IF (DIM (co_occ_grid, 1)-g_black+1 <= width) AND
           (DIM (co_occ_grid, 2)-g_black+1 <= height) THEN
          SEND.grid2co_occ (image, index_1);
          SEND.grid2co_occ (moved, index_2);
          one_co_occ := 1.0;
        ELSE
          index_1 := DIM (co_occ_grid, 1);
          index_2 := DIM (co_occ_grid, 2);
          one_co_occ := 0.0;
        END; (* IF *)
        SEND.<:index_1, index_2:>:#SUM (one_co_occ, co_occur);
      ELSE
        co_occur_help := 0.0;
        IF (DIM (grid, 1)-LOWER (grid, 1)+1 <= width) AND
           (DIM (grid, 2)-LOWER (grid, 2)+1 <= height) THEN
          one_grid := 1.0;
        ELSE
          one_grid := 0.0;
          image := DIM (grid, 1);
          moved := DIM (grid, 2);
        END; (* IF *) 
        SEND.<:image, moved:>:#SUM (one_grid, co_occur_help);
        SEND.grid2co_occ (co_occur_help, co_occur);
      END; (* IF *)
      co_occur := co_occur + transpose (co_occur);
      RETURN co_occur / REDUCE.SUM (co_occur);
    ELSE
      RETURN co_occur;
    END; (* IF *)
  END co_occur_general;

(* ------------------------------------------------------------------------ *)
(* Returns the energy coefficient of the matrix 'co_occur'.                 *)

PROCEDURE feature_energy (co_occur: co_occ_grid OF REAL): REAL;

  BEGIN (* feature_energy *)
    RETURN REDUCE.SUM (co_occur*co_occur);
  END feature_energy;

(* ------------------------------------------------------------------------ *)
(* Returns the entropy coefficient of the matrix 'co_occur'.                *)

PROCEDURE feature_entropy (co_occur: co_occ_grid OF REAL): REAL;

  BEGIN (* feature_entropy *)
    IF co_occur # 0.0 THEN 
      RETURN REDUCE.SUM (-co_occur * ln (co_occur));
    END; (* IF *)
  END feature_entropy;

(* ------------------------------------------------------------------------ *)
(* Returns the correlation coefficient of the matrix 'co_occur'.            *)

PROCEDURE feature_correlation (co_occur: co_occ_grid OF REAL): REAL;

  VAR temp:   REAL;
      temp_x: co_occ_grid OF REAL;
      temp_y: co_occ_grid OF REAL;
      mx, my: REAL;
      sx, sy: REAL;
      s:      REAL;

  BEGIN (* feature_correlation *)
    mx := REDUCE.SUM (FLOAT (DIM (co_occ_grid, 1)-g_black)*co_occur);
    my := REDUCE.SUM (FLOAT (DIM (co_occ_grid, 2)-g_black)*co_occur);
    temp_x := FLOAT (DIM (co_occ_grid, 1)-g_black) - mx;
    sx := sqrt (REDUCE.SUM (temp_x*temp_x*co_occur));
    temp_y := FLOAT (DIM (co_occ_grid, 2)-g_black) - my;
    sy := sqrt (REDUCE.SUM (temp_y*temp_y*co_occur));
    s := sx * sy;
    RETURN REDUCE.SUM (temp_x*temp_y*co_occur) / s;
  END feature_correlation;

(* ------------------------------------------------------------------------ *)
(* Returns the inertia coefficient of the matrix 'co_occur'.                *)

PROCEDURE feature_inertia (co_occur: co_occ_grid OF REAL): REAL;

  VAR temp: co_occ_grid OF INTEGER;

  BEGIN (* feature_inertia *)
    temp := DIM (co_occ_grid, 1) - DIM (co_occ_grid, 2);
    RETURN REDUCE.SUM (FLOAT (temp*temp) * co_occur);
  END feature_inertia;

(* ------------------------------------------------------------------------ *)
(* Returns the coefficient of local homogeneity of the matrix 'co_occur'.   *)

PROCEDURE feature_local_homogeneity (co_occur: co_occ_grid OF REAL): REAL;

  VAR temp: co_occ_grid OF INTEGER;

  BEGIN (* feature_local_homogeneity *)
    temp := DIM (co_occ_grid, 1) - DIM (co_occ_grid, 2);
    temp := temp * temp + 1;
    RETURN REDUCE.SUM (1.0 / FLOAT (temp) * co_occur);
  END feature_local_homogeneity;

(* ------------------------------------------------------------------------ *)

END Textures.