IMPLEMENTATION MODULE Thinning;
(* algorithms for thinning/skeletonizing of binary images:          *)
(*   1.) algorithm of Stefanelli and Rosenfeld                      *)
(*   2.) algorithm of Lue and Wang                                  *)
(*   3.) algorithm of Hall and Guo                                  *)
(* EXPORTED PROCEDURES:                                             *)
(*   stefRosen(inImage: grid OF binary): grid OF binary             *)
(*   lueWang(inImage: grid OF binary): grid OF binary               *)
(*   hallGuo(inImage: grid OF binary): grid OF binary               *)
(* LANGUAGE: Parallaxis Version 3                                   *)
(* CREATED: Aug. 95  Michael Reinhardt, Wolfgang Rapf               *)
(* RELATED FILES: Thinning.pm                                       *)
(* REMARKS: both input and output are binary images; objects to be  *)
(*   thinned have to be labelled with binary type `b_black` while   *)
(*   the background is expected to be `b_white`. In case of         *)
(*   reversed labelled images first perform an inversion operation. *)


FROM ImageIO IMPORT binary, b_white, b_black;
FROM Local IMPORT grid, left, right, up, down; 


    PROCEDURE getBoolEnvir3x3(p: grid OF BOOLEAN;
                     VAR p1, p2, p3, p4, p5, p6, p7, p8: grid OF BOOLEAN);
    (* propagate direct neighbourhood into local buffers *)
    BEGIN
      p4 := MOVE.left(p);
      p5 := MOVE.up(p4);
      p3 := MOVE.down(p4);
      p8 := MOVE.right(p);
      p7 := MOVE.up(p8);
      p1 := MOVE.down(p8);
      p2 := MOVE.down(p);
      p6 := MOVE.up(p);
    END getBoolEnvir3x3;


PROCEDURE stefRosen(inImage: grid OF binary): grid OF binary;
(* thinning algorithm of Stefanelli & Rosenfeld (4 subiterations) *)
VAR ready: BOOLEAN;
    direction: CARDINAL;
    changed,
    isContourPoint, isFinalPoint,
    p1, p2, p3,
    p8, p,  p4,
    p7, p6, p5: grid OF BOOLEAN;
    outImage: grid OF binary;


    PROCEDURE finalCond_a(p1, p2, p3, p4, p5, p6, p7, p8: grid OF BOOLEAN)
                                                        : grid OF BOOLEAN;
    BEGIN
      RETURN(   ((p1 OR p2 OR p3) AND             (* a1 *)
                 (NOT p4) AND (NOT p8) AND
                 (p5 OR p6 OR p7))
             OR ((p2 OR p3 OR p4) AND             (* a2 *)
                 (NOT p5) AND (NOT p1) AND
                 (p6 OR p7 OR p8))
             OR ((p3 OR p4 OR p5) AND             (* a3 *)
                 (NOT p2) AND (NOT p6) AND
                 (p7 OR p8 OR p1))
             OR ((p4 OR p5 OR p6) AND             (* a4 *)
                 (NOT p3) AND (NOT p7) AND
                 (p8 OR p1 OR p2)));
    END finalCond_a;


    PROCEDURE finalCond_b1(p1, p2, p3, p4, p5, p6, p7, p8: grid OF BOOLEAN)
                                                         : grid OF BOOLEAN;
    BEGIN
      RETURN ((p1 OR p2 OR p3) AND
              (NOT p4) AND p6 AND (NOT p7));
    END finalCond_b1;

    PROCEDURE finalCond_b2(p1, p2, p3, p4, p5, p6, p7, p8: grid OF BOOLEAN)
                                                         : grid OF BOOLEAN;
    BEGIN
      RETURN ((p7 OR p8 OR p1) AND
              (NOT p3) AND p4 AND (NOT p6));
    END finalCond_b2;

    PROCEDURE finalCond_b3(p1, p2, p3, p4, p5, p6, p7, p8: grid OF BOOLEAN)
                                                         : grid OF BOOLEAN;
    BEGIN
      RETURN ((p5 OR p6 OR p7) AND
              (NOT p8) AND p2 AND (NOT p3));
    END finalCond_b3;

    PROCEDURE finalCond_b4(p1, p2, p3, p4, p5, p6, p7, p8: grid OF BOOLEAN)
                                                         : grid OF BOOLEAN;
    BEGIN
      RETURN ((p3 OR p4 OR p5) AND
              (NOT p7) AND p8 AND (NOT p2));
    END finalCond_b4;


BEGIN (* stefRosen *)

  IF inImage = b_black THEN p := TRUE ELSE p := FALSE END;

  (* initialize point classification *)
  isFinalPoint    := FALSE;
  isContourPoint  := FALSE;

  direction := 0;

  ready := FALSE;
  REPEAT
    changed := FALSE;
    
    FOR direction := 0 TO 3 DO

      (* propagate direct neighbourhood into local buffers *)
      getBoolEnvir3x3(p, p1, p2, p3, p4, p5, p6, p7, p8);

      isFinalPoint := isFinalPoint OR
                      finalCond_a(p1, p2, p3, p4, p5, p6, p7, p8);

      CASE direction OF
      0   : (* from below *)
            isContourPoint := NOT p6;
            isFinalPoint := isFinalPoint OR
                            finalCond_b1(p1, p2, p3, p4, p5, p6, p7, p8) OR
                            finalCond_b2(p1, p2, p3, p4, p5, p6, p7, p8);
      | 1 : (* from above *)
            isContourPoint := NOT p2;
            isFinalPoint := isFinalPoint OR
                            finalCond_b3(p1, p2, p3, p4, p5, p6, p7, p8) OR
                            finalCond_b4(p1, p2, p3, p4, p5, p6, p7, p8);
      | 2 : (* from left *)
            isContourPoint := NOT p8;
            isFinalPoint := isFinalPoint OR
                            finalCond_b1(p1, p2, p3, p4, p5, p6, p7, p8) OR
                            finalCond_b4(p1, p2, p3, p4, p5, p6, p7, p8);
      | 3 : (* from right *)
            isContourPoint := NOT p4;
            isFinalPoint := isFinalPoint OR
                            finalCond_b2(p1, p2, p3, p4, p5, p6, p7, p8) OR
                            finalCond_b3(p1, p2, p3, p4, p5, p6, p7, p8);
      END; (* CASE *)

      IF p AND (NOT isFinalPoint) AND isContourPoint
      THEN
        p := FALSE;
        changed := TRUE;
      END;

    END; (* FOR *)

    ready := NOT REDUCE.OR(changed);

  UNTIL ready;

  IF p THEN outImage := b_black ELSE outImage := b_white END; 

  RETURN outImage;
END stefRosen;



PROCEDURE lueWang(inImage: grid OF binary): grid OF binary;
(* thinning algorithm of Lue & Wang *)
VAR ready, step: BOOLEAN;
    changed, cond_C,
    p1, p2, p3,
    p8, p,  p4,
    p7, p6, p5: grid OF BOOLEAN;
    A, B: grid OF CARDINAL;
    outImage: grid OF binary;

    PROCEDURE count_A(p1, p2, p3, p4, p5, p6, p7, p8: grid OF BOOLEAN)
                                                    : grid OF CARDINAL;
    (* count FALSE-TRUE sequences *)
    BEGIN
      RETURN(ORD((NOT p1) AND p2) + ORD((NOT p2) AND p3) +
             ORD((NOT p3) AND p4) + ORD((NOT p4) AND p5) +
             ORD((NOT p5) AND p6) + ORD((NOT p6) AND p7) +
             ORD((NOT p7) AND p8) + ORD((NOT p8) AND p1));
    END count_A;

    PROCEDURE count_B(p1, p2, p3, p4, p5, p6, p7, p8: grid OF BOOLEAN)
                                                    : grid OF CARDINAL;
   (* count TRUE-labelled neighbours *)
    BEGIN  
      RETURN(ORD(p1) + ORD(p2) + ORD(p3) +
             ORD(p8)           + ORD(p4) +
             ORD(p7) + ORD(p6) + ORD(p5));
    END count_B;


BEGIN (* lueWang *)

  IF inImage = b_black THEN p := TRUE ELSE p := FALSE END;
  
  ready := FALSE;
  REPEAT
    changed := FALSE;

    FOR step := FALSE TO TRUE DO

      (* propagate direct neighbourhood into local buffers *)
      getBoolEnvir3x3(p, p1, p2, p3, p4, p5, p6, p7, p8);

      (* calculate number of FALSE-TRUE sequences *)
      A := count_A(p1, p2, p3, p4, p5, p6, p7, p8);

      (* calculate number of TRUE-labelled neighbours *)
      B := count_B(p1, p2, p3, p4, p5, p6, p7, p8);

      (* check elimination conditions and eliminate label *) 
      IF step THEN
        cond_C := NOT(p4 AND p6 AND (p2 OR p8));
      ELSE
        cond_C := NOT(p2 AND p8 AND (p6 OR p8));
      END; (* IF *)

      IF p AND (B >= 3) AND (B <= 6) AND (A = 1) AND cond_C THEN
        p := FALSE;
        changed := TRUE;
      END; (* IF *)

    END; (* FOR *)
    ready := NOT REDUCE.OR(changed);
  UNTIL ready;

  IF p THEN outImage := b_black ELSE outImage := b_white END; 

  RETURN outImage;
END lueWang;


PROCEDURE hallGuo(inImage: grid OF binary): grid OF binary;
(* thinning algorithm of Hall & Guo *)
VAR ready: BOOLEAN;
    changed, chessLabel, cond_B,
    p1, p2, p3,
    p8, p,  p4,
    p7, p6, p5: grid OF BOOLEAN;
    B, C: grid OF CARDINAL;
    outImage: grid OF binary;

    PROCEDURE count_B(p1, p2, p3, p4, p5, p6, p7, p8: grid OF BOOLEAN)
                                                    : grid OF CARDINAL;
   (* count TRUE-labelled neighbours *)
    BEGIN  
      RETURN(ORD(p1) + ORD(p2) + ORD(p3) +
             ORD(p8)           + ORD(p4) +
             ORD(p7) + ORD(p6) + ORD(p5));
    END count_B;


    PROCEDURE count_C(p1, p2, p3, p4, p5, p6, p7, p8: grid OF BOOLEAN )
                                                    : grid OF CARDINAL;
   (* calculate connectivity number *)
    BEGIN
      RETURN(ORD(NOT p2 AND (p3 OR p4)) +
             ORD(NOT p4 AND (p5 OR p6)) +
             ORD(NOT p6 AND (p7 OR p8)) +
             ORD(NOT p8 AND (p1 OR p2)));
    END count_C;


BEGIN (* hallGuo *)

  IF inImage = b_black THEN p := TRUE ELSE p := FALSE END;

  (* chessboard-labelling of complete image *)
  IF ODD(DIM(grid, 1) + DIM(grid, 2))
  THEN chessLabel := TRUE
  ELSE chessLabel := FALSE
  END;

  ready := FALSE;
  REPEAT
    changed := FALSE;

    (* propagate direct neighbourhood into local buffers *)
    getBoolEnvir3x3(p, p1, p2, p3, p4, p5, p6, p7, p8);

    (* calculate number of TRUE-labelled neighbours *)
    B := count_B(p1, p2, p3, p4, p5, p6, p7, p8);

    (* calculate connectivity number *)
    C := count_C(p1, p2, p3, p4, p5, p6, p7, p8);

    (* check elimination conditions and eliminate label *) 
    cond_B := NOT((p1 AND p3 AND p5 AND p7) OR
                  (p2 AND p4 AND p6 AND p8));
    IF p AND chessLabel AND (C = 1) AND cond_B AND (B > 1)
    THEN
      p := FALSE;
      changed := TRUE;
    END;

    ready := NOT REDUCE.OR(changed);
    chessLabel := NOT chessLabel;
  UNTIL ready;

  IF p THEN outImage := b_black ELSE outImage := b_white END; 

  RETURN outImage;
END hallGuo;

BEGIN (* no init necessary *)
END Thinning.


