IMPLEMENTATION MODULE Trans;
(* Braunl, Jochum, Schaper, Univ. Stuttgart, Sep. 1994 *)

FROM ImageIO IMPORT color,gray,binary,
                    b_black,b_white, g_black,g_white,
                    c_black,c_white,c_red,c_green,c_blue;


PROCEDURE binary2gray(img : VECTOR OF binary; gray4black, gray4white: gray): VECTOR OF gray;
(* transform binary image to gray image *)
VAR res: VECTOR OF gray;
BEGIN
  IF img = b_white THEN res := gray4white;
                   ELSE res := gray4black;
  END;
  RETURN res;
END binary2gray;


PROCEDURE binary2color(img: VECTOR OF binary; col4black, col4white: color): VECTOR OF color;
(* transform binary image to color image *)
VAR res: VECTOR OF color;
BEGIN
  IF img = b_white THEN res := col4white
                   ELSE res := col4black; 
  END;
  RETURN res;
END binary2color;


PROCEDURE gray2binary(img: VECTOR OF gray; threshold: gray): VECTOR OF binary;
(* transform gray image to binar image. gray values above threshold *)
(* become white, gray value less or equal threshold become black    *)
BEGIN
  RETURN img <= threshold;
END gray2binary;


PROCEDURE gray2binary2(img: VECTOR OF gray; thres1,thres2: gray): VECTOR OF binary;
(* transform gray image to binary image. gray values in range *)
(* [thres1 .. thres2] become black, others become white       *)
BEGIN
  RETURN thres1 <= img <= thres2;
END gray2binary2;


PROCEDURE dither_random(img: VECTOR OF gray): VECTOR OF binary;
(* transform gray image to binary image by random dithering *)
CONST low  =  50;
      high = 200;
BEGIN
  (* RETURN img <= RandomInt(img) MOD 256 *)
  RETURN img <= low + RandomInt(img) MOD (high-low)
END dither_random;


PROCEDURE rgb2color(g_red,g_green,g_blue: VECTOR OF gray): VECTOR OF color;
(* transform gray image to color image *)
VAR res: VECTOR OF color;
BEGIN
  res.red   := g_red;
  res.green := g_green; 
  res.blue  := g_blue;
  RETURN res;
END rgb2color;


PROCEDURE cmy2color(cya,mag,yel: VECTOR OF gray): VECTOR OF color;
(* transform color image from cmy to rgb representation *)
BEGIN
  RETURN rgb2color(g_white - cya, g_white - mag, g_white - yel)
END cmy2color;


PROCEDURE hsv2color(hue,sat,val: VECTOR OF gray): VECTOR OF color;
(* transform color image from hsv to rgb representation                     *)
(* integer version of Watt: Three-Dimensional Computer Graphics, Addison-W. *)
CONST g_white2 = g_white * g_white;  (* square of max gray value *)
VAR   res            : VECTOR OF color;
      hue6,p1,p2,p3,f: VECTOR OF gray;
BEGIN
  IF sat = 0 THEN
    res := gray2color(val)
  ELSE
    (* convert hue to range [0,6) *)
    hue6 := (hue * 6) DIV 256;  (* range [0..255] *)
    f    := (hue * 6) MOD 256;  (* range [0..255] *)
    p1   := val - val * sat DIV g_white;
    p2   := val - val * sat * f DIV g_white2;
    p3   := val - val * sat * (g_white - f) DIV g_white2;

    CASE hue6 OF
      0 : res := rgb2color(val,  p3,  p1) |
      1 : res := rgb2color( p2, val,  p1) |
      2 : res := rgb2color( p1, val,  p3) |
      3 : res := rgb2color( p1,  p2, val) |
      4 : res := rgb2color( p3,  p1, val) |
      5 : res := rgb2color(val,  p1,  p2)
    END;
  END; (* if *)

  RETURN res;
END hsv2color;


PROCEDURE gray2color(img: VECTOR OF gray): VECTOR OF color;
(* transform gray image to color image *)
BEGIN
  RETURN rgb2color(img,img,img)
END gray2color;


PROCEDURE gray2rainbow(img: VECTOR OF gray): VECTOR OF color;                   
(* transform gray image to color image with rainbow color map *)
VAR bright: VECTOR OF gray;
BEGIN
  bright := 255;
  RETURN hsv2color(img * 3 DIV 4, bright,bright)
END gray2rainbow;


PROCEDURE gray2color_map(img: VECTOR OF gray; VAR map: colormap): VECTOR OF color;
(* transform gray image to color image, pass map as VAR param for efficiency *)
(* by using gray values as indices in a colormap                             *)
VAR scale: gray;
    res  : VECTOR OF color;
BEGIN
  FOR scale:=g_black TO g_white DO
    IF img = scale THEN res := map[scale] END
  END;
  RETURN res;
END gray2color_map;


PROCEDURE color2gray(img: VECTOR OF color): VECTOR OF gray;
(* transform color image to gray image *)
BEGIN
  RETURN (img.red + img.green + img.blue) DIV 3
END color2gray;


PROCEDURE color2rgb(img: VECTOR OF color; VAR res_red,res_green,res_blue: VECTOR OF gray);
(* separate color image into rgb values *)
BEGIN
  res_red   := img.red;
  res_green := img.green;
  res_blue  := img.blue;
END color2rgb;


PROCEDURE color2cmy(img: VECTOR OF color; VAR cya,mag,yel: VECTOR OF gray);
(* transform color image  from rgb to cmy represenatation *)
BEGIN
  cya := g_white - img.red;
  mag := g_white - img.green;
  yel := g_white - img.blue;
END color2cmy;


PROCEDURE int2gray(img: VECTOR OF INTEGER): VECTOR OF gray;
(* transform integer image to gray image by stretching to range [0..255] *)
VAR tmin,tmax: INTEGER;
BEGIN
  RETURN int_stretch(img,g_black,g_white)
END int2gray;


PROCEDURE color2hsv(img: VECTOR OF color; undef_hue: gray; VAR hue,sat,val: VECTOR OF gray );
(* transfrom color image from rgb to hsv representation                     *)
(* integer version of Watt: Three-Dimensional Computer Graphics, Addison-W. *)
CONST sec2 =  85;  (* 2*60 degrees from 360 mapped to 255 *)
      sec4 = 170;  (* 4*60 degrees *)
VAR   max_rgb,min_rgb,diff,
      r_dist,g_dist,b_dist: VECTOR OF gray;
BEGIN
  (* determine max and min vlaues of red,green,blue *)
  IF img.red < img.green THEN min_rgb := img.red; max_rgb := img.green END;
  IF img.blue < min_rgb      THEN min_rgb := img.blue
    ELSIF img.blue > max_rgb THEN max_rgb := img.blue
  END;
  diff := max_rgb - min_rgb;
  IF diff = 0 THEN diff := 1 END;  (* avoid division by 0 *)

  val := max_rgb;

  IF max_rgb <> 0 THEN sat := g_white * diff DIV max_rgb
                  ELSE sat := 0
  END;

  IF sat <> 0 THEN
    r_dist := g_white * (max_rgb - img.red)   DIV diff;
    g_dist := g_white * (max_rgb - img.green) DIV diff;
    b_dist := g_white * (max_rgb - img.blue)  DIV diff;
    IF     img.red   = max_rgb THEN hue := b_dist - g_dist
     ELSIF img.green = max_rgb THEN hue := sec2 + r_dist - b_dist
     ELSIF img.blue  = max_rgb THEN hue := sec4 + g_dist - r_dist
    END; (* if *)
    IF hue < 0 THEN INC(hue,g_white) END;
   ELSE hue := undef_hue  (* sat = 0 *)
  END; (* if sat *)
END color2hsv;


PROCEDURE binary_invert(img: VECTOR OF binary): VECTOR OF binary;
(* invert binary image *)
BEGIN
  RETURN NOT img
END binary_invert;


PROCEDURE gray_invert(img: VECTOR OF gray): VECTOR OF gray;
(* invert gray image *)
BEGIN
  RETURN g_white - img
END gray_invert;


PROCEDURE color_invert(img: VECTOR OF color) : VECTOR OF color;
(* invert color image *)
VAR res: VECTOR OF color;
BEGIN
  res.red   := g_white - img.red;
  res.green := g_white - img.green;
  res.blue  := g_white - img.blue;
  RETURN res;
END color_invert;


PROCEDURE gray_stretch(img: VECTOR OF gray; g_min,g_max: gray): VECTOR OF gray;
(* stretch gray values into interval [g_min..g_max] *)
VAR tmax,tmin: INTEGER;
BEGIN
  tmin := REDUCE.MIN(img);
  tmax := REDUCE.MAX(img);
  IF tmin = tmax THEN INC(tmax) END;  (* avoid division by 0 *)
  RETURN (g_max-g_min) * (img - tmin) DIV (tmax-tmin) + g_min;
END gray_stretch;


PROCEDURE gray_const(img: VECTOR OF gray; g_min,g_max,c_val: gray): VECTOR OF gray;
(* set gray values in range [g_min..g_max] to c_val *)
VAR res: VECTOR OF gray;
BEGIN
  res := img;
  IF g_min <= img <= g_max THEN res := c_val END;
  RETURN res;
END gray_const;


PROCEDURE gray_remap(img: VECTOR OF gray; VAR map: graymap): VECTOR OF gray;
(* find new gray values by using them as entries in graymap *)
(* graymap is passed as VAR for resaons of efficiency       *)
VAR scale: INTEGER;
    res  : VECTOR OF gray;
BEGIN
  FOR scale:=g_black TO g_white DO
    IF img=scale THEN res := map[scale] END;
  END;
  RETURN res;
END gray_remap;


PROCEDURE gray_reduce(img: VECTOR OF gray; num_vals: gray): VECTOR OF gray;
(* reduce number of gray values in image to num_vals *)
VAR tmax,tmin: INTEGER;
    res      : VECTOR OF gray;
BEGIN
  tmin := REDUCE.MIN(img);
  tmax := REDUCE.MAX(img);
  IF tmax-tmin+1 > num_vals > 1 THEN
    (* stretch to [0..num_vals-1] *)
    res := (num_vals-1) * (img - tmin) DIV (tmax-tmin);
    (* stretch to [tmin ..tmax]   *)
    res := (tmax-tmin) * res DIV (num_vals-1) + tmin
   ELSE res := img  (* unchanged *)
  END;
  RETURN res;
END gray_reduce;


PROCEDURE int_stretch(img: VECTOR OF INTEGER; i_min,i_max: INTEGER): VECTOR OF INTEGER;
(* stretch integer values into interval [i_min..i_max] *)
VAR tmax,tmin: INTEGER;
BEGIN
  tmin := REDUCE.MIN(img);
  tmax := REDUCE.MAX(img);
  IF tmin = tmax THEN INC(tmax) END;  (* avoid division by 0 *)
  RETURN (i_max-i_min) * (img - tmin) DIV (tmax-tmin) + i_min;
END int_stretch;
 
 
PROCEDURE gray_overlay(front, back: VECTOR OF gray): VECTOR OF gray;
(* substitute front for back at all positions where front#black *)
BEGIN
  IF front # g_black THEN back := front END;
  RETURN back;
END gray_overlay;
 
 
PROCEDURE color_overlay(front, back: VECTOR OF color): VECTOR OF color;
(* substitute front for back at all positions where front#black *)
BEGIN
  IF front # c_black THEN back := front END;
  RETURN back;
END color_overlay;

END Trans.

