IMPLEMENTATION MODULE fourier_transform;

(* ---------------------------------------------------------------------------- *)
(* Projekt:  Studienarbeit Parallele Bildtransformationen			*)
(* Funktion: Stellt Prozeduren zur Anwendung der Fourier-Transformation und zur	*)
(*	     Berechnung des Fourier-Spektrums fuer Graustufenbilder zur		*)
(* 	     Verfuegung.							*)
(* System:   SunOS 4.1.3							*)
(* Sprache:  Parallaxis III							*)
(* Autor:    Stefan Feyrer							*)
(* Beginn:   04.01.1994								*)
(* Stand:    08.11.1994								*)
(* ---------------------------------------------------------------------------- *)

FROM ImageIO            IMPORT gray;
FROM Local              IMPORT grid, left, right, up, down;
FROM complex_arithmetic IMPORT complex, complex_abs, complex_add, complex_sub,
                               complex_mult;

(* ---------------------------------------------------------------------------- *)

TYPE direction_type = (vertical, horizontal);

(* ---------------------------------------------------------------------------- *)
	  (* Interne Prozedur, die die Minima je zweier Zahlen liefert.		*)
	  (* ------------------------------------------------------------------ *)

PROCEDURE minimum (a, b: VECTOR OF CARDINAL): VECTOR OF CARDINAL;

  VAR c: VECTOR OF CARDINAL;

  BEGIN (* minimum *)

    IF a < b THEN
      c := a;
    ELSE
      c := b;
    END; (* IF *)
    
    RETURN c; 

  END minimum;

(* ---------------------------------------------------------------------------- *)
	  (* Interne Prozedur, die x-te Potenz von w=e^(-2*pi*i/size) liefert.	*)
	  (* ------------------------------------------------------------------ *)

PROCEDURE w_power (x: VECTOR OF CARDINAL; size: CARDINAL): VECTOR OF complex;

  VAR result: VECTOR OF complex;
      temp:   VECTOR OF REAL;

  BEGIN (* w_power *)

    temp := -2.0 * pi * FLOAT (x) / FLOAT (size);
    result.re := cos (temp);
    result.im := sin (temp);
    RETURN result;

  END w_power;

(* ---------------------------------------------------------------------------- *)
	  (* Interne Prozedur, die die kleinste Zweierpotenz liefert, die	*)
	  (* groesser oder gleich der uebergebenen Zahl ist.			*)
	  (* ------------------------------------------------------------------ *)

PROCEDURE get_size (n: CARDINAL): CARDINAL;

  VAR result: CARDINAL;

  BEGIN (* get_size *)

    result := 1;
    WHILE result < n DO
      result := result * 2;
    END; (* WHILE *)

    RETURN result;

  END get_size;

(* ---------------------------------------------------------------------------- *)
	  (* Interne Prozedur, die die Zahl mit der umgekehrten Bitfolge	*)
	  (* bezueglich der uebergebenen Groesse liefert.			*)
	  (* ------------------------------------------------------------------ *)

PROCEDURE reverse_number (n:    VECTOR OF CARDINAL;
                          size: CARDINAL): VECTOR OF CARDINAL;

  VAR result: VECTOR OF CARDINAL;
      power:  CARDINAL;
      i:      CARDINAL;

  BEGIN (* reverse_number *)

    IF n < size THEN

      result := 0;
      power := size;

      FOR i := 1 TO size DO
        power := power DIV 2;
        IF ODD (n) THEN
          result := result + power;
        END; (* IF *)
        n := n DIV 2;
      END; (* FOR *)

    ELSE
      result := n;
    END; (* IF *)

    RETURN result;

  END reverse_number;

(* ---------------------------------------------------------------------------- *)
	  (* Interne Prozedur, die die uebergebenen Bildwerte in der angege-	*)
	  (* benen Richtung gemaess der Bitumkehrungskunktion vertauscht.	*)
	  (* ------------------------------------------------------------------ *)

PROCEDURE reverse (VAR image:     grid OF complex;
                       size:      CARDINAL;
                       direction: direction_type);

  VAR rev: grid OF CARDINAL;

  BEGIN (* reverse *)

    IF direction = horizontal THEN
      rev := reverse_number (DIM (grid, 1)-LOWER (grid, 1), size) + LOWER (grid, 1);
      SEND.<:*, rev:> (image, image);
    ELSE
      rev := reverse_number (DIM (grid, 2)-LOWER (grid, 2), size) + LOWER (grid, 2);
      SEND.<:rev, *:> (image, image);
    END; (* IF *)

  END reverse;

(* ---------------------------------------------------------------------------- *)
	  (* Interne Prozedur, die die den eigentlichen Transformationsschritt	*)
	  (* durchfuehrt.							*)
	  (* ------------------------------------------------------------------ *)

PROCEDURE transform (VAR image:     grid OF complex;
                         size:      CARDINAL;
                         direction: direction_type);

  VAR two_power_m: CARDINAL;
      log_of_size: CARDINAL;
      m:           CARDINAL;
      p:           CARDINAL;
      q:           CARDINAL;
      e:           grid OF complex;
      image_plus:  grid OF complex;
      temp:        grid OF complex;
      zero_dim:    grid OF CARDINAL;

  BEGIN (* transform *)

    two_power_m := 1;
    log_of_size := TRUNC (ln (FLOAT (size)) / ln (2.0));

    IF direction = horizontal THEN

      zero_dim := DIM (grid, 1) - LOWER (grid, 1);
      FOR m := 1 TO log_of_size DO
        p := two_power_m;
        two_power_m := two_power_m * 2;
        q := size DIV two_power_m;
        e := w_power ((zero_dim MOD two_power_m) * q, size);
        SEND.left:p (image, image_plus);
        IF zero_dim MOD two_power_m = zero_dim MOD p THEN
          temp := complex_mult (image_plus, e);
          image_plus := complex_sub (image, temp);
          image := complex_add (image, temp);
          SEND.right:p (image_plus, image);
        END; (* IF *)
      END; (* FOR *)

    ELSE

      zero_dim := DIM (grid, 2) - LOWER (grid, 2);
      FOR m := 1 TO log_of_size DO
        p := two_power_m;
        two_power_m := two_power_m * 2;
        q := size DIV two_power_m;
        e := w_power ((zero_dim MOD two_power_m) * q, size);
        SEND.down:p (image, image_plus);
        IF zero_dim MOD two_power_m # zero_dim MOD p THEN
          temp := complex_mult (image_plus, e);
          image_plus := complex_sub (image, temp);
          image := complex_add (image, temp);
          SEND.up:p (image_plus, image);
        END; (* IF *)
      END; (* FOR *)

    END; (* IF *)

  END transform;

(* ---------------------------------------------------------------------------- *)
	  (* Interne Prozedur, die die Fourier-Transformation in der angege-	*)
	  (* benen Richtung durchfuehrt. Ist 'division' TRUE, werden die	*)
	  (* Ergebnisse durch die Laenge des Transformationsvektors dividiert.	*)
	  (* ------------------------------------------------------------------ *)

PROCEDURE complex_fft (VAR image:     grid OF complex;
                           width:     CARDINAL;
                           height:    CARDINAL;
                           direction: direction_type;
                           division:  BOOLEAN);

  VAR size: CARDINAL;

  BEGIN (* complex_fft *)

	  (* Ermitteln der Feldgroesse, auf der die Fourier-Transformation	*)
	  (* durchgefuehrt werden soll. Alle Werte in diesem Bereich, die nicht	*)
	  (* durch das Bild ausgefuellt sind, werden auf 0 gesetzt.		*)

    IF direction = horizontal THEN
      size := get_size (width);
      IF width < DIM (grid, 1)-LOWER (grid, 1)+1 <= size THEN
        image.re := 0.0;
        image.im := 0.0;
      END; (* IF *)
    ELSE
      size := get_size (height);
      IF height < DIM (grid, 2)-LOWER (grid, 2)+1 <= size THEN
        image.re := 0.0;
        image.im := 0.0;
      END; (* IF *)
    END; (* IF *)

    reverse (image, size, direction);
    transform (image, size, direction);

    IF division THEN
      image.re := image.re / FLOAT (size);
      image.im := image.im / FLOAT (size);
    END; (* IF *)

  END complex_fft;

(* ---------------------------------------------------------------------------- *)
	  (* Fourier-transformiert das Eingabebild 'image' mittels des		*)
	  (* FFT-Algorithmus. Die entstehenden Ergebnisse sind komplexwertig.	*)
	  (* ------------------------------------------------------------------ *)

PROCEDURE fft (image: grid OF gray; width, height: CARDINAL): grid OF complex;

  VAR result: grid OF complex;

  BEGIN (* fft *)

	  (* Umwandlung in ein komplexwertiges Bild. *)

    result.re  := FLOAT (image);
    result.im  := 0.0;

	  (* Transformation mit Division. *)

    complex_fft (result, width, height, horizontal, TRUE);
    complex_fft (result, width, height, vertical,   TRUE);

    RETURN result;

  END fft;

(* ---------------------------------------------------------------------------- *)
	  (* Wendet die inverse Fourier-Transformation auf das komplexwertige	*)
	  (* Eingabebild an.							*)
	  (* ------------------------------------------------------------------ *)

PROCEDURE inverse_fft (image:         grid OF complex; 
                       width, height: CARDINAL): grid OF gray;

  VAR result: grid OF gray;

  BEGIN (* inverse_fft *)

	  (* Bildung der komplex-konjugierten Werte. *)

    image.im := - image.im;

	  (* Transformation ohne Division. *)

    complex_fft (image, width, height, horizontal, FALSE);
    complex_fft (image, width, height, vertical,   FALSE);

	  (* Bildung des Absolutbetrags. *)

    result := minimum (CARDINAL (complex_abs (image)), grid (MAX (gray)));

    RETURN result;

  END inverse_fft;

(* ---------------------------------------------------------------------------- *)
	  (* Liefert das Fourier-Spektrum des uebergebenen Bildes.		*)
	  (* 'centred' gibt an, ob das Spektrum zentriert dargestellt werden	*)
	  (* soll. Im nicht-zentrierten Fall sind die niederfrequenten Anteile	*)
	  (* am Rand und die hochfrequenten in der Mitte des Bildes darge-	*)
	  (* stellt; im zentrierten Fall entsprechend umgekehrt.		*)
	  (* 'log-scale' gibt an, ob die Darstellung linear oder logarithmisch	*)
	  (* erfolgen soll.							*)
	  (* ------------------------------------------------------------------ *)

PROCEDURE fourier_spectrum (image:     grid OF gray;
                            width:     CARDINAL;
                            height:    CARDINAL;
                            centred:   BOOLEAN;
                            log_scale: BOOLEAN): grid OF gray;

  VAR temp_image:    grid OF complex;
      result:        grid OF gray;
      log_pixel:     grid OF REAL;
      max_log_pixel: REAL;

  BEGIN (* fourier_spectrum *)

	  (* Umwandlung in ein komplexwertiges Bild. *)

    temp_image.re := FLOAT (image);
    temp_image.im := 0.0;

	  (* Um eine zentrierte Darstellung zu erreichen, muss das Bild an den	*)
	  (* 'schwarzen Schachbrettstellen' invertiert werden.			*)

    IF centred THEN
      IF (DIM (grid, 1) + DIM (grid, 2)) MOD 2 = 1 THEN
        temp_image.re := -temp_image.re;
      END; (* IF *)
    END; (* IF *)

    complex_fft (temp_image, width, height, horizontal, TRUE);
    complex_fft (temp_image, width, height, vertical,   TRUE);

    IF log_scale THEN
      log_pixel := ln (1.0 + complex_abs (temp_image));
      max_log_pixel := REDUCE.MAX (log_pixel);
      result := CARDINAL (log_pixel/max_log_pixel*255.0);
    ELSE
      result := minimum (CARDINAL (complex_abs (temp_image)), grid (MAX (gray)));
    END; (* IF *)

    RETURN result;

  END fourier_spectrum;

(* ---------------------------------------------------------------------------- *)

END fourier_transform.

