IMPLEMENTATION MODULE hough_transform;

(* ---------------------------------------------------------------------------- *)
(* Projekt:  Studienarbeit Parallele Bildtransformationen			*)
(* Funktion: Stellt Prozeduren zur Anwendung der Hough-Transformation auf	*)
(*           Graustufenbilder zur Verfuegung.					*)
(* System:   SunOS 4.1.3							*)
(* Sprache:  Parallaxis III							*)
(* Autor:    Stefan Feyrer							*)
(* Beginn:   04.01.1994								*)
(* Stand:    11.02.1995								*)
(* ---------------------------------------------------------------------------- *)

FROM Local   IMPORT grid, left, right, up, down;
FROM ImageIO IMPORT binary, b_white, b_black;
FROM Storage IMPORT ALLOCATE, DEALLOCATE;

(* ---------------------------------------------------------------------------- *)

TYPE acc_element_type = RECORD
                          count: CARDINAL;
                          dist:  REAL;
                          angle: REAL;
                        END; (* RECORD *)

VAR  delta_angle: REAL;
     delta_dist:  REAL;

(* ---------------------------------------------------------------------------- *)
	  (* Interne Prozedur, die in eine verkettete Liste ein Element         *)
	  (* eintraegt.                                                         *)
	  (* ------------------------------------------------------------------ *)

PROCEDURE enter_element (VAR list: line_list_type; element: acc_element_type);

  VAR list_element: line_list_type;

  BEGIN (* enter_element *)

    NEW (list_element);
    WITH list_element^ DO
      dist  := element.dist;
      angle := element.angle;
      next  := list;
    END; (* WITH *)
    list := list_element;

  END enter_element;

(* ---------------------------------------------------------------------------- *)
	  (* Wendet die Hough-Transformation auf das uebergebene Konturbild an.	*)
	  (* 'max_size' muss dabei das Maximum von 'width' und 'height' ent-	*)
	  (* halten. Die Anzahl der zu extrahierenden Geraden wird in 'number'	*)
	  (* uebergeben. Zuruechgeliefert wird eine Liste der extrahierten	*)
	  (* Geraden.								*)
	  (* ------------------------------------------------------------------ *)

PROCEDURE hough_trans (image:    grid OF binary; 
                       width:    CARDINAL;
                       height:   CARDINAL;
                       max_size: CARDINAL;
                       number:   CARDINAL): line_list_type;

  CONFIGURATION acc_grid = grid [1..2*max_size-1],[1..2*max_size-2];

  VAR  cos_angle: acc_grid OF REAL;
       sin_angle: acc_grid OF REAL;
       temp:      acc_grid OF REAL;
       acc:       acc_grid OF acc_element_type;
       max_angle: CARDINAL;
       max_dist:  CARDINAL;
       x, y:      INTEGER;
       end:       BOOLEAN; 

  (* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *)
	  (* Bildet die 5x5-Summe der uebergebenen Vektor-Variablen.		*)
	  (* ------------------------------------------------------------------ *)

  PROCEDURE sum_5x5 (v:         acc_grid OF INTEGER; 
                     max_dist:  CARDINAL;
                     max_angle: CARDINAL): acc_grid OF INTEGER;

    VAR temp1, temp2, temp3, temp4, result: acc_grid OF INTEGER;

    BEGIN (* sum_5x5 *)

	(* Horizontale Summe.							*)

      temp1  := MOVE.left (v);
      IF DIM (acc_grid, 1) = max_angle THEN
        temp1 := 0;
      END; (* IF *)

      temp2  := MOVE.right (v);
      IF DIM (acc_grid, 1) = 1 THEN
        temp2 := 0;
      END; (* IF *)

      temp3  := MOVE.left (temp1);
      IF DIM (acc_grid, 1) = max_angle THEN
        temp3 := 0;
      END; (* IF *)

      temp4  := MOVE.right (temp2);
      IF DIM (acc_grid, 1) = 1 THEN
        temp4 := 0;
      END; (* IF *)

      result := v + temp1 + temp2 + temp3 + temp4;

	(* Vertikale Summe.							*)

      temp1  := MOVE.down (result);
      IF DIM (acc_grid, 2) = 1 THEN
        temp1 := 0;
      END; (* IF *)

      temp2  := MOVE.up (result);
      IF DIM (acc_grid, 2) = max_dist THEN
        temp2 := 0;
      END; (* IF *)

      temp3  := MOVE.down (temp1);
      IF DIM (acc_grid, 2) = 1 THEN
        temp3 := 0;
      END; (* IF *)

      temp4  := MOVE.up (temp2);
      IF DIM (acc_grid, 2) = max_dist THEN
        temp4 := 0;
      END; (* IF *)

      result := result + temp1 + temp2 + temp3 + temp4;

      RETURN (result);

    END sum_5x5;

  (* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *)
	  (* Liefert in einer verketteten Liste die 'number' laengsten Geraden  *)
	  (* des Ursprungsbildes.                                               *)
	  (* ------------------------------------------------------------------ *)

  PROCEDURE get_lines (acc:       acc_grid OF acc_element_type;
                       max_dist:  CARDINAL;
                       max_angle: CARDINAL;
                       number:    CARDINAL): line_list_type;

  VAR line_list:     line_list_type;
      maximum:       CARDINAL;
      s:             acc_grid OF CARDINAL;
      element:       acc_element_type;
      neg_count:     acc_grid OF INTEGER;
      neg_sum:       acc_grid OF INTEGER;
      d1, d2:        CARDINAL;
      i:             CARDINAL;
      nr:            CARDINAL;

    BEGIN (* get_lines *)

      line_list := NIL;

	  (* Es wird 'number'-mal der groesste 5x5-Summen-Wert gesucht. Die	*)
	  (* jeweiligen Geradenparameter werden in eine Liste eingetragen.	*)

      s := sum_5x5 (acc.count, max_dist, max_angle);
      FOR i := 1 TO number DO
        maximum := REDUCE.MAX (s);
        IF maximum = s THEN
          d1 := REDUCE.FIRST (DIM (acc_grid, 1));
          d2 := REDUCE.FIRST (DIM (acc_grid, 2));
          nr := REDUCE.FIRST (ID (acc_grid));
          element := acc <<nr>>;
          enter_element (line_list, element);
        END; (* IF *)

	  (* Die 5x5-Umgebung des gefundenen Punktes muss fuer die naechste	*)
	  (* Maximumbildung auf Null gesetzt werden.				*)
          (* Ausserdem muessen die Summenwerte der umgebenden Punkte            *)
          (* entsprechend verkleinert werden.					*)

        neg_count := 0;
        neg_sum   := 0;
        IF (d1-2 <= DIM (acc_grid, 1) <= d1+2) AND 
           (d2-2 <= DIM (acc_grid, 2) <= d2+2) THEN
          neg_count := -acc.count; 
        END; (* IF *)
        IF (d1-4 <= DIM (acc_grid, 1) <= d1+4) AND 
           (d2-4 <= DIM (acc_grid, 2) <= d2+4) THEN
          neg_sum := sum_5x5 (neg_count, max_dist, max_angle);
          s := s + neg_sum;
        END; (* IF *)
        IF (d1-2 <= DIM (acc_grid, 1) <= d1+2) AND 
           (d2-2 <= DIM (acc_grid, 1) <= d2+2) THEN
          s := 0;
        END; (* IF *)

      END; (* FOR *)

      RETURN line_list;

    END get_lines;

  (* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *)
	  (* Liefert den ersten gesetzten Bildpunkt des uebergebenen Bildes und *)
	  (* loescht diesen. 'end' wird TRUE, falls dies der letzte gesetzte	*)
	  (* Punkt des Bildes war.						*)
	  (* ------------------------------------------------------------------ *)

  PROCEDURE get_next_pixel (VAR image:  grid OF binary; 
                                width:  CARDINAL;
                                height: CARDINAL;
			    VAR x:      INTEGER;
                            VAR y:      INTEGER;
                            VAR end:    BOOLEAN);                             

    VAR nr: CARDINAL;

    BEGIN (* get_next_pixel *)

      IF (DIM (grid, 1)-LOWER (grid, 1)+1 <= width) AND 
         (DIM (grid, 2)-LOWER (grid, 2)+1 <= height) THEN
        IF image = b_white THEN

	  (* Der erste gesetzte Punkt des Bildes wird bestimmt und geloescht.	*)

          x := REDUCE.FIRST (DIM (grid, 1)) - LOWER (grid, 1) + 1 - width  DIV 2;
          y := REDUCE.FIRST (DIM (grid, 2)) - LOWER (grid, 2) + 1 - height DIV 2;
          nr := REDUCE.FIRST (ID (grid));
          image <<nr>> := b_black;
        END; (* IF *)
        end := REDUCE.AND (image); 
      END; (* IF *)

    END get_next_pixel;

  (* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *)

  BEGIN (* hough_trans *)

    max_dist      := 2*(max_size)-1;
    max_angle     := 2*(max_size)-2; 
    delta_angle   := pi/FLOAT(max_angle);
    delta_dist    := sqrt ((FLOAT(max_size)**2.0)/2.0)/FLOAT(max_dist);

	  (* Initialisierung des Akkumulators.					*)

    acc.count := 0;
    acc.dist  := FLOAT ((DIM (acc_grid, 2) - max_size))*delta_dist;
    acc.angle := -(pi/2.0) + FLOAT (DIM (acc_grid, 1))*delta_angle - delta_angle/2.0;

	  (* Sinus und Cosinus der Winkel werden oft benoetigt und daher in	*)
	  (* Variablen abgelegt.						*)

    cos_angle := cos (acc.angle);
    sin_angle := sin (acc.angle);

	  (* Der erste gesetzte Punkt des Bildes wird bestimmt.			*)

    REPEAT
      end := TRUE; 
      get_next_pixel (image, width, height, x, y, end);
 
	  (* Fuer jeden einzelnen gesetzten Punkt im Eingabebild werden die-	*)
	  (* jenigen Zellen des Akkumulators inkrementiert, die durch den	*)
	  (* jeweiligen Punkt verlaufende Geraden repraesentieren.		*)

      IF NOT end THEN
        temp := FLOAT (x) * cos_angle + FLOAT (y) * sin_angle;
        IF acc.dist-delta_dist/2.0 < temp <= acc.dist+delta_dist/2.0 THEN
          INC (acc.count);
        END; (* IF *)
      END; (* IF *)
    UNTIL end;

    RETURN get_lines (acc, max_dist, max_angle, number);

  END hough_trans;

(* ---------------------------------------------------------------------------- *)
	  (* Zeichnet die uebergebenen Geraden in ein Binaerbild ein.           *)
	  (* ------------------------------------------------------------------ *)

PROCEDURE draw_lines (VAR image:  grid OF binary;
                          width:  CARDINAL;
                          height: CARDINAL;
                          lines:  line_list_type);

  VAR x:    grid OF INTEGER;
      y:    grid OF INTEGER;
      temp: grid OF REAL;
      help: line_list_type;

  BEGIN (* draw_lines *)

    x := DIM (grid, 1) - LOWER (grid, 1) + 1 - width   DIV 2;
    y := DIM (grid, 2) - LOWER (grid, 2) + 1 - height  DIV 2;

    image := b_black;

	  (* Fuer jede Gerade der Liste wird der lokale Lotabstand berechnet	*)
	  (* und mit dem Lotabstand-Eintrag der Greadenliste verglichen. Liegen	*)
	  (* die beiden Werte nahe genug beieinander, wird der Bildpunkt	*)
	  (* gesetzt.								*)

    help := lines;
    WHILE help # NIL DO
      temp := FLOAT (x) * cos (help^.angle) + FLOAT (y) * sin (help^.angle);
      IF help^.dist-delta_dist < temp <= help^.dist+delta_dist THEN
        image := b_white;
      END; (* IF *)
      help := help^.next;
    END; (* WHILE *)

  END draw_lines;

(* ---------------------------------------------------------------------------- *)
	  (* Extrahiert die Geradenstuecke, die mit dem Ursprungsbild ueberein- *)
	  (* stimmen.                                                           *)
	  (* ------------------------------------------------------------------ *)

PROCEDURE extract_lines (VAR image:      grid OF binary;
                             orig_image: grid OF binary; 
                             width:      CARDINAL;
                             height:     CARDINAL;
                             lines:      line_list_type);

  CONST max_size = 256;

  TYPE  bin_array = ARRAY [1..max_size] OF binary;

  VAR   x:         grid OF INTEGER;
        y:         grid OF INTEGER;
        temp:      grid OF REAL;
        help:      line_list_type;
        line:      bin_array;
        number:    CARDINAL;
        max_space: CARDINAL;
 
  (* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *)
	  (* Fuegt die als lineares Feld uebergebenen Geradenstuecke zusammen,  *)
	  (* sofern sie nicht weiter als 5 Pixel voneinander entfernt sind.     *)
	  (* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *)

  PROCEDURE join_line (VAR line: bin_array; size, max_space: CARDINAL);

    VAR i, j:      CARDINAL;
        count:     CARDINAL;
  
    BEGIN (* join_line *)

      i := 1;
      count := 0;
      WHILE i <= size DO
        IF line [i] = b_white THEN
          IF count <= max_space+1 THEN
            FOR j := i-count+1 TO i-1 DO
              line [j] := b_white;
            END; (* FOR *)
          END; (* IF *)
          count := 1;
        ELSE
          IF count > 0 THEN
            INC (count);
          END; (* IF *)
        END; (* IF *)
        INC (i);
      END; (* WHILE *)

    END join_line;

  (* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *)

  BEGIN (* extract_lines *)

    x := DIM (grid, 1) - LOWER (grid, 1) + 1 - width   DIV 2;
    y := DIM (grid, 2) - LOWER (grid, 2) + 1 - height  DIV 2;

    image := b_black;

	  (* Die maximale Lueckengroesse wird anhaengig von der Bildgroesse	*)
	  (* berechnet.								*)

    max_space := (width+height) DIV 100;

	  (* Fuer jede Gerade der Liste wird der lokale Lotabstand berechnet	*)
	  (* und mit dem Lotabstand-Eintrag der Greadenliste verglichen. Auf	*)
	  (* der Strecke, auf der die beiden Lotanstaende nahe genug beiein-	*)
	  (* ander liegen, wird versucht, Luecken zu schliessen.		*)

    help := lines;
    WHILE help # NIL DO
      temp := FLOAT (x) * cos (help^.angle) + FLOAT (y) * sin (help^.angle);
      IF help^.dist-delta_dist < temp <= help^.dist+delta_dist THEN
        STORE (orig_image, line, number);
        join_line (line, number, max_space);
        LOAD (image, line);
      END; (* IF *)
      help := help^.next;
    END; (* WHILE *)

  END extract_lines;

(* ---------------------------------------------------------------------------- *)
	  (* Gibt den Speicher fuer die uebergebene Liste frei.                 *)
	  (* ------------------------------------------------------------------ *)

PROCEDURE free_list (list: line_list_type);

  VAR help: line_list_type;

  BEGIN (* free_list *)

    WHILE list # NIL DO
      help := list^.next;
      DISPOSE (list);
      list := help;
    END; (* WHILE *)

  END free_list;

(* ---------------------------------------------------------------------------- *)

END hough_transform.
