IMPLEMENTATION MODULE kl_transform;

(* ---------------------------------------------------------------------------- *)
(* Projekt:  Studienarbeit Parallele Bildtransformationen			*)
(* Funktion: Stellt Prozeduren zur Anwendung der Karhunen-Loeve-Transformation	*)
(*           zur Verfuegung.							*)
(* System:   SunOS 4.1.3							*)
(* Sprache:  Parallaxis III							*)
(* Autor:    Stefan Feyrer							*)
(* Beginn:   11.03.1994								*)
(* Stand:    28.12.1994								*)
(* ---------------------------------------------------------------------------- *)

FROM ImageIO IMPORT binary;
FROM Local   IMPORT grid;

(* ---------------------------------------------------------------------------- *)

	  (* Fuehrt die Karhunen-Loeve-Transformation auf dem uebergebenen	*)
	  (* Binaerbild aus, indem die Positionen der gesetzten Pixel als	*)
	  (* Zufallssignal interpretiert werden. Der Inhalt des Bildes wird	*)
	  (* durch die Transformation an den beiden Hauptachsen ausgerichtet.	*)
	  (* Dies kann dazu verwendet werden, Bilder, die Objekte in ver-	*)
	  (* schiedenen Positionen und Orientierungen beinhalten, standardi-	*)
	  (* siert darzustellen. Damit ist die Moeglichkeit gegeben, Diese	*)
	  (* Bilder z.B. durch Differenzbildberechnung zu miteinander zu ver-	*)
	  (* gleichen.								*)

PROCEDURE kl_trans (image: grid OF binary; width, height: CARDINAL): grid OF binary;

  TYPE vector_2 = ARRAY [1..2] OF REAL;
       matrix_2 = ARRAY [1..2] OF vector_2;

  VAR  half_width:  INTEGER;
       half_height: INTEGER;
       x, y:        grid OF INTEGER;
       count:       CARDINAL;
       ave:         vector_2;
       cov:         matrix_2;
       lambda1:     REAL;
       lambda2:     REAL;
       kl_matrix:   matrix_2;
       x_new:       grid OF INTEGER;
       y_new:       grid OF INTEGER;
       result:      grid OF binary;

  (* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *)

	  (* Liefert den zweidimensionalen Signalmittelwert.			*)

  PROCEDURE compute_average (    x, y:  grid OF INTEGER;
                                 count: CARDINAL;
                             VAR ave:   vector_2);

    BEGIN (* compute_average *)

      ave[1] := FLOAT (REDUCE.SUM (x)) / FLOAT (count);
      ave[2] := FLOAT (REDUCE.SUM (y)) / FLOAT (count);

    END compute_average;

  (* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *)

	  (* Berechnet die zweidimensionale Kovarianzmatrix.			*)

  PROCEDURE compute_cov (    x, y:  grid OF INTEGER;
                             count: CARDINAL;
                             ave:   vector_2; 
                         VAR cov:   matrix_2);

    BEGIN (* compute_cov *)

      cov[1,1] := FLOAT (REDUCE.SUM (x*x)) / FLOAT (count);
      cov[1,2] := FLOAT (REDUCE.SUM (x*y)) / FLOAT (count);
      cov[2,2] := FLOAT (REDUCE.SUM (y*y)) / FLOAT (count);

      cov[1,1] := cov[1,1] - ave[1]*ave[1];
      cov[1,2] := cov[1,2] - ave[1]*ave[2];
      cov[2,1] := cov[1,2];
      cov[2,2] := cov[2,2] - ave[2]*ave[2];

    END compute_cov;

  (* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *)

	  (* Berechnet die Eigenwerte der uebergebenen Matrix und sortiert sie	*)
	  (* der Groesse nach absteigend (lambda1 >= lambda2).			*)

  PROCEDURE compute_eigenvalues (m: matrix_2; VAR lambda1, lambda2: REAL);

    VAR temp: REAL;
        root: REAL;

    BEGIN (* compute_eigenvalues *)

	  (* Berechnung der Eigenwerte.						*)

      temp    := (m[1,1] + m[2,2]) / 2.0;
      root    := sqrt (temp*temp - m[1,1]*m[2,2] + m[1,2]*m[1,2]);
      lambda1 := temp + root;
      lambda2 := temp - root;

	  (* Anordnung der Eigenwerte der Groesse nach.				*)

      IF lambda1 < lambda2 THEN
        temp    := lambda1;
        lambda1 := lambda2;
        lambda2 := temp;
      END; (* IF *)

    END compute_eigenvalues;

  (* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *)

	  (* Berechnet die Eigenmatrix zur Kovarianzmatrix, also die eigent-	*)
	  (* liche Transformationsmatrix.					*)

  PROCEDURE compute_kl_matrix (    cov:       matrix_2;
                                   lambda1:   REAL;
                                   lambda2:   REAL;
                               VAR kl_matrix: matrix_2);

    VAR norm: REAL;

    BEGIN (* compute_kl_matrix *)

      IF cov[1,2] # 0.0 THEN

        kl_matrix[1,1] := (lambda1-cov[2,2]) / cov[1,2];
        kl_matrix[1,2] := (lambda2-cov[2,2]) / cov[1,2];

        norm := sqrt (kl_matrix[1,1]*kl_matrix[1,1]+1.0);
        kl_matrix[1,1] := kl_matrix[1,1] / norm;
        kl_matrix[2,1] := 1.0 / norm;

        norm := sqrt (kl_matrix[1,2]*kl_matrix[1,2]+1.0);
        kl_matrix[1,2] := kl_matrix[1,2] / norm;
        kl_matrix[2,2] := 1.0 / norm;

        IF ((kl_matrix[1,1] > 0.0) AND (kl_matrix[2,2] < 0.0)) OR
           ((kl_matrix[1,1] < 0.0) AND (kl_matrix[2,2] > 0.0)) THEN
          kl_matrix[1,1] := -kl_matrix[1,1];
          kl_matrix[2,1] := -kl_matrix[2,1];
        END; (* IF *)

      ELSIF cov[1,1]-lambda1 # 0.0 THEN
        kl_matrix[1,1] := 0.0;
        kl_matrix[1,2] := 1.0;
        kl_matrix[2,1] := 1.0;
        kl_matrix[2,2] := 0.0;
      ELSE
        kl_matrix[1,1] := 1.0;
        kl_matrix[1,2] := 0.0;
        kl_matrix[2,1] := 0.0;
        kl_matrix[2,2] := 1.0;
      END; (* IF *)

    END compute_kl_matrix;

  (* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *)

	  (* Berechnet die Eigenmatrix zur Kovarianzmatrix, also die eigent-	*)
	  (* liche Transformationsmatrix.					*)

  PROCEDURE do_transform (    x, y:         grid OF INTEGER; 
                              kl_matrix:    matrix_2;
                              ave:          vector_2; 
                          VAR x_new, y_new: grid OF INTEGER);

    VAR x_temp: grid OF REAL;
        y_temp: grid OF REAL;

    BEGIN (* do_transform *)

      x_temp := FLOAT (x) - ave[1];
      y_temp := FLOAT (y) - ave[2];
      x_new := TRUNC (x_temp*kl_matrix[1,1] + y_temp*kl_matrix[1,2] + 0.5);
      y_new := TRUNC (x_temp*kl_matrix[2,1] + y_temp*kl_matrix[2,2] + 0.5);

    END do_transform;

  (* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -  *)

  BEGIN (* kl_trans *)

    half_width  := width  DIV 2;
    half_height := height DIV 2;

    IF (DIM (grid, 1)-LOWER (grid, 1)+1 <= width) AND 
       (DIM (grid, 2)-LOWER (grid, 2)+1 <= height) THEN

      x := DIM (grid, 1)-LOWER (grid, 1)+1 - half_width;
      y := DIM (grid, 2)-LOWER (grid, 2)+1 - half_height;

      IF image THEN

        count := REDUCE.SUM (grid (1));

	  (* Falls das Bild leer ist, wird das Ausgangsbild zurueckgegeben.	*)

        IF count = 0 THEN
          RETURN image;
        END; (* IF *)

        compute_average (x, y, count, ave);
        compute_cov (x, y, count, ave, cov);
        compute_eigenvalues (cov, lambda1, lambda2);
        compute_kl_matrix (cov, lambda1, lambda2, kl_matrix);
        do_transform (x, y, kl_matrix, ave, x_new, y_new);

      ELSE

        x_new := x;
        y_new := y;

      END; (* IF *)

      x_new := x_new + half_width;
      y_new := y_new + half_height;

	  (* Belegen des Ergebnisbildes.					*)

      result := FALSE;
      SEND.<:y_new, x_new:>:#OR (image, result);

    END; (* IF *)

    RETURN result;

  END kl_trans;

(* ---------------------------------------------------------------------------- *)

END kl_transform.
