IMPLEMENTATION MODULE textures;

(* ---------------------------------------------------------------------------- *)
(* Projekt:  Studienarbeit Parallele Bildtransformationen			*)
(* Funktion: Stellt Prozeduren zur Texturanalyse zur Verfuegung.		*)
(* System:   SunOS 4.1.3							*)
(* Sprache:  Parallaxis III							*)
(* Autor:    Stefan Feyrer							*)
(* Beginn:   18.03.1994								*)
(* Stand:    10.02.1995								*)
(* ---------------------------------------------------------------------------- *)

FROM ImageIO           IMPORT gray, g_black, g_white;
FROM Local             IMPORT left, up, up_r, up_l, grid;
FROM fourier_transform IMPORT fourier_spectrum;

(* ---------------------------------------------------------------------------- *)

	  (* Interne Prozedur, die die transponierte Matrix liefert.		*)

PROCEDURE transpose (matrix: co_occ_grid OF REAL): co_occ_grid OF REAL;

  VAR temp: co_occ_grid OF REAL;

  BEGIN (* transpose *)

    SEND.<:g_white-DIM (co_occ_grid, 1),
           g_white-DIM (co_occ_grid, 2):> (matrix, temp);
    RETURN temp;

  END transpose;

(* ---------------------------------------------------------------------------- *)

	  (* Liefert die Co-occurrence-Matrix des uebergebenen Grauwertbildes	*)
	  (* fuer die Verschiebung um einen Pixel in die angegebene Richtung.	*)

PROCEDURE co_occur (image:     grid OF gray; 
                    width:     CARDINAL;
                    height:    CARDINAL;
                    direction: direction_type): co_occ_grid OF REAL;

  VAR moved: grid OF gray;
      
  BEGIN (* co_occur *)

    CASE direction OF
    | horizontal:      moved  := MOVE.left (image);
                       width  := width-1;

    | vertical:        moved  := MOVE.up (image);
                       height := height-1;

    | diagonal_up_r:   moved  := MOVE.up_r (image);
                       moved  := MOVE.left (moved);
                       width  := width-1;
                       height := height-1;

    | diagonal_down_r: moved  := MOVE.up_l (image);
                       width  := width-1;
                       height := height-1;
    END; (* CASE *)

    RETURN co_occur_general (image, moved, width, height);

  END co_occur;

(* ---------------------------------------------------------------------------- *)

	  (* Liefert die Co-occurrence-Matrix des uebergebenen Grauwertbildes	*)
	  (* bezueglich der uebergebenen Verschiebung.				*)

PROCEDURE co_occur_general (image:  grid OF gray;
                            moved:  grid OF gray; 
                            width:  CARDINAL;
                            height: CARDINAL): co_occ_grid OF REAL;

  CONNECTION grid2co_occ: grid[i,j] -> co_occ_grid[i,j];
  
  VAR one_co_occ:    co_occ_grid OF REAL;
      index_1:       co_occ_grid OF CARDINAL;
      index_2:       co_occ_grid OF CARDINAL;
      one_grid:      grid OF REAL;
      co_occur_help: grid OF REAL;
      co_occur:      co_occ_grid OF REAL;

  BEGIN (* co_occur_general *)

    co_occur := 0.0;

    IF LEN (grid, 1) = LEN (grid, 2) THEN

	  (* Fall1: Das Gitter der Co-Occurrence-Matrix ist groesser als das	*)
	  (* Bild.								*)

      IF g_white - g_black >= LEN (grid, 1) THEN

        IF (DIM (co_occ_grid, 1)-g_black+1 <= width) AND
           (DIM (co_occ_grid, 2)-g_black+1 <= height) THEN
          SEND.grid2co_occ (image, index_1);
          SEND.grid2co_occ (moved, index_2);
          one_co_occ := 1.0;
        ELSE
          index_1 := DIM (co_occ_grid, 1);
          index_2 := DIM (co_occ_grid, 2);
          one_co_occ := 0.0;
        END; (* IF *)
      
        SEND.<:index_1, index_2:>:#SUM (one_co_occ, co_occur);

      ELSE

	  (* Fall2: Das Gitter der Co-Occurrence-Matrix ist kleiner als das	*)
	  (* Bild.								*)

        co_occur_help := 0.0;
        IF (DIM (grid, 1)-LOWER (grid, 1)+1 <= width) AND
           (DIM (grid, 2)-LOWER (grid, 2)+1 <= height) THEN
          one_grid := 1.0;
        ELSE
          one_grid := 0.0;
          image := DIM (grid, 1);
          moved := DIM (grid, 2);
        END; (* IF *) 

        SEND.<:image, moved:>:#SUM (one_grid, co_occur_help);
        SEND.grid2co_occ (co_occur_help, co_occur);
      END; (* IF *)

      co_occur := co_occur + transpose (co_occur);
      RETURN co_occur / REDUCE.SUM (co_occur);

    ELSE
      RETURN co_occur;
    END; (* IF *)
 
  END co_occur_general;

(* ---------------------------------------------------------------------------- *)

	  (* Liefert den Energie-Koeffizienten der uebergebenen Matrix.		*)

PROCEDURE feature_energy (co_occur: co_occ_grid OF REAL): REAL;

  BEGIN (* feature_energy *)

    RETURN REDUCE.SUM (co_occur*co_occur);

  END feature_energy;

(* ---------------------------------------------------------------------------- *)

	  (* Liefert den Entropie-Koeffizienten der uebergebenen Matrix.	*)

PROCEDURE feature_entropy (co_occur: co_occ_grid OF REAL): REAL;

  BEGIN (* feature_entropy *)

    IF co_occur # 0.0 THEN 
      RETURN REDUCE.SUM (-co_occur * ln (co_occur));
    END; (* IF *)

  END feature_entropy;

(* ---------------------------------------------------------------------------- *)

	  (* Liefert den Korrelationskoeffizienten der uebergebenen Matrix.	*)

PROCEDURE feature_correlation (co_occur: co_occ_grid OF REAL): REAL;

  VAR temp:   REAL;
      temp_x: co_occ_grid OF REAL;
      temp_y: co_occ_grid OF REAL;
      mx, my: REAL;
      sx, sy: REAL;
      s:      REAL;

  BEGIN (* feature_correlation *)

    mx := REDUCE.SUM (FLOAT (DIM (co_occ_grid, 1)-g_black)*co_occur);
    my := REDUCE.SUM (FLOAT (DIM (co_occ_grid, 2)-g_black)*co_occur);

    temp_x := FLOAT (DIM (co_occ_grid, 1)-g_black) - mx;
    sx := sqrt (REDUCE.SUM (temp_x*temp_x*co_occur));

    temp_y := FLOAT (DIM (co_occ_grid, 2)-g_black) - my;
    sy := sqrt (REDUCE.SUM (temp_y*temp_y*co_occur));

    s := sx * sy;
    RETURN REDUCE.SUM (temp_x*temp_y*co_occur) / s;

  END feature_correlation;

(* ---------------------------------------------------------------------------- *)

	  (* Liefert den Traegheitskoeffizienten der uebergebenen Matrix.	*)

PROCEDURE feature_inertia (co_occur: co_occ_grid OF REAL): REAL;

  VAR temp: co_occ_grid OF INTEGER;

  BEGIN (* feature_inertia *)

    temp := DIM (co_occ_grid, 1) - DIM (co_occ_grid, 2);
    RETURN REDUCE.SUM (FLOAT (temp*temp) * co_occur);

  END feature_inertia;

(* ---------------------------------------------------------------------------- *)

	  (* Liefert den Koeffizienten fuer die lokale Homogenitaet der ueber-	*)
	  (* gebenen Matrix.							*)

PROCEDURE feature_local_homogeneity (co_occur: co_occ_grid OF REAL): REAL;

  VAR temp: co_occ_grid OF INTEGER;

  BEGIN (* feature_local_homogeneity *)

    temp := DIM (co_occ_grid, 1) - DIM (co_occ_grid, 2);
    temp := temp * temp + 1;
    RETURN REDUCE.SUM (1.0 / FLOAT (temp) * co_occur);

  END feature_local_homogeneity;

(* ---------------------------------------------------------------------------- *)

	  (* Liefert den Koeffizienten fuer den 'Kontrast' der uebergebenen	*)
	  (* Matrix (nach Nevatia).						*)

PROCEDURE feature_contrast (co_occur: co_occ_grid OF REAL): REAL;

  VAR temp: REAL;
      n:    INTEGER;
      s:    REAL;
 
  BEGIN (* feature_contrast *)

    s := 0.0;
    FOR n := 1 TO g_white - g_black DO 
      temp := 0.0;
      IF ABS (DIM (co_occ_grid, 1)-g_black 
              - DIM (co_occ_grid, 2)-g_black) = n THEN
        temp := REDUCE.SUM (co_occur);
      END; (* IF *)
      s := s + FLOAT (n) * FLOAT (n) * temp;
    END; (* FOR *)
    RETURN s;

  END feature_contrast;

(* ---------------------------------------------------------------------------- *)

	  (* Liefert die Nummer desjenigen radialen Bereichs im Fourier-	*)
	  (* Spektrum, der die groesste Wertesumme enthaelt.			*)
	  (* Der Bereich im Zentrum des Bildes hat die Nummer 1.		*)

PROCEDURE feature_fourier_radial (image:      grid OF gray; 
                                  width:      CARDINAL;
                                  height:     CARDINAL;
                                  bin_number: CARDINAL): CARDINAL;

  VAR spectrum: grid OF gray;
      i:        CARDINAL;
      temp:     CARDINAL;
      maximum:  CARDINAL;
      result:   CARDINAL;
      delta:    CARDINAL;
      radius:   grid OF CARDINAL;
      x, y:     grid OF CARDINAL;

  BEGIN (* feature_fourier_radial *)

    spectrum := fourier_spectrum (image, width, height, TRUE, TRUE);

    delta := TRUNC ((sqrt (FLOAT (width*width+height*height))) / FLOAT (bin_number));
    x := DIM (grid, 1)-LOWER (grid, 1)+1-(width  DIV 2);
    y := DIM (grid, 2)-LOWER (grid, 2)+1-(height DIV 2);
    radius := TRUNC (sqrt (FLOAT (x*x+y*y)));

    FOR i := 1 TO bin_number DO
      IF (i-1)*delta < radius <= i*delta THEN
        temp := REDUCE.SUM (spectrum);
      END; (* IF *)
      IF temp > maximum THEN
        maximum := temp;
        result := i;
      END ; (* IF *)
    END; (* FOR *)

    RETURN result;

  END feature_fourier_radial;

(* ---------------------------------------------------------------------------- *)

	  (* Liefert die Nummer desjenigen angularen Bereichs im Fourier-	*)
	  (* Spektrum, der die groesste Wertesumme enthaelt.			*)
	  (* Der Bereich direkt oberhalb der x-Achse hat die Nummer 1. Die 	*)
	  (* Numerierung erfolgt im Gegenuhrzeigersinn.				*)

PROCEDURE feature_fourier_angular (image:      grid OF gray;
                                   width:      CARDINAL;
                                   height:     CARDINAL;
                                   bin_number: CARDINAL): CARDINAL;

  VAR spectrum: grid OF gray;
      i:        CARDINAL;
      temp:     CARDINAL;
      maximum:  CARDINAL;
      result:   CARDINAL;
      delta:    REAL;
      angle:    grid OF REAL;
      x, y:     grid OF REAL;

  BEGIN (* feature_fourier_angular *)

    spectrum := fourier_spectrum (image, width, height, TRUE, TRUE);

    delta := 2.0*pi / FLOAT (bin_number);
    x := FLOAT (DIM (grid, 1)-LOWER (grid, 1)+1-(width  DIV 2));
    y := FLOAT (DIM (grid, 2)-LOWER (grid, 2)+1-(height DIV 2));
    angle := arctan2 (y,x) + pi;

    FOR i := 1 TO bin_number DO
      IF FLOAT (i-1)*delta < angle <= FLOAT (i)*delta THEN
        temp := REDUCE.SUM (spectrum);
      END; (* IF *)
      IF temp > maximum THEN
        maximum := temp;
        result := i;
      END ; (* IF *)
    END; (* FOR *)

    RETURN result;

  END feature_fourier_angular;

(* ---------------------------------------------------------------------------- *)

END textures.
