MODULE vision;
(* main program, showing the possibilities of the vision modules *)
(* Thomas Braunl, Univ. Stuttgart, Feb. 1995                     *)
(* enter the zoom factor in the command line, e.g. vision 7      *)
FROM Graphics IMPORT OpenWindow, SelectWindow, CloseWindow,
                     SetColor, Line, window, SetAreaXYZ, SetgAreaXYZ;
FROM ImageIO  IMPORT read_g_image, write_g_image, 
                     binary, gray, g_white, g_black;
IMPORT Local, Trans, Hist, Thinning, Gauss, Strings, Morph, Segm, Corners;

CONST 
 max_width   = 55;     (* maximum image width  *)
 max_height  = 26;     (* maximum image height *)
 p_size      = 11;     (* gauss edge *)
 min_energ   = 4500.0; (* gauss edge *)

CONFIGURATION image = Local.grid[1..max_height],[1..max_width];

VAR g1,g2,dummy,random : image OF gray;
    b1,b2              : image OF binary;
    w1                 : window;
    str                : Strings.string;
    his                : Hist.histogram;
    coeff1,coeff2      : image OF Gauss.coeff_type;
    nr_coeff1,nr_coeff2: image OF CARDINAL;
    r1                 : image OF REAL;
    sigma1,sigma2      : REAL;
    width,height,zoom  : INTEGER;
    answer,num,x,y     : INTEGER;
    quit               : BOOLEAN;

BEGIN
  (* command line options: 1st arg. for zoom *)
  IF argc() = 0 THEN zoom := 5  (* no command line option *)
                ELSE argv(1,str);
                     zoom := Strings.string2int(str);
                     IF zoom=0 THEN zoom := 1 END;
  END;
  WriteString("Thomas Braunl, 1995"); WriteLn;
  WriteString("zoom set to "); WriteInt(zoom,1);
  WriteString(", use command line argument"); WriteLn;
  w1:= OpenWindow("Parallaxis Show", 2*zoom*(max_width+1), zoom*max_height);

  (* command line options: 2nd arg. for image filename *)
  IF argc() >= 2 THEN argv(2,str) ELSE str := "image1.pgm" END;
  read_g_image(g1,str,width,height);  (* open image file *)
  g1 := Trans.gray_reduce(g1,32);     (* reduce gray values for 8 bit screens *)
  SetgAreaXYZ(g1, 0,0,zoom);
  quit := FALSE;

  REPEAT
    WriteLn;
    WriteString("*** PARALLAXIS Image Processing Demo ***");
    WriteLn;
    WriteString(" 1: LOAD image      2: SAVE image       3: COPY back      ");
    WriteString(" 0: QUIT"); WriteLn;
    WriteString("11: add noise      12: random gray     13: invert image   ");
    WriteString("14: CLEAR"); WriteLn;
    WriteString("21: min  3x3       22: max    3x3");
    WriteLn;
    WriteString("31: mean 3x3       32: median 3x3      33: median fast 3x3");
    WriteLn;
    WriteString("34: mean 5x5       35: median 5x5      36: median fast 5x5");
    WriteLn;
    WriteString("41: gauss          42: DoG             43: LoG            ");
    WriteString("44: gauss edges"); WriteLn;
    WriteString("51: laplace        52: sobel-x 3x3     53: sobel-y 3x3");
    WriteLn;
    WriteString("54: sobel str.     55: sobel dir.      56: corner det.");
    WriteLn;
    WriteString("61: erosion        62: dilation        63: open           ");
    WriteString("64: close"); WriteLn;
    WriteString("65: boundary       66: fill            67: connected      ");
    WriteString("68: skeleton"); WriteLn;
    WriteString("71: thin StefRos.  72: thin LueWang    73: thin HallGuo");
    WriteLn;
    WriteString("81: reduce gray    82: binary transf.  83: dither random  ");
    WriteString("84: dither odered"); WriteLn;
    WriteString("91: histogram      92: hist. equalize  93: segmentation");
    WriteLn;
    WriteString("choice: "); ReadInt(answer);

    IF 61 <= answer <= 69 THEN  (* morphological operations *)
      b1 := Trans.gray2binary(g1,128)
    END;
    IF answer=1 THEN
      read_g_image(g1,"", width,height); (* empty string: ask from terminal *)
      g1 := Trans.gray_reduce(g1,32); (* reduce gray values for 8 bit screens *)
      SetgAreaXYZ(g1, 0,0,zoom)
     ELSE
     IF (DIM(image,1)-LOWER(image,1)<=width) AND (DIM(image,2)<=height) THEN
      CASE answer OF
        0: quit := TRUE                                          |
        2: write_g_image(g2,"", width,height)                    |
        3: g1 := g2;
           SetgAreaXYZ(g1, 0,0,zoom)                             |
       11: WriteString("noise percentage: "); ReadInt(num);
           IF 1 <= num <= 100 THEN
             random := RandomInt(image) MOD (200 DIV num);
             g2 := g1;
             IF random = 0 THEN g2 := g_white END; (* "salt"   - noise *)
             IF random = 1 THEN g2 := g_black END; (* "pepper" - noise *)
            ELSE WriteString("wrong percentage"); WriteLn;
           END                                                   |
       12: g2 := RandomInt(image) MOD 256                        |
       13: g2 := g_white - g1                                    |  
       14: ALL image DO
             g2 := g_white;
             SetgAreaXYZ(g2, zoom*(max_width+1), 0, zoom);
           END                                                   |
       21: g2 := Local.min_3x3(g1)                               |  
       22: g2 := Local.max_3x3(g1)                               |  
       31: g2 := Local.mean_3x3(g1)                              |
       32: g2 := Local.median_3x3(g1)                            |
       33: g2 := Local.median_3x3fast(g1)                        |
       34: g2 := Local.mean_5x5(g1)                              |
       35: g2 := Local.median_5x5(g1)                            |
       36: g2 := Local.median_5x5fast(g1)                        |
       41: WriteString("sigma: "); ReadReal(sigma1);
           Gauss.coeff_Gauss(coeff1,nr_coeff1,sigma1);
           g2 := Gauss.gauss(g1,coeff1,nr_coeff1)                |
       42: WriteString("sigma1 sigma2: ");
           ReadReal(sigma1); ReadReal(sigma2);
           Gauss.coeff_Gauss(coeff1,nr_coeff1,sigma1);
           Gauss.coeff_Gauss(coeff2,nr_coeff2,sigma2);
           g2 := Trans.int2gray(
                 Gauss.DoG(g1,coeff1,nr_coeff1, coeff2,nr_coeff2)) |
       43: WriteString("sigma: "); ReadReal(sigma1);
           Gauss.coeff_LoG(coeff1,coeff2,nr_coeff1,sigma1);
           g2 := Trans.int2gray(round(
                 Gauss.LoG(g1,coeff1,coeff2,nr_coeff1)))         |
       44: WriteString("sigma: "); ReadReal(sigma1);
           Gauss.coeff_LoG(coeff1,coeff2,nr_coeff1,sigma1);
           r1 := Gauss.LoG(g1,coeff1,coeff2,nr_coeff1);
           g2 := Trans.binary2gray(
                  Gauss.gauss_edge(Gauss.sgn(r1)) AND
                  (Gauss.energy(r1, image(p_size)) >= min_energ),
                  g_black, g_white)                              |
       51: g2 := Trans.int2gray(Local.laplace_3x3(g1))           |
       52: g2 := Trans.int2gray(Local.sobel_x_3x3(g1))           |
       53: g2 := Trans.int2gray(Local.sobel_y_3x3(g1))           |
       54: Local.edges_sobel_3x3(g1, g2, dummy)                  |
       55: Local.edges_sobel_3x3(g1, dummy, g2)                  |
       56: b2 := Corners.detMeanvar(2, 0.1, 0.666, 5, 0.05, 1.0, g1);
           g2 := Corners.marksqr_gray(g1, b2, g_black, 3, g_white, 1);  |

       61: b2 := Morph.erosion(b1)                               |
       62: b2 := Morph.dilation(b1)                              |
       63: b2 := Morph.open(b1)                                  |
       64: b2 := Morph.close(b1)                                 |
       65: b2 := Morph.boundary(b1)                              |
       66: WriteString("enter fill position x,y: ");
           ReadInt(x); ReadInt(y);
           b2 := Morph.fill(b1, x,y)                             |
       67: WriteString("enter connected position x,y: ");
           ReadInt(x); ReadInt(y);
           b2 := Morph.connected(b1, x,y)                        |
       68: b2 := Morph.skeleton(b1)                              |

       71: g2 := Trans.binary2gray(
                 Thinning.stefRosen(g1 < 128), 0,255)            |
       72: g2 := Trans.binary2gray(
                 Thinning.lueWang(g1 < 128),   0,255)            |
       73: g2 := Trans.binary2gray(
                 Thinning.hallGuo(g1 < 128),   0,255)            |
       81: WriteString("num. gray levels: "); ReadInt(num);
           g2 := Trans.gray_reduce(g1,num)                       |
       82: WriteString("threshold: "); ReadInt(num);
           IF 0 <= num <= 255 THEN
             g2 := Trans.binary2gray(Trans.gray2binary(g1,num), 0,255)
            ELSE WriteString("wrong threshold"); WriteLn;
           END                                                   |
       83: g2 := Trans.binary2gray(Trans.dither_random(g1), 0,255)  |
       84: g2 := Trans.binary2gray(Local.dither_ordered(g1), 0,255) |
       91: his := Hist.gen_histogram_n(g1,num);
           WriteInt(num,3); WriteString(" gray levels found"); WriteLn;
           g2 := g_white;
           Hist.draw_histogram(his,g2, 0,height-1, width,height-1, g_black) |
       92: g2 := Hist.equal_histogram(g1,his)                    |
       93: WriteString("enter seed and threshold: ");
           ReadInt(x); ReadInt(y);
           g2 := Trans.int2gray( Segm.region_growing(g1,x,y) )   |
       ELSE WriteString("wrong code "); WriteInt(answer,1); WriteLn;
      END; (* case *)
      IF 61 <= answer <= 69 THEN  (* morphological operations *)
        g2 := Trans.binary2gray(b2, g_black, g_white)
      END;
      IF answer > 3 THEN
        g2 := Trans.gray_reduce(g2,32);  (* reduce gray v. for 8 bit screens *)
        SetgAreaXYZ(g2, zoom*(max_width+1), 0, zoom);
      END;
     END; (* if DIM *)
    END; (* if answer='1' *)
  UNTIL quit;
  CloseWindow(w1);
END vision.
