/*****************************************************************************
  Project: PARZ - Parallel Intermediate Code Debugger/Interpreter
  ----------------------------------------------------------------------------
  Release      : 1
  Project Part : Debugger
  Filename     : debcode1.c       
  SCCS-Path    : /tmp_mnt/user/sembach/parz/v2/SCCS/s.debcode1.c
  Version      : 1.3 
  Last changed : 9/27/93 at 14:03:50        
  Author       : Frank Sembach
  Creation date: Aug. 92
  ----------------------------------------------------------------------------
  Description  : Funktionen zur Codeerzeugung fuer den
                 symbolischen Debugger von PARZ
******************************************************************************
***      (C) COPYRIGHT University of Stuttgart - All Right Reserved        ***
*****************************************************************************/

static char sccs_id[] = "@(#)debcode1.c	1.3  9/27/93 PARZ - Debugger (Frank Sembach)";


#include "parzdefs.h"
#include "komdefs.h"
#include "y_tab.h"
#include "externs.h"
#include "rundefs.h"
#include "runexts.h"
#include "komexts.h"
#include "funcs.h"
#include "debexts.h"

int subscript_error_num;     /* Programmadresse der Arrayindex Fehlermeldung */
int pointer_error_num;       /* Programmadresse der NIL-pointer Fehlermeldung */
int setrange_error_num;      /* Programmadresse der set range Fehlermeldung */

extern STAT *stat_ptr;       /* erste freie PARZ-Anweisung */
extern int akt_max;          /* hoechste allozierte Programmadresse */

extern int (* assign_do_funcs[2][REAL + 1][STR + 1])();
                             /* Funktionen fuer Zuweisung */
extern struct func_ptrs
{ int (* do_f[2][REAL + 1])();
  int (* print_f)();
} op_funcs[OR - PLUS + 1];   /* Zeiger auf Funktionen fuer unaere und binaere Operatoren */

extern int (* red_do_funcs[STR + 1][MIN - AND + 1])();
                             /* Funktionen fuer REDUCE */

extern DECLIST *dlptr();

/****************************************************************************************/

DECLIST deb_s_decls;         /* SCALAR-Deklaration fuer sym. Debugger */
DECLIST deb_v_decls;         /* VECTOR-Deklaration fuer sym. Debugger */

DISPLAY *save_disp;          /* Sichert 'disp' waehrend symbolischem Debuggen */
SCOPETAB *save_newest_scope; /* Sichert 'newest_scope' waehrend symbolischem Debuggen */
SCOPETAB *save_akt_scope;    /* Sichert 'akt_scope' waehrend symbolischem Debuggen */
int save_akt_tiefe;          /* Sichert 'akt_tiefe' waehrend symbolischem Debuggen */
ZEILTAB *save_akt_line;      /* Sichert 'akt_line' waehrend symbolischem Debuggen */
char *save_aktive_pes;       /* Sichert 'aktive_pes' waehrend symbolischem Debuggen */
int save_called;             /* Sichert 'called' waehrend symbolischem Debuggen */
int save_call_count;         /* Sichert 'call_count' waehrend symbolischem Debuggen */
int save_ended;              /* Sichert 'ended' waehrend symbolischem Debuggen */
int save_err;                /* Sichert 'err' waehrend symbolischem Debuggen */
int save_pc;                 /* Sichert 'pc' waehrend symbolischem Debuggen */
int save_max_lev;            /* Sichert 'max_lev' waehrend symbolischem Debuggen */
int save_max_label;          /* Sichert 'max_label' waehrend symbolischem Debuggen */
int save_warnings_on;        /* Sichert 'warnings_on' waehrend symbolischem Debuggen */
int save_breaked;            /* Sichert 'breaked' waehrend symbolischem Debuggen */
int save_zeile_fertig;       /* Sichert 'zeile_fertig' waehrend symbolischem Debuggen */


/**************************************************************************
 ***                      Funktion init_debcode
 ***
 *** Initialisiert 'programm' so, dass der symbolische Debugger Code 
 *** erzeugen kann.
 *** lev         : Ebene des Debugcodes
 *** procadr     : Befehlsadresse des PROC, zu dem Code erzeugt wird
 *** *ab_abstand : Rueckgabe: Abstand des Aktivierungsblocks zu 'procadr'
 ***                          von Spitze des Aufrufstapels
 ***
 **************************************************************************/

init_debcode(lev, procadr, ab_abstand)
int lev, procadr, *ab_abstand;
{ if (disp)
  { register int i;

    debcode_start = stat_anz;
    stat_ptr = programm + --stat_anz;
    nextstat();
    null_dlist(&deb_s_decls);
    null_dlist(&deb_v_decls);
    subscript_error_num = -325;
    pointer_error_num = -328;
    setrange_error_num = -340;
    akt_index_typ = index_typ1;
    sym_debugging = TRUE;
    debug_lev = lev;
                  /* Zustand sichern */
    save_called = called;
    save_call_count = call_count;
    save_ended = ended;
    save_err = err;
    save_pc = pc;
    save_max_lev = max_lev;
    save_max_label = max_label;
    save_warnings_on = warnings_on;
    save_breaked = breaked;
    save_zeile_fertig = zeile_fertig;
    save_disp = disp;
    save_newest_scope = newest_scope;
    save_akt_scope = akt_scope;
    save_akt_tiefe = akt_tiefe;
    save_akt_line = akt_line;
    save_aktive_pes = aktive_pes;
    if (!(save_aktive_pes = malloc((size_t)(pe_anz + 1))))
    { fputs(texte[80], stdout);
      kill_debcode();
      return;
    }
    strcpy(save_aktive_pes, aktive_pes);
#ifdef DEBDEBUG
    printf("akt_tiefe=%d, top_scope=%x, new_scope=%x\n",
            akt_tiefe, disp[akt_tiefe]->top_scope, disp[akt_tiefe]->new_scope);
#endif
                /* Display duplizieren */
    max_lev = (debug_lev >= max_lev) ? debug_lev + 1 : max_lev;
    if (!(disp = (DISPLAY *)calloc((size_t)(max_lev + 1), (size_t)sizeof(DISPLAY))))
    { fputs(texte[80],stderr);
      kill_debcode();
      return;
    }
    for (i = 0; i <= save_max_lev; i++)
    { disp[i] = save_disp[i]; }
                  /* PROC-Befehl erzeugen */
    funcs(do_proc, print_proc);
    STAT_proclev(*stat_ptr) = debug_lev;
    nextstat();
                  /* evtl. Prozedurblock suchen */
    if (procadr >= 0)
    { if (STAT_print_func(programm[procadr]) != print_proc) procadr = -1;
      *ab_abstand = find_aktblock(procadr, FALSE);
      if (akt_tiefe < 0)
      { komerr(texte[309]); kill_debcode(); }
    }
  }
  else
  { komerr(texte[183]); }
}

/**************************************************************************
 ***                      Funktion end_debcode
 ***
 *** Ergaenzt den erzeugten Debug-Code, so dass er ausgefuehrt werden kann. 
 ***
 **************************************************************************/

end_debcode()
{ if (sym_debugging)
  { register STAT *pstat = programm + debcode_start;

    STAT_dscal(*pstat) = copy_declist(&deb_s_decls, 1);
    STAT_dvec(*pstat) = copy_declist(&deb_v_decls, 1);
    funcs(do_halt, print_halt);
    nextstat();
  }
}

/**************************************************************************
 ***                      Funktion kill_debcode
 ***
 *** Entfernt den seit 'init_debcode()' erzeugten Code.
 ***
 **************************************************************************/

kill_debcode()
{ if (sym_debugging)
  { STAT *debanf = programm + debcode_start;
    int anz = stat_anz - debcode_start;

    loesch_prog(&debanf, &anz, FALSE);
    stat_anz = debcode_start;
    programm = (STAT *)realloc(programm,(size_t)(stat_anz * sizeof(STAT)));
    akt_max = stat_anz - 1;
    loesch_decl(&deb_s_decls);
    loesch_decl(&deb_v_decls);
    sym_debugging = FALSE;
    called = save_called;
    call_count = save_call_count;
    ended = save_ended;
    err = save_err;
    pc = save_pc;
    max_lev = save_max_lev;
    max_label = save_max_label;
    warnings_on = save_warnings_on;
    breaked = save_breaked;
    zeile_fertig = save_zeile_fertig;
    if (disp) free(disp);
    disp = save_disp;
    newest_scope = save_newest_scope;
    akt_scope = save_akt_scope;
    akt_tiefe = save_akt_tiefe;
    akt_line = save_akt_line;
    if (aktive_pes) free(aktive_pes);
    aktive_pes = save_aktive_pes;
  }
}

/**************************************************************************
 ***                      Funktion error_num
 ***
 *** Liefert die Programmadresse einer Fehlermeldung.
 *** Ist '*error_num_ptr' negativ, so ist dies die neg. Nummer des
 *** Fehlertextes, Code fuer die Fehlermeldung wird erzeugt, 
 *** die Ansprungadresse kommt nach '*error_num_ptr'.
 *** Sonst enthaelt '*error_num_ptr' bereits die Ansprungadresse.
 *** Ergebnis ist die Ansprungadresse.
 ***
 **************************************************************************/

int error_num(error_num_ptr)
int *error_num_ptr;
{ register int num;

  if ((num = - *error_num_ptr) > 0)
  { register char *estring = malloc((size_t)(strlen(texte[num]) + 1));

    if (!estring)
    { komerr(texte[80],1); return; }
    strcpy(estring, texte[num]);
        /*     GOTO l2; */
    funcs(do_goto, print_goto);
    STAT_spr_ziel(*stat_ptr) = stat_ptr - programm + 3;
    nextstat();
        /* l1: PROC debug_lev + 1; */
    STAT_label(*stat_ptr) = ++max_label;
    funcs(do_proc, print_proc);
    STAT_proclev(*stat_ptr) = debug_lev + 1;
    STAT_dscal(*stat_ptr) = dlptr(NULL);
    STAT_dvec(*stat_ptr) = dlptr(NULL);
    *error_num_ptr = stat_ptr - programm;
    nextstat();
        /*     ERROR estring; */
    funcs(do_errorcall, print_errorcall);
    ARG_argsort(STAT_vc1(*stat_ptr)) = STR | CON;
    ARG_con_wert(STAT_vc1(*stat_ptr)).datentyp =STR;
    ARG_con_wert(STAT_vc1(*stat_ptr)).inhalt.s_val = estring;
    nextstat();
        /* l2: ... */
    STAT_label(*stat_ptr) = ++max_label;
  }
  return *error_num_ptr;
}

/**************************************************************************
 ***                      Funktion para_setloop_start
 ***
 *** Erzeugt Code, fuer den Beginn einer Schleife fuer eine SET-Operation
 *** auf Variablen des Typs 'styp'.
 *** sets   : 'anz' Zeiger auf die Argumentzeiger der SET-Variablen
 *** ptrs   : 'anz' Argumentzeiger auf Zeigervariablen
 *** count  : Zeiger auf Zaehlervariable
 ***
 **************************************************************************/

para_setloop_start(styp, sets, ptrs, count, anz)
TYPTAB *styp;
ARG **sets;
ARG *ptrs;
ARG *count;
int anz;
{ ARG von, bis;

  ARG_argsort(von) = ARG_argsort(bis) = INT | CON;
  ARG_con_wert(von).datentyp = ARG_con_wert(bis).datentyp = INT | INT_ERLAUBT;
  ARG_con_wert(von).inhalt.i_val = styp->typ_arg2;
  ARG_con_wert(bis).inhalt.i_val = styp->typ_arg3;
  new_var(count, para_typ[INT], 0);
        /* count := bis - von; */
  operation(count, &bis, MINUS, &von);
  nextstat();
  for (; anz; anz--, sets++, ptrs++)
  { if (ARG_argsort(**sets) & IND)
    { register ARGART sv;

      new_var(ptrs, para_typ[INT], sv = (ARG_indsort(**sets) & VEC));
        /* ptrs := index(*sets); */
      funcs(sv ? do_vii_zuw : do_sii_zuw, print_zuw);
      STAT_verg(*stat_ptr) = *ptrs;
      STAT_vc1(*stat_ptr) = **sets;
      ARG_argsort(STAT_vc1(*stat_ptr)) = ARG_indsort(**sets);
      nextstat();
    }
    else
    { new_var(ptrs, para_typ[INT], 0);
        /* ptrs := ADDR *sets; */
      funcs(do_sii_zuw, print_zuw);
      STAT_verg(*stat_ptr) = *ptrs;
      STAT_vc1(*stat_ptr) = **sets;
      ARG_argsort(STAT_vc1(*stat_ptr)) =
        (ARG_argsort(**sets) & ~VEC) | ((ARG_argsort(**sets) & VEC) ? ADV : ADS);
      nextstat();
    }
  }
        /* lab : ... */
  STAT_label(*stat_ptr) = ++max_label;
}

/**************************************************************************
 ***                      Funktion para_setloop_end
 ***
 *** Erzeugt Code, fuer das Ende einer Schleife fuer eine SET-Operation
 *** ptrs    : 'anz' Argumentzeiger auf Zeigervariablen
 *** count   : Zeiger auf Zaehlervariable
 *** loopadr : Programmadresse des Schleifenbeginns
 ***
 **************************************************************************/

para_setloop_end(ptrs, count, anz, loopadr)
ARG *ptrs;
ARG *count;
int anz;
int loopadr;
{ ARG temp;

        /*      IF count = 0 GOTO l2; */
  funcs(do_2_ifgoto, print_ifgoto);
  STAT_vc1(*stat_ptr) = *count;
  ARG_argsort(STAT_vc2(*stat_ptr)) = INT | CON;
  ARG_con_wert(STAT_vc2(*stat_ptr)).datentyp = INT | INT_ERLAUBT;
  ARG_con_wert(STAT_vc2(*stat_ptr)).inhalt.i_val = 0;
  STAT_rel_tok(*stat_ptr) = EQ;
  STAT_spr_ziel(*stat_ptr) = stat_ptr - programm + anz + 3;
  nextstat();
        /*      count := count - 1; */
  ARG_argsort(temp) = INT | CON;
  ARG_con_wert(temp).datentyp = INT | INT_ERLAUBT;
  ARG_con_wert(temp).inhalt.i_val = 1;
  operation(count, count, MINUS, &temp);
  nextstat();
  ARG_argsort(temp) = INT | CON | SIZ;
  ARG_sizedec(temp) = copy_declist(parz_typ(para_typ[BOOL]), 1);
  for (; anz; anz--, ptrs++)
  {     /*      ptrs := ptrs + SIZE ( B1 ); */
    operation(ptrs, ptrs, PLUS, &temp);
    nextstat();
  }
        /*      GOTO loopadr; */
  funcs(do_goto, print_goto);
  STAT_spr_ziel(*stat_ptr) = loopadr;
  nextstat();
        /* l2 : ... */
  STAT_label(*stat_ptr) = ++max_label;
}

/**************************************************************************
 ***                      Funktion para_zuw
 ***
 *** Erzeugt Debuggercode fuer 'erg := var'
 *** t : Index in Debugger-Typtabelle, Typ von 'var'
 ***
 **************************************************************************/

para_zuw(erg, var, t)
SEXPR *erg;
ARG *var;
int t;
{ register ARGART esort = ARG_argsort(erg->serg);
  register ARGART vsort = ARG_argsort(*var);
  register DECLIST *dl = parz_typ(t);
  register int v_s = sign(esort & VEC);

  if ((esort & (ADS | ADV | CON | SIZ)) ||
      (ARG_tiefe(erg->serg) == debug_lev && !(esort & IND)))
  { komerr(texte[315]); return; }
  if (!v_s && (vsort & VEC))
  { komerr(texte[67]); return; }
  if ((dl->typ_zahl[SUMM] > 1 || DECL_art(*dl->darray) != FELD) &&
      AA_typ(vsort) != STR)
  {     /* MOVE var TO erg AS dl; */
    funcs(v_s ? do_v_blockmove : do_s_blockmove, print_blockmove);
    STAT_dblock(*stat_ptr) = copy_declist(dl, 1);
    STAT_vc2(*stat_ptr) = *var;
    STAT_verg(*stat_ptr) = erg->serg;
    nextstat();
  }
  else
  {     /* erg := var; */
    if (types_table[erg->typserg - 1].typ_art == TYP_ARRAY)
    { if (!(vsort & CON))
      { komerr(texte[313]); return; }
      if (AA_typ(vsort) == CHA)
      { register char *s = malloc((size_t)2);

        if (!s) { komerr(texte[80],2); return; }
        *s = ARG_con_wert(*var).inhalt.c_val; s[1] = '\0';
        ARG_argsort(*var) = vsort = STR | CON;
        ARG_con_wert(*var).datentyp = STR;
        ARG_con_wert(*var).inhalt.s_val = s;
      }
    }
    funcs(assign_do_funcs[v_s][AA_typ(esort)][AA_typ(vsort)], print_zuw);
    STAT_vc1(*stat_ptr) = *var;
    STAT_verg(*stat_ptr) = erg->serg;
    nextstat();
  }
}

/**************************************************************************
 ***                      Funktion para_relop
 ***
 *** Erzeugt Debuggercode fuer 'erg := arg1 op arg2'
 *** t1, t2 : Indizes in Debugger-Typtabelle, Typen von arg1 bzw. arg2
 *** op     : Relop-Token
 ***
 **************************************************************************/

para_relop(erg, arg1, t1, op, arg2, t2)
ARG *erg, *arg1, *arg2;
int t1, op, t2;
{ register TYPTAB *tt2 = types_table + t2 - 1;
  register ARGART asort1 = ARG_argsort(*arg1);
  register ARGART asort2 = ARG_argsort(*arg2);
  register ARGART sv1 = asort1 & VEC;
  register ARGART sv2 = asort2 & VEC;
  register ARGART sv = sv1 | sv2;

  new_var(erg, para_typ[BOOL], sv);
  if (op == IN)
  { ARG hilf;
    ARG set_von, temp_bis, intval, nset, in_range;
    register STAT *l2_sprung;

    ARG_argsort(set_von) = ARG_argsort(temp_bis) = INT | CON;
    ARG_con_wert(set_von).datentyp =
      ARG_con_wert(temp_bis).datentyp = INT | INT_ERLAUBT;
    ARG_con_wert(set_von).inhalt.i_val = tt2->typ_arg2;
    ARG_con_wert(temp_bis).inhalt.i_val = tt2->typ_arg3;
    switch (ARG_typ(*arg1))
    { case BOOL :
        new_var(&intval, para_typ[INT], sv1);
            /*  intval := arg1 */
        funcs(sv1 ? do_vib_zuw : do_sib_zuw, print_zuw);
        STAT_verg(*stat_ptr) = intval;
        STAT_vc1(*stat_ptr) = *arg1;
        nextstat();
        break;
      case CHA :
        new_var(&intval, para_typ[INT], sv1);
            /*  intval := arg1 */
        funcs(sv1 ? do_vic_zuw : do_sic_zuw, print_zuw);
        STAT_verg(*stat_ptr) = intval;
        STAT_vc1(*stat_ptr) = *arg1;
        nextstat();
        break;
      case INT :
        intval = *arg1;
        break;
      default : bug("debug.c/para_relop : Falscher typ von 'arg1'");
    }
    new_var(&in_range, para_typ[BOOL], sv1);
    new_var(&nset, para_typ[BOOL], sv1);
        /*      in_range := intval >= set_von; */
    rel_operation(&in_range, &intval, GE, &set_von);
    nextstat();
        /*      nset := intval <= temp_bis; */
    rel_operation(&nset, &intval, LE, &temp_bis);
    nextstat();
        /*      in_range := in_range AND nset; */
    operation(&in_range, &in_range, AND, &nset);
    nextstat();
        /*      erg := FALSE; */
    funcs(sv ? do_vbb_zuw : do_sbb_zuw, print_zuw);
    STAT_verg(*stat_ptr) = *erg;
    ARG_argsort(STAT_vc1(*stat_ptr)) = BOOL | CON;
    ARG_con_wert(STAT_vc1(*stat_ptr)).datentyp = BOOL | BOOL_ERLAUBT;
    ARG_con_wert(STAT_vc1(*stat_ptr)).inhalt.b_val = FALSE;
    nextstat();
        /*      IF in_range CALL l1; */
    funcs(sv1 ? do_v_ifcall : do_s_ifcall, print_ifcall);
    STAT_vc1(*stat_ptr) = in_range;
    STAT_rel_tok(*stat_ptr) = 0;
    STAT_spr_ziel(*stat_ptr) = stat_ptr - programm + 2;
    nextstat();
        /*      GOTO l2; */
    funcs(do_goto, print_goto);
    l2_sprung = stat_ptr;
    nextstat();
        /* l1 : PROC debug_lev+1; */
    funcs(do_proc, print_proc);
    STAT_proclev(*stat_ptr) = debug_lev + 1;
    STAT_dscal(*stat_ptr) = dlptr(NULL);
    STAT_dvec(*stat_ptr) = dlptr(NULL);
    STAT_label(*stat_ptr) = ++max_label;
    nextstat();
    if (sv1 && !sv2)
    { new_var(&nset, t2, sv2 = VEC);
        /*      MOVE arg2 TO nset AS settyp; */
      funcs(do_v_blockmove, print_blockmove);
      STAT_vc2(*stat_ptr) = *arg2;
      STAT_verg(*stat_ptr) = nset;
      STAT_dblock(*stat_ptr) = copy_declist(parz_typ(t2), 1);
      nextstat();
    }
    else
    { nset = *arg2; }
    if (ARG_argsort(nset) & IND)
    { ARG_argsort(nset) = ARG_indsort(nset); }
    else
    { ARG_argsort(nset) = (ARG_argsort(nset) & ~VEC) |
                          ((ARG_argsort(nset) & VEC) ? ADV : ADS);
    }
    new_var(&hilf, para_typ[INT], sv1 | (ARG_argsort(nset) & VEC));
        /*      hilf := intval - set_von; */
    operation(&hilf, &intval, MINUS, &set_von);
    nextstat();
        /*      hilf := hilf * SIZE ( B1 ); */
    ARG_argsort(temp_bis) = INT | CON | SIZ;
    ARG_sizedec(temp_bis) = copy_declist(parz_typ(para_typ[BOOL]), 1);
    operation(&hilf, &hilf, MAL, &temp_bis);
    nextstat();
        /*      hilf := nset + hilf; */
    operation(&hilf, &nset, PLUS, &hilf);
    nextstat();
        /*      erg := (S|V)B[hilf]; */
    ARG_indsort(hilf) = ARG_argsort(hilf);
    ARG_argsort(hilf) = BOOL | IND | sv2;
    funcs(sv2 ? do_vbb_zuw : do_sbb_zuw, print_zuw);
    STAT_verg(*stat_ptr) = *erg;
    STAT_vc1(*stat_ptr) = hilf;
    nextstat();
        /*      RETURN; */
    funcs(do_return, print_return);
    nextstat();
        /* l2 : ... */
    STAT_label(*stat_ptr) = ++max_label;
    STAT_spr_ziel(*l2_sprung) = stat_ptr - programm;
  }
  else
  { register DECLIST *dec = parz_typ(t1);

    if (tt2->typ_art <= TYP_REAL || tt2->typ_art == TYP_POINTER)
    {       /* erg := arg1 op arg2; */
      rel_operation(erg, arg1, op, arg2);
      nextstat();
    }
    else if (tt2->typ_art == TYP_SET && op != EQ && op != NE)
    { ARG ptrs[2];
      ARG *sets[2];
      ARG count, temp, hilf;
      register int le = (op == LE);

      new_var(&temp, para_typ[BOOL], sv);
        /*       erg := TRUE; */
      funcs(sv ? do_vbb_zuw : do_sbb_zuw, print_zuw);
      STAT_verg(*stat_ptr) = *erg;
      ARG_argsort(STAT_vc1(*stat_ptr)) = BOOL | CON;
      ARG_con_wert(STAT_vc1(*stat_ptr)).datentyp = BOOL | BOOL_ERLAUBT;
      ARG_con_wert(STAT_vc1(*stat_ptr)).inhalt.i_val = TRUE;
      nextstat();
      sets[0] = arg1;
      sets[1] = arg2;
      para_setloop_start(tt2, sets, ptrs, &count, 2);
        /* lab : temp := NOT (S|V)B[ptrs[(0|1)]]; */
      funcs(sv ? do_vb_not : do_sb_not, print_not);
      STAT_verg(*stat_ptr) = temp;
      hilf = ptrs[le ? 0 : 1];
      ARG_indsort(hilf) = ARG_argsort(hilf);
      ARG_argsort(hilf) = BOOL | (le ? sv1 : sv2) | IND;
      STAT_vc1(*stat_ptr) = hilf;
      nextstat();
        /*       temp := temp OR (S|V)B[ptrs[(1|0)]]; */
      hilf = ptrs[le ? 1 : 0];
      ARG_indsort(hilf) = ARG_argsort(hilf);
      ARG_argsort(hilf) = BOOL | (le ? sv2 : sv1) | IND;
      operation(&temp, &temp, OR, &hilf);
      nextstat();
        /*       erg := erg AND temp; */
      operation(erg, erg, AND, &temp);
      nextstat();
      para_setloop_end(ptrs, &count, 2, (int)(stat_ptr - programm - 3));
    }
    else
    {       /* EQUAL arg1 arg2 AS dec; */
      funcs(sv ? do_v_blockequal : do_s_blockequal, print_blockequal);
      STAT_dblock(*stat_ptr) = dec;
      STAT_vc2(*stat_ptr) = *arg1;
      STAT_vc3(*stat_ptr) = *arg2;
      nextstat();
            /* erg := [NOT] (V|S)Result; */
      if (op == NE)
      { funcs(sv ? do_vb_not : do_sb_not, print_not); }
      else
      { funcs(sv ? do_vbb_zuw : do_sbb_zuw, print_zuw); }
      STAT_verg(*stat_ptr) = *erg;
      ARG_argsort(STAT_vc1(*stat_ptr)) = BOOL | sv | SPEZ;
      ARG_vartok(STAT_vc1(*stat_ptr)) = sv ? VRESULT : SRESULT;
      nextstat();
    }
  }
}

/**************************************************************************
 ***                      Funktion para_addop
 ***
 *** Erzeugt Debuggercode fuer 'erg := arg1 op arg2'
 *** t : Index in Debugger-Typtabelle, Typ von erg, arg1 und arg2
 *** op : Addop-Token
 ***
 **************************************************************************/

para_addop(erg, arg1, op, arg2, t)
ARG *erg, *arg1, *arg2;
int op, t;
{ register TYPTAB *tt = types_table + t - 1;
  register ARGART asort1 = ARG_argsort(*arg1);
  register ARGART asort2 = ARG_argsort(*arg2);
  register ARGART sv1 = asort1 & VEC;
  register ARGART sv2 = asort2 & VEC;
  register ARGART sv = sv1 | sv2;

  if ( !(tt->typ_art == TYP_SET && op == MINUS) &&
       !(asort1 & IND) && ARG_tiefe(*arg1) == debug_lev &&
       ( sv1 || sv1 == sv2) )
  { *erg = *arg1; }
  else if ( !(asort2 & IND) && ARG_tiefe(*arg2) == debug_lev &&
            ( sv2 || sv1 == sv2) )
  { *erg = *arg2; }
  else
  { new_var(erg, obertyp(t), sv); }
  if (tt->typ_art == TYP_SET)
  { ARG ptrs[3];
    ARG *sets[3];
    ARG count, hilf, hilf1, hilf2;

    sets[0] = erg;
    sets[1] = arg1;
    sets[2] = arg2;
    para_setloop_start(tt, sets, ptrs, &count, 3);
    hilf = ptrs[0];
    ARG_indsort(hilf) = ARG_argsort(hilf);
    ARG_argsort(hilf) = BOOL | sv | IND;
    hilf1 = ptrs[1];
    ARG_indsort(hilf1) = ARG_argsort(hilf1);
    ARG_argsort(hilf1) = BOOL | sv1 | IND;
    hilf2 = ptrs[2];
    ARG_indsort(hilf2) = ARG_argsort(hilf2);
    ARG_argsort(hilf2) = BOOL | sv2 | IND;
    if (op == MINUS)
    {   /* lab : (S|V)B[ptrs[0]] := NOT (S|V)B[ptrs[2]]; */
      funcs(sv2 ? do_vb_not : do_sb_not, print_not);
      STAT_verg(*stat_ptr) = hilf;
      STAT_vc1(*stat_ptr) = hilf2;
      nextstat();
        /*       (S|V)B[ptrs[0]] := (S|V)B[ptrs[1]] AND (S|V)B[ptrs[0]] ; */
      operation(&hilf, &hilf1, AND, &hilf);
      nextstat();
    }
    else
    {   /* lab : (S|V)B[ptrs[0]] := (S|V)B[ptrs[1]] OR (S|V)B[ptrs[2]]; */
      operation(&hilf, &hilf1, OR, &hilf2);
      nextstat();
    }
    para_setloop_end(ptrs, &count, 3, (int)(stat_ptr - programm - (op == MINUS ? 2 : 1)));
  }
  else
  {     /* erg := arg1 op arg2; */
    operation(erg, arg1, op, arg2);
    nextstat();
  }
}

/**************************************************************************
 ***                      Funktion para_mulop
 ***
 *** Erzeugt Debuggercode fuer 'erg := arg1 op arg2'
 *** t : Index in Debugger-Typtabelle, Obertyp von erg, arg1 und arg2
 *** op : Mulop-Token
 ***
 **************************************************************************/

para_mulop(erg, arg1, op, arg2, t)
ARG *erg, *arg1, *arg2;
int op, t;
{ register TYPTAB *tt = types_table + t - 1;
  register ARGART asort1 = ARG_argsort(*arg1);
  register ARGART asort2 = ARG_argsort(*arg2);
  register ARGART sv1 = asort1 & VEC;
  register ARGART sv2 = asort2 & VEC;
  register ARGART sv = sv1 | sv2;

  if ( !(asort1 & IND) && ARG_tiefe(*arg1) == debug_lev &&
       ( sv1 || sv1 == sv2) )
  { *erg = *arg1; }
  else if ( !(asort2 & IND) && ARG_tiefe(*arg2) == debug_lev &&
            ( sv2 || sv1 == sv2) )
  { *erg = *arg2; }
  else
  { new_var(erg, obertyp(t), sv); }
  if (tt->typ_art == TYP_SET)
  { ARG ptrs[3];
    ARG *sets[3];
    ARG count, hilf, hilf1, hilf2;

    sets[0] = erg;
    sets[1] = arg1;
    sets[2] = arg2;
    para_setloop_start(tt, sets, ptrs, &count, 3);
    hilf = ptrs[0];
    ARG_indsort(hilf) = ARG_argsort(hilf);
    ARG_argsort(hilf) = BOOL | sv | IND;
    hilf1 = ptrs[1];
    ARG_indsort(hilf1) = ARG_argsort(hilf1);
    ARG_argsort(hilf1) = BOOL | sv1 | IND;
    hilf2 = ptrs[2];
    ARG_indsort(hilf2) = ARG_argsort(hilf2);
    ARG_argsort(hilf2) = BOOL | sv2 | IND;
    if (op == DURCH)
    {   /* lab : (S|V)B[ptrs[0]] := (S|V)B[ptrs[1]] <> (S|V)B[ptrs[2]]; */
      rel_operation(&hilf, &hilf1, NE, &hilf2);
      nextstat();
    }
    else
    {   /* lab : (S|V)B[ptrs[0]] := (S|V)B[ptrs[1]] AND (S|V)B[ptrs[2]]; */
      operation(&hilf, &hilf1, AND, &hilf2);
      nextstat();
    }
    para_setloop_end(ptrs, &count, 3, (int)(stat_ptr - programm - 1));
  }
  else
  {     /* erg := arg1 op arg2; */
    operation(erg, arg1, op == DIV ? DURCH : op, arg2);
    nextstat();
  }
}

/**************************************************************************
 ***                      Funktion para_pow
 ***
 *** Erzeugt Debuggercode fuer 'erg := arg1 ^ arg2'
 *** t1, t2 : Indizes in Debugger-Typtabelle, Typen von arg1 bzw. arg2
 ***
 **************************************************************************/

para_pow(erg, arg1, t1, arg2, t2)
ARG *erg, *arg1, *arg2;
int t1, t2;
{ register TYP pt1 = first_typ(parz_typ(t1));
  register TYP pt2 = first_typ(parz_typ(t2));
  register ARGART asort1 = ARG_argsort(*arg1);
  register ARGART asort2 = ARG_argsort(*arg2);
  register ARGART sv1 = asort1 & VEC;
  register ARGART sv2 = asort2 & VEC;

  if ( !(asort1 & IND) && ARG_tiefe(*arg1) == debug_lev &&
       ( sv1 || sv1 == sv2) )
  { *erg = *arg1; }
  else if ( !(asort2 & IND) && ARG_tiefe(*arg2) == debug_lev && pt1 == pt2 &&
            ( sv2 || sv1 == sv2) )
  { *erg = *arg2; }
  else
  { new_var(erg, para_typ[pt1], sv1 | sv2); }
  if (pt1 == INT)
  { funcs(sv1 | sv2 ? do_vii_pow : do_sii_pow, print_pow); }
  else if (pt2 == INT)
  { funcs(sv1 | sv2 ? do_vri_pow : do_sri_pow, print_pow); }
  else
  { funcs(sv1 | sv2 ? do_vrr_pow : do_srr_pow, print_pow); }
  STAT_verg(*stat_ptr) = *erg;
  STAT_vc1(*stat_ptr) = *arg1;
  STAT_vc2(*stat_ptr) = *arg2;
  nextstat();
}

/**************************************************************************
 ***                      Funktion para_minus
 ***
 *** Erzeugt Debuggercode fuer 'erg := - var'
 *** typ : Index in Debugger-Typtabelle, Typ von erg und var
 ***
 **************************************************************************/

para_minus(erg, var, typ)
ARG *erg, *var;
int typ;
{ register TYP pztyp = first_typ(parz_typ(typ));
  register ARGART asort = ARG_argsort(*var);
  register ARGART sv = asort & VEC;

  if (ARG_tiefe(*var) == debug_lev && !(asort & IND))
  { *erg = *var; }
  else
  { new_var(erg, typ, sv); }
        /* erg := - arg; */
  funcs(op_funcs[UMINUS-PLUS].do_f[sv > 0][pztyp], print_minus);
  STAT_verg(*stat_ptr) = *erg;
  STAT_vc1(*stat_ptr) = *var;
  nextstat();
}

/**************************************************************************
 ***                      Funktion para_not
 ***
 *** Erzeugt Debuggercode fuer 'erg := NOT var'
 ***
 **************************************************************************/

para_not(erg, var)
ARG *erg, *var;
{ register ARGART asort = ARG_argsort(*var);
  register ARGART sv = asort & VEC;

  if (ARG_tiefe(*var) == debug_lev && !(asort & IND))
  { *erg = *var; }
  else
  { new_var(erg, para_typ[BOOL], sv); }
        /* erg := NOT arg; */
  funcs(sv ? do_vb_not : do_sb_not, print_not);
  STAT_verg(*stat_ptr) = *erg;
  STAT_vc1(*stat_ptr) = *var;
  nextstat();
}

/**************************************************************************
 ***                      Funktion para_reduce
 ***
 *** Erzeugt Debuggercode fuer 'erg := reduce . op ( var )'
 *** t : Index in Debugger-Typtabelle, Typ von erg und var
 *** op : Reduce Operator Token
 ***
 **************************************************************************/

para_reduce(erg, op, var, t)
ARG *erg, *var;
int op, t;
{ register TYP atyp = first_typ(parz_typ(t));

  new_var(erg, para_typ[atyp], 0);
        /* erg := REDUCE op OF var; */
  funcs(red_do_funcs[atyp][op - AND], print_op_reduce);
  STAT_verg(*stat_ptr) = *erg;
  STAT_vc1(*stat_ptr) = *var;
  STAT_red_fct(*stat_ptr) = op;
  nextstat();
}

/**************************************************************************
 ***                      Funktion para_stdfkt
 ***
 *** Prueft ob 'key' der Woerterbuchschluessel einer Standardfunktion oder 
 *** eines ARRAY oder RECORD-Typs ist.
 *** Argumente bzw. Komponenten muessen zu 'arglist' passen.
 *** Setzt 'typserg' und 'sexpr_art' in '*erg', und meldet evtl. Fehler.
 *** Erzeugt Code fuer 'erg := key(arglist)'
 ***
 **************************************************************************/

int param_count[F_GETPIXEL - F_ABS + 1] =   /* Stelligkeiten der Standardfunktionen */
{ 1, /* F_ABS */
  1, /* F_CAP */
  1, /* F_CHR */
  1, /* F_EVEN */
  1, /* F_FLOAT */
  1, /* F_MAX */
  1, /* F_MIN */
  1, /* F_ODD */
  1, /* F_ORD */
  1, /* F_SIZE */
  1, /* F_TRUNC */
  2, /* F_VAL */
  2, /* F_STRCMP */
  2, /* F_STREQ */
  0, /* F_SIRANDOM */
  0, /* F_SRRANDOM */
  0, /* F_SCRANDOM */
  0, /* F_SBRANDOM */
  0, /* F_VIRANDOM */
  0, /* F_VRRANDOM */
  0, /* F_VCRANDOM */
  0, /* F_VBRANDOM */
  1, /* F_IN_CONNECTED */
  1, /* F_OUT_CONNECTED */
  2, /* F_IN_LINECONNECTED */
  2, /* F_OUT_LINECONNECTED */
  1, /* F_ARCCOS */
  1, /* F_ARCSIN */
  1, /* F_ARCTAN */
  2, /* F_ARCTAN2 */
  1, /* F_COS */
  1, /* F_EXP */
  1, /* F_LN */
  1, /* F_SIN */
  1, /* F_SQRT */
  1, /* F_TAN */
  2  /* F_GETPIXEL */
};

para_stdfkt(erg, key, arglist)
SEXPR *erg;
long key;
EXPRLIST *arglist;
{ register SYMTAB *st = look_sym(key, akt_scope);
  register char *name = key_to_name(key);
  register int falsch = FALSE;

  if (st)
  { register char *elstr = exprlist_string(arglist);

    if (elstr && (erg->sexpr_str = malloc((size_t)(strlen(name) + strlen(elstr) + 3))))
    { sprintf(erg->sexpr_str, "%s(%s)", name, elstr);
      free(elstr);
      switch (st->sym_art)
      { case SY_TYPENAME :
          { register TYPTAB *tt = types_table + st->sym_type_scope - 1;

            switch (tt->typ_art)
            { case TYP_ARRAY :
                fprintf(kommandout, texte[343]);
                falsch = TRUE;
                break;
              case TYP_RECORD :
                fprintf(kommandout, texte[344]);
                falsch = TRUE;
                break;
              default :
                komerr(texte[342]);
                falsch = TRUE;
            }
          }
          break;
        case SY_IOPORTNAME :
        case SY_INPORTNAME :
        case SY_OUTPORTNAME :
          { register TYPTAB *tt = types_table + st->sym_type_scope - 1;

            if (tt->typ_art == TYP_PORTTYPE)
            { if (arglist->expr_count = 1)
              { register EXPR *exp = arglist->expr_array;

                if (!exp->typerg)
                { exp->typerg = exp->typlast; exp->erg = exp->lastval; }
                if (exp->expr_art)
                { if (exp->expr_art == EXPR_VAR &&
                      zuw_ok(exp->typerg, para_typ[INT]) &&
                      !(ARG_argsort(exp->erg) & VEC))
                  { ARG arg1, arg2;

                    erg->sexpr_art = st->sym_art - SY_IOPORTNAME + EXPR_IO_PORT;
                    erg->typserg = st->sym_type_scope;
                        /* erg := port(0) + exp; */
                    ARG_argsort(arg1) = INT | CON;
                    ARG_con_wert(arg1).datentyp = INT | INT_ERLAUBT;
                    ARG_con_wert(arg1).inhalt.i_val = st->sym_spez - tt->typ_arg2;
                    new_var(&erg->serg, para_typ[INT], 0);
                    operation(&erg->serg, &arg1, PLUS, &exp->erg);
                    nextstat();
                  }
                  else
                  { komerr(texte[350], name); falsch = TRUE; }
                }
              }
              else
              { komerr(texte[349], name); falsch = TRUE; }
            }
            else
            { komerr(texte[348], name); falsch = TRUE; }
          }
          break;
        case SY_RED_STDF :
        case SY_STDFKT   :
          { register functok = 0;
            register int ec;
            register EXPR *exp, *exp2;
            register ARGART sv1, sv2;
            register f_id = st->sym_type_scope;

            switch(f_id)
            { case MAX : f_id = F_MAX;
                         break;
              case MIN : f_id = F_MIN;
            }
            if (arglist->expr_count != param_count[f_id - F_ABS])
            { komerr(texte[345], name);
              falsch = TRUE;
              break;
            }
            for (ec = arglist->expr_count, exp = arglist->expr_array;
                 ec;
                 ec--, exp++)
            { if (!exp->typerg)
              { exp->typerg = exp->typlast; exp->erg = exp->lastval; }
            }
            if ((ec = arglist->expr_count) >= 1)
            { sv1 = ARG_argsort(arglist->expr_array->erg) & VEC;
              exp = arglist->expr_array;
            }
            if (ec >= 2)
            { sv2 = ARG_argsort(arglist->expr_array[1].erg) & VEC;
              exp2 = arglist->expr_array + 1;
            }
            switch (f_id)
            { case F_ARCTAN : functok++;
              case F_ARCCOS : functok++;
              case F_ARCSIN : functok++;
              case F_TAN    : functok++;
              case F_COS    : functok++;
              case F_SIN    : functok++;
              case F_LN     : functok++;
              case F_EXP    : functok++;
              case F_SQRT   : functok += SQRT;
                if (exp->expr_art == EXPR_VAR &&
                    zuw_ok(exp->typerg, para_typ[REAL]))
                { new_var(&erg->serg, erg->typserg = para_typ[REAL], sv1);
                        /* erg := func exp; */
                  funcs(op_funcs[functok-PLUS].do_f[sv1 == VEC][REAL],
                        op_funcs[functok-PLUS].print_f);
                  STAT_vc1(*stat_ptr) = exp->erg;
                  STAT_verg(*stat_ptr) = erg->serg;
                  nextstat();
                }
                else
                { if (exp->expr_art) komerr(texte[346], name);
                  falsch = TRUE;
                }
                break;
              case F_ABS    :
                if ( exp->expr_art == EXPR_VAR &&
                     ( zuw_ok(exp->typerg, para_typ[REAL]) ||
                       zuw_ok(exp->typerg, para_typ[INT])))
                { register TYP ptyp = first_typ(parz_typ(exp->typerg));

                  new_var(&erg->serg, erg->typserg = para_typ[ptyp], sv1);
                        /* erg := ABS exp; */
                  funcs(op_funcs[ABS-PLUS].do_f[sv1 == VEC][ptyp],
                        op_funcs[ABS-PLUS].print_f);
                  STAT_vc1(*stat_ptr) = exp->erg;
                  STAT_verg(*stat_ptr) = erg->serg;
                  nextstat();
                }
                else
                { if (exp->expr_art) komerr(texte[346], name);
                  falsch = TRUE;
                }
                break;
              case F_ARCTAN2 :
                if (exp->expr_art == EXPR_VAR &&
                    zuw_ok(exp->typerg, para_typ[REAL]) && 
                    exp2->expr_art == EXPR_VAR &&
                    zuw_ok(exp2->typerg, para_typ[REAL]))
                { new_var(&erg->serg, erg->typserg = para_typ[REAL], sv1 | sv2);
                        /* erg := ARCTANT exp exp2; */
                  funcs(op_funcs[ARCTANT-PLUS].do_f[(sv1 | sv2) == VEC][REAL],
                        op_funcs[ARCTANT-PLUS].print_f);
                  STAT_vc1(*stat_ptr) = exp->erg;
                  STAT_vc2(*stat_ptr) = exp2->erg;
                  STAT_verg(*stat_ptr) = erg->serg;
                  nextstat();
                }
                else
                { if (exp->expr_art || exp2->expr_art) komerr(texte[346], name);
                  falsch = TRUE;
                }
                break;
              case F_CAP :
                { ARG hilf1, hilf2, hilf3, temp;

                  if (exp->expr_art == EXPR_VAR &&
                      zuw_ok(exp->typerg, para_typ[CHA]))
                  { new_var(&erg->serg, erg->typserg = para_typ[CHA], sv1);
                    new_var(&hilf1, para_typ[BOOL], sv1);
                        /* hilf1 := exp >= 'a'; */
                    ARG_argsort(temp) = CHA | CON;
                    ARG_con_wert(temp).datentyp = CHA | CHA_ERLAUBT;
                    ARG_con_wert(temp).inhalt.c_val = 'a';
                    rel_operation(&hilf1, &exp->erg, GE, &temp);
                    nextstat();
                    new_var(&hilf2, para_typ[BOOL], sv1);
                        /* hilf2 := exp <= 'z'; */
                    ARG_argsort(temp) = CHA | CON;
                    ARG_con_wert(temp).datentyp = CHA | CHA_ERLAUBT;
                    ARG_con_wert(temp).inhalt.c_val = 'z';
                    rel_operation(&hilf2, &exp->erg, LE, &temp);
                    nextstat();
                        /* hilf1 := hilf1 AND hilf2; */
                    operation(&hilf1, &hilf1, AND, &hilf2);
                    nextstat();
                    new_var(&hilf2, para_typ[INT], sv1);
                        /* hilf2 := hilf1; */
                    funcs(sv1 ? do_vib_zuw : do_sib_zuw, print_zuw);
                    STAT_verg(*stat_ptr) = hilf2;
                    STAT_vc1(*stat_ptr) = hilf1;
                    nextstat();
                        /* hilf2 := hilf2 * ('a'-'A'); */
                    ARG_argsort(temp) = INT | CON;
                    ARG_con_wert(temp).datentyp = INT | INT_ERLAUBT;
                    ARG_con_wert(temp).inhalt.i_val = 'a' - 'A';
                    operation(&hilf2, &hilf2, MAL, &temp);
                    nextstat();
                    new_var(&hilf3, para_typ[INT], sv1);
                        /* hilf3 := exp; */
                    funcs(sv1 ? do_vic_zuw : do_sic_zuw, print_zuw);
                    STAT_verg(*stat_ptr) = hilf3;
                    STAT_vc1(*stat_ptr) = exp->erg;
                    nextstat();
                        /* hilf2 := hilf3 - hilf2; */
                    operation(&hilf2, &hilf3, MINUS, &hilf2);
                    nextstat();
                        /* erg := hilf2; */
                    funcs(sv1 ? do_vci_zuw : do_sci_zuw, print_zuw);
                    STAT_verg(*stat_ptr) = erg->serg;
                    STAT_vc1(*stat_ptr) = hilf2;
                    nextstat();
                  }
                  else
                  { if (exp->expr_art) komerr(texte[346], name);
                    falsch = TRUE;
                  }
                }
                break;
              case F_EVEN :
                { ARG hilf, temp;

                  if (exp->expr_art == EXPR_VAR &&
                      zuw_ok(exp->typerg, para_typ[INT]))
                  { new_var(&erg->serg, erg->typserg = para_typ[BOOL], sv1);
                    new_var(&hilf, para_typ[INT], sv1);
                          /* hilf := exp MOD 2; */
                    ARG_argsort(temp) = INT | CON;
                    ARG_con_wert(temp).datentyp = INT | INT_ERLAUBT;
                    ARG_con_wert(temp).inhalt.i_val = 2;
                    operation(&hilf, &exp->erg, MOD, &temp);
                    nextstat();
                          /* erg := hilf = 0; */
                    ARG_argsort(temp) = INT | CON;
                    ARG_con_wert(temp).datentyp = INT | INT_ERLAUBT;
                    ARG_con_wert(temp).inhalt.i_val = 0;
                    rel_operation(&erg->serg, &hilf, EQ, &temp);
                    nextstat();
                  }
                  else
                  { if (exp->expr_art) komerr(texte[346], name);
                    falsch = TRUE;
                  }
                }
                break;
              case F_ODD :
                { ARG hilf, temp;

                  if (exp->expr_art == EXPR_VAR &&
                      zuw_ok(exp->typerg, para_typ[INT]))
                  { new_var(&erg->serg, erg->typserg = para_typ[BOOL], sv1);
                    new_var(&hilf, para_typ[INT], sv1);
                          /* hilf := exp MOD 2; */
                    ARG_argsort(temp) = INT | CON;
                    ARG_con_wert(temp).datentyp = INT | INT_ERLAUBT;
                    ARG_con_wert(temp).inhalt.i_val = 2;
                    operation(&hilf, &exp->erg, MOD, &temp);
                    nextstat();
                          /* erg := hilf <> 0; */
                    ARG_argsort(temp) = INT | CON;
                    ARG_con_wert(temp).datentyp = INT | INT_ERLAUBT;
                    ARG_con_wert(temp).inhalt.i_val = 0;
                    rel_operation(&erg->serg, &hilf, NE, &temp);
                    nextstat();
                  }
                  else
                  { if (exp->expr_art) komerr(texte[346], name);
                    falsch = TRUE;
                  }
                }
                break;
              case F_MAX :
                fprintf(kommandout, texte[351], name);
                falsch = TRUE;
                break;
              case F_MIN :
                fprintf(kommandout, texte[351], name);
                falsch = TRUE;
                break;
              case F_SIZE :
                if (exp->expr_art == EXPR_VAR)
                { ARG_argsort(erg->serg) = INT | CON | SIZ;
                  ARG_sizedec(erg->serg) =
                    copy_declist(parz_typ(exp->typerg), 1);
                  erg->typserg = para_typ[INT];
                }
                else
                { if (exp->expr_art) komerr(texte[346], name);
                  falsch = TRUE;
                }
                break;
              case F_TRUNC :
                if (exp->expr_art == EXPR_VAR &&
                    zuw_ok(exp->typerg, para_typ[REAL]))
                { new_var(&erg->serg, erg->typserg = para_typ[INT], sv1);
                        /* erg := exp; */
                  funcs(sv1 ? do_vir_zuw : do_sir_zuw, print_zuw);
                  STAT_vc1(*stat_ptr) = exp->erg;
                  STAT_verg(*stat_ptr) = erg->serg;
                  nextstat();
                }
                else
                { if (exp->expr_art) komerr(texte[346], name);
                  falsch = TRUE;
                }
                break;
              case F_FLOAT :
                if (exp->expr_art == EXPR_VAR &&
                    zuw_ok(exp->typerg, para_typ[INT]))
                { new_var(&erg->serg, erg->typserg = para_typ[REAL], sv1);
                        /* erg := exp; */
                  funcs(sv1 ? do_vri_zuw : do_sri_zuw, print_zuw);
                  STAT_vc1(*stat_ptr) = exp->erg;
                  STAT_verg(*stat_ptr) = erg->serg;
                  nextstat();
                }
                else
                { if (exp->expr_art) komerr(texte[346], name);
                  falsch = TRUE;
                }
                break;
              case F_CHR :
                if (exp->expr_art == EXPR_VAR &&
                    zuw_ok(exp->typerg, para_typ[INT]))
                { new_var(&erg->serg, erg->typserg = para_typ[CHA], sv1);
                        /* erg := exp; */
                  funcs(sv1 ? do_vci_zuw : do_sci_zuw, print_zuw);
                  STAT_vc1(*stat_ptr) = exp->erg;
                  STAT_verg(*stat_ptr) = erg->serg;
                  nextstat();
                }
                else
                { if (exp->expr_art) komerr(texte[346], name);
                  falsch = TRUE;
                }
                break;
              case F_ORD :
                if (exp->expr_art == EXPR_VAR)
                { erg->typserg = para_typ[INT];
                  if (zuw_ok(exp->typerg, para_typ[INT]))
                  { erg->serg = exp->erg;
                  }
                  else if (zuw_ok(exp->typerg, para_typ[BOOL]))
                  { new_var(&erg->serg, para_typ[INT], sv1);
                        /* erg := exp; */
                    funcs(sv1 ? do_vib_zuw : do_sib_zuw, print_zuw);
                    STAT_vc1(*stat_ptr) = exp->erg;
                    STAT_verg(*stat_ptr) = erg->serg;
                    nextstat();
                  }
                  else if (zuw_ok(exp->typerg, para_typ[CHA]))
                  { new_var(&erg->serg, para_typ[INT], sv1);
                        /* erg := exp; */
                    funcs(sv1 ? do_vic_zuw : do_sic_zuw, print_zuw);
                    STAT_vc1(*stat_ptr) = exp->erg;
                    STAT_verg(*stat_ptr) = erg->serg;
                    nextstat();
                  }
                  else
                  { komerr(texte[346], name); falsch = TRUE; }
                }
                else
                { if (exp->expr_art) komerr(texte[346], name);
                  falsch = TRUE;
                }
                break;
              case F_VAL :
                fprintf(kommandout, texte[351], name);
                falsch = TRUE;
                break;
              case F_STRCMP :
                fprintf(kommandout, texte[351], name);
                falsch = TRUE;
                break;
              case F_STREQ :
                fprintf(kommandout, texte[351], name);
                falsch = TRUE;
                break;
              case F_SIRANDOM :
                new_var(&erg->serg, erg->typserg = para_typ[INT], 0);
                        /* erg := RANDOM; */
                funcs(do_si_random, print_random);
                STAT_verg(*stat_ptr) = erg->serg;
                nextstat();
                break;
              case F_SRRANDOM :
                new_var(&erg->serg, erg->typserg = para_typ[REAL], 0);
                        /* erg := RANDOM; */
                funcs(do_sr_random, print_random);
                STAT_verg(*stat_ptr) = erg->serg;
                nextstat();
                break;
              case F_SCRANDOM :
                new_var(&erg->serg, erg->typserg = para_typ[CHA], 0);
                        /* erg := RANDOM; */
                funcs(do_sc_random, print_random);
                STAT_verg(*stat_ptr) = erg->serg;
                nextstat();
                break;
              case F_SBRANDOM :
                new_var(&erg->serg, erg->typserg = para_typ[BOOL], 0);
                        /* erg := RANDOM; */
                funcs(do_sb_random, print_random);
                STAT_verg(*stat_ptr) = erg->serg;
                nextstat();
                break;
              case F_VIRANDOM :
                new_var(&erg->serg, erg->typserg = para_typ[INT], VEC);
                        /* erg := RANDOM; */
                funcs(do_vi_random, print_random);
                STAT_verg(*stat_ptr) = erg->serg;
                nextstat();
                break;
              case F_VRRANDOM :
                new_var(&erg->serg, erg->typserg = para_typ[REAL], VEC);
                        /* erg := RANDOM; */
                funcs(do_vr_random, print_random);
                STAT_verg(*stat_ptr) = erg->serg;
                nextstat();
                break;
              case F_VCRANDOM :
                new_var(&erg->serg, erg->typserg = para_typ[CHA], VEC);
                        /* erg := RANDOM; */
                funcs(do_vc_random, print_random);
                STAT_verg(*stat_ptr) = erg->serg;
                nextstat();
                break;
              case F_VBRANDOM :
                new_var(&erg->serg, erg->typserg = para_typ[BOOL], VEC);
                        /* erg := RANDOM; */
                funcs(do_vb_random, print_random);
                STAT_verg(*stat_ptr) = erg->serg;
                nextstat();
                break;
              case F_IN_CONNECTED :
                if (exp->expr_art == EXPR_IO_PORT ||
                    exp->expr_art == EXPR_IN_PORT)
                { new_var(&erg->serg, erg->typserg = para_typ[INT], VEC);
                        /* erg := CONNECTED IN exp; */
                  funcs(do_inconnected, print_inconnected);
                  STAT_verg(*stat_ptr) = erg->serg;
                  STAT_cin_port(*stat_ptr) = exp->erg;
                  nextstat();
                }
                else
                { if (exp->expr_art) komerr(texte[346], name);
                  falsch = TRUE;
                }
                break;
              case F_OUT_CONNECTED :
                if (exp->expr_art == EXPR_IO_PORT ||
                    exp->expr_art == EXPR_OUT_PORT)
                { new_var(&erg->serg, erg->typserg = para_typ[INT], VEC);
                        /* erg := CONNECTED OUT exp; */
                  funcs(do_outconnected, print_outconnected);
                  STAT_verg(*stat_ptr) = erg->serg;
                  STAT_out_port(*stat_ptr) = exp->erg;
                  nextstat();
                }
                else
                { if (exp->expr_art) komerr(texte[346], name);
                  falsch = TRUE;
                }
                break;
              case F_IN_LINECONNECTED :
                if ( (exp->expr_art == EXPR_IO_PORT ||
                      exp->expr_art == EXPR_OUT_PORT) &&
                     (exp2->expr_art == EXPR_IO_PORT ||
                      exp2->expr_art == EXPR_IN_PORT) )
                { new_var(&erg->serg, erg->typserg = para_typ[INT], VEC);
                        /* erg := CONNECTED IN exp2 OUT exp; */
                  funcs(do_line_inconnected, print_line_inconnected);
                  STAT_verg(*stat_ptr) = erg->serg;
                  STAT_out_port(*stat_ptr) = exp->erg;
                  STAT_cin_port(*stat_ptr) = exp2->erg;
                  nextstat();
                }
                else
                { if (exp->expr_art || exp2->expr_art) komerr(texte[346], name);
                  falsch = TRUE;
                }
                break;
              case F_OUT_LINECONNECTED :
                if ( (exp->expr_art == EXPR_IO_PORT ||
                      exp->expr_art == EXPR_OUT_PORT) &&
                     (exp2->expr_art == EXPR_IO_PORT ||
                      exp2->expr_art == EXPR_IN_PORT) )
                { new_var(&erg->serg, erg->typserg = para_typ[INT], VEC);
                        /* erg := CONNECTED OUT exp IN exp2; */
                  funcs(do_line_outconnected, print_line_outconnected);
                  STAT_verg(*stat_ptr) = erg->serg;
                  STAT_out_port(*stat_ptr) = exp->erg;
                  STAT_cin_port(*stat_ptr) = exp2->erg;
                  nextstat();
                }
                else
                { if (exp->expr_art || exp2->expr_art) komerr(texte[346], name);
                  falsch = TRUE;
                }
                break;
              case F_GETPIXEL :
                if (exp->expr_art == EXPR_VAR &&
                    zuw_ok(exp->typerg, para_typ[INT]) && 
                    exp2->expr_art == EXPR_VAR &&
                    zuw_ok(exp2->typerg, para_typ[INT]) &&
                    !sv1 && !sv2)
                { new_var(&erg->serg, erg->typserg = color_typ, 0);
                        /* erg := GETPIXEL exp exp2; */
                  funcs(do_getpixel, print_getpixel);
                  STAT_verg(*stat_ptr) = erg->serg;
                  STAT_vc1(*stat_ptr) = exp->erg;
                  STAT_vc2(*stat_ptr) = exp2->erg;
                  nextstat();
                }
                else
                { if (exp->expr_art || exp2->expr_art) komerr(texte[346], name);
                  falsch = TRUE;
                }
                break;
              default :
                bug("debug.c/para_std_fkt : Falsche Standardfunktion");
            }
          }
          break;
        default :
          komerr(texte[341], name);
      }
      if (falsch)
      { free(erg->sexpr_str); }
    }
    else
    { if (elstr)
      { komerr(texte[80],3);
        free(elstr);
      }
      falsch = TRUE;
    }
  }
  else
  { komerr(texte[301], name);
    falsch = TRUE;
  }
  if (falsch)
  { erg->sexpr_art = EXPR_FALSCH;
    erg->sexpr_str = NULL;
  }
  else
  { erg->sexpr_art = EXPR_VAR; }
  exprlist_free(arglist);
}
