/*****************************************************************************
  Project: PARZ - Parallel Intermediate Code Debugger/Interpreter
  ----------------------------------------------------------------------------
  Release      : 1
  Project Part : Debugger
  Filename     : debcode2.c       
  SCCS-Path    : /tmp_mnt/user/sembach/parz/v2/SCCS/s.debcode2.c
  Version      : 1.3 
  Last changed : 9/27/93 at 14:04:42        
  Author       : Frank Sembach
  Creation date: Aug. 92
  ----------------------------------------------------------------------------
  Description  : Funktionen zur Codeerzeugung fuer den
                 symbolischen Debugger von PARZ
******************************************************************************
***      (C) COPYRIGHT University of Stuttgart - All Right Reserved        ***
*****************************************************************************/

static char sccs_id[] = "@(#)debcode2.c	1.3  9/27/93 PARZ - Debugger (Frank Sembach)";


#include "parzdefs.h"
#include "komdefs.h"
#include "y_tab.h"
#include "externs.h"
#include "rundefs.h"
#include "runexts.h"
#include "komexts.h"
#include "funcs.h"
#include "debexts.h"

extern int subscript_error_num;     /* Programmadresse der Arrayindex Fehlermeldung */
extern int pointer_error_num;       /* Programmadresse der NIL-pointer Fehlermeldung */
extern int setrange_error_num;      /* Programmadresse der set range Fehlermeldung */

extern STAT *stat_ptr;       /* erste freie PARZ-Anweisung */

extern int (* assign_do_funcs[2][REAL + 1][STR + 1])();
                             /* Funktionen fuer Zuweisung */

extern DECLIST *dlptr();

#ifdef MAC
extern double myatof();
#define atof myatof
#endif

/**************************************************************************
 ***                      Funktion para_arraykomp
 ***
 *** Erzeugt Code fuer 'var[elist]',
 *** 'var' ist vom ARRAY-Typ 't'
 *** '*elist' beschreibt die Komponentenselektoren
 *** 'erg' wird Beschreibung der referenzierten Arraykomponente
 *** Ergaenzt 'erg->sexpr_str' um die Strings in 'elist'
 *** Wenn Komponente nicht existiert : Fehlermeldung,
 ***                                   'erg->sexpr_art' FALSCH setzen
 ***
 **************************************************************************/

para_arraykomp(erg, var, elist, t)
SEXPR *erg;
ARG *var;
EXPRLIST *elist;
int t;
{ register int ecount, extyp;
  register EXPR *exp;
  register TYPTAB *tt = types_table + t - 1;
  register int erradr = error_num(&subscript_error_num);
  register ARGART ind_sv, asvar, arr_sv;
  register ARG *exarg;
  ARG loopvar;

  for (exp = elist->expr_array,
         ecount = elist->expr_count,
         loopvar = *var;
       ecount;
       exp++, ecount--)
  { if (tt->typ_art == TYP_ARRAY)
    { if (exp->expr_art == EXPR_VAR)
      { extyp = (extyp = exp->typerg) ? (exarg = &exp->erg, extyp)
                                      : (exarg = &exp->lastval, exp->typlast);
        if (zuw_ok((int)tt->typ_arg1, extyp))
        { ind_sv = ARG_argsort(*exarg) & VEC;
          asvar = ARG_argsort(loopvar);
          arr_sv = asvar & VEC;
          if (arr_sv || !ind_sv)
          { register int ityp, etyp;
            register TYPTAB *ti = types_table + (ityp = tt->typ_arg1) - 1;
            register TYPTAB *te = types_table + (etyp = tt->typ_arg2) - 1;
            ARG von, bis, inc, ptr_var, urspr_ptr, exint;
            register TYP parz_ityp = first_typ(parz_typ(ityp));
            register DECLIST *edecl = parz_typ(etyp);

            erg->typserg = etyp;
            ARG_argsort(von) = ARG_argsort(bis) = INT | CON;
            ARG_con_wert(von).datentyp =
              ARG_con_wert(bis).datentyp = INT | ERLAUBT(INT);
            ARG_con_wert(von).inhalt.i_val = ti->typ_arg2;
            ARG_con_wert(bis).inhalt.i_val = ti->typ_arg3;
            switch(parz_ityp)
            { case BOOL :
                new_var(&exint, para_typ[INT], ind_sv);
                        /* exint := exp; */
                funcs(ind_sv ? do_vib_zuw : do_sib_zuw, print_zuw);
                STAT_verg(*stat_ptr) = exint;
                STAT_vc1(*stat_ptr) = *exarg;
                nextstat();
                if (ti->typ_art == TYP_BOOL)
                { ARG_con_wert(bis).inhalt.i_val = 1; }
                break;
              case CHA :
                new_var(&exint, para_typ[INT], ind_sv);
                        /* exint := exp; */
                funcs(ind_sv ? do_vic_zuw : do_sic_zuw, print_zuw);
                STAT_verg(*stat_ptr) = exint;
                STAT_vc1(*stat_ptr) = *exarg;
                nextstat();
                break;
              case INT :
                exint = *exarg;
                break;
            }
                /* IF exp < von CALL erradr; */
            funcs(ind_sv ? do_v2_ifcall : do_s2_ifcall, print_ifcall);
            STAT_rel_tok(*stat_ptr) = LT;
            STAT_vc1(*stat_ptr) = exint;
            STAT_vc2(*stat_ptr) = von;
            STAT_spr_ziel(*stat_ptr) = erradr;
            nextstat();
                /* IF exp > bis CALL erradr; */
            funcs(ind_sv ? do_v2_ifcall : do_s2_ifcall, print_ifcall);
            STAT_rel_tok(*stat_ptr) = GT;
            STAT_vc1(*stat_ptr) = exint;
            STAT_vc2(*stat_ptr) = bis;
            STAT_spr_ziel(*stat_ptr) = erradr;
            nextstat();
            if (asvar & IND)
            { urspr_ptr = loopvar;
              ARG_argsort(urspr_ptr) = ARG_indsort(urspr_ptr);
              if (ARG_tiefe(loopvar) == debug_lev) /* Hilfsvariable */
              { erg->serg = loopvar;
                ptr_var = urspr_ptr;
              }
              else
              { new_var(&ptr_var, para_typ[INT], arr_sv);
                erg->serg = ptr_var;
                ARG_indsort(erg->serg) = ARG_argsort(ptr_var);
              }
            }
            else
            { new_var(&ptr_var, para_typ[INT], arr_sv);
              erg->serg = urspr_ptr = ptr_var;
              ARG_indsort(erg->serg) = ARG_argsort(ptr_var);
                /* urspr_ptr := ADDR loopvar; */
              funcs(arr_sv ? do_vii_zuw : do_sii_zuw, print_zuw);
              STAT_verg(*stat_ptr) = urspr_ptr;
              STAT_vc1(*stat_ptr) = loopvar;
              ARG_argsort(STAT_vc1(*stat_ptr)) = (asvar & ~VEC) | ((arr_sv) ? ADV : ADS);
              nextstat();
            }
            ARG_argsort(erg->serg) = first_typ(edecl) | IND | (arr_sv);
            new_var(&inc, para_typ[INT], ind_sv);
                /* inc := exp - von; */
            funcs(ind_sv ? do_vi_sub : do_si_sub, print_sub);
            STAT_verg(*stat_ptr) = inc;
            STAT_vc1(*stat_ptr) = exint;
            STAT_vc2(*stat_ptr) = von;
            nextstat();
                /* inc := inc * SIZE ( te ); */
            funcs(ind_sv ? do_vi_mul : do_si_mul, print_mul);
            STAT_verg(*stat_ptr) = STAT_vc1(*stat_ptr) = inc;
            ARG_argsort(STAT_vc2(*stat_ptr)) = INT | CON | SIZ;
            ARG_sizedec(STAT_vc2(*stat_ptr)) = copy_declist(edecl, 1);
            nextstat();
                /* ptr_var := urspr_ptr + inc; */
            funcs(arr_sv ? do_vi_add : do_si_add, print_add);
            STAT_verg(*stat_ptr) = ptr_var;
            STAT_vc1(*stat_ptr) = urspr_ptr;
            STAT_vc2(*stat_ptr) = inc;
            nextstat();
            tt = te;
            loopvar = erg->serg;
          }
          else
          { komerr(texte[327]);
            erg->sexpr_art = EXPR_FALSCH;
            return;
          }
        }
        else
        { komerr(texte[326], exp->expr_str);
          erg->sexpr_art = EXPR_FALSCH;
          return;
        }
      }
      else
      { komerr(texte[305], "[]");
        erg->sexpr_art = EXPR_FALSCH;
        return;
      }
    }
    else
    { komerr(texte[324], erg->sexpr_str);
      erg->sexpr_art = EXPR_FALSCH;
      return;
    }
  }
}

/**************************************************************************
 ***                  Funktion find_rec_komp (rekursiv)
 ***
 *** Sucht in der Mitgl.Tabelle ab '*mt' die Recordkomponente
 *** mit WB-Schluessel 'key'.
 *** Haengt die vorausgehenden Deklarationen mit 'decl_append()'
 *** an '*adddecl' an.
 *** Ergebnis : Index in Typtabelle des Komponententyps
 ***
 **************************************************************************/

int find_rec_komp(mt, adddecl, key)
MITGTAB *mt;
DECLIST *adddecl;
long key;
{ register int fertig;

  for (fertig = FALSE; !fertig; fertig = mt->flag_last, mt++)
  { if (mt->mem_key)
    { if (mt->mem_key == key)
      { return (int)mt->typ_wert; }
      else
      { decl_append(adddecl, parz_typ((int)mt->typ_wert)); }
    }
    else                /* UNION-Komponente */
    { DECLIST testdecl;
      register MITGTAB *un_mt =
        member_table + types_table[mt->typ_wert - 1].typ_arg1 - 1;
      register int un_fertig, t;

      for (un_fertig = FALSE; !un_fertig; un_fertig = un_mt->flag_last, un_mt++)
      { null_dlist(&testdecl);
        if (t = find_rec_komp(member_table +
                                types_table[un_mt->typ_wert - 1].typ_arg1 - 1,
                              &testdecl,
                              key))
        { decl_append(adddecl, &testdecl);
          if (testdecl.darray) free(testdecl.darray);
          return t;
        }
        if (testdecl.darray) free(testdecl.darray);
      }
    }
  }
  return 0;
}

/**************************************************************************
 ***                      Funktion para_recmem
 ***
 *** Erzeugt Code fuer 'var.memkey',
 *** 'var' ist vom RECORD-Typ 't'
 *** 'memkey' ist Woerterbuchschluessel des Komponentennamens
 *** 'erg' wird Beschreibung der referenzierten Recordkomponente
 *** Wenn Komponente nicht existiert : 'erg->sexpr_art' FALSCH setzen
 ***
 **************************************************************************/

para_recmem(erg, var, memkey, t)
SEXPR *erg;
ARG *var;
long memkey;
int t;
{ DECLIST adddecl;
  register MITGTAB *mt = member_table + types_table[t - 1].typ_arg1 - 1;

  null_dlist(&adddecl);
  if (erg->typserg = t = find_rec_komp(mt, &adddecl, memkey))
  { register TYP compt = first_typ(parz_typ(t));
    register ARGART asvar = ARG_argsort(*var);
    ARG ptr_var, urspr_ptr;

    erg->sexpr_art = EXPR_VAR;
    if (asvar & IND)
    { urspr_ptr = *var;
      ARG_argsort(urspr_ptr) = ARG_indsort(urspr_ptr);
      if (ARG_tiefe(*var) == debug_lev) /* Hilfsvariable */
      { erg->serg = *var;
        ptr_var = urspr_ptr;
      }
      else
      { new_var(&ptr_var, para_typ[INT], asvar & VEC);
        erg->serg = ptr_var;
        ARG_indsort(erg->serg) = ARG_argsort(ptr_var);
      }
    }
    else
    { new_var(&ptr_var, para_typ[INT], asvar & VEC);
      erg->serg = urspr_ptr = ptr_var;
      ARG_indsort(erg->serg) = ARG_argsort(ptr_var);
                /* urspr_ptr := ADDR var; */
      funcs(asvar & VEC ? do_vii_zuw : do_sii_zuw, print_zuw);
      STAT_verg(*stat_ptr) = urspr_ptr;
      STAT_vc1(*stat_ptr) = *var;
      ARG_argsort(STAT_vc1(*stat_ptr)) = (asvar & ~VEC) | ((asvar & VEC) ? ADV : ADS);
      nextstat();
    }
    ARG_argsort(erg->serg) = compt | IND | (asvar & VEC);
                /* ptr_var := urspr_ptr + SIZE ( adddecl ); */
    funcs(asvar & VEC ? do_vi_add : do_si_add, print_add);
    STAT_verg(*stat_ptr) = ptr_var;
    STAT_vc1(*stat_ptr) = urspr_ptr;
    ARG_argsort(STAT_vc2(*stat_ptr)) = INT | CON | SIZ;
    ARG_sizedec(STAT_vc2(*stat_ptr)) = copy_declist(&adddecl, 1);
    nextstat();
    if (adddecl.darray) free(adddecl.darray);
  }
  else
  { erg->sexpr_art = EXPR_FALSCH; }
}

/**************************************************************************
 ***                      Funktion para_pointer
 ***
 *** Erzeugt Code fuer 'var^', 'var' zeigt auf Variable mit Typ 't'
 *** 'erg' wird Beschreibung der referenzierten Variable
 ***
 **************************************************************************/

para_pointer(erg, var, t)
SEXPR *erg;
ARG *var;
int t;
{ register ARGART asort = ARG_argsort(*var);
  register int erradr = error_num(&pointer_error_num);

        /* IF var = 0 CALL erradr; */
  funcs(asort & VEC ? do_v2_ifcall : do_s2_ifcall, print_ifcall);
  STAT_rel_tok(*stat_ptr) = EQ;
  STAT_vc1(*stat_ptr) = *var;
  ARG_argsort(STAT_vc2(*stat_ptr)) = INT | CON;
  ARG_con_wert(STAT_vc2(*stat_ptr)).datentyp = INT | INT_ERLAUBT;
  ARG_con_wert(STAT_vc2(*stat_ptr)).inhalt.i_val = 0;
  STAT_spr_ziel(*stat_ptr) = erradr;
  nextstat();
  erg->typserg = types_table[t - 1].typ_arg1;
  if (asort & IND)
  { ARG ptr_var;

    if (ARG_tiefe(*var) == debug_lev) /* Hilfsvariable */
    { erg->serg = *var;
      ptr_var = *var;
      asort = ARG_argsort(ptr_var) = ARG_indsort(ptr_var);
    }
    else
    { new_var(&ptr_var, para_typ[INT], asort & VEC);
      erg->serg = ptr_var;
      ARG_indsort(erg->serg) = asort = ARG_argsort(ptr_var);
    }
    ARG_argsort(erg->serg) =
      first_typ(parz_typ(erg->typserg)) | IND | (asort & VEC);
                /* ptr_var := var; */
    funcs(asort & VEC ? do_vii_zuw : do_sii_zuw, print_zuw);
    STAT_verg(*stat_ptr) = ptr_var;
    STAT_vc1(*stat_ptr) = *var;
    nextstat();
  }
  else
  { ARGART salt;

    erg->serg = *var;
    ARG_indsort(erg->serg) = salt = ARG_argsort(erg->serg);
    ARG_argsort(erg->serg) =
      first_typ(parz_typ(erg->typserg)) | IND | (salt  & (VEC | SPEZ));
  }
}

/**************************************************************************
 ***                      Funktion para_inc_ptr
 ***
 *** Erzeugt Code, der den skalaren Zeiger 'ptr_var' um den Typ '*tt'
 *** weiterzaehlt.
 ***
 **************************************************************************/

para_inc_ptr(ptr_var, tt)
ARG *ptr_var;
TYPTAB *tt;
{       /* ptr_var := ptr_var + SIZE(tt); */
  funcs(do_si_add, print_add);
  STAT_vc1(*stat_ptr) = STAT_verg(*stat_ptr) = *ptr_var;
  ARG_argsort(STAT_vc2(*stat_ptr)) = INT | CON | SIZ;
  ARG_sizedec(STAT_vc2(*stat_ptr)) = copy_declist(&tt->parz_decl, 1);
  nextstat();
}

/**************************************************************************
 ***          Funktion para_const_simple (rekursiv bei SUBRANGE)
 ***
 *** Erzeugt Code, der die einfache Variable ab '[ptr_var]' mit Typ '*tt'
 *** mit dem Wert in '*ct' belegt.
 *** Erzeugt Code, der 'ptr_var' weiterzaehlt.
 *** Zaehlt '*ct' weiter.
 ***
 **************************************************************************/

para_const_simple(ptr_var, tt, ct)
ARG *ptr_var;
TYPTAB *tt;
CONSTTAB **ct;
{ if (tt->typ_art == TYP_SUBRANGE)
  { para_const_simple(ptr_var, types_table + tt->typ_arg1 - 1, ct); }
  else
  { TYP t = first_typ(&tt->parz_decl);

    if (ITEM_typ((*ct)->const_wert) == STR)
    { switch (tt->typ_art)
      { case TYP_POINTER :
          if (strcmp((*ct)->const_wert.inhalt.s_val, "NIL") != 0)
          { komerr(texte[320]); return; }
          (*ct)->const_wert.datentyp = INT | INT_ERLAUBT;
          (*ct)->const_wert.inhalt.i_val = 0;
          break;
        case TYP_BOOL :
          (*ct)->const_wert.datentyp = BOOL | BOOL_ERLAUBT;
          (*ct)->const_wert.inhalt.b_val =
            !!atoi((*ct)->const_wert.inhalt.s_val);
          break;
        case TYP_CHA :
          (*ct)->const_wert.datentyp = CHA | CHA_ERLAUBT;
          (*ct)->const_wert.inhalt.c_val =
            (unsigned char)atoi((*ct)->const_wert.inhalt.s_val);
          break;
        case TYP_INT :
        case TYP_ENUM :
          (*ct)->const_wert.datentyp = INT | INT_ERLAUBT;
          (*ct)->const_wert.inhalt.i_val =
            atol((*ct)->const_wert.inhalt.s_val);
          break;
        case TYP_REAL :
          (*ct)->const_wert.datentyp = REAL | REAL_ERLAUBT;
          (*ct)->const_wert.inhalt.r_val =
            atof((*ct)->const_wert.inhalt.s_val);
      }
    }
        /* St[ptr_var] := ct->const_wert; */
    funcs(assign_do_funcs[0][t][t], print_zuw);
    STAT_verg(*stat_ptr) = *ptr_var;
    ARG_indsort(STAT_verg(*stat_ptr)) = ARG_argsort(STAT_verg(*stat_ptr));
    ARG_argsort(STAT_verg(*stat_ptr)) = t | IND;
    ARG_argsort(STAT_vc1(*stat_ptr)) = t | CON | (SPEZ * (tt->typ_art == TYP_POINTER));
    ARG_con_wert(STAT_vc1(*stat_ptr)) = ((*ct)++)->const_wert;
    nextstat();
    para_inc_ptr(ptr_var, tt);
  }
}

/**************************************************************************
 ***                Funktion para_const_string
 ***
 *** Erzeugt Code, der die Stringvariable ab '[ptr_var]' mit Typ '*tt'
 *** mit dem Wert in '*ct' belegt.
 *** Erzeugt Code, der 'ptr_var' weiterzaehlt.
 *** Zaehlt '*ct' weiter.
 ***
 **************************************************************************/

para_const_string(ptr_var, tt, ct)
ARG *ptr_var;
TYPTAB *tt;
CONSTTAB **ct;
{ register char *sneu = malloc((size_t)(strlen((*ct)->const_wert.inhalt.s_val) + 1));

  if (!sneu)
  { komerr(texte[80],4); return; }
  strcpy(sneu, ((*ct)++)->const_wert.inhalt.s_val);
        /* SC[ptr_var] := ct->const_wert; */
  funcs(do_scstr_zuw, print_zuw);
  STAT_verg(*stat_ptr) = *ptr_var;
  ARG_indsort(STAT_verg(*stat_ptr)) = ARG_argsort(STAT_verg(*stat_ptr));
  ARG_argsort(STAT_verg(*stat_ptr)) = CHA | IND;
  ARG_argsort(STAT_vc1(*stat_ptr)) = CON | STR;
  ARG_con_wert(STAT_vc1(*stat_ptr)).datentyp = STR;
  ARG_con_wert(STAT_vc1(*stat_ptr)).inhalt.s_val = sneu;
  nextstat();
  para_inc_ptr(ptr_var, tt);
}

/**************************************************************************
 ***                      Funktion para_const_array
 ***
 *** Erzeugt Code, der die ARRAY-Variable vom Typ '*tt', auf die
 *** '*ptr_var' zeigt mit den Werten belegt, die ab '**ct' in der
 *** Konstantentabelle stehen.
 *** '*ct' wird weitergezaehlt
 ***
 **************************************************************************/

para_const_array(ptr_var, tt, ct)
ARG *ptr_var;
TYPTAB *tt;
CONSTTAB **ct;
{ register TYPTAB *ti = types_table + tt->typ_arg1 - 1;
  register TYPTAB *te = types_table + tt->typ_arg2 - 1;
  register int i;
  register int von = ti->typ_arg2;
  register int bis = ti->typ_arg3;
  register int etyp = te->typ_art;

  switch (etyp)
  { case TYP_CHA :
      if ((*ct)->string_flag)
      { para_const_string(ptr_var, tt, ct); }
      else
      { for (i = von; i <= bis; i++)
        { para_const_simple(ptr_var, te, ct); }
      }
      break;
    case TYP_BOOL :
    case TYP_INT :
    case TYP_REAL :
    case TYP_SUBRANGE :
    case TYP_ENUM :
    case TYP_POINTER :
      for (i = von; i <= bis; i++)
      { para_const_simple(ptr_var, te, ct); }
      break;
    case TYP_SET :
      for (i = von; i <= bis; i++)
      { ARG var;

        var = *ptr_var;
        ARG_indsort(var) = ARG_argsort(var);
        ARG_argsort(var) = BOOL | IND;
        para_const_set(&var, ((*ct)++)->const_wert.inhalt.s_val);
        para_inc_ptr(ptr_var, te);
      }
      break;
    case TYP_ARRAY :
      for (i = von; i <= bis; i++)
      { para_const_array(ptr_var, te, ct); }
      break;
    case TYP_RECORD :
      for (i = von; i <= bis; i++)
      { para_const_record(ptr_var, te, ct); }
      break;
    default : bug("debug.c/para_const_array : Illegaler Elementtyp");
  }
}

/**************************************************************************
 ***                      Funktion para_const_record
 ***
 *** Erzeugt Code, der die RECORD-Variable vom Typ '*tt', auf die
 *** '*ptr_var' zeigt mit den Werten belegt, die ab '**ct' in der
 *** Konstantentabelle stehen.
 *** '*ct' wird weitergezaehlt
 ***
 **************************************************************************/

para_const_record(ptr_var, tt, ct)
ARG *ptr_var;
TYPTAB *tt;
CONSTTAB **ct;
{ register MITGTAB *mt = member_table + tt->typ_arg1 - 1;
  register int fertig;

  for (fertig = FALSE; !fertig; fertig = mt->flag_last, mt++)
  { register TYPTAB *tm = types_table + mt->typ_wert - 1;

    switch (tm->typ_art)
    { case TYP_BOOL :
      case TYP_CHA :
      case TYP_INT :
      case TYP_REAL :
      case TYP_SUBRANGE :
      case TYP_ENUM :
      case TYP_POINTER :
        para_const_simple(ptr_var, tm, ct);
        break;
      case TYP_SET :
        { ARG var;

          var = *ptr_var;
          ARG_indsort(var) = ARG_argsort(var);
          ARG_argsort(var) = BOOL | IND;
          para_const_set(&var, ((*ct)++)->const_wert.inhalt.s_val);
          para_inc_ptr(ptr_var, tm);
        }
        break;
      case TYP_ARRAY :
        para_const_array(ptr_var, tm, ct);
        break;
      case TYP_RECORD :
        para_const_record(ptr_var, tm, ct);
        break;
      default :
        bug("debug.c/para_const_record : Unerlaubter Komponententyp");
    }
  }
}

/**************************************************************************
 ***                      Funktion para_const_set
 ***
 *** Erzeugt Code, der den SET ab Variable '*var' mit 'st' belegt
 ***
 **************************************************************************/

para_const_set(var, st)
ARG *var;
char *st;
{ register char *sneu = malloc((size_t)(strlen(st) + 1));

  if (!sneu)
  { komerr(texte[80],5); return; }
  strcpy(sneu, st);
        /* var := INITSET st; */
  funcs(do_s_initset, print_initset);
  STAT_verg(*stat_ptr) = *var;
  STAT_setbits(*stat_ptr) = sneu;
  nextstat();
}

/**************************************************************************
 ***                      Funktion para_desig_ident
 ***
 *** Erzeugt in '*v' Informationen zum WB-Schluessel 'key'
 ***
 **************************************************************************/

para_desig_ident(v, key)
SEXPR *v;
long key;
{ register SYMTAB *st;
  register TYP t;
  register char *name = key_to_name(key);

  if (sym_debugging)
  { if (disp)
    { if (st = look_sym(key, disp[akt_tiefe]->top_scope))
      { if (v->sexpr_str = malloc((size_t)(strlen(name) + 1)))
        { strcpy(v->sexpr_str, name);
          switch (st->sym_art)
          { case SY_CONSTNAME :
              { register TYPTAB *tt = types_table + st->sym_type_scope - 1;
                CONSTTAB *ct;
                register ARG *var = &v->serg;
                ARG hilf_ptr;

                v->sexpr_art = EXPR_VAR;
                v->typserg = st->sym_type_scope;
                t = first_typ(parz_typ(v->typserg));
                switch (tt->typ_art)
                { case TYP_BOOL :
                  case TYP_CHA :
                  case TYP_INT :
                  case TYP_ENUM :
                  case TYP_SUBRANGE :
                    ARG_argsort(*var) = t | CON;
                    ARG_con_wert(*var).datentyp = t | ERLAUBT(t);
                    switch (t)
                    { case BOOL :
                        ARG_con_wert(*var).inhalt.b_val = !!st->sym_spez;
                        break;
                      case CHA :
                        ARG_con_wert(*var).inhalt.c_val =
                                             (unsigned char)st->sym_spez;
                        break;
                      case INT :
                        ARG_con_wert(*var).inhalt.i_val = st->sym_spez;
                    }
                    break;
                  case TYP_REAL :
                    ARG_argsort(*var) = REAL | CON;
                    ct = constant_table + st->sym_spez - 1;
                    if (ITEM_typ(ct->const_wert) != REAL)
                    { float f = atof(ct->const_wert.inhalt.s_val);
                      free(ct->const_wert.inhalt.s_val);
                      ct->const_wert.datentyp = REAL | REAL_ERLAUBT;
                      ct->const_wert.inhalt.r_val = f;
                    }
                    ARG_con_wert(*var) = ct->const_wert;
                    break;
                  case TYP_STRING :
                    ARG_argsort(*var) = STR | CON;
                    ARG_con_wert(*var) =
                      constant_table[st->sym_spez - 1].const_wert;
                    { register char *sneu =
                        malloc((size_t)(strlen(ARG_con_wert(*var).inhalt.s_val) + 1));

                      if (!sneu)
                      { komerr(texte[80],6); return; }
                      strcpy(sneu, ARG_con_wert(*var).inhalt.s_val);
                      ARG_con_wert(*var).inhalt.s_val = sneu;
                    }
                    break;
                  case TYP_ARRAY :
                    new_var(var, st->sym_type_scope, 0);
                    ct = constant_table + st->sym_spez - 1;
                    new_var(&hilf_ptr, para_typ[INT], 0);
                        /* hilf_ptr := ADDR erg; */
                    funcs(do_sii_zuw, print_zuw);
                    STAT_vc1(*stat_ptr) = *var;
                    ARG_argsort(STAT_vc1(*stat_ptr)) |= ADS;
                    STAT_verg(*stat_ptr) = hilf_ptr;
                    nextstat();
                    para_const_array(&hilf_ptr, tt, &ct);
                    break;
                  case TYP_RECORD :
                    new_var(var, st->sym_type_scope, 0);
                    ct = constant_table + st->sym_spez - 1;
                    new_var(&hilf_ptr, para_typ[INT], 0);
                        /* hilf_ptr := ADDR erg; */
                    funcs(do_sii_zuw, print_zuw);
                    STAT_vc1(*stat_ptr) = *var;
                    ARG_argsort(STAT_vc1(*stat_ptr)) |= ADS;
                    STAT_verg(*stat_ptr) = hilf_ptr;
                    nextstat();
                    para_const_record(&hilf_ptr, tt, &ct);
                    break;
                  case TYP_SET :
                    new_var(var, st->sym_type_scope, 0);
                    ct = constant_table + st->sym_spez - 1;
                    para_const_set(var, ct->const_wert.inhalt.s_val);
                    break;
                  default : bug("debug.c/para_desig_ident : Falscher Konstantentyp");
                }
              }
              break;
            case SY_TYPENAME :
              v->sexpr_art = EXPR_TYP;
              v->typserg = st->sym_type_scope;
              break;
            case SY_SCALNAME :
              v->sexpr_art = EXPR_VAR;
              v->typserg = st->sym_type_scope;
              ARG_argsort(v->serg) = t = first_typ(parz_typ(v->typserg));
              ARG_tiefe(v->serg) = st->sym_lev;
              ARG_num(v->serg) = st->sym_spez;
              ARG_gut_proc(v->serg) = -2;
              if (st->sym_lev == 0)
              { if (!(set_absoffs(t, (int)st->sym_spez, &v->serg, &glob_s_decl, -1)))
                { bug("debug.c/para_desig_ident : Falsche Skalarvariable"); }
              }
              if (st->ind_flag)
              { ARG_argsort(v->serg) |= IND;
                ARG_indsort(v->serg) = INT;
              }
              break;
            case SY_VECNAME :
              v->sexpr_art = EXPR_VAR;
              v->typserg = st->sym_type_scope;
              ARG_argsort(v->serg) = (t = first_typ(parz_typ(v->typserg))) | VEC;
              ARG_tiefe(v->serg) = st->sym_lev;
              ARG_num(v->serg) = st->sym_spez;
              ARG_gut_proc(v->serg) = -2;
              if (st->sym_lev == 0)
              { if (!(set_absoffs(t, (int)st->sym_spez, &v->serg, &glob_v_decl, -1)))
                { bug("debug.c/para_desig_ident : Falsche Vektorvariable,t = %d, sym_spez = %ld", t, st->sym_spez); }
              }
              if (st->ind_flag)
              { ARG_argsort(v->serg) |= IND;
                ARG_indsort(v->serg) = INT | VEC;
              }
              break;
            case SY_IOPORTNAME :
              { register TYPTAB *tt = types_table + st->sym_type_scope - 1;

                if (tt->typ_art == TYP_PORTTYPE)
                { komerr(texte[347], name);
                  v->sexpr_art = EXPR_FALSCH;
                  free(v->sexpr_str); v->sexpr_str = NULL;
                }
                else
                { v->sexpr_art = EXPR_IO_PORT;
                  v->typserg = st->sym_type_scope;
                  ARG_argsort(v->serg) = INT | CON;
                  ARG_con_wert(v->serg).datentyp = INT | INT_ERLAUBT;
                  ARG_con_wert(v->serg).inhalt.i_val = st->sym_spez;
                }
              }
              break;
            case SY_INPORTNAME :
              { register TYPTAB *tt = types_table + st->sym_type_scope - 1;

                if (tt->typ_art == TYP_PORTTYPE)
                { komerr(texte[347], name);
                  v->sexpr_art = EXPR_FALSCH;
                  free(v->sexpr_str); v->sexpr_str = NULL;
                }
                else
                { v->sexpr_art = EXPR_IN_PORT;
                  v->typserg = st->sym_type_scope;
                  ARG_argsort(v->serg) = INT | CON;
                  ARG_con_wert(v->serg).datentyp = INT | INT_ERLAUBT;
                  ARG_con_wert(v->serg).inhalt.i_val = st->sym_spez;
                }
              }
              break;
            case SY_OUTPORTNAME :
              { register TYPTAB *tt = types_table + st->sym_type_scope - 1;

                if (tt->typ_art == TYP_PORTTYPE)
                { komerr(texte[347], name);
                  v->sexpr_art = EXPR_FALSCH;
                  free(v->sexpr_str); v->sexpr_str = NULL;
                }
                else
                { v->sexpr_art = EXPR_OUT_PORT;
                  v->typserg = st->sym_type_scope;
                  ARG_argsort(v->serg) = INT | CON;
                  ARG_con_wert(v->serg).datentyp = INT | INT_ERLAUBT;
                  ARG_con_wert(v->serg).inhalt.i_val = st->sym_spez;
                }
              }
              break;
            case SY_SPEZ :
              v->sexpr_art = EXPR_VAR;
              switch (st->sym_type_scope)
              { case ID :
                  ARG_argsort(v->serg) = SPEZ | VEC | INT;
                  ARG_vartok(v->serg) = ID;
                  v->typserg = para_typ[INT];
                  break;
                case TERMCH :
                  ARG_argsort(v->serg) = SPEZ | CHA;
                  ARG_vartok(v->serg) = TERMCH;
                  v->typserg = para_typ[CHA];
                  break;
                case DONE :
                  ARG_argsort(v->serg) = SPEZ | BOOL;
                  ARG_vartok(v->serg) = DONE;
                  v->typserg = para_typ[BOOL];
                  break;
                case NIL:
                  ARG_argsort(v->serg) = CON | INT | SPEZ;
                  ARG_con_wert(v->serg).datentyp = INT | INT_ERLAUBT;
                  ARG_con_wert(v->serg).inhalt.i_val = 0;
                  v->typserg = nil_typ;
                  break;
                default :
                  bug("debug.c/para_desig_ident : Falsches SY_SPEZ-Token %d",
                      st->sym_type_scope);
              }
              parz_typ(v->typserg);
              break;
            default :
              komerr(texte[308], name);
              v->sexpr_art = EXPR_FALSCH;
              free(v->sexpr_str); v->sexpr_str = NULL;
              break;
          }
        }
        else
        { komerr(texte[80],7);
          v->sexpr_art = EXPR_FALSCH;
        }
      }
      else
      { komerr(texte[301], name);
        v->sexpr_art = EXPR_FALSCH;
        v->sexpr_str = NULL;
      }
    }
    else
    { komerr(texte[183]);
      v->sexpr_art = EXPR_FALSCH;
      v->sexpr_str = NULL;
    }
  }
}

/**************************************************************************
 ***                      Funktion para_set_clear
 ***
 *** Erzeugt Code, der eine Set-Hilfsvariable vom Typ 't' anlegt und loescht.
 *** '*var' : Wird die Hilfsvariable
 ***
 **************************************************************************/

int para_set_clear(var, t)
ARG *var;
int t;
{ register TYPTAB *tt = types_table + t - 1;
  register char *sinit = malloc((size_t)(tt->typ_arg3 - tt->typ_arg2 + 2));

  if (sinit)
  { new_var(var, t, 0);
    sprintf(sinit, "%0*d", (int)(tt->typ_arg3 - tt->typ_arg2 + 1), 0);
    para_const_set(var, sinit);
    free(sinit);
    return TRUE;
  }
  else
  { komerr(texte[80],8); return FALSE; }
}

/**************************************************************************
 ***                      Funktion para_add_range
 ***
 *** Erzeugt Code, der die SET-Hilfsvariable in 'set' um
 *** den Bereich 'von..bis' erweitert.
 ***
 **************************************************************************/

para_add_range(set, von, bis)
SEXPR *set;
ARG *von, *bis;
{ register int erradr = error_num(&setrange_error_num);
  register TYPTAB *styp = types_table + set->typserg - 1;
  register int setsv = ARG_argsort(set->serg) & VEC;
  register int vonsv = ARG_argsort(*von) & VEC;
  register int bissv = ARG_argsort(*bis) & VEC;
  register int elsv = vonsv | bissv;
  ARG hilf, count, set_von, temp_bis, von_intval, bis_intval, nset;

  ARG_argsort(set_von) = ARG_argsort(temp_bis) = INT | CON;
  ARG_con_wert(set_von).datentyp = ARG_con_wert(temp_bis).datentyp = INT | INT_ERLAUBT;
  ARG_con_wert(set_von).inhalt.i_val = styp->typ_arg2;
  ARG_con_wert(temp_bis).inhalt.i_val = styp->typ_arg3;
  switch (ARG_typ(*von))
  { case BOOL :
      new_var(&von_intval, para_typ[INT], vonsv);
        /* von_intval := von */
      funcs(vonsv ? do_vib_zuw : do_sib_zuw, print_zuw);
      STAT_verg(*stat_ptr) = von_intval;
      STAT_vc1(*stat_ptr) = *von;
      nextstat();
      break;
    case CHA :
      new_var(&von_intval, para_typ[INT], vonsv);
        /* von_intval := von */
      funcs(vonsv ? do_vic_zuw : do_sic_zuw, print_zuw);
      STAT_verg(*stat_ptr) = von_intval;
      STAT_vc1(*stat_ptr) = *von;
      nextstat();
      break;
    case INT :
      von_intval = *von;
      break;
    default : bug("debug.c/para_incl_set : Falscher typ von 'von'");
  }
  switch (ARG_typ(*bis))
  { case BOOL :
      new_var(&bis_intval, para_typ[INT], bissv);
        /* bis_intval := bis */
      funcs(bissv ? do_vib_zuw : do_sib_zuw, print_zuw);
      STAT_verg(*stat_ptr) = bis_intval;
      STAT_vc1(*stat_ptr) = *bis;
      nextstat();
      break;
    case CHA :
      new_var(&bis_intval, para_typ[INT], bissv);
        /* bis_intval := bis */
      funcs(bissv ? do_vic_zuw : do_sic_zuw, print_zuw);
      STAT_verg(*stat_ptr) = bis_intval;
      STAT_vc1(*stat_ptr) = *bis;
      nextstat();
      break;
    case INT :
      bis_intval = *bis;
      break;
    default : bug("debug.c/para_incl_set : Falscher typ von 'bis'");
  }
        /* IF von_intval < set_von CALL erradr; */
  funcs(vonsv ? do_v2_ifcall : do_s2_ifcall, print_ifcall);
  STAT_vc1(*stat_ptr) = von_intval;
  STAT_rel_tok(*stat_ptr) = LT;
  STAT_vc2(*stat_ptr) = set_von;
  STAT_spr_ziel(*stat_ptr) = erradr;
  nextstat();
        /* IF von_intval > temp_bis CALL erradr; */
  funcs(vonsv ? do_v2_ifcall : do_s2_ifcall, print_ifcall);
  STAT_vc1(*stat_ptr) = von_intval;
  STAT_rel_tok(*stat_ptr) = GT;
  STAT_vc2(*stat_ptr) = temp_bis;
  STAT_spr_ziel(*stat_ptr) = erradr;
  nextstat();
        /* IF bis_intval < set_von CALL erradr; */
  funcs(bissv ? do_v2_ifcall : do_s2_ifcall, print_ifcall);
  STAT_vc1(*stat_ptr) = bis_intval;
  STAT_rel_tok(*stat_ptr) = LT;
  STAT_vc2(*stat_ptr) = set_von;
  STAT_spr_ziel(*stat_ptr) = erradr;
  nextstat();
        /* IF bis_intval > temp_bis CALL erradr; */
  funcs(bissv ? do_v2_ifcall : do_s2_ifcall, print_ifcall);
  STAT_vc1(*stat_ptr) = bis_intval;
  STAT_rel_tok(*stat_ptr) = GT;
  STAT_vc2(*stat_ptr) = temp_bis;
  STAT_spr_ziel(*stat_ptr) = erradr;
  nextstat();
  if (elsv && !setsv)
  { new_var(&nset, set->typserg, setsv = VEC);
        /* MOVE set TO nset AS settyp; */
    funcs(do_v_blockmove, print_blockmove);
    STAT_vc2(*stat_ptr) = set->serg;
    STAT_verg(*stat_ptr) = nset;
    STAT_dblock(*stat_ptr) = copy_declist(parz_typ(set->typserg), 1);
    nextstat();
    set->serg = nset;
  }
  else
  { nset = set->serg; }
  if (ARG_argsort(nset) & IND)
  { ARG_argsort(nset) = ARG_indsort(nset); }
  else
  { ARG_argsort(nset) = (ARG_argsort(nset) & ~VEC) |
                        ((ARG_argsort(nset) & VEC) ? ADV : ADS);
  }
  new_var(&hilf, para_typ[INT], elsv | (ARG_argsort(nset) & VEC));
        /* hilf := von_intval - set_von; */
  operation(&hilf, &von_intval, MINUS, &set_von);
  nextstat();
        /* hilf := hilf * SIZE ( B1 ); */
  ARG_argsort(temp_bis) = INT | CON | SIZ;
  ARG_sizedec(temp_bis) = copy_declist(parz_typ(para_typ[BOOL]), 1);
  operation(&hilf, &hilf, MAL, &temp_bis);
  nextstat();
        /* hilf := nset + hilf; */
  operation(&hilf, &nset, PLUS, &hilf);
  nextstat();
  if (!(ARG_argsort(von_intval) & IND) && ARG_tiefe(von_intval) == debug_lev)
  { count = von_intval; }
  else
  { new_var(&count, para_typ[INT], elsv);
          /* count := von_intval; */
    funcs(elsv ? do_vii_zuw : do_sii_zuw , print_zuw);
    STAT_verg(*stat_ptr) = count;
    STAT_vc1(*stat_ptr) = von_intval;
    nextstat();
  }
  if (elsv)
  { ARG laeuft, ska_laeuft;

    new_var(&laeuft, para_typ[BOOL], VEC);
    new_var(&ska_laeuft, para_typ[BOOL], 0);
        /*      laeuft := count <= bis_intval; */
    rel_operation(&laeuft, &count, LE, &bis_intval);
    nextstat();
        /* l1 : ska_laeuft := FALSE; */
    funcs(do_sbb_zuw, print_zuw);
    STAT_verg(*stat_ptr) = ska_laeuft;
    ARG_argsort(STAT_vc1(*stat_ptr)) = BOOL | CON;
    ARG_con_wert(STAT_vc1(*stat_ptr)).datentyp = BOOL | BOOL_ERLAUBT;
    ARG_con_wert(STAT_vc1(*stat_ptr)).inhalt.b_val = FALSE;
    STAT_label(*stat_ptr) = ++max_label;
    nextstat();
        /*      IF laeuft CALL l2; */
    funcs(do_v_ifcall, print_ifcall);
    STAT_vc1(*stat_ptr) = laeuft;
    STAT_rel_tok(*stat_ptr) = 0;
    STAT_spr_ziel(*stat_ptr) = stat_ptr - programm + 2;
    nextstat();
        /*      GOTO l3; */
    funcs(do_goto, print_goto);
    STAT_spr_ziel(*stat_ptr) = stat_ptr - programm + 8;
    nextstat();
        /* l2 : PROC debug_lev+1; */
    funcs(do_proc, print_proc);
    STAT_proclev(*stat_ptr) = debug_lev + 1;
    STAT_dscal(*stat_ptr) = dlptr(NULL);
    STAT_dvec(*stat_ptr) = dlptr(NULL);
    STAT_label(*stat_ptr) = ++max_label;
    nextstat();
        /*      VB[hilf] := TRUE; */
    funcs(do_vbb_zuw, print_zuw);
    STAT_verg(*stat_ptr) = hilf;
    ARG_indsort(STAT_verg(*stat_ptr)) = ARG_argsort(hilf);
    ARG_argsort(STAT_verg(*stat_ptr)) = BOOL | IND | VEC;
    ARG_argsort(STAT_vc1(*stat_ptr)) = BOOL | CON;
    ARG_con_wert(STAT_vc1(*stat_ptr)).datentyp = BOOL | BOOL_ERLAUBT;
    ARG_con_wert(STAT_vc1(*stat_ptr)).inhalt.b_val = TRUE;
    nextstat();
        /*      laeuft := count < bis_intval; */
    rel_operation(&laeuft, &count, LT, &bis_intval);
    nextstat();
        /*      ska_laeuft := TRUE; */
    funcs(do_sbb_zuw, print_zuw);
    STAT_verg(*stat_ptr) = ska_laeuft;
    ARG_argsort(STAT_vc1(*stat_ptr)) = BOOL | CON;
    ARG_con_wert(STAT_vc1(*stat_ptr)).datentyp = BOOL | BOOL_ERLAUBT;
    ARG_con_wert(STAT_vc1(*stat_ptr)).inhalt.b_val = TRUE;
    nextstat();
        /*      hilf := hilf + SIZE ( B1 ); */
    ARG_argsort(temp_bis) = INT | CON | SIZ;
    ARG_sizedec(temp_bis) = copy_declist(parz_typ(para_typ[BOOL]), 1);
    operation(&hilf, &hilf, PLUS, &temp_bis);
    nextstat();
        /*      count := count + 1; */
    ARG_argsort(temp_bis) = INT | CON;
    ARG_con_wert(temp_bis).datentyp = INT | INT_ERLAUBT;
    ARG_con_wert(temp_bis).inhalt.i_val = 1;
    operation(&count, &count, PLUS, &temp_bis);
    nextstat();
        /*      RETURN; */
    funcs(do_return, print_return);
    nextstat();
        /* l3 : IF ska_laeuft GOTO l1; */
    funcs(do_ifgoto, print_ifgoto);
    STAT_vc1(*stat_ptr) = ska_laeuft;
    STAT_spr_ziel(*stat_ptr) = stat_ptr - programm - 10;
    STAT_label(*stat_ptr) = ++max_label;
    nextstat();
  }
  else
  {     /*      IF count > bis_intval GOTO l2; */
    funcs(do_2_ifgoto, print_ifgoto);
    STAT_vc1(*stat_ptr) = count;
    STAT_vc2(*stat_ptr) = bis_intval;
    STAT_rel_tok(*stat_ptr) = GT;
    STAT_spr_ziel(*stat_ptr) = stat_ptr - programm + 6;
    nextstat();
        /* l1 : (S|V)B[hilf] := TRUE; */
    funcs(setsv ? do_vbb_zuw : do_sbb_zuw, print_zuw);
    STAT_verg(*stat_ptr) = hilf;
    ARG_indsort(STAT_verg(*stat_ptr)) = ARG_argsort(hilf);
    ARG_argsort(STAT_verg(*stat_ptr)) = BOOL | IND | setsv;
    ARG_argsort(STAT_vc1(*stat_ptr)) = BOOL | CON;
    ARG_con_wert(STAT_vc1(*stat_ptr)).datentyp = BOOL | BOOL_ERLAUBT;
    ARG_con_wert(STAT_vc1(*stat_ptr)).inhalt.b_val = TRUE;
    STAT_label(*stat_ptr) = ++max_label;
    nextstat();
        /*      IF count >= bis_intval GOTO l2; */
    funcs(do_2_ifgoto, print_ifgoto);
    STAT_vc1(*stat_ptr) = count;
    STAT_vc2(*stat_ptr) = bis_intval;
    STAT_rel_tok(*stat_ptr) = GE;
    STAT_spr_ziel(*stat_ptr) = stat_ptr - programm + 4;
    nextstat();
        /*      hilf := hilf + SIZE( B1 ); */
    ARG_argsort(temp_bis) = INT | CON | SIZ;
    ARG_sizedec(temp_bis) = copy_declist(parz_typ(para_typ[BOOL]), 1);
    operation(&hilf, &hilf, PLUS, &temp_bis);
    nextstat();
        /*      count := count + 1; */
    ARG_argsort(temp_bis) = INT | CON;
    ARG_con_wert(temp_bis).datentyp = INT | INT_ERLAUBT;
    ARG_con_wert(temp_bis).inhalt.i_val = 1;
    operation(&count, &count, PLUS, &temp_bis);
    nextstat();
        /*      GOTO l1; */
    funcs(do_goto, print_goto);
    STAT_spr_ziel(*stat_ptr) = stat_ptr - programm - 4;
    nextstat();
        /* l2 : ... */
    STAT_label(*stat_ptr) = ++max_label;
  }
}

/**************************************************************************
 ***                      Funktion para_incl_set
 ***
 *** Erzeugt Code, der die SET-Hilfsvariable in 'set' um
 *** den Wert von 'inc' erweitert
 ***
 **************************************************************************/

para_incl_set(set, inc)
SEXPR *set;
ARG *inc;
{ register int erradr = error_num(&setrange_error_num);
  register int t = set->typserg;
  register TYPTAB *styp = types_table + t - 1;
  register int setsv = ARG_argsort(set->serg) & VEC;
  register int elsv = ARG_argsort(*inc) & VEC;
  ARG set_von, temp_bis, intval, nset;
  ARG hilf;

  ARG_argsort(set_von) = ARG_argsort(temp_bis) = INT | CON;
  ARG_con_wert(set_von).datentyp = ARG_con_wert(temp_bis).datentyp = INT | INT_ERLAUBT;
  ARG_con_wert(set_von).inhalt.i_val = styp->typ_arg2;
  ARG_con_wert(temp_bis).inhalt.i_val = styp->typ_arg3;
  switch (ARG_typ(*inc))
  { case BOOL :
      new_var(&intval, para_typ[INT], elsv);
        /* intval := inc */
      funcs(elsv ? do_vib_zuw : do_sib_zuw, print_zuw);
      STAT_verg(*stat_ptr) = intval;
      STAT_vc1(*stat_ptr) = *inc;
      nextstat();
      break;
    case CHA :
      new_var(&intval, para_typ[INT], elsv);
        /* intval := inc */
      funcs(elsv ? do_vic_zuw : do_sic_zuw, print_zuw);
      STAT_verg(*stat_ptr) = intval;
      STAT_vc1(*stat_ptr) = *inc;
      nextstat();
      break;
    case INT :
      intval = *inc;
      break;
    default : bug("debug.c/para_incl_set : Falscher typ von 'inc'");
  }
        /* IF intval < set_von CALL erradr; */
  funcs(elsv ? do_v2_ifcall : do_s2_ifcall, print_ifcall);
  STAT_vc1(*stat_ptr) = intval;
  STAT_rel_tok(*stat_ptr) = LT;
  STAT_vc2(*stat_ptr) = set_von;
  STAT_spr_ziel(*stat_ptr) = erradr;
  nextstat();
        /* IF intval > temp_bis CALL erradr; */
  funcs(elsv ? do_v2_ifcall : do_s2_ifcall, print_ifcall);
  STAT_vc1(*stat_ptr) = intval;
  STAT_rel_tok(*stat_ptr) = GT;
  STAT_vc2(*stat_ptr) = temp_bis;
  STAT_spr_ziel(*stat_ptr) = erradr;
  nextstat();
  if (elsv && !setsv)
  { new_var(&nset, t, setsv = VEC);
        /* MOVE set->serg TO nset AS settyp; */
    funcs(do_v_blockmove, print_blockmove);
    STAT_vc2(*stat_ptr) = set->serg;
    STAT_verg(*stat_ptr) = nset;
    STAT_dblock(*stat_ptr) = copy_declist(parz_typ(t), 1);
    nextstat();
  }
  else
  { nset = set->serg; }
  if (ARG_argsort(nset) & IND)
  { ARG_argsort(nset) = ARG_indsort(nset); }
  else
  { ARG_argsort(nset) = (ARG_argsort(nset) & ~VEC) |
                        ((ARG_argsort(nset) & VEC) ? ADV : ADS);
  }
  new_var(&hilf, para_typ[INT], elsv | (ARG_argsort(nset) & VEC));
        /* hilf := intval - set_von; */
  operation(&hilf, &intval, MINUS, &set_von);
  nextstat();
        /* hilf := hilf * SIZE ( B1 ); */
  ARG_argsort(temp_bis) = INT | CON | SIZ;
  ARG_sizedec(temp_bis) = copy_declist(parz_typ(para_typ[BOOL]), 1);
  operation(&hilf, &hilf, MAL, &temp_bis);
  nextstat();
        /* hilf := ADDR nset + hilf; */
  operation(&hilf, &nset, PLUS, &hilf);
  nextstat();
        /* (S|V)B[hilf] := TRUE; */
  ARG_indsort(hilf) = ARG_argsort(hilf);
  ARG_argsort(hilf) = BOOL | IND | setsv;
  funcs(setsv ? do_vbb_zuw : do_sbb_zuw, print_zuw);
  STAT_verg(*stat_ptr) = hilf;
  ARG_argsort(STAT_vc1(*stat_ptr)) = BOOL | CON;
  ARG_con_wert(STAT_vc1(*stat_ptr)).datentyp = BOOL | BOOL_ERLAUBT;
  ARG_con_wert(STAT_vc1(*stat_ptr)).inhalt.b_val = TRUE;
  nextstat();
}
