/*****************************************************************************
  Project: PARZ - Parallel Intermediate Code Debugger/Interpreter
  ----------------------------------------------------------------------------
  Release      : 1
  Project Part : Debugger
  Filename     : debugio.c       
  SCCS-Path    : /tmp_mnt/user/sembach/parz/v2/SCCS/s.debugio.c
  Version      : 1.3 
  Last changed : 9/27/93 at 14:04:45        
  Author       : Frank Sembach
  Creation date: Aug. 92
  ----------------------------------------------------------------------------
  Description  : Funktionen zum Einlesen und Ausgeben der symbolischen
                 Debugger-Informationen von PARZ
******************************************************************************
***      (C) COPYRIGHT University of Stuttgart - All Right Reserved        ***
*****************************************************************************/

static char sccs_id[] = "@(#)debugio.c	1.3  9/27/93 PARZ - Debugger (Frank Sembach)";


#include "parzdefs.h"
#include "komdefs.h"
#include "y_tab.h"
#include "externs.h"
#include "rundefs.h"
#include "runexts.h"
#include "komexts.h"
#include "toup.h"
#include "debexts.h"

#define MAX_SRC_LN_LENGTH 257
#define SRC_ALLOC_PART 100

#ifdef MAC
char *inbuf; /* [BMAX + 1] */
char *xrf_source; /* [BMAX] */

init_debio_buffers()
{ if (!(inbuf = (char *)calloc(BMAX + 1, sizeof(char))))
  { fatal("not enough memory"); parzexit(1); }
  if (!(xrf_source = (char *)calloc(BMAX, sizeof(char))))
  { fatal("not enough memory"); parzexit(1); }
}
#else
char inbuf[BMAX + 1];
char xrf_source[BMAX];
#endif


/**************************************************************************
 ***                      Funktion hash
 ***
 *** Berechnet den Hashcode zur Zeichenkette 'z'.
 ***
 **************************************************************************/

int hash(z)
char * z;
{   register int h = 0;

    while (*z)
    {  if (uppercase_flag)
       { *z = mytoupper(*z); }
       h = ((h * factor) + *z++) % prime;
    }
    return h;
}

/**************************************************************************
 ***                      Funktion name_to_key
 ***
 *** Ermittelt Woerterbuchschluessel und Token zur Zeichenkette 's'
 *** *tok : liefert das Token zurueck
 *** Ergebnis : Woerterbuchschluessel, 0 bei Fehler
 ***
 **************************************************************************/

long name_to_key(s, tok)
char *s;
int *tok;
{ register int h = hash(s);
  register int n;
  register char **sptr;
  char ns[200];

  strcpy(ns,s);
  if (wbuch)
  { for (n = 0, sptr = wbuch[h].start_ptr; n < wbuch[h].wcount; n++, sptr++)
    { if (strcmp(s, *sptr) == 0)
      { *tok = wbuch[h].start_tok[n];
        return n * prime + h + 1;
      }
    }
  }
  *tok = 0;
  return 0l;
}

/**************************************************************************
 ***                      Funktion key_to_name
 ***
 *** Ermittelt die Zeichenkette zum Woerterbuchschluessel 'key'
 *** Im Fehlerfall: Ergebnis 'NULL'
 ***
 **************************************************************************/

char *key_to_name(key)
long key;
{ register int n = (key - 1) / prime;
  register int h = (key - 1) % prime;

  if (!wbuch || !key || n >= wbuch[h].wcount) return NULL;
  else return wbuch[h].start_ptr[n];
}

/**************************************************************************
 ***                      Funktion look_sym
 ***
 *** Sucht ab 'startscope' einen Symboltabelleneintrag zum
 *** Woerterbuchschluessel 'key'
 *** Im Fehlerfall: Ergebnis == NULL
 ***
 **************************************************************************/

SYMTAB *look_sym(key, startscope)
long key;
SCOPETAB *startscope;
{ register SYMTAB *st;
  register int h = (key - 1) % prime;
  register long tkey;
  register int i;

  if (startscope)
  { for (; startscope; startscope = startscope->sym_parent)
    {
#ifdef SCDEBUG
      printf("Scope %d   ", startscope - scope_table + 1);
#endif
      for (i = startscope->anz_sym, st = sym_table + startscope->von_sym - 1;
           i && (st->name_key - 1) % prime != h;
           i--, st++);
      for (; i && ((tkey = st->name_key) - 1) % prime == h; i--, st++)
      { if (tkey == key)
        {
#ifdef SCDEBUG
          putchar('\n');
#endif
          return st;
        }
      }
    }
#ifdef SCDEBUG
    putchar('\n');
#endif
  }
  return NULL;
}

/**************************************************************************
 ***                      Funktion loesch_consttab
 ***
 *** loescht die Konstantentabelle, die von 'read_xrf()' aufgebaut wurde.
 ***
 **************************************************************************/

loesch_consttab()
{ register CONSTTAB *ct;
  register int i;

  if (constant_table)
  { for (i = constcount, ct = constant_table; i; i--, ct++)
    { if (ct->const_wert.datentyp == STR)
      { if (ct->const_wert.inhalt.s_val) free(ct->const_wert.inhalt.s_val); }
    }
    free(constant_table); constant_table = NULL;
  }
  constcount = 0;
}

/**************************************************************************
 ***                      Funktion loesch_deb
 ***
 *** loescht alle Informationen, die von 'read_xrf()' erzeugt wurden
 ***
 **************************************************************************/

loesch_deb()
{ register STAT *sptr;
  register int i;

  geltcount = 0;
  if (name_string)
  { free(name_string); name_string = NULL; }
  if (name_ptrs)
  { free(name_ptrs); name_ptrs = NULL; }
  if (tok_ptrs)
  { free(tok_ptrs); tok_ptrs = NULL; }
  if (wbuch)
  { free(wbuch); wbuch = NULL; }
  stringcount = 0;
  if (sym_table)
  { free(sym_table); sym_table = NULL; }
  symbolcount = 0;
  system_name = "";
  if (scope_table)
  { free(scope_table); scope_table = NULL; }
  geltcount = 0;
  std_scope.von_sym = std_scope.anz_sym = 0;
  if (lines_table)
  { free(lines_table); lines_table = NULL; }
  zeilcount = 0;
  if (types_table)
  { free(types_table); types_table = NULL; }
  typcount = 0;
  if (member_table)
  { free(member_table); member_table = NULL; }
  mitgcount = 0;
  loesch_consttab();
  if (programm)
  { for (i = stat_anz, sptr = programm; i; i--, sptr++)
    { STAT_src_next(*sptr) = FALSE;
      if (STAT_src_ln(*sptr))
      { free(STAT_src_ln(*sptr)); STAT_src_ln(*sptr) = NULL; }
      if (STAT_scopes(*sptr))
      { free(STAT_scopes(*sptr)); STAT_scopes(*sptr) = NULL; }
      STAT_src_ln_count(*sptr) = STAT_scopes_count(*sptr) = 0;
    }
  }
  if (source_text)
  { register char **sp;
    for (sp = source_text; source_count; source_count--, sp++)
    { if (*sp) free(*sp); }
    free(source_text); source_text = NULL;
  }
}

#ifdef MAC
static char **invalid_keys;
static char **invalid_standard_idents;
static struct std_info
{ char *name;
  int art;
  int info;
} *standard_idents;

init_debio_ids()
{ register char **strptr;
  register struct std_info *siptr;

  if (!(invalid_keys = (char **)malloc(41 * sizeof(char *))))
  { fatal("not enough memory"); parzexit(1); }
  if (!(invalid_standard_idents = (char **)malloc(43 * sizeof(char *))))
  { fatal("not enough memory"); parzexit(1); }
  if (!(standard_idents = (struct std_info *)malloc(46 * sizeof(struct std_info))))
  { fatal("not enough memory"); parzexit(1); }

  strptr = invalid_keys;
  *strptr++ = "SCALAR";
  *strptr++ = "SEND";
  *strptr++ = "LOOP";
  *strptr++ = "ELSE";
  *strptr++ = "UNTIL";
  *strptr++ = "POINTER";
  *strptr++ = "OF";
  *strptr++ = "REPEAT";
  *strptr++ = "LOAD";
  *strptr++ = "PROCEDURE";
  *strptr++ = "TO";
  *strptr++ = "VECTOR";
  *strptr++ = "CONST";
  *strptr++ = "ARRAY";
  *strptr++ = "ELSIF";
  *strptr++ = "FROM";
  *strptr++ = "RECORD";
  *strptr++ = "TYPE";
  *strptr++ = "VAR";
  *strptr++ = "RECEIVE";
  *strptr++ = "WITH";
  *strptr++ = "SET";
  *strptr++ = "END";
  *strptr++ = "EXIT";
  *strptr++ = "FOR";
  *strptr++ = "DO";
  *strptr++ = "SYSTEM";
  *strptr++ = "STORE";
  *strptr++ = "ENDPARALLEL";
  *strptr++ = "IF";
  *strptr++ = "CASE";
  *strptr++ = "THEN";
  *strptr++ = "PROPAGATE";
  *strptr++ = "PARALLEL";
  *strptr++ = "WHILE";
  *strptr++ = "BY";
  *strptr++ = "RETURN";
  *strptr++ = "CONNECTION";
  *strptr++ = "BEGIN";
  *strptr++ = "CONFIGURATION";
  *strptr++ = NULL;

  strptr = invalid_standard_idents;
  *strptr++ = "EXCL";
  *strptr++ = "WriteReal";
  *strptr++ = "DrawReal";
  *strptr++ = "ReadCard";
  *strptr++ = "Write";
  *strptr++ = "Draw";
  *strptr++ = "WriteBool";
  *strptr++ = "DrawBool";
  *strptr++ = "WriteLn";
  *strptr++ = "DEC";
  *strptr++ = "SelectWindow";
  *strptr++ = "Line";
  *strptr++ = "HALT";
  *strptr++ = "OpenWindow";
  *strptr++ = "OpenAbsWindow";
  *strptr++ = "OpenInput";
  *strptr++ = "INCL";
  *strptr++ = "WriteCard";
  *strptr++ = "DrawCard";
  *strptr++ = "CloseInput";
  *strptr++ = "SetPixel";
  *strptr++ = "OpenOutput";
  *strptr++ = "WriteString";
  *strptr++ = "DrawString";
  *strptr++ = "TRACE";
  *strptr++ = "NOTRACE";
  *strptr++ = "DISPOSE";
  *strptr++ = "WriteInt";
  *strptr++ = "DrawInt";
  *strptr++ = "SetColor";
  *strptr++ = "ReadString";
  *strptr++ = "CloseOutput";
  *strptr++ = "INC";
  *strptr++ = "DEBUG";
  *strptr++ = "NEW";
  *strptr++ = "ReadInt";
  *strptr++ = "ReadReal";
  *strptr++ = "ReadBool";
  *strptr++ = "Read";
  *strptr++ = "WriteFixPt";
  *strptr++ = "DrawFixPt";
  *strptr++ = "WindowSize";
  *strptr++ = NULL;

  siptr = standard_idents;
  siptr->name = "FIRST";             siptr->art = SY_REDOPER; siptr++->info = FIRST;
  siptr->name = "LAST";              siptr->art = SY_REDOPER; siptr++->info = LAST;
  siptr->name = "SUM";               siptr->art = SY_REDOPER; siptr++->info = SUM;
  siptr->name = "PRODUCT";           siptr->art = SY_REDOPER; siptr++->info = PRODUCT;
  siptr->name = "MAX";               siptr->art = SY_RED_STDF; siptr++->info = MAX ;
  siptr->name = "MIN";               siptr->art = SY_RED_STDF; siptr++->info = MIN ;
  siptr->name = "id_no";             siptr->art = SY_SPEZ;   siptr++->info = ID ;
  siptr->name = "termCH";            siptr->art = SY_SPEZ;   siptr++->info = TERMCH ;
  siptr->name = "Done";              siptr->art = SY_SPEZ;   siptr++->info = DONE ;
  siptr->name = "NIL";               siptr->art = SY_SPEZ;   siptr++->info = NIL ;
  siptr->name = "ABS";               siptr->art = SY_STDFKT; siptr++->info = F_ABS;
  siptr->name = "CAP";               siptr->art = SY_STDFKT; siptr++->info = F_CAP;
  siptr->name = "CHR";               siptr->art = SY_STDFKT; siptr++->info = F_CHR;
  siptr->name = "EVEN";              siptr->art = SY_STDFKT; siptr++->info = F_EVEN;
  siptr->name = "FLOAT";             siptr->art = SY_STDFKT; siptr++->info = F_FLOAT;
  siptr->name = "ODD";               siptr->art = SY_STDFKT; siptr++->info = F_ODD;
  siptr->name = "ORD";               siptr->art = SY_STDFKT; siptr++->info = F_ORD;
  siptr->name = "SIZE";              siptr->art = SY_STDFKT; siptr++->info = F_SIZE;
  siptr->name = "TRUNC";             siptr->art = SY_STDFKT; siptr++->info = F_TRUNC;
  siptr->name = "VAL";               siptr->art = SY_STDFKT; siptr++->info = F_VAL;
  siptr->name = "STRCMP";            siptr->art = SY_STDFKT; siptr++->info = F_STRCMP;
  siptr->name = "STREQ";             siptr->art = SY_STDFKT; siptr++->info = F_STREQ;
  siptr->name = "SIRandom";          siptr->art = SY_STDFKT; siptr++->info = F_SIRANDOM;
  siptr->name = "SRRandom";          siptr->art = SY_STDFKT; siptr++->info = F_SRRANDOM;
  siptr->name = "SCRandom";          siptr->art = SY_STDFKT; siptr++->info = F_SCRANDOM;
  siptr->name = "SBRandom";          siptr->art = SY_STDFKT; siptr++->info = F_SBRANDOM;
  siptr->name = "VIRandom";          siptr->art = SY_STDFKT; siptr++->info = F_VIRANDOM;
  siptr->name = "VRRandom";          siptr->art = SY_STDFKT; siptr++->info = F_VRRANDOM;
  siptr->name = "VCRandom";          siptr->art = SY_STDFKT; siptr++->info = F_VCRANDOM;
  siptr->name = "VBRandom";          siptr->art = SY_STDFKT; siptr++->info = F_VBRANDOM;
  siptr->name = "IN_Connected";      siptr->art = SY_STDFKT; siptr++->info = F_IN_CONNECTED;
  siptr->name = "OUT_Connected";     siptr->art = SY_STDFKT; siptr++->info = F_OUT_CONNECTED;
  siptr->name = "IN_Lineconnected";  siptr->art = SY_STDFKT; siptr++->info = F_IN_LINECONNECTED;
  siptr->name = "OUT_Lineconnected"; siptr->art = SY_STDFKT; siptr++->info = F_OUT_LINECONNECTED;
  siptr->name = "ArcCos";            siptr->art = SY_STDFKT; siptr++->info = F_ARCCOS;
  siptr->name = "ArcSin";            siptr->art = SY_STDFKT; siptr++->info = F_ARCSIN;
  siptr->name = "ArcTan";            siptr->art = SY_STDFKT; siptr++->info = F_ARCTAN;
  siptr->name = "ArcTan2";           siptr->art = SY_STDFKT; siptr++->info = F_ARCTAN2;
  siptr->name = "Cos";               siptr->art = SY_STDFKT; siptr++->info = F_COS;
  siptr->name = "Exp";               siptr->art = SY_STDFKT; siptr++->info = F_EXP;
  siptr->name = "Ln";                siptr->art = SY_STDFKT; siptr++->info = F_LN;
  siptr->name = "Sin";               siptr->art = SY_STDFKT; siptr++->info = F_SIN;
  siptr->name = "Sqrt";              siptr->art = SY_STDFKT; siptr++->info = F_SQRT;
  siptr->name = "Tan";               siptr->art = SY_STDFKT; siptr++->info = F_TAN;
  siptr->name = "GetPixel";          siptr->art = SY_STDFKT; siptr++->info = F_GETPIXEL;
  siptr->name = NULL;                siptr->art = 0;         siptr++->info = 0;
}
#else

static char *invalid_keys[] = /* Fuer Debugger ungueltige Schluesselworte */
{ "SCALAR", "SEND", "LOOP", "ELSE", "UNTIL", "POINTER", "OF", "REPEAT",
  "LOAD", "PROCEDURE", "TO", "VECTOR", "CONST", "ARRAY", "ELSIF", "FROM",
  "RECORD", "TYPE", "VAR", "RECEIVE", "WITH", "SET", "END", "EXIT", "FOR",
  "DO", "SYSTEM", "STORE", "ENDPARALLEL", "IF", "CASE", "THEN", "PROPAGATE",
  "PARALLEL", "WHILE", "BY", "RETURN", "CONNECTION", "BEGIN", "CONFIGURATION",
  NULL
};

static char *invalid_standard_idents[] = /* Fuer Debugger ungueltige Standardbezeichner */
{ "EXCL", "WriteReal", "DrawReal", "ReadCard", "Write", "Draw", "WriteBool",
  "DrawBool", "WriteLn", "DEC", "SelectWindow", "Line", "HALT", "OpenWindow",
  "OpenAbsWindow",
  "OpenInput", "INCL", "WriteCard", "DrawCard", "CloseInput", "SetPixel",
  "OpenOutput", "WriteString", "DrawString", "TRACE", "NOTRACE", "DISPOSE",
  "WriteInt", "DrawInt", "SetColor", "ReadString", "CloseOutput", "INC",
  "DEBUG", "NEW", "ReadInt", "ReadReal", "ReadBool", "Read", "WriteFixPt",
  "DrawFixPt", "WindowSize",
  NULL
};

static struct std_info
{ char *name;
  int art;
  int info;
} standard_idents[] =   /* Standardbezeichner */
{ { "FIRST",             SY_REDOPER, FIRST},
  { "LAST",              SY_REDOPER, LAST},
  { "SUM",               SY_REDOPER, SUM},
  { "PRODUCT",           SY_REDOPER, PRODUCT},
  { "MAX",               SY_RED_STDF, MAX },
  { "MIN",               SY_RED_STDF, MIN },
  { "id_no",             SY_SPEZ, ID },
  { "termCH",            SY_SPEZ, TERMCH },
  { "Done",              SY_SPEZ, DONE },
  { "NIL",               SY_SPEZ, NIL },
  { "ABS",               SY_STDFKT, F_ABS},
  { "CAP",               SY_STDFKT, F_CAP},
  { "CHR",               SY_STDFKT, F_CHR},
  { "EVEN",              SY_STDFKT, F_EVEN},
  { "FLOAT",             SY_STDFKT, F_FLOAT},
  { "ODD",               SY_STDFKT, F_ODD},
  { "ORD",               SY_STDFKT, F_ORD},
  { "SIZE",              SY_STDFKT, F_SIZE},
  { "TRUNC",             SY_STDFKT, F_TRUNC},
  { "VAL",               SY_STDFKT, F_VAL},
  { "STRCMP",            SY_STDFKT, F_STRCMP},
  { "STREQ",             SY_STDFKT, F_STREQ},
  { "SIRandom",          SY_STDFKT, F_SIRANDOM},
  { "SRRandom",          SY_STDFKT, F_SRRANDOM},
  { "SCRandom",          SY_STDFKT, F_SCRANDOM},
  { "SBRandom",          SY_STDFKT, F_SBRANDOM},
  { "VIRandom",          SY_STDFKT, F_VIRANDOM},
  { "VRRandom",          SY_STDFKT, F_VRRANDOM},
  { "VCRandom",          SY_STDFKT, F_VCRANDOM},
  { "VBRandom",          SY_STDFKT, F_VBRANDOM},
  { "IN_Connected",      SY_STDFKT, F_IN_CONNECTED},
  { "OUT_Connected",     SY_STDFKT, F_OUT_CONNECTED},
  { "IN_Lineconnected",  SY_STDFKT, F_IN_LINECONNECTED},
  { "OUT_Lineconnected", SY_STDFKT, F_OUT_LINECONNECTED},
  { "ArcCos",            SY_STDFKT, F_ARCCOS},
  { "ArcSin",            SY_STDFKT, F_ARCSIN},
  { "ArcTan",            SY_STDFKT, F_ARCTAN},
  { "ArcTan2",           SY_STDFKT, F_ARCTAN2},
  { "Cos",               SY_STDFKT, F_COS},
  { "Exp",               SY_STDFKT, F_EXP},
  { "Ln",                SY_STDFKT, F_LN},
  { "Sin",               SY_STDFKT, F_SIN},
  { "Sqrt",              SY_STDFKT, F_SQRT},
  { "Tan",               SY_STDFKT, F_TAN},
  { "GetPixel",          SY_STDFKT, F_GETPIXEL},
  { NULL, 0, 0}
};                                  

#endif

/**************************************************************************
 ***                      Funktion name_tok
 ***
 *** Ordnet der Zeichenkette 's' im Woerterbuch das Token 'tok' zu
 ***
 **************************************************************************/

name_tok(s, tok)
char *s;
int tok;
{ register int h = hash(s);
  register int n;
  register char **sptr;

  for (n = 0, sptr = wbuch[h].start_ptr; n < wbuch[h].wcount; n++, sptr++)
  { if (strcmp(s, *sptr) == 0)
    { wbuch[h].start_tok[n] = tok; }
  }
}

/**************************************************************************
 ***                      Funktion init_keywords
 ***
 *** Ordnet allen Schluesselworten im Woerterbuch ihr Token zu
 ***
 **************************************************************************/

init_keywords()
{ register char **nptr;

  name_tok("IN", IN);
  name_tok("OR", OR);
  name_tok("DIV", DIV);
  name_tok("MOD", MOD);
  name_tok("AND", AND);
  name_tok("NOT", NOT);
  name_tok("REDUCE", REDUCE);
  for (nptr = invalid_keys; *nptr; nptr++)
  { name_tok(*nptr, 0); }
}

/**************************************************************************
 ***                      Funktion init_std_idents
 ***
 *** Ergaenzt Informationen in der Tabelle der Standardbezeichner
 ***
 **************************************************************************/

init_std_idents()
{ int dummytok;
  register SYMTAB *st;
  { register char **inames;

    for (inames = invalid_standard_idents; *inames; inames++)
    { st = look_sym(name_to_key(*inames, &dummytok), &std_scope);
      st->sym_art = 0;
    }
  }
  { register struct std_info *si;

    for (si = standard_idents; si->name; si++)
    { st = look_sym(name_to_key(si->name, &dummytok), &std_scope);
      st->sym_art = si->art;
      st->sym_type_scope = si->info;
    }
  }
}

/**************************************************************************
 ***                      Funktion nextline
 ***
 *** Liest die erste nichtleere Zeile (maximal 'BMAX' Zeichen)
 *** aus 'f' nach 'inbuf'
 ***
 **************************************************************************/

nextline(f)
FILE *f;
{ do
  { if (!fgets(inbuf, BMAX, f))
    { inbuf[0] = '\0'; return; }
  } while (inbuf[0] <= ' ');
}

/**************************************************************************
 ***                      Funktion read_xrf
 ***
 *** Liest die Informationen fuer den symbolischen Debugger aus 'f'
 ***
 **************************************************************************/

int read_xrf(f)
FILE *f;
{ register int i;

                /* Magic-number testen */
  nextline(f);
  if (strcmp(inbuf, MAGIC)) return MAGIC_FALSCH;

                /* optionalen Quelldateiname lesen */
  nextline(f);
  if (inbuf[0] == 'Q')
  { sscanf(&inbuf[1], " %s", xrf_source);
    nextline(f);
  }
  else
  { xrf_source[0] = '\0'; }

                /* Woerterbuch lesen */
  if (inbuf[0] != 'W') return WB_FALSCH;
  nextline(f);
  { long lensum;
    register int h_code, max_h_code = -1;
    register char *n_ptr, **np_ptr;
    register int *t_ptr;

    if (sscanf(inbuf, "%ld %ld %d %d %d",
               &stringcount, &lensum, &prime, &factor, &uppercase_flag) != 5)
    { loesch_deb(); return WB_FALSCH; }
    if (!(name_string = calloc((size_t)(stringcount + lensum), (size_t)sizeof(char))))
    { loesch_deb(); return NO_MEM; }
    if (!(name_ptrs = (char **)calloc((size_t)stringcount, (size_t)sizeof(char *))))
    { loesch_deb(); return NO_MEM; }
    if (!(tok_ptrs = (int *)calloc((size_t)stringcount, (size_t)sizeof(int))))
    { loesch_deb(); return NO_MEM; }
    if (!(wbuch = (WB *)calloc((size_t)prime, (size_t)sizeof(WB))))
    { loesch_deb(); return NO_MEM; }
    for (i = stringcount, n_ptr = name_string, np_ptr= name_ptrs, t_ptr = tok_ptrs; 
         i;
         i--, *(np_ptr++) = n_ptr, *(t_ptr++) = IDENT, n_ptr += strlen(n_ptr) + 1)
    { nextline(f);
      sscanf(inbuf, "%s", n_ptr);
      h_code = hash(n_ptr);
      if (h_code > max_h_code)
      { wbuch[h_code].start_ptr = np_ptr;
        wbuch[h_code].start_tok = t_ptr;
        max_h_code = h_code;
      }
      else if (h_code < max_h_code)
      { loesch_deb(); return WB_FALSCH; }
      wbuch[h_code].wcount++;
    }
  }
  init_keywords();  /* Schluesselworte mit Token versehen */

                /* Symboltabelle lesen */
  nextline(f);
  if (sscanf(inbuf, "S %ld", &symbolcount) != 1)
  { loesch_deb(); return SYM_FALSCH; }
  if (!(sym_table = (SYMTAB *)calloc((size_t)symbolcount, (size_t)sizeof(SYMTAB))))
  { loesch_deb(); return NO_MEM; }
  { register SYMTAB *st;
    int test_art;
    char s[BMAX];
    register int erg = 0;

    for (i = symbolcount, st = sym_table; i; i--, st++)
    { nextline(f);
      if (sscanf(inbuf, "%d", &test_art) != 1)
      { loesch_deb(); return SYM_FALSCH; }
      switch(st->sym_art = (test_art & ALL_SYMS))
      { case SY_UNDEF     : 
        case SY_SYSNAME   :
        case SY_STDFKT    :
        case SY_STDPROC   :
        case SY_SPEZ      :
          if (sscanf(inbuf, "%d %ld", &test_art, &st->name_key) != 2)
          { erg = SYM_FALSCH; }
          else if (test_art == SY_SYSNAME)
          { system_name = key_to_name(st->name_key); }
          break;
        case SY_TYPENAME  :
        case SY_PROCNAME  :
          if (sscanf(inbuf, "%d %ld %d", &test_art, &st->name_key,
                                         &st->sym_type_scope) != 3)
          { erg = SYM_FALSCH; }
          break;
        case SY_CONSTNAME :
        case SY_CONFNAME  :
        case SY_SCALNAME  :
        case SY_VECNAME   :
        case SY_IOPORTNAME  :
        case SY_INPORTNAME  :
        case SY_OUTPORTNAME  :
          if (sscanf(inbuf, "%d %ld %d %ld", &test_art, &st->name_key,
                                             &st->sym_type_scope,
                                             &st->sym_spez) != 4)
          { erg = SYM_FALSCH; break; }
          break;
      }
      if (erg)
      { loesch_deb(); return erg; }
      st->ind_flag = !!(test_art & SY_INDFLAG);
      st->sym_lev = test_art / SY_LEVUNIT;
    }
  }

                /* Geltungsbereiche lesen */
  nextline(f);
  if (sscanf(inbuf, "G %ld", &geltcount) != 1)
  { loesch_deb(); return GELT_FALSCH; }
  if (!(scope_table = (SCOPETAB *)calloc((size_t)geltcount, (size_t)sizeof(SCOPETAB))))
  { loesch_deb(); return NO_MEM; }
  { register SCOPETAB *sct, **neu;
    register int erg = 0;
    register STAT *sptr;
    int von_l, entry_l, next_l, parnum;

    for (i = geltcount, sct = scope_table; i; i--, sct++)
    { nextline(f);
      if (sscanf(inbuf, "%d %d %d %d %ld %d %d %d %d",
                        &sct->von_sym,
                        &sct->anz_sym,
                        &parnum,
                        &sct->sc_lev,
                        &sct->proc_key,
                        &sct->par_flag,
                        &von_l,
                        &entry_l,
                        &next_l) != 9)
      { erg = GELT_FALSCH; break; }
      sct->parent = parnum ? scope_table + parnum - 1 : NULL;
      if ((sct->von_lab = ziel(-von_l)) < 0)
      { erg = GELT_FALSCH; break; }
      if ((sct->entry_lab = ziel(-entry_l)) < 0)
      { erg = GELT_FALSCH; break; }
      if ((sct->next_lab = ziel(-next_l)) < 0)
      { erg = GELT_FALSCH; break; }
      sct->sym_parent = NULL;
    }
    if (erg)
    { loesch_deb(); return erg; }
                        /* Scope fuer Standardnamen anlegen */
    std_scope.von_sym = (sct - 1)->von_sym + (sct - 1)->anz_sym;
    std_scope.anz_sym = symbolcount - std_scope.von_sym + 1;
    init_std_idents();
                        /* Geltungsbereiche in Programm eintragen */
    for (i = geltcount, sct = scope_table + i - 1; i; i--, sct--)
    { if (sct->von_lab != sct->next_lab)
      { sptr = programm + sct->entry_lab;
        if (!(neu = (SCOPETAB **)(STAT_scopes(*sptr)                              ?
                                    realloc(STAT_scopes(*sptr),
                                            (size_t)(++STAT_scopes_count(*sptr) *
                                                     sizeof(SCOPETAB *)))         :
                                    calloc((size_t)++STAT_scopes_count(*sptr),
                                           (size_t)sizeof(SCOPETAB *)))))
        { loesch_deb(); return NO_MEM; }
        neu[STAT_scopes_count(*sptr) - 1] = sct;
        STAT_scopes(*sptr) = neu;
      }
    }
  }

                /* Zeilennummern lesen */
  nextline(f);
  if (sscanf(inbuf, "Z %ld", &zeilcount) != 1)
  { loesch_deb(); return ZNR_FALSCH; }
  if (!(lines_table = (ZEILTAB *)calloc((size_t)zeilcount, (size_t)sizeof(ZEILTAB))))
  { loesch_deb(); return NO_MEM; }
  { register ZEILTAB *zt, **neu;
    register int erg = 0;
    register int alt_l, offs;
    register STAT *sptr;
    int von_l, next_l;

    for (i = zeilcount, zt = lines_table + i - 1; i; i--, zt--)
    { nextline(f);
      if (sscanf(inbuf, "%d %d %d", &zt->source_line, &von_l, &next_l) != 3)
      { erg = ZNR_FALSCH; break; }
      if ((zt->prog_von = von_l = ziel(-von_l)) < 0)
      { erg = ZNR_FALSCH; break; }
      if ((zt->prog_next = next_l = ziel(-next_l)) < 0)
      { erg = ZNR_FALSCH; break; }
      zt->parent_line = NULL;
      if (von_l != next_l) STAT_src_next(programm[next_l]) = TRUE;
    }
    if (erg)
    { loesch_deb(); return erg; }
                        /* Zeilennummern in Programm eintragen */
    for (i = zeilcount, zt = lines_table, alt_l = offs = 0; i; i--, zt++)
    { if (zt->prog_von != zt->prog_next)
      { sptr = programm + zt->prog_von;
        if (zt->source_line == alt_l)
        { zt->stat_in_line = ++offs; }
        else
        { alt_l = zt->source_line; offs = 0; }
        if (!(neu = (ZEILTAB **)(STAT_src_ln(*sptr)                             ?
                                   realloc(STAT_src_ln(*sptr),
                                           (size_t)(++STAT_src_ln_count(*sptr) *
                                                    sizeof(ZEILTAB *)))         :
                                   calloc((size_t)++STAT_src_ln_count(*sptr),
                                          (size_t)sizeof(ZEILTAB *)))))
        { loesch_deb(); return NO_MEM; }
        neu[STAT_src_ln_count(*sptr) - 1] = zt;
        STAT_src_ln(*sptr) = neu;
      }
    }
  }

                /* Typtabelle lesen */
  nextline(f);
  if (sscanf(inbuf, "T %ld", &typcount) != 1)
  { loesch_deb(); return TYP_FALSCH; }
  if (!(types_table = (TYPTAB *)calloc((size_t)typcount + 4, (size_t)sizeof(TYPTAB))))
  { loesch_deb(); return NO_MEM; }
  { register TYPTAB *tt;
    register int erg = 0;
    int typ_art;

    for (i = typcount, tt = types_table; i; i--, tt++)
    { nextline(f);
      if (sscanf(inbuf, "%d", &typ_art) != 1)
      { loesch_deb(); return TYP_FALSCH; }
      switch (typ_art)
      { case TYP_UNDEF    :
        case TYP_BOOL     :
        case TYP_CHA      :
        case TYP_INT      :
        case TYP_REAL     :
          if (sscanf(inbuf, "%d %ld", &tt->typ_art, &tt->typ_key) != 2)
          { erg = TYP_FALSCH; }
          break;
        case TYP_STRING   :
        case TYP_RECORD   :
        case TYP_UNION    :
        case TYP_POINTER  :
        case TYP_ENUM     :
        case TYP_PORT     :
          if (sscanf(inbuf, "%d %ld %ld",
                            &tt->typ_art, &tt->typ_key, &tt->typ_arg1) != 3)
          { erg = TYP_FALSCH; }
          break;
        case TYP_ARRAY    :
          if (sscanf(inbuf, "%d %ld %ld %ld",
                            &tt->typ_art, &tt->typ_key,
                            &tt->typ_arg1, &tt->typ_arg2) != 4)
          { erg = TYP_FALSCH; }
          break;
        case TYP_SET      : 
        case TYP_SUBRANGE :
        case TYP_PORTTYPE :
          if (sscanf(inbuf, "%d %ld %ld %ld %ld",
                            &tt->typ_art, &tt->typ_key,
                            &tt->typ_arg1, &tt->typ_arg2, &tt->typ_arg3) != 5)
          { erg = TYP_FALSCH; }
          break;
      }
      if (erg)
      { loesch_deb(); return erg; }
    }
        /* Standardtypen stehen in fester Reihenfolge am Ende der Typtabelle */
    para_typ[INT] = typcount;
    para_typ[BOOL] = typcount - 1;
    para_typ[CHA] = typcount - 2;
    cardinal_typ = typcount - 3;
    para_typ[REAL] = typcount - 4;
    bitset_typ = typcount - 5;
    nil_typ = typcount - 7;
    color_typ = typcount - 8;
        /* Vier zusaetzliche Typeintraege fuer Strings, die im Debugger
           eingegeben werden */
    /* index_typ1 : variabler Subrange */
    tt->typ_art = TYP_SUBRANGE;
    tt->typ_arg1 = para_typ[INT];
    tt->typ_arg2 = 0;
    index_typ1 = typcount + 1;
    /* string_typ1 : String[typcount + 1] */
    (++tt)->typ_art = TYP_STRING;
    tt->typ_arg1 = index_typ1;
    string_typ1 = typcount + 2;
    /* index_typ2 : variabler Subrange */
    (++tt)->typ_art = TYP_SUBRANGE;
    tt->typ_arg1 = para_typ[INT];
    tt->typ_arg2 = 0;
    index_typ2 = typcount + 3;
    /* string_typ2 : String[typcount + 1] */
    (++tt)->typ_art = TYP_STRING;
    tt->typ_arg1 = index_typ2;
    string_typ2 = typcount + 4;
  }

                /* Mitgliedertabelle lesen */
  nextline(f);
  if (sscanf(inbuf, "M %ld", &mitgcount) != 1)
  { loesch_deb(); return MITG_FALSCH; }
  if (!(member_table = (MITGTAB *)calloc((size_t)mitgcount, (size_t)sizeof(MITGTAB))))
  { loesch_deb(); return NO_MEM; }
  { register MITGTAB *mt;

    for (i = mitgcount, mt = member_table; i; i--, mt++)
    { nextline(f);
      if (sscanf(inbuf, "%ld %ld %d",
                        &mt->mem_key, &mt->typ_wert, &mt->flag_last) != 3)
      { loesch_deb(); return MITG_FALSCH; }
    }
  }

                /* Konstantentabelle lesen */
  nextline(f);
  if (sscanf(inbuf, "C %ld", &constcount) != 1)
  { loesch_deb(); return CONST_FALSCH; }
  if (!(constant_table = (CONSTTAB *)calloc((size_t)constcount, (size_t)sizeof(CONSTTAB))))
  { loesch_deb(); return NO_MEM; }
  { register CONSTTAB *ct;
    register int fl, len, nlen;
    register char *s, *neu, first;

    for (i = constcount, ct = constant_table; i; i--, ct++)
    { nextline(f);
      if ((fl = *inbuf - '0') < 0 || fl > (LAST_FLAG | STRING_FLAG))
      { loesch_deb(); return CONST_FALSCH; }
      ct->last_flag = fl & LAST_FLAG;
      ct->string_flag = fl & STRING_FLAG;
      if (inbuf[1] != ' ' || inbuf[2] != '"')
      { loesch_deb(); return CONST_FALSCH; }
      if (!(s = malloc((size_t)(strlen(inbuf) - 2))))
      { loesch_deb(); return NO_MEM; }
      len = strlen(inbuf) - 4;
      strncpy(s, inbuf + 3, (size_t)len);
      while (inbuf[strlen(inbuf) - 2] != '"')
      { if (!fgets(inbuf, BMAX, f))
        { loesch_deb(); free(s); return CONST_FALSCH; }
        if (!(neu = realloc(s, (size_t)(len + strlen(inbuf)))))
        { loesch_deb(); free(s); return NO_MEM; }
        s = neu; neu = s + len;
        nlen = strlen(inbuf) - 1;
        strncpy(neu, inbuf, (size_t)nlen); len += nlen;
      }
      s[len - 1] = '\0';
      ct->const_wert.datentyp = STR;
      ct->const_wert.inhalt.s_val = s;
    }
  }

                /* Endekennung pruefen */
  nextline(f);
  if (*inbuf != 'E')
  { loesch_deb(); return KEIN_E; }

                /* Standard-PARZ-Typen eintragen */
  parz_typ(para_typ[BOOL]);
  parz_typ(para_typ[CHA]);
  parz_typ(para_typ[INT]);
  parz_typ(para_typ[REAL]);

  return 0;
}

/**************************************************************************
 ***                      Funktion read_source
 ***
 *** Liest den Parallaxis Quelltext aus der Datei 'the_sourcename'
 ***
 **************************************************************************/

read_source()
{ char buf[MAX_SRC_LN_LENGTH];
  register char *zeile, **neu;
  register int source_max;
  register FILE *f;

  source_count = 0; source_text = NULL;
  if (!(f = fopen(the_sourcename, "r")))
  { loesch_deb(); fputs(texte[300], stderr); return; }
  if (!(source_text = (char **)calloc((size_t)SRC_ALLOC_PART, (size_t)sizeof(char *))))
  { loesch_deb(); fatal(texte[299]); return; }
  source_max = SRC_ALLOC_PART - 1;
  while (fgets(buf, MAX_SRC_LN_LENGTH - 2, f))
  { if (!(zeile = calloc((size_t)(strlen(buf) + 1), (size_t)sizeof(char))))
    { loesch_deb(); fatal(texte[299]); return; }
    strcpy(zeile, buf);
    if (source_count > source_max)
    { if (!(neu =
            (char **)realloc(source_text,
                             (size_t)((source_count + SRC_ALLOC_PART) *
                                      sizeof(char *)))))
      { if (!(neu = (char **)realloc(source_text,
                                     (size_t)((source_count + 1) * sizeof(char *)))))
        { loesch_deb(); free(zeile); fatal(texte[299]); return; }
        else
        { source_max += 1; }
      }
      else
      { source_max += SRC_ALLOC_PART; }
      source_text = neu;
    }
    source_text[source_count++] = zeile;
  }
  source_text = (char **)realloc(source_text, (size_t)(source_count * sizeof(char *)));
  fclose(f);
}

/**************************************************************************
 ***                      Funktion proc_lines
 ***
 *** sucht in 'sym_table' nach Prozedureintraegen mit Name 'key'
 *** Die Eintrittszeilen der zugehoerigen Geltungsbereiche werden im
 *** dynamisch verlaengerbaren Feld 'zl' von '*zeilen' eingetragen.
 *** 'zcount' in '*zeilen' zaehlt die Eintraege.
 ***
 **************************************************************************/

proc_lines(key, zeilen)
long key;
ZEILE *zeilen;
{ register long i;
  register SYMTAB *sptr;
  register int *neu, zc;

  zeilen->list_flag = 1;
  zeilen->zval.zlist.zcount = 0;
  if (!(zeilen->zval.zlist.zl = (int *)calloc((size_t)1,(size_t)sizeof(int))))
  { komerr(texte[80],34); return; }
  for (sptr = sym_table, i = symbolcount; i; i--, sptr++)
  { if ((sptr->sym_art == SY_PROCNAME || sptr->sym_art == SY_SYSNAME) &&
        sptr->name_key == key)
    { if (zc = zeilen->zval.zlist.zcount)
      { if (!(neu =
              (int *)realloc(zeilen->zval.zlist.zl, (size_t)((zc + 1) * sizeof(int)))))
        { komerr(texte[80],35);
          free(zeilen->zval.zlist.zl); zeilen->zval.zlist.zl = NULL;
          zeilen->zval.zlist.zcount = 0;
          return;
        }
        zeilen->zval.zlist.zl = neu;
      }
      zeilen->zval.zlist.zl[zc] = scope_table[sptr->sym_type_scope - 1].entry_lab;
      zeilen->zval.zlist.zcount++;
    }
  }
}

#ifdef DEBDEBUG

/****************** XRF-Ausgabe **************************************/

char *sart_name[] =
{ "undef",
  "system-name",
  "const-name",
  "type-name",
  "conf-name",
  "scalar-name",
  "vector-name",
  "proc-name",
  "io-port-name",
  "in-port-name",
  "out-port-name",
  "std_function",
  "std_procedure",
  "special",

  "spez-var-const",
  "redoper/stdfkt",
  "redoper"
};

xrf_out(f)
FILE *f;
{ register int i, j;
  SCOPETAB *sct;
  SYMTAB *st;

  if (the_sourcename) fprintf(f, "Quelldatei : %s\n", the_sourcename);
  fputs("Geltungsbereiche \n", f);
  for (i = 1, sct = scope_table; i <= geltcount + 1; i++, sct = (i <= geltcount) ? sct + 1 : &std_scope)
  { fprintf(f, "\nGB Nr %d, VaterGB : %d, Ebene : %d\nProc Name : %s, %s\nvon %d eingang %d nach %d\n",
               i, sct->parent ? sct->parent - scope_table + 1 : 0, sct->sc_lev, 
               sct->proc_key ? key_to_name(sct->proc_key) : "---", sct->par_flag ? "PARALLEL" : "NORMAL",
               STAT_label(programm[sct->von_lab]), STAT_label(programm[sct->entry_lab]), STAT_label(programm[sct->next_lab]));
    for (j = sct->anz_sym, st = sym_table + sct->von_sym - 1; j; j--, st++)
    { fprintf(f, "Name : %s \tKey : %d\n  Art : %s %s",
                 st->name_key ? key_to_name(st->name_key) : "---", st->name_key,
                 sart_name[st->sym_art],
                 st->ind_flag ? "IND " : "");
      switch(st->sym_art)
      { case SY_UNDEF :
        case SY_SYSNAME :
        case SY_STDPROC : putc('\n', f);
                          break;
        case SY_PROCNAME : fprintf(f, "  Geltungsb. : %d\n", st->sym_type_scope);
                           break;
        case SY_TYPENAME : fputs("\n  Typ : ", f);
                           typ_out(st->sym_type_scope, 1, f);
                           putc('\n', f);
                           break;
        case SY_CONSTNAME :
        case SY_CONFNAME :
        case SY_IOPORTNAME :
        case SY_INPORTNAME :
        case SY_OUTPORTNAME : fputs("\n  Typ : ", f);
                              typ_out(st->sym_type_scope, 0, f);
                              putc('\n', f);
                              break;
        case SY_SCALNAME :
        case SY_VECNAME :     fprintf(f,"Ebene : %d\n  Typ : ", st->sym_lev);
                              typ_out(st->sym_type_scope, 0, f);
                              putc('\n', f);
                              break;
        case SY_STDFKT :
        case SY_SPEZ :
        case SY_RED_STDF :
        case SY_REDOPER :     fprintf(f,"Info : %d\n", st->sym_type_scope);
      }
      switch(st->sym_art)
      {
        case SY_CONSTNAME : fputs("  Wert : ", f);
                            switch (types_table[st->sym_type_scope - 1].typ_art)
                            { case TYP_BOOL :
                              case TYP_CHA :
                              case TYP_INT :
                              case TYP_SUBRANGE :
                              case TYP_ENUM : fprintf(f, "%ld\n", st->sym_spez);
                                              break;
                              default : const_out(st->sym_spez, f);
                            }
                            break;
        case SY_CONFNAME : fprintf(f, "  erste PE : %ld\n", st->sym_spez);
                           break;
        case SY_SCALNAME :
        case SY_VECNAME : fprintf(f, "  rel-Nr : %ld\n", st->sym_spez);
                          break;
        case SY_IOPORTNAME :
        case SY_INPORTNAME :
        case SY_OUTPORTNAME : fprintf(f, "  Port-Nr : %ld\n", st->sym_spez);
      }
    }
  }
}

typ_out(typnum, tdef_flag, f)
int typnum;
FILE *f;
{ TYPTAB *typptr = types_table + typnum - 1;

  if (!typnum) fputs("undef", f);
  else if (typptr->typ_key && !tdef_flag)
  { fputs(key_to_name(typptr->typ_key), f); }
  else
  switch (typptr->typ_art)
  { case TYP_UNDEF    : fputs("undef", f); break;
    case TYP_BOOL     : fputs("BOOLEAN", f); break;
    case TYP_CHA      : fputs("CHAR", f); break;
    case TYP_INT      : fputs("INTEGER", f); break;
    case TYP_REAL     : fputs("REAL", f); break;
    case TYP_STRING   : fputs("STRING(", f);
                        typ_out((int)typptr->typ_arg1, 0, f);
                        fputs(")", f);
                        break;
    case TYP_ARRAY    : fputs("ARRAY ", f);
                        typ_out((int)typptr->typ_arg1, 0, f);
                        fputs(" OF ", f);
                        typ_out((int)typptr->typ_arg2, 0, f);
                        break;
    case TYP_RECORD   : fputs("RECORD ", f);
                        memb_out((int)typptr->typ_arg1, 1, f);
                        fputs("ENDREC", f);
                        break;
    case TYP_UNION    : fputs("UNION ", f);
                        memb_out((int)typptr->typ_arg1, 1, f);
                        fputs("ENDUN", f);
                        break;
    case TYP_SET      : fputs("SET OF ", f);
                        typ_out((int)typptr->typ_arg1, 0, f);
                        fprintf(f, "[%ld .. %ld]", typptr->typ_arg2, typptr->typ_arg3);
                        break;
    case TYP_POINTER  : fputs("POINTER TO ", f);
                        typ_out((int)typptr->typ_arg1, 0, f);
                        break;
    case TYP_SUBRANGE : typ_out((int)typptr->typ_arg1, 0, f);
                        fprintf(f, "[%ld .. %ld]", typptr->typ_arg2, typptr->typ_arg3);
                        break;
    case TYP_ENUM     : fputs("enum(", f);
                        memb_out((int)typptr->typ_arg1, 0, f);
                        fputs(")", f);
                        break;
    case TYP_PORT     : fprintf (f, "in %s", key_to_name(typptr->typ_arg1));
                        break;
    case TYP_PORTTYPE : fprintf (f, "(%ld .. %ld) in %s",
                                    typptr->typ_arg2, typptr->typ_arg3,
                                    key_to_name(typptr->typ_arg1));
                        break;
  }
}

memb_out(membnum, tflag, f)
int membnum;
int tflag;
FILE *f;
{ MITGTAB *membptr = member_table + membnum - 1;

  do
  { fprintf(f, "%s %c ", membptr->mem_key ? key_to_name(membptr->mem_key) : "---",
                         tflag ? ':' : '=');
    if (tflag)
    { typ_out((int)membptr->typ_wert, 0, f);
      fputs(";\n          ", f);
    }
    else fprintf(f, "%ld%s", membptr->typ_wert, membptr->flag_last ? "" : ", ");
  } while (!(membptr++)->flag_last);
}

const_out(constnum, f)
long constnum;
FILE *f;
{ CONSTTAB *constptr = constant_table + constnum - 1;

  beim_listing = TRUE;
  do
  { print_wert(&constptr->const_wert, f, STR);
    if (!constptr->last_flag) fputs(", ", f);
  } while (!(constptr++)->last_flag);
  putc('\n', f);
  beim_listing = FALSE;
}

parz_typs_out(f)
FILE *f;
{ register TYPTAB *tt;
  register int i;

  for (i = 1, tt = types_table; i <= typcount; i++, tt++)
  { fprintf(f, "Typ Nr. %d :", i);
    if (tt->parz_decl.dcount)
    { list_decl(&tt->parz_decl, f);
      fprintf(f, " | %d", tt->ausg_zeilen);
    }
    putc('\n', f);
  }
}

/**************************************************************************
 ***             Funktion debcode_out
 ***
 *** Ausgabe des Debugcodes (nur zu Debugzwecken )
 ***
 **************************************************************************/

debcode_out()
{ RANGE r;

  more_mod = FALSE;
  if ((r.von_n = debcode_start) <= (r.bis_n = stat_anz - 1))
  { list_prog(kommandout, &r); }
  more_mod = more_mod_save;
}

#endif
