/*****************************************************************************
  Project: PARZ - Parallel Intermediate Code Debugger/Interpreter
  ----------------------------------------------------------------------------
  Release      : 1
  Project Part : Simulation
  Filename     : do_1fu.c       
  SCCS-Path    : /tmp_mnt/user/sembach/parz/v2/SCCS/s.do_1fu.c
  Version      : 1.3 
  Last changed : 9/27/93 at 14:04:46        
  Author       : Frank Sembach
  Creation date: Aug. 92
  ----------------------------------------------------------------------------
  Description  : Funktionen zur Simulation der PARZ-Befehle

******************************************************************************
***      (C) COPYRIGHT University of Stuttgart - All Right Reserved        ***
*****************************************************************************/

static char sccs_id[] = "@(#)do_1fu.c	1.3  9/27/93 PARZ - Simulation (Frank Sembach)";


#include "parzdefs.h"
#include "rundefs.h"
#include "komdefs.h"
#include "y_tab.h"
#include "externs.h"
#include "runexts.h"
#include "toup.h"

extern FILE *yyerfp;            /* Ausgabedatei fuer Fehlermeldungen */

extern char *wert_string();     /* aus Datei print_fu.c */

/**************************************************************************
 ***                      Funktion do_return
 ***
 *** fuehrt RETURN-Befehl aus,
 *** wird als Vektorbefehl protokolliert, wenn
 ***    1. Prozessorzustaende auf dem Stapel lagen oder
 ***    2. Vektorvariablen deklariert waren.
 ***
 **************************************************************************/

int do_return(bef)
STAT *bef;
{ char *parpush;
  int vecflag;
  register int r;

  if (akt_tiefe < 1)
  { runerr(texte[93]); return; }
  r = akt_return(&parpush, &vecflag);
  call_count--;
  if (parpush)
  { protokoll('R', aktive_pes);
    free(aktive_pes); aktive_pes = parpush; last_akt_pe = last(aktive_pes,&anz_akt_pes);
  }
  else 
  { if (vecflag) protokoll('R', aktive_pes);
    else protokoll('r', NULL);
  }
  pc = r - 1;
}

/**************************************************************************
 ***   Funktionen do_halt, do_end
 ***
 *** fuehren die entsprechenden Befehle aus.
 ***
 **************************************************************************/

int do_halt(bef)
STAT *bef;
{ protokoll('h', NULL);
  ended = TRUE;
}

int do_end(bef)
STAT *bef;
{ protokoll('h', NULL);
  ended = TRUE;
}

/**************************************************************************
 ***       Funktionen do_closeinput, do_closeoutput
 ***
 *** fuehren Dateischliessbefehle aus.
 ***
 **************************************************************************/

int do_closeinput(bef)
STAT *bef;
{ protokoll('o', NULL);
  if (runin != stdin)
  { fclose(runin);
    runin = stdin;
    s_spez_vars[TERMCH_VAR].datentyp = CHA_ERLAUBT | KEINER;
  }
}

int do_closeoutput(bef)
STAT *bef;
{ protokoll('o', NULL);
  if (runout != stdout)
  { fclose(runout);
    runout = stdout;
  }
}

/**************************************************************************
 ***                      Funktionen do_txy_zuw
 ***
 *** fuehren Zuweisungs- und Typumwandlungs-Befehl 'verg := vc1' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** x bezeichnet Zieltyp
 *** y bezeichnet Quelltyp
 *** Typkennzeichner : b : BOOLEAN
 ***                   c : CHAR
 ***                   i : INTEGER
 ***                   r : REAL
 ***                   str : konstante Zeichenkette
 ***
 **************************************************************************/

int do_sbb_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('z', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, BOOL, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_b_item(ziel, quell->inhalt.b_val, uflag);
}

int do_sbi_zuw(bef)
STAT*bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('z', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, INT, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_b_item(ziel, quell->inhalt.i_val != 0l, uflag);
}

int do_sbr_zuw(bef)
STAT*bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('z', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, REAL, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_b_item(ziel, quell->inhalt.r_val != 0.0, uflag);
}

int do_scb_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('v', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, BOOL, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_c_item(ziel, quell->inhalt.b_val ? 'T' : 'F', uflag);
}

int do_scc_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('z', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, CHA, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_c_item(ziel, quell->inhalt.c_val, uflag);
}

int do_sci_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('z', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, INT, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_c_item(ziel, (unsigned char)quell->inhalt.i_val, uflag);
}

int do_scstr_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;

  protokoll('b', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  { register char *sp = quell->inhalt.s_val;
    register VARBLOCK *bl = zielbs->blocks[0];
    register int fertig;

    if ((ziel - bl->vars) + bl->von + strlen(sp) > bl->bis)
    { runerr(texte[128]); return; }
    for (fertig = FALSE; !fertig; sp++, ziel++)
    { if (!(ziel->datentyp & CHA_ERLAUBT))
      { runerr(texte[129]); return; }
      set_c_item(ziel, (unsigned char)*sp, 0);
      fertig = !(*sp);
    }
  }
}

int do_sib_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('v', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, BOOL, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_i_item(ziel, (long)quell->inhalt.b_val, uflag);
}

int do_sic_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('z', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, CHA, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_i_item(ziel, (long)quell->inhalt.c_val, uflag);
}

int do_sii_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('z', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, INT, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_i_item(ziel, quell->inhalt.i_val, uflag);
}

int do_sir_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('z', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, REAL, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_i_item(ziel, (long)quell->inhalt.r_val, uflag);
}

int do_srb_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('z', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, BOOL, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_r_item(ziel, (float)quell->inhalt.b_val, uflag);
}

int do_sri_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('z', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, INT, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_r_item(ziel, (float)quell->inhalt.i_val, uflag);
}

int do_srr_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('z', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, REAL, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_r_item(ziel, quell->inhalt.r_val, uflag);
}

int do_vbb_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('Z', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, BOOL, 91, 283);
      set_b_item(ziel, quell->inhalt.b_val, uflag);
    }
  }
}

int do_vbi_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('Z', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, INT, 91, 283);
      set_b_item(ziel, quell->inhalt.i_val != 0l, uflag);
    }
  }
}

int do_vbr_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('Z', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, REAL, 91, 283);
      set_b_item(ziel, quell->inhalt.r_val != 0.0, uflag);
      if (fperror) return;
    }
  }
}

int do_vcb_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('Z', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, BOOL, 91, 283);
      set_c_item(ziel, quell->inhalt.b_val ? 'T' : 'F', uflag);
    }
  }
}

int do_vcc_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('Z', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, CHA, 91, 283);
      set_c_item(ziel, quell->inhalt.c_val, uflag);
    }
  }
}

int do_vci_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('Z', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, INT, 91, 283);
      set_c_item(ziel, (unsigned char)quell->inhalt.i_val, uflag);
    }
  }
}

int do_vcstr_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register VARBLOCK **blpt;

  protokoll('B', aktive_pes);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  blpt = zielbs->blocks;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++, blpt += zielbs->inc)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { register char *sp = quell->inhalt.s_val;
      register VARBLOCK *bl = *blpt;
      register int fertig;
     
      if (((ziel - bl->vars) / pe_anz + bl->von + strlen(sp)) > bl->bis)
      { runerr(texte[128]); return; }
      for (fertig = FALSE; !fertig; sp++, ziel += pe_anz)
      { if (!(ziel->datentyp & CHA_ERLAUBT))
        { runerr(texte[129]); return; }
        set_c_item(ziel, (unsigned char)*sp, 0);
        fertig = !(*sp);
      }
    }
  }
}

int do_vib_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('Z', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, BOOL, 91, 283);
      set_i_item(ziel, (long)quell->inhalt.b_val, uflag);
    }
  }
}

int do_vic_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('Z', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, CHA, 91, 283);
      set_i_item(ziel, (long)quell->inhalt.c_val, uflag);
    }
  }
}

int do_vii_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('Z', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, INT, 91, 283);
      set_i_item(ziel, quell->inhalt.i_val, uflag);
    }
  }
}

int do_vir_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('Z', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, REAL, 91, 283);
      set_i_item(ziel, (long)quell->inhalt.r_val, uflag);
      if (fperror) return;
    }
  }
}

int do_vrb_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('Z', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, BOOL, 91, 283);
      set_r_item(ziel, (float)quell->inhalt.b_val, uflag);
      if (fperror) return;
    }
  }
}

int do_vri_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('Z', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, INT, 91, 283);
      set_r_item(ziel, (float)quell->inhalt.i_val, uflag);
      if (fperror) return;
    }
  }
}

int do_vrr_zuw(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('Z', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, REAL, 91, 283);
      set_r_item(ziel, quell->inhalt.r_val, uflag);
      if (fperror) return;
    }
  }
}

/**************************************************************************
 ***           Funktionen do_tx_minus
 ***
 *** fuehren Negationsbefehl 'verg := - vc1' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** x bezeichnet den Typ der Operanden :
 ***    i : INTEGER
 ***    r : REAL
 ***
 **************************************************************************/

int do_si_minus(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('a', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, INT, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_i_item(ziel,- quell->inhalt.i_val, uflag);
}

int do_sr_minus(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('a', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, REAL, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_r_item(ziel,- quell->inhalt.r_val, uflag);
}

int do_vi_minus(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('A', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, INT, 91, 283);
      set_i_item(ziel,- quell->inhalt.i_val, uflag);
    }
  }
}

int do_vr_minus(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('A', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, REAL, 91, 283);
      set_r_item(ziel,- quell->inhalt.r_val, uflag);
      if (fperror) return;
    }
  }
}

/**************************************************************************
 ***             Funktionen do_tb_not
 ***
 *** fuehren Befehl 'verg := NOT vc1' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 ***
 **************************************************************************/

int do_sb_not(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('a', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, BOOL, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_b_item(ziel, ! quell->inhalt.b_val, uflag);
}

int do_vb_not(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('A', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, BOOL, 91, 283);
      set_b_item(ziel, ! quell->inhalt.b_val, uflag);
    }
  }
}

/**************************************************************************
 ***             Funktionen do_ti_new
 ***
 *** fuehren Befehl 'verg := NEW dnew' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 ***
 **************************************************************************/

int do_si_new(bef)
STAT *bef;
{ register ITEM *ziel;
  register VARBLOCK *vb;

  protokoll('n', NULL);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  vb = make_vars(STAT_dnew(*bef), 1, &s_min_h_adr, -1, &s_heap, NULL);
  if (err) return;
  set_i_item(ziel, vb ? (long)vb->von : 0l, 0);
}

int do_vi_new(bef)
STAT *bef;
{ register ITEM *ziel;
  register long erg;
  register VARBLOCK *vb;
  register char *max_p;

  protokoll('N', aktive_pes);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  if (!(max_p = calloc((size_t)(pe_anz + 1), (size_t)sizeof(char))))
  { runerr(texte[5]); return; }
  for (erg = 0l; erg < (long)pe_anz; max_p[erg++] = '1');
  max_p[erg] = '\0';
  vb = make_vars(STAT_dnew(*bef), pe_anz, &v_min_h_adr, -1, &v_heap, max_p);
  if (err) return;
  erg = vb ? (long)vb->von : 0l;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    {  set_i_item(ziel, erg, 0); }
  }
}

/**************************************************************************
 ***           Funktion do_status
 ***
 *** fuehrt Befehl 'verg := STATUS' aus,
 *** wird als skalarer Blockbefehl protokolliert
 ***
 **************************************************************************/

int do_status(bef)
STAT *bef;
{ register int i;
  register ITEM *ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  register VARBLOCK *bl;

  protokoll('b', NULL);
  if (err) return;
  bl = zielbs->blocks[0];
  if ((ziel - bl->vars) + bl->von + pe_anz - 1 > bl->bis)
  { runerr(texte[160]); return; }
  for (i = 0; i < pe_anz; i++, ziel++)
  { if (!(ziel->datentyp & BOOL_ERLAUBT))
    { runerr(texte[161]); return; }
    set_b_item(ziel, aktive_pes[i] == '1', 0);
  }
}

/**************************************************************************
 ***                  Funktionen do_tx_random
 ***
 *** fuehren Befehl 'verg := RANDOM' aus,
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** x bezeichnet Typ:
 ***    b : BOOLEAN
 ***    c : CHAR
 ***    i : INTEGER
 ***    r : REAL
 *** Bei Vektor-RANDOM erhaelt jeder Prozessor einen eigenen Wert
 ***
 **************************************************************************/

int do_sb_random(bef)
STAT *bef;
{ register ITEM *ziel;

  protokoll('a', NULL);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_b_item(ziel, !(random() & 256), 0);
}

int do_sc_random(bef)
STAT *bef;
{ register ITEM *ziel;

  protokoll('a', NULL);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_c_item(ziel, (unsigned char)(random() % 95 + 32), 0);
}

/*long spread(i)
int i;
{ register long erg;
  register int count;

  erg = 0;
  for (count = sizeof(int) * 8 - 1; count >= 0; count--)
  { erg += (i & (1<<count)) << count; }
  return erg;
}
*/
int do_si_random(bef)
STAT *bef;
{ register ITEM *ziel;
  register long erg;

  protokoll('a', NULL);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  erg = random();
/*  if (sizeof(int) < sizeof(long))
  { erg = spread((int)erg)<<1 + spread(random()); }  */
  set_i_item(ziel, erg, 0);
}

int do_sr_random(bef)
STAT *bef;
{ register ITEM *ziel;

  protokoll('a', NULL);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_r_item(ziel, (float)random() / ((1<<(sizeof(int)*8-1)) - 1), 0);
}

int do_vb_random(bef)
STAT *bef;
{ register ITEM *ziel;

  protokoll('A', aktive_pes);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { set_b_item(ziel, !(random() & 256), 0); }
  }
}

int do_vc_random(bef)
STAT *bef;
{ register ITEM *ziel;

  protokoll('A', aktive_pes);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { set_c_item(ziel, (unsigned char)(random() % 95 + 32), 0); }
  }
}

int do_vi_random(bef)
STAT *bef;
{ register ITEM *ziel;
  register long erg;

  protokoll('A', aktive_pes);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { erg = random();
/*      if (sizeof(int) < sizeof(long))
      { erg = spread((int)erg)<<1 + spread(random()); }  */
      set_i_item(ziel, erg, 0);
    }
  }
}

int do_vr_random(bef)
STAT *bef;
{ register ITEM *ziel;

  protokoll('A', aktive_pes);
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { set_r_item(ziel, (float)random() / ((1<<(sizeof(int)*8-1)) - 1), 0);
      if (fperror) return;
    }
  }
}

/**************************************************************************
 ***         Funktion test_union (rekursiv ueber test_decl)
 ***
 *** 1) Testet die Speicherbereiche ab '*i1' und '*i2' auf Uebereinstimmung
 *** mit '*upt'. Uebereinstimmung, wenn alle Typen aus '*upt' in den
 *** Speicherbereichen erlaubt sind, sonst Fehlermeldung.
 *** 'inc1' und 'inc2' sind die Variablenlaengen in den Speicherbereichen.
 *** 2) Markiert ('MARKE' in 'datentyp') alle Speicherstellen ab '*i1', bei
 *** 'mark2 == TRUE' auch ab '*i2', deren tatsaechlicher Typ in '*upt' vorkommt.
 *** Ist 'unmark == TRUE', werden am Anfang alle Marken geloescht.
 ***
 **************************************************************************/

test_union(upt, i1, i2, inc1, inc2, unmark, mark2)
UNION_LIST *upt;
ITEM *i1, *i2;
int inc1, inc2;
int unmark, mark2;
{ register int count;
  ITEM *it1, *it2;
  register DECLIST *dl;

  if (unmark)
  { for (count = upt->utyp_anz[SUMM], it1 = i1, it2 = i2;
         count;
         count--, it1 += inc1, it2 += inc2)
    { it1->datentyp &= ~MARKE;
      it2->datentyp &= ~MARKE;
    }
  }
  for (count = upt->ucount, dl = upt->uarray; count; count--, dl++)
  { it1 = i1; it2 = i2;
    test_decl(dl, &it1, &it2, inc1, inc2, mark2);
    if (err) return;
  }
}

/**************************************************************************
 ***        Funktion test_decl (rekursiv, auch ueber test_union)
 ***
 *** 1) Testet die Speicherbereiche ab '**i1' und '**i2' auf Uebereinstimmung
 *** mit '*dpt'. Uebereinstimmung, wenn alle Typen aus '*dpt' in den
 *** Speicherbereichen erlaubt sind, sonst Fehlermeldung.
 *** 'inc1' und 'inc2' sind die Variablenlaengen in den Speicherbereichen.
 *** 2) Markiert ('MARKE' in 'datentyp') alle Speicherstellen ab '*i1', bei
 *** 'mark2 == TRUE' auch ab '*i2', deren tatsaechlicher Typ in '*dpt' vorkommt.
 *** '*i1' und '*i2' werden weitergezaehlt.
 ***
 **************************************************************************/

test_decl(dpt, i1pt, i2pt, inc1, inc2, mark2)
DECLIST *dpt;
ITEM **i1pt, **i2pt;
int inc1, inc2;
int mark2;
{ register int count;
  register DECL *dec;

  for (count = dpt->dcount, dec = dpt->darray;
       count;
       count--, dec++)
  { switch (DECL_art(*dec))
    { case FELD  :
      { register int i;
        register TYP t = DECL_t(*dec);
        register int z = DECL_zahl(*dec);

        for (i = z; i; i--, *i1pt += inc1, *i2pt += inc2)
        { if (!((*i1pt)->datentyp & ERLAUBT(t)) || !((*i2pt)->datentyp & ERLAUBT(t)))
          { runerr(texte[141]); return; }
          if (ITEM_typ(**i1pt) == t) (*i1pt)->datentyp |= MARKE;
          if (mark2 && ITEM_typ(**i2pt) == t) (*i2pt)->datentyp |= MARKE;
        }
      }
      break;
      case KLAM  :
      { register int i;
        register int w = DECL_wiederh(*dec);
        register DECLIST *di = &DECL_klamdecl(*dec);

        for (i = w; i; i--)
        { test_decl(di, i1pt, i2pt, inc1, inc2, mark2);
          if (err) return;
        }
      }
      break;
      case UNION :
      { register int len = DECL_ul(*dec).utyp_anz[SUMM];

        test_union(&DECL_ul(*dec), *i1pt, *i2pt, inc1, inc2, FALSE, mark2);
        if (err) return;
        *i1pt += len * inc1;
        *i2pt += len * inc2;
      }
    }
  }
}

/**************************************************************************
 ***                 Funktion move_as_decl (rekursiv)
 ***
 *** Kopiert einen Speicherbereich der Form 'decl'
 *** qinc, zinc : Variablenabstand im Quell- bzw. Zielbereich
 ***              '1' fuer skalar, 'pe_anz' fuer vektor 
 *** *quellpt, *zielpt : Zeiger auf Quell- bzw. Zielbereich
 ***                     werden unabhaengig von Rekursion weitergezaehlt
 *** Bei nicht uebereinstimmenden Typen : vorzeitiges Ende mit Fehlermeldung
 ***
 **************************************************************************/

move_as_decl(decl, qinc, zinc, quellpt, zielpt)
DECLIST *decl;
int qinc, zinc;
ITEM **quellpt, **zielpt;
{ register int i, count;
  register DECL *dec;

  for (dec = decl->darray, count = decl->dcount; count; dec++, count--)
                             /* Schleife ueber oberste Ebene der Deklaration */
  { switch (DECL_art(*dec))
    { case FELD :            /* Deklarationselement 'typ_t num_z' */
      { register TYP t = DECL_t(*dec);
        register int z = DECL_zahl(*dec);
        register TYP it;

        for (i = z; i; i--, *quellpt += qinc, *zielpt += zinc)
        { if ((!((*quellpt)->datentyp & ERLAUBT(t))) ||
              (!((*zielpt)->datentyp & ERLAUBT(t))))
          { runerr(texte[141]); return; }
          set_item_egal(*zielpt, *quellpt);
          if ((it = ITEM_typ(**quellpt)) != t && it != KEINER)
          { runwarn(texte[283]);
            (*zielpt)->datentyp = CH_TYP((*zielpt)->datentyp, t);
          }
          if (err) return;
        }
      }
      break ;
      case KLAM :            /* Deklarationselement 'num_w( decl_di )' */
      { register int w = DECL_wiederh(*dec);
        register DECLIST *di = &DECL_klamdecl(*dec);

        for (i = w; i; i--)           /* Schleife fuer 'w' Wiederholungen */
        { move_as_decl(di, qinc, zinc, quellpt, zielpt);
          if (err) return;
        }
      }
      break ;
      case UNION :           /* Deklarationselement 'U(decl , ... )' */
      { register int count;

        test_union(&DECL_ul(*dec), *quellpt, *zielpt, qinc, zinc, TRUE, FALSE);
        if (err) return;
        for (count = DECL_ul(*dec).utyp_anz[SUMM];
             count;
             count--, *quellpt += qinc, *zielpt += zinc)
        { set_item_egal(*zielpt, *quellpt);
          if (!((*quellpt)->datentyp & MARKE) && ITEM_typ(**quellpt) != KEINER)
          { runwarn(texte[283]);
            (*zielpt)->datentyp = CH_TYP((*zielpt)->datentyp, KEINER);
          }
        }
      }
    }
  }
}

/**************************************************************************
 ***              Funktionen do_t_blockmove
 ***
 *** fuehren Befehl 'MOVE vc2 TO verg AS dblock' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** verwendet 'move_as_decl'
 ***
 **************************************************************************/

int do_s_blockmove(bef)
STAT *bef;
{ ITEM *quell, *ziel;
  register int z;
  register VARBLOCK *bq, *bz;

  protokoll('b', NULL);
  z = STAT_dblock(*bef)->typ_zahl[SUMM];
  quell = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  bq = dummybs->blocks[0];
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  bz = zielbs->blocks[0];
  if ((quell - bq->vars) + bq->von + z - 1 > bq->bis)
  { runerr(texte[95]); return; }
  if ((ziel - bz->vars) + bz->von + z - 1 > bz->bis)
  { runerr(texte[97]); return; }
  move_as_decl(STAT_dblock(*bef), 1, 1, &quell, &ziel);
}

int do_v_blockmove(bef)
STAT *bef;
{ ITEM *quell, *ziel;
  register int z;
  register VARBLOCK **bqpt, **bzpt, *bq, *bz;
  register ITPTRS *quellits;
  register int qlen;

  protokoll('B', aktive_pes);
  z = STAT_dblock(*bef)->typ_zahl[SUMM];
  quellits = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  qlen = quellits->n_it_func == gl_item ? 1 : pe_anz;
  bqpt = dummybs->blocks;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  bzpt = zielbs->blocks;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++, bqpt += dummybs->inc, bzpt += zielbs->inc)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { bq = *bqpt; bz = *bzpt;
      if ((quell - bq->vars) / qlen + bq->von + z - 1 > bq->bis)
      { runerr(texte[95]); return; }
      if ((ziel - bz->vars) / pe_anz + bz->von + z - 1 > bz->bis)
      { runerr(texte[97]); return; }
      move_as_decl(STAT_dblock(*bef), qlen, pe_anz, &quell, &ziel);
      if (err || fperror) return;
    }
  }
}

/**************************************************************************
 ***                Funktion compare_as_decl (rekursiv)
 ***
 *** Vergleicht zwei Speicherbereiche der Form 'decl'
 *** p1inc, p2inc : Variablenabstaende in den Speicherbereichen
 ***                '1' fuer skalar, 'pe_anz' fuer vektor 
 *** *p1pt, *p2pt : Zeiger auf Speicherbereiche
 ***                werden unabhaengig von Rekursion weitergezaehlt
 *** Bei nicht uebereinstimmenden Typen : vorzeitiges Ende mit Fehlermeldung
 ***
 **************************************************************************/

     /**************************************************************************
      ***                Hilfsfunktion comp_vergleich
      ***
      **************************************************************************/

     int comp_vergleich(t, p1, p2)
     TYP t;
     ITEM *p1, *p2;
     { switch (t)
       { case BOOL : return (p1->inhalt.b_val == p2->inhalt.b_val);
         case CHA  : return (p1->inhalt.c_val == p2->inhalt.c_val);
         case INT  : return (p1->inhalt.i_val == p2->inhalt.i_val);
         case REAL : return (p1->inhalt.r_val == p2->inhalt.r_val);
         default : bug("do_1fu.c/comp_vergleich : falscher Typ");
       }
     }

int compare_as_decl(decl, p1inc, p2inc, p1pt, p2pt, uflag)
DECLIST *decl;
int p1inc, p2inc;
ITEM **p1pt, **p2pt;
int *uflag;
{ register int i, erg, count;
  register DECL *dec;

  for (dec = decl->darray, count = decl->dcount ; count; dec++, count--)
                             /* Schleife ueber oberste Ebene der Deklaration */
  { switch (DECL_art(*dec))
    { case FELD :                    /* Deklarationselement 'typ_t num_z' */
      { register TYP t = DECL_t(*dec);
        register int z = DECL_zahl(*dec);

        for (i = z; i; i--, *p1pt += p1inc, *p2pt += p2inc)
        { register TYP it1 = ITEM_typ(**p1pt), it2 = ITEM_typ(**p2pt);

          *uflag |= (*p1pt)->datentyp | (*p2pt)->datentyp;
          if ((!((*p1pt)->datentyp & ERLAUBT(t))) ||
              (!((*p2pt)->datentyp & ERLAUBT(t))))
          { runerr(texte[141]); return FALSE; }
          if (it1 != t && it1 != KEINER)
          { runwarn(texte[284]); *uflag |= UNDEF_ERG;}
          if (it2 != t && it2 != KEINER)
          { runwarn(texte[285]); *uflag |= UNDEF_ERG;}
          if (it1 == KEINER)
          { *uflag |= UNDEF_ERG;
            erg = (it2 == KEINER);
          }
          else if (it2 == KEINER)
          { *uflag |= UNDEF_ERG;
            erg = FALSE;
          }
          else
          { erg = comp_vergleich(t, *p1pt, *p2pt); }
          if (!erg) return FALSE;
        }
      }
      break;
      case KLAM :                    /* Deklarationselement 'num_w( decl_di )' */
      { register int w = DECL_wiederh(*dec);
        register DECLIST *di = &DECL_klamdecl(*dec);

        for (i = w; i; i--)           /* Schleife fuer 'w' Wiederholungen */
        { erg = compare_as_decl(di, p1inc, p2inc, p1pt, p2pt, uflag);
          if (err || !erg) return FALSE;
        }
      }
      break;
      case UNION :                   /* Deklarationselement 'U(decl , ... )' */
      { register int count;

        test_union(&DECL_ul(*dec), *p1pt, *p2pt, p1inc, p2inc, TRUE, TRUE);
        if (err) return FALSE;
        for (count = DECL_ul(*dec).utyp_anz[SUMM];
             count;
             count--, *p1pt += p1inc, *p2pt += p2inc)
        { register TYP t1 = (*p1pt)->datentyp, t2 = (*p2pt)->datentyp;
          register TYP it1 = AA_typ(t1), it2 = AA_typ(t2);

          *uflag |= t1 | t2;
          if (!(t1 & MARKE) && it1 != KEINER)
          { runwarn(texte[284]); *uflag |= UNDEF_ERG; }
          if (!(t2 & MARKE) && it2 != KEINER)
          { runwarn(texte[285]); *uflag |= UNDEF_ERG;}
          if (it1 == KEINER)
          { *uflag |= UNDEF_ERG;
            erg = (it2 == KEINER);
          }
          else if (it1 != it2)
          { *uflag |= UNDEF_ERG;
            erg = FALSE;
          }
          else
          { erg = comp_vergleich(it1, *p1pt, *p2pt); }
          if (!erg) return FALSE;
        }
      }
    }
  }
  return TRUE;
}

/**************************************************************************
 ***              Funktionen do_t_blockequal
 ***
 *** fuehren Befehl 'EQUAL vc2 vc3 AS dblock' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** verwendet 'compare_as_decl'
 *** Ergebnis steht in Variable 'SResult' bzw. 'VResult'
 ***
 **************************************************************************/

int do_s_blockequal(bef)
STAT *bef;
{ ITEM *p1, *p2;
  register int z;
  register VARBLOCK *bp1, *bp2;
  int uflag = 0;

  protokoll('b', NULL);
  z = STAT_dblock(*bef)->typ_zahl[SUMM];
  p1 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  bp1 = dummybs->blocks[0];
  p2 = item_ptr(&STAT_vc3(*bef), &zielbs, disp)->itps[0];
  if (err) return;
  bp2 = zielbs->blocks[0];
  if ((p1 - bp1->vars) + bp1->von + z - 1 > bp1->bis)
  { runerr(texte[105]); return; }
  if ((p2 - bp2->vars) + bp2->von + z - 1 > bp2->bis)
  { runerr(texte[106]); return; }
  z = compare_as_decl(STAT_dblock(*bef), 1, 1, &p1, &p2, &uflag);
  set_b_item(&s_spez_vars[SRESULT_VAR], z, uflag);
}

int do_v_blockequal(bef)
STAT *bef;
{ ITEM *p1, *p2;
  register int z, erg;
  register VARBLOCK **bp1pt, **bp2pt, *bp1, *bp2;
  register ITPTRS *p1its, *p2its;
  register int p1len, p2len;
  int uflag;

  protokoll('B', aktive_pes);
  z = STAT_dblock(*bef)->typ_zahl[SUMM];
  p1its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  p1len = p1its->n_it_func == gl_item ? 1 : pe_anz;
  bp1pt = dummybs->blocks;
  p2its = item_ptr(&STAT_vc3(*bef), &zielbs, disp);
  if (err) return;
  p2len = p2its->n_it_func == gl_item ? 1 : pe_anz;
  bp2pt = zielbs->blocks;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++, bp1pt += dummybs->inc, bp2pt += zielbs->inc)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    if (aktive_pes[pec] == '1')
    { bp1 = *bp1pt; bp2 = *bp2pt;
      if ((p1 - bp1->vars) / p1len + bp1->von + z - 1 > bp1->bis)
      { runerr(texte[105]); return; }
      if ((p2 - bp2->vars) / p2len + bp2->von + z - 1 > bp2->bis)
      { runerr(texte[106]); return; }
      uflag = 0;
      erg = compare_as_decl(STAT_dblock(*bef), p1len, p2len, &p1, &p2, &uflag);
      set_b_item(v_spez_vars + pe_anz * VRESULT_VAR + pec, erg, uflag);
      if (err || fperror) return;
    }
  }
}

/**************************************************************************
 ***               Funktionen do_tx_add
 ***
 *** fuehren Befehl 'verg := vc1 + vc2' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** x bezeichnet Typ:
 ***    i : INTEGER
 ***    r : REAL
 ***
 **************************************************************************/

int do_si_add(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register int uflag;

  protokoll('a', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, INT, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, INT, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_i_item(ziel, p1->inhalt.i_val + p2->inhalt.i_val, uflag);
}

int do_sr_add(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register int uflag;

  protokoll('a', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, REAL, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, REAL, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_r_item(ziel, p1->inhalt.r_val + p2->inhalt.r_val, uflag);
}

int do_vi_add(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register int uflag;

  protokoll('A', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, INT, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, INT, 102, 285);
      set_i_item(ziel, p1->inhalt.i_val + p2->inhalt.i_val, uflag);
    }
  }
}

int do_vr_add(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register int uflag;

  protokoll('A', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, REAL, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, REAL, 102, 285);
      set_r_item(ziel, p1->inhalt.r_val + p2->inhalt.r_val, uflag);
      if (fperror) return;
    }
  }
}

/**************************************************************************
 ***               Funktionen do_tx_sub
 ***
 *** fuehren Befehl 'verg := vc1 - vc2' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** x bezeichnet Typ:
 ***    i : INTEGER
 ***    r : REAL
 ***
 **************************************************************************/

int do_si_sub(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register int uflag;

  protokoll('a', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, INT, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, INT, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_i_item(ziel, p1->inhalt.i_val - p2->inhalt.i_val, uflag);
}

int do_sr_sub(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register int uflag;

  protokoll('a', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, REAL, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, REAL, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_r_item(ziel, p1->inhalt.r_val - p2->inhalt.r_val, uflag);
}

int do_vi_sub(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register int uflag;

  protokoll('A', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, INT, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, INT, 102, 285);
      set_i_item(ziel, p1->inhalt.i_val - p2->inhalt.i_val, uflag);
    }
  }
}

int do_vr_sub(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register int uflag;

  protokoll('A', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, REAL, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, REAL, 102, 285);
      set_r_item(ziel, p1->inhalt.r_val - p2->inhalt.r_val, uflag);
      if (fperror) return;
    }
  }
}

