/*****************************************************************************
  Project: PARZ - Parallel Intermediate Code Debugger/Interpreter
  ----------------------------------------------------------------------------
  Release      : 1
  Project Part : Simulation
  Filename     : do_2fu.c       
  SCCS-Path    : /tmp_mnt/user/sembach/parz/v2/SCCS/s.do_2fu.c
  Version      : 1.3 
  Last changed : 9/27/93 at 14:04:47        
  Author       : Frank Sembach
  Creation date: Aug. 92
  ----------------------------------------------------------------------------
  Description  : Funktionen zur Simulation der PARZ-Befehle

******************************************************************************
***      (C) COPYRIGHT University of Stuttgart - All Right Reserved        ***
*****************************************************************************/

static char sccs_id[] = "@(#)do_2fu.c	1.3  9/27/93 PARZ - Simulation (Frank Sembach)";


#include "parzdefs.h"
#include "rundefs.h"
#include "komdefs.h"
#include "y_tab.h"
#include "externs.h"
#include "runexts.h"
#include "toup.h"

extern FILE *yyerfp;            /* Ausgabedatei fuer Fehlermeldungen */

extern char *wert_string();     /* aus Datei print_fu.c */

/**************************************************************************
 ***              Funktionen do_tx_mul
 ***
 *** fuehren Befehl 'verg := vc1 * vc2' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** x bezeichnet Typ:
 ***    i : INTEGER
 ***    r : REAL
 ***
 **************************************************************************/

int do_si_mul(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register int uflag;

  protokoll('a', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, INT, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, INT, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_i_item(ziel, p1->inhalt.i_val * p2->inhalt.i_val, uflag);
}

int do_sr_mul(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register int uflag;

  protokoll('a', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, REAL, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, REAL, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_r_item(ziel, p1->inhalt.r_val * p2->inhalt.r_val, uflag);
}

int do_vi_mul(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register int uflag;

  protokoll('A', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, INT, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, INT, 102, 285);
      set_i_item(ziel, p1->inhalt.i_val * p2->inhalt.i_val, uflag);
    }
  }
}

int do_vr_mul(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register int uflag;

  protokoll('A', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, REAL, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, REAL, 102, 285);
      set_r_item(ziel, p1->inhalt.r_val * p2->inhalt.r_val, uflag);
      if (fperror) return;
    }
  }
}

/**************************************************************************
 ***              Funktionen do_tx_div
 ***
 *** fuehren Befehl 'verg := vc1 / vc2' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** x bezeichnet Typ:
 ***    i : INTEGER
 ***    r : REAL
 ***
 **************************************************************************/

int do_si_div(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register int uflag;

  protokoll('a', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, INT, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, INT, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  if (!p2->inhalt.i_val)
  { runerr(texte[151]); return; }
  set_i_item(ziel, p1->inhalt.i_val / p2->inhalt.i_val, uflag);
}

int do_sr_div(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register int uflag;

  protokoll('a', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, REAL, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, REAL, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  if (!p2->inhalt.r_val)
  { runerr(texte[151]); return; }
  set_r_item(ziel, p1->inhalt.r_val / p2->inhalt.r_val, uflag);
}

int do_vi_div(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register int uflag;

  protokoll('A', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, INT, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, INT, 102, 285);
      if (!p2->inhalt.i_val)
      { runerr(texte[151]); return; }
      set_i_item(ziel, p1->inhalt.i_val / p2->inhalt.i_val, uflag);
    }
  }
}

int do_vr_div(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register int uflag;

  protokoll('A', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, REAL, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, REAL, 102, 285);
      if (!p2->inhalt.r_val)
      { runerr(texte[151]); return; }
      set_r_item(ziel, p1->inhalt.r_val / p2->inhalt.r_val, uflag);
      if (fperror) return;
    }
  }
}

/**************************************************************************
 ***            Funktion ihochi
 ***
 *** berechnet ib ^ ie
 *** Fehlermeldung bei: neg. Exponent,
 ***                    0 ^ 0
 ***
 **************************************************************************/

long ihochi(ib, ie)
long ib, ie;
{ register int i;
  register long erg;

  if (ie < 0)
  { runerr(texte[147]); return 0l; }
  if (!ib && !ie)
  { runerr(texte[148]); return 0l; }
  if (!ie) return 1l;
  if (!ib) return 0l;
  for (erg = 1; ie; ie >>= 1, ib *= ib)
  { if (ie & 1) erg *= ib; }
  return erg;
}

/**************************************************************************
 ***             Funktion rhochi
 ***
 *** berechnet rb ^ ie
 *** Fehlermeldung bei: 0 ^ 0
 ***                    0 ^ neg. Exponent
 ***
 **************************************************************************/

float rhochi(rb, ie)
float rb;
long ie;
{ register int i;
  register long exp;
  float erg;

  if (!rb && !ie)
  { runerr(texte[148]); return 0.0; }
  if (!rb && ie < 0)
  { runerr(texte[150]); return 0.0; }
  if (!ie) return 1.0;
  if (!rb) return 0.0;
  for (erg = 1.0, exp = labs(ie); exp; exp >>= 1, rb *= rb)
  { if (fperror) break;
    if (exp & 1) erg *= rb;
    if (fperror) break;
  }
  if (ie < 0)
  { if (fperror)
    { fperror = FALSE; return 0.0; }
    else return 1.0 / erg;
  }
  else if (fperror)
  { return HUGE; }
  else
  { return erg; }
}

/**************************************************************************
 ***              Funktion rhochr
 ***
 *** berechnet rb ^ re
 *** Fehlermeldung bei: neg. Basis,
 ***                    0 ^ 0,
 ***                    0 ^ neg.Exponent
 ***
 **************************************************************************/

float rhochr(rb, re)
float rb, re;
{ float erg;

  if (rb < 0)
  { runerr(texte[149]); return HUGE; }
  if (!rb && !re)
  { runerr(texte[148]); return 0; }
  if (!rb && re < 0)
  { runerr(texte[150]); return 0; }
  if (!re) return 1.0;
  if (!rb) return 0.0;
  errno = 0;
  erg = pow(rb,re);
  if (erg && (errno == ERANGE || errno == EDOM)) fperror = TRUE;
  return erg;
}

/**************************************************************************
 ***            Funktionen do_txy_pow
 ***
 *** fuehren Befehl 'verg := vc1 ^ vc2' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** x bezeichnet Basistyp
 *** y bezeichnet Exponententyp
 *** Typkennzeichner : i : INTEGER
 ***                   r : REAL
 ***
 **************************************************************************/

int do_sii_pow(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register int uflag;

  protokoll('a', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, INT, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, INT, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_i_item(ziel, ihochi(p1->inhalt.i_val, p2->inhalt.i_val), uflag);
}

int do_sri_pow(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register int uflag;

  protokoll('a', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, REAL, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, INT, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_r_item(ziel, rhochi(p1->inhalt.r_val, p2->inhalt.i_val), uflag);
}

int do_srr_pow(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register int uflag;

  protokoll('a', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, REAL, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, REAL, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_r_item(ziel, rhochr(p1->inhalt.r_val, p2->inhalt.r_val), uflag);
}

int do_vii_pow(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register int uflag;

  protokoll('A', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, INT, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, INT, 102, 285);
      set_i_item(ziel, ihochi(p1->inhalt.i_val, p2->inhalt.i_val), uflag);
      if (err) return;
    }
  }
}

int do_vri_pow(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register int uflag;

  protokoll('A', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, REAL, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, INT, 102, 285);
      set_r_item(ziel, rhochi(p1->inhalt.r_val, p2->inhalt.i_val), uflag);
      if (fperror || err) return;
    }
  }
}

int do_vrr_pow(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register int uflag;

  protokoll('A', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, REAL, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, REAL, 102, 285);
      set_r_item(ziel, rhochr(p1->inhalt.r_val, p2->inhalt.r_val), uflag);
      if (fperror || err) return;
    }
  }
}

/**************************************************************************
 ***             Funktionen do_ti_mod
 ***
 *** fuehren Befehl 'verg := vc1 MOD vc2' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 ***
 **************************************************************************/

int do_si_mod(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register long erg;
  register int uflag;

  protokoll('a', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, INT, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, INT, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  if (!p2->inhalt.i_val)
  { runerr(texte[151]); return; }
  set_i_item(ziel, (erg = p1->inhalt.i_val % p2->inhalt.i_val) < 0
                   ? erg + p2->inhalt.i_val
                   : erg, uflag);
}

int do_vi_mod(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register long erg;
  register ITPTRS *p1its, *p2its;
  register int uflag;

  protokoll('A', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, INT, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, INT, 102, 285);
      if (!p2->inhalt.i_val)
      { runerr(texte[151]); return; }
      set_i_item(ziel, (erg = p1->inhalt.i_val % p2->inhalt.i_val) < 0
                       ? erg + p2->inhalt.i_val
                       : erg, uflag);
    }
  }
}

/**************************************************************************
 ***            Funktionen do_tb_and
 ***
 *** fuehren Befehl 'verg := vc1 AND vc2' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 ***
 **************************************************************************/

int do_sb_and(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register int uflag;

  protokoll('a', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, BOOL, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, BOOL, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_b_item(ziel, p1->inhalt.b_val && p2->inhalt.b_val, uflag);
}

int do_vb_and(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register int uflag;

  protokoll('A', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, BOOL, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, BOOL, 102, 285);
      set_b_item(ziel, p1->inhalt.b_val && p2->inhalt.b_val, uflag);
    }
  }
}

/**************************************************************************
 ***            Funktionen do_tb_or
 ***
 *** fuehren Befehl 'verg := vc1 OR vc2' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 ***
 **************************************************************************/

int do_sb_or(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register int uflag;

  protokoll('a', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, BOOL, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, BOOL, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_b_item(ziel, p1->inhalt.b_val || p2->inhalt.b_val, uflag);
}

int do_vb_or(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register int uflag;

  protokoll('A', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, BOOL, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, BOOL, 102, 285);
      set_b_item(ziel, p1->inhalt.b_val || p2->inhalt.b_val, uflag);
    }
  }
}

/**************************************************************************
 ***                  Funktionen do_t_op
 ***
 *** fuehren Vergleichsbefehle 'verg := vc1 rel vc2' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** op = eq   : rel = '='
 *** op = ne   : rel = '<>'
 *** op = lt   : rel = '<'
 *** op = le   : rel = '<='
 *** op = gt   : rel = '>'
 *** op = ge   : rel = '>='
 ***
 **************************************************************************/

int do_s_eq(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));
  register int uflag;

  protokoll('v', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, t, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, t, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  switch (t)
  { case BOOL : set_b_item(ziel, p1->inhalt.b_val == p2->inhalt.b_val, uflag);
                break;
    case CHA  : set_b_item(ziel, p1->inhalt.c_val == p2->inhalt.c_val, uflag);
                break;
    case INT  : set_b_item(ziel, p1->inhalt.i_val == p2->inhalt.i_val, uflag);
                break;
    case REAL : set_b_item(ziel, p1->inhalt.r_val == p2->inhalt.r_val, uflag);
                break;
    default : bug("do_2fu.c/do_s_eq : falscher Typ");
  }
}

int do_v_eq(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));
  register int uflag;

  protokoll('V', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, t, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, t, 102, 285);
      switch (t)
      { case BOOL : set_b_item(ziel, p1->inhalt.b_val == p2->inhalt.b_val, uflag);
                    break;
        case CHA  : set_b_item(ziel, p1->inhalt.c_val == p2->inhalt.c_val, uflag);
                    break;
        case INT  : set_b_item(ziel, p1->inhalt.i_val == p2->inhalt.i_val, uflag);
                    break;
        case REAL : set_b_item(ziel, p1->inhalt.r_val == p2->inhalt.r_val, uflag);
                    break;
        default : bug("do_2fu.c/do_v_eq : falscher Typ");
      }
      if (fperror) return;
    }
  }
}

int do_s_ne(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));
  register int uflag;

  protokoll('v', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, t, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, t, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  switch (t)
  { case BOOL : set_b_item(ziel, p1->inhalt.b_val != p2->inhalt.b_val, uflag);
                break;
    case CHA  : set_b_item(ziel, p1->inhalt.c_val != p2->inhalt.c_val, uflag);
                break;
    case INT  : set_b_item(ziel, p1->inhalt.i_val != p2->inhalt.i_val, uflag);
                break;
    case REAL : set_b_item(ziel, p1->inhalt.r_val != p2->inhalt.r_val, uflag);
                break;
    default : bug("do_2fu.c/do_s_ne : falscher Typ");
  }
}

int do_v_ne(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));
  register int uflag;

  protokoll('V', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, t, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, t, 102, 285);
      switch (t)
      { case BOOL : set_b_item(ziel, p1->inhalt.b_val != p2->inhalt.b_val, uflag);
                    break;
        case CHA  : set_b_item(ziel, p1->inhalt.c_val != p2->inhalt.c_val, uflag);
                    break;
        case INT  : set_b_item(ziel, p1->inhalt.i_val != p2->inhalt.i_val, uflag);
                    break;
        case REAL : set_b_item(ziel, p1->inhalt.r_val != p2->inhalt.r_val, uflag);
                    break;
        default : bug("do_2fu.c/do_v_ne : falscher Typ");
      }
      if (fperror) return;
    }
  }
}

int do_s_lt(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));
  register int uflag;

  protokoll('v', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, t, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, t, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  switch (t)
  { case BOOL : set_b_item(ziel, p1->inhalt.b_val < p2->inhalt.b_val, uflag);
                break;
    case CHA  : set_b_item(ziel, p1->inhalt.c_val < p2->inhalt.c_val, uflag);
                break;
    case INT  : set_b_item(ziel, p1->inhalt.i_val < p2->inhalt.i_val, uflag);
                break;
    case REAL : set_b_item(ziel, p1->inhalt.r_val < p2->inhalt.r_val, uflag);
                break;
    default : bug("do_2fu.c/do_s_lt : falscher Typ");
  }
}

int do_v_lt(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));
  register int uflag;

  protokoll('V', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, t, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, t, 102, 285);
      switch (t)
      { case BOOL : set_b_item(ziel, p1->inhalt.b_val < p2->inhalt.b_val, uflag);
                    break;
        case CHA  : set_b_item(ziel, p1->inhalt.c_val < p2->inhalt.c_val, uflag);
                    break;
        case INT  : set_b_item(ziel, p1->inhalt.i_val < p2->inhalt.i_val, uflag);
                    break;
        case REAL : set_b_item(ziel, p1->inhalt.r_val < p2->inhalt.r_val, uflag);
                    break;
        default : bug("do_2fu.c/do_v_lt : falscher Typ");
      }
      if (fperror) return;
    }
  }
}

int do_s_le(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));
  register int uflag;

  protokoll('v', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, t, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, t, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  switch (t)
  { case BOOL : set_b_item(ziel, p1->inhalt.b_val <= p2->inhalt.b_val, uflag);
                break;
    case CHA  : set_b_item(ziel, p1->inhalt.c_val <= p2->inhalt.c_val, uflag);
                break;
    case INT  : set_b_item(ziel, p1->inhalt.i_val <= p2->inhalt.i_val, uflag);
                break;
    case REAL : set_b_item(ziel, p1->inhalt.r_val <= p2->inhalt.r_val, uflag);
                break;
    default : bug("do_2fu.c/do_s_le : falscher Typ");
  }
}

int do_v_le(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));
  register int uflag;

  protokoll('V', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, t, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, t, 102, 285);
      switch (t)
      { case BOOL : set_b_item(ziel, p1->inhalt.b_val <= p2->inhalt.b_val, uflag);
                    break;
        case CHA  : set_b_item(ziel, p1->inhalt.c_val <= p2->inhalt.c_val, uflag);
                    break;
        case INT  : set_b_item(ziel, p1->inhalt.i_val <= p2->inhalt.i_val, uflag);
                    break;
        case REAL : set_b_item(ziel, p1->inhalt.r_val <= p2->inhalt.r_val, uflag);
                    break;
        default : bug("do_2fu.c/do_v_le : falscher Typ");
      }
      if (fperror) return;
    }
  }
}

int do_s_gt(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));
  register int uflag;

  protokoll('v', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, t, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, t, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  switch (t)
  { case BOOL : set_b_item(ziel, p1->inhalt.b_val > p2->inhalt.b_val, uflag);
                break;
    case CHA  : set_b_item(ziel, p1->inhalt.c_val > p2->inhalt.c_val, uflag);
                break;
    case INT  : set_b_item(ziel, p1->inhalt.i_val > p2->inhalt.i_val, uflag);
                break;
    case REAL : set_b_item(ziel, p1->inhalt.r_val > p2->inhalt.r_val, uflag);
                break;
    default : bug("do_2fu.c/do_s_gt : falscher Typ");
  }
}

int do_v_gt(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));
  register int uflag;

  protokoll('V', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, t, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, t, 102, 285);
      switch (t)
      { case BOOL : set_b_item(ziel, p1->inhalt.b_val > p2->inhalt.b_val, uflag);
                    break;
        case CHA  : set_b_item(ziel, p1->inhalt.c_val > p2->inhalt.c_val, uflag);
                    break;
        case INT  : set_b_item(ziel, p1->inhalt.i_val > p2->inhalt.i_val, uflag);
                    break;
        case REAL : set_b_item(ziel, p1->inhalt.r_val > p2->inhalt.r_val, uflag);
                    break;
        default : bug("do_2fu.c/do_v_gt : falscher Typ");
      }
      if (fperror) return;
    }
  }
}

int do_s_ge(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));
  register int uflag;

  protokoll('v', NULL);
  p1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(p1->datentyp, t, 101, 284);
  p2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(p2->datentyp, t, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  switch (t)
  { case BOOL : set_b_item(ziel, p1->inhalt.b_val >= p2->inhalt.b_val, uflag);
                break;
    case CHA  : set_b_item(ziel, p1->inhalt.c_val >= p2->inhalt.c_val, uflag);
                break;
    case INT  : set_b_item(ziel, p1->inhalt.i_val >= p2->inhalt.i_val, uflag);
                break;
    case REAL : set_b_item(ziel, p1->inhalt.r_val >= p2->inhalt.r_val, uflag);
                break;
    default : bug("do_2fu.c/do_s_ge : falscher Typ");
  }
}

int do_v_ge(bef)
STAT *bef;
{ register ITEM *p1, *p2, *ziel;
  register ITPTRS *p1its, *p2its;
  register TYP t = ARG_valtyp(STAT_vc1(*bef));
  register int uflag;

  protokoll('V', aktive_pes);
  p1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  p2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { p1 = (* p1its->n_it_func)(p1its);
    p2 = (* p2its->n_it_func)(p2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(p1->datentyp, t, 101, 284);
      uflag |= ini_typ_test(p2->datentyp, t, 102, 285);
      switch (t)
      { case BOOL : set_b_item(ziel, p1->inhalt.b_val >= p2->inhalt.b_val, uflag);
                    break;
        case CHA  : set_b_item(ziel, p1->inhalt.c_val >= p2->inhalt.c_val, uflag);
                    break;
        case INT  : set_b_item(ziel, p1->inhalt.i_val >= p2->inhalt.i_val, uflag);
                    break;
        case REAL : set_b_item(ziel, p1->inhalt.r_val >= p2->inhalt.r_val, uflag);
                    break;
        default : bug("do_2fu.c/do_v_ge : falscher Typ");
      }
      if (fperror) return;
    }
  }
}

/**************************************************************************
 ***            Funktion sr_math
 ***
 *** fuehrt Skalarbefehl 'verg := func vc1' aus:
 *** 'func' ist Zeiger auf mathematische Funktion.
 *** Wird aufgerufen von den Funktionen 'do_sr_func'
 ***
 **************************************************************************/

sr_math(bef,func,tnum)
STAT *bef;
double (* func)();
int tnum;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('a', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, REAL, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  errno = 0;
  set_r_item(ziel, (* func)(quell->inhalt.r_val), uflag);
  if (errno == EDOM || errno == ERANGE)
  { runerr(texte[257],texte[tnum],quell->inhalt.r_val); fperror = FALSE; }
}

/**************************************************************************
 ***            Funktion vr_math
 ***
 *** fuehrt Vektorbefehl 'verg := func vc1' aus:
 *** 'func' ist Zeiger auf mathematische Funktion.
 *** Wird aufgerufen von den Funktionen 'do_vr_func'
 ***
 **************************************************************************/

vr_math(bef,func,tnum)
STAT *bef;
double (* func)();
int tnum;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('A', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, REAL, 91, 283);
      errno = 0;
      set_r_item(ziel, (* func)(quell->inhalt.r_val), uflag);
      if (errno == EDOM || errno == ERANGE)
      { runerr(texte[257],texte[tnum],quell->inhalt.r_val); fperror = FALSE; }
      if (fperror) return;
    }
  }
}

/**************************************************************************
 ***            Funktionen do_tr_func
 ***
 *** fuehren Befehl 'verg := func vc1' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** func bezeichnet mathematische Funktion
 ***   sqrt : Wurzel, exp : e-Potenz, ln : natuerlicher Logarithmus
 ***   sin, cos, tan : trigonometrische Funktionen
 ***   arcsin, arccos, arctan : trigonometrische Umkehrfunktionen
 ***
 **************************************************************************/

int do_sr_sqrt(bef)
STAT *bef;
{ sr_math(bef,sqrt,259); }

int do_vr_sqrt(bef)
STAT *bef;
{ vr_math(bef,sqrt,259); }

int do_sr_exp(bef)
STAT *bef;
{ sr_math(bef,exp,260); }

int do_vr_exp(bef)
STAT *bef;
{ vr_math(bef,exp,260); }

int do_sr_ln(bef)
STAT *bef;
{ sr_math(bef,log,261); }

int do_vr_ln(bef)
STAT *bef;
{ vr_math(bef,log,261); }

int do_sr_sin(bef)
STAT *bef;
{ sr_math(bef,sin,262); }

int do_vr_sin(bef)
STAT *bef;
{ vr_math(bef,sin,262); }

int do_sr_cos(bef)
STAT *bef;
{ sr_math(bef,cos,263); }

int do_vr_cos(bef)
STAT *bef;
{ vr_math(bef,cos,263); }

int do_sr_tan(bef)
STAT *bef;
{ sr_math(bef,tan,264); }

int do_vr_tan(bef)
STAT *bef;
{ vr_math(bef,tan,264); }

int do_sr_arcsin(bef)
STAT *bef;
{ sr_math(bef,asin,265); }

int do_vr_arcsin(bef)
STAT *bef;
{ vr_math(bef,asin,265); }

int do_sr_arccos(bef)
STAT *bef;
{ sr_math(bef,acos,266); }

int do_vr_arccos(bef)
STAT *bef;
{ vr_math(bef,acos,266); }

int do_sr_arctan(bef)
STAT *bef;
{ sr_math(bef,atan,267); }

int do_vr_arctan(bef)
STAT *bef;
{ vr_math(bef,atan,267); }

/**************************************************************************
 ***            Funktionen do_tr_arctant
 ***
 *** fuehren Befehl 'verg := arctant vc1 vc2' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** Mathematische Bedeutung : arctan(vc1/vc2) (-PI < Erg < PI)
 ***
 **************************************************************************/

int do_sr_arctant(bef)
STAT *bef;
{ register ITEM *arg1, *arg2, *ziel;
  register int uflag;

  protokoll('a', NULL);
  arg1 = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(arg1->datentyp, REAL, 101, 284);
  arg2 = item_ptr(&STAT_vc2(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag |= ini_typ_test(arg2->datentyp, REAL, 102, 285);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  errno = 0;
  set_r_item(ziel, atan2(arg1->inhalt.r_val,arg2->inhalt.r_val), uflag);
  if (errno == EDOM || errno == ERANGE)
  { runerr(texte[258],arg1->inhalt.r_val,arg2->inhalt.r_val); fperror = FALSE; }
}

int do_vr_arctant(bef)
STAT *bef;
{ register ITEM *arg1, *arg2, *ziel;
  register ITPTRS *arg1its, *arg2its;
  register int uflag;

  protokoll('A', aktive_pes);
  arg1its = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  arg2its = item_ptr(&STAT_vc2(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { arg1 = (* arg1its->n_it_func)(arg1its);
    arg2 = (* arg2its->n_it_func)(arg2its);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(arg1->datentyp, REAL, 101, 284);
      uflag |= ini_typ_test(arg2->datentyp, REAL, 102, 285);
      errno = 0;
      set_r_item(ziel, atan2(arg1->inhalt.r_val,arg2->inhalt.r_val), uflag);
      if (errno == EDOM || errno == ERANGE)
      { runerr(texte[258],arg1->inhalt.r_val,arg2->inhalt.r_val); fperror = FALSE; }
      if (fperror) return;
    }
  }
}

/**************************************************************************
 ***            Funktionen do_tx_abs
 ***
 *** fuehren Befehl 'verg := ABS vc1' aus:
 *** t = s : Skalarbefehl, t = v : Vektorbefehl
 *** x bezeichnet Typ:
 ***    i : INTEGER
 ***    r : REAL
 ***
 **************************************************************************/

int do_si_abs(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register int uflag;

  protokoll('a', NULL);
  quell = item_ptr(&STAT_vc1(*bef), &dummybs, disp)->itps[0];
  if (err) return;
  uflag = ini_typ_test(quell->datentyp, INT, 91, 283);
  ziel = (zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp))->itps[0];
  if (err) return;
  set_i_item(ziel, labs(quell->inhalt.i_val), uflag);
}

int do_sr_abs(bef)
STAT *bef;
{ sr_math(bef,fabs,268); }

int do_vi_abs(bef)
STAT *bef;
{ register ITEM *quell, *ziel;
  register ITPTRS *quellits;
  register int uflag;

  protokoll('A', aktive_pes);
  quellits = item_ptr(&STAT_vc1(*bef), &dummybs, disp);
  if (err) return;
  zielits = item_ptr(&STAT_verg(*bef), &zielbs, disp);
  if (err) return;
  vec_bef = TRUE;
  for (pec = 0; pec < last_akt_pe; pec++)
  { quell = (* quellits->n_it_func)(quellits);
    ziel = (* zielits->n_it_func)(zielits);
    if (aktive_pes[pec] == '1')
    { uflag = ini_typ_test(quell->datentyp, INT, 91, 283);
      set_i_item(ziel, labs(quell->inhalt.i_val), uflag);
    }
  }
}

int do_vr_abs(bef)
STAT *bef;
{ vr_math(bef,fabs,268); }

