/*****************************************************************************
  Project: PARZ - Parallel Intermediate Code Debugger/Interpreter
  ----------------------------------------------------------------------------
  Release      : 1
  Project Part : Kommandointerpreter
  Filename     : kommand.c       
  SCCS-Path    : /tmp_mnt/user/sembach/parz/v2/SCCS/s.kommand.c
  Version      : 1.3 
  Last changed : 9/27/93 at 14:04:55        
  Author       : Frank Sembach
  Creation date: Aug. 92
  ----------------------------------------------------------------------------
  Description  : Funktionen des Kommandointerpreters fuer PARZ

******************************************************************************
***      (C) COPYRIGHT University of Stuttgart - All Right Reserved        ***
*****************************************************************************/

static char sccs_id[] = "@(#)kommand.c	1.3  9/27/93 PARZ - Kommandointerpreter (Frank Sembach)";


#ifndef MAC
#include <sys/types.h>
#include <sys/stat.h>
#endif
#include "parzdefs.h"
#include "komdefs.h"
#include "y_tab.h"
#include "externs.h"
#include "rundefs.h"
#include "runexts.h"
#include "debexts.h"

char stepping_to;               /* 'STEP TO label'-Kommando wurde ausgefuehrt */
char beim_listing;              /* CHAR-Ausgabe wie im Listing */
char ganzes_listing;            /* Leerzeilen vor und hinter manchen Befehlen */

VARBLOCK *setbl;                /* Variablenblock fuer 'SET'-Befehl */
int setvarlen;                  /* 1 fuer Skalarvar., pe_anz fuer Vektorvar.
                                   noetig fuer Stringzuweisung */

char *default_pelist = NULL;    /* PEs die standardmaessig von DEBUG, TRACE und
                                   Kommandointerpreter betrachtet werden */
char *pe_list_string = NULL;    /* PEs die in Kommandozeile ausgewaehlt wurden */

STAT *kommand_traces = NULL;    /* TRACE-Befehle die im Kommandointerpreter
                                   eingegeben wurden */
int akt_kom_tr = 0;             /* Zahl der Befehle in kommand_traces */
int max_kom_tr = 0;             /* Groesse des Arrays kommand_traces
                                   wird bei Bedarf verlaengert */

char typ_char[REAL + 1] = {'B', 'C', 'I', 'R'};
char *var_nam[VRESULT - MAXTRANS + 1] =
  {"MaxTrans", "ActTrans", "Done", "termCH", "SResult", "ID", "VResult"};

char loadfilename[257];         /* Name der zu ladenden bzw. compilierenden Datei */

/*************************************************************************/

int max_item_breite;            /* Maximale Breite einer INTEGER- oder REAL-Zahl */

int *to_break_lines = NULL;     /* Befehlsnummern, bei denen nach STEP TO und GO TO
                                   angehalten wird */
int to_count = 0;               /* Zahl der Zeilen in 'to_break_lines' */

char block_breaked;             /* gibt an, ob schon eine Meldung Funktionsabbruch
                                   ausgegeben wurde */
extern FILE *yyin;              /* Eingabefile fuer Parser */

extern char *wert_string();     /* liefert Ausgabestring zu einem ITEM */

extern int print_proc();
extern int do_trace(), print_trace();

/**************************************************************************
 ***                  Funktion init_breite
 ***
 *** ermittelt maximale Ausgabebreite fuer INTEGERS und REALS
 ***
 **************************************************************************/

init_breite()
{ register int n;
  char s[255];

  sprintf(s,"%ld", 1l<<(8 * sizeof(long) - 1));
  n = strlen(s);
  max_item_breite = (n > 6) ? n : 6;
  sprintf(s,"%#g", -0.00000123456789123456789);
  n = strlen(s);
  max_item_breite = (n > max_item_breite) ? n : max_item_breite;
}

/**************************************************************************
 ***                      Funktion to_break
 ***
 *** setzt die TO_BREAK-Marke auf den Befehl mit Adresse 'statnum'.
 *** Immer nur ein Befehl ist so markiert
 ***
 **************************************************************************/

to_break(statnums)
ZEILE *statnums;
{ if (to_break_lines)       /* Alte Marken loeschen */
  { set_breakpoints(-1, to_count, to_break_lines, 0, TO_BREAK);
    free(to_break_lines);
  }
  if (statnums)
  { if (statnums->list_flag)
    { to_break_lines = statnums->zval.zlist.zl;
      to_count = statnums->zval.zlist.zcount;
    }
    else
    { if (!(to_break_lines = (int *)calloc((size_t)1,(size_t)sizeof(int))))
      { komerr(texte[80],39);
        to_break_lines = NULL; to_count = 0; stepping_to = FALSE;
        return;
      }
      *to_break_lines = statnums->zval.znr;
      to_count = 1;
    }
    set_breakpoints(1, to_count, to_break_lines, 0, TO_BREAK);
  }
  else
  { to_break_lines = NULL; to_count = 0; }
  stepping_to = TRUE;
}

/**************************************************************************
 ***   Funktion to_unbreak
 ***
 *** loescht die TO_BREAK-Marke
 ***
 **************************************************************************/

to_unbreak()
{ if (to_break_lines)
  { set_breakpoints(-1, to_count, to_break_lines, 0, TO_BREAK);
    free(to_break_lines);
  }
  to_break_lines = NULL; to_count = 0;
  stepping_to = FALSE;
}

/**************************************************************************
 ***                    Funktion laden
 ***
 *** ex == DO_LOAD    : Neues Programm laden
 *** ex == DO_COMPILE : Neues Programm compilieren und laden
 *** Dateiname steht in 'loadfilename',
 *** tatsaechlicher Name steht nachher in 'the_filename'
 *** wird compiliert, steht der Sourcename nachher in 'the_sourcename'
 *** sonst wird versucht, dort einen eventuellen Sourcenamen abzulegen
 *** protout : wenn TRUE, dann Programmwechsel protokollieren
 *** infoout : wenn TRUE, dann Eingabedateiname ausgeben
 *** Bei Erfolg wird file_gelesen == TRUE
 ***
 **************************************************************************/

laden(ex, protout, infoout)
int ex, protout, infoout;
{ register int filename_da = FALSE;
#ifndef MAC
  struct stat filestate;
  register time_t z_time, p_time, xrf_time;
#endif
  register FILE *xrfin;
  char xrffilename[257];

  switch (ex)
  { case DO_LOAD    :
      if ((strcmp(loadfilename + strlen(loadfilename) - 2, ".p")
           != 0) &&          /* Endung ist nicht .p */
          (strcmp(loadfilename + strlen(loadfilename) - 4, ".pre")
           != 0))            /* Endung ist nicht .pre */
      { if (yyin = parzopen(loadfilename))
        { filename_da = TRUE;
          uebersetzen = FALSE;
          if (protout && prot_offen)
          { fprintf(protokollout, texte[203], the_filename); }
          if (!strcmp(the_filename + strlen(the_filename) - 2, ".z"))
          { strcpy(the_sourcename, the_filename);
            the_sourcename[strlen(the_sourcename) - 1] = 'p';
            strcpy(xrffilename, the_filename);
            strcpy(xrffilename + strlen(xrffilename) - 1, "xrf");
            sourcename_gilt = TRUE;
          }
          else
          { sourcename_gilt = FALSE;
            strcpy(xrffilename, the_filename);
            strcpy(xrffilename + strlen(xrffilename), ".xrf");
          }
          sourcename_sicher = FALSE;
          break;
        }
      }
    case DO_COMPILE :
      {
# ifndef PC
        char com[300];
        register int pre_flag;

        unlink("$xrf"); strcpy(xrffilename, "$xrf");
        if (!(pre_flag = parallaxistry(loadfilename)))
        { fprintf(stderr, texte[157], loadfilename); }
        else
        { sprintf(com,texte[254], fehlermax, (pre_flag < 0) ? texte[255] : "", the_filename);
          if (!(yyin = popen(com,"r")))
          { fprintf(stderr,texte[166]); }
          else
          { filename_da = TRUE;
            uebersetzen = TRUE;
            strcpy(the_sourcename, the_filename);
            sourcename_gilt = sourcename_sicher = TRUE;
            if (protout && prot_offen)
            { fprintf(protokollout, texte[203], the_filename); }
          }
        }
# else
        if (ex == DO_LOAD)
        { fprintf(stderr, texte[157], loadfilename); }
        else
        { fprintf(stderr, texte[269]); }
        filename_da = FALSE;
# endif
      }
      break;
    default         : bug("kommand.c/laden : falscher Parameter ex");
  }
  if (filename_da)
  { if (infoout) fprintf(stderr,texte[165],the_filename);
    file_gelesen = parse(uebersetzen);
    loesch_deb();
    new_def_pelist();
    if (!uebersetzen)
    {
#ifndef MAC
      stat(the_filename, &filestate);
      z_time = filestate.st_mtime;
      if (stat(xrffilename, &filestate))
      { sourcename_gilt = FALSE;
        if (infoout) fputs(texte[332], stderr);
      }
      else
      { xrf_time = filestate.st_mtime;
        sourcename_gilt = (xrf_time >= z_time);
        if (infoout && !sourcename_gilt) fputs(texte[333], stderr);
      }
#endif
    }
    if (sourcename_gilt)
    { if (!(xrfin = fopen(xrffilename, "r")))
      { sourcename_gilt = FALSE;
        if (infoout) fputs(texte[334], stderr);
      }
      else
      { register int x;
  
        if (x = read_xrf(xrfin))
        { if (infoout) fputs(texte[288-x], stderr);
          sourcename_gilt = FALSE;
        }
        fclose(xrfin);
        if (uebersetzen) unlink("$xrf");
        else if (!x)
        { strcpy(the_sourcename, xrf_source);
#ifndef MAC
          if (stat(the_sourcename, &filestate))
          { sourcename_gilt = FALSE;
            if (infoout) fputs(texte[335], stderr);
          }
          else
          { p_time = filestate.st_mtime;
            sourcename_gilt = (p_time <= z_time);
            if (infoout && !sourcename_gilt) fputs(texte[336], stderr);
          }
#endif
        }
      }
    }
    if (sourcename_gilt)
    { read_source();
#ifdef DEBDEBUG
      xrf_out(stderr);
#endif
    }
    else loesch_deb();
  }
}

/**************************************************************************
 ***                      Funktion kommandoloop
 ***
 *** Endlosschleife, die Kommandoparser aufruft und Ladekommandos uebernimmt
 ***
 **************************************************************************/

kommandoloop()
{ register int ex;

  new_def_pelist();
  do
  { breaked = FALSE;
    ex = kommandoparse();
    if (ex == DO_LOAD || ex == DO_COMPILE)
    { laden(ex, TRUE, TRUE); }
  } while (1);
}

/**************************************************************************
 ***                  Funktion new_def_pelist
 ***
 *** Erzeugt Standard-PE-String und setzt darin alle PEs aktiv
 ***
 **************************************************************************/

new_def_pelist()
{ register int n;
  register char *sp;

  if (default_pelist) free(default_pelist);
  if (!(default_pelist = calloc((size_t)(pe_anz + 1), (size_t)sizeof(char))))
  { fatal(texte[4]); parzexit(1); }
  for (n = 0, sp = default_pelist; n < pe_anz; n++, *(sp++) = '1');
  *sp = '\0';
}

/**************************************************************************
 ***           Funktion set_item_cast
 ***
 *** setzt '*ziel' auf den Wert von '*quell',
 *** fuehrt dabei die noetigen Typumwandlungen durch.
 *** vorzeichen : Bezeichnet das auf Kommandoebene eingegeben Vorzeichen
 ***              Werte : 0 (kein Vorzeichen), '-', NOT
 *** Fehlermeldung bei nicht zuweisbaren Typen.
 ***
 **************************************************************************/

set_item_cast(ziel,tz,quell,tq,vorzeichen)
ITEM *ziel;
TYP tz;
ITEM *quell;
TYP tq;
int vorzeichen;
{ register int uflag = 0;

  if (ITEM_typ(*quell) == KEINER)
  { ziel->datentyp = CH_TYP(ziel->datentyp, KEINER | UNDEF_ERG);
    tr_flag = tr_flag || (ziel->datentyp >= TRACED);
  }
  else
  { if (ITEM_typ(*quell) != tq)
    { komwarn(texte[283]); uflag = UNDEF_ERG; }
    switch (tz)
    { case BOOL :
        switch (tq)
        { case BOOL :
            if (vorzeichen == '-')
            { komerr(texte[228]); return; }
            set_b_item(ziel, vorzeichen ? !quell->inhalt.b_val
                                        : quell->inhalt.b_val, uflag);
            return;
          case CHA  :
            komerr(texte[99]);
            return;
          case INT  :
            if (vorzeichen == NOT)
            { komerr(texte[229]); return; }
            vorzeichen = vorzeichen ? -1 : 1;
            set_b_item(ziel, vorzeichen * quell->inhalt.i_val != 0, uflag);
            return;
          case REAL :
            if (vorzeichen == NOT)
            { komerr(texte[229]); return; }
            vorzeichen = vorzeichen ? -1 : 1;
            set_b_item(ziel, (float)vorzeichen * quell->inhalt.r_val != 0.0, uflag);
            return;
          case STR  :
            komerr(texte[30]);
            return;
        }
      case CHA  :
        switch (tq)
        { case BOOL :
            if (vorzeichen == '-')
            { komerr(texte[228]); return; }
            set_c_item(ziel, (vorzeichen ? !quell->inhalt.b_val
                                         : quell->inhalt.b_val) ? 'T' : 'F', uflag);
            return;
          case CHA  :
            if (vorzeichen)
            { komerr(texte[230]); return; }
            set_c_item(ziel, quell->inhalt.c_val, uflag);
            return;
          case INT  :
            if (vorzeichen == NOT)
            { komerr(texte[229]); return; }
            vorzeichen = vorzeichen ? -1 : 1;
            set_c_item(ziel, (unsigned char)(vorzeichen * quell->inhalt.i_val), uflag);
            return;
          case REAL :
            komerr(texte[100]);
            return;
          case STR  :
            if (vorzeichen)
            { komerr(texte[230]); return; }
            { register char *sp = quell->inhalt.s_val;
              register int fertig;

              if ((ziel - setbl->vars) / setvarlen + setbl->von + strlen(sp) > setbl->bis)
              { komerr(texte[128]); return; }
              for (fertig = FALSE; !fertig; sp++, ziel += setvarlen)
              { if (!(ziel->datentyp & CHA_ERLAUBT))
                { komerr(texte[129]); return; }
                set_c_item(ziel, (unsigned char)*sp, 0);
                fertig = !(*sp);
              }
            }
            return;
        }
      case INT  :
        switch (tq)
        { case BOOL :
            if (vorzeichen == '-')
            { komerr(texte[228]); return; }
            set_i_item(ziel, (long) (vorzeichen ? !quell->inhalt.b_val
                                                : quell->inhalt.b_val), uflag);
            return;
          case CHA  :
            if (vorzeichen)
            { komerr(texte[230]); return; }
            set_i_item(ziel, (long)quell->inhalt.c_val, uflag);
            return;
          case INT  :
            if (vorzeichen == NOT)
            { komerr(texte[229]); return; }
            vorzeichen = vorzeichen ? -1 : 1;
            set_i_item(ziel, (long)vorzeichen * quell->inhalt.i_val, uflag);
            return;
          case REAL :
            if (vorzeichen == NOT)
            { komerr(texte[229]); return; }
            vorzeichen = vorzeichen ? -1 : 1;
            set_i_item(ziel, (long)((float)vorzeichen * quell->inhalt.r_val), uflag);
            return;
          case STR  :
            komerr(texte[30]);
            return;
        }
      case REAL :
        switch (tq)
        { case BOOL :
            if (vorzeichen == '-')
            { komerr(texte[228]); return; }
            set_r_item(ziel, (float)(vorzeichen ? !quell->inhalt.b_val
                                                : quell->inhalt.b_val), uflag);
            return;
          case CHA  :
            komerr(texte[103]);
            return;
          case INT  :
            if (vorzeichen == NOT)
            { komerr(texte[229]); return; }
            vorzeichen = vorzeichen ? -1 : 1;
            set_r_item(ziel, (float)(vorzeichen * quell->inhalt.i_val), uflag);
            return;
          case REAL :
            if (vorzeichen == NOT)
            { komerr(texte[229]); return; }
            vorzeichen = vorzeichen ? -1 : 1;
            set_r_item(ziel, (float)vorzeichen * quell->inhalt.r_val, uflag);
            return;
          case STR  :
            komerr(texte[30]);
            return;
        }
    }
  }
}

/**************************************************************************
 ***                   Funktion new_tr_stat
 ***
 *** erzeugt Platz fuer einen neuen Programmbefehl und initialisiert
 *** diesen als TRACE-Befehl. (Benoetigt auf Kommandoebene)
 ***
 **************************************************************************/

STAT *new_tr_stat()
{ register STAT *erg;

  if (!kommand_traces)          /* noch keine TRACE-Befehle aus Kommandoebene */
  { if (!(kommand_traces = (STAT *)calloc((size_t)TR_STAT_BLOCK, (size_t)sizeof(STAT))))
    { komerr(texte[80],40); return NULL; }
    akt_kom_tr = 0; max_kom_tr = TR_STAT_BLOCK - 1;
  }
  if (akt_kom_tr > max_kom_tr)  /* Befehlsblock muss verlaengert werden */
  { if (!(erg = 
          (STAT *)realloc(kommand_traces,
                          (size_t)(((max_kom_tr += TR_STAT_BLOCK) + 1) *
                                   sizeof(STAT)))))
    { if (!(erg = 
            (STAT *)realloc(kommand_traces,
                            (size_t)(((max_kom_tr = akt_kom_tr) + 1) *
                                     sizeof(STAT)))))
      { komerr(texte[80],41); return NULL; }
    }
    kommand_traces = erg;
  }
  erg = kommand_traces + akt_kom_tr++;
  STAT_do_func(*erg) = do_trace;
  STAT_print_func(*erg) = print_trace;
  return erg;
}

/**************************************************************************
 ***                          Funktion show_pes
 ***
 *** gibt auf 'f' den Text 'text' und den PE-Aktivierungsstring 'pes' aus.
 *** Wird 'pes' auf mehrere Zeilen verteilt, werden alle Zeilen entsprechend
 *** 'text' eingerueckt.
 ***
 **************************************************************************/

show_pes(pes,text,f)
char *pes, *text;
FILE *f;
{ register int i;
  register char *tp, *pp;
  char praefix[255];

  if (pe_anz > 0)
  { for (tp = text, pp = praefix; *tp; *(pp++) = ' ', tp++);
    *pp = '\0';
    fputs(text, f);
    for (i = 0; i < pe_anz; i++)
    { if (i && !(i % pe_width))
      { putc('\n', f); ifmore(f);
        if (quitted) return;
        fputs(praefix, f);
      }
      putc(pes[i], f);
    }
  }
}

int zaehl[REAL + 1];            /* Zaehler fuer Offset zu jedem Typ */
char **vnames;                  /* Feld von Variablennamen fuer 'var_names()' */
int vn_len = 0;                 /* Laenge von 'vnames' */

/**************************************************************************
 ***                  Funktion var_names
 ***
 *** Erzeugt in 'vnames' ein Feld von Variablenbezeichnern, in dem
 *** zu jeder in 'dl' deklarierten Speicherstelle alle zugeordneten
 *** Variablennamen aufgefuehrt sind.
 *** s_v   : 'S' oder 'V'
 *** tiefe : Variablenebene
 *** Jeder Eintrag hat die Form
 ***      <s_v><typ><tiefe>:<offset>[,<s_v><typ><tiefe>:<offset>...]
 ***
 **************************************************************************/

  /**************************************************************************
   ***              Hilfsfunktion put_vnames (rekursiv)
   ***
   *** arbeitet 'dl' rekursiv ab und setzt die Variablennamen in 'vnames' ein.
   *** '*addr' zaehlt den absoluten Offset mit.
   ***
   **************************************************************************/

  put_vnames(dl, s_v, tiefe, addr)
  DECLIST *dl;
  char s_v;
  int tiefe, *addr;
  { register int count;
    register DECL *dec;

    for (count = dl->dcount, dec = dl->darray; count; count--, dec++)
    { switch (DECL_art(*dec))
      { case FELD  :
        { register TYP t = DECL_t(*dec);
          register int i;
          register char *neu;

          for (i = DECL_zahl(*dec); i; i--, (*addr)++)
          { register char *vpt = vnames[*addr];

            if (!(neu = vpt ? realloc(vpt, (size_t)((strlen(vpt) + 25) * sizeof(char))) :
                              calloc((size_t)25, (size_t)sizeof(char))))
            { kill_vnames(); return; }
            sprintf(neu + strlen(neu), "%s%c%c%d:%d",
                    *neu ? "," : "", s_v, typ_char[t], tiefe, ++zaehl[t]);
            vnames[*addr] = neu;
          }
        }
        break;
        case KLAM  :
        { register DECLIST *di = &DECL_klamdecl(*dec);
          register int i;

          for (i = DECL_wiederh(*dec); i; i--)
          { put_vnames(di, s_v, tiefe, addr);
            if (!vnames) return;
          }
        }
        break;
        case UNION :
        { register DECLIST *upt = DECL_ul(*dec).uarray;
          register int i;
          int lokaddr;

          for (i = DECL_ul(*dec).ucount; i; i--, upt++)
          { lokaddr = *addr;
            put_vnames(upt, s_v, tiefe, &lokaddr);
            if (!vnames) return;
          }
          *addr += DECL_ul(*dec).utyp_anz[SUMM];
        }
      }
    }
  }

var_names(dl, s_v, tiefe)
DECLIST *dl;
char s_v;
int tiefe;
{ register int i;
  int addr = 0;

  if (vn_len = dl->typ_zahl[SUMM])
  { if (!(vnames = (char **)calloc((size_t)vn_len, (size_t)sizeof(char *))))
    { vn_len = 0; return; }
    for (i = 0; i < vn_len; i++)
    { if (!(vnames[i] = calloc((size_t)1, (size_t)sizeof(char))))
      { kill_vnames(); return; }
      *vnames[i] = '\0';
    }
    for (i = BOOL; i <= REAL; zaehl[i++] = 0);
    put_vnames(dl, s_v, tiefe, &addr);
  }
}

/**************************************************************************
 ***                  Funktion kill_vnames
 ***
 *** Gibt allen Speicherplatz in 'vnames' frei
 ***
 **************************************************************************/

kill_vnames()
{ register char **vnptr;

  if (vnames)
  { for (vnptr = vnames; vn_len; vnptr++, vn_len--)
    { if (*vnptr) free(*vnptr); }
    free(vnames);
    vnames = NULL;
  }
  else vn_len = 0;
}

/**************************************************************************
 ***                  Funktion show_scal_varblock
 ***
 *** gibt einen Block von Skalarvariablen auf 'f' aus.
 *** '*aptr'    : Aktivierungsblock mit auszugebendem Block
 *** 'min_adr'  : Adresse ab der ausgegeben wird (evtl. nicht im Block)
 *** 'max_adr'  : Adresse bis zu der ausgegeben wird (evtl. nicht im Block)
 *** 'tiefe'    : Ebene der Variablen im Block
 *** 'proc_str' : evtl. Text, der Prozedurlabel enthaelt
 *** Ausgabeformat jeder Variablen :
 ***    <abs.adresse> : S<typ><tiefe>:<offset>[,...]<proc_str> = <inhalt>
 *** Ergebnis : Zahl der ausgegebenen Zeilen
 ***
 **************************************************************************/

int show_scal_varblock(aptr, min_adr, max_adr, tiefe, proc_str, f)
AKTBLOCK *aptr;
int min_adr, max_adr;
int tiefe;
char *proc_str;
FILE *f;
{ register int i, j, erg = 0;
  register ITEM *iptr;
  register VARBLOCK *vptr = aptr->s_vars;
  register DECLIST *dl = aptr->procadr < 0 ? &glob_s_decl
                                           : STAT_dscal(programm[aptr->procadr]);

  block_breaked = FALSE;
  if (vptr)
  { beim_listing = TRUE;
    var_names(dl, 'S', tiefe);
    if (!vnames)
    { beim_listing = FALSE;
      fputs(texte[80], f);
      return erg;
    }
    for (i = vptr->von, iptr = vptr->vars, j = 0; i <= vptr->bis; i++, iptr++, j++)
    { if (i >= min_adr && i <= max_adr)
      { putc('\n', f); ifmore(f); erg++;
        if (!quitted)
        { fprintf(f,"%8d : %s%s = ", i, vnames[j], proc_str);
          print_wert(iptr, f, KEINER);
        }
        if (quitted)
        { fputs(texte[122],f);
          quitted--; block_breaked = TRUE;      /* Nur diesen Block abbrechen */
          beim_listing = FALSE;
          kill_vnames();
          return erg;
        }
      }
    }
    beim_listing = FALSE;
    kill_vnames();
  }
  else
  { fputs(texte[204], f); }
  return erg;
}

/**************************************************************************
 ***                      Funktion show_vecvar
 ***
 *** gibt eine Vektorvariable auf 'f' aus.
 *** '*ipptr' : Zeiger auf auszugebendes ITEM (wird weitergezaehlt)
 *** 'string' : Variablenbezeichnung
 *** 'max_pes', 'start_pes' : PEs aus Schnittmenge sind auszugeben
 *** Ausgabeformat :
 *** <string>PE<m> : <inhalt>   PE<n> : <inhalt>   ...
 ***         ...
 *** Ergebnis : Zahl der ausgegebenen Zeilen
 ***
 **************************************************************************/

int show_vecvar(ipptr, string, max_pes, start_pes, f)
ITEM **ipptr;
char *string;
char *max_pes, *start_pes;
FILE *f;
{ register int i, j, sc, len, erg = 0;
  register char *s;
  register ITEM *ip;
  register int pelen = !pe_anz ? 1 : (int)log10((double)pe_anz) + 1;
  register int flen, cols;
  int dummy;
  register int last_pe = last(max_pes,&dummy);
  register int last_s_pe = last(start_pes,&dummy);

  if (last_s_pe < last_pe) last_pe = last_s_pe;
                /* Maximale Ausgabelaenge eines ITEMs in 'flen' ermitteln */
  for (j = 0, flen = 0, ip = *ipptr; j < last_pe; j++, ip++)
  { if (start_pes[j] == '1' && max_pes[j] == '1')
    { if (!(s = wert_string(ip, KEINER))) return erg;
      len = strlen(s);
      flen = (len > flen) ? len : flen;
      free(s);
    }
  }

  j = CRTCOLS - strlen(string);         /* Platz fuer Variablenwerte */
  cols = j / (flen + pelen + 8);        /* Zahl der Var. nebeneinander */

                /* Ausgabe der Variable in 'cols' Spalten */
  for (j = 0, sc = 0; j < last_pe; )
  { if (start_pes[j] == '1' && max_pes[j] == '1')
    { putc('\n', f); ifmore(f); erg++;
      if (quitted) return erg;
      fputs(string, f);
      if (!sc)          /* 'string' mit Blanks ueberschreiben */
      { register char *sp;

        for (sp = string; *sp; *(sp++) = ' ');
        sc++;
      }
                /* Ausgabe einer Zeile */
      for (i = 0; i < cols && j < last_pe; j++, (*ipptr)++)
      { if (start_pes[j] == '1' && max_pes[j] == '1')
        { if (!(s = wert_string(*ipptr, KEINER))) return erg;
          fprintf(f,"Pe%0*d : %-*s   ", pelen, j + 1, flen, s);
          free(s);
          i++;
        }
      }
    }
    else
    { j++; (*ipptr)++; }
    if (quitted) break;
  }
  *ipptr += pe_anz - j;
  return erg;
}

/**************************************************************************
 ***                  Funktion show_vec_varblock
 ***
 *** gibt einen Block von Vektorvariablen auf 'f' aus.
 *** '*aptr'    : Aktivierungsblock mit auszugebendem Block
 *** 'min_adr'  : Adresse ab der ausgegeben wird (evtl. nicht im Block)
 *** 'max_adr'  : Adresse bis zu der ausgegeben wird (evtl. nicht im Block)
 *** 'tiefe'    : Ebene der Variablen im Block
 *** 'max_pes', 'start_pes' : PEs aus Schnittmenge sind auszugeben
 *** 'proc_str' : evtl. Text, der Prozedurlabel enthaelt
 *** verwendet show_vecvar, uebergibt als 'string' :
 ***    '<adresse> : V<typ><tiefe>:<offset>[,...]<proc_str> = '
 *** Ergebnis : Zahl der ausgegebenen Zeilen
 ***
 **************************************************************************/

int show_vec_varblock(aptr, min_adr, max_adr, tiefe,
                      max_pes, start_pes, proc_str, f)
AKTBLOCK *aptr;
int min_adr, max_adr;
int tiefe;
char *max_pes, *start_pes;
char *proc_str;
FILE *f;
{ register int i, j, erg = 0;
  register VARBLOCK *vptr = aptr->v_vars;
  register DECLIST *dl = aptr->procadr < 0 ? &glob_v_decl
                                           : STAT_dvec(programm[aptr->procadr]);
  ITEM *iptr;
  char *s;

  block_breaked = FALSE;
  if (vptr)
  { beim_listing = TRUE;
    var_names(dl, 'V', tiefe);
    if (!vnames)
    { beim_listing = FALSE;
      fputs(texte[80], f);
      return erg;
    }
    for (i = vptr->von, iptr = vptr->vars, j = 0; i <= vptr->bis; i++, j++)
    { if (i >= min_adr && i <= max_adr)
      { if (!(s = calloc((size_t)(18 + strlen(vnames[j]) + strlen(proc_str)), 
                         (size_t)sizeof(char))))
        { fputs(texte[80], f);
          beim_listing = FALSE;
          kill_vnames();
          return erg ;
        }
        sprintf(s,"%8d : %s%s = ", i, vnames[j], proc_str);
        erg += show_vecvar(&iptr, s, max_pes, start_pes, f);
        free(s);
        if (quitted)
        { fputs(texte[122],f);
          quitted--; block_breaked = TRUE;      /* nur diesen Block abbrechen */
          beim_listing = FALSE;
          kill_vnames();
          return erg ;
        }
      }
      else iptr += pe_anz;
    }
    beim_listing = FALSE;
    kill_vnames();
  }
  else
  { fputs(texte[204], f); }
  return erg;
}

/**************************************************************************
 ***                      Funktion show_aktstack
 ***
 *** gibt den Aktivierungsstack oder Teile davon auf 'f' aus:
 *** opt : Optionen bei SHOW CALLSTACK : CHAIN_OPT : ganze Aufrufkette ausgeben
 ***                                     ACTIVITY_OPT : Prozessorzustaende auf 
 ***                                                    Stapel anzeigen
 *** sstack : Tiefe, bis zu der Skalarvariablen ausgegeben werden
 ***          -1 : alle, -2 : alle, falls -2 in vstack und keine Optionen
 *** vstack : analog zu sstack
 *** st_pes : wenn TRUE, dann gilt 'pe_list_string'
 ***
 **************************************************************************/

show_aktstack(f, opt, sstack, vstack, st_pes)
FILE *f;
int opt, sstack, vstack, st_pes;
{ register int i, lev;
  register DISPLAY *dupdisp;
  register AKTBLOCK *abptr;
  register int stief = sstack;
  register int vtief = vstack;
  register int sganz = (stief == -1 || (stief == -2 && vtief == -2 && !opt));
  register int vganz = (vtief == -1 || (stief == -2 && vtief == -2 && !opt));
  register char *max_pes = (st_pes) ? pe_list_string : default_pelist;

  quitted = 0;
  more_count = 0;
  block_breaked = FALSE;
  if (!disp)
  { fputs(texte[183],f); putc('\n', f);
    return;
  }
  if (!(dupdisp = (DISPLAY *)calloc((size_t)(max_lev + 1), (size_t)sizeof(DISPLAY))))
  { fputs(texte[80],stderr);
    return;
  }
                /* Zur Zeit aktive PEs anzeigen */
  show_pes(aktive_pes, texte[179] ,f);
                /* Display kopieren */
  for (i = 0; i <= max_lev; i++)
  { dupdisp[i] = disp[i]; }
  lev = akt_tiefe;
                /* Alle gewuenschten Prozeduren auf dem Aufrufstack abarbeiten */
  while (lev > 0 && (sganz || stief > 0 || vganz || vtief > 0 || opt))
  { if (abptr = dupdisp[lev])
    { if ((opt & CHAIN_OPT) || sganz || stief > 0 || vganz || vtief > 0)
                /* Prozedurlabel, evtl. Prozedurname und Ruecksprungadresse */
      { putc('\n', f); ifmore(f); 
        if (!quitted)
        { register char *sp;
          register ZEILTAB *p;
          register int i;
          register SCOPETAB **st;
          char ss[30];

          if (st = STAT_scopes(programm[abptr->procadr]))
          { sp = key_to_name((*st)->proc_key); }
          else
          { sp = ""; }
          if (p = abptr->para_call_zeile)
          { sprintf(ss,"%d",p->source_line);
            if (i = p->stat_in_line) sprintf(ss+strlen(ss), texte[298], i);
          }
          else ss[0] = '\0';
          fprintf(f,texte[205],
                  sp,
                  STAT_label(programm[abptr->procadr]),
                  ss,
                  STAT_label(programm[abptr->rueck - 1]));
        }
      }

      if (!quitted && (sganz || stief > 0))
                /* Skalarvariablen ausgeben */
      { putc('\n', f); ifmore(f);
        if (!quitted)
        { fputs(texte[206],f);
          show_scal_varblock(abptr, 0, s_max_b_adr, lev, "", f);
        }
      }

      if (!quitted && (vganz || vtief > 0))
                /* Vektorvariablen ausgeben */
      { putc('\n', f); ifmore(f);
        if (!quitted)
        { fputs(texte[207] ,f);
          show_vec_varblock(abptr, 0, v_max_b_adr,
                            lev, max_pes, abptr->start_aktiv, "", f);
        }
      }

      if (!quitted &&
          ((opt & ACTIVITY_OPT) || sganz || stief > 0 || vganz || vtief > 0))
                /* Fruehere Prozessorzustaende ausgeben */
      { if (abptr->vorh_aktiv)
        { putc('\n', f); ifmore(f);
          if (!quitted)
          { show_pes(abptr->vorh_aktiv, texte[208], f); }
        }
      }
                /* Naechsten Aktivierungsblock von dupliziertem Display */
      dupdisp[lev] = abptr->dispalt;
      lev = abptr->aufr_tiefe;
      if (stief > 0) stief--;
      if (vtief > 0) vtief--;

      if (quitted) lev = 0;
    }
    else
    { fputs(texte[209], f); more_count++; ifmore(f); }
  }
                /* Analog fuers Hauptprogramm */
  if (!quitted)
  { if (abptr = dupdisp[lev])
    { if (!quitted &&
          ((opt & CHAIN_OPT) || sganz || stief > 0 || vganz || vtief > 0))
                /* Hauptprogramm */
      { putc('\n', f); ifmore(f);
        if (!quitted) fprintf(f,texte[210], system_name);
      }

      if (!quitted && (sganz || stief > 0))
                /* Skalarvariablen */
      { putc('\n', f); ifmore(f);
        if (!quitted)
        { fputs(texte[211], f);
          show_scal_varblock(abptr, 0, s_max_b_adr, 0, "", f);
        }
      }

      if (!quitted && (vganz || vtief > 0))
                /* Vektorvariablen */
      { putc('\n', f); ifmore(f);
        if (!quitted)
        { fputs(texte[212], f);
          show_vec_varblock(abptr, 0, v_max_b_adr,
                            0, max_pes, abptr->start_aktiv, "", f);
        }
      }
    }
    else
    { fputs(texte[213], f);
      more_count++; ifmore(f);
    }
  }
  if (quitted && !block_breaked) fputs(texte[122],f);
  free(dupdisp);
  no_linefeed = !quitted;
}

/**************************************************************************
 ***                   Funktion print_s_heap
 ***
 *** gibt den skalaren Heap zwischen 'vonaddr' und 'bisaddr' auf 'f' aus
 ***
 **************************************************************************/

print_s_heap(f, vonaddr, bisaddr)
FILE *f;
int vonaddr, bisaddr;
{ register VARBLOCK *heap;
  register int bcount = 0;

  beim_listing = TRUE;
  block_breaked = FALSE;
                /* Schleife ueber alle Bloecke auf dem Heap */
  for (heap = s_heap; heap; heap = heap->nblock)
  { register int i_von, i_bis;
    register ITEM *iptr;
                /* i_von, i_bis werden existierende Adressen im Block */
    if (vonaddr <= heap->von)
    { i_von = heap->von; iptr = heap->vars; }
    else if (vonaddr <= heap->bis)
    { i_von = vonaddr; iptr = heap->vars + i_von - heap->von; }
    else continue;      /* nichts aus dem Block ausgeben */
    if (bisaddr >= heap->bis) i_bis = heap->bis;
    else if (bisaddr >= heap->von) i_bis = bisaddr;
    else continue;      /* nichts aus dem Block ausgeben */
    if (i_bis >= i_von)
    { register int i;

      if (bcount++)
      { putc('\n', f); ifmore(f);
        if (quitted)
        { fputs(texte[122],f);
          beim_listing = FALSE;
          quitted--; block_breaked = TRUE;
          return;
        }
      }
      for (i = i_von; i <= i_bis; i++, iptr++)
      { putc('\n', f); ifmore(f);
        if (!quitted)
        { fprintf(f,"%8d : ", i);
          print_wert(iptr, f, KEINER);
        }
        if (quitted)
        { fputs(texte[122],f);
          beim_listing = FALSE;
          quitted--; block_breaked = TRUE;
          return;
        }
      }
    }
  }
  beim_listing = FALSE;
}

/**************************************************************************
 ***               Funktion print_v_heap
 ***
 *** gibt den vektoriellen Heap fuer die PEs in 'pes'
 *** zwischen 'vonaddr' und 'bisaddr' auf 'f' aus
 ***
 **************************************************************************/

print_v_heap(f, vonaddr, bisaddr, pes)
FILE *f;
int vonaddr, bisaddr;
char *pes;
{ register VARBLOCK *heap;
  register int bcount = 0;

  beim_listing = TRUE;
  block_breaked = FALSE;
                /* Schleife ueber alle Bloecke auf dem Heap */
  for (heap = v_heap; heap; heap = heap->nblock)
  { register int i_von, i_bis;
    ITEM *iptr;
                /* i_von, i_bis werden existierende Adressen im Block */
    if (vonaddr <= heap->von)
    { i_von = heap->von; iptr = heap->vars; }
    else if (vonaddr <= heap->bis)
    { i_von = vonaddr; iptr = heap->vars + (i_von - heap->von) * pe_anz; }
    else continue;      /* nichts aus dem Block ausgeben */
    if (bisaddr >= heap->bis) i_bis = heap->bis;
    else if (bisaddr >= heap->von) i_bis = bisaddr;
    else continue;      /* nichts aus dem Block ausgeben */
    if (i_bis >= i_von)
    { register int i;

      if (bcount++)
      { putc('\n', f); ifmore(f);
        if (quitted)
        { fputs(texte[122],f);
          beim_listing = FALSE;
          quitted--; block_breaked = TRUE;
          return;
        }
      }
      for (i = i_von; i <= i_bis; i++)
      { char s[13];

        sprintf(s,"%8d : ", i);
        show_vecvar(&iptr, s, pes, pes, f);
        if (quitted)
        { fputs(texte[122],f);
          beim_listing = FALSE;
          quitted--; block_breaked = TRUE;
          return;
        }
      }
    }
  }
  beim_listing = FALSE;
}

/**************************************************************************
 ***                      Funktion show_heap
 ***
 *** Gibt Teile von skalarem und vektroriellem Heap auf 'f' aus
 *** smem : Liste der Speicherbereiche auf dem skalaren Heap
 *** vmem : Liste der Speicherbereiche auf dem vektoriellen Heap
 *** mem_pes : TRUE, wenn pe_list_string gilt
 ***
 **************************************************************************/

show_heap(f, smem, vmem, mem_pes)
FILE *f;
RANGELIST *smem, *vmem;
int mem_pes;
{ quitted = 0;
  if (!smem ||
      (smem == (RANGELIST *)(-1) && vmem == (RANGELIST *)(-1)))
                /* skalaren Heap ganz ausgeben */
  { fputs(texte[214], f);
    if (!s_heap)
    { fputs(texte[215], f); }
    else
    { print_s_heap(f, s_min_h_adr, -1);
      if (quitted)
      { if (!block_breaked) fputs(texte[122],f);
        no_linefeed = FALSE;
        return;
      }
    }
  }
  else if (smem && smem != (RANGELIST *)(-1))
                /* Teile des skalaren Heap ausgeben */
  { fputs(texte[214], f);
    if (!s_heap)
    { fputs(texte[215], f); }
    else
                /* Liste fuer skalaren Heap abarbeiten */
    { for (; smem; smem = smem->next_range)
      { if (smem->r.bis_n < s_min_h_adr || smem->r.von_n > -1)
                /* Bereich ganz ausserhalb */
        { putc('\n', f); ifmore(f);
          if (quitted)
          { fputs(texte[122],f);
            no_linefeed = FALSE;
            return;
          }
          fprintf(f,(smem->r.von_n == smem->r.bis_n) ? texte[226] : texte[199],
                    smem->r.von_n,smem->r.bis_n);
        }
        else
        { if (smem->r.von_n < s_min_h_adr) smem->r.von_n = s_min_h_adr;
          if (smem->r.bis_n > -1) smem->r.bis_n = -1;
                /* Bereich beschraenkt auf Heap */
          putc('\n', f); ifmore(f);
          if (!quitted)
          { fprintf(f, (smem->r.von_n == smem->r.bis_n) ? "%d :" : "%d..%d :",
                    smem->r.von_n, smem->r.bis_n);
            print_s_heap(f, smem->r.von_n, smem->r.bis_n);
          }
          if (quitted)
          { if (!block_breaked) fputs(texte[122],f);
            no_linefeed = FALSE;
            return;
          }
        }
      }
    }
  }
  block_breaked = FALSE;
  if (!vmem || (vmem == (RANGELIST *)(-1) && smem == (RANGELIST *)(-1)))
                /* vektoriellen Heap ganz ausgeben */
  { putc('\n', f); ifmore(f);
    if (quitted)
    { fputs(texte[122],f);
      no_linefeed = FALSE;
      return;
    }
    fputs(texte[216], f);
    if (!v_heap)
    { fputs(texte[215], f); }
    else
    { print_v_heap(f, v_min_h_adr, -1, mem_pes ? pe_list_string
                                               : default_pelist);
      if (quitted)
      { if (!block_breaked) fputs(texte[122],f);
        no_linefeed = FALSE;
        return;
      }
    }
  }
  else if (vmem && vmem != (RANGELIST *)(-1))
                /* Teile des skalaren Heap ausgeben */
  { putc('\n', f); ifmore(f);
    if (quitted)
    { fputs(texte[122],f);
      no_linefeed = FALSE;
      return;
    }
    fputs(texte[216], f);
    if (!v_heap)
    { fputs(texte[215], f); }
    else
    { register RANGELIST *vm;
                /* Liste fuer skalaren Heap abarbeiten */
      for (vm = vmem; vm; vm = vm->next_range)
      { if (vm->r.bis_n < v_min_h_adr || vm->r.von_n > -1)
                /* Bereich ganz ausserhalb */
        { putc('\n', f); ifmore(f);
          if (quitted)
          { fputs(texte[122],f);
            no_linefeed = FALSE;
            return;
          }
          fprintf(f,(vm->r.von_n == vm->r.bis_n) ? texte[226] : texte[199],
                    vm->r.von_n,vm->r.bis_n);
        }
        else
        { if (vm->r.von_n < v_min_h_adr) vm->r.von_n = v_min_h_adr;
          if (vm->r.bis_n > -1) vm->r.bis_n = -1;
                /* Bereich beschraenkt auf Heap */
          putc('\n', f); ifmore(f);
          if (!quitted)
          { fprintf(f,(vm->r.von_n == vm->r.bis_n) ? "%d :" : "%d..%d :",
                    vm->r.von_n, vm->r.bis_n);
            print_v_heap(f, vm->r.von_n, vm->r.bis_n, mem_pes ? pe_list_string
                                                              : default_pelist);
          }
          if (quitted)
          { if (!block_breaked) fputs(texte[122],f);
            no_linefeed = FALSE;
            return;
          }
        }
      }
    }
  }
  no_linefeed =TRUE;
}

DISPLAY *dupdisp = NULL;        /* Duplikat des Display fuer 'show mem' */

/**************************************************************************
 ***                Funktion print_s_mem_rec (rekursiv)
 ***
 *** gibt den skalaren Speicherbereich zwischen 'vonaddr' und 'bisaddr'
 *** auf 'f' aus.
 *** Die Funktion startet bei Prozedurtiefe 'lev' auf dem Display 'dupdisp',
 *** laeuft rekursiv durch den Aufrufstapel in Richtung Hauptprogramm
 *** bis 'vonaddr' und gibt den Speicherinhalt auf dem Rueckweg
 *** in der richtigen Reihenfolge aus.
 ***
 **************************************************************************/

print_s_mem_rec(f, lev, vonaddr, bisaddr)
FILE *f;
int lev, vonaddr, bisaddr;
{ AKTBLOCK *abptr;
  VARBLOCK *vbptr;

  if (abptr = dupdisp[lev])
  { vbptr = abptr->s_vars;
    if (!vbptr || vbptr->von > vonaddr)
                /* weiter in Richtung Hauptprogramm */
    { dupdisp[lev] = abptr->dispalt;
      print_s_mem_rec(f, abptr->aufr_tiefe, vonaddr, bisaddr);
      if (quitted) return;
    }
    if (vbptr && vonaddr <= vbptr->bis && bisaddr >= vbptr->von)
    { char *ps;

      if (abptr->procadr >= 0)
                /* Prozedurnummer fuer Variablen im Block */
      { register STAT *bef = programm + abptr->procadr;
        register char *sc = NULL;
        register SCOPETAB **st;

        if (st = STAT_scopes(*bef))
        { sc = key_to_name((*st)->proc_key); }
        if (sc)
        { if (ps = malloc((size_t)(strlen(sc) + 25)))
          { sprintf(ps, "/%s(%d :)", sc, STAT_label(*bef)); }
        }
        else
        { if (ps = malloc((size_t)25))
          { sprintf(ps, "/%d", STAT_label(*bef)); }
        }
      }
      else
      { ps = calloc((size_t)2, (size_t)sizeof(char)); }
      if (!ps)
      { komerr(texte[80],42); return; }
      if (show_scal_varblock(abptr, vonaddr, bisaddr, lev, ps, f))
      { if (vbptr->bis < bisaddr) { putc('\n', f); ifmore(f); } }
      free(ps);
    }
  }
}

/**************************************************************************
 ***                Funktion print_v_mem_rec (rekursiv)
 ***
 *** gibt den vektoriellen Speicherbereich zwischen 'vonaddr' und 'bisaddr'
 *** auf 'f' aus.
 *** Funktioniert analog zu print_s_mem_rec.
 *** pes : zu betrachtende Prozessoren
 ***
 **************************************************************************/

print_v_mem_rec(f, lev, vonaddr, bisaddr, pes)
FILE *f;
int lev, vonaddr, bisaddr;
char *pes;
{ AKTBLOCK *abptr;
  VARBLOCK *vbptr;

  if (abptr = dupdisp[lev])
  { vbptr = abptr->v_vars;
    if (!vbptr || vbptr->von > vonaddr)
                /* weiter in Richtung Hauptprogramm */
    { dupdisp[lev] = abptr->dispalt;
      print_v_mem_rec(f, abptr->aufr_tiefe, vonaddr, bisaddr, pes);
      if (quitted) return;
    }
    if (vbptr && vonaddr <= vbptr->bis && bisaddr >= vbptr->von)
    { char *ps;

      if (abptr->procadr >= 0)
                /* Prozedurnummer fuer Variablen im Block */
      { register STAT *bef = programm + abptr->procadr;
        register char *sc = NULL;
        register SCOPETAB **st;

        if (st = STAT_scopes(*bef))
        { sc = key_to_name((*st)->proc_key); }
        if (sc)
        { if (ps = malloc((size_t)(strlen(sc) + 25)))
          { sprintf(ps, "/%s(%d :)", sc, STAT_label(*bef)); }
        }
        else
        { if (ps = malloc((size_t)25))
          { sprintf(ps, "/%d", STAT_label(*bef)); }
        }
      }
      else
      { ps = calloc((size_t)2, (size_t)sizeof(char)); }
      if (!ps)
      { komerr(texte[80],43); return; }
      if (show_vec_varblock(abptr, vonaddr, bisaddr, lev, pes, vbptr->max_pes, ps, f))
      { if (vbptr->bis < bisaddr) { putc('\n', f); ifmore(f); } }
      free(ps);
    }
  }
}

/**************************************************************************
 ***                      Funktion print_s_mem
 ***
 *** gibt den skalaren Speicherbereich zwischen 'vonaddr' und 'bisaddr'
 *** auf 'f' auf:
 *** kopiert 'disp' nach 'dupdisp' und ruft dann 'print_s_mem_rec' auf
 ***
 **************************************************************************/

print_s_mem(f, vonaddr, bisaddr)
FILE *f;
int vonaddr, bisaddr;
{ register int i;

  if (!disp) return;
  if (dupdisp) free(dupdisp);
  if (!(dupdisp = (DISPLAY *)calloc((size_t)(max_lev + 1), (size_t)sizeof(DISPLAY))))
  { fputs(texte[80],stderr);
    return;
  }
  for (i = 0; i <= max_lev; i++)
  { dupdisp[i] = disp[i]; }
  print_s_mem_rec(f, akt_tiefe, vonaddr, bisaddr, 0);
}

/**************************************************************************
 ***                      Funktion print_v_mem
 ***
 *** gibt den vektoriellen Speicherbereich zwischen 'vonaddr' und 'bisaddr'
 *** auf 'f' auf:
 *** kopiert 'disp' nach 'dupdisp' und ruft dann 'print_v_mem_rec' auf
 *** pes : zu betrachtende Prozessoren
 ***
 **************************************************************************/

print_v_mem(f, vonaddr, bisaddr, pes)
FILE *f;
int vonaddr, bisaddr;
char *pes;
{ register int i;

  if (!disp) return;
  if (dupdisp) free(dupdisp);
  if (!(dupdisp = (DISPLAY *)calloc((size_t)(max_lev + 1), (size_t)sizeof(DISPLAY))))
  { fputs(texte[80],stderr);
    return;
  }
  for (i = 0; i <= max_lev; i++)
  { dupdisp[i] = disp[i]; }
  print_v_mem_rec(f, akt_tiefe, vonaddr, bisaddr, pes, 0);
}

/**************************************************************************
 ***                      Funktion show_mem
 ***
 *** Gibt Teile von skalarem und vektroriellem Speicher auf 'f' aus
 *** smem : Liste der Speicherbereiche des skalaren Speichers
 *** vmem : Liste der Speicherbereiche des vektoriellen Speichers
 *** mem_pes : TRUE, wenn pe_list_string gilt
 *** Betrachtet sowohl Heap als auch Aufrufstack
 ***
 **************************************************************************/

show_mem(f, smem, vmem, mem_pes)
FILE *f;
RANGELIST *smem, *vmem;
int mem_pes;
{ quitted = 0;
  if (!smem ||
      (smem == (RANGELIST *)(-1) && vmem == (RANGELIST *)(-1)))
                /* skalaren Speicher ganz ausgeben */
  { fputs(texte[217], f);
    if (!(s_heap || s_blocks))
    { fputs(texte[215], f); }
    else
    { if (s_heap)
                /* skalaren Heap ausgeben */
      { print_s_heap(f, s_min_h_adr, -1);
        if (!quitted && s_blocks) { putc('\n', f); ifmore(f); }
      }
      if (quitted)
      { if (!block_breaked) fputs(texte[122],f);
        no_linefeed = FALSE;
        return;
      }
                /* skalaren Stack ausgeben */
      if (s_blocks) print_s_mem(f, 1, s_max_b_adr);
      if (quitted)
      { if (!block_breaked) fputs(texte[122],f);
        no_linefeed = FALSE;
        return;
      }
    }
  }
  else if (smem && smem != (RANGELIST *)(-1))
                /* Teile des skalaren Speichers ausgeben */
  { fputs(texte[217], f);
    if (!(s_heap || s_blocks))
    { fputs(texte[215], f); }
    else
                /* Liste fuer skalaren Speicher abarbeiten */
    { for (; smem; smem = smem->next_range)
      { if (smem->r.bis_n < s_min_h_adr || smem->r.von_n > s_max_b_adr ||
            (smem->r.von_n == 0 && smem->r.bis_n == 0))
                /* Bereich ganz ausserhalb */
        { putc('\n', f); ifmore(f);
          if (quitted)
          { fputs(texte[122],f);
            no_linefeed = FALSE;
            return;
          }
          fprintf(f,(smem->r.von_n == smem->r.bis_n) ? texte[224] : texte[198],
                    smem->r.von_n,smem->r.bis_n);
        }
        else
        { if (smem->r.von_n < s_min_h_adr) smem->r.von_n = s_min_h_adr;
          if (smem->r.von_n == 0) smem->r.von_n = 1;
          if (smem->r.bis_n > s_max_b_adr) smem->r.bis_n = s_max_b_adr;
          if (smem->r.bis_n == 0) smem->r.bis_n = -1;
                /* Bereich beschraenkt auf vorhandenen Speicher */
          putc('\n', f); ifmore(f);
          if (quitted)
          { fputs(texte[122],f);
            no_linefeed = FALSE;
            return;
          }
          fprintf(f, (smem->r.von_n == smem->r.bis_n) ? "%d :" : "%d..%d :",
                  smem->r.von_n, smem->r.bis_n);
          if (s_heap && smem->r.von_n < 0)
                /* skalarer Heap */
          { print_s_heap(f, smem->r.von_n,
                            (smem->r.bis_n >= 0) ? -1 : smem->r.bis_n);
            if (!quitted && s_blocks && smem->r.bis_n > 0)
            { putc('\n', f); ifmore(f); }
            if (quitted)
            { if (!block_breaked) fputs(texte[122],f);
              no_linefeed = FALSE;
              return;
            }
          }
          if (s_blocks && smem->r.bis_n > 0)
                /* skalarer Stack */
          { print_s_mem(f, (smem->r.von_n <= 0) ? 1 : smem->r.von_n,
                           smem->r.bis_n);
            if (quitted)
            { if (!block_breaked) fputs(texte[122],f);
              no_linefeed = FALSE;
              return;
            }
          }
        }
      }
    }
  }
  block_breaked = FALSE;
  if (!vmem || (vmem == (RANGELIST *)(-1) && smem == (RANGELIST *)(-1)))
                /* vektoriellen Speicher ganz ausgeben */
  { putc('\n', f); ifmore(f);
    if (quitted)
    { fputs(texte[122],f);
      no_linefeed = FALSE;
      return;
    }
    fputs(texte[218], f);
    if (!(v_heap || v_blocks))
    { fputs(texte[215], f); }
    else
    { if (v_heap)
                /* vektoriellen Heap ausgeben */
      { print_v_heap(f, v_min_h_adr, -1, mem_pes ? pe_list_string
                                                 : default_pelist);
        if (!quitted && v_blocks) { putc('\n', f); ifmore(f); }
      }
      if (quitted)
      { if (!block_breaked) fputs(texte[122],f);
        no_linefeed = FALSE;
        return;
      }
                /* vektoriellen Stack ausgeben */
      if (v_blocks) print_v_mem(f, 1, v_max_b_adr, mem_pes ? pe_list_string
                                                           : default_pelist);
      if (quitted)
      { if (!block_breaked) fputs(texte[122],f);
        no_linefeed = FALSE;
        return;
      }
    }
  }
  else if (vmem && vmem != (RANGELIST *)(-1))
                /* Teile des vektoriellen Speichers ausgeben */
  { putc('\n', f); ifmore(f);
    if (quitted)
    { fputs(texte[122],f);
      no_linefeed = FALSE;
      return;
    }
    fputs(texte[218], f);
    if (!(v_heap || v_blocks))
    { fputs(texte[215], f); }
    else
    { register RANGELIST *vm;
                /* Liste fuer vektoriellen Speicher abarbeiten */
      for (vm = vmem; vm; vm = vm->next_range)
      { if (vm->r.bis_n < v_min_h_adr || vm->r.von_n > v_max_b_adr ||
            (vm->r.von_n == 0 && vm->r.bis_n == 0))
                /* Bereich ganz ausserhalb */
        { putc('\n', f); ifmore(f);
          if (quitted)
          { fputs(texte[122],f);
            no_linefeed = FALSE;
            return;
          }
          fprintf(f,(vm->r.von_n == vm->r.bis_n) ? texte[224] : texte[198],
                    vm->r.von_n,vm->r.bis_n);
        }
        else
        { if (vm->r.von_n < v_min_h_adr) vm->r.von_n = v_min_h_adr;
          if (vm->r.von_n == 0) vm->r.von_n = 1;
          if (vm->r.bis_n > v_max_b_adr) vm->r.bis_n = v_max_b_adr;
          if (vm->r.bis_n == 0) vm->r.bis_n = -1;
                /* Bereich beschraenkt auf vorhandenen Speicher */
          putc('\n', f); ifmore(f);
          if (quitted)
          { fputs(texte[122],f);
            no_linefeed = FALSE;
            return;
          }
          fprintf(f,(vm->r.von_n == vm->r.bis_n) ? "%d :" : "%d..%d :",
                  vm->r.von_n, vm->r.bis_n);
          if (v_heap && vm->r.von_n < 0)
                /* vektorieller Heap */
          { print_v_heap(f, vm->r.von_n,
                            (vm->r.bis_n >= 0) ? -1 : vm->r.bis_n,
                            mem_pes ? pe_list_string : default_pelist);
            if (!quitted && v_blocks && vm->r.bis_n > 0)
            { putc('\n', f); ifmore(f); }
            if (quitted)
            { if (!block_breaked) fputs(texte[122],f);
              no_linefeed = FALSE;
              return;
            }
          }
          if (v_blocks && vm->r.bis_n > 0)
                /* vektorieller Stack */
          { print_v_mem(f, (vm->r.von_n <= 0) ? 1 : vm->r.von_n,
                           vm->r.bis_n,
                           mem_pes ? pe_list_string : default_pelist);
            if (quitted)
            { if (!block_breaked) fputs(texte[122],f);
              no_linefeed = FALSE;
              return;
            }
          }
        }
      }
    }
  }
  no_linefeed = TRUE;
}

/**************************************************************************
 ***                      Funktion show_stack
 ***
 *** Gibt 'stack' bis zur maximalen Tiefe 'tiefe' auf 'f' aus
 *** 'tiefe < 0' : Ganzen Stack ausgeben
 *** Format : erste Zeile   : * ToS *
 ***          weitere Zeilen: Stackinhalt
 ***          letzte Zeile  : * BoS * , wenn Stack ganz ausgegeben
 ***                            ...   , sonst
 ***
 **************************************************************************/

show_stack(stack, tiefe, f)
STACK *stack;
int tiefe;
FILE *f;
{ register int tos;

  block_breaked = FALSE;
  putc('\n', f); ifmore(f);
  if (!quitted) fputs(texte[219],f);
  if (quitted)
  { fputs(texte[122],f);
    quitted --; block_breaked = TRUE;
    return;
  }
  if (stack) tos = stack->top;
  while (stack && tiefe)
  { if (tos--)
    { putc('\n',f); ifmore(f);
      if (!quitted) print_wert(&stack->speicher[tos], f, KEINER);
      if (quitted)
      { fputs(texte[122],f);
        quitted--; block_breaked = TRUE;
        return;
      }
      if (tiefe > 0) tiefe--;
    }
    else
    { stack = stack->n_stack_part; tos = STACKPARTLEN; }
  }
  if (!tos && stack) stack = stack->n_stack_part;
  putc('\n', f); ifmore(f);
  if (!quitted)
  { fputs(stack ? texte[225] : texte[220], f); }
}

/**************************************************************************
 ***                      Funktion show_vec_stacks
 ***
 *** Gibt die vektoriellen Parameterstacks der PEs 'pes' bis zur Tiefe 'tiefe'
 *** auf 'f' aus
 *** Format wie bei 'show_stack', aber mehrere Stacks nebeneinander,
 ***                                   ueber jedem Stack : 'Pexxx :'
 ***
 **************************************************************************/

show_vec_stacks(tiefe, pes, f)
int tiefe;
char *pes;
FILE *f;
{ register int j;
  register int cols;
  register int pelen = !pe_anz ? 1 : (int)log10((float)pe_anz) + 1;
  register int br = pelen + 6;
  int dummy;
  register int last_pe = last(pes,&dummy);
  STACK *stacks[CRTCOLS / 7];
  int tops[CRTCOLS / 7];
  char fertig[CRTCOLS / 7];

  block_breaked = FALSE;
  br = (max_item_breite + 1 >= br) ? max_item_breite + 2 : br;
  cols = CRTCOLS / br;
                /* Schleife ueber alle PEs */
  for (j = 0; j < pe_anz && !quitted; )
  { register int i, maxcol, t, printing;

    putc('\n', f); ifmore(f);
    if (quitted) break;
    block_breaked = FALSE;
                /* Schleife ermittelt naechste Zeile von Stacks */
    for (maxcol = 0; maxcol < cols && j < last_pe; j++)
    { if (pes[j] == '1')
      { stacks[maxcol] = vec_stacks[j];
        tops[maxcol] = stacks[maxcol] ? stacks[maxcol]->top : STACKPARTLEN;
        fertig[maxcol++] = FALSE;
        fprintf(f,"Pe%0*d :%*s", pelen, j + 1, br - pelen - 4, " ");
      }
    }
    if (maxcol)
    { putc('\n', f); ifmore(f);
      if (quitted)
      { fputs(texte[122], f);
        quitted--; block_breaked = TRUE;
        continue;
      }
                /* '* ToS *' fuer jeden Stack */
      for (i = 0; i < maxcol; i++)
      { fprintf(f,"%-*s", br, texte[219]); }
                /* Schleife ueber Ausgabezeilen */
      for (t = tiefe, printing = TRUE; t && printing; )
      { printing = FALSE;
        if (!quitted) { putc('\n', f); ifmore(f); }
        if (quitted)
        { fputs(texte[122], f);
          quitted--; block_breaked = TRUE;
          break;
        }
                /* Schleife ueber PEs in einer Zeile */
        for (i = 0; i < maxcol; i++)
        { register char *s;

          if (!tops[i])
          { stacks[i] = stacks[i]->n_stack_part; tops[i] = STACKPARTLEN; }
          if (stacks[i])
                /* Wert ausgeben */
          { tops[i]--;
            s = wert_string(&stacks[i]->speicher[tops[i]], KEINER);
            fprintf(f," %-*s", br - 1, s);
            free(s); printing = TRUE;
          }
          else
                /* Stack ist zuende */
          { if (fertig[i])
            { fprintf(f,"%*s", br, " "); }
            else
            { fprintf(f,"%-*s", br, texte[220]); fertig[i] = TRUE; }
          }
        }
        if (t > 0) t--;
      }
      if (printing)
      { if (!quitted) { putc('\n', f); ifmore(f); }
        if (quitted)
        { fputs(texte[122], f);
          quitted--; block_breaked = TRUE;
          continue;
        }
                /* Hinter Stacks Zeile mit '* BoS *' oder '...' oder ' ' ausgeben */
        for (i = 0; i < maxcol; i++)
        { if (!tops[i] && stacks[i]) stacks[i] = stacks[i]->n_stack_part;
          fprintf(f,"%-*s", br, stacks[i] ? texte[225]
                                          : (fertig[i] ? " " : texte[220]));
        }
      }
    }
  }
  if (quitted && !block_breaked) fputs(texte[122], f);
}

/**************************************************************************
 ***                      Funktion show_parstack
 ***
 *** Gibt Teile der skalaren und vektoriellen Parameterstacks auf 'f' aus
 *** sstack : max Ausgabetiefe fuer skalaren Stack
 ***          ganz ausgeben, wenn (sstack == -1 ||
 ***                               (sstack == -2 && vstack == -2))
 *** vstack : analog zu sstack fuer vektorielle Stacks
 *** st_pes : TRUE, wenn pe_list_string gilt
 ***
 **************************************************************************/

show_parstack(f, sstack, vstack, st_pes)
FILE *f;
int sstack, vstack, st_pes;
{ quitted = 0;
  more_count = 0;
  beim_listing = TRUE;
  block_breaked = FALSE;
  if (sstack != -2 || vstack == -2)
  { fputs(texte[221],f);
    show_stack(scal_stack, sstack, f);
    if (quitted)
    { if (!block_breaked) fputs(texte[122],f);
      beim_listing = FALSE;
      no_linefeed = FALSE;
      return;
    }
  }
  if (vstack != -2 || sstack == -2)
  { register char *akt_pes = st_pes ? pe_list_string : default_pelist;

    putc('\n', f); ifmore(f);
    if (!quitted) fprintf(f,texte[222]);
    if (quitted)
    { fputs(texte[122],f);
      beim_listing = FALSE;
      no_linefeed = FALSE;
      return;
    }
    show_vec_stacks(vstack, akt_pes, f);
  }
  beim_listing = FALSE;
  no_linefeed = !quitted;
}

extern DECL *neu_decl();

/**************************************************************************
 ***                      Funktion show_var
 ***
 *** Gibt Variablenwert auf 'f' aus :
 *** *argpt         : Beschreibung der auszugebenden Variablen
 *** o_proc->p_lab  : Label der Prozedur,
 ***                  aus deren Sicht die Variable betrachtet wird
 ***                  < 0 ==> aus der Sicht der aktiven Prozedur
 *** o_proc->p_tief : Anzahl der betrachteten Aktivierungen der Prozedur
 ***                  von der Spitze des Aufrufstapels aus
 *** vpes           : TRUE : pe_list_string gilt,
 ***                         Vektorvariablen werden auf allen PEs ausgegeben
 ***                         auf denen sie existieren
 *** o_decl         : Deklaration der Speicherstruktur der Variablen
 ***
 **************************************************************************/

show_var(f, argpt, o_proc, v_pes, o_decl)
FILE *f;
ARG *argpt;
OPTPROC *o_proc;
int v_pes;
DECLIST *o_decl;
{ if (!disp)
  { fputs(texte[183],f); no_linefeed = TRUE; return; }
  quitted = 0;
  block_breaked = FALSE;
  if (!o_decl->dcount)
                /* passende Deklaration der Laenge 1 erzeugen */
  { register DECL *neu;

    o_decl->typ_zahl[ARG_typ(*argpt)] = 1;
    o_decl->typ_zahl[SUMM] = 1;
    neu = neu_decl(o_decl,FELD);
    if (!abbruch)
    { DECL_t(*neu) = ARG_typ(*argpt);
      DECL_zahl(*neu) = 1;
    }
  }
  if (!abbruch)
  { register char *akt_pe_save = aktive_pes;
    register int last_akt_pe_save = last_akt_pe;
    register int anz_akt_save = anz_akt_pes;
    register char *def_pe_save = default_pelist;

    if (v_pes) default_pelist = pe_list_string;
    if (o_proc->p_lab < 0)
                /* Anzeige aus aktueller Sicht */
    { aktive_pes = disp[ v_pes ? ARG_tiefe(*argpt) : akt_tiefe ]->start_aktiv;
      last_akt_pe = last(aktive_pes,&anz_akt_pes);
      zielits = item_ptr(argpt, &zielbs, disp);
      if (err) err = FALSE;
      else deb_out(zielits,zielbs,NULL,o_decl,0,kommandout);
      if (quitted) fputs(texte[122], f);
    }
    else
                /* Anzeige aus Sicht von 'p_lab : PROC ... ' */
    { register int i, lev, s_tiefe;
      register DISPLAY *dupdisp;
      register AKTBLOCK *abptr;
      register int tief = o_proc->p_tief;

                /* Display kopieren nach 'dupdisp' */
      if (!(dupdisp = (DISPLAY *)calloc((size_t)(max_lev + 1), (size_t)sizeof(DISPLAY))))
      { fputs(texte[80],stderr);
        no_linefeed = FALSE;
        aktive_pes = akt_pe_save; last_akt_pe = last_akt_pe_save; anz_akt_pes = anz_akt_save;
        default_pelist = def_pe_save;
        return;
      }
      for (i = 0; i <= max_lev; i++)
      { dupdisp[i] = disp[i]; }
      lev = akt_tiefe; s_tiefe = 0;
                /* 'tief' mal Anzeige aus Sicht von 'plab : PROC ...' */
      while (lev > 0 && tief)
      { if (abptr = dupdisp[lev])
        { if (abptr->procadr == o_proc->p_lab)
                /* Aktivierungsblock der gewuenschten Prozedur */
          { register int akt_tiefe_save = akt_tiefe;

            block_breaked = FALSE;
            akt_tiefe = lev;
            if (no_linefeed)
            { putc('\n', f); ifmore(f); }
            if (quitted)
            { fputs(texte[122], f);
              quitted--; block_breaked = TRUE;
            }
            else
            { fprintf(f, texte[223], s_tiefe); ifmore(f);
              if (quitted)
              { fputs(texte[122], f);
                quitted--; block_breaked = TRUE;
              }
              else
              { bicount = -1;
                aktive_pes = (v_pes ? dupdisp[ARG_tiefe(*argpt)]
                                    : abptr)->start_aktiv;
                last_akt_pe = last(aktive_pes,&anz_akt_pes);
                zielits = item_ptr(argpt, &zielbs, dupdisp);
                if (err) err = FALSE;
                else
                { deb_out(zielits,zielbs,NULL,o_decl,0,kommandout);
                  no_linefeed = TRUE;
                }
                if (quitted)
                { fputs(texte[122], f);
                  quitted--; block_breaked = TRUE;
                }
              }
            }
            tief--;
            akt_tiefe = akt_tiefe_save;
          }
                /* zum naechsten Aktivierungsblock auf 'dupdisp' */
          dupdisp[lev] = abptr->dispalt;
          lev = abptr->aufr_tiefe;
          s_tiefe++;

          if (quitted) lev = 0;
        }
        else
        { fputs(texte[209], f); more_count++; ifmore(f); }
      }
      if (quitted && !block_breaked) fputs(texte[122],f);
      free(dupdisp);
    }
    aktive_pes = akt_pe_save; last_akt_pe = last_akt_pe_save; anz_akt_pes = anz_akt_save;
    default_pelist = def_pe_save;
  }
  loesch_decl(o_decl);
  no_linefeed = !quitted;
}

/**************************************************************************
 ***                      Funktion show_spez
 ***
 *** gibt Inhalte von Spezialvariablen auf 'f' aus
 *** sp_vars : jedes Bit (0 .. 6) steht fuer eine Spezialvariable :
 ***           MAXTRANS ACTTRANS DONE TERMCH SRESULT ID VRESULT
 ***           ID wird nicht beachtet
 *** sp_pes :  wenn TRUE, dann gilt pe_list_string
 ***
 **************************************************************************/

show_spez(f, sp_vars, sp_pes)
FILE *f;
int sp_vars, sp_pes;
{ register int tok;

  quitted = 0;
  beim_listing = TRUE;
                /* Ausgabe der Skalaren Spezialvariablen */
  for (tok = MAXTRANS; tok <= SRESULT; tok++)
  { if (sp_vars & (1 << (tok - MAXTRANS)))
    { fprintf(f,"%s = ", var_nam[tok-MAXTRANS]);
      print_wert(&s_spez_vars[tok-MAXTRANS], f, KEINER);
      putc('\n', f); ifmore(f);
      if (quitted)
      { fputs(texte[122], f);
        beim_listing = FALSE;
        no_linefeed = FALSE;
        return;
      }
    }
  }
                /* evtl. Ausgabe von VResult */
  if ((sp_vars & (1 << (VRESULT - MAXTRANS))) && v_spez_vars)
  { char s[30], *pes;
    ITEM *it = v_spez_vars + VRESULT_VAR * pe_anz;

    sprintf(s,"%s = ", var_nam[VRESULT - MAXTRANS]);
    pes = sp_pes ? pe_list_string : default_pelist;
    show_vecvar(&it, s, pes, pes, f);
    if (quitted)
    { fputs(texte[122], f); }
  }
  beim_listing = FALSE;
  no_linefeed = !quitted;
}

/**************************************************************************
 ***                      Funktion list_pat_prog
 ***
 *** gibt die Programmzeilen (ohne Verbindungsliste) auf 'f' aus, die ein 
 *** Textmuster enthalten.
 *** pat         : das Textmuster
 *** stringmatch : TRUE  : 'pat' wird als String im Programmtext gesucht
 ***               FALSE : 'pat' wird als Muster fuer grep aufgefasst
 ***
 **************************************************************************/

list_pat_prog(f, pat, stringmatch)
FILE *f;
char *pat;
int stringmatch;
{
#ifndef PC
  char n[257], pu[520];
  register char *tempnam;
  register FILE *tempfile;
  register char *puptr;
                /* Alle Sonderzeichen im Pattern mit '\' versehen */
  for (puptr = pu; *pat; *puptr++ = *pat++)
  { if ((*pat < 'A' || *pat > 'Z') &&
        (*pat < 'a' || *pat > 'z'))
    { *puptr++ = '\\'; }
  }
  *puptr = '\0';
  if (stat_anz == 0)
  { fputs(texte[196],stderr); return; }
  strcpy(n, home_path); strcat(n, "/grep.XXXXXX");
  tempnam = mktemp(n);
  if (!(tempfile = fopen(tempnam,"w")))
  { komerr(texte[194]); }
  else
  { RANGE tr;
    char st[257];

    tr.von_n = 0;
    tr.bis_n = stat_anz - 1;
    more_mod = FALSE;
    list_prog(tempfile, &tr);  /* Ausgabe von Programmtext */
    more_mod = more_mod_save; more_count = 0;
    fclose(tempfile);
                /* (f)grep Kommando erzeugen */
    sprintf(st,"%sgrep -i %s %s", stringmatch ? "f" : "", pu, tempnam);
    if (!(tempfile = popen(st,"r")))
                /* grep schlaegt fehl */
    { komerr(texte[194]); }
    else
                /* Ausgabe von grep lesen und ausgeben */
    { register char *erg = fgets(st, 255, tempfile);

      while (erg)
      { fputs(erg, f); ifmore(f);
        if (quitted)
        { fputs(texte[122], f); pclose(tempfile); unlink(tempnam); return; }
        erg = fgets(st, 255, tempfile);
      }
      pclose(tempfile);
    }
    unlink(tempnam);
  }
#else
  fprintf(f,texte[269]);
#endif
}

/**************************************************************************
 ***                      Funktion to_list
 ***
 *** liefert TRUE, wenn die Verbindung 'vpe vpo TO npe npo'
 *** von der Verbindungsliste 'vlist' abgedeckt wird
 *** d.h. liefert u.a. TRUE fuer 'vlist == NULL'
 ***
 **************************************************************************/

int to_list(vpe, vpo, npe, npo, vlist)
int vpe, vpo, npe, npo;
CONLIST *vlist;
{ if (!vlist) return TRUE;
  else
  { while (vlist)
    { CONLIST con;

      con = *vlist;
      if ( con.v_pe.von_n <= vpe && vpe <= con.v_pe.bis_n &&
           con.v_po.von_n <= vpo && vpo <= con.v_po.bis_n &&
           con.n_pe.von_n <= npe && npe <= con.n_pe.bis_n &&
           con.n_po.von_n <= npo && npo <= con.n_po.bis_n )
      { return TRUE; }
      vlist = vlist->conlist_tail;
    }
  }
  return FALSE;
}

/**************************************************************************
 ***                      Funktion list_connections
 ***
 *** gibt eine Liste von Portverbindungen auf 'f' aus
 *** verb_list : Beschreibt die auszugebenden Verbindungen
 ***             NULL : alle ausgeben
 ***             sonst: alle ausgeben, die auf ein Muster der Liste passen
 ***
 **************************************************************************/

list_connections(f, verb_list)
FILE *f;
CONLIST *verb_list;
{ register int put_nl = FALSE;
               /* Verbindungen aus Zeigerstruktur der Ports rekonstruieren */
  if (portarray)
  { register int pen, pon, kopf_da, count;
    register PORT *pptr, **zpptr;

    quitted = more_count = 0;
    for (pen = 1; pen <= pe_anz; pen++)
    { for (pon = 1; pon <= port_anz; pon++)
      { if ((pptr = &PNR(pen,pon))->zielarr)
        { kopf_da = FALSE;
          for (count = pptr->out_count, zpptr = pptr->zielarr;
               count;
               count--, zpptr++)
          { register int dif = (*zpptr - portarray);
            register int zpe = dif / port_anz + 1;
            register int zpo = dif % port_anz + 1;
                            /* jede Verbindung mit verb_list testen */
            if (to_list(pen, pon, zpe, zpo, verb_list))
            { if (kopf_da)
              { fprintf(f,", %d %d", zpe, zpo); }
              else
              { if (put_nl)
                { putc('\n', f); ifmore(f);
                  if (quitted)
                  { fputs(texte[122],f); 
                    beim_listing = ganzes_listing = FALSE;
                    return;
                  }
                }
                fprintf(f,"%d %d TO %d %d",pen,pon,zpe,zpo);
                kopf_da = TRUE; put_nl = TRUE;
              }
            }
          }
        }
      }
    }
  }
  if (put_nl)
  { putc('\n', f); }
  else
  { fputs(texte[288], f); }
}

/**************************************************************************
 ***                      Funktion list_prog
 ***
 *** gibt das Programm oder Teile davon auf 'f' aus
 *** *text_bereich : Bereich der auszugebenden Befehle
 *** Programmkopf wird nur zusammen mit der ersten Zeile ausgegeben,
 *** STOP nur mit der letzten.
 ***
 **************************************************************************/

list_prog(f, text_bereich)
FILE *f;
RANGE *text_bereich;
{ register STAT *sptr;
  register int z, von = text_bereich->von_n, bis = text_bereich->bis_n;

  if (stat_anz == 0)
  { fputs(texte[196],stderr); return; }
  beim_listing = ganzes_listing = TRUE;
  quitted = more_count = 0;
                /* START etc. nur ausgeben, wenn erste Zeile ausgegeben wird */
  if (von == 0)
  { fputs("START\n",f); ifmore(f);
    fprintf(f,"%d PE\n",pe_anz); ifmore(f);
    fprintf(f,"%d PORTS\n",port_anz); ifmore(f);
    fputs("SCALAR",f);
    list_decl(&glob_s_decl,f); putc('\n',f); ifmore(f);
    fputs("VECTOR",f);
    list_decl(&glob_v_decl,f); putc('\n',f); ifmore(f);
  }
                /* ausgewaehlten Textbereich ausgeben */
  beim_listing = FALSE;
  for (z = von, sptr = programm + von; z <= bis; z++, sptr++)
  { list_stat(sptr,"",f);
    if (quitted)
    { fputs(texte[122],f);
      ganzes_listing = FALSE;
      return;
    }
  }
  if (bis == stat_anz - 1)
  { fputs("STOP\n",f); }
  ganzes_listing = FALSE;
}

/**************************************************************************
 ***                  Funktion list_breakpoints
 ***
 *** gibt alle Zeilen auf 'f' aus, auf die ein Breakpoint gesetzt ist
 ***
 **************************************************************************/

list_breakpoints(f)
FILE *f;
{ register STAT *sptr;
  register int z;

  if (stat_anz == 0)
  { fputs(texte[196],stderr); return; }
  beim_listing = TRUE;
  quitted = 0;
  more_count = 0;
  for (z = stat_anz, sptr = programm; z; z--, sptr++)
  { if (STAT_bpoint(*sptr) & USER_BREAK)
    { list_stat(sptr,"",f);
      if (quitted)
      { fputs(texte[122],f);
        return;
      }
    }
  }
}

/**************************************************************************
 ***                      Funktion loesch_conlist
 ***
 *** gibt den Speicherplatz fuer die verkettete Liste frei, die an 'cl' haengt.
 ***
 **************************************************************************/

loesch_conlist(cl)
CONLIST *cl;
{ register CONLIST *clneu;

  while (cl && cl != (CONLIST *)(-1))
  { clneu = cl->conlist_tail;
    free(cl);
    cl = clneu;
  }
}

/**************************************************************************
 ***                      Funktion loesch_rangelist
 ***
 *** gibt den Speicherplatz fuer die verkettete Liste frei, die an 'rl' haengt.
 ***
 **************************************************************************/

loesch_rangelist(rl)
RANGELIST *rl;
{ register RANGELIST *rlneu;

  while (rl && rl != (RANGELIST *)(-1))
  { rlneu = rl->next_range;
    free(rl);
    rl = rlneu;
  }
}

/**************************************************************************
 ***                      Funktion new_rangelist
 ***
 *** liefert einen Zeiger auf eine neuallozierte Struktur RANGELIST.
 *** Fehlermeldung, wenn kein Platz vorhanden ist.
 ***
 **************************************************************************/

RANGELIST *new_rangelist()
{ register RANGELIST *erg = (RANGELIST *)calloc((size_t)1,(size_t)sizeof(RANGELIST));

  if (erg) erg->next_range = NULL;
  else
  { komerr(texte[80],44); }
  return erg;
}

/**************************************************************************
 ***              Funktion source_ziel
 ***
 *** Ermittelt Befehlsnummer mit Sourcezeile 'zeile' 
 *** 'zeile' unbekannt : naechsthoehere Zeile
 *** keine hoehere Zeile : Ergebnis -1
 ***
 **************************************************************************/

int source_ziel(zeile)
int zeile;
{ register ZEILTAB *zptr;
  register long z, qz;
  register int next = -1;
  register int nextbef = -1;

  for (zptr = lines_table, z = zeilcount; z; zptr++, z--)
  { if ((qz = zptr->source_line) == zeile)
    { return zptr->prog_von; }
    else if (qz > zeile && (next == -1 || qz < next))
    { next = qz; nextbef = zptr->prog_von; }
  }
  return nextbef;
}

/**************************************************************************
 ***              Funktion source_out
 ***
 *** Gibt die drei Sourcezeilen 'zeile - 1' .. 'zeile + 1' auf 'f' aus,
 *** falls die Sourcedatei eingelesen wurde und mindestens 'zeile' Zeilen hat.
 ***
 **************************************************************************/

source_out(zeile, f)
int zeile;
FILE *f;
{ register char *s;
  register int numlen = (int)log10((double)(zeile + 1)) + 1;

  if (source_text && (source_count >= zeile))
  { if (zeile > 1)
    { fprintf(f, texte[241], numlen, zeile - 1, source_text[zeile - 2]);
      ifmore(f);
      if (quitted) return;
    }
    fprintf(f, texte[241], numlen, zeile, s = source_text[zeile - 1]);
    if (s[strlen(s) - 1] != '\n') putc('\n', f);
    ifmore(f);
    if (quitted) return;
    if (zeile < source_count)
    { fprintf(f, texte[241], numlen, zeile + 1, s = source_text[zeile]);
      if (s[strlen(s) - 1] != '\n') putc('\n', f);
      ifmore(f);
    }
  }
}

/**************************************************************************
 ***                      Funktion set_breakpoints
 ***
 *** Setzt/entfernt (entspr. 'sign') den Breakpoint 'sort' (USER_BREAK oder
 *** TO_BREAK) aus den 'count' zeilen in 'lines' mit Offset 'offset'.
 *** Kommentar zur Aktion, falls 'sort == USER_BREAK'
 ***
 **************************************************************************/

set_breakpoints(sign, count, lines, offset, sort)
int sign;
int count;
int *lines;
int offset;
int sort;
{ register int i, l;

  if (sort == USER_BREAK)
  { fputs(texte[sign > 0 ? 239 : 240], kommandout); ifmore(kommandout); }
  for (i = count; i; i--, lines++)
  { if ((l = *lines) >= 0)
    { l += offset;
      if (l < 0 || l >= stat_anz)
      { komerr(texte[174], offset); }
      else
      { l += (STAT_print_func(programm[l]) == print_proc);
        if (sign > 0)
        { STAT_bpoint(programm[l]) |= sort; }
        else
        { STAT_bpoint(programm[l]) &= ~sort; }
        if (sort == USER_BREAK)
        { list_stat(programm + l, "", kommandout); }
      }
    }
  }
}

