/*****************************************************************************
  Project: PARZ - Parallel Intermediate Code Debugger/Interpreter
  ----------------------------------------------------------------------------
  Release      : 1
  Project Part : Simulation
  Filename     : print_fu.c       
  SCCS-Path    : /tmp_mnt/user/sembach/parz/v2/SCCS/s.print_fu.c
  Version      : 1.3 
  Last changed : 9/27/93 at 14:05:05        
  Author       : Frank Sembach
  Creation date: Aug. 92
  ----------------------------------------------------------------------------
  Description  : Funktionen zum Programmlisting fuer PARZ

******************************************************************************
***      (C) COPYRIGHT University of Stuttgart - All Right Reserved        ***
*****************************************************************************/

static char sccs_id[] = "@(#)print_fu.c	1.3  9/27/93 PARZ - Simulation (Frank Sembach)";

/******* Datei print_fu.c ******
 *
 *      Funktionen zum Programmlisting fuer PARZ
 *      Frank Sembach
 */

#include "parzdefs.h"
#include "komdefs.h"
#include "y_tab.h"
#include "rundefs.h"
#include "externs.h"
#include "komexts.h"

char *fehl_str = "@@";  /* kennzeichnet Fehlerhafte Befehlsteile */

/**************************************************************************
 ***                      Funktionen print_xxx
 ***
 *** Zu jeder Befehlsart 'xxx' gibt es eine Ausgabefunktion 'print_xxx'
 ***
 **************************************************************************/

int print_return(bef,f)
STAT *bef;
FILE *f;
{ if (STAT_falsch(*bef)) fputs(fehl_str,f);
  fputs("RETURN",f);
}

int print_halt(bef,f)
STAT *bef;
FILE *f;
{ fputs("HALT",f);
}

int print_end(bef,f)
STAT *bef;
FILE *f;
{ if (STAT_falsch(*bef)) fputs(fehl_str,f);
  fputs("END",f);
}

int print_closeinput(bef,f)
STAT *bef;
FILE *f;
{ fputs("CLOSEINPUT",f);
}

int print_closeoutput(bef,f)
STAT *bef;
FILE *f;
{ fputs("CLOSEOUTPUT",f);
}

int print_zuw(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_minus(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := - ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_not(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := NOT ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_new(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := NEW",f);
  list_decl(STAT_dnew(*bef),f);
}

int print_status(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := STATUS",f);
}

int print_random(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := RANDOM",f);
}

int print_blockmove(bef,f)
STAT *bef;
FILE *f;
{ fputs("MOVE ", f);
  list_arg(&STAT_vc2(*bef),f);
  fputs(" TO ",f);
  list_arg(&STAT_verg(*bef),f);
  fputs(" AS",f);
  list_decl(STAT_dblock(*bef),f);
}

int print_blockequal(bef,f)
STAT *bef;
FILE *f;
{ fputs("EQUAL ", f);
  list_arg(&STAT_vc2(*bef),f);
  fputs("  ",f);
  list_arg(&STAT_vc3(*bef),f);
  fputs(" AS",f);
  list_decl(STAT_dblock(*bef),f);
}

int print_add(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" + ",f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_sub(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" - ",f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_mul(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" * ",f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_div(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" / ",f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_pow(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" ^ ",f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_mod(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" MOD ",f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_and(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" AND ",f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_or(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" OR ",f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_eq(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" = ",f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_ne(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" <> ",f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_lt(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" < ",f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_le(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" <= ",f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_gt(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" > ",f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_ge(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" >= ",f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_sqrt(bef, f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := SQRT ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_exp(bef, f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := EXP ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_ln(bef, f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := LN ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_sin(bef, f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := SIN ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_cos(bef, f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := COS ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_tan(bef, f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := TAN ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_arcsin(bef, f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ARCSIN ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_arccos(bef, f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ARCCOS ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_arctan(bef, f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ARCTAN ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_arctant(bef, f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ARCTANT ",f);
  list_arg(&STAT_vc1(*bef),f);
  putc(' ', f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_abs(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := ABS ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_strcmp(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := STRCMP ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs("  ",f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_connect(bef,f)
STAT *bef;
FILE *f;
{ fputs("CONNECT ",f);
  list_arg(&STAT_out_port(*bef),f);
  fputs(" TO ",f);
  list_arg(&STAT_in_port(*bef),f);
  fputs(" AT ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_biconnect(bef,f)
STAT *bef;
FILE *f;
{ fputs("BICONNECT ",f);
  list_arg(&STAT_out_port(*bef),f);
  fputs(" TO ",f);
  list_arg(&STAT_in_port(*bef),f);
  fputs(" AT ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_disconnect(bef,f)
STAT *bef;
FILE *f;
{ fputs("DISCONNECT ",f);
}

int print_one_disconnect(bef,f)
STAT *bef;
FILE *f;
{ fputs("DISCONNECT ",f);
  list_arg(&STAT_out_port(*bef),f);
}

                /* Relationale Operatoren */
char *rel_nam[GE - EQ + 1] = {" = ", " <> ", " < ", " <= ", " > ", " >= "};

p_i_w(s1,s2,bef,f)              /* Hilfsfunktion fuer if/while_call/goto */
char *s1, *s2;
STAT *bef;
FILE *f;
{ register int rt;

  fputs(s1,f);
  list_arg(&STAT_vc1(*bef),f);
  if (rt = STAT_rel_tok(*bef))
  { fputs(rel_nam[rt - EQ],f);
    list_arg(&STAT_vc2(*bef),f);
  }
  fputs(s2,f);
  list_sprziel(STAT_spr_ziel(*bef),f);
}

int print_ifcall(bef,f)
STAT *bef;
FILE *f;
{ p_i_w("IF "," CALL ",bef,f); }

int print_ifgoto(bef,f)
STAT *bef;
FILE *f;
{ p_i_w("IF "," GOTO ",bef,f); }

int print_whilecall(bef,f)
STAT *bef;
FILE *f;
{ p_i_w("WHILE "," CALL ",bef,f); }

int print_goto(bef,f)
STAT *bef;
FILE *f;
{ fputs("GOTO ",f);
  list_sprziel(STAT_spr_ziel(*bef),f);
}

int print_read(bef,f)
STAT *bef;
FILE *f;
{ fputs("READ ",f);
  list_arg(&STAT_verg(*bef),f);
}

int print_read_string(bef,f)
STAT *bef;
FILE *f;
{ fputs("READ ",f);
  list_arg(&STAT_verg(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_1_write(bef,f)
STAT *bef;
FILE *f;
{ fputs("WRITE ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_2_write(bef,f)
STAT *bef;
FILE *f;
{ fputs("WRITE ",f);
  list_arg(&STAT_vc1(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_3_write(bef,f)
STAT *bef;
FILE *f;
{ fputs("WRITE ",f);
  list_arg(&STAT_vc1(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc2(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc3(*bef),f);
}

int print_1_draw(bef,f)
STAT *bef;
FILE *f;
{ fputs("DRAW ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_2_draw(bef,f)
STAT *bef;
FILE *f;
{ fputs("DRAW ",f);
  list_arg(&STAT_vc1(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_3_draw(bef,f)
STAT *bef;
FILE *f;
{ fputs("DRAW ",f);
  list_arg(&STAT_vc1(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc2(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc3(*bef),f);
}

int print_1_writeln(bef,f)
STAT *bef;
FILE *f;
{ fputs("WRITELN ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_2_writeln(bef,f)
STAT *bef;
FILE *f;
{ fputs("WRITELN ",f);
  list_arg(&STAT_vc1(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_3_writeln(bef,f)
STAT *bef;
FILE *f;
{ fputs("WRITELN ",f);
  list_arg(&STAT_vc1(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc2(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc3(*bef),f);
}

int print_load(bef,f)
STAT *bef;
FILE *f;
{ register int n;

  fputs("LOAD ",f);
  list_arg(&STAT_verg(*bef),f);
  fputs(" WITH ",f);
  list_arg(&STAT_vc1(*bef),f);
  switch (sign(n = STAT_pe_nr(*bef)))
  { case 1  : if (n > pe_anz)
              { fprintf(f," PE %s0", fehl_str); }
              else
              { fprintf(f," PE %d", n); }
              break;
    case -1 : fprintf(f," PE %s%d", fehl_str, -n);
  }
}

int print_blockload(bef,f)
STAT *bef;
FILE *f;
{ fputs("LOAD ",f);
  list_arg(&STAT_verg(*bef),f);
  fputs(" WITH ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" AS",f);
  list_decl(STAT_dload(*bef),f);
}

int print_store(bef,f)
STAT *bef;
FILE *f;
{ register int n;

  fputs("STORE ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" TO ",f);
  list_arg(&STAT_verg(*bef),f);
  switch (sign(n = STAT_pe_nr(*bef)))
  { case 1  : if (n > pe_anz)
              { fprintf(f," PE %s0", fehl_str); }
              else
              { fprintf(f," PE %d", n); }
              break;
    case -1 : fprintf(f," PE %s%d", fehl_str, -n);
  }
}

int print_blockstore(bef,f)
STAT *bef;
FILE *f;
{ fputs("STORE ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" TO ",f);
  list_arg(&STAT_verg(*bef),f);
  fputs(" AS",f);
  list_decl(STAT_dload(*bef),f);
}

                /* Standard Reduce-Funktionen */
char *qual_nam[MIN - AND + 1] = {"AND", "OR", "FIRST", "LAST", "SUM", "PRODUCT", "MAX", "MIN"};

int print_op_reduce(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := REDUCE ",f);
  fputs(qual_nam[STAT_red_fct(*bef) - AND],f);
  fputs(" OF ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_proc_reduce(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := REDUCE ",f);
  list_sprziel(STAT_red_fct(*bef),f);
  fputs(" OF ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_pushs(bef,f)
STAT *bef;
FILE *f;
{ fputs("PUSHS ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_pushv(bef,f)
STAT *bef;
FILE *f;
{ fputs("PUSHV ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_pops(bef,f)
STAT *bef;
FILE *f;
{ fputs("POPS ",f);
  list_arg(&STAT_verg(*bef),f);
}

int print_popv(bef,f)
STAT *bef;
FILE *f;
{ fputs("POPV ",f);
  list_arg(&STAT_verg(*bef),f);
}

int print_call(bef,f)
STAT *bef;
FILE *f;
{ fputs("CALL ",f);
  list_sprziel(STAT_spr_ziel(*bef),f);
}

int print_proc(bef,f)
STAT *bef;
FILE *f;
{ if (STAT_proclev(*bef) <= 0) fputs(fehl_str,f);
  fprintf(f,"PROC %d",abs(STAT_proclev(*bef)));
  if (STAT_dscal(*bef)->darray)
  { fputs(" SCALAR",f);
    list_decl(STAT_dscal(*bef),f);
  }
  if (STAT_dvec(*bef)->darray)
  { fputs(" VECTOR",f);
    list_decl(STAT_dvec(*bef),f);
  }
}

char *para_offset;

int print_parbit(bef,f)
STAT *bef;
FILE *f;
{ register char *bp;
  register int n;
  register char c;

  fprintf(f,"PARALLEL %s",
          (strlen(STAT_parbits(*bef)) == pe_anz) ? "  " : fehl_str);
  for(bp = STAT_parbits(*bef), n = 0; c = *bp; bp++, n++)
  { if (n && !(n % pe_width))
    { putc('\n', f); ifmore(f);
      if (quitted) return;
      fputs(para_offset, f); fprintf(f,"%*s",16 + szlen + lablen, " ");
    }
    putc(c,f);
  }
}

int print_parvar(bef,f)
STAT *bef;
FILE *f;
{ fputs("PARALLEL ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_initset(bef,f)
STAT *bef;
FILE *f;
{ register char *sb;
  register int n;
  register char c;

  list_arg(&STAT_verg(*bef), f);
  fputs(" := INITSET ", f);
  for(sb = STAT_setbits(*bef), n = 0; c = *sb; sb++, n++)
  { if (n && !(n % pe_width))
    { putc('\n', f); ifmore(f);
      if (quitted) return;
      fputs(para_offset, f); fprintf(f,"%*s",24 + szlen + lablen, " ");
    }
    putc(c,f);
  }
}

int print_propagate(bef,f)
STAT *bef;
FILE *f;
{ fputs("PROPAGATE ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" OUT ",f);
  list_arg(&STAT_out_port(*bef),f);
  fputs(" IN ",f);
  list_arg(&STAT_in_port(*bef),f);
}

int print_to_propagate(bef,f)
STAT *bef;
FILE *f;
{ fputs("PROPAGATE ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" TO ",f);
  list_arg(&STAT_verg(*bef),f);
  if (STAT_red_fct(*bef) > AND)
  { fputs(" REDUCE ",f);
    fputs(qual_nam[STAT_red_fct(*bef) - AND],f);
  }
}

int print_redproc_propagate(bef,f)
STAT *bef;
FILE *f;
{ fputs("PROPAGATE ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" TO ",f);
  list_arg(&STAT_verg(*bef),f);
  fputs(" REDUCE ",f);
  list_sprziel(STAT_red_fct(*bef),f);
}

int print_send(bef,f)
STAT *bef;
FILE *f;
{ fputs("SEND ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" TO ",f);
  list_arg(&STAT_verg(*bef),f);
  if (STAT_red_fct(*bef) > AND)
  { fputs(" REDUCE ",f);
    fputs(qual_nam[STAT_red_fct(*bef) - AND],f);
  }
}

int print_redproc_send(bef,f)
STAT *bef;
FILE *f;
{ fputs("SEND ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" TO ",f);
  list_arg(&STAT_verg(*bef),f);
  fputs(" REDUCE ",f);
  list_sprziel(STAT_red_fct(*bef),f);
}

int print_receive(bef,f)
STAT *bef;
FILE *f;
{ fputs("RECEIVE ",f);
  list_arg(&STAT_verg(*bef),f);
  fputs(" FROM ",f);
  list_arg(&STAT_vc1(*bef),f);
  if (STAT_red_fct(*bef) > AND)
  { fputs(" REDUCE ",f);
    fputs(qual_nam[STAT_red_fct(*bef) - AND],f);
  }
}

int print_redproc_receive(bef,f)
STAT *bef;
FILE *f;
{ fputs("RECEIVE ",f);
  list_arg(&STAT_verg(*bef),f);
  fputs(" FROM ",f);
  list_arg(&STAT_vc1(*bef),f);
  fputs(" REDUCE ",f);
  list_sprziel(STAT_red_fct(*bef),f);
}

int print_inconnected(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := CONNECTED IN ",f);
  list_arg(&STAT_cin_port(*bef),f);
}

int print_outconnected(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := CONNECTED OUT ",f);
  list_arg(&STAT_out_port(*bef),f);
}

int print_line_inconnected(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := CONNECTED IN ",f);
  list_arg(&STAT_cin_port(*bef),f);
  fputs(" OUT ",f);
  list_arg(&STAT_out_port(*bef),f);
}

int print_line_outconnected(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := CONNECTED OUT ",f);
  list_arg(&STAT_out_port(*bef),f);
  fputs(" IN ",f);
  list_arg(&STAT_cin_port(*bef),f);
}

int print_openinput(bef,f)
STAT *bef;
FILE *f;
{ fputs("OPENINPUT ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_openoutput(bef,f)
STAT *bef;
FILE *f;
{ fputs("OPENOUTPUT ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_errorcall(bef,f)
STAT *bef;
FILE *f;
{ fputs("ERROR ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_getpixel(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := GETPIXEL ",f);
  list_arg(&STAT_vc1(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_openw(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := OPENW ",f);
  list_arg(&STAT_vc1(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_openabsw(bef,f)
STAT *bef;
FILE *f;
{ list_arg(&STAT_verg(*bef),f);
  fputs(" := OPENABSW ",f);
  list_arg(&STAT_vc1(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_moveto(bef,f)
STAT *bef;
FILE *f;
{ fputs("MOVETO ",f);
  list_arg(&STAT_vc1(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_lineto(bef,f)
STAT *bef;
FILE *f;
{ fputs("LINETO ",f);
  list_arg(&STAT_vc1(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_setpixel(bef,f)
STAT *bef;
FILE *f;
{ fputs("SETPIXEL ",f);
  list_arg(&STAT_vc1(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_wsize(bef,f)
STAT *bef;
FILE *f;
{ fputs("WSIZE ",f);
  list_arg(&STAT_vc1(*bef),f);
  putc(' ',f);
  list_arg(&STAT_vc2(*bef),f);
}

int print_setcolor(bef,f)
STAT *bef;
FILE *f;
{ fputs("SETCOLOR ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_selectw(bef,f)
STAT *bef;
FILE *f;
{ fputs("SELECTW ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_closew(bef,f)
STAT *bef;
FILE *f;
{ fputs("CLOSEW ",f);
  list_arg(&STAT_vc1(*bef),f);
}

int print_debug(bef,f)
STAT *bef;
FILE *f;
{ fputs("DEBUG ",f);
  list_arg(&STAT_vc2(*bef),f);
  putc (' ', f);
  list_arg(&STAT_vc3(*bef),f);
  fputs (" AS", f);
  list_decl(STAT_dblock(*bef), f);
}

int print_trace(bef,f)
STAT *bef;
FILE *f;
{ fputs("TRACE ",f);
  list_arg(&STAT_vc2(*bef),f);
  putc (' ', f);
  list_arg(&STAT_vc3(*bef),f);
  fputs (" AS", f);
  list_decl(STAT_dblock(*bef), f);
}

int print_notrace(bef,f)
STAT *bef;
FILE *f;
{ fputs("NOTRACE ",f);
  if (ARG_typ(STAT_vc1(*bef)) != KEINER)
  { list_arg(&STAT_vc1(*bef),f); }
}

int print_nop(bef,f)
STAT *bef;
FILE *f;
{ fputs("NOP",f);
}

int print_falsch(bef,f)
STAT *bef;
FILE *f;
{ fputs(fehl_str,f);
}

static int lz_flag = FALSE;     /* Flag : Leerzeile schon ausgegeben */

/**************************************************************************
 ***                          Funktion bps
 ***
 *** erzeugt aus 'b' (bpoint-Komponente eines Befehls)
 *** und 'pz' (para_zeile eines Befehls) einen String,
 *** der Compiler- und User-Breakpoints, Sourcezeile und Protokoll-Flags anzeigt
 *** 'bps' schreibt nach 's' und liefert 's'
 ***
 **************************************************************************/

char *bps(b, pz, s)
int b, pz;
char *s;
{ if (max_source_zeile)
  { char sz[20];

    if (pz)
    { sprintf(sz,"%*d",szlen,pz); }
    else
    { sprintf(sz,"%*s",szlen," "); }
    sprintf(s,"%c%s%c%c", (b & COMP_BREAK) ? '!' : ' ',
                          sz,
                          (b & USER_BREAK) ? 'B' : ' ',
                          (b & PROT_ON) ? 'R' : ' ');
  }
  else
  { *s++ = (b & COMP_BREAK) ? '!' : ' ';
    *s++ = (b & USER_BREAK) ? 'B' : ' ';
    *s++ = (b & PROT_ON) ? 'R' : ' ';
    *s = '\0';
    s -= 3;
  }
  return s;
}

/**************************************************************************
 ***                   Funktion list_stat
 ***
 *** gibt die Programmzeile '*sptr' auf 'f' aus.
 *** Bei PARALLEL wird offset zusaetzlich ab der zweiten Zeile ausgegeben.
 *** ganzes_listing == TRUE : Leerzeile vor PROC
 ***                          Leerzeile nach RETURN, HALT, END
 ***
 **************************************************************************/

list_stat(sptr,offset,f)
STAT *sptr;
char *offset;
FILE *f;
{ register int lab = STAT_label(*sptr);
  char b[30];

  beim_listing = TRUE;
  if (ganzes_listing)
  { if (STAT_print_func(*sptr) == print_proc && !lz_flag)
    { putc('\n',f); ifmore(f); }
  }
  if (lab < 0)
  { bps((int)STAT_bpoint(*sptr), STAT_para_zeile(*sptr), b);
    fprintf(f, "%*s%s  ", lablen, "", b);
  }
  else
  { bps((int)STAT_bpoint(*sptr), STAT_para_zeile(*sptr), b);
    fprintf(f, "%*d%s: ", lablen, lab, b);
  }
  para_offset = offset;
  (* STAT_print_func(*sptr))(sptr,f);
  fprintf(f,"; %s\n",STAT_comm_text(*sptr) ? STAT_comm_text(*sptr) : "");
  ifmore(f);
  if (ganzes_listing)
  { if (STAT_print_func(*sptr) == print_return ||
        STAT_print_func(*sptr) == print_halt ||
        STAT_print_func(*sptr) == print_end)
    { putc('\n',f); ifmore(f);
      lz_flag = TRUE;
    }
    else lz_flag = FALSE;
  }
  beim_listing = FALSE;
}

/**************************************************************************
 ***           Funktion list_decl
 ***
 *** gibt die Deklaration 'dl' auf 'f' aus
 ***
 **************************************************************************/

list_decl(dl,f)
DECLIST *dl;
FILE *f;
{ register int i, u;
  register DECL *dec;
  register DECLIST *ud;

  for (dec = dl->darray, i = dl->dcount; i; dec++, i--)
  { switch (DECL_art(*dec))
    { case FELD  : fprintf(f," %c%d", typ_char[DECL_t(*dec)], DECL_zahl(*dec));
                   break;
      case KLAM  : fprintf(f," %d(",DECL_wiederh(*dec));
                   list_decl(&DECL_klamdecl(*dec),f);
                   fputs(" )",f);
                   break;
      case UNION : fputs(" U(", f);
                   for (ud = DECL_ul(*dec).uarray, u = DECL_ul(*dec).ucount; u; ud++, u--)
                   { list_decl(ud, f);
                     if (u > 1) fputs(" ,", f);
                   }
                   fputs(" )",f);
    }
  }
}

/**************************************************************************
 ***               Funktion list_arg
 ***
 *** gibt das Befehlsargument '*aptr' auf 'f' aus
 ***
 **************************************************************************/

list_arg(aptr,f)
ARG *aptr;
FILE *f;
{ register int sort;

  if ((sort = ARG_argsort(*aptr)) & (ADS | ADV))
  { fputs("ADDR ",f);
    if (sort & ADV)
    { sort |= VEC; }
  }
  if (sort & FALSCH)
  { fputs(fehl_str,f); }
  if (sort & CON)
  { if ((sort & SPEZ) && AA_typ(sort) == INT)
    { fputs("NIL",f); }
    else if (sort & SIZ)
    { fputs("SIZE(", f);
      list_decl(ARG_sizedec(*aptr), f);
      fputs(" )", f);
    }
    else
    { print_wert(&ARG_con_wert(*aptr),f,ARG_typ(*aptr)); }
  }
  else
  { if (sort & INDL)
    { fprintf(f,"%c%c%d:[",
              (sort & VEC) ? 'V' : 'S',
              typ_char[AA_typ(sort)],
              ARG_vartiefe(*aptr));
      if ((sort = ARG_indsort(*aptr)) & FALSCH)
      { fputs(fehl_str,f); }
    }
    else if(sort & IND)
    { fprintf(f,"%c%c[",
              (sort & VEC) ? 'V' : 'S',
              typ_char[AA_typ(sort)]);
      if ((sort = ARG_indsort(*aptr)) & FALSCH)
      { fputs(fehl_str,f); }
    }
    if (sort & SPEZ)
    { fputs(var_nam[ARG_vartok(*aptr) - MAXTRANS],f); }
    else
    { fprintf(f,"%c%c%d:%d",
              (sort & VEC) ? 'V' : 'S',
              typ_char[AA_typ(sort)],
              ARG_tiefe(*aptr),ARG_num(*aptr));
    }
    if (ARG_argsort(*aptr) & (INDL | IND))
    { putc(']',f); }
  }
}

/**************************************************************************
 ***                        Funktion list_sprziel
 ***
 *** gibt 'z' als Sprungziel auf 'f' aus : Sprungziel < 0 gilt als falsch
 ***
 **************************************************************************/

list_sprziel(z,f)
int z;
FILE *f;
{ if (z < 0)
  { fprintf(f,"%s%d",fehl_str,-z); }
  else
  { fprintf(f,"%d",STAT_label(programm[z])); }
}

/**************************************************************************
 ***             Funktion wert_string
 ***
 *** liefert Zeiger auf Ausgabestring fuer '*iptr'
 *** alloziert neuen Speicherplatz fuer Ergebnis
 *** t : 'Soll'typ
 ***
 **************************************************************************/

char *wert_string(iptr, t)
ITEM *iptr;
TYP t;
{ register unsigned char c;
  register char *st, *erg;
  register size_t l;
  register TYP ityp = ITEM_typ(*iptr);

  if (!(erg = malloc((size_t)300)))
  { fprintf(kommandout, "\n%s", texte[80]); return NULL; }
  if (ityp != KEINER)
  { if (t != KEINER && t != ityp)
    { sprintf(erg, "%c in %c", typ_char[ityp], typ_char[t]); }
    else if (t == BOOL || (t == KEINER && ityp == BOOL))
    { strcpy(erg, iptr->inhalt.b_val ? "TRUE" : "FALSE"); }
    else if (t == CHA || (t == KEINER && ityp == CHA))
    { st = beim_listing ? "\'" : "";
      c = iptr->inhalt.c_val;
      if (beim_listing)
      { if (c == EOL_CHAR)
        { strcpy(erg, "EOL"); goto reterg; }
        else if (c == TERMS_CHAR)
        { strcpy(erg, "termS"); goto reterg; }
        else if (c == '\'')
        { strcpy(erg, "''''"); goto reterg; }
        else if (c < MIN_PRINTING_CHAR || c > MAX_PRINTING_CHAR)
        { sprintf(erg, "CHR(%d)", (int)c); goto reterg; }
      }
      sprintf(erg,"%s%c%s", st, c, st);
    }
    else if (t == INT || (t == KEINER && ityp == INT))
    { sprintf(erg,"%ld",iptr->inhalt.i_val); }
    else if (t == REAL || (t == KEINER && ityp == REAL))
    { if (fabs(iptr->inhalt.r_val) >= HUGE)
      { strcpy(erg, "<NaN>"); }
      else
#  ifdef SUN
      { sprintf(erg, iptr->inhalt.r_val ? "%#g" : "%#f", iptr->inhalt.r_val); }
#  else
      { sprintf(erg,"%#g",iptr->inhalt.r_val); }
#  endif
    }
    else if (ityp == STR)
    { if (st = iptr->inhalt.s_val)
      { if (beim_listing)
                /* '"' verdoppeln */
        { register char *ergpt = erg;

          *ergpt++ = '"';
          for (; *st; st++)
          { *ergpt++ = *st;
            if (*st == '"') *ergpt++ = '"';
          }
          *ergpt++ = '"'; *ergpt = '\0';
        }
        else
        { strcpy(erg, st); }
      }
      else
      { strcpy(erg, fehl_str); }
    }
    else bug("print_fu.c/wert_string : falscher Typ");
  }
  else
  { register TYP d = iptr->datentyp;

    if (t == KEINER)
    { sprintf(erg,"%s%s%s%s : ??",d & BOOL_ERLAUBT ? "B" : "",
                                  d & CHA_ERLAUBT ? "C" : "",
                                  d & INT_ERLAUBT ? "I" : "",
                                  d & REAL_ERLAUBT ? "R" : "");
    }
    else
    { sprintf(erg,"%c : ??", typ_char[t]); }
  }
reterg :
  if (beim_listing && ityp != KEINER && (iptr->datentyp & UNDEF_ERG))
  { strcpy(erg + strlen(erg), "?"); }
  return realloc(erg, (size_t)(strlen(erg) + 1));
}

/**************************************************************************
 ***         Funktion print_wert
 ***
 *** gibt Wert von '*iptr' auf 'f' aus
 ***
 **************************************************************************/

print_wert(iptr,f,t)
ITEM *iptr;
FILE *f;
TYP t;
{ register char *s;

  if (s = wert_string(iptr, t))
  { fputs(s,f);
    free(s);
  }
}
