/*****************************************************************************
  Project: PARZ - Parallel Intermediate Code Debugger/Interpreter
  ----------------------------------------------------------------------------
  Release      : 1
  Project Part : Simulation
  Filename     : run.c       
  SCCS-Path    : /tmp_mnt/user/sembach/parz/v2/SCCS/s.run.c
  Version      : 1.3 
  Last changed : 9/27/93 at 14:05:03        
  Author       : Frank Sembach
  Creation date: Aug. 92
  ----------------------------------------------------------------------------
  Description  : Hauptprogramm des Simulationsteils von PARZ

******************************************************************************
***      (C) COPYRIGHT University of Stuttgart - All Right Reserved        ***
*****************************************************************************/

static char sccs_id[] = "@(#)run.c	1.3  9/27/93 PARZ - Simulation (Frank Sembach)";


#include "parzdefs.h"
#include "rundefs.h"
#include "komdefs.h"
#include "y_tab.h"
#include "komexts.h"
#include "debexts.h"
#include "externs.h"
#include "funcs.h"

/*************************************************************************/

int pc;                 /* Programmzaehler */
int pec;                /* Prozessorzaehler */
int akt_tiefe;          /* momentane Prozedurebene */
int call_count;         /* zaehlt Zahl der aktiven CALL-Aufrufe */
long ausgefuehrt;       /* Zahl der ausgefuehrten Befehle */
SCOPETAB *newest_scope; /* neuester Geltungsbereich */
SCOPETAB *akt_scope;    /* aktueller Geltungsbereich*/
ZEILTAB *akt_line;      /* aktueller Quellzeileneintrag */
char called;            /* Flag : Befehl erreicht durch CALL */
char reducing;          /* Flag : REDUCE wird gerade ausgefuehrt */
char ended;             /* Flag : Programm fertig */
char err;               /* Flag : Fehler ist aufgetreten */
char vec_bef;           /* Flag : Vektorbefehl wird ausgefuehrt */
char zeile_fertig;      /* Flag : Zeile von stdin fertiggelesen */
char forced_break;      /* Flag : gewaltsamer Programmabbruch */
char no_linefeed;       /* Flag : Ausgabezeile noch nicht fertig */

char im_programm = FALSE; /* Flag : Simulation laeuft ab */

DISPLAY *disp;          /* Stack-Display */

VARBLOCK *s_blocks,
         *v_blocks,     /* Listen der fuer lokale Variablen allozierten Bloecke */
         *s_heap,
         *v_heap;       /* Listen der mit NEW allozierten Bloecke */

int s_max_b_adr, v_max_b_adr; /* bisher maximal verwendete Adressen in lok. Var. */
int s_min_h_adr, v_min_h_adr; /* bisher minimal verwendete Adressen in Heap */

ITEM s_spez_vars[5] =   /* Variablen MAXTRANS, ACTTRANS, DONE, TERMCH, SRESULT */
{ { KEINER | INT_ERLAUBT },
  { KEINER | INT_ERLAUBT },
  { KEINER | BOOL_ERLAUBT },
  { KEINER | CHA_ERLAUBT },
  { KEINER | BOOL_ERLAUBT }
};
VARBLOCK s_spez_block = /* Pseudo-Variablenblock fuer Skalar-Spezialvariablen */
{ s_spez_vars,
  1,
  5,
  NULL,
  NULL,
  NULL
};
ITEM *v_spez_vars;      /* Variablen ID, VRESULT */
VARBLOCK v_spez_block = /* Pseudo-Variablenblock fuer Vektor-Spezialvariablen */
{ NULL,
  1,
  2,
  NULL,
  NULL,
  NULL
};

int bicount;            /* Zaehler fuer Aufrufe von 'item_ptr()' in einem Befehl */

char *aktive_pes;       /* Bitstring : gibt momentan aktive PEs an */
char *dummy_aktive_pes; /* Prozessoraktivierung fuer LOAD und STORE mit einem PE */
int last_akt_pe;        /* Nummer des letzten aktiven PEs */
int anz_akt_pes;        /* Zahl der momentan aktiven PEs */

AKTBLOCK *step_over_block;      /* Aktivierungsblock in dem gesteppt wird */
AKTBLOCK *akt_step_block;       /* Aktueller echter Aktivierungsblock */

int tr_flag;            /* Flag : In Befehl wurde TRACE-Variable veraendert */

ITPTRS *zielits;        /* Beschreibung der Zielvariable des aktuellen Befehls */
VBLOCKS *zielbs, *dummybs; /* Blockbeschreibungen fuer ziel und quelle */

int exit_code;          /* Rueckgabewert eines 'start()'-Aufrufs in REDUCE */

AKTBLOCK *new_aktblock();
int akt_return();
ITPTRS *item_ptr();
VARBLOCK *make_vars();

/*************************************************************************/

int anz_runwarnings;    /* Zahl der ausgegebenen Warnungen zu einem Befehl */

int pcalt;              /* Wert von 'pc' vor der Befehlsausfuehrung */

VBLOCKS oneblock[3] = {{0}, {0}, {0}};  /* Platz fuer Blockbeschreibungen mit 
                                           je einem Block */
VBLOCKS *moreblocks[3]; /* Zeiger auf Blockbeschreibungen mit je mehreren Bloecken */

ITPTRS oneitem[3];      /* Platz fuer Variablenbeschreibungen mit je einem ITEM */
ITPTRS *moreitems[3];   /* Zeiger auf Variablenbeschreibungen mit je mehreren ITEMS */

ITEM hilf_item[3];      /* Hilfs-Speicherstellen fuer Konstanten */

extern char *wert_string();

extern FILE *yyerfp;    /* Ausgabestream fuer Fehlermeldungen */

extern int graph_bef_exist;
extern int init_x();

/**************************************************************************
 ***                       Funktion run_prog
 ***
 *** Initialisiert Simulator fuer Programmausfuehrung und startet Programm
 *** Fehler, wenn kein ausfuehrbares Programm geladen
 *** Ergebnis ist Exit-Code der Programmausfuehrung
 ***
 **************************************************************************/

int run_prog()
{ register int ex;

  if (!file_gelesen)
  { fputs(texte[3],stderr); return 0; }
  aufraeumen();
  init_ausfuehrung();
  if (err) return 0;
  yyerfp = (kommandout == stdout) ? stderr : kommandout;
  ex = start(0);
  if (no_linefeed && runout == stdout)
  { putc('\n', runout); no_linefeed = FALSE; }
  if (prot && prot_offen) putc('\n', protokollout);
  return ex;
}

/**************************************************************************
 ***          Funktion loesch_trace
 ***
 *** gibt die TRACE-Information in 'tv' frei
 ***
 **************************************************************************/

loesch_trace(tv)
TRACEVAR *tv;
{ TRACEVAR *tvneu;

  while (tv)
  { tvneu = tv->t_naechst;
    free(tv);
    tv = tvneu;
  }
}

/**************************************************************************
 ***                     Funktion free_heap
 ***
 *** gibt durch NEW allozierte Bloecke in der Liste 'vbptr' frei
 ***
 **************************************************************************/

free_heap(vbptr)
VARBLOCK *vbptr;
{ register VARBLOCK *vp;

  while (vbptr)
  { free(vbptr->vars);
    loesch_trace(vbptr->t_info);
    if (vbptr->max_pes) free(vbptr->max_pes);
    vp = vbptr->nblock;
    free(vbptr);
    vbptr = vp;
  }
}

/**************************************************************************
 ***     Funktion STACK_loeschen
 ***
 *** gibt den Stack '*stackptr' frei
 ***
 **************************************************************************/

STACK_loeschen(stackptr)
STACK **stackptr;
{ register STACK *st;

  while (*stackptr)
  { st = (*stackptr)->n_stack_part;
    free(*stackptr);
    *stackptr = st;
  }
}

/**************************************************************************
 ***                   Funktion aufraemen
 ***
 *** Bereinigt Speicher vor Programmausfuehrung:
 *** Gibt allen erreichbaren allozierten Speicherplatz frei.
 ***
 **************************************************************************/

aufraeumen()
{ char *dummy;
  int dumflag;
  register int i;
                /* Aktivierungsstack abraeumen */
  if (disp)
  { while (akt_tiefe >= 0)
    { akt_return(&dummy, &dumflag);
      if (dummy) free(dummy);
    }
    free(disp);
    disp = NULL;
  }
                /* Aktivierungsstrings entfernen */
  if (aktive_pes)
  { free(aktive_pes);
    aktive_pes = NULL; last_akt_pe = 0; anz_akt_pes = 0;
  }
  if (dummy_aktive_pes)
  { free(dummy_aktive_pes);
    dummy_aktive_pes = NULL;
  }
                /* TRACE-Infos fuer Spezialvariablen entfernen */
  loesch_trace(s_spez_block.t_info);
  loesch_trace(v_spez_block.t_info);
                /* Zusaetzliche TRACE-Befehle entfernen */
  if (kommand_traces)
  { loesch_prog(&kommand_traces, &akt_kom_tr, TRUE);
    max_kom_tr = 0;
  }
                /* Verbindungen entfernen */
  if (portarray)
  { register PORT *pptr;
    
    for (pptr = portarray + pe_anz * port_anz - 1; pptr >= portarray; pptr--)
    { if (pptr->zielarr)
      { free(pptr->zielarr);
        pptr->zielarr = NULL;
        pptr->out_count = 0;
      }
    }
  }
                /* Parameterstacks entfernen */
  STACK_loeschen(&scal_stack);
  if (vec_stacks)
  { for (i = 0; i < pe_anz; STACK_loeschen(vec_stacks + i++)); }
                /* Vektorielle Spezialvariablen entfernen */
  if (v_spez_block.vars)
  { free(v_spez_block.vars);
    v_spez_block.vars = v_spez_vars = NULL;
  }
                /* Argumentbeschreibungen entfernen */
  for (i = 0; i < 3; i++)
  { if (moreblocks[i])
    { free(moreblocks[i]);
      moreblocks[i] = NULL;
    }
    if (moreitems[i])
    { free(moreitems[i]);
      moreitems[i] = NULL;
    }
  }
                /* Heap entfernen */
  free_heap(s_heap); s_heap = NULL;
  free_heap(v_heap); v_heap = NULL;
  v_min_h_adr = s_min_h_adr = 0;
}

/**************************************************************************
 ***                   Funktion init_ausfuehrung
 ***
 *** Initialisiert Simulator zur Ausfuehrung des geladenen Programms
 ***
 **************************************************************************/

init_ausfuehrung()
{ register int i;
  register char *cptr, *cdptr;
  register ITEM *iptr;

  called = FALSE;
  reducing = FALSE;
  call_count = 0;
  ended = FALSE;
  err = FALSE;
  no_linefeed = FALSE;
  zeile_fertig = TRUE;
  forced_break = FALSE;
  pc = -1;
  ausgefuehrt = 0;
  akt_scope = newest_scope = NULL;
  akt_line = NULL;
                /* Display der Laenge 'max_lev + 1' erzeugen */
  if (!(disp = (DISPLAY *)calloc((size_t)(max_lev + 1), (size_t)sizeof(DISPLAY))))
  { runerr(texte[4]); return; }
                /* Aktivierungsstrings erzeugen und vorbelegen */
  if (!(aktive_pes = calloc((size_t)(pe_anz + 1), (size_t)sizeof(char))))
  { runerr(texte[4]);
    free(disp); disp = NULL;
    return;
  }
  if (!(dummy_aktive_pes = calloc((size_t)(pe_anz + 1), (size_t)sizeof(char))))
  { runerr(texte[4]);
    free(disp); disp = NULL;
    free(aktive_pes); aktive_pes = NULL;
    return;
  }
  for (i = pe_anz, cptr = aktive_pes, cdptr = dummy_aktive_pes;
       i;
       *(cptr++) = '1', *(cdptr++) = '0', i--);
  *cptr = *cdptr = '\0'; last_akt_pe = pe_anz; anz_akt_pes = pe_anz;
                /* Spezialvariablen erzeugen und vorbesetzen */
  if (!(v_spez_vars = (ITEM *)calloc((size_t)(pe_anz ? 2 * pe_anz : 1), (size_t)sizeof(ITEM))))
  { runerr(texte[5]);
    free(disp); disp = NULL;
    free(aktive_pes); aktive_pes = NULL; last_akt_pe = 0; anz_akt_pes = 0;
    free(dummy_aktive_pes); dummy_aktive_pes = NULL;
    return;
  }
  for (i = MAXTRANS_VAR;
       i <= SRESULT_VAR;
       s_spez_vars[i].datentyp = CH_TYP(s_spez_vars[i].datentyp, KEINER), i++);
  v_spez_block.vars = v_spez_vars;
  for (i = 1, iptr = v_spez_vars; i <= pe_anz; i++, iptr++)     /* ID */
  { iptr->datentyp = (INT | INT_ERLAUBT);
    iptr->inhalt.i_val = i;
  }
  for (i = 1; i <= pe_anz; i++, (iptr++)->datentyp = KEINER | BOOL_ERLAUBT);     /* VRESULT */
                /* Adresszaehler fuer Aktivierungsstack <- 0 */
  v_max_b_adr = 0;
  s_max_b_adr = 0;
                /* Variablen- und Blockbeschreibungen fuer Argumente erzeugen 
                   und vorbesetzen */
  for (i = 0; i < 3; i++)
  { if (!(moreblocks[i] =
            (VBLOCKS *)calloc((size_t)1,
                              (size_t)(sizeof(VBLOCKS) +
                                       (pe_anz - 1) * sizeof(VARBLOCK *)))))
    { register int j;

      runerr(texte[4]);
      free(disp); disp = NULL;
      free(aktive_pes); aktive_pes = NULL; last_akt_pe = 0; anz_akt_pes = 0;
      free(dummy_aktive_pes); dummy_aktive_pes = NULL;
      free(v_spez_vars); v_spez_vars = NULL;
      for (j = 0; j < i; free(moreblocks[j]), moreblocks[j++] = NULL);
      return;
    }
    moreblocks[i]->inc = 1;
  }
  for (i = 0; i < 3; i++)
  { if (!(moreitems[i] =
          (ITPTRS *)calloc((size_t)1,
                           (size_t)(sizeof(ITPTRS) + (pe_anz - 1) * sizeof(ITEM *)))))
    { register int j;

      runerr(texte[4]);
      free(disp); disp = NULL;
      free(aktive_pes); aktive_pes = NULL; last_akt_pe = 0; anz_akt_pes = 0;
      free(dummy_aktive_pes); dummy_aktive_pes = NULL;
      free(v_spez_vars); v_spez_vars = NULL;
      for (j = 0; j < 3; free(moreblocks[j]), moreblocks[j++] = NULL);
      for (j = 0; j < i; free(moreitems[j]), moreitems[j++] = NULL);
      return;
    }
    moreitems[i]->n_it_func = n_item_ptr;
  }
                /* Ein-/Ausgabedateien auf stdio setzen */
  if (runin != stdin) { fclose(runin); runin = stdin; }
  if (runout != stdout) { fclose(runout); runout = stdout; }
                /* Aktivierungsblock fuer Hauptprogramm erzeugen */
  akt_tiefe = -1;
  akt_step_block = NULL;
  fill_aktblock(new_aktblock(0, NULL, TRUE), 0, -1, &glob_s_decl, &glob_v_decl);
  step_over_block = akt_step_block;
  if (graph_bef_exist) init_x();
}

/**************************************************************************
 ***                         Funktion last
 ***
 *** liefert die Nummer des letzten aktiven PEs im Aktivierungsstring 'pes',
 *** 0 fuer 'pes == NULL'
 ***
 **************************************************************************/

int last(pes,anz)
char *pes;
int *anz;
{ if (pes)
  { register int erg;
    register char *peptr;

    *anz = 0;
    for (erg = pe_anz, peptr = pes + pe_anz - 1;
         erg && *peptr != '1';
         erg--, peptr--);
    if (erg)
      for (; peptr >= pes; *anz += (*(peptr--) == '1') );
    return erg;
  }
  else
  { return 0; }
}

/**************************************************************************
 ***                   Funktion deb_print (rekursiv)
 ***
 *** gibt DEBUG-Information fuer einen Skalaren Variablenbereich auf 'f' aus
 *** *itpt    : Zeiger auf Anfang des Variablenbereichs (durchlaeuft Variablen)
 *** dptr     : Zeiger auf Deklaration des Variablenbereichs
 *** indent   : Rekursionstiefe fuer Einrueckung
 *** set_tr   : Flag: soll TRACE-Zaehler in Variablen erhoeht werden
 *** Ergebnis : Zahl der ausgegebenen Zeilen
 *** Fehler bei Typdifferenzen zw. Deklaration und Speicher
 ***
 **************************************************************************/

int deb_print(itpt, dlptr, indent, set_tr, f, sep)
ITEM **itpt;
DECLIST *dlptr;
int indent, set_tr;
FILE *f;
char sep;
{ register int i, count, erg = 0;
  register DECL *dptr;

  for (count = dlptr->dcount, dptr = dlptr->darray; count; dptr++, count--)    /* Schleife ueber Deklaration */
  { switch(DECL_art(*dptr))
    { case FELD  :                /* echte Deklaration */
      { register TYP t = DECL_t(*dptr);
        register int z = DECL_zahl(*dptr);

        if (t == CHA && z > 1)                    /* Ausgabe als String */
        { register int auslen = 0;

          if (deb_mod && !quitted)
          { if (erg > 0 || indent > 0) { putc('\n', f); ifmore(f); }
            if (indent) fprintf(f,"%*s", 2 * indent, " ");
            putc('"', f);
            auslen = 2 * indent + 1;
          }
          for (i = z; i && *itpt; i--, (*itpt)++)       /* Schleife ueber String */
          { if (!((*itpt)->datentyp & ERLAUBT(t)))
            { runerr(texte[141]); return 0; }
            (*itpt)->datentyp += TRACED * set_tr;
            if (deb_mod && !quitted)
            { register int kflag;
              register char *ws;

              if (kflag = (ITEM_typ(**itpt) != t ||
                           ((*itpt)->datentyp & UNDEF_ERG) ||
                           (*itpt)->inhalt.c_val == EOL_CHAR ||
                           (*itpt)->inhalt.c_val == TERMS_CHAR ||
                           (*itpt)->inhalt.c_val < MIN_PRINTING_CHAR ||
                           (*itpt)->inhalt.c_val > MAX_PRINTING_CHAR)) /* Sonderform */
              { beim_listing = TRUE;
                ws = wert_string(*itpt, CHA);
                if (auslen + 2 + strlen(ws) > CRTCOLS - 2 * indent - 1) /* zu lang */
                { putc('\n', f); ifmore(f);
                  if (!quitted)
                  { fprintf(f,"%*s", 2 * indent + 1, " ");
                    auslen = 2 * indent + 1;
                  }
                }
                putc('<', f); auslen += 1;
              }
              else                        /* ein Buchstabe */
              { ws = wert_string(*itpt, CHA);
                if (auslen + strlen(ws) > CRTCOLS - 2 * indent - 1) /* zu lang */
                { putc('\n', f); ifmore(f);
                  if (!quitted)
                  { fprintf(f,"%*s", 2 * indent + 1, " ");
                    auslen = 2 * indent + 1;
                  }
                }
              }
              if (!quitted)
              { fputs(ws, f); auslen += strlen(ws); }
              free(ws);
              if (kflag)                          /* Sonderform Ende */
              { beim_listing = FALSE;
                if (!quitted)
                { putc('>', f); auslen += 1; }
              }
            }
          }
          if (deb_mod && !quitted) 
          { putc('"', f); erg++; }        /* Stringende */
        }
        else                              /* Ausgabe als Einzelvariablen */
        { for (i = z; i && *itpt; i--, (*itpt)++)
          { if (!((*itpt)->datentyp & ERLAUBT(t)))
            { runerr(texte[141]); return 0; }
            (*itpt)->datentyp += TRACED * set_tr;
            if (deb_mod && !quitted)
            { if (erg > 0 || indent > 0) { putc('\n', f); ifmore(f); }
              if (indent) fprintf(f,"%*s", 2 * indent, " ");
              beim_listing = TRUE;
              if (!quitted) print_wert(*itpt, f, t); erg++;
              beim_listing = FALSE;
            }
          }
        }
      }
      break;
      case KLAM  :
      { register int w = DECL_wiederh(*dptr);
        register DECLIST *di = &DECL_klamdecl(*dptr);

        for (i = w;
             i && *itpt;
             i--, erg += deb_print(itpt, di, indent + 1, set_tr, f, '|'));
      }
      break;
      case UNION :
      { register int count = DECL_ul(*dptr).ucount;
        register DECLIST *di;
        ITEM *lok_itpt = *itpt;

        for (i = count, di = DECL_ul(*dptr).uarray;
             i && lok_itpt;
             i--)
        { lok_itpt = *itpt;
          erg += deb_print(&lok_itpt, di++, indent + 1,
                           set_tr, f, i == 1 ? ';' : ',');
        }
        *itpt += DECL_ul(*dptr).utyp_anz[SUMM];
      }
    }
  }
  if (deb_mod && !quitted && (erg > 1 || sep != '|') && indent)
  { putc(' ', f); putc(sep, f); }
  return erg;
}

int strichzahl, akt_striche;    /* Zaehler fuer Striche hinter Debug-Zeilen */

/**************************************************************************
 ***                    Funktion deb_breite (rekursiv)
 ***
 *** Abwandlung von 'deb_print()': ermittelt max. Ausgabebreite,
 *** max. Zahl der Striche hinter Variablenwerten, Anz. auszugebende Zeilen
 *** fuer Vektorvariablenbereich auf einem PE
 *** *zeilen  : Rueckgabewert: Anzahl auszugebende Zeilen
 *** *itpt    : Zeiger auf Variablenbereich (laeuft mit)
 *** dptr     : Zeiger auf Deklaration
 *** indent   : Rekursionstiefe fuer Einruecken
 *** set_tr   : Flag: soll TRACE-Zaehler in Variablen hochgezaehlt werden?
 *** Ergebnis : maximale Ausgabebreite ohne Striche (-1 bei Fehler)
 *** maximale Strichzahl steht in 'strichzahl'
 *** Fehlermeldung bei Typkonflikt zw. Deklaration und Speicher
 ***
 **************************************************************************/

int deb_breite(zeilen, itpt, dlptr, indent, set_tr, sep)
int *zeilen;
ITEM **itpt;
DECLIST *dlptr;
int indent, set_tr;
char sep;
{ register int i, count, len, erg = 0;
  register char *sneu;
  register DECL *dptr;

  *zeilen = 0;
  for (count = dlptr->dcount, dptr = dlptr->darray; count; dptr++, count--)    /* Schleife ueber Deklaration */
  { switch(DECL_art(*dptr))
    { case FELD  :                /* echte Deklaration */
      { register TYP t = DECL_t(*dptr);
        register int z = DECL_zahl(*dptr);

        if (t == CHA && z > 1)
        { len = 0;
          if (akt_striche > strichzahl) strichzahl = akt_striche;
          akt_striche = 0;
          len++;  /* '"' */
          for (i = z; i && *itpt; i--, *itpt += pe_anz)
          { register int kflag;

            if (!((*itpt)->datentyp & ERLAUBT(t)))
            { runerr(texte[141]); return -1; }
            (*itpt)->datentyp += TRACED * set_tr;
            if (kflag = (ITEM_typ(**itpt) != t ||
                         ((*itpt)->datentyp & UNDEF_ERG) ||
                         (*itpt)->inhalt.c_val == EOL_CHAR ||
                         (*itpt)->inhalt.c_val == TERMS_CHAR ||
                         (*itpt)->inhalt.c_val < MIN_PRINTING_CHAR ||
                         (*itpt)->inhalt.c_val > MAX_PRINTING_CHAR))
            { beim_listing = TRUE;
              len++; /* '<' */
            }
            sneu = wert_string(*itpt, t);
            len += strlen(sneu);
            free(sneu);
            if (kflag)
            { beim_listing = FALSE;
              len++; /* '>' */
            }
          }
          len++; /* '"' */
          if (len > erg) erg = len;
          (*zeilen)++;
        }
        else
        { for (i = z; i && *itpt; i--, *itpt += pe_anz)
          { if (!((*itpt)->datentyp & ERLAUBT(t)))
            { runerr(texte[141]); return -1; }
            (*itpt)->datentyp += TRACED * set_tr;
            if (akt_striche > strichzahl) strichzahl = akt_striche;
            akt_striche = 0;
            beim_listing = TRUE;
            len = strlen(sneu = wert_string(*itpt, t));
            if (len > erg) erg = len;
            free(sneu);
            beim_listing = FALSE;
            (*zeilen)++;
          }
        }
      }
      break;
      case KLAM  :
      { register int w = DECL_wiederh(*dptr);
        register DECLIST *di = &DECL_klamdecl(*dptr);
        int zeil;

        for (i = w; i && *itpt; i--)
        { len = 2 + deb_breite(&zeil, itpt, di, indent + 1, set_tr, '|');
          if (len < 2) return -1;
          *zeilen += zeil;
          if (len > erg) erg = len;
        }
      }
      break;
      case UNION :
      { register int count = DECL_ul(*dptr).ucount;
        register DECLIST *di;
        ITEM *lok_itpt = *itpt;
        int zeil;

        for (i = count, di = DECL_ul(*dptr).uarray;
             i && lok_itpt;
             i--)
        { lok_itpt= *itpt;
          len = 2 + deb_breite(&zeil, &lok_itpt, di++, indent + 1, set_tr, ',');
          if (len < 2) return -1;
          *zeilen += zeil;
          if (len > erg) erg = len;
        }
        *itpt += pe_anz * DECL_ul(*dptr).utyp_anz[SUMM];
      }
    }
  }
  if ((*zeilen > 1 || sep != '|') && indent) { akt_striche++; }
  return erg;
}

char ausbuf[CRTCOLS + 1];       /* Ausgabepuffer fuer 'deb_vec_print()' */

/**************************************************************************
 ***           Funktion outbuf
 ***
 *** gibt '\n' und 'ausbuf' auf 'f' aus
 *** Ergebnis : Zahl der ausgegebenen Zeilen (0 oder 1)
 ***
 **************************************************************************/

int outbuf(f)
FILE *f;
{ if (!quitted && ausbuf[0])
  { putc('\n', f); ifmore(f);
    if (!quitted) fputs(ausbuf, f);
    return 1;
  }
  return 0;
}

/**************************************************************************
 ***                  Funktion deb_vec_print (rekursiv)
 ***
 *** gibt DEBUG-Information fuer einen Vektoriellen Variablenbereich auf 'f' aus
 *** itpts    : Feld aus Zeigern auf Variablenbereiche die nebeneinander
 ***            ausgegeben werden (Zeiger laufen mit)
 *** exist    : Feld aus Flags: Variable der Spalte existiert
 ***            (noetig bei indirekten Adressen: Zeiger existiert, Variable nicht)
 *** cols     : Zahl der Spalten
 *** breite   : Breite jeder Spalte ohne Striche
 *** dptr     : Deklaration des Variablenbereichs
 *** indent   : Rekursionstiefe fuer Einruecken
 *** Ergebnis : Zahl der ausgegebenen Zeilen
 ***
 **************************************************************************/

int deb_vec_print(itpts, exist, cols, breite, dlptr, indent, f, sep)
ITEM *itpts[];
int exist[];
int cols, breite;
DECLIST *dlptr;
int indent;
FILE *f;
char sep;
{ register int i, j, count, erg = 0;
  register DECL *dptr;
  register char *sp, *sneu;
  register int colbr = breite + 2 * strichzahl + 2;   /* Breite mit Strichen */
  register int maxcol = CRTCOLS - 3 - 2 * strichzahl; /* max. Bildschirmspalte */

  for (count = dlptr->dcount, dptr = dlptr->darray; count; dptr++, count--)    /* Schleife ueber Deklaration */
  { switch(DECL_art(*dptr))
    { case FELD  :                /* echte Deklaration */
      { register TYP t = DECL_t(*dptr);
        register int z = DECL_zahl(*dptr);

        if (t == CHA && z > 1)            /* Stringausgabe */
        { erg += outbuf(f);
          if (quitted) return erg;
          sprintf(ausbuf,"%*s", cols * colbr, " "); /* Puffer loeschen */
          for (j = 0; j < cols; j++)      /* Schleife ueber Spalten */
          { sp = ausbuf + j * colbr + 2 * indent;
            if (exist[j])
            { *sp++ = '"';
              for (i = z; i && itpts[j]; i--, itpts[j] += pe_anz)
                                          /* Schleife ueber String */
              { register int kflag;
  
                if (kflag = (ITEM_typ(*itpts[j]) != t ||
                             ((itpts[j])->datentyp & UNDEF_ERG) ||
                             (itpts[j])->inhalt.c_val == EOL_CHAR ||
                             (itpts[j])->inhalt.c_val == TERMS_CHAR ||
                             (itpts[j])->inhalt.c_val < MIN_PRINTING_CHAR ||
                             (itpts[j])->inhalt.c_val > MAX_PRINTING_CHAR))
                          /* Spezialausgabe */
                { beim_listing = TRUE;
                  sneu = wert_string(itpts[j], t);
                  if (sp + 2 + strlen(sneu) > ausbuf + maxcol) /* String zu lang */
                  { outbuf(f);
                    if (quitted) return erg;
                    sprintf(ausbuf,"%*s", cols * colbr, " ");
                    sp = ausbuf + 2 * indent + 1;
                  }
                  *sp++ = '<';
                }
                else
                          /* Normales Zeichen */
                { sneu = wert_string(itpts[j], t);
                  if (sp + strlen(sneu) > ausbuf + maxcol) /* String zu lang */
                  { outbuf(f);
                    if (quitted) return erg;
                    sprintf(ausbuf,"%*s", cols * colbr, " ");
                    sp = ausbuf + 2 * indent + 1;
                  }
                }
                strcpy(sp, sneu); sp += strlen(sp);
                free(sneu);
                if (kflag)
                { beim_listing = FALSE;
                  *sp++ = '>';
                }
              }
              *sp = '"';
            }
            else
            { strncpy(sp,"<non-ex>",(size_t)5); }
          }
        }
        else              /* Ausbabe als einzelne Variablen */
        { for (i = z; i; i--) /* Schleife ueber Variablen */
          { erg += outbuf(f);
            if (quitted) return erg;
            sprintf(ausbuf,"%*s", cols * colbr, " "); /* Puffer loeschen */
            for (j = 0; j < cols && itpts[j]; itpts[j++] += pe_anz)
                          /* Schleife ueber Spalten */
            { sp = ausbuf + j * colbr + 2 * indent;
              if (exist[j])
              { register size_t l;
              
                beim_listing = TRUE;
                sneu = wert_string(itpts[j], t);
                l = strlen(sneu);
                strncpy(sp, sneu, (size_t)l);
                free(sneu);
                beim_listing = FALSE;
              }
              else
              { strncpy(sp,"<non-ex>",(size_t)5); }
            }
          }
        }
      }
      break;
      case KLAM  :               /* Unterdeklaration */
      { register int w = DECL_wiederh(*dptr);
        register DECLIST *di = &DECL_klamdecl(*dptr);

        for (i = w;
             i;
             i--, erg += deb_vec_print(itpts, exist, cols, breite,
                                       di, indent + 1, f, '|'));
      }
      break;
      case UNION :
      { register int count = DECL_ul(*dptr).ucount;
        register DECLIST *di;
        ITEM *lok_itpts[CRTCOLS / 7];

        for (i = count, di = DECL_ul(*dptr).uarray; i; i--)
        { for (j = 0; j < cols; j++)
          { lok_itpts[j] = itpts[j]; }
          erg += deb_vec_print(lok_itpts, exist, cols, breite,
                               di++, indent + 1, f, i == 1 ? ';' : ',');
        }
        for (j = 0; j < cols; j++)
        { itpts[j] += pe_anz * DECL_ul(*dptr).utyp_anz[SUMM]; }
      }
    }
  }
  if ((erg > 1 || sep != '|') && indent)        /* Strich in Puffer schreiben */
  { for (j = 0; j < cols; j++)  /* in alle Spalten */
    { sp = ausbuf + j * colbr + breite + 1;
      while (*sp != ' ') sp += 2;       /* hinter bisher letzten Strich */
      *sp = sep;
    }
  }
  return erg;
}

/**************************************************************************
 ***                       Funktion deb_out
 ***
 *** gibt DEBUG-Information fuer einen Variablenbereich auf 'f' aus
 *** *varits : Beschreibung der ersten Variable
 *** *varbls : Beschreibung der Bloecke aus der die Variablen sind
 *** str     : Ueberschrift fuer die Ausgabe (kann NULL sein)
 *** *decl   : Deklaration des Variablenbereichs
 *** set_tr  : Flag: sollen TRACE-Zaehler im Speicher hochgezaehlt werden?
 *** Fehlermeldung bei Konflikt zwischen Deklaration und Speicher
 ***
 **************************************************************************/

deb_out(varits, varbls, str, decl, set_tr, f)
ITPTRS *varits;
VBLOCKS *varbls;
char *str;
DECLIST *decl;
int set_tr;
FILE *f;
{ register VARBLOCK *bl;        /* Zeiger auf Block der Variable */
  ITEM *prit;                   /* Zeiger auf Variable fuer Druckfunktionen */
  ITEM *itpts[CRTCOLS / 7];     /* Variablenzeiger fuer mehrspaltige Ausgabe */
  int existiert[CRTCOLS / 7];   /* Flags fuer mehrspaltige Ausgabe */

  varits->ip_count = 0;         /* Variablenbeschreibung auf erste PE setzten */
  if (deb_mod && str)
  { fprintf(f, "%s :\n", str); ifmore(f); }
  if (varits->n_it_func == gl_item)     /* Skalarvariable */
  { bl = varbls->blocks[0];
    prit = varits->itps[0];
    if ((prit - bl->vars) + bl->von + decl->typ_zahl[SUMM] - 1 > bl->bis)
    { runerr(texte[139]); return; }
    deb_print(&prit, decl, 0, set_tr, f, '|');
  }
  else if (deb_mod || set_tr)   /* Vektorvariable : ausgeben oder Trace setzen */
  { register int breite = 0;    /* Ziel fuer maximale Spaltenbreite */
    register int br;            /* Spaltenbreite */
    register int ganze_breite;  /* Spaltenbreite incl. Striche */
    register int pelen = !pe_anz ? 1 : (int)log10((double)pe_anz) + 1;
                                /* Stellenzahl der PE-Nummer */
    register int cols;          /* Spaltenzahl */
    int dummy;                  /* Platz fuer unbeachteten Rueckgabewert */
    register int last_pe = last(default_pelist,&dummy); /* letzte betrachtete PE */

    if (last_akt_pe < last_pe) last_pe = last_akt_pe;
    vec_bef = TRUE;
    for (pec = 0; pec < pe_anz; pec++)  /* Breite ermitteln */
    { prit = (* varits->n_it_func)(varits);
      if (aktive_pes[pec] == '1' && default_pelist[pec] == '1') /* Ausgabe noetig */
      { bl = varbls->blocks[pec * varbls->inc];
        if ((prit - bl->vars) / pe_anz + bl->von + decl->typ_zahl[SUMM] - 1
            > bl->bis)
        { runerr(texte[139]); return; }
        strichzahl = akt_striche = 0;
        if ((br = deb_breite(&dummy, &prit, decl, 0, set_tr, '|')) < 0) return;
        if (br > breite) breite = br;
      }
      else if (set_tr)          /* nur TRACE-Zaehler hochzaehlen */
      { bl = varbls->blocks[pec * varbls->inc];
        if ((prit - bl->vars) / pe_anz + bl->von + decl->typ_zahl[SUMM] - 1
            > bl->bis)
        { runerr(texte[139]); return; }
        if (deb_breite(&dummy, &prit, decl, 0, set_tr, '|') < 0) return;
      }
    }
    if (breite + 2 * strichzahl < pelen + 4)
    { breite = pelen + 4 - 2 * strichzahl; }
    ganze_breite = breite + 2 * strichzahl + 2;
    cols = CRTCOLS / ganze_breite;
    if (!cols)          /* zu breit : Strings muessen aufgespalten werden */
    {  cols = 1;
       ganze_breite = CRTCOLS;
       breite = CRTCOLS - 2 - 2 * strichzahl;
    }

    varits->ip_count = 0;       /* Variablenbeschreibung auf erste PE setzten */
    if (deb_mod)                /* Ausgabe noetig */
    { register int ausgegeben = FALSE;  /* Kopfzeile wurde ausgegeben */

      for (pec = 0; pec < last_pe; )    /* Schleife ueber alle PEs */
      { register int max;

        if (!quitted && pec > 0)
        { putc('\n', f); ifmore(f); }
        if (quitted) return;
        for (max = 0; max < cols && pec < last_pe; pec++) /* Schleife ueber Spalten */
        { itpts[max] = (* varits->n_it_func)(varits); /* Variablenzeiger speichern */
          if (aktive_pes[pec] == '1' && default_pelist[pec] == '1')
          { if (!quitted)
            { fprintf(f, "Pe%0*d :%*s",
                      pelen, pec + 1, ganze_breite - pelen - 4, " ");
              ausgegeben = TRUE;
            }
            existiert[max] =  /* Existenzflags besetzen : max. aktive PEs im Block */
              ( !varbls->blocks[pec * varbls->inc]->max_pes || varbls->blocks[pec * varbls->inc]->max_pes[pec] == '1' );
            max++;
          }
        }
        if (ausgegeben)         /* keine Kopfzeile -> keine aktiven PEs gefunden */
        { sprintf(ausbuf, "\0%*s", CRTCOLS - 1, " ");
          deb_vec_print(itpts, existiert, max, breite, decl, 0, f, '|');
          if (deb_mod && !quitted) outbuf(f);
        }
      }
      if (deb_mod && !ausgegeben && !quitted)
      { putc('\n', f); ifmore(f);
        if (!quitted)
        { fprintf(f, texte[233]); }
      }
    }
  }
  no_linefeed = TRUE;
}

/**************************************************************************
 ***                       Funktion tr_off
 ***
 *** schaltet TRACE fuer alle Variablenbereiche aus, die mit der Variablen
 *** beginnen, die durch '*varits' und '*varbls' beschrieben ist.
 *** Warnung, wenn Variable nicht getraced war.
 ***
 **************************************************************************/

tr_off(varits, varbls)
ITPTRS *varits;
VBLOCKS *varbls;
{ register int i;
  register ITEM *trit;
  register TRACEVAR **trlistpt = &varbls->blocks[0]->t_info;
  register int max, anz = (varits->n_it_func == gl_item) ? 1 : pe_anz;
  register ITEM *suchit = varits->itps[0];
  register char *max_p = varbls->blocks[0]->max_pes;
  register int gefunden = FALSE;

  while (*trlistpt)     /* Schleife ueber TRACE-Informationen des Variablenblocks */
  { trit = (*trlistpt)->varpts.itps[0]; /* Variable in TRACE-Info */
    if (trit > suchit) break;   /* Suche beendet */
    if (trit == suchit)         /* TRACE-Info zu Variable gefunden */
    { register TRACEVAR *talt = *trlistpt;

      gefunden = TRUE;
      max = (*trlistpt)->tdec->typ_zahl[SUMM] * anz;
      for (i = 0; i < max; i++, trit++) /* Tracezaehler in Speicherstellen runterzaehlen */
      { if (!max_p || max_p[i % anz] == '1')
        { trit->datentyp -= TRACED;
          if (trit->datentyp < 0) bug("run.c/tr_off : nicht getracete Variable");
        }
      }
      *trlistpt = talt->t_naechst;      /* TRACE-Info aushaengen */
      free(talt);
    }
    else                        /* weitersuchen */
    { trlistpt = &(*trlistpt)->t_naechst; }
  }
  if (!gefunden)
  { runwarn(texte[59]); }
}

/**************************************************************************
 ***                      Funktion tr_block_off
 ***
 *** schaltet TRACE fuer alle Variablen der Variablenblockliste 'vbl' aus
 *** anz : Laenge der Variablen ('1' oder 'pe_anz')
 ***
 **************************************************************************/

tr_block_off(vbl, anz)
VARBLOCK *vbl;
int anz;
{ register int i, max;
  register ITEM *it;

  for (; vbl; vbl = vbl->nblock)        /* Schleife ueber alle Bloecke */
  { register TRACEVAR *tvneu, *tv = vbl->t_info;

    while (tv)          /* Schleife ueber TRACE-Infos des Blocks */
    { max = tv->tdec->typ_zahl[SUMM] * anz;
      for (it = tv->varpts.itps[0], i = 0; i < max; i++, it++) /* Tracezaehler loeschen */
      { it->datentyp &= (TRACED - 1); }
      tvneu = tv->t_naechst;            /* TRACE-Info aushaengen */
      free(tv);
      tv = tvneu;
    }
    vbl->t_info = NULL;
  }
}

int gelisted;           /* Programmzeile zu Trace-Ausgabe schon gelisted ? */

/**************************************************************************
 ***                       Funktion look
 ***
 *** schaut nach ob die Speicherstelle '*it' in der Traceliste von '*block'
 *** enthalten ist, gibt TRACE-Information aller Variablenbereiche auf 'f' aus,
 *** die '*it' enthalten.
 *** len : Laenge der Variablen ('1' oder 'pe_anz')
 ***
 **************************************************************************/

look (block, it, len)
VARBLOCK *block;
ITEM *it;
int len;
{ register ITEM *trit;                    /* Anfangsvar. der Tracebloecke */
  register TRACEVAR *tv = block->t_info;  /* laeuft durch Traceinformation */

  for ( ; tv; tv = tv->t_naechst)       /* ganze Traceinfo ansehen */
  { if ((trit = tv->varpts.itps[0]) > it) break; /* schon zu weit */
    if (trit + tv->tdec->typ_zahl[SUMM] * len > it) /* gefunden */
    { if (im_programm && !gelisted)     /* Programmzeile ausgeben */
      { if (no_linefeed) putc('\n', debugout); no_linefeed = FALSE;
        if (!quitted) list_stat(programm + pcalt, "", debugout);
        gelisted = TRUE;
      }
      if (deb_mod && no_linefeed)
      { putc('\n', debugout); ifmore(debugout); }
      fputs("[TRACE] ", debugout);
      deb_out(&tv->varpts, &tv->varbls, tv->t_string, tv->tdec, 0, debugout);
    }
  }
}

/**************************************************************************
 ***                       Funktion tr_out
 ***
 *** gibt TRACE-Information zum abgelaufenen Befehl auf 'f' aus 
 ***
 **************************************************************************/

tr_out(f)
FILE *f;
{ register int i;

  if (deb_mod)
  { gelisted = FALSE;
    if (zielits->n_it_func == n_item_ptr)       /* indirekte Vektoradresse */
    { zielits->ip_count = 0;
      for (i = 0; i < last_akt_pe; i++)
      { if (aktive_pes[i] == '1')
        { look(zielbs->blocks[i * zielbs->inc],
               zielits->itps[i],
               pe_anz,
               f);
        }
      }
    }
    else
    { look(zielbs->blocks[0],
           zielits->itps[0],
           (zielits->n_it_func == gl_item) ? 1 : pe_anz,
           f);
    }
    putc('\n', f);
    no_linefeed = FALSE;
    if (im_programm) putc('\n', f);
  }
}

/**************************************************************************
 ***                   Funktion saveinp
 ***
 *** speichert den Rest der Eingabezeile von 'stdin' in '*buf'
 ***
 **************************************************************************/

saveinp(buf)
char *buf;
{ if (zeile_fertig)
  { buf[0] = '\0'; }
  else
  { fgets(buf, 257, stdin); }
}

/**************************************************************************
 ***                   Funktion restoreinp
 ***
 *** setzt 'stdin' wieder auf den Zustand vor 'saveinp(buf)'
 ***
 **************************************************************************/

restoreinp(buf)
char *buf;
{ if (*buf)
  { register int i;

    for (i = strlen(buf) - 1; i >= 0; i--)
    { ungetc(buf[i], stdin); }
  }
}

/**************************************************************************
 ***                   Funktion enter_scopes
 ***
 *** betritt die 'sccount' Geltungsbereiche in 'sclist'
 ***
 **************************************************************************/

enter_scopes(sclist, sccount)
SCOPETAB **sclist;
int sccount;
{ register int count;

#ifdef SCDEBUG
int xxxx = 0;
#endif
  for (count = 0, sclist = sclist + sccount - 1;
       count < sccount && *sclist != newest_scope;
       count++, sclist --);
  sclist++;
  for (; count; sclist++, count--)
  { if ((*sclist)->par_flag)
    { register SCOPETAB **scp;
      register int lev = (*sclist)->sc_lev;

      for (scp = &akt_scope;
           !((*scp)->proc_key && (*scp)->sc_lev == lev);
           scp = &(*scp)->sym_parent);
      (*sclist)->sym_parent = *scp;
      if (disp[lev]->top_scope == *scp) disp[lev]->top_scope = *sclist;
      *scp = *sclist;
      akt_scope = disp[akt_tiefe]->top_scope;
    }
    else
    { (*sclist)->sym_parent = akt_scope;
      disp[akt_tiefe]->top_scope = akt_scope = *sclist;
    }
    disp[akt_tiefe]->new_scope = newest_scope = *sclist;
    disp[akt_tiefe]->entered_scopes++;
#ifdef SCDEBUG
printf("-> %s: GB %d  ", newest_scope->par_flag ? "PAR" : "WITH", newest_scope - scope_table + 1);
xxxx++;
#endif
  }
#ifdef SCDEBUG
if (xxxx) putchar('\n');
#endif
}

/**************************************************************************
 ***                   Funktion exit_scope
 ***
 *** verlaesst den neuesten Geltungsbereich
 ***
 **************************************************************************/

exit_scope()
{
#ifdef SCDEBUG
printf("<- GB %d\n..", newest_scope - scope_table + 1);
#endif
 if (newest_scope->par_flag)
  { register SCOPETAB **scp;
    register int lev = newest_scope->sc_lev;

    for (scp = &akt_scope; *scp != newest_scope; scp = &(*scp)->sym_parent);
    *scp = newest_scope->sym_parent;
    if (disp[lev]->top_scope == newest_scope) disp[lev]->top_scope = *scp;
  }
  else
  { disp[akt_tiefe]->top_scope = akt_scope = akt_scope->sym_parent; }
  disp[akt_tiefe]->new_scope = newest_scope = newest_scope->parent;
  disp[akt_tiefe]->entered_scopes--;
}

/**************************************************************************
 ***                      Funktion start
 ***
 *** startet das geladene Programm bei der Befehlsnummer 'adr'
 *** Ergebnis : Exit-Code von REDUCE bzw. Kommandoparser
 ***
 **************************************************************************/

int start(adr)
int adr;
{ char inpbuf[257];     /* Puffer fuer 'saveinp'/'restoreinp' */
  register int ex;      /* Exit-Code */
  register char komm;   /* Flag: Kommandointerpreter aufrufen */
  register STAT *bef;   /* aktueller Befehl */

  pc = adr;
  bef = programm + pc;
  call_count = 1;
  quitted = 0;
  do
  {            /* beginnende Geltungsbereiche betreten */
    { register SCOPETAB **scl = STAT_scopes(*bef);

      if (scl)
      { if(!(*scl)->par_flag && (*scl)->proc_key)
        { if (!(*scl)->sc_lev)          /* Hauptprogramm */
          { (*scl)->sym_parent = &std_scope;
            newest_scope = disp[akt_tiefe]->new_scope =
            akt_scope = disp[akt_tiefe]->top_scope = *scl;
            disp[akt_tiefe]->entered_scopes++;
#ifdef SCDEBUG
printf("-> HP: GB %d  ", newest_scope - scope_table + 1);
#endif
            enter_scopes(scl + 1, STAT_scopes_count(*bef) - 1);
          }
        }
        else
        { enter_scopes(scl, STAT_scopes_count(*bef)); }
      }
    }
                /* beginnende Quellzeilen betreten */
    if (STAT_src_ln(*bef))
    { register int i;
      register ZEILTAB **zt;

      for (i = STAT_src_ln_count(*bef), zt = STAT_src_ln(*bef);
           i;
           i--, zt++)
      { if (akt_line != *zt)
        { (*zt)->parent_line = akt_line;
          akt_line = *zt;
        }
      }
    }
    if (ended) break;
    if (!sym_debugging)
    { komm = FALSE; more_count = 0;
      if (STAT_print_func(*bef) != print_proc)
      { more_mod = FALSE;
        if (STAT_bpoint(*bef) & USER_BREAK)       /* B-Breakpoint */
        { fflush(stdout); fflush(stderr);
          fprintf(yyerfp, texte[175]);
          more_count++;
          komm = TRUE;
        }
        if (STAT_bpoint(*bef) & TO_BREAK)         /* TO-Breakpoint */
        { fflush(stdout); fflush(stderr);
          fprintf(yyerfp, texte[176]);
          more_count++;
          to_unbreak();
          komm = TRUE;
        }
        if (quitted)                              /* Programmunterbrechung */
        { fflush(stdout); fflush(stderr);
          fprintf(yyerfp, texte[177]);
          more_count++;
          komm = TRUE; forced_break = TRUE;
        }
        if ( (ss_mod & STEP_INTER) ||
             ( (ss_mod & STEP_COMP) && ( STAT_src_ln(*bef)
                                         || ( STAT_bpoint(*bef) & COMP_BREAK ) ) ) )
                                                  /* Singlestep */
        { if (!(ss_mod & STEP_OVER) || step_over_block == akt_step_block)
          { fflush(stdout); fflush(stderr);
            fprintf(yyerfp, texte[178]);
            more_count++;
            if (ss_fast && !komm)         /* nicht anhalten */
            { putc('\n', yyerfp);
              list_stat(bef, "", yyerfp);
              if (akt_line)
              { register int i, j;

                fprintf(yyerfp, texte[234], i = akt_line->source_line);
                if (j = akt_line->stat_in_line) fprintf(yyerfp, texte[298], j);
                putc('\n', yyerfp);
                source_out(i, yyerfp);
              }
              show_pes(aktive_pes, texte[179], yyerfp); putc('\n', yyerfp);
              no_linefeed = FALSE;
            }
            else
            { komm = TRUE; }
          }
        }
        if (komm)                         /* anhalten */
        { quitted = 0;
          saveinp(inpbuf);
          if (more_mod = more_mod_save) more_on();
          putc('\n', yyerfp); ifmore(yyerfp);
          if (!quitted)
          { list_stat(bef, "", yyerfp);
            if (!quitted)
            { if (akt_line)
              { register int i,j;

                fprintf(yyerfp, texte[234], i = akt_line->source_line);
                if (j = akt_line->stat_in_line) fprintf(yyerfp, texte[298], j);
                putc('\n', yyerfp); ifmore(yyerfp);
                if (!quitted)
                { source_out(i, yyerfp); }
              }
              if (!quitted)
              { show_pes(aktive_pes, texte[179], yyerfp); }
            }
          }
          putc('\n', yyerfp); no_linefeed = FALSE; quitted = 0;
          if (more_mod) more_off();
          breaked = TRUE;
          switch (ex = kommandoparse())   /* Kommandointerpreter */
          { case DO_STOP    : 
            case DO_LOAD    : 
            case DO_COMPILE : ended = TRUE;
                              break;
            case DO_GO      : 
            case DO_STEP    : bef = programm + pc;
                              break;
          }
          restoreinp(inpbuf);
          forced_break = FALSE; quitted = 0;
        }
        if (ended) break;
      }
    }
                        /* Schritt ausfuehren */
    more_mod = more_mod_save;
    im_programm = TRUE;
    tr_flag = FALSE;            /* wird evtl. von Befehl gesetzt */
    bicount = -1;
    if (!sym_debugging) more_count = 0;
    pcalt = pc;
    vec_bef = FALSE;
    exit_code = 0;
    anz_runwarnings = 0;
    (* STAT_do_func(*bef))(bef);        /* Befehl */
    if (fperror)                /* Fliesskommafehler : Ergebnis auf HUGE setzen */
    { register ITEM *it;

      fperror = FALSE;
      runerr(texte[111]);
      zielits->ip_count--;
      it = (* zielits->n_it_func)(zielits);
      if (it->datentyp & REAL_ERLAUBT)
      { it->inhalt.r_val = HUGE;
        it->datentyp = CH_TYP(it->datentyp, REAL);
        tr_flag = (tr_flag || it->datentyp >= TRACED);
      }
    }
    if (tr_flag && deb_mod)     /* Trace-Ausgabe noetig */
    { more_mod = FALSE;
      tr_out(debugout);
      more_mod = more_mod_save;
    }
#ifdef MAC
    if (!( ausgefuehrt % 32)) ctrl_c_test();
#endif
    im_programm = FALSE;
    pc++; ex = exit_code;  
    bef = programm + pc;
    if (!sym_debugging)
                /* beendete Quellzeilen verlassen */
    { register int pcnext = pcalt + 1;
      register STAT *befnext = programm + pcnext;

      if (STAT_src_next(*befnext))
      { while (akt_line && akt_line->prog_next == pcnext &&
               (pc <= akt_line->prog_von || pc >= akt_line->prog_next))
        { akt_line = akt_line->parent_line; }
      }
      if (STAT_src_next(*bef))
      { while (akt_line && akt_line->prog_next == pc)
        { akt_line = akt_line->parent_line; }
      }
                /* beendete Geltungsbereiche verlassen */
#ifdef SCDEBUG
{int xxxx = 0;
#endif
      if (STAT_print_func(*bef) != print_proc)
      { while (newest_scope && disp[akt_tiefe]->entered_scopes &&
               (pc < newest_scope->von_lab || pc >= newest_scope->next_lab))
        { exit_scope();
#ifdef SCDEBUG
xxxx = 1;
#endif
        }
      }
#ifdef SCDEBUG
if (xxxx) putchar('\n');
}
#endif
    }
  } while (!(err || ended) && call_count > 0);
  if (!sym_debugging && !reducing) ueberlies();
  breaked = FALSE;
  return ex;
}

/**************************************************************************
 ***                 Funktion init_vars (rekursiv)
 ***
 *** traegt Variablentypen entsprechend der Deklaration '*dptr' in den
 *** Variablenblock ein, auf den 'vptr' zeigt.
 *** Alle Variablen werden uninitialisiert.
 *** anz      : Laenge einer Variable ('1'  oder 'pe_anz')
 *** Ergebnis : Zeiger auf erste Variable hinter Block.
 ***
 **************************************************************************/

ITEM *init_vars(dptr, vptr, anz)
DECLIST *dptr;
ITEM *vptr;
int anz;
{ register int i, count;
  register DECL *dp;

  for (count = dptr->dcount, dp = dptr->darray; count; dp++, count--)
  { switch (DECL_art(*dp))
    { case FELD :
      { register int n;
        register TYP t = DECL_t(*dp);
        register int z = DECL_zahl(*dp);

        for (i = 0; i < z; i++)
        { for (n = 0; n < anz; n++, (vptr++)->datentyp |= KEINER | ERLAUBT(t)); }
      }
      break;
      case KLAM :
      { register int w = DECL_wiederh(*dp);
        register DECLIST *di = &DECL_klamdecl(*dp);

        for (i = 0; i < w; i++, vptr = init_vars(di, vptr, anz));
      }
      break;
      case UNION :
      { register int count = DECL_ul(*dp).ucount;
        register DECLIST *uptr = DECL_ul(*dp).uarray;
        register ITEM *endi, *endmax = NULL;

        for(; count; count--, uptr++)
        { endi = init_vars(uptr, vptr, anz);
          if (!endmax || endi > endmax) endmax = endi;
        }
        vptr = endmax;
      }
    }
  }
  return vptr;
}

/**************************************************************************
 ***                       Funktion make_vars
 ***
 *** erzeugt einen Variablenblock entsprechend '*dec'
 *** anz      : Laenge einer Variablen ('1' oder 'pe_anz')
 *** *grenze  : Adressgrenze nach oben/unten (abhaengeig von 'richtung')
 *** richtung : Vorzeichen und Zaehlrichtung der Adressen (-1 : Heap, 1 : Stack)
 *** *alist   : Allozierungsliste in die neuer Block eingehaengt wird
 *** max_p    : Aktivierungsstring: PEs fuer die die Variablen existieren
 ***
 **************************************************************************/

VARBLOCK *make_vars(dec, anz, grenze, richtung, alist, max_p)
DECLIST *dec;
int anz;
int *grenze;
int richtung;
VARBLOCK **alist;
char *max_p;
{ register VARBLOCK *block;
  register int sum = dec->typ_zahl[SUMM];

  if (sum)
  {             /* neuer VARBLOCK */
    if (!(block = (VARBLOCK *)calloc((size_t)1, (size_t)sizeof(VARBLOCK))))
    { runerr(texte[5]); return NULL; }
                /* neue Speicherstellen */
    if (!(block->vars = (ITEM *)calloc((size_t)(anz ? sum * anz : 1), (size_t)sizeof(ITEM))))
    { runerr(texte[5]); return NULL; }
                /* Typen eintragen */
    init_vars(dec, block->vars, anz);
    switch (richtung)           /* grenze eintragen */
    { case  1 : block->von = *grenze + 1;
                block->bis = (*grenze += sum);
                break;
      case -1 : block->bis = *grenze - 1;
                block->von = (*grenze -= sum);
                break;
      default : bug("run.c/make_vars : richtung");
    }
    block->max_pes = max_p;
    block->nblock = *alist;
    *alist = block;
    return block;
  }
  else return NULL;
}

/**************************************************************************
 ***                    Funktion release_vars
 ***
 *** gibt den Variablenblock 'block' frei, wenn 'block' existiert,
 *** ist er der erste Variablenblock in der Verketteten Liste '*alist'
 *** und wird dort ausgehaengt.
 *** Neue Maximaladresse in '*alist' kommt nach '*grenze'.
 ***
 **************************************************************************/

release_vars(block, alist, grenze)
VARBLOCK *block;
VARBLOCK **alist;
int *grenze;
{ if (block)
  { free(block->vars);
    *grenze = block->von - 1;
    *alist = block->nblock;
    loesch_trace(block->t_info);
    free(block);
  }
}

/**************************************************************************
 ***                         Funktion new_aktblock
 ***
 *** erzeugt neuen Aktivierungsblock, traegt ein, was bei CALL bekannt ist
 *** rueckspr       : Adresse des Befehls nach CALL
 *** parpush        : zu pushende Parallel-Information bei IF...CALL, WHILE...CALL
 *** new_step_block : Flag: echter CALL  aus Sicht von Step
 *** Ergebnis       : Zeiger auf neuen Aktivierungsblock
 ***
 **************************************************************************/

AKTBLOCK *new_aktblock(rueckspr, parpush, new_step_block)
int rueckspr;
char *parpush;
int new_step_block;
{ register AKTBLOCK *new_block;

  if (!(new_block = (AKTBLOCK *)calloc((size_t)1, (size_t)sizeof(AKTBLOCK))))
  { runerr(texte[5]); return NULL;}
  new_block->rueck = rueckspr;
  new_block->para_call_zeile = akt_line;
  new_block->aufr_tiefe = akt_tiefe;
  new_block->vorh_aktiv = parpush;
  new_block->alt_step_block = akt_step_block;
  if (new_step_block)
  { akt_step_block = new_block;
    akt_line = NULL;
  }
  return new_block;
}

/**************************************************************************
 ***                       Funktion fill_aktblock
 ***
 *** traegt in den Aktivierungsblock '*block' Informationen ein,
 *** die bei PROC bekannt sind und traegt ihn im  Display ein.
 *** tiefe : Schachtelungstiefe der Prozedur
 *** procnum : Befehlsnummer des PROC-Befehls
 *** *s_decl : Lokale SCALAR-Deklaration
 *** *v_decl : Lokale Vector-Deklaration
 ***
 **************************************************************************/

fill_aktblock(block, tiefe, procnum, s_decl, v_decl)
AKTBLOCK *block;
int tiefe;
int procnum;
DECLIST *s_decl, *v_decl;
{ STAT *procbef = (procnum >= 0 ? programm + procnum : NULL);
  SCOPETAB **pscopes, *scptr;

  block->dispalt = disp[tiefe];
  block->procadr = procnum;
  if (!(block->start_aktiv = calloc((size_t)(pe_anz + 1), (size_t)sizeof(char))))
  { runerr(texte[121]); return; }
  strcpy(block->start_aktiv, aktive_pes);
  block->s_vars =
     make_vars(s_decl, 1, &s_max_b_adr, 1, &s_blocks, NULL);
  block->v_vars =
     make_vars(v_decl, pe_anz, &v_max_b_adr, 1, &v_blocks, block->start_aktiv);
  disp[tiefe] = block;
  akt_tiefe = tiefe;
  if (tiefe > block->aufr_tiefe + 1)
  { runerr(texte[96], tiefe, block->aufr_tiefe); return; }
                /* beginnende Geltungsbereiche betreten */
  if (procbef)
  { if (pscopes = STAT_scopes(*procbef))
    { newest_scope = block->new_scope = akt_scope = block->top_scope = *pscopes;
      block->entered_scopes = 1;
      for (scptr = disp[tiefe - 1]->top_scope;
           scptr && !scptr->par_flag && !scptr->proc_key;
           scptr = scptr->sym_parent);
      akt_scope->sym_parent = scptr;
#ifdef SCDEBUG
printf("-> UP: GB %d", newest_scope - scope_table + 1);
#endif
      enter_scopes(pscopes + 1, STAT_scopes_count(*procbef) - 1);
    }
    else
    { block->new_scope = newest_scope; block->top_scope = akt_scope;
      block->entered_scopes = 0;
    }
  }
}

/**************************************************************************
 ***                    Funktion akt_return
 ***
 *** setzt bei RETURN Aktivierungsstack auf Zustand vor CALL
 *** *parptr  : Rueckgabe des gepushten Aktivierungsstrings
 *** *vecflag : Rueckgabeflag : Lokale Vektorvariablen vorhanden?
 *** Ergebnis : Ruecksprungadresse
 ***
 **************************************************************************/

int akt_return(parptr, vecflag)
char **parptr;
int *vecflag;
{ DISPLAY d = disp[akt_tiefe];
  int rsp = d->rueck;           /* Ruecksprungadresse */

                /* beendete Geltungsbereiche verlassen */
#ifdef SCDEBUG
if (d->entered_scopes) printf("<- UPGB : %d\n", newest_scope - scope_table + 1);
#endif
  if (step_over_block == akt_step_block) /* Step-Over-Ebene verlassen : mitnehmen */
  { step_over_block = d->alt_step_block; }
  disp[akt_tiefe] = d->dispalt; /* Display zuruecksetzen */
  akt_step_block = d->alt_step_block;
                /* Variablen freigeben */
  release_vars(d->s_vars, &s_blocks, &s_max_b_adr);
  *vecflag = (int)d->v_vars;
  release_vars(d->v_vars, &v_blocks, &v_max_b_adr);
  *parptr = d->vorh_aktiv;
  akt_tiefe = d->aufr_tiefe;
  akt_line = d->para_call_zeile;
                /* Aktivierungsblock freigeben */
  free(d->start_aktiv);
  free(d);
  if (akt_tiefe >= 0)
  { newest_scope = (d = disp[akt_tiefe])->new_scope;
    akt_scope = d->top_scope;
  }
  else
  { newest_scope = akt_scope = NULL; }
  return rsp;
}

/**************************************************************************
 ***                       Funktion ini_typ_test
 ***
 *** testet ob die Typkomponente in 'dt' initialisiert und gleich 'solltyp' ist.
 *** Bei Uninitialisierter Speicherstelle: Warnung 'texte[ini_nr]'
 *** Bei Falschem Datentyp, d.h. falscher Variante: Warnung 'texte[typ_nr]'
 *** Ergebnis : Datentyp mit evtl gesetztem Flag 'UNDEF_ERG'
 ***
 **************************************************************************/
     
int ini_typ_test(dt, solltyp, ini_nr, typ_nr)
TYP dt, solltyp;
int ini_nr, typ_nr;
{ register TYP t;
  if ((t = dt & ALLTYPS) == KEINER)
  { runwarn(texte[ini_nr]);
    return dt | UNDEF_ERG;
  }
  else if (t != solltyp)
  { runwarn(texte[typ_nr]);
    return dt | UNDEF_ERG;
  }
  else return dt;
}

/**************************************************************************
 ***                       Funktion gl_item
 ***
 *** ermittelt "Variable fuer naechste PE" in '*ip' bei Skalarvariablen
 ***
 **************************************************************************/

ITEM *gl_item(ip)
ITPTRS *ip;
{ return ip->itps[0];
}

/**************************************************************************
 ***                      Funktion n_item
 ***
 *** ermittelt "Variable fuer naechste PE" in '*ip'
 *** bei Vektorvariablen in einem Block ('ip->itps' hat Laenge 1)
 ***
 **************************************************************************/

ITEM *n_item(ip)
ITPTRS *ip;
{ return ip->itps[0] + ip->ip_count++;
}

/**************************************************************************
 ***                       Funktion n_item_ptr
 ***
 *** ermittelt "Variable fuer naechste PE" in '*ip'
 *** bei Vektorvariablen aus mehreren Bloecken ('ip->itps' hat Laenge 'pe_anz')
 ***
 **************************************************************************/

ITEM *n_item_ptr(ip)
ITPTRS *ip;
{ register ITEM *it = ip->itps[ip->ip_count] + ip->ip_count;

  ip->ip_count++;
  return it;
}

/**************************************************************************
 ***                       Funktion get_block
 ***
 *** Sucht in der verketteten Variablenblockliste 'list' nach dem Block,
 *** der die Variable mit Absolutadresse 'index' enthaelt.
 *** Liefert NULL, wenn kein Block gefunden, sonst Zeiger auf Block.
 ***
 **************************************************************************/

VARBLOCK *get_block(index, list)
long index;
VARBLOCK *list;
{ switch((int)lsign(index))
  { case  0 : return NULL;  /* falsche Adresse */
    case  1 :               /* Stackadresse : Adressen nehmen in Verkettungsrichtung ab */
      if (!list || list->bis < index) return NULL;
      for( ; list && (list->von > index); list = list->nblock);
      return list;
    case -1 :               /* Heapadresse : Adressen nehmen in Verkettungsrichtung zu */
      if (!list || list->von > index) return NULL;
      for( ; list && (list->bis < index); list = list->nblock);
      return list;
  }
}

/**************************************************************************
 ***                       Funktion item_ptr
 ***
 *** ermittelt die Variablenbeschreibung zum Argument '*argptr'
 *** *blptr   : Rueckgabe: Bloecke fuer einzelne PEs (noetig fuer TRACE)
 *** disp     : gueltiges Display-Feld
 *** Ergebnis : Zeiger auf Variablenbeschreibung fuer alle PEs
 *** Fehlermeldung bei unzulaessigem Argument
 *** Ergebnisse werden in den von 'bicount' selektierten Komponenten von
 *** 'oneitem' bzw. 'moreitems' und 'oneblock' bzw. 'moreblocks' abgelegt.
 *** Ergebnis und '*blptr' zeigen dann in diese Variablen.
 ***
 **************************************************************************/

ITPTRS *item_ptr(argptr, blptr, disp)
ARG *argptr;
VBLOCKS **blptr;
DISPLAY *disp;
{ register ARGART asort = ARG_argsort(*argptr); /* Typangabe des Arguments */
  register int isvec = asort & VEC;     /* Flag: Vektorargument */
  register int v_len = isvec ? pe_anz : 1;      /* Variablenlaenge */
  register int pad;     /* Befehlsnummer von PROC-Befehl zu 'ab' */
  register AKTBLOCK *ab; /* Aktivierungsblock, der Variable bzw. Indexvariable enthaelt */
  register VARBLOCK *bl; /* Variablenblock, der Variable bzw. Variablenteil enthaelt */
  register int isind = asort & (IND | INDL); /* Flag: Argument mit indirekter Adresse */
  register ARGART indsort = ARG_indsort(*argptr); /* Typ der indirekten Adresse */
  register int tief; /* Ebenennummer von Variable bzw. Indexvariable */

  bicount++;
  if (asort & CON)              /* Argument ist Konstante : oneblock, oneitem */
  { oneblock[bicount].blocks[0] = NULL; /* kein Variablenblock */
    *blptr = &oneblock[bicount];
    if (asort & SIZ)            /* SIZE-Konstante : erzeugen in 'hilf_item' */
    { hilf_item[bicount].datentyp = (INT | INT_ERLAUBT);
      hilf_item[bicount].inhalt.i_val = (long)ARG_sizedec(*argptr)->typ_zahl[SUMM];
      oneitem[bicount].itps[0] = &hilf_item[bicount];
    }
    else        /* Normale Konstante : steht im Programmtext */
    { oneitem[bicount].itps[0] = &ARG_con_wert(*argptr); }
    oneitem[bicount].n_it_func = gl_item;
    return &oneitem[bicount];
  }
  else if (asort & SPEZ)        /* Argument ist Spezialvariable : oneblock, oneitem */
  { register int tok;

    switch (tok = ARG_vartok(*argptr))
    { case MAXTRANS :
      case ACTTRANS :
      case DONE     :
      case TERMCH   :
      case SRESULT  : oneitem[bicount].itps[0] = &s_spez_vars[tok - MAXTRANS];
                      oneitem[bicount].n_it_func = gl_item;
                      oneblock[bicount].blocks[0] = &s_spez_block;
                      break;
      case ID       :
      case VRESULT  : oneitem[bicount].itps[0] = v_spez_vars + (tok - ID) * pe_anz;
                      oneitem[bicount].n_it_func = n_item;
                      oneblock[bicount].blocks[0] = &v_spez_block;
                      break;
      default       : bug("run.c/item_ptr : falsche Spezialvariable");
    }
    *blptr = &oneblock[bicount];
    oneitem[bicount].ip_count = 0;
    return &oneitem[bicount];
  }
  else if (!disp)       /* Fehler : kein Display vorhanden */
  { runerr(texte[183]); return oneitem; }
  else if ((tief = ARG_tiefe(*argptr)) > akt_tiefe) /* Fehler : falsche Ebene */
  { runerr(texte[184], tief, akt_tiefe); return oneitem; }

  ab = disp[tief];      /* Aktivierungsblock der Variablentiefe */
  if ((pad = ab->procadr) >= 0) /* 'ab' gehoert nicht zum Hauptprogramm :
                                   Absolutadresse ermitteln. Fehler, wenn unmoeglich */
  { if (!set_absoffs(isind ? AA_typ(indsort) : AA_typ(asort),
                     ARG_num(*argptr),
                     argptr,
                     (isind ? indsort & VEC : asort & (VEC | ADV))
                        ? STAT_dvec(programm[pad])
                        : STAT_dscal(programm[pad]),
                     pad))
    { runerr(texte[92]); return oneitem; }
  }
  if (asort & IND)      /* Argument ist global indirekt adressiert */
  { if (indsort & VEC)  /* Vektor-Index : moreblocks, moreitems */
    { register ITEM *indptr = ab->v_vars->vars +
                              v_len * (ARG_absoffs(*argptr) - 1); /* Zeiger auf Index-Var */
      register long index; /* Wert der Indexvariablen */
      register ITEM *it;   /* referenzierte Speicherstelle */

      vec_bef = TRUE;
      for (pec = 0; pec < last_akt_pe; pec++, indptr++)
      { if (aktive_pes[pec] == '1')
        { if (ITEM_typ(*indptr) != INT)       /* Fehler : Index unin. oder falsche Variante */
          { runerr(texte[90]); return oneitem; }
          index = indptr->inhalt.i_val;
          if (!(moreblocks[bicount]->blocks[pec] =
                bl = get_block(index, (index >= 0) ? v_blocks : v_heap)))
          { runerr(texte[53], index); return oneitem; } /* Fehler : kein Variablenblock */
          moreitems[bicount]->itps[pec] =
                it = bl->vars + (index - bl->von) * v_len;
          if (!(it->datentyp & ERLAUBT(AA_typ(asort))))    /* Fehler : Typkonflikt */
          { runerr(texte[162], index); return oneitem; }
        }
      }
      *blptr = moreblocks[bicount];
      moreitems[bicount]->ip_count = 0;
      vec_bef = FALSE;
      return moreitems[bicount];
    }
    else                /* Skalar-Index : oneblock, oneitem */
    { register ITEM *ip = ab->s_vars->vars + ARG_absoffs(*argptr) - 1; /* Indexvariable */
      register long index;      /* Indexwert */
      register ITEM *it;        /* referenzierte Speicherstelle */

      if (ITEM_typ(*ip) != INT)       /* Fehler : Index unin. oder falsche Variante*/
      { runerr(texte[90]); return oneitem; }
      index = ip->inhalt.i_val;
      if (!(oneblock[bicount].blocks[0] =
            bl = get_block(index, (index >= 0) ? (isvec ? v_blocks : s_blocks)
                                               : (isvec ? v_heap : s_heap))))
      { runerr(texte[53], index); return oneitem; } /* Fehler : kein Variablenblock */
      *blptr = &oneblock[bicount];
      oneitem[bicount].itps[0] = it = bl->vars + (index - bl->von) * v_len;
      if (!(it->datentyp & ERLAUBT(AA_typ(asort))))        /* Fehler : Typkonflikt */
      { runerr(texte[162], index); return oneitem; }
      oneitem[bicount].ip_count = 0;
      oneitem[bicount].n_it_func = isvec ? n_item : gl_item;
      return &oneitem[bicount];
    }
  }
  else if (asort & INDL)        /* Argument ist lokal indirekt adressiert : oneblock */
  { register AKTBLOCK *vb;      /* Aktivierungsblock der Variablen */
    register DECLIST *decl;     /* Deklaration des Variablenblocks */

    if ((tief = ARG_vartiefe(*argptr)) > akt_tiefe) /* Fehler : falsche Variablentiefe */
    { runerr(texte[184], tief, akt_tiefe); return oneitem; }
    vb = disp[tief];
    oneblock[bicount].blocks[0] = bl = isvec ? vb->v_vars : vb->s_vars;
    *blptr = &oneblock[bicount];
    if (indsort & VEC)          /* Vektorindex : moreitems */
    { register ITEM *indptr = ab->v_vars->vars +
                              v_len * (ARG_absoffs(*argptr) - 1); /* Indexvariable */
      register long index;      /* Indexwert */
      register int absind;      /* Offset der referenzierten Variablen */

      decl = (vb->procadr >= 0) ? STAT_dvec(programm[vb->procadr])
                                : &glob_v_decl;
      vec_bef = TRUE;
      for (pec = 0; pec < last_akt_pe; pec++, indptr++)
      { if (aktive_pes[pec] == '1')
        { if (ITEM_typ(*indptr) != INT)       /* Fehler : Index unin. oder falsche Variante */
          { runerr(texte[90]); return oneitem; }
          index = indptr->inhalt.i_val;
          if (!(absind = get_absoffs(AA_typ(asort),
                                     index,
                                     decl)))    /* Fehler : ref. Variable ex. nicht */
          { runerr(texte[53],index); return oneitem; }
          moreitems[bicount]->itps[pec] = bl->vars + v_len * (absind - 1);
        }
      }
      moreitems[bicount]->ip_count = 0;
      vec_bef = FALSE;
      return moreitems[bicount];
    }
    else                        /* Skalarindex : oneitem */
    { register ITEM *ip = ab->s_vars->vars + ARG_absoffs(*argptr) - 1; /* Indexvariable */
      register long index;      /* Indexwert */
      register int absind;      /* Offset der referenzierten Variablen */

      if (ITEM_typ(*ip) != INT)       /* Fehler : Index unin. oder falsche Variante */
      { runerr(texte[90]); return oneitem; }
      index = ip->inhalt.i_val;
      decl = (vb->procadr >= 0) ? (isvec ? STAT_dvec(programm[vb->procadr])
                                         : STAT_dscal(programm[vb->procadr]))
                                : (isvec ? &glob_v_decl : &glob_s_decl);
      if (!(absind = get_absoffs(AA_typ(asort),
                                 index,
                                 decl)))        /* Fehler : ref. Variable ex. nicht */
      { runerr(texte[53],index); return oneitem; }
      oneitem[bicount].itps[0] = bl->vars + v_len * (absind - 1);
      oneitem[bicount].n_it_func = isvec ? n_item : gl_item;
      oneitem[bicount].ip_count = 0;
      return &oneitem[bicount];
    }
  }
  else if (asort & (ADS | ADV)) /* Argument ist Adressoperator : oneblock, oneitem 
                                   Wert wird in 'hilf_item' bereitgestellt */
  { oneblock[bicount].blocks[0] = NULL; /* kein Variablenblock noetig */
    *blptr = &oneblock[bicount];
    hilf_item[bicount].datentyp = (INT | INT_ERLAUBT);
    hilf_item[bicount].inhalt.i_val = ((asort & ADV) ? ab->v_vars->von : ab->s_vars->von)
                                      - 1 + ARG_absoffs(*argptr);
    oneitem[bicount].itps[0] = &hilf_item[bicount];
    oneitem[bicount].n_it_func = gl_item;
    return &oneitem[bicount];
  }
  else                          /* Argument ist normale Variable : oneblock, oneitem */
  { oneblock[bicount].blocks[0] = bl = isvec ? ab->v_vars : ab->s_vars;
    *blptr = &oneblock[bicount];
    oneitem[bicount].itps[0] = bl->vars + v_len * (ARG_absoffs(*argptr) - 1);
    oneitem[bicount].n_it_func = isvec ? n_item : gl_item;
    oneitem[bicount].ip_count = 0;
    return &oneitem[bicount];
  }
}

/**************************************************************************
 ***                    Funktionen set_x_item 
 ***
 *** mit 'x' : Datentyp : b, c, i, r
 *** traegt 'wert' in '*ziel' als Inhalt ein, setzt '*ziel' gueltig.
 *** uflag : gibt an, ob 'wert' aus undefinierten Werten entstanden ist.
 ***
 **************************************************************************/

set_b_item(ziel, wert, uflag)
ITEM *ziel;
int wert, uflag;
{ ziel->inhalt.b_val = wert;
  ziel->datentyp = CH_TYP(ziel->datentyp, BOOL);
  ziel->datentyp = UNDEF_CPY(ziel->datentyp, uflag);
  tr_flag = (tr_flag || (ziel->datentyp >= TRACED));
}

set_c_item(ziel, wert, uflag)
ITEM *ziel;
unsigned char wert;
int uflag;
{ ziel->inhalt.c_val = wert;
  ziel->datentyp = CH_TYP(ziel->datentyp, CHA);
  ziel->datentyp = UNDEF_CPY(ziel->datentyp, uflag);
  tr_flag = (tr_flag || (ziel->datentyp >= TRACED));
}

set_i_item(ziel, wert, uflag)
ITEM *ziel;
long wert;
int uflag;
{ ziel->inhalt.i_val = wert;
  ziel->datentyp = CH_TYP(ziel->datentyp, INT);
  ziel->datentyp = UNDEF_CPY(ziel->datentyp, uflag);
  tr_flag = (tr_flag || (ziel->datentyp >= TRACED));
}

set_r_item(ziel, wert, uflag)
ITEM *ziel;
float wert;
int uflag;
{ if (fabs(ziel->inhalt.r_val = wert) >= HUGE && !err) fperror = TRUE;
  ziel->datentyp = CH_TYP(ziel->datentyp, REAL);
  ziel->datentyp = UNDEF_CPY(ziel->datentyp, uflag);
  tr_flag = (tr_flag || (ziel->datentyp >= TRACED));
}

/**************************************************************************
 ***                       Funktion set_item
 ***
 *** kopiert Inhalt von '*quell' nach '*ziel', setzt '*ziel' gueltig
 *** Datentyp von '*ziel' wird 't'.
 *** Warnung, wenn '*quell' uninitialisiert war.
 ***
 **************************************************************************/

set_item(ziel, quell, t)
ITEM *ziel, *quell;
TYP t;
{ register int uflag = ini_typ_test(quell->datentyp, t, 91, 283);

  ziel->inhalt = quell->inhalt;
  ziel->datentyp = CH_TYP(ziel->datentyp, t);
  ziel->datentyp = UNDEF_CPY(ziel->datentyp, uflag);
  tr_flag = (tr_flag || (ziel->datentyp >= TRACED));
}

/**************************************************************************
 ***                     Funktion set_item_egal
 ***
 *** kopiert Inhalt und aktuellen Typ (evtl KEINER) von '*quell' nach '*ziel',
 ***
 **************************************************************************/

set_item_egal(ziel, quell)
ITEM *ziel, *quell;
{ register int t;

  ziel->inhalt = quell->inhalt;
  ziel->datentyp = CH_TYP(ziel->datentyp, t = ITEM_typ(*quell));
  ziel->datentyp = UNDEF_CPY(ziel->datentyp,
                             quell->datentyp | (UNDEF_ERG * (t == KEINER)));
  tr_flag = (tr_flag || (ziel->datentyp >= TRACED));
}

/**************************************************************************
 ***                      Funktion protokoll
 ***
 *** protokolliert den aktiven Befehl mit Kennbuchstabe 'c' und
 *** Aktivierungsstring 'akt'
 ***
 **************************************************************************/

protokoll(c, akt)
char c, *akt;
{ register int i;
  register int anz;
  register int lokprot = prot;

  if (!sym_debugging)
  { ausgefuehrt++;
    if (prot_offen)
    { if (!lokprot && (STAT_bpoint(programm[pc]) & PROT_ON))
      { lokprot = alt_prot; }
      if (lokprot == 1)
      { fprintf(protokollout, reducing ? "%10ld : %cu " : "%10ld : %c ", ausgefuehrt, c);
        if (akt)
        { fprintf(protokollout, "%d", anz_akt_pes); }
        fputs(" ;\n", protokollout);
        prot_geschrieben = TRUE;
      }
      else if (lokprot == 2)
      { fprintf(protokollout, "%10ld : ", ausgefuehrt);
        list_stat(programm + pc, "             ", protokollout);
        if (akt)
        { fputs("             ", protokollout);
          if (pe_anz <= 100)
          { fputs(akt, protokollout); }
          else
          { for (i = 0; i < pe_anz; i++)
            { if (i && (i % 100 == 0)) fputs("\n             ", protokollout);
              putc(akt[i], protokollout);
            }
          }
          fputs(" ;\n", protokollout);
        }
        prot_geschrieben = TRUE;
      }
    }
  }
}
