#include <memory.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <math.h>

#include "SYSTEM_C.h"

/* Implementation of standard functions *************************************/

INTEGER ABSI
#ifdef __STDC__
(register INTEGER x)
#else
(x) register INTEGER x;
#endif
{
  return (x < 0 ? -x : x);
}

REAL ABSR
#ifdef __STDC__
(register REAL x)
#else
(x) register REAL x;
#endif
{
  return (x < 0.0 ? -x : x);
}

CHAR CAP
#ifdef __STDC__
(register CHAR ch)
#else
(ch) register CHAR ch;
#endif
{
   return (ch >= 'a' && ch <= 'z' ? ch - 'a' + 'A' : ch);
}

CARDINAL MOD1
#ifdef __STDC__
(INTEGER x, INTEGER y)
#else
(x, y)
INTEGER x, y;
#endif
{
  return ((x % y < 0) ? x % y + (y < 0 ? -y : y) : x % y );
}

/* Implementation of parallaxis standard functions **************************/

REAL ArcTan2
#ifdef __STDC__
(REAL x, REAL y)
#else
(x, y)
REAL x, y;
#endif
{
  return x == 0 && y == 0 ? 0 : atan2(x, y);
}

CARDINAL argc()
{
  return SYSTEM_argc;
}

/* Parallaxis specific functions ------------------------------------------ */

CARDINAL DIM2ID
#ifdef __STDC__
(CONFIGURATION Config, INTEGER *Dim)
#else
(Config, Dim) CONFIGURATION Config; INTEGER *Dim;
#endif
/* Expects an index and returns the corresponding ID number in the range */
/* from 1 to LEN(Config). 'Dim' is in the range from 1 to RANK(Config).  */
{
  CARDINAL Id, Dimension;

  Id = 0;
  if (IS_IN_CONFIG(Config, Dim)) {
    for (Dimension = RANK(Config); Dimension >= 1; Dimension--) {
      Id = Id * LEN1(Config, Dimension) +
        (Dim[Dimension - 1] - LOWER(Config, Dimension));
    }
    Id += 1;
  }
  return Id;
}

INTEGER *DIM
#ifdef __STDC__
(CONFIGURATION Config, CARDINAL Dim)
#else
(Config, Dim) CONFIGURATION Config; CARDINAL Dim;
#endif
/* Returns a vector with the indices of the dimension 'Dim'. 'Dim' is in */
/* the range from 1 to RANK(Config).                                     */
{
  INTEGER *Result;
  CARDINAL Dimension, Divisor;

  Result = (INTEGER *) malloc(sizeof(INTEGER) * LEN(Config));
  Divisor = 1;
  for (Dimension = 1; Dimension < Dim; Dimension++) {
    Divisor *= LEN1(Config, Dimension);
  }
  if (1 <= Dim && Dim <= RANK(Config)) {
    for (PE = 0; PE < LEN(Config); PE++) {
      Result[PE] = (PE / Divisor) % LEN1(Config, Dim) + LOWER(Config, Dim);
    }
  }
  return Result;
}

CARDINAL *ID
#ifdef __STDC__
(CONFIGURATION Config)
#else
(Config) CONFIGURATION Config;
#endif
/* Returns a vector with the ID. */
{
  CARDINAL *Result;

  Result = (CARDINAL *) malloc(sizeof(CARDINAL) * LEN(Config));
  for (PE = 0; PE < LEN(Config); PE++) {
    Result[PE] = PE + 1;
  }
  return Result;
}


/* Implementation of standard procedures ************************************/

/* Implementation of parallaxis standard procedures *************************/

void argv
#ifdef __STDC__
(CARDINAL index, CHAR arg[])
#else
(index, arg)
CARDINAL index;
CHAR *arg;
#endif
{
  if (index <= argc()) {
    (void) strcpy(arg, SYSTEM_argv[index]);
  }
}

/* MODULE IO -------------------------------------------------------------- */

void CloseInput()
{
  Done = (fclose(SYSTEM_stdin) == 0);
  SYSTEM_stdin = stdin;
}

void CloseOutput()
{
  Done = (fclose(SYSTEM_stdout) == 0);
  SYSTEM_stdout = stdout;
}

void OpenInput
#ifdef __STDC__
(CHAR *s)
#else
(s) CHAR *s;
#endif
{
  char filename[1024];

  if (strlen(s) == 0) {
    fprintf(stdout, "\nin> ");
    fflush(stdout);
    fscanf(stdin, "%s", filename);
  } else {
    strcpy(filename, s);
  }
  Done = ((SYSTEM_stdin = fopen(filename, "r")) != NULL);
}

void OpenOutput
#ifdef __STDC__
(CHAR *s)
#else
(s) CHAR *s;
#endif
{
  char filename[1024];

  if (strlen(s) == 0) {
    fprintf(stdout, "\nout> ");
    fflush(stdout);
    fscanf(stdin, "%s", filename);
  } else {
    strcpy(filename, s);
  }
  Done = ((SYSTEM_stdout = fopen(filename, "w")) != NULL);
}

void Read
#ifdef __STDC__
(CHAR *c)
#else
(c) CHAR *c;
#endif
{
  Done = (fscanf(SYSTEM_stdin, "%c", c) == 1);
}

void Read1
#ifdef __STDC__
(CONFIGURATION Config, CHAR *c)
#else
(Config, c) CONFIGURATION Config; CHAR *c;
#endif
{
  CARDINAL Divisor, Dimension;
  CHAR dummy;
  BOOLEAN DoneSum;

  DoneSum = TRUE;
  for (PE = 0; PE < LEN(Config); PE++) {
    if (IS_ACTIVE(Config, PE)) {
      Read(&(c[PE]));
      DoneSum = DoneSum && Done;
    }
    for (Dimension = 1, Divisor = 1; Dimension <= RANK(Config); Dimension++) {
      Divisor *= LEN1(Config, Dimension);
      if (((PE + 1) % Divisor) == 0) {
        Read(&dummy); /* deletes new line */
        DoneSum = DoneSum && Done;
      }
    }
  }
  Done = DoneSum;
}

void ReadBool
#ifdef __STDC__
(BOOLEAN *b)
#else
(b) BOOLEAN *b;
#endif
{
  char s[1024];

  ReadString(s);
  if (strcmp("TRUE", s) == 0) {
    *b = TRUE;
    Done = TRUE;
  } else if (strcmp("FALSE", s) == 0) {
    *b = FALSE;
    Done = TRUE;
  } else {
    Done = FALSE;
  }
  termCH = getc(SYSTEM_stdin);
  ungetc(termCH, SYSTEM_stdin);
}

void ReadBool1
#ifdef __STDC__
(CONFIGURATION Config, BOOLEAN *b)
#else
(Config, b) CONFIGURATION Config; BOOLEAN *b;
#endif
{
  BOOLEAN DoneSum;

  DoneSum = TRUE;
  for (PE = 0; PE < LEN(Config); PE++) {
    if (IS_ACTIVE(Config, PE)) {
      ReadBool(&(b[PE]));
      DoneSum = DoneSum && Done;
    }
  }
  Done = DoneSum;
}

void ReadCard
#ifdef __STDC__
(CARDINAL *c)
#else
(c) CARDINAL *c;
#endif
{
  Done = (fscanf(SYSTEM_stdin, "%lu", c) == 1);
  termCH = getc(SYSTEM_stdin);
  ungetc(termCH, SYSTEM_stdin);
}

void ReadCard1
#ifdef __STDC__
(CONFIGURATION Config, CARDINAL *c)
#else
(Config, c) CONFIGURATION Config; CARDINAL *c;
#endif
{
  BOOLEAN DoneSum;

  DoneSum = TRUE;
  for (PE = 0; PE < LEN(Config); PE++) {
    if (IS_ACTIVE(Config, PE)) {
      ReadCard(&(c[PE]));
      DoneSum = DoneSum && Done;
    }
  }
  Done = DoneSum;
}

void ReadInt
#ifdef __STDC__
(INTEGER *i)
#else
(i) INTEGER *i;
#endif
{
  Done = (fscanf(SYSTEM_stdin, "%ld", i) == 1);
  termCH = getc(SYSTEM_stdin);
  ungetc(termCH, SYSTEM_stdin);
}

void ReadInt1
#ifdef __STDC__
(CONFIGURATION Config, INTEGER *i)
#else
(Config, i) CONFIGURATION Config; INTEGER *i;
#endif
{
  BOOLEAN DoneSum;

  DoneSum = TRUE;
  for (PE = 0; PE < LEN(Config); PE++) {
    if (IS_ACTIVE(Config, PE)) {
      ReadInt(&(i[PE]));
      DoneSum = DoneSum && Done;
    }
  }
  Done = DoneSum;
}

void ReadReal
#ifdef __STDC__
(REAL *r)
#else
(r) REAL *r;
#endif
{
  Done = (fscanf(SYSTEM_stdin, "%lf", r) == 1);
  termCH = getc(SYSTEM_stdin);
  ungetc(termCH, SYSTEM_stdin);
}

void ReadReal1
#ifdef __STDC__
(CONFIGURATION Config, REAL *r)
#else
(Config, r) CONFIGURATION Config; REAL *r;
#endif
{
  BOOLEAN DoneSum;

  DoneSum = TRUE;
  for (PE = 0; PE < LEN(Config); PE++) {
    if (IS_ACTIVE(Config, PE)) {
      ReadReal(&(r[PE]));
      DoneSum = DoneSum && Done;
    }
  }
  Done = DoneSum;
}

void ReadString
#ifdef __STDC__
(CHAR *s)
#else
(s) CHAR *s;
#endif
{
  Done = (fscanf(SYSTEM_stdin, "%s", s) == 1);
  termCH = getc(SYSTEM_stdin);
  ungetc(termCH, SYSTEM_stdin);
}

void ReadString1
#ifdef __STDC__
(CONFIGURATION Config, CHAR **s)
#else
(Config, s) CONFIGURATION Config; CHAR *s;
#endif
{
  BOOLEAN DoneSum;

  DoneSum = TRUE;
  for (PE = 0; PE < LEN(Config); PE++) {
    if (IS_ACTIVE(Config, PE)) {
      ReadString(s[PE]);
      DoneSum = DoneSum && Done;
    }
  }
  Done = DoneSum;
}

void Write
#ifdef __STDC__
(CHAR c)
#else
(c) CHAR c;
#endif
{
  Done = (fprintf(SYSTEM_stdout, "%c", c) != EOF);
}

void Write1
#ifdef __STDC__
(CONFIGURATION Config, CHAR c[])
#else
(Config, c) CONFIGURATION Config; CHAR c[];
#endif
{
  CARDINAL Divisor, Dimension;
  BOOLEAN DoneSum;

  DoneSum = TRUE;
  for (PE = 0; PE < LEN(Config); PE++) {
    if (IS_ACTIVE(Config, PE)) {
      Write(c[PE]);
      DoneSum = DoneSum && Done;
    }
    for (Dimension = 1, Divisor = 1; Dimension <= RANK(Config); Dimension++) {
      Divisor *= LEN1(Config, Dimension);
      if (((PE + 1) % Divisor) == 0) {
        WriteLn();
        DoneSum = DoneSum && Done;
      }
    }
  }
  Done = DoneSum;
}

void WriteBool
#ifdef __STDC__
(BOOLEAN b, CARDINAL l)
#else
(b, l) BOOLEAN b; CARDINAL l;
#endif
{
  CARDINAL Spaces;
  BOOLEAN DoneSum;

  if (b) {
    DoneSum = (fprintf(SYSTEM_stdout, "TRUE") != EOF);
    for (Spaces = 4; Spaces < l; Spaces++) {
      Write(' ');
      DoneSum = DoneSum && Done;
    }
  } else {
    DoneSum = (fprintf(SYSTEM_stdout, "FALSE") != EOF);
    for (Spaces = 5; Spaces < l; Spaces++) {
      Write(' ');
      DoneSum = DoneSum && Done;
    }
  }
  Done = DoneSum;
}

void WriteBool1
#ifdef __STDC__
(CONFIGURATION Config, BOOLEAN b[], CARDINAL l)
#else
(Config, b, l) CONFIGURATION Config; BOOLEAN b[]; CARDINAL l;
#endif
{
  CARDINAL Divisor, Dimension;
  BOOLEAN DoneSum;

  DoneSum = TRUE;
  for (PE = 0; PE < LEN(Config); PE++) {
    if (IS_ACTIVE(Config, PE)) {
      WriteBool(b[PE], l);
      DoneSum = DoneSum && Done;
      Write(' ');
      DoneSum = DoneSum && Done;
    }
    for (Dimension = 1, Divisor = 1; Dimension <= RANK(Config); Dimension++) {
      Divisor *= LEN1(Config, Dimension);
      if (((PE + 1) % Divisor) == 0) {
        WriteLn();
        DoneSum = DoneSum && Done;
      }
    }
  }
  Done = DoneSum;
}

void WriteCard
#ifdef __STDC__
(CARDINAL c, CARDINAL l)
#else
(c, l) CARDINAL c; CARDINAL l;
#endif
{
  char format[16];

  /* output format for CARDINAL depends on the macro for CARDINAL in the
   * SYSTEM_C headerfile
   */
  strcpy(format, "%");
  sprintf(&(format[1]), "%lu", l);
  strcat(format, "lu");

  Done = (fprintf(SYSTEM_stdout, format, c) != EOF);
}

void WriteCard1
#ifdef __STDC__
(CONFIGURATION Config, CARDINAL c[], CARDINAL l)
#else
(Config, c, l) CONFIGURATION Config; CARDINAL c[]; CARDINAL l;
#endif
{
  CARDINAL Divisor, Dimension;
  BOOLEAN DoneSum;

  DoneSum = TRUE;
  for (PE = 0; PE < LEN(Config); PE++) {
    if (IS_ACTIVE(Config, PE)) {
      WriteCard(c[PE], l);
      DoneSum = DoneSum && Done;
      Write(' ');
      DoneSum = DoneSum && Done;
    }
    for (Dimension = 1, Divisor = 1; Dimension <= RANK(Config); Dimension++) {
      Divisor *= LEN1(Config, Dimension);
      if (((PE + 1) % Divisor) == 0) {
        WriteLn();
        DoneSum = DoneSum && Done;
      }
    }
  }
  Done = DoneSum;
}

void WriteFixPt
#ifdef __STDC__
(REAL r, CARDINAL l, CARDINAL m)
#else
(r, l, m) REAL r; CARDINAL l, m;
#endif
{
  char format[32];

  /* output format for REAL depends on the macro for REAL in the
   * SYSTEM_C headerfile
   */
  strcpy(format, "%");
  sprintf(&(format[1]), "%lu", l);
  strcat(format, ".");
  sprintf(&(format[strlen(format)]), "%lu", m);
  strcat(format, "f");

  Done = (fprintf(SYSTEM_stdout, format, r) != EOF);
}

void WriteFixPt1
#ifdef __STDC__
(CONFIGURATION Config, REAL r[], CARDINAL l, CARDINAL m)
#else
(Config, r, l, m) CONFIGURATION Config; REAL r[]; CARDINAL l, m;
#endif
{
  CARDINAL Divisor, Dimension;
  BOOLEAN DoneSum;

  DoneSum = TRUE;
  for (PE = 0; PE < LEN(Config); PE++) {
    if (IS_ACTIVE(Config, PE)) {
      WriteFixPt(r[PE], l, m);
      DoneSum = DoneSum && Done;
      Write(' ');
      DoneSum = DoneSum && Done;
    }
    for (Dimension = 1, Divisor = 1; Dimension <= RANK(Config); Dimension++) {
      Divisor *= LEN1(Config, Dimension);
      if (((PE + 1) % Divisor) == 0) {
        WriteLn();
        DoneSum = DoneSum && Done;
      }
    }
  }
  Done = DoneSum;
}

void WriteInt
#ifdef __STDC__
(INTEGER i, CARDINAL l)
#else
(i, l) INTEGER i; CARDINAL l;
#endif
{
  char format[16];

  /* output formats for INTEGER and CARDINAL depend on the macro for INTEGER
   * and CARDINAL in the SYSTEM_C headerfile
   */
  strcpy(format, "%");
  sprintf(&(format[1]), "%lu", l);
  strcat(format, "ld");

  Done = (fprintf(SYSTEM_stdout, format, i) != EOF);
}

void WriteInt1
#ifdef __STDC__
(CONFIGURATION Config, INTEGER i[], CARDINAL l)
#else
(Config, i, l) CONFIGURATION Config; INTEGER i[]; CARDINAL l;
#endif
{
  CARDINAL Divisor, Dimension;
  BOOLEAN DoneSum;

  DoneSum = TRUE;
  for (PE = 0; PE < LEN(Config); PE++) {
    if (IS_ACTIVE(Config, PE)) {
      WriteInt(i[PE], l);
      DoneSum = DoneSum && Done;
      Write(' ');
      DoneSum = DoneSum && Done;
    }
    for (Dimension = 1, Divisor = 1; Dimension <= RANK(Config); Dimension++) {
      Divisor *= LEN1(Config, Dimension);
      if (((PE + 1) % Divisor) == 0) {
        WriteLn();
        DoneSum = DoneSum && Done;
      }
    }
  }
  Done = DoneSum;
}

void WriteLn()
{
  fprintf(SYSTEM_stdout, "\n");
}

void WriteReal
#ifdef __STDC__
(REAL r, CARDINAL l)
#else
(r, l) REAL r; CARDINAL l;
#endif
{
  char format[16];
  char RealString[512];
  long Position;

  /* number of digits before the dot */
  sprintf(RealString, "%1f", r);
  Position = 0;
  while (RealString[Position] != '.') {
    Position++;
  }
  /* output format for REAL depends on the macro for REAL in the
   * SYSTEM_C headerfile
   */
  strcpy(format, "%");
  sprintf(&(format[1]), "%lu", l);
  strcat(format, ".");
  sprintf(&(format[strlen(format)]), "%lu",
    Position >= l ? 0 : l - Position - 5);
  strcat(format, "e");

  Done = (fprintf(SYSTEM_stdout, format, r) != EOF);
}

void WriteReal1
#ifdef __STDC__
(CONFIGURATION Config, REAL r[], CARDINAL l)
#else
(Config, r, l) CONFIGURATION Config; REAL r[]; CARDINAL l;
#endif
{
  CARDINAL Divisor, Dimension;
  BOOLEAN DoneSum;

  DoneSum = TRUE;
  for (PE = 0; PE < LEN(Config); PE++) {
    if (IS_ACTIVE(Config, PE)) {
      WriteReal(r[PE], l);
      DoneSum = DoneSum && Done;
      Write(' ');
      DoneSum = DoneSum && Done;
    }
    for (Dimension = 1, Divisor = 1; Dimension <= RANK(Config); Dimension++) {
      Divisor *= LEN1(Config, Dimension);
      if (((PE + 1) % Divisor) == 0) {
        WriteLn();
        DoneSum = DoneSum && Done;
      }
    }
  }
  Done = DoneSum;
}

void WriteString
#ifdef __STDC__
(CHAR *s)
#else
(s) CHAR *s;
#endif
{
  Done = (fprintf(SYSTEM_stdout, "%s", s) != EOF);
}

void WriteString1
#ifdef __STDC__
(CONFIGURATION Config, CHAR **s)
#else
(Config, s) CONFIGURATION Config; CHAR **s;
#endif
{
  CARDINAL Divisor, Dimension;
  BOOLEAN DoneSum;

  DoneSum = TRUE;
  for (PE = 0; PE < LEN(Config); PE++) {
    if (IS_ACTIVE(Config, PE)) {
      WriteString(s[PE]);
      DoneSum = DoneSum && Done;
      Write(' ');
      DoneSum = DoneSum && Done;
    }
    for (Dimension = 1, Divisor = 1; Dimension <= RANK(Config); Dimension++) {
      Divisor *= LEN1(Config, Dimension);
      if (((PE + 1) % Divisor) == 0) {
        WriteLn();
        DoneSum = DoneSum && Done;
      }
    }
  }
  Done = DoneSum;
}

void WriteStringChar1
#ifdef __STDC__
(CONFIGURATION Config, CHAR s[])
#else
(Config, s) CONFIGURATION Config; CHAR s[];
#endif
{
  CARDINAL Divisor, Dimension;
  BOOLEAN DoneSum;

  DoneSum = TRUE;
  for (PE = 0; PE < LEN(Config); PE++) {
    if (IS_ACTIVE(Config, PE)) {
      Write(s[PE]);
      DoneSum = DoneSum && Done;
      Write(' ');
      DoneSum = DoneSum && Done;
    }
    for (Dimension = 1, Divisor = 1; Dimension <= RANK(Config); Dimension++) {
      Divisor *= LEN1(Config, Dimension);
      if (((PE + 1) % Divisor) == 0) {
        WriteLn();
        DoneSum = DoneSum && Done;
      }
    }
  }
  Done = DoneSum;
}

/* Parallaxis specific procedures ----------------------------------------- */

void ID2DIM
#ifdef __STDC__
(CONFIGURATION Config, CARDINAL Id, INTEGER Dim[])
#else
(Config, Id, Dim) CONFIGURATION Config; CARDINAL Id; INTEGER Dim[];
#endif
{
  CARDINAL Dimension;

  if (1 <= Id && Id <= LEN(Config)) {
    Id -= 1;
    for (Dimension = 1 ; Dimension <= RANK(Config); Dimension++) {
      Dim[Dimension - 1] = Id % LEN1(Config, Dimension) +
        LOWER(Config, Dimension);
      Id /= LEN1(Config, Dimension);
    }
  }
}

/* Implementation of set operators ******************************************/

CARDINAL SET_RANGE1
#ifdef __STDC__
(register CARDINAL lo, register CARDINAL hi)
#else
(lo, hi) register CARDINAL lo, hi;
#endif
{
  return (lo <= hi ? ~0X0UL >> lo << (lo + SYSTEM_MaxSet - hi) >> (SYSTEM_MaxSet - hi) : 0X0UL);
}

/* Implementation of compiler functions *************************************/

void CaseError
#ifdef __STDC__
(char file[], int line)
#else
(file, line) char file[]; int line;
#endif
{
  (void) fprintf (stderr, "\"%s\", line %1d: case expression out of range\n",
    file, line);
  exit(1);
}

void ReturnError
#ifdef __STDC__
(char file[], int line)
#else
(file, line) char file[]; int line;
#endif
{
  (void) fprintf (stderr, "\"%s\", line %1d: missing return from function\n",
    file, line);
  exit(1);
}

INTEGER CHECK_PE_NUMBER
#ifdef __STDC__
(CONFIGURATION Config, INTEGER PeNumber, char file[], int line)
#else
(Config, PeNumber, file, line)
CONFIGURATION Config; INTEGER PeNumber;  char file[]; int line;
#endif
/* Checks, if a PE-number is in the range of the configuration. */
/* On success, the PE-number is returned, otherwise the programm */
/* terminates. */
{
  if (0 <= PeNumber && PeNumber < LEN(Config)) {
    return PeNumber;
  } else {
    (void) fprintf (stderr, "\"%s\", line %1d: processor number out of range\n",
      file, line);
    exit(1);
  }
}

/* Implementation of parallaxis compiler functions **************************/

/* Functions for configurations ------------------------------------------- */

void INIT_CONFIGURATION
#ifdef __STDC__
(char Name[], CONFIGURATION *Config, CARDINAL Rank)
#else
(Name, Config, Rank) char Name[]; CONFIGURATION *Config; CARDINAL Rank;
#endif
/* Initialises the configuration with its rank and allocates memory. */
{
  strncpy(Config->Name, Name, 256);
  Config->Rank = Rank;
  Config->Len = 1;
  Config->Lower = (INTEGER *) malloc(sizeof(INTEGER) * Rank);
  Config->Upper = (INTEGER *) malloc(sizeof(INTEGER) * Rank);
  Config->DimLen = (INTEGER *) malloc(sizeof(INTEGER) * Rank);
  Config->Generics = NULL;
  Config->GenericsParam = NULL;
}

void SET_BOUNDS
#ifdef __STDC__
(CONFIGURATION *Config, CARDINAL Dim, INTEGER Lwb, INTEGER Upb)
#else
(Config, Dim, Lwb, Upb)
CONFIGURATION *Config; CARDINAL Dim; INTEGER Lwb; INTEGER Upb;
#endif
/* Sets the bounds of one dimension 'Dim'. Dim starts with 0 and */
/* represents the most left dimension.                           */
{
  Config->Lower[Dim] = Lwb;
  Config->Upper[Dim] = Upb;
  Config->DimLen[Dim] = Upb - Lwb + 1;
  Config->Len *= Config->DimLen[Dim];
}

void INIT_STACKS
#ifdef __STDC__
(CONFIGURATION *Config)
#else
(Config) CONFIGURATION *Config;
#endif
/* Initialises all stacks of the configuration. */
{
  INIT_ACTIVE_STACK(Config);
  INIT_ELSE_STACK(Config);
}

void COPY_STACKS
#ifdef __STDC__
(CONFIGURATION *DestConfig, CONFIGURATION *SourceConfig)
#else
(DestConfig, SourceConfig)
CONFIGURATION *DestConfig; CONFIGURATION *SourceConfig;
#endif
/* Copies the references to the original stacks of the configuration group */
/* which are hold from the first declared configuration in the group.      */
{
  DestConfig->ActiveStack = SourceConfig->ActiveStack;
  DestConfig->ElseStack   = SourceConfig->ElseStack;
}

void FREE_CONFIGURATION
#ifdef __STDC__
(CONFIGURATION *Config, BOOLEAN FreeStacks)
#else
(Config, FreeStack) CONFIGURATION *Config; BOOLEAN FreeStacks;
#endif
/* Frees memory allocated for stacks and bounds. */
{
  if (FreeStacks) {
    FREE_ACTIVE_STACK(Config);
    FREE_ELSE_STACK(Config);
  }
  free (Config->Lower);
  free (Config->Upper);
  free (Config->DimLen);
}

/* Functions for connections ---------------------------------------------- */

/* connections without index ----- */

void INIT_CONNECTION
#ifdef __STDC__
(CONNECTION *Connection, CONFIGURATION *SourceConfig,
  CONFIGURATION *DestinationConfig)
#else
(Connection, SourceConfig, DestinationConfig)
CONNECTION *Connection; CONFIGURATION *SourceConfig;
CONFIGURATION *DestinationConfig;
#endif
/* Initialises the connection with the source and destination configuration. */
{
  Connection->SourceConfig = SourceConfig;
  Connection->DestinationConfig = DestinationConfig;
  /*  Connection->Links = NULL;
   *  Do not do it! In case the connection declaration is surrounded by a
   *  FOR-loop, all up to now computed connections will be lost!
   *  The Pointer is set to NULL when the configuration is declared.
   */
}

void REGISTER_LINK
#ifdef __STDC__
(CONNECTION *Connection, INTEGER *SourceArray, INTEGER *DestinationArray)
#else
(Connection, SourceArray, DestinationArray)
CONNECTION *Connection; INTEGER *SourceArray; INTEGER *DestinationArray;
#endif
/* Stores a link of the connection. */
{
  LINK *Link;

  if (IS_IN_CONFIG(*(Connection->SourceConfig), SourceArray) &&
      IS_IN_CONFIG(*(Connection->DestinationConfig), DestinationArray)) {
    Link = (LINK *) malloc(sizeof(LINK));
    Link->Next = Connection->Links;
    Link->SourceID = DIM2ID(*(Connection->SourceConfig), SourceArray) - 1;
    Link->DestinationID = DIM2ID(*(Connection->DestinationConfig),
      DestinationArray) - 1;
    Connection->Links = Link;
  }
}

LINK *GET_LINKS
#ifdef __STDC__
(CONNECTION Connection)
#else
(Connection)
CONNECTION Connection;
#endif
{
  return Connection.Links;
}

void FREE_CONNECTION
#ifdef __STDC__
(CONNECTION *Connection)
#else
(Connection)
CONNECTION *Connection;
#endif
{
  LINK *Link, *Link2;

  Link = Connection->Links;
  while (Link != NULL) {
    Link2 = Link;
    Link = Link->Next;
    free(Link2);
  }
  Connection->Links = NULL;
}

/* connections with index ----- */

void INIT_PARAMETER_CONNECTION
#ifdef __STDC__
(PARAMETER_CONNECTION *Connection, CONFIGURATION *SourceConfig,
  CONFIGURATION *DestinationConfig)
#else
(Connection, SourceConfig, DestinationConfig)
PARAMETER_CONNECTION *Connection;
CONFIGURATION *SourceConfig;
CONFIGURATION *DestinationConfig;
#endif
/* Initialises the connection with the source and destination configuration. */
{
  Connection->SourceConfig = SourceConfig;
  Connection->DestinationConfig = DestinationConfig;
  /*  Connection->Links = NULL;
   *  Do not do it! In case the connection declaration is surrounded by a
   *  FOR-loop, all up to now computet connections will be lost!
   *  The Pointer is set to NULL when the configuration is declared.
   */
}

void REGISTER_PARAMETER_LINK
#ifdef __STDC__
(PARAMETER_CONNECTION *Connection, INTEGER Index, INTEGER *SourceArray,
  INTEGER *DestinationArray)
#else
(Connection, Index, SourceArray, DestinationArray)
CONNECTION *Connection; INTEGER Index; INTEGER *SourceArray;
INTEGER *DestinationArray;
#endif
/* Stores a link of the connection. */
{
  PARAMETER_LINK *ParameterLink;
  LINK *Link;

  if (IS_IN_CONFIG(*(Connection->SourceConfig), SourceArray) &&
      IS_IN_CONFIG(*(Connection->DestinationConfig), DestinationArray)) {
    /* search for the connection with the index 'Index' */
    ParameterLink = Connection->ParameterLinks;
    while (ParameterLink != NULL && ParameterLink->Parameter != Index) {
      ParameterLink = ParameterLink->Next;
    }
    /* if index not found, then creat a new connection with this index */
    if (ParameterLink == NULL) {
      ParameterLink = (PARAMETER_LINK *) malloc(sizeof(PARAMETER_LINK));
      ParameterLink->Next = Connection->ParameterLinks;
      ParameterLink->Parameter = Index;
      Connection->ParameterLinks = ParameterLink;
    }
    Link = (LINK *) malloc(sizeof(LINK));
    Link->Next = ParameterLink->Links;
    Link->SourceID = DIM2ID(*(Connection->SourceConfig), SourceArray) - 1;
    Link->DestinationID = DIM2ID(*(Connection->DestinationConfig),
      DestinationArray) - 1;
    ParameterLink->Links = Link;
  }
}

LINK *GET_PARAMETER_LINKS
#ifdef __STDC__
(PARAMETER_CONNECTION Connection, INTEGER Parameter)
#else
(Connection, Parameter)
PARAMETER_CONNECTION Connection; INTEGER Parameter;
#endif
{
  PARAMETER_LINK *ParameterLinks;

  ParameterLinks = Connection.ParameterLinks;
  while (ParameterLinks != NULL && ParameterLinks->Parameter != Parameter) {
    ParameterLinks = ParameterLinks->Next;
  }
  if (ParameterLinks != NULL) {
    return ParameterLinks->Links;
  } else {
    return NULL;
  }
}

void FREE_PARAMETER_CONNECTION
#ifdef __STDC__
(PARAMETER_CONNECTION *Connection)
#else
(Connection)
PARAMETER_CONNECTION *Connection;
#endif
{
  PARAMETER_LINK *ParaLink, *ParaLink2;
  LINK *Link, *Link2;

  ParaLink = Connection->ParameterLinks;
  while (ParaLink != NULL) {  
    Link = ParaLink->Links;
    while (Link != NULL) {
      Link2 = Link;
      Link = Link->Next;
      free(Link2);
    }
    ParaLink2 = ParaLink;
    ParaLink = ParaLink->Next;
    free(ParaLink2);
  }
  Connection->ParameterLinks = NULL;
}

/* generic connections without index ----- */

void INIT_GENERIC_CONNECTION
#ifdef __STDC__
(char Connection[], CONFIGURATION *Configuration)
#else
(Connection, Configuration)
char Connection[];
CONFIGURATION *Configuration;
#endif
{
  GENERIC_CONNECTIONS *Connect;

  Connect = (GENERIC_CONNECTIONS *) malloc(sizeof(GENERIC_CONNECTIONS));
  strncpy(Connect->ConnectionName, Connection, 256);
  Connect->Configuration = Configuration;
  Connect->Links = NULL;
  Connect->Next = ALL_GENERIC_CONNECTIONS;
  ALL_GENERIC_CONNECTIONS = Connect;
}

void REGISTER_GENERIC_LINK
#ifdef __STDC__
(char ConnectionName[], CONFIGURATION *Configuration, INTEGER *SourceArray,
  INTEGER *DestinationArray)
#else
(ConnectionName, Configuration, SourceArray, DestinationArray)
char ConnectionName[]; CONFIGURATION *Configuration;
INTEGER *SourceArray; INTEGER *DestinationArray;
#endif
/* Stores a link of the generic connection. */
{
  GENERIC_CONNECTIONS *Connection;
  LINK *Link;

  /* search generic connection */
  Connection = ALL_GENERIC_CONNECTIONS;
  while (Connection != NULL && (
         strcmp(ConnectionName, Connection->ConnectionName) != 0 ||
         strcmp(Configuration->Name, Connection->Configuration->Name) != 0)) {
    Connection = Connection->Next;
  }
  /* check if link is in configuration */
  if (IS_IN_CONFIG(*(Connection->Configuration), SourceArray) &&
      IS_IN_CONFIG(*(Connection->Configuration), DestinationArray)) {
    Link = (LINK *) malloc(sizeof(LINK));
    Link->Next = Connection->Links;
    Link->SourceID = DIM2ID(*(Connection->Configuration), SourceArray) - 1;
    Link->DestinationID = DIM2ID(*(Connection->Configuration),
      DestinationArray) - 1;
    Connection->Links = Link;
  }
}

LINK *GET_GENERIC_LINKS
#ifdef __STDC__
(char Connection[], CONFIGURATION *Configuration)
#else
(Connection, Configuration)
char Connection[];
CONFIGURATION *Configuration;
#endif
{
  GENERIC_CONNECTIONS *All_Generic;

  All_Generic = ALL_GENERIC_CONNECTIONS;
  while (All_Generic != NULL && (
         strcmp(Connection, All_Generic->ConnectionName) != 0 ||
         strcmp(Configuration->Name, All_Generic->Configuration->Name) != 0)) {
    All_Generic = All_Generic->Next;
  }
  if (All_Generic != NULL) {
    return All_Generic->Links;
  } else {
    return NULL;
  }
}

void FREE_GENERIC_CONNECTION
#ifdef __STDC__
(char Connection[], CONFIGURATION *Configuration)
#else
(Connection, Configuration)
char Connection[];
CONFIGURATION *Configuration;
#endif
{
  GENERIC_CONNECTIONS *Connect1, *Connect2;
  LINK *Link1, *Link2;

  /* search generic connection */
  Connect1 = ALL_GENERIC_CONNECTIONS;
  Connect2 = NULL;
  while (Connect1 != NULL && (
         strcmp(Connection, Connect1->ConnectionName) != 0 ||
         strcmp(Configuration->Name, Connect1->Configuration->Name) != 0)) {
    Connect2 = Connect1; /* pointer to predecessor */
    Connect1 = Connect1->Next;
  }
  /* take connection out of the list */
  if (Connect2 == NULL) { /* connection is the first in the list */
    ALL_GENERIC_CONNECTIONS = Connect1->Next;
  } else {
    Connect2->Next = Connect1->Next;
  }
  /* free links */
  Link1 = Connect1->Links;
  while (Link1 != NULL) {
    Link2 = Link1;
    Link1 = Link1->Next;
    free(Link2);
  }
  /* free connection */
  free(Connect1);  
}

void FREE_ALL_GENERIC_CONNECTIONS()
{
  GENERIC_CONNECTIONS *Connect1, *Connect2;
  LINK *Link1, *Link2;

  Connect1 = ALL_GENERIC_CONNECTIONS;
  while (Connect1 != NULL) {
    /* free links */
    Link1 = Connect1->Links;
    while (Link1 != NULL) {
      Link2 = Link1;
      Link1 = Link1->Next;
      free(Link2);
    }
    Connect2 = Connect1;
    Connect1 = Connect1->Next;
    /* free connection */
    free(Connect2);
  }
  ALL_GENERIC_CONNECTIONS = NULL;
}

/* generic connections with index ----- */

void INIT_GENERIC_PARAMETER_CONNECTION
#ifdef __STDC__
(char ConnectionName[], CONFIGURATION *Configuration)
#else
(Connection, Configuration)
char Connection[];
CONFIGURATION *Configuration;
#endif
{
  GENERIC_PARAMETER_CONNECTIONS *Connection;

  Connection = (GENERIC_PARAMETER_CONNECTIONS *)
    malloc(sizeof(GENERIC_PARAMETER_CONNECTIONS));
  strncpy(Connection->ConnectionName, ConnectionName, 256);
  Connection->Configuration = Configuration;
  Connection->ParameterLinks = NULL;
  Connection->Next = ALL_GENERIC_PARAMETER_CONNECTIONS;
  ALL_GENERIC_PARAMETER_CONNECTIONS = Connection;
}

void REGISTER_GENERIC_PARAMETER_LINK
#ifdef __STDC__
(char ConnectionName[], CONFIGURATION *Configuration, INTEGER Parameter,
  INTEGER *SourceArray, INTEGER *DestinationArray)
#else
(ConnectionName, Configuration, Parameter, SourceArray, DestinationArray)
char ConnectionName[]; CONFIGURATION *Configuration; INTEGER Parameter;
INTEGER *SourceArray; INTEGER *DestinationArray;
#endif
/* Stores a link of the generic parameter connection. */
{
  GENERIC_PARAMETER_CONNECTIONS *Connection;
  PARAMETER_LINK *ParameterLink;
  LINK *Link;

  /* search generic connection */
  Connection = ALL_GENERIC_PARAMETER_CONNECTIONS;
  while (Connection != NULL && (
         strcmp(ConnectionName, Connection->ConnectionName) != 0 ||
         strcmp(Configuration->Name, Connection->Configuration->Name) != 0)) {
    Connection = Connection->Next;
  }
  /* check if link is in configuration */
  if (IS_IN_CONFIG(*(Connection->Configuration), SourceArray) &&
      IS_IN_CONFIG(*(Connection->Configuration), DestinationArray)) {
    /* search for the connection with the index 'Parameter' */
    ParameterLink = Connection->ParameterLinks;
    while (ParameterLink != NULL && ParameterLink->Parameter != Parameter) {
      ParameterLink = ParameterLink->Next;
    }
    /* if index not found, then creat a new connection with this index */
    if (ParameterLink == NULL) {
      ParameterLink = (PARAMETER_LINK *) malloc(sizeof(PARAMETER_LINK));
      ParameterLink->Next = Connection->ParameterLinks;
      ParameterLink->Parameter = Parameter;
      Connection->ParameterLinks = ParameterLink;
    }
    Link = (LINK *) malloc(sizeof(LINK));
    Link->Next = ParameterLink->Links;
    Link->SourceID = DIM2ID(*(Connection->Configuration), SourceArray) - 1;
    Link->DestinationID = DIM2ID(*(Connection->Configuration),
      DestinationArray) - 1;
    ParameterLink->Links = Link;
  }
}

LINK *GET_GENERIC_PARAMETER_LINKS
#ifdef __STDC__
(char Connection[], CONFIGURATION *Configuration, INTEGER Parameter)
#else
(Connection, Configuration, Parameter)
char Connection[];
CONFIGURATION *Configuration;
INTEGER Parameter;
#endif
{
  GENERIC_PARAMETER_CONNECTIONS *All_Generic;
  PARAMETER_LINK *ParameterLinks;

  All_Generic = ALL_GENERIC_PARAMETER_CONNECTIONS;
  while (All_Generic != NULL && (
         strcmp(Connection, All_Generic->ConnectionName) != 0 ||
         strcmp(Configuration->Name, All_Generic->Configuration->Name) != 0)) {
    All_Generic = All_Generic->Next;
  }
  ParameterLinks = All_Generic->ParameterLinks;
  while (ParameterLinks != NULL && ParameterLinks->Parameter != Parameter) {
    ParameterLinks = ParameterLinks->Next;
  }
  if (ParameterLinks != NULL) {
    return ParameterLinks->Links;
  } else {
    return NULL;
  }
}

void FREE_GENERIC_PARAMETER_CONNECTION
#ifdef __STDC__
(char Connection[], CONFIGURATION *Configuration)
#else
(Connection, Configuration)
char Connection[];
CONFIGURATION *Configuration;
#endif
{
  GENERIC_PARAMETER_CONNECTIONS *Connect1, *Connect2;
  PARAMETER_LINK *ParaLink1, *ParaLink2;
  LINK *Link1, *Link2;

  /* search generic parameter connection */
  Connect1 = ALL_GENERIC_PARAMETER_CONNECTIONS;
  Connect2 = NULL;
  while (Connect1 != NULL && (
         strcmp(Connection, Connect1->ConnectionName) != 0 ||
         strcmp(Configuration->Name, Connect1->Configuration->Name) != 0)) {
    Connect2 = Connect1; /* pointer to predecessor */
    Connect1 = Connect1->Next;
  }
  /* found -> take connection out of the list */
  if (Connect1 != NULL) {
    if (Connect2 == NULL) { /* connection is the first in the global list */
      ALL_GENERIC_PARAMETER_CONNECTIONS = Connect1->Next;
    } else {
      Connect2->Next = Connect1->Next;
    }
    ParaLink1 = Connect1->ParameterLinks;
    while (ParaLink1 != NULL) {  
      Link1 = ParaLink1->Links;
      while (Link1 != NULL) {
        Link2 = Link1;
        Link1 = Link1->Next;
        free(Link2);
      }
      ParaLink2 = ParaLink1;
      ParaLink1 = ParaLink1->Next;
      free(ParaLink2);
    }
    free(Connect1); 
  }
}

void FREE_ALL_GENERIC_PARAMETER_CONNECTIONS()
{
  GENERIC_PARAMETER_CONNECTIONS *Connect1, *Connect2;
  PARAMETER_LINK *ParaLink1, *ParaLink2;
  LINK *Link1, *Link2;

  Connect1 = ALL_GENERIC_PARAMETER_CONNECTIONS;
  while (Connect1 != NULL) {
    ParaLink1 = Connect1->ParameterLinks;
    while (ParaLink1 != NULL) {  
      Link1 = ParaLink1->Links;
      while (Link1 != NULL) {
        Link2 = Link1;
        Link1 = Link1->Next;
        free(Link2);
      }
      ParaLink2 = ParaLink1;
      ParaLink1 = ParaLink1->Next;
      free(ParaLink2);
    }
    Connect2 = Connect1;
    Connect1 = Connect1->Next;
    free(Connect2); 
  }
  ALL_GENERIC_CONNECTIONS = NULL;
}

/* Functions for checks --------------------------------------------------- */

BOOLEAN IS_IN_CONFIG
#ifdef __STDC__
(CONFIGURATION Config, INTEGER *IndexArray)
#else
(Config, IndexArray) CONFIGURATION Config; INTEGER *IndexArray;
#endif
/* Checks, whether an index is in the configuration bounds or not. */
{
  BOOLEAN IsInConfig;
  CARDINAL Dimension;

  IsInConfig = TRUE;
  for (Dimension = 1; Dimension <= RANK(Config); Dimension++) {
    IsInConfig = IsInConfig &&
      LOWER(Config, Dimension) <= IndexArray[Dimension - 1] &&
      IndexArray[Dimension - 1] <= UPPER(Config, Dimension);
  }
  return IsInConfig;
}

/* Functions for active stack --------------------------------------------- */

void INIT_ACTIVE_STACK
#ifdef __STDC__
(CONFIGURATION *Config)
#else
(Config) CONFIGURATION *Config;
#endif
/* Initialises the processor status stack (active/inactive) and sets all */
/* processors to activ.                                                  */
{
  Config->ActiveStack = (ACTIVE_STACK_PTR) malloc(sizeof(ACTIVE_STACK));
  /* assuming eight bits per byte */
  Config->ActiveStack->ElemSize = (LEN(*Config) + 7) / 8;
  Config->ActiveStack->StackElem = NULL;
  PUSH_ACTIVE_ALL(*Config); /* create stackelement and set all to active */
}

void FREE_ACTIVE_STACK
#ifdef __STDC__
(CONFIGURATION *Config)
#else
(Config) CONFIGURATION *Config;
#endif
/* Frees the processor status stack. */
{
  while (Config->ActiveStack->StackElem != NULL) {
    POP_ACTIVE(*Config);
  }
  free(Config->ActiveStack);
}

void PUSH_ACTIVE_ALL
#ifdef __STDC__
(CONFIGURATION Config)
#else
(Config) CONFIGURATION Config;
#endif
/* Push the actual processor status and sets all processors to active. */
{
  BOOLEAN_ELEM_PTR StackElem;

  StackElem = (BOOLEAN_ELEM_PTR) malloc(sizeof(BOOLEAN_ELEM));
  StackElem->Elem = (CHAR *) malloc(Config.ActiveStack->ElemSize);
  /* assuming eight bits per char, all bits are filled with '1' */
  (void) memset(StackElem->Elem, 255, Config.ActiveStack->ElemSize);
  StackElem->Previous = Config.ActiveStack->StackElem;
  Config.ActiveStack->StackElem = StackElem;
}

void PUSH_ACTIVE
#ifdef __STDC__
(CONFIGURATION Config)
#else
(Config) CONFIGURATION Config;
#endif
/* Push the actual processor status. */
{
  BOOLEAN_ELEM_PTR StackElem;

  StackElem = (BOOLEAN_ELEM_PTR) malloc(sizeof(BOOLEAN_ELEM));
  StackElem->Elem = (CHAR *) malloc(Config.ActiveStack->ElemSize);
  /* assuming eight bits per char, bits are copied from old stack */
  (void) memcpy(StackElem->Elem, Config.ActiveStack->StackElem->Elem,
    Config.ActiveStack->ElemSize);
  StackElem->Previous = Config.ActiveStack->StackElem;
  Config.ActiveStack->StackElem = StackElem;
}

void POP_ACTIVE
#ifdef __STDC__
(CONFIGURATION Config)
#else
(Config) CONFIGURATION Config;
#endif
/* Pop the previous processor status. */
{
  BOOLEAN_ELEM_PTR StackElem;

  StackElem = Config.ActiveStack->StackElem;
  Config.ActiveStack->StackElem = StackElem->Previous;
  free(StackElem->Elem);
  free(StackElem);
}

void TOP_ACTIVE
#ifdef __STDC__
(CONFIGURATION Config)
#else
(Config) CONFIGURATION Config;
#endif
/* Is same as POP_ACTIVE(); PUSH_ACTIVE();. Just copies the active status */
/* from the previous stack element into the actual stack element. */
{
  (void) memcpy(Config.ActiveStack->StackElem->Elem,
    Config.ActiveStack->StackElem->Previous->Elem,
    Config.ActiveStack->ElemSize);
}

BOOLEAN_ELEM_PTR READ_ACTIVE
#ifdef __STDC__
(CONFIGURATION Config)
#else
(Config) CONFIGURATION Config;
#endif
/* Reads the pointer to the actual active stack element */
{
  return Config.ActiveStack->StackElem;
}

void RESET_ACTIVE
#ifdef __STDC__
(CONFIGURATION Config, BOOLEAN_ELEM_PTR Pointer)
#else
(Config, Pointer) CONFIGURATION Config; BOOLEAN_ELEM_PTR Pointer;
#endif
/* Resets the active stack to the element with the pointer */
{
  while (Config.ActiveStack->StackElem != Pointer) {
    POP_ACTIVE(Config);
  }
}

void SET_INACTIVE
#ifdef __STDC__
(CONFIGURATION Config, CARDINAL Processor)
#else
(Config, Processor) CONFIGURATION Config; CARDINAL Processor; 
#endif
/* Sets a processor in the range from 0 to LEN(Config) - 1 inactive. */
{
  Config.ActiveStack->StackElem->Elem[Processor / 8] &=
    (~(1 << (Processor % 8)));
}

void EXIT_ALL_ACTIVES
#ifdef __STDC__
(CONFIGURATION Config, BOOLEAN_ELEM_PTR ActivePtr, BOOLEAN_ELEM_PTR ElsePtr)
#else
(Config, Pointer) CONFIGURATION Config; BOOLEAN_ELEM_PTR Pointer; 
#endif
/* Sets all active processors of the configuration up to the (active */
/* stack) pointer to inactive. Also for all active processors their  */
/* else-stack is deactivated. This is used for the parallel LOOP to  */
/* disactivate PEs up to the end of the LOOP.                        */
{
  BOOLEAN_ELEM_PTR TempPtr;

  FOR_ALL_ACTIVES(Config) {
    TempPtr = Config.ActiveStack->StackElem;
    while (TempPtr != ActivePtr) {
      TempPtr->Elem[PE / 8] &= (~(1 << (PE % 8)));
      TempPtr = TempPtr->Previous;
    };
    TempPtr = Config.ElseStack->StackElem;
    while (TempPtr != ElsePtr) {
      TempPtr->Elem[PE / 8] &= (~(1 << (PE % 8)));
      TempPtr = TempPtr->Previous;
    };
  }
}

BOOLEAN IS_ACTIVE
#ifdef __STDC__
(CONFIGURATION Config, CARDINAL Processor)
#else
(Config, Processor) CONFIGURATION Config; CARDINAL Processor; 
#endif
/* Checks, whether a processor in the range from 0 to LEN(Config) - 1 is */
/* active or not.                                                        */
{
  if (Processor < LEN(Config)) {
    return (Config.ActiveStack->StackElem->Elem[Processor / 8] &
            (1 << (Processor % 8))) != 0;
  } else {
    return FALSE;
  }
}

BOOLEAN IS_ANY_ACTIVE
#ifdef __STDC__
(CONFIGURATION Config)
#else
(Config) CONFIGURATION Config; 
#endif
/* Checks, whether at least one processor of the configuration is active. */
{
  CARDINAL Index, Processor;

  /* up to the last but one element it is possible to check eight processors
   * simultaneously, because if at least one is active, the value is not null
   */
  for (Index = 0; Index < Config.ActiveStack->ElemSize - 1; Index++) {
    if (Config.ActiveStack->StackElem->Elem[Index] != 0) {
      return TRUE;
    }
  }
  /* the remaining prozessors are checked one after another */
  for (Processor = ((LEN(Config) - 1) / 8) * 8; Processor < LEN(Config);
       Processor++) {
    if (IS_ACTIVE(Config, Processor)) {
      return TRUE;
    }
  }
  return FALSE;
}

/* Functions for else stack ----------------------------------------------- */

void INIT_ELSE_STACK
#ifdef __STDC__
(CONFIGURATION *Config)
#else
(Config) CONFIGURATION *Config;
#endif
/* Initialises the else stack */
{
  Config->ElseStack = (ELSE_STACK_PTR) malloc(sizeof(ELSE_STACK));
  /* assuming eight bits per byte */
  Config->ElseStack->ElemSize = (LEN(*Config) + 7) / 8;
  Config->ElseStack->StackElem = NULL;
}

void FREE_ELSE_STACK
#ifdef __STDC__
(CONFIGURATION *Config)
#else
(Config) CONFIGURATION *Config;
#endif
/* Frees the else stack. */
{
  while (Config->ElseStack->StackElem != NULL) {
    POP_ELSE(*Config);
  }
  free(Config->ElseStack);
}

void PUSH_ELSE
#ifdef __STDC__
(CONFIGURATION Config)
#else
(Config) CONFIGURATION Config;
#endif
/* Push the actual else status and copies the actual active status. */
{
  BOOLEAN_ELEM_PTR StackElem;

  StackElem = (BOOLEAN_ELEM_PTR) malloc(sizeof(BOOLEAN_ELEM));
  StackElem->Elem = (CHAR *) malloc(Config.ElseStack->ElemSize);
  /* assuming eight bits per char, bits are copied from the active stack */
  (void) memcpy(StackElem->Elem, Config.ActiveStack->StackElem->Elem,
    Config.ElseStack->ElemSize);
  StackElem->Previous = Config.ElseStack->StackElem;
  Config.ElseStack->StackElem = StackElem;
}

void POP_ELSE
#ifdef __STDC__
(CONFIGURATION Config)
#else
(Config) CONFIGURATION Config;
#endif
/* Pop the previous else status. */
{
  BOOLEAN_ELEM_PTR StackElem;

  StackElem = Config.ElseStack->StackElem;
  Config.ElseStack->StackElem = StackElem->Previous;
  free(StackElem->Elem);
  free(StackElem);
}

BOOLEAN_ELEM_PTR READ_ELSE
#ifdef __STDC__
(CONFIGURATION Config)
#else
(Config) CONFIGURATION Config;
#endif
/* Reads the pointer to the actual else stack element */
{
  return Config.ElseStack->StackElem;
}

void RESET_ELSE
#ifdef __STDC__
(CONFIGURATION Config, BOOLEAN_ELEM_PTR Pointer)
#else
(Config, Pointer) CONFIGURATION Config; BOOLEAN_ELEM_PTR Pointer;
#endif
/* Resets the else stack to the element with the pointer */
{
  while (Config.ElseStack->StackElem != Pointer) {
    POP_ACTIVE(Config);
  }
}

void COPY_ELSE_TO_ACTIVE
#ifdef __STDC__
(CONFIGURATION Config)
#else
(Config) CONFIGURATION Config;
#endif
/* Copy the status in the else stack into the active stack. */
{
  /* assuming eight bits per char, bits are copied from the else stack */
  (void) memcpy(Config.ActiveStack->StackElem->Elem,
    Config.ElseStack->StackElem->Elem, Config.ElseStack->ElemSize);
}

void UNSET_ELSE
#ifdef __STDC__
(CONFIGURATION Config, CARDINAL Processor)
#else
(Config, Processor) CONFIGURATION Config; CARDINAL Processor; 
#endif
/* Sets a processor in the range from 0 to LEN(Config) - 1 for the */
/* ELSE-part of a CASE-statement to inactive. */
{
  Config.ElseStack->StackElem->Elem[Processor / 8] &=
    (~(1 << (Processor % 8)));
}


/* Main program *************************************************************/

/* The main()-function is generated in the program module itself. Directly */
/* after the start of main() there's called the function INIT_MODULE() to */
/* initialize some variables, e.g. the argument pointers. Then the code of */
/* the parallaxis program is executed, and before the end of the main() */
/* function, the function EXIT_MODULE() is called to free some memory. */

unsigned int SYSTEM_argc;
char **SYSTEM_argv;
char **SYSTEM_envp;

FILE *SYSTEM_stdin;
FILE *SYSTEM_stdout;

BOOLEAN Done;
CHAR termCH;

CARDINAL PE;
INTEGER *SOURCE_ARRAY;
INTEGER *DESTINATION_ARRAY;
CONFIGURATION CONFIG_1;

GENERIC_CONNECTIONS *ALL_GENERIC_CONNECTIONS;
GENERIC_PARAMETER_CONNECTIONS *ALL_GENERIC_PARAMETER_CONNECTIONS;



void INIT_MODULE
#ifdef __STDC__
(int argc, char *argv[], char *envp[])
#else
(argc, argv, envp)
int argc;
char *argv[], *envp[];
#endif
{
  SYSTEM_argc = argc - 1;
  SYSTEM_argv = argv;
  SYSTEM_envp = envp;

  SYSTEM_stdin = stdin;
  SYSTEM_stdout = stdout;

  INIT_CONFIGURATION("CONFIG_1", &CONFIG_1, 1);
  SET_BOUNDS(&CONFIG_1, 0, 0, 0);
  INIT_STACKS(&CONFIG_1);

  ALL_GENERIC_CONNECTIONS = NULL;
  ALL_GENERIC_PARAMETER_CONNECTIONS = NULL;

  srand48(time(NULL)); /* init random number generator */
}

void EXIT_MODULE
#ifdef __STDC__
(void)
#else
()
#endif
{
/*  FREE_ALL_GENERIC_CONNECTIONS();
  FREE_ALL_GENERIC_PARAMETER_CONNECTIONS();

  FREE_CONFIGURATION(&CONFIG_1, TRUE); */

  exit(0);
}
