/********************************************************************************
 *  Projektname		: AERO
 *  Filename		: gleichungsloeser.c
 *  Filetyp		: C-Source
 ********************************************************************************
 *  Modulname		: gleichungsloeser.o
 *  letzte Aenderung	: Dienstag, 27. August 1996, 17:47:10
 *  Autor		: Horst Stolz (HUS)
 *  Status:		: testet (lcptest.c)
 *  
 *  Beschreibung:
 *    Dieses Modul stellt Funktion(en) zur Loesung von Linearen Gleichungs-
 *    systemen bereit.
 *
 *  Exportierte Funktionen:
 *    TBoolean LinGleichLoeser(int n, TReal *a, TReal *x)
 *
 *  Noch zu machen:
 *
 *  Versionsgeschichte:
 *    05.03.93       LexicMinRatioTest in LCP eingebaut -> keine Endloschleifen
 *  
 ********************************************************************************/


#include <stdio.h>
#include <stdlib.h>

#include "fehler.h"
#include "grundtypen.h"
#include "gleichungsloeser.h"


int gl_debug_level = 0;


/* Matrixgroessen: a [n][n+1], x[n]
 *
 */



#define A(y,x) a[(y)*(n+1)+(x)]


TBoolean LinGleichLoeser(int n, TReal *a, TReal *x)
{
    int i, j, k, max;
    TReal t, p, q;

    /* Gauss-Elimination mit partiellem pivotieren
     */

    /*
     * Elimination
     */

    for (i=0; i<n;  i++) {
	max = i;

#ifndef NODEBUG
	if (gl_debug_level>=2) {
	    int i, j;
	    printf("LinGl-Matrix, Pivot-Spalte %d\n", max);
	    for (i=0; i<n; i++) {
	      for (j=0; j<=n; j++)
		printf("%+2.2f ", A(i,j));
	      printf("\n");
	  }
	}
#endif
        /* Groesstes Element in Spalte i bestimmen
	 */
	for (j=i+1; j<n; j++)
	  if (fabs(A(j,i)) > fabs(A(max,i)))
	    max = j;

#ifndef NODEBUG
	if (gl_debug_level>=1) printf("MaxElem(Spalte %d)=%f(Zeile %d)\n", i, A(max,i), max);
#endif

	/* tausche Zeile i mit Zeile des groesste Spalten-Elements
	 */ 
	for (k=i; k<=n; k++) {
	    t = A(i,k);
	    A(i,k) = A(max,k);
	    A(max,k) = t;
	}

	if (fabs(A(i,i)) <= 10* REAL_EPSILON ) {
	    A(i,i) = 0.0;   /* frei Waehlbar */
	    continue;
	}

	/* von alle darunterliegenden Zeilen Zeile i subtrahieren
	 * um Spalte i zu 0 zu machen
	 */
	p = A(i,i);                   /* Pivot-Element */
	for (j=i+1; j<n; j++) {
	    q = A(j,i)/p;
	    if (q!=0.0) 
	      for (k=n; k>i; k--)
		A(j,k) -= A(i,k)*q;
	}
    }

    /*
     * Substitution
     */
    for (j=n-1; j>=0; j--) {
	if (A(j,j)==0.0) {
	    if (fabs(A(j, n)) > 20*REAL_EPSILON) {
#ifndef NODEBUG
		if (gl_debug_level>=0)
		  printf("Abbruch wegen A(%d, %d) = %e <> 0.0(%e)\n",
			 j, n , A(j, n), 10*REAL_EPSILON);
#endif
		return TRUE;
	    }
	    A(j,j)=1.0;
	}
	t = 0.0;
	for (k=j+1; k<n; k++)
	  t += A(j,k)*x[k];
	x[j] = (A(j,n)-t)/A(j,j);
#ifndef NODEBUG
	if (gl_debug_level>=2) printf("x[%d] = %f\n", j, x[j]);
#endif
    }

    return FALSE;
}





/* Loesungs-Prozeduren fuer LCP 
 */



TReal *lcp_x = NULL;	/* Inverse I, Matrix M, Vektor Q ->Buch */
TReal *lcp_w = NULL;    /* Ergebnisvektoren w und z */
TReal *lcp_z = NULL;

int lcp_n = 0;
static lcp_maxn = 0;

static int  *lcp_b = NULL;	/* intern fuer Loesungsalgorithmus */
static TBoolean *lcp_s = NULL;  /* fuer lexico min test */


#ifndef NODEBUG
static void printx(int *b, TReal *x, int n)
{
    int i, j, nn;

    nn = 2*n+2;
    for (i=0; i<n; i++) {
	printf("%c%d(%d)   ", (b[i]>=n) ? 'z' : 'w', ((b[i]==2*n)? 0: (b[i] % n + 1)), b[i]);
	for (j=0; j<nn; j++)
	  printf("%+2.2f  ", x[j*n+i]);
	printf("\n");
    }
      
}
#endif

/* x - (2n+n)*n Matrix von w,z,z0,q
 * r - Pivotzeile  0<r<n
 * c - Pivotspalte 0<c<2n+1
 * n
 */
static void xadd (TReal *x, int r, int c, int n)
{
    int nn, i, j;
    TReal p, f;

    nn = 2*n+2;
    p = x[c*n+r];   /* Pivotelement != 0.0 */


    /* Pivotzeile zu allen anderen Zeilen dazuaddieren
     * mit Faktor - Pivot(zeile)/Pivotelement
     */
    for (i=0; i<n; i++)
      if (i != r) {
	  f = - x[c*n+i] / p;
	  for (j=0; j<nn; j++) {
	      x[j*n+i] += f * x[j*n+r];
	  }
	}
    /*
     * Pivotelement durch Pivotzeile teilen
     */

    for (j=0; j<nn; j++)
      x[j*n+r] /= p;
    
}




static int min_row(TReal *x, int c, int n)
/* Liefert Zeile(0..n-1) dessen q am kleinsten ist
 */
{
    int i;
    int min;
    int q;

    q = (2*n+1)*n;
    min = 0;

    /* Zeile mit kleinstem q waehlen
     */
    for (i=1; i<n; i++) {
	if (x[q+i] < x[q+min]) min = i;
    }

    return min;
}




static int lexico_min_ratio_row(int *b, TReal *x, int c, int n)
/* liefert die Zeilennummer(0..n-1)
 * deren Lexic-Ratio (q_der_Zeile / Pivot_Ele der Zeile)
 * am kleinsten ist. (Pivot_Ele muss > 0 sein!!)
 * Wenn Pivotspalte <= 0 ist der Rueckgabewert <0!!!! =>ray termination
 */
{
    int i, j, k;
    int min, amin;
    TReal min_ratio = 0.0; /* Initialisierung, um Warning zu vermeiden */
    TReal ratio;
    int q, z0;
    int jn, cn;

    z0 = 2*n;
    q = z0+1;
    cn = c*n;

    
    /* Alle Zeilen selektieren desses Pivotelement >0.0 ist */
    for (i=0; i<n; i++)
      if (x[cn+i]>0.0) lcp_s[i] = TRUE; else lcp_s[i] = FALSE;

    /* Als erstes die q-Spalte betrachten, Danach die B-Matrix-Spalten */
    /* Dabei beachten das z0 eingestreut ist */
    j = q;

    do {
	min = -1;
	amin = 0;

        if (b[j] == z0)
	  jn = z0 * n;
	else
	  jn = j * n;

       /* Minimum der Spalte bestimmen und alle Zeilen die >Min_Ratio sind
         * mit FALSE deselektieren
         */
	for (i=0; i<n; i++)
	  if (lcp_s[i]) {   /* Nur selektierte Zeilen betrachten */
	      ratio = x[jn+i]/x[cn+i];
	      if (min>=0) {
		  if (ratio<min_ratio) {
		      for (k=min; k<i; k++)
			if (lcp_s[k]) lcp_s[k] = FALSE;
		      min_ratio = ratio;
		      min = i;
		      amin = 1;
		  } 
		  else if (ratio==min_ratio) {
		      amin++;
		  }
		  else lcp_s[i] = FALSE;   /* ratio>min_ratio */
	      }
	      else {
		  min_ratio = ratio;
		  min = i;
		  amin = 1;
	      }
	  }
	
	if (j == q) j=0;
	else j++;
	
    } while(amin>1 && j<n);

#ifndef NODEBUG
    if (gl_debug_level>=2)     printf("min_ratio=%e\n", min_ratio);
#endif

    return min;
}




static int min_ratio_row(TReal *x, int c, int n)
/* liefert die Zeilennummer(0..n-1)
 * deren Ratio (q_der_Zeile / Pivot_Ele der Zeile)
 * am kleinsten ist. (Pivot_Ele muss > 0 sein!!)
 * Wenn Pivotspalte <= 0 ist der Rueckgabewert <0!!!! =>ray termination
 */
{
    int i;
    int min;
    TReal ratio;
    TReal min_ratio = 0.0; /* Initialisierung, um Warning zu vermeiden */
    int q;

    q = 2*n+1;
    min = -1;

    for (i=0; i<n; i++)
	if (x[c*n+i]>0.0) {
	    ratio = x[q*n+i]/x[c*n+i];
	    if (min>=0) {
		if (ratio<min_ratio) {
		    min_ratio = ratio;
		    min = i;
		}
	    }
	    else {
		min_ratio = ratio;
		min = i;
	    }
	}

#ifndef NODEBUG
    if (gl_debug_level>=2)     printf("min_ratio=%e\n", min_ratio);
#endif

    return min;
}





void LCP_Init(int n) 
{
    int i, j;


    FehlerOrt("LCP_Init()");

    if (n<=0) Fehler("Nichts zum loesen?!?");

    lcp_n = n;
    if (lcp_n > lcp_maxn) {
	if (lcp_maxn != 0) LCP_Free();
	lcp_maxn = lcp_n;

	lcp_x = (TReal *)malloc((n*2+2)*n*sizeof(TReal));
	lcp_w = (TReal *)malloc(n*sizeof(TReal));
	lcp_z = (TReal *)malloc(n*sizeof(TReal));
	lcp_b = (int *)malloc(n*sizeof(int));
	lcp_b = (int *)malloc(n*sizeof(int));
	lcp_s = (TBoolean *)malloc(n*sizeof(TBoolean));

	if (lcp_x==NULL || lcp_w==NULL || lcp_z==NULL || lcp_b==NULL || lcp_s==NULL)
	    Fehler("Kann nicht genuegend Speicher allokieren");
    }

    for (i=0; i<n; i++) {
	LCP_SET_Q(i, 0.0);
	lcp_x[(2*n)*n +i] = -1.0;
	lcp_b[i] = i;

	for (j=0; j<n; j++) {
	    lcp_x[j*n +i] = (i==j)? 1.0 : 0.0;
	    LCP_SET_M(i, j, 0.0);
	}
    }
}



void LCP_Free()
{
    if (lcp_x) free(lcp_x);
    if (lcp_w) free(lcp_w);
    if (lcp_z) free(lcp_z);
    if (lcp_b) free(lcp_b);
    if (lcp_s) free(lcp_s);

    lcp_x = NULL;
    lcp_w = NULL;
    lcp_z = NULL;
    lcp_b = NULL;
    lcp_s = NULL;
}



TBoolean LCP_Loeser()
{
    int i, j;
    int c, r;

    FehlerOrt("LCP_Loeser()");


#ifndef NODEBUG
    if (gl_debug_level>=2) printx(lcp_b, lcp_x, lcp_n);
#endif

    c = 2*lcp_n;

    r = min_row(lcp_x, c, lcp_n);
/*
    r = lexico_min_ratio_row(lcp_b, lcp_x, c, lcp_n);
*/

    do {
#ifndef NODEBUG
	if (gl_debug_level>=1) {
	printf("Pivot: Zeile=%d, Spalte=%d\n", r, c); 
	printf("xadd()\n");
	}
#endif
	xadd(lcp_x, r, c, lcp_n);

	i = lcp_b[r];	
	lcp_b[r] = c;
	if (i<lcp_n)	/* umgekehrt var nehmen */
	  c = i+lcp_n;	/* statt z w und umgekehrt! */
	else		/* COMPLEMENTARY PIVOT !! */
	   c = i-lcp_n;

#ifndef NODEBUG
	if (gl_debug_level>=2) printx(lcp_b, lcp_x, lcp_n);
#endif
	if (i == 2*lcp_n) break;   /* z0 aus basic vektor entfernt */

	r = lexico_min_ratio_row(lcp_b, lcp_x, c, lcp_n);
/*
	r = min_ratio_row(lcp_x, c, lcp_n);
*/
	if (r<0) {
#ifndef NODEBUG
	    if (gl_debug_level>=1) printf("LCP - Ray-termination(nicht loesbar\n");
#endif
	    return TRUE;	/* Ray-Termination => nicht loesbar */
	}


    } while(1);


    for (i=0; i<lcp_n; i++) {
	lcp_w[i] = 0.0;
	lcp_z[i] = 0.0;
    }

    /* Ergebnis nach w und z schreiben
     */
    j = 2*lcp_n+1;
    for (i=0; i<lcp_n; i++) {
	r = lcp_b[i];
	if (r<lcp_n) lcp_w[r] = lcp_x[j*lcp_n +i];
	else lcp_z[r-lcp_n] = lcp_x[j*lcp_n +i];
    }

#ifndef NODEBUG
    if (gl_debug_level>=1) {
    for (i=0; i<lcp_n; i++) {
	printf("w[%d] = %f, z[%d] = %f\n", i, lcp_w[i], i, lcp_z[i]);
    }
    }
#endif

    return FALSE;  /* Loesung erfolgreich gefunden! */
}




/* Matrixgroessen: a [n][m], x[n] mit m>=n
 * n rows, m columns
 */
#undef A
#define A(y,x) a[(y)*(m)+(x)]


int linear_equation_solver(int n, int m, double *a /*, double *x */)
{
    int i, j, k, l, max;
    double t, p, q;

    /* Gauss-Elimination mit partiellem pivotieren
     */

    /*
     * Elimination
     */

    for (i=0; i<n;  i++) {
	max = i;

#ifdef DEBUG
	{
	    int i, j;
	    printf("LinGl-Matrix, Pivot-Spalte %d\n", max);
	    for (i=0; i<n; i++) {
	      for (j=0; j<m; j++)
		printf("%+2.2f ", A(i,j));
	      printf("\n");
	  }
	}
#endif
        /* Groesstes Element in Spalte i bestimmen
	 */
	for (j=i+1; j<n; j++)
	  if (fabs(A(j,i)) > fabs(A(max,i)))
	    max = j;

#ifdef DEBUG
	printf("MaxElem(Spalte %d)=%f(Zeile %d)\n", i, A(max,i), max);
#endif

	/* tausche Zeile i mit Zeile des groesste Spalten-Elements
	 */
	for (k=i; k<m; k++) {
	    t = A(i,k);
	    A(i,k) = A(max,k);
	    A(max,k) = t;
	}

	if (fabs(A(i,i)) <= 10 * DBL_EPSILON ) {
	    A(i,i) = 0.0;   /* frei Waehlbar */
	    continue;
	}

	/* von alle darunterliegenden Zeilen Zeile i subtrahieren
	 * um Spalte i zu 0 zu machen
	 */
	p = A(i,i);                   /* Pivot-Element */
	for (j=i+1; j<n; j++) {
	    q = A(j,i)/p;
	    if (q!=0.0)
	      for (k=m-1; k>i; k--)
		A(j,k) -= A(i,k)*q;
	}
    }

    /*
     * Substitution
     */
    for (j=n-1; j>=0; j--) {
	if (A(j,j)==0.0) {
	    if (fabs(A(j, n)) > 20*DBL_EPSILON) {
#ifdef DEBUG
		printf("Abbruch wegen A(%d, %d) = %e <> 0.0(%e)\n",
			 j, n , A(j, n), 20*DBL_EPSILON);
#endif
		return TRUE;
	    }
	    A(j,j)=1.0;
	}

	for (l=n; l<m; l++) {
 	    t = 0.0;
	    for (k=j+1; k<n; k++)
	       t += A(j,k)*A(k,l);

	    A(j,l) = (A(j,l)-t)/A(j,j);
	}
/*
	x[j] = (A(j,n)-t)/A(j,j);
*/
    }

    return FALSE;
}

/*
int main(int argc, char **argv)
{
    double *matrix;
    int rows, cols, row, col;

    if (argc<3) {
	fprintf(stderr, "USAGE: %s <cols> <rows> [varnames..] <matrix\n", argv[0]);
	exit(10);
    }

    cols = atoi(argv[1]);
    rows = atoi(argv[2]);
    if (rows < 1 || cols < rows) {
	fprintf(stderr, "*** argument error or not enough columns\n");
	exit(10);
    }

    matrix = malloc(sizeof(double) * rows * (cols==rows ? cols+1 : cols));


    for (row=0; row<rows; row++)
	for (col=0; col<cols; col++) {
	    if (1 != scanf("%lf", &matrix[row*cols+col])) {
		fprintf(stderr, "*** not enough datas in the %d x %d matrix\n",
				cols, rows);
		exit(10);
            }
    }

    if (cols==rows) {
	cols++;
	for (row=0; row<rows; row++)
            matrix[row*cols+cols-1] = 0.0;
    }

    if (linear_equation_solver(rows, cols, matrix)) {
	fprintf(stderr, "*** no solution found\n");
	exit(5);
    }

    for (row=0; row<rows; row++) {

	if (argc-3-row > 0)
	    printf("%s = ", argv[3+row]);

	for (col = rows; col < cols; col++)
	    printf("%+01.010e%s", matrix[row*cols+col], (col+1==cols)? "\n" : " ");
    }

    return 0;
}
*/
