/********************************************************************************
 *  Projektname		: AERO
 *  Filename		: integration.c
 *  Filetyp		: C-Source
 ********************************************************************************
 *  Modulname		: integration.o
 *  Version           	: 0.1.2
 *  letzte Aenderung	: 20.12.92
 *  Autor		: Horst Stolz (HUS)
 *  Status:		: lauffaehig
 *  
 *  Beschreibung:
 *    Aufbauend auf dem Grundtyp TReal liefert dieses Modul
 *    Numerische Integration nach einem eingebetteten RK-Verfahren mit
 *    Fehlersteuerung.
 *  
 *  Benutzung:
 *    1) mit IInit(numdgls) wird die Anzahl der Diffgls. festgelegt die integriert
 *       werden sollen. Als Rueckgabewert erhaelt man ein Feld in das die aktuellen
 *       y einzutragen/auszulesen sind.
 *    2) Mit INach...
 *
 *  
 *  Noch zu machen:
 *    Schrittweitensteuerung+Interpolation geht noch nicht richtig
 *
 *  Versionsgeschichte:
 *    20.12.92    RK4(4)7FM eingebaut => Genauigkeitsgewinn
 *  
 ********************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include "grundtypen.h"
#include "fehler.h"
#include "integration.h"
#include <setjmp.h>


int i_debug_level = 0;
static TBoolean schrittsteuerung = FALSE;

#define SMAX 7

static int ng = 0;       /* Anzahl von Diffgln. die zu integrieren sind */
static int maxng = 0;    /* Aufgrund der derzeitigen Speicherbelegung */
                         /* max. moegliche Gleichungsanzahl ohne den  */
                         /* alten Speicher freizugeben und neuen zu belegen */

static TReal dh = 0.001; /* Zeitschrittweite */
static TReal dh_min = 0.0000001;	/* Schrittweitengrenzen */
static TReal dh_max = 1.0;		/* bei Schrittsteuerung */

static TReal eps = 0.0001;   /* Genauigkeit pro Schritt */

static void (*f)(TReal *, TReal *, TReal);	/* Funktionsauswertungsroutine */
static void (*fi)(TReal *ytemp, TReal t);	/* Interpol.nachbearbeitung */
static void (*fischritt)(TReal t);              /* Aufruf bei ISchritt (vor f()) */
static void (*fischrittwieder)(TReal t);        


static TReal *y = 0;     /* Anfangsvektor y[maxng] davon nur ng benutzt */
static TReal *yneu = 0;  /* Ergebnisvektor von ISchritt */
static TReal *yipol = 0; /* Ergebnisvektor der Interpolation */
static TReal *ytemp = 0; /* ytemp[maxng] */
static TReal *k = 0;     /* k[s * maxng] */
static TReal *aa = 0;    /* aa[maxng * (d+1) - Zwischenspeicher fuer Intepolation */

static int p;                /* Ordnung des Verfahrens */
static TBoolean fsal;	/* First Same As Last - manche Integrationsverfahren erlauben
                         * es statt das erste Element k[0] neu zu berechnen die
                         * werte des letzten k[s-1] zu nehmen (fuer Integrationsschritt)
                         */
static TReal potenz;

/* Koeffizienten fuer eingebettetes Runga-Kutta-Verfahren */
static int s = 4;        /* Zahl der Zwischenschritte */
static TReal a[SMAX][SMAX];    
static TReal c[SMAX];       
static TReal b[SMAX];
static TReal b2[SMAX];
/* 
 * Interpolationskoeffizienten
 */
static TReal beta[7][7];
static int d = 5;

static TReal dh_last;               /* Schrittweite des letzten ISchritt() */
static TReal t_last_start = 0.0;    /* Start-Zeit des letzten ISchritt() */
static TReal tschritt = - REAL_MAX; /* (End-)Zeit des letzten ISchritt() */
/* Schrittweite und (Uhr)Zeit des letzten Integrationschrittes (ISchritt)
 */

static TBoolean kinit = FALSE;

/* 
 */
TReal *ystart = 0;
TReal *yschritt = 0;
TReal *youtput = 0;


/*
 * Statistische Daten die waehrend der Integration gesammelt werden
 */

static long nfail = 0;		/* Anzahl zu ungenauer Integrationsversuche */
static long nstep = 0;		/* Anzahl ausgefuehrter Integrationsschritte (ISchritt) */
static long nipol = 0;		/* Anzahl der Aufrufe der Interpolationsroutine */
static long nfunk = 0;		/* Anzahl Funktionsaufrufe */


static TBoolean istoppen = FALSE;


static jmp_buf jump_icont;


static TBoolean  neue_poly_noetig = TRUE;


void KoefInit_RK5_4_7FM()
/*
 * RK5(4)7FM von Dormand und Prince
 */ 
{
    s = 7;
    p = 4;
    d = 5;
    potenz = 1.0 / (p + 1.0);
    fsal = TRUE;

    c[0] = 0.0;
    c[1] = 1.0 / 5.0;
    c[2] = 3.0 / 10.0;
    c[3] = 4.0 / 5.0;
    c[4] = 8.0 / 9.0;
    c[5] = 1.0;
    c[6] = 1.0;
    
    a[1][0] =       1.0 / 5.0;
    a[2][0] =       3.0 / 40.0;
    a[2][1] =       9.0 / 40.0;
    a[3][0] =      44.0 / 45.0;
    a[3][1] =    - 56.0 / 15.0;
    a[3][2] =      32.0 / 9.0;
    a[4][0] =   19372.0 / 6561.0;
    a[4][1] =   - 25360 / 2187.0;
    a[4][2] =   64448.0 / 6561.0;
    a[4][3] =   - 212.0 / 729.0;
    a[5][0] =    9017.0 / 3168.0;
    a[5][1] =   - 355.0 / 33.0;
    a[5][2] =   46732.0 / 5247.0;
    a[5][3] =      49.0 / 176.0;
    a[5][4] =  - 5103.0 / 18656.0;
    a[6][0] =      35.0 / 384.0;
    a[6][1] =       0.0;
    a[6][2] =     500.0 / 1113.0;
    a[6][3] =     125.0 / 192.0;
    a[6][4] =  - 2187.0 / 6784.0;
    a[6][5] =      11.0 / 84.0;
    
    b[0] =     35.0 / 384.0;
    b[1] = 0.0;
    b[2] =    500.0 / 1113.0;
    b[3] =    125.0 / 192.0;
    b[4] = - 2187.0 / 6784.0;
    b[5] =     11.0 / 84.0;
    b[6] = 0.0;

    b2[0] =    5179.0 / 57600;
    b2[1] = 0.0;
    b2[2] =    7571.0 / 16695.0;
    b2[3] =     393.0 / 640.0;
    b2[4] = - 92097.0 / 339200.0;
    b2[5] =     187.0 / 2100.0;
    b2[6] =       1.0 / 40.0;
    
    beta[0][0] = 1.0;
    beta[1][0] = - 4034104133.0 / 1410260304.0;
    beta[2][0] = 105330401.0 / 33982176.0;
    beta[3][0] =  - 13107642775.0 / 11282082432.0;
    beta[4][0] = 6542295.0 / 470086768.0;
    beta[0][1] = 0.0;
    beta[1][1] = 0.0;
    beta[2][1] = 0.0;
    beta[3][1] = 0.0;
    beta[4][1] = 0.0;
    beta[0][2] = 0.0;
    beta[1][2] = 132343189600.0 / 32700410799.0;
    beta[2][2] = - 833316000.0 / 131326951.0;
    beta[3][2] = 91412856700.0 / 32700410799.0;
    beta[4][2] = - 523383600.0 / 10900136933.0;
    beta[0][3] = 0.0;
    beta[1][3] = - 115792950.0 / 29380423.0;
    beta[2][3] = 185270875.0 / 16991088.0;
    beta[3][3] = - 12653452475.0 / 1880347072.0;
    beta[4][3] = 98134425.0 / 235043384.0;
    beta[0][4] = 0.0;
    beta[1][4] = 70805911779.0 / 24914598704.0;
    beta[2][4] = - 4531260609.0 / 600351776.0;
    beta[3][4] = 988140236175.0 / 199316789632.0; 
    beta[4][4] = - 14307999165.0 / 24914598704.0;
    beta[0][5] = 0.0;
    beta[1][5] = - 331320693.0 / 205662961.0;
    beta[2][5] = 31361737.0 / 7433601.0;
    beta[3][5] = - 2426908385.0 / 822651844.0;
    beta[4][5] = 97305120.0 / 205662961.0;
    beta[0][6] = 0.0;
    beta[1][6] = 44764047.0 / 29380423.0;
    beta[2][6] = - 1532549.0 / 353981.0;
    beta[3][6] = 90730570.0 / 29380423.0;
    beta[4][6] = - 8293050.0 / 29380423.0;
}


void KoeffInit_RK4()
{
/* Klassisches Runga-Kutta-Verfahren */

    FehlerOrt("KoeffInit()");
    
    s = 4;
    p = 2;
    potenz = 1.0 / (p + 1.0);
    fsal = FALSE;

    c[0] = 0.0;
    c[1] = 0.5;
    c[2] = 0.5;
    c[3] = 1.0;
    a[1][0] = 0.5;
    a[2][0] = 0.0;
    a[2][1] = 0.5;
    a[3][0] = 0.0;
    a[3][1] = 0.0;
    a[3][2] = 1.0;
    b[0] = b2[0] = 1.0 / 6.0;
    b[1] = b2[1] = 2.0 / 6.0;
    b[2] = b2[2] = 2.0 / 6.0;
    b[3] = b2[3] = 1.0 / 6.0;
}


void KoeffInit()
{
    TReal sb, sb2, sbeta, sc, diff;
    int i, j;

/*    KoeffInit_RK4(); erlaubt keine Interpolation -> beta nicht gesetzt! */
    KoefInit_RK5_4_7FM();
    potenz = 1.0 / (p + 1.0);
    tschritt = 0.0;
    /* stellt sicher das bei ersten Aufruf von INach(t=0)  ISchritt() aufgerufen wird u.
     * der Funktionswert y(t) berechnet wird und ansonsten wenn moeglich k[0]=k[s-1]
     */
    kinit = FALSE;

/*
    printf("real_epsilon=%le\n", REAL_EPSILON);
*/
    
    sb = 0.0;
    sb2 = 0.0;
    for (i=0; i<s; i++) {
	sb += b[i];
	sb2 += b2[i];
	
	sc = 0.0;
	for (j=0; j<i; j++)
	    sc += a[i][j];

	sbeta = 0.0;
	for (j=0; j<d; j++)
	    sbeta += beta[j][i];

	if ((diff=fabs(sc-c[i])) > 10*REAL_EPSILON)
	    printf("c[%d] falsch, %e != %e, diff=%e\n", i, sc, c[i], diff);

	if ((diff=fabs(sbeta-b[i])) > 10*REAL_EPSILON)
	    printf("beta[][%d] falsch, beta[][%d]=%e, b%d=%e, diff=%e\n",
				   i, i, sbeta, i, b[i], diff);
    }

    if ((diff=fabs(sb-1.0)) > 10*REAL_EPSILON)
	printf("b falsch, SUM(b)=%e, diff=%e\n", sb, diff);
    if ((diff=fabs(sb2-1.0)) > 10*REAL_EPSILON)
	printf("b2 falsch, SUM(b2)=%e, diff=%e\n", sb2, diff);
}


TReal *IErg()
{
    FehlerOrt("IErg()");

    return youtput;
}


TReal *IInit(int numdgls, void (*funk)(TReal *, TReal *, TReal), 
             void (*inach)(TReal *, TReal), void (*fschritt)(TReal ),
	     void (*fiswieder)(TReal))
{

    FehlerOrt("IInit() - Integrationsverfahren-Initialisierung");

    f = funk;
    fi = inach;
    fischritt = fschritt;
    fischrittwieder = fiswieder;

    ng = numdgls;
    kinit  = FALSE;

    if (maxng == 0) {
	KoeffInit();
    }

    if (ng > maxng) {
	if (y) free(y);
	if (y) free(yneu);
	if (y) free(yipol);
	if (k) free(k);
	if (ytemp) free(ytemp);
	if (aa) free(aa);

	y      = (TReal *)malloc(ng * sizeof(TReal));
	yneu   = (TReal *)malloc(ng * sizeof(TReal));
	yipol  = (TReal *)malloc(ng * sizeof(TReal));
	ytemp  = (TReal *)malloc(ng * sizeof(TReal));
	k      = (TReal *)malloc(s * ng * sizeof(TReal));
	aa     = (TReal *)malloc((d+1) * ng * sizeof(TReal));
 
	if (y == NULL || yneu == NULL || yipol == NULL ||
	    ytemp == NULL || k == NULL || aa == NULL)
	  Fehler("Kann nicht genuegend Speicher allozieren");

	maxng = ng;
    }

    yschritt = yneu;

    return (ystart = youtput = y);
}


TReal ISchritt(TReal t)
/****************************************************************************
 * Zweck:
 *   Fuehrt ein Integrationsschritt mit einem eingebettetem RK-Verfahren
 *   durch. ystart(t) -> yschritt(Rueckgabewert von ISchritt)
 *   
 * Parameter:
 *   TReal t   ... Startzeit des Integrationsschrittes
 *   
 * Rueckgabewert:
 *   TReal     ... Neu Zeit nach der Integration
 *
 * Benutzte Variablen:
 *   Lesend     : ystart - Startvektor
 *                
 *   Aufrufend  : f - Funktionsauswertungsfunktion
 *   Veraendernd: k - speicherung der RK-Zwischenwerte,
 *                    stehen danach fuer Interpolation zur Verfuegung!
 *                ytemp - Zwischenvektor 
 *   Schreibend : yschritt - Nach Aufruf steht hier der Ergebnisvektor!
 *
 * Beispiel:
 *   
 * Siehe weiter:
 *   
 * Fehler:
 *   
 */

{
    TReal h, yy, ydiff;    /* Hilfvariable */
    int n, i, j;           /* Schleifenvars */
    TReal err, hfakt;
    int zu_ungenau = FALSE;
    int ifail = 0;


    FehlerOrt("Integration()");
 
    fischritt(t);

/*
    printf("ystart(%f): ", t);
    for (n=0; n<ng; n++)
      printf("%f ", ystart[n]);
    printf("\n");
*/
    if (i_debug_level)
	printf("Integration(%e)\n", t);

    /* UBERNEHMEN VON k[s-1] DES LETZTEN SCHRITTES NACH k[0] FALLS MOEGLICH:
     * Wenn die Zeitdifferenz zwischen 
     * dem k[s-1] vom letzten Schritt(tschritt) mit der neuen Zeit(t) des k[0]
     * vom diesem Integrationsschritt nicht zu GROSS ist und fsal (First Same As Last)
     * erlaubt ist so kopiere die Werte von k[s-1] nach k[0], andernfalls werden
     * die Funktionswerte von k[0] - Zeit t berechnet.
     */
    if (!kinit || !fsal) { 
	if (i_debug_level)
	    printf("  Neues k[0] ausrechnen\n");

	(*f)(k, ystart, t);
	nfunk++;
    	kinit = TRUE;
    }
    else {
	if (i_debug_level)
	    printf("  Alte k[s-1] nach k[0] kopieren\n");

	memcpy(k, &k[(s-1)*ng], ng * sizeof(TReal));
    }


    /* Wiederhole solange den Integrationsschritt bis die Genauigkeit(EPS) stimmt!
     * (Bem: die Werte fuer k[0]  bleiben natuerlich gleich da sich t von k[0] sich
     * nicht aendert.)
     */
    do {
	zu_ungenau = FALSE;

	if (i_debug_level) {
	    printf("  k[1,..,s-1] ausrechnen\n");
/*
	    for (i=0; i<ng; i++)
		printf("ystart[%d] = %lf\n", i, ystart[i]);
*/
	}
        /* Zwischenwerte k[1] .. k[s-1] berechnen
         */
	for (i=1; i<s; i++) {             /* k0 ... k(s-1) berechnen fur ein y */
	    for (n=0; n<ng; n++) { /* Schleife ueber all Dgls */           
		h = 0.0;
		for (j=0; j<i; j++)
		  h += a[i][j] * k[j*ng + n];
		ytemp[n] = ystart[n] + dh * h;
	    }
	    (*f)(&(k[i*ng]), ytemp, t+dh*c[i]);  
	    nfunk++;
	    /* von allen Dgls. */
	}



	/* Ergebnis y aus k[0]..k[s-1] zusammenrechnen.
	 * Fehlerabschaetzung machen um Schrittweite anzupassen
	 */
	err = 0.0; 
	for (n=0; n<ng; n++) {          /* Schleife ueber alle Dgls */
	    ydiff = yy = 0.0;                   /* neues y(t+dh) bestimmen */
	    for (i=0; i<s; i++) {
		yy += b[i] * k[i*ng + n];
		ydiff += b2[i] * k[i*ng + n];
	    }
	    ydiff = fabs(dh * (ydiff-yy));
	    yschritt[n] = ystart[n] + dh * yy;
	    
	    err = (ydiff > err) ? ydiff : err;
	}

/*		
	if (i_debug_level) {
	    for (i=0; i<ng; i++)
		printf("yschritt[%d] = %lf\n", i, yschritt[i]);
	}
*/
	dh_last = dh;

        if (schrittsteuerung) {

	    /* Liegt die Fehlerabschaetzung unter der geforderten
             * Schrittgenauigkeit eps, so muss eine Wiederholung 
             * eingeleitet werden.
	     */
	    if ((zu_ungenau = (err>eps) && (dh>dh_min)))  ifail++;

	    if (i_debug_level)
		printf("err>eps?  %f>%f?\n", err, eps);

	    if (ifail>5) {
		/* Fehler("Schrittweise schon 5 mal erfolglos verkleinert"); 
		printf("***Integration: Schrittweise schon 5 mal erfolglos verkleinert\n");
		*/
		dh = dh_min;
		break;
	    }
	    else {
		/* neuer Schrittweitenfaktor bestimmen
		 */
		hfakt = 0.9 * pow(eps/err, potenz);
		
		/* Schrittweitenaenderung begrenzen
		 */
		if (hfakt < 0.1) hfakt = 0.1; 
		if (hfakt > 5.0) hfakt = 5.0; 
		
		
		/* Schrittweite darf bei zu geringer Genauigkeit nicht vergroessert werden
		 */
		if (!ifail || hfakt < 1.0) {
		    dh *= hfakt;
		    
		    
		    /* Schrittweite begrenzen zwischen dh_min <= dh <= dh_max
		     */
		    if (dh < dh_min) dh = dh_min;
		    else if (dh > dh_max) dh = dh_max;
		}
	    }
	}

	if (zu_ungenau) fischrittwieder(t);

    } while (zu_ungenau);

    if (zu_ungenau) {
/*
	printf("***integration: timesteprate not accurate\n");
	*/
    }

    if (i_debug_level)
	printf("altes dh=%f, neues dh=%f\n", dh_last, dh);

    nfail += ifail;		/* statistische Daten sammeln */
    nstep++;

    neue_poly_noetig = TRUE;
/*
    printf("yschritt: ");
    for (n=0; n<ng; n++)
      printf("%lf ", yschritt[n]);
    printf("\n");
*/

    return t + dh_last;
}




void IInterpolation(TReal t)
{
    int n, i, j;
    TReal r, sigma;


    FehlerOrt("IInterpolation()");
    nipol++;

    if (i_debug_level)
	printf("Interpolation(%f)\n", t);

    if (neue_poly_noetig) {
	/*
	 *
	 */
	/* Anfangswerte fuer Interpolation merken */
        /* alter startwert vom letztem ISchritt nehmen! */
	memcpy(aa, yschritt, ng*sizeof(TReal));
	
	if (i_debug_level) {
	    printf("  Neue Polynom-Koeffizienten aa[]\n"); 
	}
	
	for (n=0; n<ng; n++) {
	    for (i=1; i<=d; i++) {
		r = 0.0;
		for (j=0; j<s; j++)
		    r += beta[i-1][j]*k[n+j*ng];
		aa[n+i*ng] = r * dh_last;
	    }
	}

	neue_poly_noetig = FALSE;
    }


    sigma = (t - t_last_start) / dh_last;
    if (sigma < 0.0 || sigma > 1.0)
	Fehler("Interpolation ist ausserhalb des letzten Schrittes (sigma != 0..1)");

    if (i_debug_level)
	printf("  sigma=%f\n", sigma);

   
    /* Interpolation der Zwischenwerte durch ein Polynom
     * -> Horner Schema zum effizienten Ausrechnen
     */
    for (n=0; n<ng; n++) {
	yipol[n] = aa[n+d*ng];
	for (i=d-1; i>=0; i--)
	    yipol[n] = yipol[n] * sigma + aa[n+i*ng];
    }
}


void INach(TReal *t, TReal tend)
/**************************************************************************************
 * Integration des Zustandes von der Zeit t bis zur Zeit tend (tend >= t !)
 * In t steht danach tend.
 * Anfangswerte stehen im Vektor y, Funktionsaufruf mittels externer Funktion.
 * Ergebnis danach in youtput.
 * 
 */
{
    istoppen = FALSE;

    if (i_debug_level)
	printf("INach({von}%f,{nach}%f)\n", *t, tend); 

    if (!schrittsteuerung) {
	TReal dh0;

	/* ohne Schrittsteuerung (feste Schrittweite!)
	 */

        tschritt = *t;

	if (setjmp(jump_icont) == 0) {}
	else {
	}

	youtput = yschritt = ystart;

	while (tend - (tschritt+dh) > 0.0001*dh) {
	    t_last_start = tschritt;
	    tschritt = ISchritt(t_last_start);

	    if (istoppen) {		/* fruehzeitiges Abbrechen der Integration */
		*t = tschritt;
		return;	    
	    }
	}

	dh0 = dh;
	dh = tend - tschritt;
	tschritt = ISchritt(tschritt);
	if (istoppen) {
	    *t = tschritt;
	    return;	    
	}
	dh = dh0;
        *t = tend;

        return;
    }

    /* Fortlaufende Integration?
     */
    if (*t < t_last_start || !kinit) {
	tschritt = *t;
    }

    if (setjmp(jump_icont) == 0) {}
    else {
    }

    /* Schrittweise Integration bis das die geforderte Zeit tend innerhalb 
     * des letzten Integrationsschrittes liegt.
     */
    while (tend > tschritt) {
	t_last_start = tschritt;
	tschritt = ISchritt(tschritt);
	youtput = yschritt;
	yschritt = ystart;
	ystart = youtput;

	if (istoppen) {			/* fruehzeitiges Abbrechen der Integration */
	    youtput = ystart;
	    *t = tschritt;
	    return;
	}	
    }
	

    /* Ist die Verlangte Zeit innerhalb des letzten Integrationsschrittes? (tend<tschritt!)
     * Dann Interpoliere den Zustand zur Zeit tnext. 
     * Bei tend==tschritt ist keine Intepolation notwendig!
     */
    if (tend < tschritt) {
	IInterpolation(tend);
	youtput = yipol;
	fi(yipol, tend);
    }
    else if (tend == tschritt) {
	youtput = ystart;
    }

    *t = tend;
}


TReal IMinInterpolZeit()
{
    if (schrittsteuerung)
	return t_last_start;
    else 
	return tschritt;
}


void IStoppen()
/* Integration am naechsten moeglichen Zeitpunkt stoppen
 * Aufruf aus f(..) aus!!
 */
{
    istoppen = TRUE;
}


TReal IConfig(TConfig was, TReal r)
/* Setzte interne Parameter des Integrationsverfahrens
 */
{
    TReal h = 0.0;


    FehlerOrt("IConfig()");

    switch(was) {
      case CONSTSTEP:
	schrittsteuerung = FALSE;
	break;
      case STEPCONTROL:
	schrittsteuerung = TRUE;
	break;
      case EPS:
	h = eps;
	eps = r;
	break;
      case ISCHRITT:
	h = dh;
	dh = r;
	break;
      case ISCHRITT_MIN:
	h = dh_min;
	dh_min = r;
	break;
      case ISCHRITT_MAX:
	h = dh_max;
	dh_max = r;
	break;
      default:
	Fehler("unbekannte Configurationsnummer");
	break;
    }
    return h;
}



void IRuecksetzen(TReal t, TReal *ty)
/* Ruecksetzten des Integrationsverfahrens auf vorige Start-Zeit
 */
{
    int n;

    tschritt = t;
    


    if (i_debug_level>2) {
	printf("ty: ");
	for (n=0; n<ng; n++)
	  printf("%f ", ty[n]);
	printf("\n");
    }


    /*
     * Augenblickliche Ausgabe-Var von ISchritt als naechsten Startpunkt waehlen.
     */
    memcpy(ystart, ty, ng * sizeof(TReal));


    kinit = FALSE;
}





void INeustart(TReal t, TReal *ty)
/* Neustart des Integrationsverfahrens von Zeit t mit Anfangswerten ty
 */
{
    int n;

    tschritt = t;
    


    if (i_debug_level>2) {
	printf("ty: ");
	for (n=0; n<ng; n++)
	  printf("%f ", ty[n]);
	printf("\n");
    }


    /*
     * Augenblickliche Ausgabe-Var von ISchritt als naechsten Startpunkt waehlen.
     */
    memcpy(ystart, ty, ng * sizeof(TReal));


    kinit = FALSE;

    longjmp(jump_icont, 1);
}



void IStatistik(TIStatistik *istat)
/* Ausgabe der mitprotokollierten Statistischen Daten
 */
{
    if (istat) {
	istat->AnzFehlversuche     = nfail;
	istat->AnzSchritte         = nstep;
	istat->AnzInterpolAufrufe  = nipol;
	istat->AnzFunkAuswertungen = nfunk;
    }
    nfail = 0;
    nstep = 0;
    nipol = 0;
    nfunk = 0;
}


