/********************************************************************************
 *  Projektname:	AERO
 *  Filename:		invserv.c
 *  Filetyp:		Prozess
 ********************************************************************************
 *  WICHTIG:            Dieser Quelltext benoetigt einen Compiler, der Struktur-
 *                      Zuweisungen beherrscht (z.B. ANSI-C, gcc). Sind also z.B.
 *                      a und b Strukturen gleichen Typs, muss bei b=a; der ge-
 *                      samte Inhalt von a nach b kopiert werden.
 ********************************************************************************
 *  Modulname:		Inventor-Server
 *  Version:		1
 *  letzte Aenderung:	Donnerstag, 16. Maerz 1995, 02:05:13
 *  Autor:  		Hartmut 
 *  Status:		Test
 *
 *  imp. Bezeichner:
 *  ----------------
 *
 *  exp. Bezeichner:
 *  ----------------
 *                      
 *
 *  Beschreibung:
 *  -------------
 *  Server fuer Inventor. Extra Programm.
 *
 *  Fehler:
 *  -------
 *  
 *
 *  Versionsgeschichte:
 *  -------------------
 *  0:	Vorversion
 *  1:  Synchronisation eingefuegt; kommentiert
 *
 *   in Arbeit:
 *   ----------
 *   
 *   zu tun:
 *   -------
 *   - Synchronisierung von invserv und aero selbst.
 *   - Uebernahme der Kameraposition aus Inventor nach AERO ueber speziellen
 *     Button
 *   - Umsetzen des Aspect Ratio bei 3D-Modus
 *   - Setzen der Farben von Verbindungen
 *   - Umsortieren und dokumentieren der Funktionen
 *   - Erfinden von zwei kleinen Grafiken, die die Zustaende "Kamera von AERO"
 *     und "Kamera lokal" anzeigen. Binden an den extra Button. Bis jetzt: X
 *     und O.
 ********************************************************************************/

#include <stdio.h>			  /* e.g. fprintf() */
#include <stdlib.h>
#include <math.h>			  /* e.g. acos() */
#include <unistd.h>
#include <alloca.h>			  /* alloca() */

#include <X11/Intrinsic.h>		  /* basic declarations */
#include <Inventor_c/Xt/SoXt.h>
#include <Inventor_c/nodes/SoSeparator.h>
#include <Inventor_c/nodes/SoCoordinate3.h>

#include <Inventor_c/nodes/SoCube.h>	  /* shapes */
#include <Inventor_c/nodes/SoSphere.h>
#include <Inventor_c/nodes/SoCylinder.h>
#include <Inventor_c/nodes/SoIndexedFaceSet.h>

#include <Inventor_c/nodes/SoBaseColor.h> /* material */
#include <Inventor_c/nodes/SoMaterial.h>

#include <Inventor_c/nodes/SoRotationXYZ.h> /* transformations */
#include <Inventor_c/nodes/SoTranslation.h>
#include <Inventor_c/nodes/SoTransform.h>
#include <Inventor_c/nodes/SoScale.h>

#include <Inventor_c/fields/SoSFVec3f.h>  /* values */
#include <Inventor_c/fields/SoMFVec3f.h>
#include <Inventor_c/fields/SoSFRotation.h>
#include <Inventor_c/fields/SoMFFloat.h>
#include <Inventor_c/fields/SoMFLong.h>

#include <Inventor_c/nodes/SoPerspectiveCamera.h> /* viewer */
#include <Inventor_c/Xt/viewers/SoXtExaminerViewer.h>

#include <Xm/PushBG.h>			  /* PushButtonGadget */

#include "pvm3.h"			  /* PVM */

#include "inventor.h"			  /* shared definitions */

#define PI M_PI
#define EPSILON 1e-7
#define NEAR_0(x) (fabs(x) < EPSILON)

#undef INVDEBUG



/*------ Variables ---------------------------------------------------------*/

static Widget mainWindow;
static Widget cameraButton;        /* extra button for camera handling */
static SoSep  *root;		   /* root of object tree */
static SoXtExamVwr  *viewer;	   /* examiner viewer */
static XmString LabelX, LabelO;    /* strings for extra button labeling */

static int my_tid, parent_tid;     /* pvm task-identifiers */
static int syncflag = TRUE;	   /* run synchronously with AERO or not */

static int UseCamera = 1;	   /* start with camera linked to AERO */
static double campos[3], camq[4], camzoom; /* camera data received from AERO */
					   /* (this gets recorded even if */
					   /* only local camera is active) */

/*------ Functions --------------------------------------------------------*/

/*****************************************************************************
 *  Function:       void QuaternionFromVector(double v[3], double len, 
 *                                                             double q[4])
 *
 *  Parameters:     v:       target vector
 *                  len:     length of v
 *                     
 *  Result:         q:       quaternion which rotates (0,0,len) to v
 *
 *  Import. idents:
 *
 *  Description:
 *  ------------
 *  From the given vector v and its length len, there is computed a
 *  quaternion q which rotates the vector (0, 0, len), i.e. a vector along
 *  the z-axis, to the desired orientation v. This is used in calculating
 *  the orientation of a AERO link where only the two connection points are
 *  known.
 *****************************************************************************/

static void QuaternionFromVector(double v[3], double len, double q[4])
{
     if (NEAR_0(len))			  /* extremely short vector? */
     {					  /* use identity quaternion */
	  q[0] = 1;
	  q[1] = 0;
	  q[2] = 0;
	  q[3] = 0;
     }
     else if (NEAR_0(len+v[2]))		  /* v looking in -z direction? */
     {
	  q[0] = 0;			  /* the computation below would be */
	  q[1] = 1;			  /* ambiguous, so we simply decide */
	  q[2] = 0;			  /* to use a rotation about x-axis */
	  q[3] = 0;
     }
     else				  /* compute quaternion */
     {
	  q[0] = sqrt((v[2]+len)/(2*len));
	  q[1] = v[1]/sqrt(2*len*(v[2]+len));
	  q[2] = -v[0]/sqrt(2*len*(v[2]+len));
	  q[3] = 0;
     }
}     



void linkunpack(SoSep *sep)
{
     double p1[3], p2[3], q[4], len, length;
     SoXf *trans;
     double rotangle;
     SbVec3f axis;
     SoScale *scale;
    
     pvm_upkdouble(p1, 3, 1);
     pvm_upkdouble(p2, 3, 1);
				    
     p2[0]-=p1[0];
     p2[1]-=p1[1];
     p2[2]-=p1[2];
    
     length = sqrt(p2[0]*p2[0]+p2[1]*p2[1]+p2[2]*p2[2]);
    
     QuaternionFromVector(p2, length, q);
     trans = SoXfCreate();
     SoSV3fSetX_Y_Z(&(trans->translation), p1[0], p1[1], p1[2]);
     len = sqrt(q[1] * q[1] + q[2] * q[2] + q[3] * q[3]);
     if (len<EPSILON) len=1.0;
     q[1] /= len;
     q[2] /= len;
     q[3] /= len;
     SbV3fSetX_Y_Z(axis, q[1], q[2], q[3]);

     rotangle = -2.0 * acos(q[0]);
     SoSRotSetAxisAngle(&(trans->rotation), 
			&axis, rotangle);
   
     SoSepAddChild(sep, (SoNode *) trans);
   
     scale = SoScaleCreate();
     SoSV3fSetX_Y_Z(&(scale->scaleFactor), 1.0, 1.0, length);
     SoSepAddChild(sep, (SoNode *) scale); 
}

void unpackinfo(SoSep *sep)
{
     double pos[3], q[4];
     unsigned short xR, xG, xB;
     double R, G, B;
     SoXf *trans;
     SoMaterial *col;
     double len;
     double rotangle;
     SbVec3f axis;
   
     pvm_upkdouble(pos, 3, 1);
     pvm_upkdouble(q, 4, 1);
     trans = SoXfCreate();
     SoSV3fSetX_Y_Z(&(trans->translation), pos[0], pos[1], pos[2]);
     len = sqrt(q[1]*q[1] + q[2]*q[2] + q[3]*q[3]);
     if (len<1e-9) len=1.0;
     q[1] /= len;
     q[2] /= len;
     q[3] /= len;
     SbV3fSetX_Y_Z(axis, q[1], q[2], q[3]);

     rotangle = -2.0 * acos(q[0]);
     SoSRotSetAxisAngle(&(trans->rotation), 
			&axis, rotangle);
   
/*
   SoSRotSetQ_U_A_T(&(trans->rotation), q0, q1, q2, q3);
*/
     SoSepAddChild(sep,  (SoNode *) trans);
     pvm_upkushort(&xR, 1, 1);
     pvm_upkushort(&xG, 1, 1);
     pvm_upkushort(&xB, 1, 1);

     R = ((double)xR)/65535.0;
     G = ((double)xG)/65535.0;
     B = ((double)xB)/65535.0;

     col = SoMtlCreate();
     SoMColSetR_G_B(&(col->diffuseColor), 0.8*R, 0.8*G, 0.8*B);
     SoMColSetR_G_B(&(col->ambientColor), 0.2*R, 0.2*G, 0.2*B);
     SoMColSetR_G_B(&(col->specularColor), 0.5*R, 0.5*R, 0.5*R);
     SoMColSetR_G_B(&(col->emissiveColor), 0, 0, 0);
     SoMFloatSet(&(col->shininess), 0.2);
     SoMFloatSet(&(col->transparency), 0);
     SoSepAddChild(sep,  (SoNode *) col);
}


static void SetSphere()
{
     SoSphere *sphere;
     SoSep    *sep;
     double   radius;

     sphere = SoSphCreate();		  /* create sphere */
     pvm_upkdouble(&radius, 1, 1);
     SoSFloatSet(&(sphere->radius), radius);

     sep = SoSepCreateN(3);		  /* 1 here + 2 in unpackinfo */
     unpackinfo(sep);			  /* set orientation and position */

     SoSepAddChild(sep,  (SoNode *) sphere);
     SoSepAddChild(root, (SoNode *) sep); /* add sphere to tree */

#ifdef INVDEBUG
     printf("Sphere with radius %f\n",radius);
#endif
}

static void SetCuboid()
{
     double size[3];
     SoCube *cube;
     SoSep  *sep;

     cube = SoCubeCreate();		  /* create cuboid */
     pvm_upkdouble(size, 3, 1);
     SoSFloatSet(&(cube->width),  size[0]);
     SoSFloatSet(&(cube->height), size[1]);
     SoSFloatSet(&(cube->depth),  size[2]);

     sep = SoSepCreateN(3);		  /* 1 here + 2 in unpackinfo */
     unpackinfo(sep);			  /* set orientation and position */

     SoSepAddChild(sep,  (SoNode *) cube);
     SoSepAddChild(root, (SoNode *) sep); /* add cuboid to tree */

#ifdef INVDEBUG
     printf("Cuboid with edges (WxHxD): %f,%f,%f\n",size[0], size[1], size[2]);
#endif
}
	     
static void SetCylinder()
{
     double        radius, height;
     SoCyl         *cyl;
     SoRotationXYZ *rot;
     SoSep         *sep;
		
     cyl = SoCylCreate();		  /* create cylinder */
     pvm_upkdouble(&radius, 1, 1);
     pvm_upkdouble(&height, 1, 1);
     SoSFloatSet(&(cyl->radius), radius);
     SoSFloatSet(&(cyl->height), height);

     sep = SoSepCreateN(4);		  /* 2 here + 2 in unpackinfo */
     unpackinfo(sep);			  /* set orientation and position */

     rot = SoRotXYZCreate();		  /* the cylinder stands in y-axis */
     SoSFloatSet(&(rot->angle), PI/2);	  /* in inventor; AERO needs it in */
     SoSEnumSet(&(rot->axis), SO_ROT_XYZ_X); /* z-axis; so rotate it. */
     SoSepAddChild(sep,  (SoNode *) rot);
     SoSepAddChild(sep,  (SoNode *) cyl);

     SoSepAddChild(root, (SoNode *) sep); /* add cylinder to tree */

#ifdef INVDEBUG
     printf("Cylinder with radius %f and height %f\n", radius, height);
#endif
}

static void SetPlane()
{
     SoCube *plane;
     SoSep  *sep;

     plane = SoCubeCreate();		  /* create plane as thin cuboid */
     SoSFloatSet(&(plane->width), 3000);
     SoSFloatSet(&(plane->height), 0.01);
     SoSFloatSet(&(plane->depth), 3000);

     sep = SoSepCreateN(3);		  /* 1 here + 2 in unpackinfo */
     unpackinfo(sep);			  /* set orientation and position */

     SoSepAddChild(sep,  (SoNode *) plane);
     SoSepAddChild(root, (SoNode *) sep); /* add plane to tree */

#ifdef INVDEBUG
     printf("PLANE\n");
#endif
}

static void SetPoint()
{
     SoSphere *point;
     SoSep    *sep;
		
     point = SoSphCreate();		  /* create point as small sphere */
     SoSFloatSet(&(point->radius), 0.01);

     sep = SoSepCreateN(3);		  /* 1 here + 2 in unpackinfo */
     unpackinfo(sep);			  /* set position (and orientation) */

     SoSepAddChild(sep,  (SoNode *) point);
     SoSepAddChild(root, (SoNode *) sep); /* add point to tree */

#ifdef INVDEBUG
     printf("POINT\n");
#endif
}


/*****************************************************************************
 *  Function:       void SetSpring()
 *
 *  Parameters:     p1, p2:      end points of spring      
 *                  restlength:  resting length of spring (has influence of
 *                               number of gos)
 *                  springconst: stiffness of the spring (should have
 *                               influence on spring diameter and/or
 *                               material thickness -- but has not yet)
 *                     
 *  Result:
 *
 *  Import. idents:
 *
 *  Description:
 *  ------------
 *  Between the given points a spiral spring is drawn. The spring is
 *  represented by a single inventor object, described by points and
 *  rectangles. The number of gos (and probably sometimes the diameter and
 *  thickness of the spring material) is determined by the parameters of the
 *  spring. The spring consists of a starting part, the gos
 *  of the spring (each go represented by several parts interpolating a
 *  circle) and a final part. The accuracy is controlled by the number of
 *  parts per go (should be >10) and the number of polygons representing the
 *  cylindrical material (depends on size, should be >8).
 *  Some difficulties arise as the points where the starting and ending
 *  parts meet the gos of the spring have to be calculated explicitly. To
 *  avoid sharp edges the starting and ending parts have been rotated
 *  slightly (the last part of the go is shortened).
 *****************************************************************************/

#define PointsPerCircle 12
#define PartsPerGo 12
#define GoRadius 0.05
#define CylRadius 0.01
#define GosPerLen 30
#define goindex PointsPerCircle*2

static void SetSpring()
{
     typedef float point[3];

     SoSep *sep;
     SoXf  *trans;
     double rotangle;
     SbVec3f axis;

     SoCoord3 *controlPts;
     SoIndFSet *spring;
     point *pts;
     long  *index;

     int   ngos, npoints, nindex;
     double length, goway, gostep, gobase;
     int go, i, j, k;
     double sj, cj, si, ci;
     double alpha, beta, sx, sy;
     double q[4], len;
     double restlength, springconst;
     double p1[3],p2[3];

     pvm_upkdouble(&restlength, 1, 1);
     pvm_upkdouble(&springconst, 1, 1);
     pvm_upkdouble(p1, 3, 1);
     pvm_upkdouble(p2, 3, 1);

     /* get length of spring */
     p2[0]-=p1[0];
     p2[1]-=p1[1];
     p2[2]-=p1[2];
     length = sqrt(p2[0]*p2[0]+p2[1]*p2[1]+p2[2]*p2[2]);
    
     if (NEAR_0(length)) return;	  /* too short, do nothing */

     sep = SoSepCreate();		  /* spring as whole */

     /* set orientation and position of spring */
     QuaternionFromVector(p2, length, q);
     trans = SoXfCreate();
     SoSV3fSetX_Y_Z(&(trans->translation), p1[0], p1[1], p1[2]);
     len = sqrt(q[1] * q[1] + q[2] * q[2] + q[3] * q[3]);
     if (len<EPSILON) len=1.0;
     q[1] /= len;
     q[2] /= len;
     q[3] /= len;
     SbV3fSetX_Y_Z(axis, q[1], q[2], q[3]);

     rotangle = -2.0 * acos(q[0]);
     SoSRotSetAxisAngle(&(trans->rotation), 
			&axis, rotangle);
   
     SoSepAddChild(sep, (SoNode *) trans);


     /* compute number of gos, number of used points and index elements */
     ngos = restlength * GosPerLen;
     npoints= (ngos*PartsPerGo+PartsPerGo/2)*PointsPerCircle+5*PointsPerCircle;
     nindex = (ngos*PartsPerGo+PartsPerGo/2+4)*PointsPerCircle*5;

     /* compute increase steps for one go and one part of a go */
     gostep = (length*0.7)/(ngos*PartsPerGo+PartsPerGo/2);
     goway  = gostep*PartsPerGo;
     gobase = (length*0.15);		  /* starting point */

     /* allocate memory for points and indices */
     pts = (point*)(alloca(sizeof(point)*npoints));
     index = (long *)(alloca(sizeof(long)*nindex));

     /* generate points for the full gos */
     for (go=0; go<ngos; go++)
     {
	  for (j=0; j<PartsPerGo; j++)
	  {
	       sj = sin(j*2*PI/PartsPerGo);
	       cj = cos(j*2*PI/PartsPerGo);
	       
	       for (i=0; i<PointsPerCircle; i++)
	       {
		    si = sin(i*2*PI/PointsPerCircle);
		    ci = cos(i*2*PI/PointsPerCircle);
		    k = (go*PartsPerGo+j)*PointsPerCircle+i+goindex;
		    
		    pts[k][0] = (GoRadius+CylRadius*ci)*cj;
		    pts[k][1] = (GoRadius+CylRadius*ci)*sj;
		    pts[k][2] = CylRadius*si+go*goway+j*gostep+gobase;
	       }
	  }
     }
     /* generate points for an additional half go at the end */
     for (j=0; j<PartsPerGo/2; j++)
     {
	  sj = sin(j*2*PI/PartsPerGo);
	  cj = cos(j*2*PI/PartsPerGo);

	  for (i=0; i<PointsPerCircle; i++)
	  {
	       si = sin(i*2*PI/PointsPerCircle);
	       ci = cos(i*2*PI/PointsPerCircle);
	       k = (ngos*PartsPerGo+j)*PointsPerCircle+i+goindex;

	       pts[k][0] = (GoRadius+CylRadius*ci)*cj;
	       pts[k][1] = (GoRadius+CylRadius*ci)*sj;
	       pts[k][2] = CylRadius*si+ngos*goway+j*gostep+gobase;
	  }
     }

     /* rotation due to number of parts in one spring go */
     alpha = PI/PartsPerGo;
     /* rotation of the end parts due to the shortened last part of the go */
     beta = atan(CylRadius/(CylRadius+GoRadius));
     
     /* constants for cropping points */
     sx = CylRadius*(sin(alpha)+cos(alpha)*cos(beta))/cos(alpha-beta);
     sy = CylRadius*cos(alpha)*(1-sin(beta))/cos(alpha-beta);

     /* crop spiral part at 45 deg at start */
     j = 0;
     k = goindex;
     sj = 0;				  /*=sin(0*2*PI/PartsPerGo)*/
     cj = 1;				  /*=cos(0*2*PI/PartsPerGo)*/
     for (i=0; i<PointsPerCircle; i++)
     {
	  si = sin(i*2*PI/PointsPerCircle);
	  ci = cos(i*2*PI/PointsPerCircle);
	  pts[k+i][0] = (GoRadius+CylRadius-(1-ci)*sx)*cj;
	  pts[k+i][1] = (GoRadius+CylRadius)*sj+(1-ci)*sy;
	  pts[k+i][2] = CylRadius*si+gobase;
     }

     /* add start part */
     k = PointsPerCircle;
     for (i=0; i<PointsPerCircle; i++)
     {
	  si = sin(i*2*PI/PointsPerCircle-beta);
	  ci = cos(i*2*PI/PointsPerCircle-beta);
	  pts[k+i][0] = -(CylRadius*si);
	  pts[k+i][1] = -(CylRadius*ci);
	  pts[k+i][2] = (CylRadius*si)+gobase;
	  pts[i][0] = -(CylRadius*si);
	  pts[i][1] = -(CylRadius*ci);
	  pts[i][2] = 0;
     }
	  
     /* crop spiral part at 45 deg at end */
     j = PartsPerGo/2;
     k=(ngos*PartsPerGo+j)*PointsPerCircle+goindex;
     sj = 0;			     /*=sin(PartsPerGo/2*2*PI/PartsPerGo)*/
     cj = -1;			     /*=cos(PartsPerGo/2*2*PI/PartsPerGo)*/
     for (i=0; i<PointsPerCircle; i++)
     {
	  si = sin(i*2*PI/PointsPerCircle);
	  ci = cos(i*2*PI/PointsPerCircle);
	  pts[k+i][0] = (GoRadius+CylRadius-(1-ci)*sx)*cj;
	  pts[k+i][1] = (GoRadius+CylRadius)*sj+(1-ci)*sy;
	  pts[k+i][2] = CylRadius*si+ngos*goway+j*gostep+gobase;
     }

     /* add end part */
     k = goindex+(ngos*PartsPerGo+PartsPerGo/2)*PointsPerCircle+PointsPerCircle;
     for (i=0; i<PointsPerCircle; i++)
     {
	  si = sin(i*2*PI/PointsPerCircle+beta);
	  ci = cos(i*2*PI/PointsPerCircle+beta);
	  pts[k+i][0] = -(CylRadius*si);
	  pts[k+i][1] = -(CylRadius*ci);
	  pts[k+i][2] = (CylRadius*si)+length-gobase;
	  pts[k+i+PointsPerCircle][0] = -(CylRadius*si);
	  pts[k+i+PointsPerCircle][1] = -(CylRadius*ci);
	  pts[k+i+PointsPerCircle][2] = length;
     }
	  

     /* define rectangular polygons from the points to build the spring */
     /* shape. */
     for (j=0; j<(PartsPerGo*ngos+PartsPerGo/2+4); j++)
     {
	  for (i=0; i<PointsPerCircle; i++)
	  {
	       k = (j*PointsPerCircle+i)*5;
	       index[k+0] = j*PointsPerCircle+(i+1)%PointsPerCircle;
	       index[k+1] = j*PointsPerCircle+i;
	       index[k+2] = (j+1)*PointsPerCircle+i;
	       index[k+3] = (j+1)*PointsPerCircle+(i+1)%PointsPerCircle;
	       index[k+4] = SO_END_FACE_INDEX;
	  }
     }

     /* add the points node */
     controlPts = SoCoord3Create();
     SoMV3fSetXYZs( &(controlPts->point), 0, npoints, pts);
     SoSepAddChild(sep, (SoNode *) controlPts);

     /* add the indexed face shape (whole spring) */
     spring = SoIndFSetCreate();
     SoMLongSetN(&(spring->coordIndex), 0, nindex, index);
     SoSepAddChild(sep, (SoNode *) spring);

     /* add the spring including transformation to the tree */
     SoSepAddChild(root, (SoNode *) sep);

#ifdef INVDEBUG
     printf("SPRING\n");
#endif
}

static void SetDamper()
{
     SoCyl   *rod;
     SoCyl   *cyl;
     SoRotationXYZ *rot;
     SoXlate *move;
     SoSep   *sep;
				
     rod = SoCylCreate();		  /* create damper as a thin rod */
     SoSFloatSet(&(rod->radius), 0.01);	  /* with a bigger cylinder in the */
     SoSFloatSet(&(rod->height), 1.0);	  /* middle */
     cyl = SoCylCreate();
     SoSFloatSet(&(cyl->radius), 0.04);
     SoSFloatSet(&(cyl->height), 0.5);

     rot = SoRotXYZCreate();		  /* again the orientation is wrong */
     SoSFloatSet(&(rot->angle), PI/2);	  /* correct it by rotating about x */
     SoSEnumSet(&(rot->axis), SO_ROT_XYZ_X);
     move = SoXlateCreate();		  /* also move to [0..1] instead of */
     SoSV3fSetX_Y_Z(&(move->translation), 0.0, 0.0, 0.5); /* [-0.5..0.5] */

     sep = SoSepCreateN(6);		  /* 4 here + 2 in linkunpack */
     linkunpack(sep);			  /* set orientation and position */
     SoSepAddChild(sep, (SoNode *) move);
     SoSepAddChild(sep, (SoNode *) rot);
     SoSepAddChild(sep, (SoNode *) rod);
     SoSepAddChild(sep, (SoNode *) cyl);
     SoSepAddChild(root, (SoNode *) sep); /* add damper to tree */
		
#ifdef INVDEBUG
     printf("DAMPER\n");
#endif
}

static void SetRod()
{
     SoCyl   *rod;
     SoRotationXYZ *rot;
     SoXlate *move;
     SoSep   *sep;
				
     rod = SoCylCreate();		  /* create rod as thin cylinder */
     SoSFloatSet(&(rod->radius), 0.01);
     SoSFloatSet(&(rod->height), 1.0);

     rot = SoRotXYZCreate();		  /* again the orientation is wrong */
     SoSFloatSet(&(rot->angle), -PI/2);	  /* correct it by rotating about x */
     SoSEnumSet(&(rot->axis), SO_ROT_XYZ_X);
     move = SoXlateCreate();		  /* move to [0..1] */
     SoSV3fSetX_Y_Z(&(move->translation), 0.0, 0.0, 0.5);

     sep = SoSepCreateN(5);		  /* 3 here + 2 in linkunpack */
     linkunpack(sep);			  /* set orientation and position */
     SoSepAddChild(sep, (SoNode *) move);
     SoSepAddChild(sep,  (SoNode *) rot);
     SoSepAddChild(sep,  (SoNode *) rod);
     SoSepAddChild(root, (SoNode *) sep); /* add rod to tree */
#ifdef INVDEBUG
     printf("ROD\n");
#endif
}

static void SetJoint()
{
     SoSphere *point1, *point2;
     SoCyl    *minirod;
     SoRotationXYZ *rot;
     SoXlate  *move, *mp1, *mp2;
     SoSep    *sep;
		
     minirod = SoCylCreate();		  /* create joint as thin rod with */
     SoSFloatSet(&(minirod->radius), 0.005); /* two small spheres at the ends */
     SoSFloatSet(&(minirod->height), 1.0);
     point1 = SoSphCreate();
     SoSFloatSet(&(point1->radius), 0.01);
     point2 = SoSphCreate();
     SoSFloatSet(&(point2->radius), 0.01);
     mp1 = SoXlateCreate();
     SoSV3fSetX_Y_Z(&(mp1->translation), 0.0, -0.5, 0.0);
     mp2 = SoXlateCreate();
     SoSV3fSetX_Y_Z(&(mp2->translation), 0.0, 1.0, 0.0);

     rot = SoRotXYZCreate();
     SoSFloatSet(&(rot->angle), -PI/2);
     SoSEnumSet(&(rot->axis), SO_ROT_XYZ_X);
     move = SoXlateCreate();
     SoSV3fSetX_Y_Z(&(move->translation), 0.0, 0.0, 0.5);

     sep = SoSepCreateN(9);		  /* 7 here + 2 in linkunpack */
     linkunpack(sep);			  /* set orientation and position */

     SoSepAddChild(sep, (SoNode *) move);
     SoSepAddChild(sep, (SoNode *) rot);
     SoSepAddChild(sep, (SoNode *) minirod);
     SoSepAddChild(sep, (SoNode *) mp1);
     SoSepAddChild(sep, (SoNode *) point1);
     SoSepAddChild(sep, (SoNode *) mp2);
     SoSepAddChild(sep, (SoNode *) point2);
     SoSepAddChild(root, (SoNode *) sep); /* add joint to tree */

#ifdef INVDEBUG
     printf("JOINT\n");
#endif
}

static void SetCamera()
{
     SbVec3f axis;
     SoCam *camera;
     double len;
     double rotangle;
     SoType camtype;

     len = sqrt(camq[1]*camq[1] + camq[2]*camq[2] + camq[3]*camq[3]);
     if (len<1e-9) len=1.0;
     camq[1] /= len;
     camq[2] /= len;
     camq[3] /= len;
     SbV3fSetX_Y_Z(axis, camq[1],  camq[2],  camq[3]);
     rotangle = -2.0 * acos(camq[0]);
		     
     camera = SoXtExamVwrGetCam(viewer);
     SoSV3fSetX_Y_Z(&(camera->position), campos[0], campos[1], campos[2]);
     SoSRotSetAxisAngle(&(camera->orientation), &axis, rotangle);
     SoSFloatSet(&(camera->nearDistance), 0.06);
     SoSFloatSet(&(camera->farDistance), 100.00);
     SoSFloatSet(&(camera->focalDistance), 5);

     camtype = SoXtVwrGetCamType((SoXtVwr*) viewer);
     if (SoTypeIsEq(&camtype,SoPerspCamGetClassTypeId()))
     {
	  /* Zoom */
	  SoSFloatSet(&(((SoPerspCam*)camera)->heightAngle), 0.5/camzoom);
     }
     else
     {
	  SoXtExamVwrResetToHomePos(viewer);
     }
     SoXtExamVwrStopAnim(viewer);
}

void ReturnAck(unsigned long framenumber)
{
     /* Send framenumber via PVM back to AERO */
     pvm_initsend(PvmDataDefault);
     pvm_pkulong(&framenumber, 1, 1);
     pvm_send(parent_tid, ACK);
}

void DrawPic()
{
     int object;
     
     SoSepRemoveAllChildren(root);
     while (1)
     {
	  pvm_unpackf("%d", &object);
          if (object==NOOBJECT)		  /* end of frame? */
	  {
#ifdef INVDEBUG
	       printf("NOOBJECT\n");
#endif
	       break;			  /* leave the while loop */
	  }
	  switch (object)
	  {
	    case SPHERE:		  /* AERO objects */
	       SetSphere();
	       break;
		
	    case CUBOID:
	       SetCuboid();
	       break;

	    case CYLINDER:
	       SetCylinder();
	       break;
	     
	    case PLANE:
	       SetPlane();
	       break;

	     case POINT:
	       SetPoint();
	       break;

	     case SPRING:		  /* AERO links */
	       SetSpring();
	       break;

	     case DAMPER:
	       SetDamper();
	       break;

	     case ROD:
	       SetRod();
	       break;

	     case JOINT:
	       SetJoint();
	       break;

	     case CAMERA:
	     {
		  pvm_upkdouble(campos, 3, 1);
		  pvm_upkdouble(camq, 4, 1);
		  pvm_upkdouble(&camzoom, 1, 1);

		  if (UseCamera) {
		       SetCamera();
		  }

#ifdef INVDEBUG
		  printf("CAMERA\n");
#endif
		  break;
	     }
	     
	     default:
#ifdef INVDEBUG
	        printf("Undefined object %d\n", object);
#endif
	        break;
	  }
     }
}


static void cameraButtonCallback(Widget w, XtPointer dummy, XtPointer info)
{
     Arg cbChangeArglist[1];

     UseCamera=!UseCamera;
     XtVaSetValues(w, XmNlabelString, UseCamera ? LabelX : LabelO, NULL);
     if (UseCamera) SetCamera();

}

/****************************************************************************/

static void ServerLoop_CB(XtPointer cli, int *src, XtInputId *id)
{
     int bufid, bytes, msgtag, client_tid;

     while(pvm_probe(-1, -1)) {

	  bufid = pvm_recv(-1,-1);
	  pvm_bufinfo(bufid, &bytes, &msgtag, &client_tid);

	  switch(msgtag) {

	    case OPEN_INVENTOR:
	       break;

	    case CLOSE_INVENTOR:
	       break;

	    case EXIT_INVENTOR:
	       exit (0);

	    case SYNC_ON:
	       syncflag = TRUE;
	       break;

	    case SYNC_OFF:
	       syncflag = FALSE;
	       break;

	    case FRAME:
	       {
		    unsigned long framenumber;

		    pvm_upkulong(&framenumber, 1, 1);
#if 0		   
		    /* ???? SoXtExamVwrIsStereoViewing() doesn't exist in */
		    /* the library!?!?!?! */
		    if (SoXtExamVwrIsStereoViewing(viewer)!=stereo)
		    {
			 SoCam *camera;
			 stereo = SoXtExamVwrIsStereoViewing(viewer);
			 camera = SoXtExamVwrGetCam(viewer);
			 if (stereo)
			      SoSFloatSet(&(camera->aspectRatio), 2.0);
			 else
			      SoSFloatSet(&(camera->aspectRatio), 1.0);
		    }
#endif
		    
/*		    SoXtExamVwrSetAutoRedraw(viewer, FALSE);*/
#ifdef INVDEBUG
		    printf("BeginFrame No. %lu\n",framenumber);
		    DrawPic(root);
		    printf("EndFrame\n");
#else
		    DrawPic(root);
#endif
		    SoXtExamVwrRender(viewer);
/*		    SoXtExamVwrSetAutoRedraw(viewer, TRUE);*/
/*		    SoXtExamVwrSetRedrawPriority(viewer, 0);
		    SoSepTouch(root);
*/
		    if (syncflag)
		    {
			 ReturnAck(framenumber); 
		    }
	       }
	       break;
	    default:
	       break;
	   
	  } /* switch */
     } /* while */
}



/*----------------------------------------------------------------------*/

/*****************************************************************************
 *  Synopsis:       void main(int argc, char **argv)
 *
 *  Parameters:     argc:    number of command line arguments
 *                  argv:    array of command line arguments
 *                     
 *  Result:         return value for shell
 *
 *  Import. idents:
 *
 *  Description:
 *  ------------
 *  This is the main routine. This starts the Xt, creates callback functions
 *  for all input sources (PVM, extra buttons), starts an examiner viewer
 *  window and goes in the main event loop of the inventor toolkit.
 *****************************************************************************/

#define argcount 1			  /* number of arguments in the */
					  /* widget creation call */

void main(int argc, char **argv)
{
     XtAppContext app_context;
     int n, *fds;

     Arg cameraButtonArglist[argcount];


     /* Get own PVM-task-ID and of parent process (AERO) */
     my_tid = pvm_mytid();
     parent_tid = pvm_parent();

     if (my_tid < 0)
     {
	  fprintf(stderr, "%s: PVM is not running.",argv[0]);
	  exit(-1);
     }

     if (parent_tid < 0) {
	 fprintf(stderr, "%s: no AERO parent process, this makes no sense!\n", argv[0]);
	 fprintf(stderr, "%s: my PVM-tid is %d\n", argv[0], my_tid);
     }

     /* Initialize Xt and Inventor, add callback for PVM-messages */
     mainWindow = SoXtInit( argv[0], "Inventor" );
     if ( mainWindow == NULL ) exit( 1 );

     app_context = SoXtGetAppContext();

     n = pvm_getfds(&fds);
     while(n-- > 0) {
	  XtAppAddInput(app_context, fds[n], (XtPointer) XtInputReadMask,
			ServerLoop_CB, (XtPointer) 0);
     }

     /* Create root-node for all objects. */
     root = SoSepCreate();
     SoSepRef( root );

     /* Create labels for extra button */
     LabelX = XmStringCreate("X", XmFONTLIST_DEFAULT_TAG);
     LabelO = XmStringCreate("O", XmFONTLIST_DEFAULT_TAG);

     XtSetArg(cameraButtonArglist[0], XmNlabelString, LabelX);


     /* Setup viewer with an extra button, add callback for button handling */
     viewer = SoXtExamVwrCreateStd(mainWindow, NULL);
     cameraButton =
	  XmCreatePushButtonGadget(SoXtExamVwrGetAppPushBtnParent(viewer),
				   "cameraButton", cameraButtonArglist,
				   argcount);
     XtAddCallback(cameraButton, XmNactivateCallback, cameraButtonCallback, NULL);
     SoXtExamVwrAddAppPushBtn(viewer, cameraButton);

     /* Link viewer to root node, set title bar and start viewer */
     SoXtExamVwrSetScene( viewer, (SoNode *) root );
     SoXtExamVwrSetTitle(viewer, "XAERO Viewer");
     SoXtExamVwrScheduleRedraw(viewer);
     SoXtExamVwrShow( viewer );

     SoXtShow(mainWindow);

     /* Mainloop, wait forever */
     SoXtMainLoop();
}
