/********************************************************************************
 *  Projektname		: AERO
 *  Filename		: materialien.c
 *  Filetyp		: C-Source
 ********************************************************************************
 *  Modulname		: materialien.o
 *  letzte Aenderung	: 2.11.93
 *  Autor		: Horst Stolz (HUS) / Hartmut Keller
 *  Status:		: erfolgreich compiliert
 *  
 *  Beschreibung:
 *    liefert eine einfache Materialtabelle
 *
 *  Noch zu mache:
 *    - mehr Materialien
 *    - Reibungs- und Elastizitaetskoeffizienten
 *
 *  Versionsgeschichte:
 *  -------------------
 *  03.12.92   File erstellt
 *  25.01.93   Mit "unrichtige" Elastizitaetswerte belegt
 *  18.05.93   Default-Werte fuer Raytracer eingefuegt
 *  13.07.93   Default-Farbwerte fuer ErgaenzeMatTab auf 0(Schwarz) gesetzt 
 *  11.05.94   Compiler Warnings rausgemacht
 ********************************************************************************/

#include <stdlib.h>

#include "materialien.h"
#include "fehler.h"

static char *MatNamen[33] = {
    "iron", "plastic", "cork", "ice", "glass", "granite", "gold", "alu", "wood",
    "concrete", "brick", "rubber", NULL
};

static TReal MatDichte[32] = {
    7.8e3, 0.9e3, 0.27e3, 0.917e3, 2.5e3, 2.8e3,
    19.29e3, 2.7e3, 0.6e3, 2.3e3, 1.5e3, 0.94e3
};

static TReal MatGleitfaktor[32] = {
	1.0, 0.4, 0.5, 0.014, 0.02, 0.6, 0.9, 0.98, 0.5, 0.2, 1.2, 3.5    
};

static TReal MatHaftfaktor[32] = {
    1.3, 0.5, 0.6, 0.002, 0.04, 0.8, 0.95, 1.05, 0.65, 0.3, 1.5, 4.0
};

static TReal MatStossfaktor[32] = {
    0.9, 0.95, 0.8, 0.8, 0.97, 0.3, 0.2, 0.92, 0.7, 0.5, 0.4, 0.99
};

static TReal MatDehnfaktor[32] = {
    0.5e5, 0.5e5, 0.5e5, 0.5e5, 0.5e5, 0.5e5, 0.5e5, 0.5e5, 0.5e5, 0.5e5, 0.5e5, 0.5e5

};

static TReal MatDaempffaktor[32] = {
    0.5e3, 0.5e3, 0.5e3, 0.5e3, 0.5e3, 0.5e3, 0.5e3, 0.5e3, 0.5e3, 0.5e3, 0.5e3, 0.5e3
};

static unsigned short MatDefaultR[32] = {
    23550, 16383, 37887, 52735, 64000, 26111, 52428, 56319, 38911, 32767, 65535, 48639
};

static unsigned short MatDefaultG[32] = {
    27135, 50687, 29695, 63487, 64000, 24063, 32639, 55295, 24574, 34815, 13107, 44031
};

static unsigned short MatDefaultB[32] = {
    36862, 11263, 20991, 65535, 65535, 21503, 12850, 56831, 0, 37375, 1542, 36351
};

static TReal MatDurchsichtigkeit[32] = {
    0.2, 0.1, 0.45, 0.6, 0.9, 0.4, 0.8, 0.3, 0.35, 0.5, 0.5, 0.3
};

static TReal MatReflexion[32] = {
    0.1, 0.0, 0.0, 0.3, 0.7, 0.0, 0.25, 0.1, 0.0, 0.0, 0.0, 0.0
};

static TReal MatRauhigkeit[32] = {
    0.0, 0.03, 0.5, 0.0, 0.0, 0.1, 0.0, 0.03, 0.05, 0.8, 0.8, 0.6
};

static unsigned short MatFlags[32];


TMaterialTabelle GrundMaterialien = {
    12,
    32,
    MatNamen,
    MatHaftfaktor,
    MatGleitfaktor, 
    MatStossfaktor,
    MatDichte,
    MatDehnfaktor,
    MatDaempffaktor,
    MatDefaultR,
    MatDefaultG,
    MatDefaultB,
    MatDurchsichtigkeit,
    MatReflexion,
    MatRauhigkeit,
    MatFlags
};




TReal Elastizitaet(TMaterialTabelle *mt, TMaterial m1, TMaterial m2)
{
    return mt->Stossfaktor[m1] * mt->Stossfaktor[m2];
}


TReal Gleitreibung(TMaterialTabelle *mt, TMaterial m1, TMaterial m2)
{
    return mt->Gleitreibungsfaktor[m1] * mt->Gleitreibungsfaktor[m2];
}


TReal Haftreibung(TMaterialTabelle *mt, TMaterial m1, TMaterial m2)
{
    return mt->Haftreibungsfaktor[m1] * mt->Haftreibungsfaktor[m2];
}


TReal Dehnungsfaktor(TMaterialTabelle *mt, TMaterial m1, TMaterial m2)
{
    return mt->Dehnungsfaktor[m1] + mt->Dehnungsfaktor[m2];
}


TReal Daempfungsfaktor(TMaterialTabelle *mt, TMaterial m1, TMaterial m2)
{
    return mt->Daempfungsfaktor[m1] + mt->Daempfungsfaktor[m2];
}


TReal Dichte(TMaterialTabelle *mt, TMaterial m)
{
    return mt->Dichte[m];
}

unsigned short DefaultR(TMaterialTabelle *mt, TMaterial m)
{
     return mt->DefaultR[m];
}

unsigned short DefaultG(TMaterialTabelle *mt, TMaterial m)
{
     return mt->DefaultG[m];
}

unsigned short DefaultB(TMaterialTabelle *mt, TMaterial m)
{
     return mt->DefaultB[m];
}

TReal DefaultDurchsichtigkeit(TMaterialTabelle *mt, TMaterial m)
{
     return mt->Durchsichtigkeit[m];
}

TReal DefaultReflexion(TMaterialTabelle *mt, TMaterial m)
{
     return mt->Reflexion[m];
}

TReal DefaultRauhigkeit(TMaterialTabelle *mt, TMaterial m)
{
     return mt->Rauhigkeit[m];
}



char **MaterialNamen(TMaterialTabelle *mt)
{
    return (mt->MaterialNamen);
}


int AnzahlMaterialien(TMaterialTabelle *mt)
{
    return mt->AnzahlMaterialien;
}


void LoescheMatTab(TMaterialTabelle *mt)
{
    int i;

    if (mt && mt != &GrundMaterialien) {
	for (i=mt->AnzahlMaterialien-1; i>=0; i--)
	    if (mt->MaterialNamen[i]) free(mt->MaterialNamen[i]);

	free(mt->MaterialNamen);
	free(mt->Dichte);
	free(mt->Haftreibungsfaktor);
	free(mt->Gleitreibungsfaktor);
	free(mt->Stossfaktor);
	free(mt->Dehnungsfaktor);
	free(mt->Daempfungsfaktor);
	free(mt->DefaultR);
	free(mt->DefaultG);
	free(mt->DefaultB);
	free(mt->Durchsichtigkeit);
	free(mt->Reflexion);
	free(mt->Rauhigkeit);
	free(mt->Flags);

	free(mt);
    }
}


TMaterialTabelle *AllokiereMatTab(int MaxMatAnz)
{
    int i;
    TMaterialTabelle *mt;

    FehlerOrt("AllokiereMatTab()");

    mt = (TMaterialTabelle *)malloc(sizeof(TMaterialTabelle));
    mt->MaxMatAnz = MaxMatAnz;
    mt->AnzahlMaterialien = 0;

    mt->MaterialNamen = (char **)malloc((MaxMatAnz+1)*sizeof(char *));
    mt->Dichte = (TReal *)malloc(MaxMatAnz*sizeof(TReal));
    mt->Haftreibungsfaktor = (TReal *)malloc(MaxMatAnz*sizeof(TReal));
    mt->Gleitreibungsfaktor = (TReal *)malloc(MaxMatAnz*sizeof(TReal));
    mt->Stossfaktor = (TReal *)malloc(MaxMatAnz*sizeof(TReal));
    mt->Dehnungsfaktor = (TReal *)malloc(MaxMatAnz*sizeof(TReal));
    mt->Daempfungsfaktor = (TReal *)malloc(MaxMatAnz*sizeof(TReal));
    mt->DefaultR = (unsigned short *)malloc(MaxMatAnz*sizeof(unsigned short));
    mt->DefaultG = (unsigned short *)malloc(MaxMatAnz*sizeof(unsigned short));
    mt->DefaultB = (unsigned short *)malloc(MaxMatAnz*sizeof(unsigned short));
    mt->Durchsichtigkeit = (TReal *)malloc(MaxMatAnz*sizeof(TReal));
    mt->Reflexion = (TReal *)malloc(MaxMatAnz*sizeof(TReal));
    mt->Rauhigkeit = (TReal *)malloc(MaxMatAnz*sizeof(TReal));
    mt->Flags = (unsigned short *)malloc(MaxMatAnz*sizeof(unsigned short));


    if (mt->MaterialNamen==NULL ||
        mt->Dichte==NULL ||
	mt->Haftreibungsfaktor==NULL ||
	mt->Gleitreibungsfaktor==NULL ||
	mt->Stossfaktor==NULL ||
	mt->Dehnungsfaktor==NULL ||
        mt->Daempfungsfaktor==NULL ||
	mt->DefaultR==NULL ||
	mt->DefaultG==NULL ||
	mt->DefaultB==NULL ||
	mt->Durchsichtigkeit==NULL ||
	mt->Reflexion==NULL ||
	mt->Rauhigkeit==NULL ||
	mt->Flags==NULL) {
	Fehler("No memory for table of materials!");
    }

    for (i=MaxMatAnz; i>=0; i--)
    	mt->MaterialNamen[i] = NULL;

    return mt;
}


TBoolean ErgaenzeMatTab2(TMaterialTabelle *mt, char *name,
			TReal Dichte, TReal Gleitfaktor, TReal Haftfaktor,
			TReal Stossfaktor, TReal Dehnfaktor, TReal Daempffaktor,
			unsigned short DefaultR, unsigned short DefaultG, 
			unsigned short DefaultB, TReal Durchsichtigkeit, 
			TReal Reflexion, TReal Rauhigkeit)
{
    int i;

    FehlerOrt("ErgaenzeMatTab()");
    
    
    if (mt==NULL) Fehler("Parameter mt==NULL!!!");

    i = mt->AnzahlMaterialien;

    if (i >= mt->MaxMatAnz) return TRUE;

    mt->MaterialNamen[i] = malloc(sizeof(char) * (strlen(name)+1));
    if (mt->MaterialNamen[i] == NULL)
	Fehler("No memory for material names!");

    strcpy(mt->MaterialNamen[i], name);

    mt->Dichte[i] = Dichte;
    mt->Gleitreibungsfaktor[i] = Gleitfaktor;
    mt->Haftreibungsfaktor[i] = Haftfaktor;
    mt->Stossfaktor[i] = Stossfaktor;
    mt->Dehnungsfaktor[i] = Dehnfaktor;
    mt->Daempfungsfaktor[i] = Daempffaktor;
    mt->DefaultR[i] = DefaultR;
    mt->DefaultG[i] = DefaultG;
    mt->DefaultB[i] = DefaultB;
    mt->Durchsichtigkeit[i] = Durchsichtigkeit;
    mt->Reflexion[i] = Reflexion;
    mt->Rauhigkeit[i] = Rauhigkeit;
    mt->Flags[i] = 0;

    mt->AnzahlMaterialien = i + 1;
    
    return FALSE;
}




TBoolean ErgaenzeMatTab(TMaterialTabelle *mt, char *name,
			TReal Dichte, TReal Gleitfaktor, TReal Haftfaktor,
			TReal Stossfaktor, TReal Dehnfaktor, TReal Daempffaktor)
{
    int i;

    FehlerOrt("ErgaenzeMatTab()");
    
    
    if (mt==NULL) Fehler("Parameter mt==NULL!!!");

    i = mt->AnzahlMaterialien;

    if (i >= mt->MaxMatAnz) return TRUE;

    mt->MaterialNamen[i] = malloc(sizeof(char) * (strlen(name)+1));
    if (mt->MaterialNamen[i] == NULL)
	Fehler("Kann Speicher fuer MaterialNamen nicht allokieren!");

    strcpy(mt->MaterialNamen[i], name);

    mt->Dichte[i] = Dichte;
    mt->Gleitreibungsfaktor[i] = Gleitfaktor;
    mt->Haftreibungsfaktor[i] = Haftfaktor;
    mt->Stossfaktor[i] = Stossfaktor;
    mt->Dehnungsfaktor[i] = Dehnfaktor;
    mt->Daempfungsfaktor[i] = Daempffaktor;

    mt->DefaultR[i] = 0;
    mt->DefaultG[i] = 0;
    mt->DefaultB[i] = 0;
    mt->Durchsichtigkeit[i] = 0.0;
    mt->Reflexion[i] = 0.0;
    mt->Rauhigkeit[i] = 0.0;
    mt->Flags[i] = 0;

    mt->AnzahlMaterialien = i + 1;
    
    return FALSE;
}
