/**
 ** code.c
 **
 ** Copyright 1995 by Kurt Konolige
 **
 ** The author hereby grants to SRI permission to use this software.
 ** The author also grants to SRI permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by SRI, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor SRI make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Kurt Konolige
 ** Senior Computer Scientist
 ** SRI International
 ** 333 Ravenswood Avenue
 ** Menlo Park, CA 94025
 ** E-mail:  konolige@ai.sri.com
 **
 **/


#include "core.h"
#define TRUE 1
#define FALSE 0

Int code_debug = 1;
Int code_inline = 0;
Int native_flag = 0;

/*
 * opcodes
 */

mops allcodes[] = {
  {"JMP IMM", 0x7E, 0},
  {"BRA", 0x20, 0},
  {"BRZ", 0x27, 0},
  {"BRNZ", 0x26, 0},
  {"BRGT", 0x2E, 0},
  {"BRLT", 0x2D, 0},
  {"BRGE", 0x2C, 0},
  {"BRLE", 0x2F, 0},
  {"CPD VAL", 0x1A, 0xB3},
  {"CPD X", 0x1A, 0xA3},
  {"CPD IMM", 0x1A, 0x83},
  {"CLRA", 0x4F, 0},
  {"CLRB", 0x5F, 0},
  {"NEGA", 0x40, 0},
  {"NEGB", 0x50, 0},
  {"COMA", 0x43, 0},
  {"COMB", 0x53, 0},
  {"ADCA", 0x89, 0},
  {"ADCB", 0xC9, 0},
  {"LDD VAL", 0xFC, 0},
  {"LDB VAL", 0xF6, 0},
  {"LDA VAL", 0xB6, 0},
  {"LDX VAL", 0xFE, 0},
  {"LDY VAL", 0x18, 0xFE},
  {"LDD IMM", 0xCC, 0},
  {"LDA IMM", 0x86, 0},
  {"LDB IMM", 0xC6, 0},
  {"LDX IMM", 0xCE, 0},
  {"LDY IMM", 0x18, 0xCE},
  {"LDY X", 0x1A, 0xEE},
  {"LDB Y", 0x18, 0xE6},
  {"LDB X", 0xE6, 0},
  {"LDA Y", 0x18, 0xA6},
  {"LDA X", 0xA6, 0},
  {"LDD X", 0xEC, 0},
  {"LDD Y", 0x18, 0xEC},
  {"STD IMM", 0xFD, 0},
  {"STA IMM", 0xB7, 0},
  {"STB IMM", 0xF7, 0},
  {"STD Y", 0x18, 0xED},
  {"STA Y", 0x18, 0xA7},
  {"STB Y", 0x18, 0xE7},
  {"STD X", 0xED, 0},
  {"STA X", 0xA7, 0},
  {"STB X", 0xE7, 0},
  {"INX", 0x08, 0},
  {"INY", 0x18, 0x08},
  {"DEX", 0x09, 0},
  {"DEY", 0x18, 0x09},
  {"XCDY", 0x18, 0x8F},
  {"XCDX", 0x8F, 0},
  {"ADD VAL", 0xF3, 0},
  {"ADD X", 0xE3, 0},
  {"ADD Y", 0x18, 0xE3},
  {"ADD IMM", 0xC3, 0},
  {"ADDB VAL", 0xFB, 0},
  {"ADDB X", 0xEB, 0},
  {"ADDB Y", 0x18, 0xEB},
  {"ADDB IMM", 0xCB, 0},
  {"SUB VAL", 0xB3, 0},
  {"SUB X", 0xA3, 0},
  {"SUB Y", 0x18, 0xA3},
  {"SUB IMM", 0x83, 0},
  {"SUBB VAL", 0xF0, 0},
  {"SUBB X", 0xE0, 0},
  {"SUBB Y", 0x18, 0xE0},
  {"SUBB IMM", 0xC0, 0},
  {"ANDA VAL", 0xB4, 0},
  {"ANDA X", 0xA4, 0},
  {"ANDA Y", 0x18, 0xA4},
  {"ANDA IMM", 0x84, 0},
  {"ANDB VAL", 0xF4, 0},
  {"ANDB X", 0xE4, 0},
  {"ANDB Y", 0x18, 0xE4},
  {"ANDB IMM", 0xC4, 0},
  {"ORA VAL", 0xBA, 0},
  {"ORA X", 0xAA, 0},
  {"ORA Y", 0x18, 0xAA},
  {"ORA IMM", 0x8A, 0},
  {"ORB VAL", 0xFA, 0},
  {"ORB X", 0xEA, 0},
  {"ORB Y", 0x18, 0xEA},
  {"ORB IMM", 0xCA, 0},
  {"XORA VAL", 0xB8, 0},
  {"XORA X", 0xA8, 0},
  {"XORA Y", 0x18, 0xA8},
  {"XORA IMM", 0x88, 0},
  {"XORB VAL", 0xF8, 0},
  {"XORB X", 0xE8, 0},
  {"XORB Y", 0x18, 0xE8},
  {"XORB IMM", 0xC8, 0},
  {"PSHA", 0x36, 0},
  {"PSHB", 0x37, 0},
  {"PSHX", 0x3C, 0},
  {"PSHY", 0x18, 0x3C},
  {"PULA", 0x32, 0},
  {"PULB", 0x33, 0},
  {"PULX", 0x38, 0},
  {"PULY", 0x18, 0x38},
  {"JSR IMM", 0xBD, 0},
  {"RTS", 0x39, 0},

} ;


/*
 * Test on arguments
 */

Int
undefined(sexp *s)
{
  if (type_id(s->type) == undef_id) return 1;
  else return 0;
}

Int
ref_type(sexp *s)
{
  if (type_id(s->type) != pointer_id) return -1;
  else return type_id(type_Pointer_deref(s->type));
}



Int
loadable_arg(Int n)		/* 0 is top of stack, 1 is next element, etc */
{
  Type_id t;
  if (n < sem_stack_n)
    {
      t = type_id(sem_stack[sem_stack_n - n - 1]->type);
      if (t == int_id || t == char_id || t == long_id || t == pointer_id)
	return(1);
    }
  return(0);
}

Int
address_arg(sexp *s)		/* returns 1 if s is an effective address */
{
  etype e;
  Type_id t;

  t = type_id(s->type);
  e = s->exp;
  if (e == cnst && 
      (t == int_id || t == char_id || t == long_id || t == pointer_id ))
    return(1);
  else
    return(0);
}


Int
indirect_arg(sexp *s)		/* returns 1 if s is an indirect address */
{
  etype e;
  Type_id t;

  t = type_id(s->type);
  e = s->exp;
  if (e == val)
    {
      if (!s->arg1 && (t == int_id || t == char_id || t == long_id ))
	return(1);
      else if (s->arg1->exp == cnst) 
	return(1);
    }
  else
    return(0);
}

Int
ref_arg(sexp *s)		/* returns 1 if s is an effective address */
{
  etype e;
  Type_id t;

  t = type_id(s->type);
  e = s->exp;
  if ((e == ref || e == cnst) && 
      (t == int_id || t == char_id || t == long_id || t == pointer_id ))
    return(1);
  else
    return(0);
}

/*
 * machine op generators
 */

void
code_mop(mcodes op)		/* stores one machine op */
{
  if (spew_debug) 
    debug_out(("\n%x:[%s ", code_current, allcodes[op].name));
  spew_byte(allcodes[op].first);
  if (allcodes[op].second)
    spew_byte(allcodes[op].second);
  if (spew_debug) 
    debug_out(("]"));
}

void
code_m2op(mcodes op, Int byte)	/* stores one machine op */
{
  if (spew_debug) 
    debug_out(("\n%x:[%s ", code_current, allcodes[op].name));
  spew_byte(allcodes[op].first);
  if (allcodes[op].second)
    spew_byte(allcodes[op].second);
  spew_byte(byte);
  if (spew_debug) 
    debug_out(("]"));
}


/*
 * some simple immediate code generators
 */

void				/* negates D accum */
code_negate()
{
  code_mop(negb);
  code_m2op(adca, 0);
  code_mop(nega);
}

void				/* complements D accum */
code_bitcomp()
{
  code_mop(comb);
  code_mop(coma);
}

void
code_lognot()			/* logical NOT */
{
  code_m2op(brnz,5);	/* if zero, 1 */
  code_mop(lddi);
  spew_word(1);
  code_m2op(bra,2);
  code_mop(clra);	/* if 1, zero */
  code_mop(clrb);
}

void				/* immediate integer */
code_load_int(Int i)
{
  code_mop(lddi);
  spew_word(i);
}

void				/* do a conditional load of 1 or 0 */
code_load_cond(mcodes mc)
{
  code_m2op(mc,4);
  code_mop(clra);
  code_mop(clrb);
  code_m2op(bra,3);
  code_load_int(1);
}

void				/* immediate reference pointer */
code_load_ref(Symbol *sym)
{
  code_mop(lddi);
  spew_label_reference(sym);
}

void				/* value of symbol */
code_load_val(Symbol *sym, Int offset, Int stack_offset)
{
  Value *value= symtab_get(sym);

  switch(value_loctype(value)) 
    {
    case global_location:
    case zero_page_global_location:
      code_mop(ldd);
      spew_label_reference_offset(sym,offset);
      break;
    case stack_location:
      code_m2op(lddx, stack_offset + offset + current_stack_size - value_address(value));
      break;
    default:
      die(("Illegal location_type in define_get_symbol"));
    }
}


void				/* takes most recent stack entry to D accum */
code_load_stack()
{
  Type *t = define_pop();

  switch(type_sizeof(t))
    {
    case 1:			/* pop one byte */
      code_m2op(ldbx, 0);
      code_mop(clra);
      code_mop(inx);
      break;
    case 2:			/* pop two bytes */
      code_m2op(lddx, 0);
      code_mop(inx);
      code_mop(inx);
      break;
    case 4:			/* pop four bytes */
      code_m2op(lddx, 2);
      code_mop(inx);
      code_mop(inx);
      code_mop(inx);
      code_mop(inx);
      break;
    default:
      die(("Shouldn't be popping %d bytes\n", type_sizeof(t)));
    }
}

void
code_pop_n(Int n)		/* pops N bytes off stack */
{
  while (n--) code_mop(inx);
}


void				/* takes most recent stack off */
code_pop_stack()
{
  Type *t = define_pop();

  switch(type_sizeof(t))
    {
    case 1:			/* pop one byte */
      code_mop(inx);
      break;
    case 2:			/* pop two bytes */
      code_mop(inx);
      code_mop(inx);
      break;
    case 4:			/* pop four bytes */
      code_mop(inx);
      code_mop(inx);
      code_mop(inx);
      code_mop(inx);
      break;
    default:
      die(("Shouldn't be popping %d bytes\n", type_sizeof(t)));
    }
}


void
code_store_stack()
{
  code_mop(dex);
  code_mop(dex);
  code_m2op(stdx, 0);
}

void				/* long at address i */
code_load_long_val(Int i)	/* automatically pushes it */
{
  code_mop(ldd);
  spew_word(i);
  code_store_stack();
  code_mop(ldd);
  spew_word(i+2);
  code_store_stack();
  define_push(type_Long());
}

void				/* int at address i */
code_load_int_val(Int i)
{
  code_mop(ldd);
  spew_word(i);
}




void
code_jump(Symbol *sym)		/* jump immediately to this symbolic address */
{
  code_mop(jmp);
  spew_label_reference(sym);
}

void				/* pops two-byte run stack entry to Y reg */
code_pop_to_y()
{
  code_m2op(ldyx, 0);
  code_mop(inx);
  code_mop(inx);
}

void
code_store1_to_y()		/* stores one byte to Y loc */
{
  code_m2op(stby, 0);
}

void
code_store_to_y()		/* stores two bytes to Y loc */
{
  code_m2op(stdy, 0);
}

void
code_load1_from_y()		/* loads one byte from Y loc */
{
  code_m2op(ldby, 0);
  code_mop(clra);
}

void
code_load_from_y()		/* loads two bytes from Y loc */
{
  code_m2op(lddy, 0);
}


void
code_dup_val()			/* duplicate value of top of stack */
{
  Type *type = define_stack_type(0);
  if (type_id(type) != pointer_id)
    {
      if (type_id(type) == undef_id)
	{
	  define_push(type_Undef());
	  return;
	}
      else
	die(("Attempt to take value of non-pointer on stack"));
    }

  code_mop(dex);
  code_mop(dex);
  code_m2op(ldyx, 2);
  code_m2op(lddy, 0);
  code_m2op(stdx, 0);
  define_push(type_Pointer_deref(type));
}


void
code_long_to_int()
{
  code_mop(inx);
  code_mop(inx);
  define_pop();
  define_push(type_Int());
}

/*
 * evaluate and generate code
 */

void
code_imm_jump(Symbol *sym)	/* jump to this symbol def */
{
  if (!code_inline)
    spew_jump(sym);
  else
    code_jump(sym);
}


void
code_load_addr(sexp *s, Int size) /* loads D acc from sexp */
{
  if (s->exp == val)		/* have an indirect address */
    {
      Symbol *sym = (Symbol *)s->val.p;
      Value *value = symtab_get(sym);

      switch(value_loctype(value))
	{
	case global_location:
	case zero_page_global_location:
	  code_mop(ldy);
	  spew_label_reference(sym);
	  break;
	case stack_location:
	  code_mop(ldyx);
	  spew_byte(current_stack_size - value_address(value));
	  break;
	}
      if (size == 1)
	code_m2op(ldby, 0);
      else
	code_m2op(lddy, 0);
    }
  else
    {
      if (size == 1)
	code_mop(ldb);
      else
	code_mop(ldd);
      switch(type_id(s->type))
	{
	case int_id:
	  spew_word(s->val.i);
	  break;

	case long_id:
	  spew_word(s->val.l);
	  break;

	default:
	  die(("Should only load an int or long address\n"));
	  break;
	}
    }
  if (size == 1) code_mop(clra);
}


void
code_store_addr(sexp *s, Int size) /* direct operation on sexp */
{
  if (s->exp == val || s->exp == ref) /* have an indirect address */
    {
      Symbol *sym = (Symbol *)s->val.p;
      Value *value = symtab_get(sym);

      switch(value_loctype(value))
	{
	case global_location:
	case zero_page_global_location:
	  code_mop(ldy);
	  spew_label_reference(sym);
	  break;
	case stack_location:
	  code_mop(ldyx);
	  spew_byte(current_stack_size - value_address(value));
	  break;
	}
      if (size == 1)
	code_m2op(stby, 0);
      else
	code_m2op(stdy, 0);
    }
  else
    {
      if (size == 1)
	code_mop(stb);
      else
	code_mop(std);
      switch(type_id(s->type))
	{
	case int_id:
	  spew_word(s->val.i);
	  break;

	case long_id:
	  spew_word(s->val.l);
	  break;

	default:
	  die(("Should directly address only an int or long\n"));
	  break;
	}
    }
}



void
code_op_addr(sexp *s, mcodes op, mcodes opind, mcodes opx) /* direct operation on sexp */
{
  if (s->exp == val && s->arg1) s = s->arg1; /* constant memory reference */
  if (s->exp == val || s->exp == ref) /* have an indirect address */
    {
      Symbol *sym = (Symbol *)s->val.p;
      Value *value = symtab_get(sym);

      switch(value_loctype(value))
	{
	case global_location:
	case zero_page_global_location:
	  code_mop(opind);
	  spew_label_reference(sym);
	  break;
	case stack_location:
	  code_mop(opx);
	  spew_byte(current_stack_size - value_address(value));
	  break;
	}
    }
  else if (s->exp == cnst)	/* have a constant memory reference */
    {
      if (s->op == minus && type_id(s->type) == pointer_id) 
				/* constant memory reference */
	code_mop(opind);
      else
	code_mop(op);
      spew_word(s->val.i);
    }
  else
    {
      code_mop(op);
      switch(type_id(s->type))
	{
	case char_id:
	case int_id:
	  spew_word(s->val.i);
	  break;

	case long_id:
	  spew_word(s->val.l);
	  break;

	default:
	  die(("Should directly address only an int or long\n"));
	  break;
	}
    }
}

void
code_store_to_addr(sexp *s)
{
  if (ref_type(s) == char_id)
    code_op_addr(s,stb,stb,stbx);
  else
    code_op_addr(s,std,std,stdx);
}

void
code_add_addr(sexp *s)
{
  if (ref_type(s) == char_id)
    code_op_addr(s,addi,addb,addbx);
  else
    code_op_addr(s,addi,add,addx);
}

void
code_sub_addr(sexp *s)
{
  if (ref_type(s) == char_id)
    code_op_addr(s,subi,subb,subbx);
  else
  code_op_addr(s,subi,sub,subx);
}



void				/* same as before, only operate on A, B separately */
code_op1_addr(sexp *s, mcodes opa, mcodes opb,
	      mcodes opinda, mcodes opindb, 
	      mcodes opxa, mcodes opxb) /* direct operation on sexp */
{
  if (s->exp == val && s->arg1) s = s->arg1; /* constant memory reference */
  if (s->exp == val || s->exp == ref) /* have an indirect address */
    {
      Symbol *sym = (Symbol *)s->val.p;
      Value *value = symtab_get(sym);

      switch(value_loctype(value))
	{
	case global_location:
	case zero_page_global_location:
	  if (ref_type(s) == char_id) /* just do lower half */
	    {
	      code_mop(opindb);
	      spew_label_reference(sym);
	      code_mop(opa);
	      spew_byte(0);
	    }
	  else
	    {
	      code_mop(opinda);
	      spew_label_reference(sym);
	      code_mop(opindb);
	      spew_label_reference_offset(sym,1);
	    }
	  break;
	case stack_location:
	  if (ref_type(s) == char_id) /* just do lower half */
	    {
	      code_mop(opindb);
	      spew_byte(current_stack_size - value_address(value));
	      code_mop(opa);
	      spew_byte(0);
	    }
	  else
	    {
	      code_mop(opxa);
	      spew_byte(current_stack_size - value_address(value));
	      code_mop(opxb);
	      spew_byte(1 + current_stack_size - value_address(value));
	    }
	  break;
	}
    }
  else if (s->exp == cnst)	/* have a constant memory reference */
    {
      if (s->op == minus && type_id(s->type) == pointer_id)
				/* constant memory reference */
	{
	  if (ref_type(s) == char_id) /* just do lower half */
	    {
	      code_mop(opindb);
	      spew_word(s->val.i);	  
	      code_mop(opa);
	      spew_byte(0);
	    }
	  else
	    {
	      code_mop(opinda);
	      spew_word(s->val.i);	  
	      code_mop(opindb);
	      spew_word(1+s->val.i);	  
	    }
	}
      else
	{
	  code_mop(opa);
	  spew_byte((s->val.i)>>8);
	  code_mop(opb);
	  spew_byte(s->val.i);
	}
    }
  else
    {
      switch(type_id(s->type))
	{
	case int_id:
	  code_mop(opb);
	  spew_byte(s->val.i);
	  code_mop(opa);
	  spew_byte((s->val.i)>>8);
	  break;

	case long_id:
	  code_mop(opb);
	  spew_byte(s->val.l);
	  code_mop(opa);
	  spew_byte((s->val.l)>>8);
	  break;

	default:
	  die(("Should directly address only an int or long\n"));
	  break;
	}
    }
}

void				/* immediate logical operation */
code_bit_addr(sexp *a1, eop op)
{
  switch(op)
    {
    case bitand:
    case bitandequal:
      code_op1_addr(a1, andai, andbi, anda, andb, andax, andbx);
      break;
    case bitior:
    case bitorequal:
      code_op1_addr(a1, orai, orbi, ora, orb, orax, orbx);
      break;
    case bitxor:
    case bitxorequal:
      code_op1_addr(a1, eorai, eorbi, eora, eorb, eorax, eorbx);
      break;
    }
}


/*
 * Comparison coding
 */

void				/* load 1 or 0, depending on C register + op */
code_compare_result(eop op, Int sw)
{
  switch(op)
    {
    case eequal:
      code_load_cond(brz);
      break;
    case nequal:
      code_load_cond(brnz);
      break;
    case lessthan:
      code_load_cond(sw ? brlt : brgt);
      break;
    case greaterthan:
      code_load_cond(sw ? brgt : brlt);
      break;
    case ltequal:
      code_load_cond(sw ? brle : brge);
      break;
    case gtequal:
      code_load_cond(sw ? brge : brle);
      break;
    }      
}

void				/* do a compare, then load 0 or 1 appropriately */
code_compare_addr(sexp *s, eop op, Int sw)
{
  code_op_addr(s, cpdi, cpd, cpdx);
  code_compare_result(op,sw);
}


/*
 * Push an integer onto stack
 */
void
code_push_int(Int i)
{
  if (!code_inline) spew_push_int(i);
  else
    {
      code_load_int(i);
      code_store_stack();
      define_push(type_Int());
    }
}

/* 
 * constants
 * takes constant value and type
 * in pcode generation, places value on stack
 * in inline generation, checks value of code_push to see if it
 *   returns a value or stuffs it on the stack
 */

void
code_const(sexp *s, Int code_push)
{
  if (!code_inline)		/* generate pcode here */
    {
      switch(type_id(s->type))
	{
	case char_id:
	  spew_push_int(s->val.i);
	  break;
	case int_id:
	  spew_push_int(s->val.i);
	  break;
	case long_id:
	  spew_push_long(s->val.l);
	  break;
	case float_id:
	  spew_push_float(s->val.d);
	  break;
	case pointer_id:
	  {
	    if (s->op == plus)	/* have a string to store */
	      {
		int i, len= strlen((char *)s->val.p) - 1, ref;
		Symbol *end_of_string= define_temp_symbol();
		
		spew_jump(end_of_string);
		ref = code_current;
		spew_word(len);
		for (i= 1; ((char *)(s->val.p))[i+1]; i++) spew_byte(((char *)(s->val.p))[i]);
		spew_byte(0);
		spew_label_definition(end_of_string);
		s->op = minus;	/* say we've stored it */
		s->val.i = ref;
	      }
	    spew_op(Ppush2);
	    spew_word(s->val.i);
	    define_push(s->type);
	  }
	  break;
	}
    }

  else				/* generate inline code */
    {
      switch(type_id(s->type))
	{
	case char_id:
	case int_id:
	  code_load_int(s->val.i);
	  if (code_push)
	    {
	      code_store_stack();
	      define_push(type_Int());
	    }
	  break;

	case float_id:
	  s->val.l = spew_float_6811_convert(s->val.d);
	case long_id:
	  code_load_int(s->val.l); /* load the lower half */
	  if (code_push)
	    {			/* have to store upper half too */
	      code_store_stack();
	      code_load_int((s->val.l)>>16); /* load the upper half */
	      code_store_stack();
	      if (type_id(s->type) == float_id)
		define_push(type_Float());
	      else 
		define_push(type_Long());
	    }
	  else if (type_id(s->type) == float_id)
	    user_error(("Illegal type: float\n"));
	  break;

	case pointer_id:
	  {
	    if (s->op == plus)	/* have a string to store */
	      {
		int i, len= strlen((char *)s->val.p) - 1, ref;
		Symbol *end_of_string= define_temp_symbol();
		
		code_jump(end_of_string);
		ref = code_current;
		spew_word(len);
		for (i= 1; ((char *)(s->val.p))[i+1]; i++) spew_byte(((char *)(s->val.p))[i]);
		spew_byte(0);
		spew_label_definition(end_of_string);
		s->op = minus;	/* say we've stored it */
		s->val.i = ref;
	      }
	    code_load_int(s->val.i);
	    if (code_push)
	      {
		code_store_stack();
		define_push(s->type);
	      }
	  }
	  break;
	}
    }
}


/* References */

void
code_ref(Symbol *sym, Type *type, Int code_push)
{
  Value *value = symtab_get(sym);

  if (!value) 
    {
      if (code_push) define_push(type_Undef());
      return;
    }
  if (value->loctype == constant) /* can't assign a constant variable */
    {
      user_error(("Attempt to assign the constant %s\n",
		  symbol_name(sym)));
      define_push(type_Pointer(value_type(value)));
    }
  else
    {
      if (!code_inline)		/* generate pcode here */
	define_get_symbol_addr(sym);
      
      else				/* generate inline code */
	{
	  switch(value_loctype(value))
	    {
	    case global_location:
	    case zero_page_global_location:
	      code_mop(lddi);
	      spew_label_reference(sym);
	      break;
	    case stack_location:
	      code_mop(pshx);	/* get X into D */
	      code_mop(pula);
	      code_mop(pulb);
	      code_mop(addi);
	      spew_word(current_stack_size - value_address(value));
	      break;
	    }
	  code_store_stack();
	  define_push(type_Pointer(value_type(value)));
	}
    }
}


/* Dereferences */

void
code_deref(sexp *arg, Type *type, Symbol *sym, Int code_push)
{
  if (!code_inline)		/* do pcodes */
    {
      if (!arg)			/* symbol deref, get it */
	code_ref(sym,type,code_push);
      else			/* else evaluate argument */
	code_sexp_push(arg);
      spew_peek();
    }
				
  else				/* do native thang */
    {				/* just load it using D accum, stores */
      if (arg && arg->exp != cnst)
	{
	  code_sexp_load(arg);	/* get address in D accum */
	  code_mop(xcdy);	/* put into Y register */
	}
      switch(type_id(type))
	{
	case char_id:		/* load a char */
	  if (arg && arg->exp !=cnst) /* pointer */
	    code_m2op(ldby, 0);
	  else if (arg)		/* constant pointer */
	    {
	      code_mop(ldb);
	      spew_word(arg->val.i);
	    }
	  else			/* variable */
	    code_load_val(sym,0,0);
	  code_mop(clra);
	  if (code_push)
	    {
	      code_store_stack();
	      define_push(type);
	    }
	  break;

	case int_id:		/* load an int */
	case pointer_id:	/* or a pointer */
	  if (arg && arg->exp !=cnst) /* pointer */
	    code_m2op(lddy, 0);
	  else if (arg)		/* constant pointer */
	    {
	      code_mop(ldd);
	      spew_word(arg->val.i);
	    }
	  else			/* variable */
	    code_load_val(sym,0,0);
	  if (code_push)
	    {
	      code_store_stack();
	      define_push(type);
	    }
	  break;

	case long_id:		/* load 4 bytes */
	case float_id:
	  if (code_push)
	    {
	      if (arg && arg->exp != cnst)
		code_m2op(lddy, 2);
	      else if (arg)
		{
		  code_mop(ldd);
		  spew_word(arg->val.i);
		}
	      else
		code_load_val(sym,2,0);
	      code_store_stack();
	      if (arg && arg->exp != cnst)
		code_m2op(lddy, 0);
	      else if (arg)
		{
		  code_mop(ldd);
		  spew_word(arg->val.i+0);
		}
	      else
		code_load_val(sym,0,2);
	      code_store_stack();
	      define_push(type);
	    }
	  else if (type_id(type) == float_id) 
	    user_error(("Illegal type: float\n"));
	  else			/* load lower half of long here */
	    {
	      if (arg && arg->exp != cnst)
		code_m2op(lddy, 2);
	      else if (arg)
		{
		  code_mop(ldd);
		  spew_word(arg->val.i+2);
		}
	      else
		code_load_val(sym,2,0);
	      define_push(type);
	    }
	  break;

	default:
	  user_error(("Illegal type in reference\n"));
	  define_push(type);
	  break;
	}
    }
}

/* array dereference, currently only with nonnative code */

void
code_aderef(sexp *arr, sexp *arg, Type *type, Int code_push)
{
  if (!code_inline)		/* do pcodes */
    {
      code_sexp_push(arr);
      code_sexp_push(arg);
      spew_array_reference();
      spew_peek();
    }
  else
    {
      user_error(("Can't reference arrays in native code\n"));
      define_push(type_Undef());
    }
}

void
code_aref(sexp *arr, sexp *arg, Type *type, Int code_push)
{
  if (!code_inline)		/* do pcodes */
    {
      code_sexp_push(arr);
      code_sexp_push(arg);
      spew_array_reference();
    }
  else
    {
      user_error(("Can't reference arrays in native code\n"));
      define_push(type_Undef());
    }
}

/* PEEK exp */

void
code_sexp_peek(sexp *a, Int size, Int code_push)
{
  if (!code_inline)
    {
      code_sexp_push(a);
      if (size == 1)
	define_cast(type_Pointer(type_Char()));
      else
	define_cast(type_Pointer(type_Int()));
      spew_peek();
    }
  else
    {
      if (address_arg(a) || indirect_arg(a)) /* argument can be an address */
	code_load_addr(a,size);
      else
	{
	  code_sexp_load(a);	/* code into D register */
	  code_mop(xcdy);	/* get into Y reg */
	  if (size == 1)
	    code_load1_from_y(); /* load using y reg */
	  else
	    code_load_from_y();	/* load using y reg */
	}
    }
}

/*
 * Unary operations 
 */

void
code_sexp_unop(eop op, sexp *a1, Type_id type, Int code_push)
{
  if (!code_inline ||		/* do the pcode thang */
      (type != int_id && type != char_id && type != pointer_id)) /* can't optimize */
    {
      code_sexp_push(a1);
      switch(op)
	{
	case neg:
	  spew_unop('-');
	  break;
	case lognot:
	  spew_unop('!');
	  break;
	case bitcomp:
	  spew_unop('~');
	  break;

	case int_to_long:
	case long_to_int:
	case int_to_float:
	case long_to_float:
	case float_to_long:
	case float_to_int:
	  define_coerce(type_Simple(type));
	  break;

	case ssin:
	  spew_unop('s');
	  break;
	case scos:
	  spew_unop('c');
	  break;
	case stan:
	  spew_unop('t');
	  break;
	case ssqrt:
	  spew_unop('q');
	  break;
	case satan:
	  spew_unop('a');
	  break;
	case slog10:
	  spew_unop('L');
	  break;
	case sloge:
	  spew_unop('l');
	  break;
	case sexp10:
	  spew_unop('E');
	  break;
	case sexpe:
	  spew_unop('e');
	  break;
	  

	default:
	  user_error(("No expression coding for operation %s\n", name_of_eop(op)));
	}
      if (!code_push)
	code_load_stack();	/* this pops the runtime stack */
    }
  else
    {
      if (op == float_to_int)
	{
	  code_sexp_push(a1);
	  define_coerce(type_Int());
	  if (!code_push) code_load_stack();
	}
      else if (op == long_to_int)
	{
	  code_sexp_push(a1);
	  code_long_to_int();
	  if (!code_push) code_load_stack();
	}

      else
	{
	  code_sexp_load(a1);
	  switch(op)
	    {
	    case neg:
	      code_negate();
	      break;
	    case lognot:
	      code_lognot();
	      break;
	    case bitcomp:
	      code_bitcomp();
	      break;
	  
	    default:
	      user_error(("No expression coding for operation %s\n", name_of_eop(op)));
	    }
	  if (code_push)
	    {
	      code_store_stack();
	      define_push(type_Int());
	    }
	}
    }
}


/*
 * Logical operations
 */

void
code_logop_branch(Symbol *l1, eop op)
{
  code_mop(cpdi);		/* Bleachh, may have had a pop stack */
  spew_word(0);
  switch(op)
    {
    case logior:
      if (l1)
	{
	  code_m2op(brz, 6);	/* 3 to load int, 3 to jump */
	  code_load_int(1);
	  code_mop(jmp);
	  spew_label_reference(l1);
	}
      else
	{
	  code_m2op(brz, 3);	/* 3 to load int */
	  code_load_int(1);
	}
      break;
    case logand:
      if (l1)
	{
	  code_m2op(brnz,3);
	  code_mop(jmp);
	  spew_label_reference(l1);
	}
      else
	{
	  code_m2op(brz,3);
	  code_load_int(1);
	}
    }
}


void
code_logop(eop op, sexp *a1, sexp *a2)
{
  Symbol *l1;

  code_sexp_load(a1);		/* get it in the D reg */
  l1 = define_temp_symbol();
  code_logop_branch(l1, op);	/* set a 0 or 1 and branch appropriately */
  code_sexp_load(a2);
  code_logop_branch(NULL, op);	/* don't have to jump at the end */
  spew_label_definition(l1);
}

/*
 * Binary operations 
 */

void
code_sexp_bop(eop op, sexp *a1, sexp *a2, Type_id type, Int code_push)
{
  Symbol *l1;

/*  if (undefined(a1) || undefined(a2)) 
    {
      define_push(type_Undef());
      return;
    }
*/
  if (!code_inline ||		/* do the pcode thang */
      type != int_id || 
	  (op != plus && op != minus && op != bitand && op != bitior &&
	   op != bitxor && op != eequal && op != nequal && op != ltequal &&
	   op != gtequal && op != lessthan && op != greaterthan &&
	   op != logior && op != logand) )
				/* can't do optimization */
    {
      code_sexp_push(a1);
      switch(op)
	{
	case plus:
	  code_sexp_push(a2);
	  spew_binop('+');
	  break;
	case times:
	  code_sexp_push(a2);
	  spew_binop('*');
	  break;
	case divide:
	  code_sexp_push(a2);
	  spew_binop('/');
	  break;
	case minus:
	  code_sexp_push(a2);
	  spew_binop('-');
	  break;
	case modulo:
	  code_sexp_push(a2);
	  if (type_id(define_stack_type(0)) == float_id &&
	      pass == 1) {
	    user_error(("The operator %% is not an floating point operator\n"));
	  }
	  spew_speek(1); spew_speek(1); spew_binop('/');
	  spew_binop('*');  spew_binop('-');
	  break;
	case eequal:
	  code_sexp_push(a2);
	  spew_binop('=');
	  break;
	case nequal:
	  code_sexp_push(a2);
	  spew_binop('=');
	  spew_unop('!');
	  break;
	case lessthan:
	  code_sexp_push(a2);
	  spew_binop('<');
	  break;
	case greaterthan:
	  code_sexp_push(a2);
	  spew_binop('>');
	  break;
	case ltequal:
	  code_sexp_push(a2);
	  spew_binop('>');
	  spew_unop('!');
	  break;
	case gtequal:
	  code_sexp_push(a2);
	  spew_binop('<');
	  spew_unop('!');
	  break;
	case bitand:
	  code_sexp_push(a2);
	  spew_binop('&');
	  break;
	case bitior:
	  code_sexp_push(a2);
	  spew_binop('|');
	  break;
	case bitxor:
	  code_sexp_push(a2);
	  spew_binop('^');
	  break;
	case lshift:
	  code_sexp_push(a2);
	  spew_binop('l');
	  break;
	case rshift:
	  code_sexp_push(a2);
	  spew_unop('-');
	  spew_binop('l');
	  break;
	case logand:
	  l1 = define_temp_symbol();
	  spew_conditional_jump(Pjpfalse,l1);
	  code_sexp_push(a2);
	  spew_unop('=');
	  spew_label_definition(l1);
	  break;
	case logior:
	  l1 = define_temp_symbol();
	  spew_conditional_jump(Pjptrue,l1);
	  code_sexp_push(a2);
	  spew_unop('=');
	  spew_label_definition(l1);
	  break;
	  
	default:
	  user_error(("No expression coding for operation %s\n", name_of_eop(op)));
	}
      if (code_inline && !code_push)
	code_load_stack();	/* this pops the runtime stack */
    }
  else
    {

      if (op == logior || op == logand) /* handle these specially */
	{
	  code_logop(op, a1, a2);
	  if (code_push)
	    {
	      code_store_stack();
	      define_push(type_Int());
	    }
	}

      else if (address_arg(a2) || indirect_arg(a2)) /* second argument is loadable */
	{
	  code_sexp_load(a1);	/* code it, leave in acc D */
	  switch(op)
	    {
	    case plus:
	      code_add_addr(a2);
	      break;
	    case minus:
	      code_sub_addr(a2);
	      break;
	    case bitand:
	    case bitior:
	    case bitxor:
	      code_bit_addr(a2,op);
	      break;

	    case eequal:
	    case nequal:
	    case ltequal:
	    case gtequal:
	    case lessthan:
	    case greaterthan:
	      code_compare_addr(a2,op,1);
	      break;

	    default:
	      user_error(("No expression coding for operation %s\n", name_of_eop(op)));
	    }
	  if (code_push)
	    {
	      code_store_stack();
	      define_push(type_Int());
	    }
	}
      
      else if (address_arg(a1) || indirect_arg(a1)) /* first argument is loadable */
	{
	  /* *** NOTE: not strictly correct if assignments are made *** */
	  code_sexp_load(a2);	/* code it, leave in acc D */
	  switch(op)
	    {
	    case plus:
	      code_add_addr(a1);
	      break;
	    case minus:
	      code_negate();
	      code_add_addr(a1);
	      break;

	    case bitand:
	    case bitior:
	    case bitxor:
	      code_bit_addr(a1,op);
	      break;

	    case eequal:
	    case nequal:
	    case ltequal:
	    case gtequal:
	    case lessthan:
	    case greaterthan:
	      code_compare_addr(a1,op,0);
	      break;

	    default:
	      user_error(("No expression coding for operation %s\n", name_of_eop(op)));
	    }
	  if (code_push)
	    {
	      code_store_stack();
	      define_push(type_Int());
	    }
	}
      
      else
	{
	  code_sexp_push(a2);
	  code_sexp_load(a1);
	  switch(op)
	    {
	    case plus:
	      code_m2op(addx, 0);
	      break;
	    case minus:
	      code_m2op(subx, 0);
	      break;
	    case bitand:
	      code_m2op(andax, 0);
	      code_m2op(andbx, 0);
	      break;
	    case bitior:
	      code_m2op(orax, 0);
	      code_m2op(orbx, 0);
	      break;
	    case bitxor:
	      code_m2op(eorax, 0);
	      code_m2op(eorbx, 0);
	      break;

	    case eequal:
	    case nequal:
	    case ltequal:
	    case gtequal:
	    case lessthan:
	    case greaterthan:
	      code_m2op(cpdx, 0);
	      code_compare_result(op,1);
	      break;

	    default:
	      user_error(("No expression coding for operation %s\n", name_of_eop(op)));
	    }
	  if (code_push)
	    code_m2op(stdx, 0);
	  else
	    code_pop_stack();
	}
      
    }
}


/*
 * Assignment operations
 */

void
code_sexp_assign(eop op, sexp *a1, sexp *a2, Type_id type, Int code_push)
{
  if (!code_inline)		/* do the pcode thang */
    {
      code_sexp_push(a1);
      if (op == equal)
	{
	  code_sexp_push(a2);
	  spew_poke(FALSE);
	}
      else
	{
	  spew_dup();
	  spew_peek();
	  code_sexp_push(a2);
	  switch(op)
	    {
	    case plusequal:
	      spew_binop('+');
	      break;
	    case minusequal:
	      spew_binop('-');
	      break;
	    case timesequal:
	      spew_binop('*');
	      break;
	    case divideequal:
	      spew_binop('/');
	      break;
	    case bitandequal:
	      spew_binop('&');
	      break;
	    case bitorequal:
	      spew_binop('|');
	      break;
	    case bitxorequal:
	      spew_binop('^');
	      break;
	    case lshiftequal:
	      spew_binop('l');
	      break;
	    case rshiftequal:
	      spew_unop('-');
	      spew_binop('l');
	      break;
	    default:
	      user_error(("No expression coding for operation %s\n", name_of_eop(op)));	      
	    }
	  spew_poke(FALSE);
	}
    }
  else
    {
      if ((type_id(a1->type) == int_id || type_id(a1->type) == pointer_id
	                               || type_id(a1->type) == char_id) &&
	  (op == equal || op == plusequal || op == minusequal ||
	   op == bitandequal || op == bitorequal || op == bitxorequal))
	{			/* do a 2-byte poke */
	  if (ref_arg(a1))
				/* first arg is address */
	    {
	      code_sexp_load(a2); /* code it, leave in acc D */
	      switch(op)
		{
		case equal:
		  break;
		case minusequal:
		  code_negate();
		case plusequal:
		  code_add_addr(a1);
		  break;
		case bitandequal:
		case bitorequal:
		case bitxorequal:
		  code_bit_addr(a1,op);
		  break;
		}
	      code_store_to_addr(a1); /* code store to address */
	    }

	  else			/* can do a simple addition */
	    {
	      code_sexp_push(a1); /* code and push */
	      define_cast(type_Pointer(type_Int())); 
	      code_sexp_load(a2); /* code it, leave in acc D */
	      code_pop_to_y();	/* get stack into Y reg */
	      define_pop();
	      switch(op)
		{
		case equal:
		  break;
		case minusequal:
		  code_negate();
		case plusequal:
		  code_m2op(addy, 0);
		  break;
		}
	      code_store_to_y(); /* store using y reg */
	    }
	  if (code_push)
	    {
	      code_store_stack();
	      define_push(type_Int());
	    }
	}

      else
	/* *** COULD CHECK HERE FOR SYMBOL ASSIGNMENT, BUT I'M TOO TIRED */
	{
	  code_sexp_push(a1);
	  switch(op)
	    {
	    case equal:
	      code_sexp_push(a2);
	      spew_poke(FALSE);
	      break;
	    case plusequal:
	      code_dup_val();
	      code_sexp_push(a2);
	      spew_binop('+');
	      spew_poke(FALSE);
	      break;
	    case minusequal:
	      code_dup_val();
	      code_sexp_push(a2);
	      spew_binop('-');
	      spew_poke(FALSE);
	      break;
	    case timesequal:
	      code_dup_val();
	      code_sexp_push(a2);
	      spew_binop('*');
	      spew_poke(FALSE);
	      break;
	    case divideequal:
	      code_dup_val();
	      code_sexp_push(a2);
	      spew_binop('/');
	      spew_poke(FALSE);
	      break;
	    default:
	      user_error(("No expression coding for operation %s\n", name_of_eop(op)));
	      return;
	    }
	  if (!code_push)
	    code_load_stack();	/* this pops the runtime stack */
	}
    }
}

/* Function calling */

void
code_begin_call(Symbol *func, Symbol *temp)
{
  if (!code_inline)
    spew_begin_call(func,temp);
  else
    {
      Value *func_val= symtab_get(func);
    
      if (func_val &&
	  type_id(value_type(func_val)) == func_id) {

	switch (type_Func_calling_convention(value_type(func_val))) 
	  {
	  case pcode_calling_convention:
	    code_mop(lddi);
	    spew_label_reference(temp);
	    code_store_stack();
	    break;
	  case ml_calling_convention:
	    code_mop(lddi);
	    spew_label_reference(func);
	    code_store_stack();
	    break;
	  default:
	    die(("Illegal calling convention in begin_call"));
	}
      }
      define_push(type_Label());
      define_begin_block();
    }
}

void
code_end_call(Symbol *func, Symbol *temp)
{
  if (!code_inline)
    spew_end_call(func,temp);
  else
    {  
      Type  *proc_type;
      Value *func_val= symtab_get(func);
      Int nat = 0;

      proc_type= spew_check_funcall(func);
    
      if (func_val &&
	  type_id(value_type(func_val)) == func_id) {

	switch (type_Func_calling_convention(value_type(func_val))) 
	  {
	  case pcode_calling_convention: /* load Y with function address, return */
	    code_mop(ldyi);
	    spew_label_reference(func);
	    code_mop(rts);	/* go back to Pcode interpreter */
	    nat = 1;
	    break;
	  case ml_calling_convention:
	    spew_op(Pcallml);
	    break;
	  }
      }
      define_end_block_no_pop();
      define_pop(); /* pop call address or return address */
      define_push(proc_type);
      spew_label_definition(temp);
      if (nat)
	{
	  code_inline = 0;
	  spew_op(Pnative);	/* return here... */
	  code_inline = 1;
	}
    }
}


void
code_sexp_call(Symbol *func, sexp *s)
{
  Symbol *l = define_temp_symbol();

  code_begin_call(func,l);
  while(s) { code_sexp_push(s); s = s->next; }
  code_end_call(func,l);
}

void
code_printf(sexp *s)
{
  int nargs = 1, m = 0;

  define_begin_block();
  while(s) { code_sexp_push(s); s = s->next; }  
  for (nargs= 0; type_id(define_stack_type(nargs)) != block_id; nargs++);
  code_push_int((Int) (define_stack_offset(nargs) - 2));
  spew_op(Pprintf);
  define_end_block_no_pop();
  define_push(type_Int());
}

void
code_start_process(sexp *s, Symbol *sym)
{
  sexp *a;
  if (code_inline) 
    user_error(("Can't use START_PROCESS in native mode"));
  else
    {
      spew_begin_start_process(sym);
      a = s->arg1;
      s = s->next;
      while (a) { code_sexp_push(a); a = a->next; }
      spew_continue_start_process(sym);
      code_sexp_push(s);
      s = s->next;
      code_sexp_push(s);
      spew_end_start_process(sym);
    }
}

void
code_kill_process(sexp *s)
{
  if (code_inline) user_error(("Can't use KILL_PROCESS in native mode"));
  else
    {
      code_sexp_push(s);
      spew_kill_process();
    }
}

/*
 * Nilary operations
 */

void
code_sexp_nilop(eop op, Type_id type, Int code_push)
{
  switch (op)
    {
    case mseconds:
      if (code_inline)
	{
	  code_load_long_val(0x12); /* this is the mseconds counter */
	}
      else
	{
	  spew_op(Psystime);  
	  define_push(type_Long());   
	}
      break;
    case benchmark:
      if (code_inline)
	printf("BENCHMARK not supported in native mode\n");
      else
	spew_op(Pbench);  
      define_push(type_Int());
      break;
    }
}

/*
 * Bit set and clear
 */

void
code_bit(Int which)		/* 0 for clear, 1 for set */
{
  sexp *s = pop_sexp();
  if (!code_inline)
    {
      code_sexp_push(pop_sexp());
      code_sexp_push(s);
      spew_bit_set_clr(which ? Pbitset : Pbitclr);
    }
  else
    {
      pop_sexp();
      printf("BITSET and BITCLR not defined in native mode; \n \
please use *a |= x and *a &= x constructions\n");
    }
}


/* 
 * Expression coding.  Pops one sexp off the stack, codes it.
 */

void
code_sexp_push(sexp *s)
{
  code_sexp(s,1);
}

void
code_sexp_load(sexp *s)
{
  code_sexp(s,0);
}

void
code_sexp(sexp *s, Int code_push)
{
  etype e = s->exp;

  switch(e)
    {
    case cnst:			/* do the constant thang */
      code_const(s, code_push);
      break;
      
    case ref:			/* get a reference here */
      code_ref((Symbol *)s->val.p, s->type, code_push);
      break;

    case val:			/* get a dereference */
      code_deref(s->arg1, s->type, (Symbol *)s->val.p, code_push);
      break;

    case aref:			/* get an array reference here */
      code_aref(s->arg1, s->arg2, s->type, code_push);
      break;

    case aval:			/* get an array dereference */
      code_aderef(s->arg1, s->arg2, s->type, code_push);
      break;

    case peek:			/* get a dereference */
      code_sexp_peek(s->arg1, s->op, code_push);
      break;

    case nilop:
      code_sexp_nilop(s->op, type_id(s->type), code_push);
      break;

    case unop:
      if (s->op == kill_process)
	code_kill_process(s->arg1);
      else
	code_sexp_unop(s->op, s->arg1, type_id(s->type), code_push);
      break;

    case bop:
      code_sexp_bop(s->op, s->arg1, s->arg2, type_id(s->type), code_push);
      break;

    case assign:
      code_sexp_assign(s->op, s->arg1, s->arg2, type_id(s->type), code_push);
      break;

    case call:
      switch(s->op)
	{
	case prntf:
	  code_printf(s->arg1);
	  break;
	case start_process:
	  code_start_process(s->arg1, s->arg1->val.p);
	  break;
	default:
	  code_sexp_call((Symbol *)s->val.p, s->arg1);
	}
      break;

    default:
      die(("No code generation for sexp %d\n", e));
      break;
    }
}

/* POKE exp exp */

void
code_poke_it(sexp *a1, sexp *a2, Int size)
{
  if (address_arg(a1) || indirect_arg(a1)) /* first argument can be an address */
    {
      code_sexp_load(a2);	/* code it, leave in acc D */
      code_store_addr(a1,size);	/* code store to address */
    }
  else
    {
      code_sexp_push(a1);	/* code and push */
      define_cast(type_Pointer(type_Char())); 
      code_sexp_load(a2);	/* code it, leave in acc D */
      code_pop_to_y();		/* get stack into Y reg */
      define_pop();
      if (size == 1)
	code_store_to_y();	/* store using y reg */
      else
	code_store1_to_y();	/* store using y reg */
    }
}

void
code_poke(Int size)
{
  sexp *a2 = pop_sexp();
  sexp *a1 = pop_sexp();

  if (!code_inline)
    {
      code_sexp_push(a1);
      if (size == 1)
	define_cast(type_Pointer(type_Char()));
      else
	define_cast(type_Pointer(type_Int()));
      code_sexp_push(a2);
      spew_poke(TRUE);
    }
  else
    code_poke_it(a1,a2,size);
}


/* Conditionals, finally... */

void
code_sexp_cond(sexp *a, Symbol *s, Int branch)
{
      code_sexp_load(a);
      if (branch)		/* branch when true */
	{
	  code_mop(subi);	/* test just to make sure, gag */
	  spew_word(0);
	  code_m2op(brz, 3);	/* skip around jump */
	}
      else
	{
	  code_mop(subi);	/* test just to make sure, gag */
	  spew_word(0);
	  code_m2op(brnz, 3);	/* skip around jump */
	}
      code_jump(s);
}



void				/* expects an expression on the sexp stack */
code_sexp_conditional(Int branch, Symbol *s)
{
  /*** should check suitability of sem expression here ****/

  if (!code_inline)
    {
      code_sexp_push(pop_sexp());
      if (branch)		/* branch when true */
	spew_conditional_jump(Pjtrue, s);
      else
	spew_conditional_jump(Pjfalse, s);
    }
  else
    code_sexp_cond(pop_sexp(),s,branch);
}


void				/* expects a jump on the sexp stack */
code_sexp_jump(Int jump)
{
  sexp *s = pop_sexp();
  if (jump)			/* yes, do the jump */
    code_imm_jump((Symbol *)s->arg1);
  spew_label_definition((Symbol *)s->arg2);
}


/* expressions as statements */

void
code_sexp_noval()		/* evals sexp, no return value */
{
  if (!code_inline)
    {
      code_sexp_push(pop_sexp());
      spew_pop();
    }
  else
    code_sexp_load(pop_sexp());
}

void
code_sexp_val()			/* evals sexp, return value */
{
  code_sexp_push(pop_sexp());
}

void
code_sexp_list(Int n)		/* code n sexps and push on stack, in reverse order */
{				/* if n=0, look for define block */
  sexp *s;
  int m = 0;

  if (n == 0)
    while (top_sexp(m++)->exp != callmark) {} /* now at right part of stack */
  else
    m = n+1;			/* or use arg */
  while(--m) code_sexp_push(top_sexp(m-1)); /* do args */
  if (n == 0) while (pop_sexp()->exp != callmark) {} /* pop stack */
  else while (n--) pop_sexp();
}


/* Procedure definition coding */

void
code_push_garbage(Type *type)	/* set aside space for variables */
{
  if (!code_inline)
    spew_push_garbage(type);

  else				/* *** NEED TO DO ARRAYS AT SOME POINT... */
    {				/* *** NO INITIALIZATION... */
      int size = type_sizeof(type);
      for (; size>0; size--) code_mop(dex);
      define_push(type);
    }
}

void
code_sexp_return(Int val)	/* val = 1 for return value expression */
{
  if (val) code_sexp_push(pop_sexp());
  if (native_flag)		/* pcodes again!!! */
    {
      code_mop(ldyi);
      spew_word(code_current+5);	/* 2 arg bytes, 3 JMP mret bytes */
    }
  spew_return();
}



/* Popping the stack */

void code_pop(void)
{
    Type *top_of_stack= define_pop();
    Int  size= type_sizeof(top_of_stack);

    if (size)
      {
	if (!code_inline)
	  spew_pop_op(size);
	else
	  while(size--) code_mop(inx);
      }
}


/* Pcode-only statements */

void
code_pcode(Opcode op)
{
  if (code_inline)
    user_error(("Operation %s only available in pcodes\n", opcode_name(op)));
  else
    switch(op)
      {
      case Pbitclr:
	spew_bit_set_clr(Pbitclr);
	break;
      case Pbitset:
	spew_bit_set_clr(Pbitset);
	break;
      case Pdefer:
	spew_op(Pdefer);
	break;
      case Pinitint:
	spew_op(Pinitint);
	break;
      }
}
